/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/referencecollection', [
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/deps/dependencyutils'
], function (IteratorUtils, SimpleMap, SheetUtils, DependencyUtils) {

    'use strict';

    // convenience shortcuts
    var RangeArray = SheetUtils.RangeArray;

    // class ReferenceCollection ==============================================

    /**
     * A data structure that stores information about the source references a
     * formula expression depends on.
     *
     * The collection contains entries for the following references:
     * - Range addresses of the cell references contained in the formula.
     * - The models of the [defined name]s contained in the formula.
     * - The models of the [table range]s contained in the formula.
     *
     * @constructor
     *
     * @param {SpreadsheetModel} docModel
     *  The document model containing all formulas.
     *
     * @param {Object} dependencies
     *  The resolved dependencies of a token array, as returned by the method
     *  TokenArray.getDependencies().
     */
    function ReferenceCollection(docModel, dependencies) {

        // maps sheet UIDs to range arrays
        this._rangesMap = new SimpleMap();
        // maps name keys to instances of NameModel
        this._nameMap = new SimpleMap();
        // maps table keys to instances of TableModel
        this._tableMap = new SimpleMap();

        // separate all source ranges by sheets (also the root ranges of cell formulas, and other formulas)
        dependencies.depRanges.concat(dependencies.rootRanges).unify().forEach(function (range3d) {
            var range2d = range3d.toRange();
            var sheetIterator = docModel.createSheetIntervalIterator(range3d.sheet1, range3d.sheet2, { types: 'worksheet' });
            IteratorUtils.forEach(sheetIterator, function (sheetModel) {
                this._rangesMap.getOrConstruct(sheetModel.getUid(), RangeArray).push(range2d);
            }, this);
        }, this);

        // merge the range addresses to prevent duplicates and overlapping ranges
        this._rangesMap.forEach(function (ranges, sheetUid) {
            this._rangesMap.insert(sheetUid, ranges.merge());
        }, this);

        // register the keys for all defined names referred by the token array
        _.each(dependencies.names, function (nameModel) {
            this._nameMap.insert(nameModel.getUid(), nameModel);
        }, this);

        // register the keys for all table ranges referred by the token array
        _.each(dependencies.tables, function (tableModel) {
            this._tableMap.insert(tableModel.getUid(), tableModel);
        }, this);

    } // class ReferenceCollection

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this collection is empty.
     *
     * @returns {Boolean}
     *  Whether this collection is empty.
     */
    ReferenceCollection.prototype.empty = function () {
        return this._rangesMap.empty() && this._nameMap.empty() && this._tableMap.empty();
    };

    /**
     * Invokes the passed callback function for each cell range contained in
     * this collection.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each cell range in this
     *  collection. Receives the following parameters:
     *  (1) {String} sheetUid
     *      The UID of the sheet containing the cell range currently visited.
     *  (2) {Range} range
     *      The cell range address.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    ReferenceCollection.prototype.forEachRange = function (callback, context) {
        this._rangesMap.forEach(function (ranges, sheetUid) {
            ranges.forEach(function (range) {
                callback.call(context, sheetUid, range);
            });
        });
    };

    /**
     * Invokes the passed callback function for the keys of all defined names
     * contained in this collection.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each key of a defined
     *  name in this collection. Receives the following parameters:
     *  (1) {String} nameKey
     *      The map key of a defined name (its UID), as used in the dependency
     *      manager.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    ReferenceCollection.prototype.forEachNameKey = function (callback, context) {
        this._nameMap.forEachKey(function (nameKey) { callback.call(context, nameKey); });
    };

    /**
     * Invokes the passed callback function for the keys of all table ranges
     * contained in this collection.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each key of a table
     *  range in this collection. Receives the following parameters:
     *  (1) {String} tableKey
     *      The map key of a table range (its UID), as used in the dependency
     *      manager.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    ReferenceCollection.prototype.forEachTableKey = function (callback, context) {
        this._tableMap.forEachKey(function (tableKey) { callback.call(context, tableKey); });
    };

    /**
     * Returns whether the specified name model is contained in this reference
     * collection.
     *
     * @param {NameModel} nameModel
     *  The model instance of a defined name.
     *
     * @returns {Boolean}
     *  Whether the name model is contained in this reference collection.
     */
    ReferenceCollection.prototype.containsName = function (nameModel) {
        return this._nameMap.has(nameModel.getUid());
    };

    /**
     * Returns whether the specified table model is contained in this reference
     * collection.
     *
     * @param {TableModel} tableModel
     *  The model instance of a table range.
     *
     * @returns {Boolean}
     *  Whether the table model is contained in this reference collection.
     */
    ReferenceCollection.prototype.containsTable = function (tableModel) {
        return this._tableMap.has(tableModel.getUid());
    };

    /**
     * Creates the string representation of all contents of this collection for
     * debug logging.
     *
     * @returns {String}
     *  The string representation of all contents of this collection.
     */
    ReferenceCollection.prototype.toString = function () {
        var labels = [];
        this.forEachRange(function (sheetUid, range) { labels.push(sheetUid + '!' + range); });
        this._nameMap.forEach(function (nameModel) { labels.push(DependencyUtils.getNameLabel(nameModel)); });
        this._tableMap.forEach(function (tableModel) { labels.push(tableModel.getName()); });
        return labels.join();
    };

    // exports ================================================================

    return ReferenceCollection;

});
