/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/formuladictionary', [
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (SimpleMap, SheetUtils) {

    'use strict';

    // convenience shortcuts
    var RangeSet = SheetUtils.RangeSet;

    // class FormulaDictionary ================================================

    /**
     * A container for all registered formula descriptors in a spreadsheet
     * document. The descriptors will be mapped by their source references, in
     * order to provide efficient methods for looking up formulas depending on
     * changed cells, or changed ancestor formulas in a dependency chain.
     *
     * @constructor
     */
    function FormulaDictionary() {

        // several formula descriptor maps attached to ranges in range sets, mapped by sheet UID
        this._rangeSets = new SimpleMap();
        // formula descriptors depending on a defined name, mapped by name key
        this._nameMaps = new SimpleMap();
        // formula descriptors depending on a table range, mapped by table key
        this._tableMaps = new SimpleMap();

    } // class FormulaDictionary

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this dictionary is completely empty.
     *
     * @returns {Boolean}
     *  Whether this dictionary is completely empty.
     */
    FormulaDictionary.prototype.empty = function () {
        return this._rangeSets.empty() && this._nameMaps.empty() && this._tableMaps.empty();
    };

    /**
     * Creates a clone of this formula dictionary, referring to the same
     * formula descriptors as this dictionary.
     *
     * @returns {FormulaDictionary}
     *  A clone of this formula dictionary.
     */
    FormulaDictionary.prototype.clone = function () {

        // start with an empty dictionary
        var clone = new FormulaDictionary();

        // clone the formula maps of the cell range sets
        clone._rangeSets = this._rangeSets.clone(function (rangeSet) {
            return rangeSet.clone(function (newRange, oldRange) {
                newRange.formulaMap = oldRange.formulaMap.clone();
            });
        });

        // clone the formula descriptors mapped by name keys
        clone._nameMaps = this._nameMaps.clone(function (formulaMap) {
            return formulaMap.clone();
        });

        // clone the formula descriptors mapped by table keys
        clone._tableMaps = this._tableMaps.clone(function (formulaMap) {
            return formulaMap.clone();
        });

        return clone;
    };

    /**
     * Inserts a new formula descriptor into this dictionary. The formula
     * descriptor will be mapped by all of its source references.
     *
     * @param {String} formulaKey
     *  The key of the passed formula descriptor.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The descriptor of a formula to be inserted into this dictionary.
     */
    FormulaDictionary.prototype.insertFormula = function (formulaKey, formulaDesc) {

        // insert the formula descriptor into the formula maps for all source ranges
        formulaDesc.references.forEachRange(function (sheetUid, range) {
            var rangeSet = this._rangeSets.getOrConstruct(sheetUid, RangeSet);
            range = rangeSet.insert(range);
            var formulaMap = range.formulaMap || (range.formulaMap = new SimpleMap());
            formulaMap.insert(formulaKey, formulaDesc);
        }, this);

        // insert the formula descriptor into the formula maps for all defined names
        formulaDesc.references.forEachNameKey(function (nameKey) {
            this._nameMaps.getOrConstruct(nameKey, SimpleMap).insert(formulaKey, formulaDesc);
        }, this);

        // insert the formula descriptor into the formula maps for all table ranges
        formulaDesc.references.forEachTableKey(function (tableKey) {
            this._tableMaps.getOrConstruct(tableKey, SimpleMap).insert(formulaKey, formulaDesc);
        }, this);
    };

    /**
     * Removes a formula descriptor from this dictionary.
     *
     * @param {String} formulaKey
     *  The key of the passed formula descriptor.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The descriptor of a formula to be removed form this dictionary.
     */
    FormulaDictionary.prototype.removeFormula = function (formulaKey, formulaDesc) {

        // remove the formula descriptor from the formula maps for all source ranges
        formulaDesc.references.forEachRange(function (sheetUid, range) {
            this._rangeSets.with(sheetUid, function (rangeSet) {
                if ((range = rangeSet.get(range, null))) {
                    var formulaMap = range.formulaMap;
                    formulaMap.remove(formulaKey);
                    if (formulaMap.empty()) { rangeSet.remove(range); }
                }
            }, this);
        }, this);

        // remove the formula descriptor from the formula maps for all defined names
        formulaDesc.references.forEachNameKey(function (nameKey) {
            this._nameMaps.with(nameKey, function (formulaMap) {
                formulaMap.remove(formulaKey);
                if (formulaMap.empty()) { this._nameMaps.remove(nameKey); }
            }, this);
        }, this);

        // remove the formula descriptor from the formula maps for all table ranges
        formulaDesc.references.forEachTableKey(function (tableKey) {
            this._tableMaps.with(tableKey, function (formulaMap) {
                formulaMap.remove(formulaKey);
                if (formulaMap.empty()) { this._tableMaps.remove(tableKey); }
            }, this);
        }, this);
    };

    /**
     * Returns the descriptors of all formulas that depend directly on the
     * specified cell address.
     *
     * @param {String} sheetUid
     *  The UID of the sheet containing the cell with the passed address.
     *
     * @param {Address} address
     *  The address of the cell to look up formula descriptors for.
     *
     * @returns {SimpleMap<FormulaDescriptor>}
     *  All formula descriptors depending on the specified cell address, mapped
     *  by formula key.
     */
    FormulaDictionary.prototype.findFormulasForAddress = function (sheetUid, address) {

        // the collected formula descriptors
        var formulaMap = new SimpleMap();

        // collect formula descriptors depending on all cell ranges containing the address
        this._rangeSets.with(sheetUid, function (rangeSet) {
            rangeSet.findByAddress(address).forEach(function (range) {
                formulaMap.merge(range.formulaMap);
            });
        }, this);

        return formulaMap;
    };

    /**
     * Returns the descriptors of all formulas that depend directly on any cell
     * in the specified cell range address.
     *
     * @param {String} sheetUid
     *  The UID of the sheet containing the cell range with the passed address.
     *
     * @param {Range} range
     *  The address of the cell range to look up formula descriptors for.
     *
     * @returns {SimpleMap<FormulaDescriptor>}
     *  All formula descriptors depending on the specified cell range address,
     *  mapped by formula key.
     */
    FormulaDictionary.prototype.findFormulasForRange = function (sheetUid, range) {

        // the collected formula descriptors
        var formulaMap = new SimpleMap();

        // collect formula descriptors depending on all cell ranges overlapping with the range
        this._rangeSets.with(sheetUid, function (rangeSet) {
            rangeSet.findRanges(range).forEach(function (range) {
                formulaMap.merge(range.formulaMap);
            });
        }, this);

        return formulaMap;
    };

    /**
     * Returns the descriptors of all formulas that depend directly on the
     * formula represented by the specified formula descriptor.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The descriptor of a formula to look up formula descriptors for.
     *
     * @returns {SimpleMap<FormulaDescriptor>|Null}
     *  All formula descriptors depending on the specified formula, mapped by
     *  formula key; or null, if no other formula depends on the formula.
     */
    FormulaDictionary.prototype.findFormulasForFormula = function (formulaDesc) {

        // formulas can depend on cell formulas only (including shared formulas and matrix formulas)
        if (!formulaDesc.cellAddress) { return null; }

        // find all formulas depending on any cell covered by a matrix formula
        if (formulaDesc.matrixRange) {
            return this.findFormulasForRange(formulaDesc.sheetUid, formulaDesc.matrixRange);
        }

        // find all formulas depending on the cell represented by the passed formula descriptor
        return this.findFormulasForAddress(formulaDesc.sheetUid, formulaDesc.cellAddress);
    };

    /**
     * Returns the descriptors of all formulas that depend directly on the
     * defined name with the specified key.
     *
     * @param {String} nameKey
     *  The key of the defined name to look up formula descriptors for.
     *
     * @returns {SimpleMap<FormulaDescriptor>|Null}
     *  All formula descriptors depending on the specified defined name, mapped
     *  by formula key; or null, if no other formula depends on the defined
     *  name.
     */
    FormulaDictionary.prototype.findFormulasForName = function (nameKey) {
        return this._nameMaps.get(nameKey, null);
    };

    /**
     * Returns the descriptors of all formulas that depend directly on the
     * table range with the specified key.
     *
     * @param {String} tableKey
     *  The key of the table range to look up formula descriptors for.
     *
     * @returns {SimpleMap<FormulaDescriptor>|Null}
     *  All formula descriptors depending on the specified table range, mapped
     *  by formula key; or null, if no other formula depends on the table
     *  range.
     */
    FormulaDictionary.prototype.findFormulasForTable = function (tableKey) {
        return this._tableMaps.get(tableKey, null);
    };

    // exports ================================================================

    return FormulaDictionary;

});
