/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/dependencymanager', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/utils/deferredutils',
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/celloperationsbuilder',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/deps/dependencyutils',
    'io.ox/office/spreadsheet/model/formula/deps/timerhelper',
    'io.ox/office/spreadsheet/model/formula/deps/formuladescriptor',
    'io.ox/office/spreadsheet/model/formula/deps/sheetdescriptor',
    'io.ox/office/spreadsheet/model/formula/deps/formuladictionary',
    'io.ox/office/spreadsheet/model/formula/deps/pendingcollection'
], function (Utils, IteratorUtils, DeferredUtils, SimpleMap, TriggerObject, TimerMixin, Config, SheetUtils, CellOperationsBuilder, FormulaUtils, DependencyUtils, TimerHelper, FormulaDescriptor, SheetDescriptor, FormulaDictionary, PendingCollection) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var AddressArray = SheetUtils.AddressArray;
    var RangeArray = SheetUtils.RangeArray;
    var Scalar = FormulaUtils.Scalar;
    var getCellKey = DependencyUtils.getCellKey;

    // class ResultCollection =================================================

    /**
     * Encapsulates all data that will be collected during a single update
     * cycle of a dependency manager.
     *
     * @constructor
     *
     * @property {SimpleMap<FormulaDescriptor>} formulaMap
     *  All dirty formula descriptors collected by an update cycle of a
     *  dependency manager. This map contains all formula descriptors contained
     *  in the properties 'cellFormulaMap' and 'ruleFormulaMaps'.
     *
     * @property {SimpleMap<FormulaDescriptor>} cellFormulaMap
     *  The descriptors of all dirty cell formulas collected by an update cycle
     *  of a dependency manager. All formulas in this map will be calculated
     *  while the calculation of another formula looks up the cell contents of
     *  that dirty formulas. After calculation of a formula has finished, the
     *  formula result will be stored in the property 'resultMaps', and will be
     *  returned when looking up the cell value the next time.
     *
     * @property {SimpleMap<SimpleMap<FormulaDescriptor>>} ruleFormulaMaps
     *  The descriptors of all dirty formulas of formatting rules collected by
     *  an update cycle of a dependency manager, mapped as submaps by sheet
     *  UID. The target ranges of all rule models will be repainted after the
     *  update cycle has been completed.
     *
     * @property {SimpleMap<SimpleMap<Object>>} resultMaps
     *  The results of all dirty formulas (normal formulas, shared formulas,
     *  and matrix formulas) that have been calculated, mapped by cell address
     *  key without sheet UID. Each object in the map contains the cell address
     *  (property 'address'), the scalar result value (property 'value'), and a
     *  timestamp (property 'time') used to detect timeouts during formula
     *  calculation.
     *
     * @property {Array<Array<FormulaDescriptor>>} referenceCycles
     *  All formula cycles found during calculation of formulas. Each element
     *  in the array is another array with formula descriptors representing the
     *  reference cycle.
     *
     * @property {Boolean} hasErrors
     *  Whether any of the formulas could not be calculated correctly, e.g. due
     *  to unimplemented functions.
     */
    function ResultCollection() {

        this.formulaMap = new SimpleMap();
        this.cellFormulaMap = new SimpleMap();
        this.ruleFormulaMaps = new SimpleMap();
        this.resultMaps = new SimpleMap();
        this.referenceCycles = [];
        this.hasErrors = false;

    } // class ResultCollection

    // public methods ---------------------------------------------------------

    /**
     * Inserts a new formula descriptor into the internal collections of this
     * instance.
     *
     * @param {String} formulaKey
     *  The unique key of the formula descriptor.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The descriptor for the new formula to be registered.
     *
     * @returns {ResultCollection}
     *  A reference to this instance.
     */
    ResultCollection.prototype.insertFormula = function (formulaKey, formulaDesc) {
        this.formulaMap.insert(formulaKey, formulaDesc);
        if (formulaDesc.cellAddress) {
            this.cellFormulaMap.insert(formulaKey, formulaDesc);
        } else if (formulaDesc.ruleModel) {
            this.ruleFormulaMaps.getOrConstruct(formulaDesc.sheetUid, SimpleMap).insert(formulaKey, formulaDesc);
        }
        return this;
    };

    // class DependencyManager ================================================

    /**
     * Collects all change events in the document, and recalculates all formula
     * expressions depending on these changes.
     *
     * Triggers the following events:
     * - 'recalc:start'
     *      When the background task for calculating the dependent formulas has
     *      started.
     * - 'recalc:end'
     *      When the background task for calculating the dependent formulas has
     *      been finished successfully. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event}
     *          The jQuery event object.
     *      (2) {Array<AddressArray>} referenceCycles
     *          An array of arrays of cell addresses with sheet UIDs. Each of
     *          the inner arrays represents a reference cycle found during
     *          calculation of dirty formulas.
     * - 'recalc:cancel'
     *      When the background task for calculating the dependent formulas has
     *      been aborted due to new document changes.
     * - 'recalc:overflow'
     *      Will be triggered once, if the total number of formulas in the
     *      document exceeds a specific limit in the server configuration.
     *      After that, the dependency manager will not process any formulas in
     *      the document anymore.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    var DependencyManager = TriggerObject.extend({ constructor: function (docModel) {

        // private properties -------------------------------------------------

        // self reference
        var self = this;

        // the application containing this dependency manager
        var app = docModel.getApp();

        // A helper instance to execute background timer loops with different
        // priorities (interval durations).
        //
        var timerHelper = new TimerHelper(app);

        // Stores the descriptors of all registered formulas in the document.
        //
        // The formula descriptors (instances of the class FormulaDescriptor)
        // will be mapped by their formula keys.
        //
        var formulaDescMap = new SimpleMap();

        // The total number of all registered formulas currently contained in
        // the spreadsheet document.
        //
        var formulaCount = 0;

        // Stores a sheet descriptor object per existing sheet in the document.
        //
        // The sheet descriptors (instances of the class SheetDescriptor) will
        // be mapped by the UID of the sheet model.
        //
        var sheetDescMap = new SimpleMap();

        // A dictionary for all formula descriptors of this dependency manager.
        //
        // This is the central storage of this dependency manager that will be
        // used to find formulas depending on specific dirty cells, or other
        // changed document contents, in a very efficient way.
        //
        var formulaDictionary = new FormulaDictionary();

        // Collected data for all document change events, and other information
        // needed to decide which formulas need to be recalculated in the next
        // update cycle.
        //
        var pendingCollection = new PendingCollection();

        // All dirty formula descriptors collected by an update cycle of this
        // dependency manager, and the formula results already calculated.
        //
        var resultCollection = null;

        // Whether this dependency manager did not finish its initial update
        // cycle after importing the document yet.
        //
        // Will be set to false, after the first update cycle has finished
        // successfully.
        //
        var initialUpdateCycle = true;

        // Specifies whether the initial update cycle needs to recalculate all
        // formulas in the document.
        //
        // This flag will be initialized after importing the document, and will
        // be reset after the first successful update cycle.
        //
        var calcOnLoad = false;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Creates a new operations generator, and invokes the passed callback
         * function. If the spreadsheet document is in edit mode, the generated
         * operations will be sent to the server.
         *
         * @param {Function} callback
         *  The callback function. Receives the operations generator as first
         *  parameter.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         */
        function generateAndSendOperations(callback, context) {

            // create a new generator that applies the generated operations immediately
            var generator = docModel.createOperationsGenerator({ applyImmediately: true });

            // invoke the callback function with the new generator
            callback.call(context, generator);

            // send the operations to the server, but only if the document is in edit mode
            if (docModel.getEditMode()) {
                docModel.sendOperations(generator, { defer: !app.isLocallyModified() });
            }
        }

        /**
         * Sends the 'calcOnLoad' document flag to the server, if it has not
         * been set yet.
         */
        function setCalcOnLoadFlag() {
            if (!calcOnLoad) {
                generateAndSendOperations(function (generator) {
                    generator.generateDocAttrsOperation({ calcOnLoad: true });
                });
                calcOnLoad = true;
            }
        }

        /**
         * Inserts a new formula descriptor into all collections of this
         * dependency manager.
         *
         * @param {String} formulaKey
         *  The unique key of the formula descriptor.
         *
         * @param {FormulaDescriptor} formulaDesc
         *  The descriptor for the new formula to be registered.
         *
         * @returns {Boolean}
         *  Whether the formula descriptor has been inserted into the internal
         *  formula collections. If the return value is false, this method has
         *  triggered a 'recalc:overflow' event, and has stopped the entire
         *  recalculation cycle.
         */
        function insertFormula(formulaKey, formulaDesc) {

            // update the total number of formulas
            formulaCount += 1;

            // If the maximum number of formulas has been exceeded, detach this dependency manager
            // from all document model events (this prevents processing any further edit actions),
            // trigger the 'recalc:overflow' event (this will abort the update cycle currently
            // running), and set the 'calcOnLoad' document flag.
            if (formulaCount > Config.MAX_FORMULA_COUNT) {
                self.stopListeningTo(docModel);
                pendingCollection = new PendingCollection();
                self.trigger('recalc:overflow');
                setCalcOnLoadFlag();
                return false;
            }

            // register the formula descriptor in the global map
            formulaDescMap.insert(formulaKey, formulaDesc);

            // insert the formula descriptor into the sheet descriptor maps
            sheetDescMap.get(formulaDesc.sheetUid).insertFormula(formulaKey, formulaDesc);

            // insert the formula descriptor into the dictionary
            formulaDictionary.insertFormula(formulaKey, formulaDesc);

            return true;
        }

        /**
         * Removes a formula descriptor from all collections of this dependency
         * manager.
         *
         * @param {String} formulaKey
         *  The unique key of the formula descriptor.
         *
         * @returns {FormulaDescriptor|Null}
         *  The formula descriptor that has been removed from the collections;
         *  or null, if no formula descriptor was registered for the passed
         *  formula key.
         */
        function removeFormula(formulaKey) {

            // remove the formula descriptor from the global map
            var formulaDesc = formulaDescMap.remove(formulaKey);
            if (!formulaDesc) { return null; }

            // update the total number of formulas in this dependency manager, if the formula exists
            formulaCount -= 1;

            // remove the formula descriptor from the sheet descriptor maps (check existence,
            // sheet descriptor has been removed already while deleting a sheet in the document)
            sheetDescMap.with(formulaDesc.sheetUid, function (sheetDesc) {
                sheetDesc.removeFormula(formulaKey);
            });

            // remove the formula descriptor from the dictionary
            formulaDictionary.removeFormula(formulaKey, formulaDesc);
            return formulaDesc;
        }

        /**
         * Refreshes the dependency settings of a formula descriptor that is
         * registered for this dependency manager.
         *
         * @param {String} formulaKey
         *  The unique key of the formula descriptor.
         *
         * @returns {FormulaDescriptor|Null}
         *  The formula descriptor that has been refreshed; or null, if no
         *  formula descriptor was registered for the passed formula key.
         */
        function refreshFormula(formulaKey) {

            // find and remove an existing formula descriptor
            var formulaDesc = removeFormula(formulaKey);
            if (!formulaDesc) { return null; }

            // refresh the source dependencies, and reinsert the descriptor
            formulaDesc.refreshReferences();
            insertFormula(formulaKey, formulaDesc);
            return formulaDesc;
        }

        /**
         * Registers a new or changed formatting rule.
         *
         * @param {RuleModel} ruleModel
         *  The model instance of the new or changed formatting rule.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.update=false]
         *      If set to true, this method will update an existing formula
         *      descriptor, otherwise it will insert a new formula descriptor.
         */
        var registerRuleModel = DependencyUtils.profileMethod('registerRuleModel()', function (ruleModel, options) {

            // remove old settings when updating an existing formula descriptor
            if (Utils.getBooleanOption(options, 'update', false)) {
                unregisterRuleModel(ruleModel);
            }

            // register the rule model for the next update cycle
            pendingCollection.registerRuleModel(ruleModel);
        });

        /**
         * Unregisters a deleted formatting rule.
         *
         * @param {RuleModel} ruleModel
         *  The model instance of the formatting rule about to be deleted.
         */
        var unregisterRuleModel = DependencyUtils.profileMethod('unregisterRuleModel()', function (ruleModel) {

            // remove the rule model from the pending settings
            pendingCollection.unregisterRuleModel(ruleModel);

            // remove all formula descriptors registered for the rule model
            var sheetDesc = sheetDescMap.get(ruleModel.getSheetModel().getUid());
            sheetDesc.ruleFormulaMaps.with(ruleModel.getUid(), function (formulaMap) {
                formulaMap.forEachKey(removeFormula);
            });
        });

        /**
         * Updates the settings of this dependency manager for a new sheet in
         * the spreadsheet document.
         *
         * @param {SheetModel} sheetModel
         *  The model instance of the new sheet.
         */
        var registerSheetModel = DependencyUtils.profileMethod('registerSheetModel()', function (sheetModel) {
            DependencyUtils.withLogging(function () {
                DependencyUtils.log('sheet="' + sheetModel.getName() + '" key=' + sheetModel.getUid());
            });

            // create the sheet descriptor object for the new sheet
            var sheetDesc = new SheetDescriptor(sheetModel);
            sheetDescMap.insert(sheetModel.getUid(), sheetDesc);

            // initialize the pending settings of the sheet for the next update cycle
            pendingCollection.registerSheetModel(sheetModel);
        });

        /**
         * Updates the settings of this dependency manager before a sheet in
         * the spreadsheet document will be deleted.
         *
         * @param {SheetModel} sheetModel
         *  The model instance of the deleted sheet.
         */
        var unregisterSheetModel = DependencyUtils.profileMethod('unregisterSheetModel()', function (sheetModel) {
            DependencyUtils.withLogging(function () {
                DependencyUtils.log('sheet="' + sheetModel.getName() + '" key=' + sheetModel.getUid());
            });

            // unregister the pending settings of the sheet
            pendingCollection.unregisterSheetModel(sheetModel);

            // remove the sheet descriptor object for the deleted sheet (sheet descriptor may not
            // exist yet, e.g. when inserting/deleting a sheet very quickly in unit tests)
            var sheetDesc = sheetDescMap.remove(sheetModel.getUid());
            if (!sheetDesc) { return; }

            // remove all formulas of the deleted sheet from the internal structures
            sheetDesc.formulaMap.forEach(function (formulaDesc, formulaKey) {
                removeFormula(formulaKey, formulaDesc);
            });
        });

        /**
         * Processes all document change events that have been collected in the
         * pending collection.
         *
         * @returns {jQuery.Promise}
         *  An abortable promise that will be resolved after all collected
         *  document changes have been processed.
         */
        var processPendingCollection = DependencyUtils.profileAsyncMethod('processPendingCollection()', function () {

            // process the dirty addresses and moved ranges of all sheets
            return timerHelper.iterate(pendingCollection.changedSheetMap.iterator(), function (pendingSheetData) {

                // the model, UID, and cell collection of the affected sheet
                var sheetModel = pendingSheetData.sheetModel;
                var sheetUid = sheetModel.getUid();
                var cellCollection = sheetModel.getCellCollection();
                // the addresses of the dirty (moved) cell ranges
                var dirtyRanges = pendingSheetData.dirtyRanges = pendingSheetData.dirtyRanges.merge();
                // the addresses of the changed formula cells
                var dirtyCells = pendingSheetData.dirtyFormulaCells;
                // used to prevent double registration from 'dirtyRanges' and 'dirtyCells'
                var registeredCellSet = {};

                // removes the specified cell formulas from the internal collections
                function unregisterFormulaCells(addresses) {
                    return timerHelper.iterate(addresses.iterator(), function (address) {
                        removeFormula(getCellKey(sheetUid, address));
                    });
                }

                // inserts the specified cell formulas into the internal collections
                function registerFormulaCells(iterator, recalc) {
                    return timerHelper.iterate(iterator, function (address) {

                        // the cell key will be used as formula descriptor key (prevent double registration)
                        var formulaKey = getCellKey(sheetUid, address);
                        if (formulaKey in registeredCellSet) { return; }
                        registeredCellSet[formulaKey] = true;

                        // create a new descriptor for the token array (nothing more to do, if there is no
                        // token array available for the key, e.g. deleted cell formulas)
                        var tokenDesc = cellCollection.getTokenArray(address);
                        if (!tokenDesc) { return; }
                        var formulaDesc = new FormulaDescriptor(docModel, sheetUid, tokenDesc.tokenArray, tokenDesc.refAddress, address, tokenDesc.matrixRange);

                        // formula cells without a valid result value must be recalculated
                        recalc = recalc || (cellCollection.getValue(address) === null);

                        // ignore the formula if it does not contain any dependencies
                        if (recalc || !formulaDesc.isFixed()) {
                            // immediately escape the loop, if the number of formulas exceeds the configuration limit
                            if (!insertFormula(formulaKey, formulaDesc)) { return Utils.BREAK; }
                        }

                        // register the new formula for recalculation
                        if (recalc) { pendingSheetData.recalcFormulaCells.push(address); }
                    });
                }

                // remove formula descriptors of all dirty formula cells
                var promise = unregisterFormulaCells(dirtyCells);

                // remove all formula descriptors inside the dirty ranges
                promise = promise.then(function () {
                    var sheetDesc = sheetDescMap.get(sheetUid, null);
                    return timerHelper.iterate(dirtyRanges.iterator(), function (dirtyRange) {
                        return unregisterFormulaCells(sheetDesc.cellFormulaSet.findAddresses(dirtyRange));
                    });
                });

                // register the changed formula cells (before dirty ranges; always recalculate all changed formula cells)
                promise = promise.then(function () {
                    return registerFormulaCells(dirtyCells.iterator(), true).done(function () { dirtyCells.clear(); });
                });

                // collect the formula cells in the moved ranges at their current positions
                promise = promise.then(function () {
                    var iterator = cellCollection.createAddressIterator(dirtyRanges, { type: 'formula', covered: true });
                    var recalc = initialUpdateCycle && calcOnLoad;
                    return registerFormulaCells(iterator, recalc).done(function () { dirtyRanges.clear(); });
                });

                // register the formulas of all formatting rules
                if (!pendingSheetData.dirtyRulesMap.empty()) {
                    promise = promise.then(function () {
                        return timerHelper.iterate(pendingSheetData.dirtyRulesMap.iterator(), function (ruleModel) {
                            var ruleUid = ruleModel.getUid();
                            var refAddress = ruleModel.getRefAddress();
                            IteratorUtils.forEach(ruleModel.createTokenArrayIterator(), function (tokenArray, result) {
                                var formulaDesc = new FormulaDescriptor(docModel, sheetUid, tokenArray, refAddress, null, null, ruleModel);
                                if (!formulaDesc.isFixed()) { insertFormula(ruleUid + '!' + result.key, formulaDesc); }
                            });
                        });
                    });
                }

                return promise;
            });
        });

        /**
         * Collects the descriptors of all dirty formulas depending directly on
         * changed cells, or indirectly on other dirty formulas, in the class
         * member variable 'resultCollection', recalculates all collected dirty
         * formulas, and writes the calculated results into the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all dirty formulas have been
         *  collected and recalculated, and the cells in the document have been
         *  updated.
         */
        var recalcDirtyFormulas = DependencyUtils.profileAsyncMethod('recalcDirtyFormulas()', function () {

            // realculate-all mode: calculate and collect the results of all cell formulas in the document
            if (pendingCollection.recalcAll) {
                return timerHelper.iterate(docModel.createSheetIterator(), function (sheetModel) {
                    var iterator = sheetModel.getCellCollection().createAddressIterator(docModel.getSheetRange(), { type: 'formula', covered: true });
                    return timerHelper.iterate(iterator, function (address) {
                        self.getCellValue(sheetModel, address);
                    });
                });
            }

            // copy of the formula dictionary that will be reduced during lookup to improve performance
            var lookupDictionary = formulaDictionary.clone();
            // staring point of the abortable promise chain
            var promise = self.createResolvedPromise(null);

            // inserts a new formula descriptor into the dirty formula map, and removes it from the lookup dictionary
            function registerDirtyFormula(formulaKey, formulaDesc) {
                lookupDictionary.removeFormula(formulaKey, formulaDesc);
                resultCollection.insertFormula(formulaKey, formulaDesc);
            }

            // inserts a map of formula descriptors into the dirty formula map, and removes them from the lookup dictionary
            function registerDirtyFormulas(formulaMap) {
                if (formulaMap) { formulaMap.forEachKey(registerDirtyFormula); }
            }

            // Collect all formulas depending on defined names with changed formula expression.
            if (!pendingCollection.changedNamesMap.empty()) {
                promise = promise.then(DependencyUtils.profileAsyncMethod('collecting formulas for dirty names...', function () {
                    return timerHelper.iterate(pendingCollection.changedNamesMap.keyIterator(), function (nameKey) {
                        registerDirtyFormulas(lookupDictionary.findFormulasForName(nameKey));
                    });
                }));
            }

            // Collect all formulas depending on table ranges with changed target range.
            if (!pendingCollection.changedTablesMap.empty()) {
                promise = promise.then(DependencyUtils.profileAsyncMethod('collecting formulas for dirty table ranges...', function () {
                    return timerHelper.iterate(pendingCollection.changedTablesMap.keyIterator(), function (tableKey) {
                        registerDirtyFormulas(lookupDictionary.findFormulasForTable(tableKey));
                    });
                }));
            }

            // Refresh the dependencies of all found formulas for dirty names and tables. The dependencies will include the new
            // source references of the changed defined names, and the target ranges of the changed tables.
            promise = promise.then(function () {
                if (!resultCollection.formulaMap.empty()) { // bug 48014: check MUST be inside promise.then()!
                    return DependencyUtils.takeAsyncTime('refreshing formulas for dirty names and tables...', function () {
                        return timerHelper.iterate(resultCollection.formulaMap.keyIterator(), refreshFormula);
                    });
                }
            });

            // Collect all formulas with a #NAME? error, after a defined name changed its label, and refresh
            // their dependencies too, in order to resolve #NAME? errors to valid name references.
            if (!pendingCollection.changedLabelsMap.empty()) {
                promise = promise.then(DependencyUtils.profileAsyncMethod('refreshing formulas for dirty name labels...', function () {
                    // visit all sheets (each sheet descriptor contains its own map of formulas with missing names)...
                    return timerHelper.iterate(sheetDescMap.iterator(), function (sheetDesc) {
                        // process all defined names with changed label for the current sheet...
                        return timerHelper.iterate(pendingCollection.changedLabelsMap.iterator(), function (nameModel) {
                            // if the sheet contains formulas with #NAME? errors using the current label...
                            return sheetDesc.missingNamesMap.with(nameModel.getKey(), function (destFormulaMap) {
                                // visit all these formulas with #NAME? errors...
                                return timerHelper.iterate(destFormulaMap.iterator(), function (formulaDesc, result) {
                                    // the formula may have been refreshed already (multiple unresolved names)
                                    var formulaKey = result.key;
                                    if (!resultCollection.formulaMap.has(formulaKey)) {
                                        refreshFormula(formulaKey);
                                        if (formulaDesc.references.containsName(nameModel)) {
                                            registerDirtyFormula(formulaKey, formulaDesc);
                                        }
                                    }
                                });
                            });
                        });
                    });
                }));
            }

            // Collect all initially dirty formulas from all sheets in the document.
            promise = promise.then(DependencyUtils.profileAsyncMethod('collecting initially dirty formulas...', function () {
                return timerHelper.iterate(sheetDescMap.iterator(), function (sheetDesc) {

                    // Add all formulas that are permanently dirty (recalculation mode 'always'). These formulas will
                    // always be recalculated, regardless whether they depend on any dirty cells.
                    if (initialUpdateCycle || pendingCollection.recalcVolatile) {
                        registerDirtyFormulas(sheetDesc.recalcAlwaysMap);
                    }

                    // Add all formulas that are initially dirty (recalculation mode 'once'). These formulas will be
                    // recalculated once after importing the document.
                    if (initialUpdateCycle) {
                        registerDirtyFormulas(sheetDesc.recalcOnceMap);
                    }

                    // the model of the affected sheet
                    var sheetModel = sheetDesc.sheetModel;
                    // the sheet UID, used as map key
                    var sheetUid = sheetModel.getUid();
                    // collected cell addresses in the sheet from document change actions
                    var pendingSheetData = pendingCollection.getPendingSheetData(sheetModel);

                    // the addresses of all formula cells that will be recalculated
                    var recalcFormulaCells = pendingSheetData ? pendingSheetData.recalcFormulaCells.clone() : new AddressArray();

                    // Collect the dirty value cells. Dirty value cells that are formulas by themselves need
                    // to be recalculated (the correct formula result may have been destroyed).
                    var dirtyValueCells = pendingSheetData ? pendingSheetData.dirtyValueCells.reject(function (address) {
                        if (formulaDescMap.has(getCellKey(sheetUid, address))) {
                            recalcFormulaCells.push(address);
                            return true;
                        }
                    }) : new AddressArray();

                    // nothing more to do without changed formula nor value cells
                    if (recalcFormulaCells.empty() && dirtyValueCells.empty()) { return; }

                    // remove duplicates from the formula cell addresses
                    recalcFormulaCells = recalcFormulaCells.unify();

                    // join the value cells to ranges for faster lookup of dependent formulas
                    var dirtyValueRanges = RangeArray.mergeAddresses(dirtyValueCells);

                    DependencyUtils.withLogging(function () {
                        var sheetName = sheetModel.getName();
                        if (!recalcFormulaCells.empty()) {
                            DependencyUtils.log('outdated formula results in sheet ' + sheetName + ': ' + RangeArray.mergeAddresses(recalcFormulaCells));
                        }
                        if (!dirtyValueRanges.empty()) {
                            DependencyUtils.log('dirty cells in sheet ' + sheetName + ': ' + dirtyValueRanges);
                        }
                    });

                    // staring point of the abortable promise chain
                    var promise2 = self.createResolvedPromise(null);

                    // Collect all dirty cell formulas. Cells with changed formula expressions must be recalculated in
                    // order to get a valid initial result.
                    if (!recalcFormulaCells.empty()) {
                        promise2 = promise2.then(function () {
                            return timerHelper.iterate(recalcFormulaCells.iterator(), function (address) {
                                var formulaKey = getCellKey(sheetUid, address);
                                sheetDesc.formulaMap.with(formulaKey, function (formulaDesc) {
                                    registerDirtyFormula(formulaKey, formulaDesc);
                                });
                            });
                        });
                    }

                    // Collect all formulas depending directly on dirty value cells.
                    if (!dirtyValueRanges.empty()) {
                        promise2 = promise2.then(function () {
                            return timerHelper.iterate(dirtyValueRanges.iterator(), function (range) {
                                registerDirtyFormulas(lookupDictionary.findFormulasForRange(sheetUid, range));
                            });
                        });
                    }

                    return promise2;
                }).done(function () {
                    DependencyUtils.logFormulaMap(resultCollection.formulaMap);
                });
            }));

            // Extend the map of dirty formulas with other formulas depending on them.
            promise = promise.then(function () {

                // descriptors of all cell formulas to collect dependent formulas for
                var pendingFormulaMap = resultCollection.cellFormulaMap.clone();

                // repeat as long as new formulas can be found depending on the formulas in 'pendingFormulaMap'
                return timerHelper.loop(function (index) {

                    // exit the (endless) loop, if the last iteration step did not find a new formula
                    if (pendingFormulaMap.empty()) { return Utils.BREAK; }

                    return DependencyUtils.takeAsyncTime('collecting dependent formulas for chain depth: ' + (index + 1) + '...', function () {

                        // all new formula descriptors not yet contained in the dirty formula map
                        var newFormulaMap = new SimpleMap();

                        // process all formulas whose dependencies have not been calculated yet
                        var promise2 = timerHelper.iterate(pendingFormulaMap.iterator(), function (sourceFormulaDesc) {

                            // find all formulas that depend directly on the current formula (ignore formulas of formatting rules)
                            var destFormulaMap = lookupDictionary.findFormulasForFormula(sourceFormulaDesc);
                            if (!destFormulaMap) { return; }

                            // process all found formulas depending on the current formula
                            return timerHelper.iterate(destFormulaMap.iterator(), function (destFormulaDesc, destResult) {

                                // nothing more to do, if the formula has already been collected
                                var destFormulaKey = destResult.key;
                                if (resultCollection.formulaMap.has(destFormulaKey)) { return; }

                                // insert the formula descriptor into the maps
                                registerDirtyFormula(destFormulaKey, destFormulaDesc);
                                newFormulaMap.insert(destFormulaKey, destFormulaDesc);
                            });
                        });

                        // initialize the next iteration cycle
                        return promise2.done(function () {
                            DependencyUtils.logFormulaMap(newFormulaMap);
                            pendingFormulaMap = newFormulaMap;
                        });
                    });
                });
            });

            // calculate and collect all formula results, as long as dirty formulas are contained in the map
            promise = promise.then(DependencyUtils.profileAsyncMethod('recalculating all dirty formulas...', function () {
                return timerHelper.loop(function () {

                    // get a formula descriptor from the dirty formula map
                    var formulaDesc = resultCollection.cellFormulaMap.getAny(null);
                    if (!formulaDesc) { return Utils.BREAK; }

                    // calculate and store the result of the formula (this may recursively calculate other dirty
                    // formulas as well, which will be removed from the dirty formula map)
                    var sheetModel = sheetDescMap.get(formulaDesc.sheetUid).sheetModel;
                    self.getCellValue(sheetModel, formulaDesc.cellAddress);
                });
            }));

            return promise;
        });

        /**
         * Writes the collected formula results into the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the cells in the document have
         *  been updated.
         */
        var sendFormulaResults = DependencyUtils.profileAsyncMethod('sendFormulaResults()', function () {

            // wait for another operations generator process currently running (needed to prevent internal
            // application error due to nested operation generators), this may even abort the entire update
            // cycle, e.g. after deleting a sheet which makes the calculated formula results useless
            var promise = docModel.waitForActionsProcessed();

            // write the collected results (synchronously) into the document
            promise.always(DependencyUtils.profileMethod('updating document contents...', function () {

                // generate the cell operations, and send them to the server
                generateAndSendOperations(function (generator) {

                    // update the 'calcOnLoad' flag in the original spreadsheet document
                    var newCalcOnLoad = resultCollection.hasErrors ? true : initialUpdateCycle ? false : calcOnLoad;
                    if (newCalcOnLoad !== calcOnLoad) {
                        generator.generateDocAttrsOperation({ calcOnLoad: newCalcOnLoad });
                        calcOnLoad = newCalcOnLoad;
                    }

                    // create a 'changeCell' operation per sheet
                    resultCollection.resultMaps.forEach(function (resultMap, sheetUid) {

                        // generate cell operations for the remaining results
                        var sheetModel = sheetDescMap.get(sheetUid).sheetModel;
                        generator.setSheetIndex(sheetModel.getIndex());

                        // the cell operation builder
                        var operationsBuilder = new CellOperationsBuilder(sheetModel, generator, { skipShared: true });

                        // sort the formula results by address (needed by cell operation builder)
                        var resultDescs = resultMap.values().sort(function (desc1, desc2) {
                            return Address.compare(desc1.address, desc2.address);
                        });

                        // add the new result value to the 'changeCells' operation
                        resultDescs.forEach(function (resultDesc) {
                            operationsBuilder.createCells(resultDesc.address, { v: resultDesc.value }, 1);
                        });

                        // generate the cell operations
                        operationsBuilder.finalizeOperations({ createUndo: false, filter: true });
                    });
                });
            }));

            // refresh all formatting rules containing dirty formulas locally (without operations)
            promise.always(DependencyUtils.profileMethod('updating conditional formatting rules...', function () {

                var dirtyRuleMaps = new SimpleMap();

                function registerRuleModels(sheetUid, formulaMap) {
                    formulaMap.forEach(function (formulaDesc) {
                        var ruleModel = formulaDesc.ruleModel;
                        dirtyRuleMaps.getOrConstruct(sheetUid, SimpleMap).insert(ruleModel.getUid(), ruleModel);
                        formulaDesc.tokenArray.clearResultCache();
                    });
                }

                if (pendingCollection.recalcAll) {
                    sheetDescMap.forEach(function (sheetDesc, sheetUid) {
                        sheetDesc.ruleFormulaMaps.forEach(function (formulaMap) {
                            registerRuleModels(sheetUid, formulaMap);
                        });
                    });
                } else {
                    resultCollection.ruleFormulaMaps.forEachKey(registerRuleModels);
                }

                dirtyRuleMaps.forEach(function (ruleMap, sheetUid) {
                    var ruleRanges = ruleMap.reduce(new RangeArray(), function (ranges, ruleModel) {
                        return ranges.append(ruleModel.getTargetRanges());
                    });
                    sheetDescMap.get(sheetUid).sheetModel.refreshRanges(ruleRanges);
                });
            }));

            return promise;
        });

        /**
         * Starts the background task that processes all collected and pending
         * document events, refreshes the dependency chains, recalculates all
         * dirty formulas, and updates the model contents (cells, conditional
         * formattings, etc.).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.continue=false]
         *      If set to true, an update cycle currently running will be
         *      continued. By default, a running update cycle will be aborted,
         *      and a fresh update cycle will be started.
         *  @param {Boolean} [options.priority=false]
         *      If set to true, the update cycle will be started with high
         *      timer priority. If used together with the option 'continue',
         *      the continued update cycle will be swiched from low to high
         *      priority.
         *  @param {Boolean} [options.volatile=true]
         *      If set to false (default is true!), volatile formulas will NOT
         *      be recalculated (unless they depend on dirty cells). This
         *      option will affect new update cycles only, but not a running
         *      continued update cycle.
         */
        var startUpdateCycle = (function () {

            // the timer promise representing the update cycle currently running
            var updateTimer = null;
            // first update after import will be delayed further
            var firstUpdate = true;
            // deferred object for a full cycle (remains pending until any cycle has been completed)
            var fullCycleDef = null;
            // the abortable promise of the deferred to be returned to external callers
            var fullCyclePromise = null;

            // aborts the update cycle currently running
            function abortUpdateCycle() {
                if (updateTimer) {
                    DependencyUtils.log('cancelling update cycle');
                    updateTimer.abort(); // additionally, this sets 'updateTimer' to null
                }
            }

            // immediately abort an update cycle, if a formula count overflow has been detected
            self.on('recalc:overflow', abortUpdateCycle);

            // starts a new background task (debounced)
            var runUpdateCycle = self.createDebouncedMethod(_.noop, Utils.profileAsyncMethod('dependency update cycle', function () {

                // trigger the start event (additional delay for initial update after import)
                var delay = firstUpdate ? DependencyUtils.IMPORT_UPDATE_DELAY : 0;
                updateTimer = self.executeDelayed(function () { self.trigger('recalc:start'); }, { delay: delay, infoString: 'DependencyManager: startUpdateCycle', app: app });
                firstUpdate = false;

                // collect all dirty formulas and the results, used later e.g. in the public method getCellValue()
                resultCollection = new ResultCollection();

                // Process all pending document changes (changed value cells, changed formula
                // expressions, moved cells, etc.), before starting to update the token arrays.
                updateTimer = updateTimer.then(processPendingCollection);

                // Collect and recalculate all dirty token arrays (all token arrays referring to
                // dirty cells, unless all formulas have to be recalculated).
                updateTimer = updateTimer.then(recalcDirtyFormulas);

                // Update the values of the formula cells in the document model.
                updateTimer = updateTimer.then(sendFormulaResults);

                // Notify listeners, and clean up the caches after the update cycle has finished successfully.
                updateTimer.done(function () {

                    // notify the listeners
                    self.trigger('recalc:end', resultCollection.referenceCycles);

                    // clean up all internal data
                    pendingCollection = new PendingCollection();
                    resultCollection = null;
                    initialUpdateCycle = false;

                    // resolve the deferred object that represents a full calculation cycle
                    fullCycleDef.resolve();
                });

                // notify listeners if the update cycle has been aborted; 'fullCycleDef' remains pending
                updateTimer.fail(function () { self.trigger('recalc:cancel'); });

                // returning the promise is required by Logger.profileAsyncMethod() method
                return updateTimer.always(function () { updateTimer = null; });

            }), { delay: DependencyUtils.UPDATE_DEBOUNCE_DELAY, infoString: 'DependencyManager: runUpdateCycle', app: app });

            // the implementation of the method startUpdateCycle() returned from local scope
            return function (options) {

                // abort a running background task (unless the task shall be continued)
                if (!Utils.getBooleanOption(options, 'continue', false)) {
                    abortUpdateCycle();
                }

                // start (or continue) timers with high priority if specified
                if (Utils.getBooleanOption(options, 'priority', false)) {
                    timerHelper.setPriority('high');
                }

                // create the promise that will be returned to external callers from public methods
                // (remains in pending state until an update cycle can be finished successfully)
                if (!fullCycleDef) {
                    fullCycleDef = DeferredUtils.createDeferred(app, 'DependencyManager: startUpdateCycle');
                    fullCyclePromise = self.createAbortablePromise(fullCycleDef);
                    fullCyclePromise.done(function () { timerHelper.setPriority('low'); });
                    fullCyclePromise.always(function () { fullCycleDef = null; });
                }

                // whether to recalculate all volatile formulas that do NOT depend on dirty cells
                pendingCollection.recalcVolatile = Utils.getBooleanOption(options, 'volatile', true);

                // start the new background task (unless an update cycle exists already and will be reused)
                if (!updateTimer) { runUpdateCycle();  }

                return fullCyclePromise;
            };
        }());

        // event handlers -----------------------------------------------------

        /**
         * Handler for 'insert:sheet:after' events sent by the document model.
         * Registers a new sheet and all its formula cells in this dependency
         * manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the new sheet model.
         *
         * @param {SheetModel} sheetModel
         *  The new sheet model.
         */
        function insertSheetHandler(event, sheet, sheetModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: inserted sheet "' + sheetModel.getName() + '"'); });

            // register the new sheet model for lazy update
            registerSheetModel(sheetModel);
            startUpdateCycle();
        }

        /**
         * Handler for 'delete:sheet:before' events sent by the document model.
         * Removes all settings of a sheet that will be deleted from the
         * document.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model to be deleted.
         *
         * @param {SheetModel} sheetModel
         *  The sheet model the will be deleted.
         */
        function deleteSheetHandler(event, sheet, sheetModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: deleted sheet "' + sheetModel.getName() + '"'); });

            // delete the sheet model and related data from all collections
            unregisterSheetModel(sheetModel);
            startUpdateCycle();
        }

        /**
         * Handler for 'move:sheet:before' events sent by the document model.
         */
        function moveSheetHandler(event, from, to, sheetModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: moved sheet "' + sheetModel.getName() + '"'); });

            // simply recalculate all volatile formulas, nothing more to do
            startUpdateCycle();
        }

        /**
         * Handler for 'rename:sheet' events sent by the document model.
         */
        function renameSheetHandler(event, sheet, sheetName) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: changed sheet name to "' + sheetName + '"'); });

            // simply recalculate all volatile formulas, nothing more to do
            startUpdateCycle();
        }

        /**
         * Handler for 'insert:name' events sent by the document model. Inserts
         * all settings for a new defined name into this dependency manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number|Null} sheet
         *  The index of the sheet model containing the defined name, or null
         *  for globally defined names.
         *
         * @param {NameModel} nameModel
         *  The model of the new defined name.
         */
        function insertNameHandler(event, sheet, nameModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: inserted name "' + DependencyUtils.getNameLabel(nameModel) + '"'); });

            // register the model of the new defined name for lazy update
            pendingCollection.registerNameModel(nameModel, true, true);
            startUpdateCycle();
        }

        /**
         * Handler for 'delete:name' events sent by the document model. Removes
         * all settings of a deleted defined name from this dependency manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number|Null} sheet
         *  The index of the sheet model containing the defined name, or null
         *  for globally defined names.
         *
         * @param {NameModel} nameModel
         *  The model of the defined name that will be deleted.
         */
        function deleteNameHandler(event, sheet, nameModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: deleted name "' + DependencyUtils.getNameLabel(nameModel) + '"'); });

            // delete the name model from the collection of pending document actions
            pendingCollection.unregisterNameModel(nameModel);
            startUpdateCycle();
        }

        /**
         * Handler for 'change:name' events sent by the document model. Updates
         * the settings of an existing defined name, whose formula expression
         * has been changed.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number|Null} sheet
         *  The index of the sheet model containing the defined name, or null
         *  for globally defined names.
         *
         * @param {NameModel} nameModel
         *  The model of the defined name with a changed formula expression.
         *
         * @param {Object} changeFlags
         *  A flag set that specifies what has been changed. Must contain the
         *  boolean properties 'formula' and 'label'.
         */
        function changeNameHandler(event, sheet, nameModel, changeFlags) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: changed name "' + DependencyUtils.getNameLabel(nameModel) + '"'); });

            // register the model of the changed defined name for lazy update
            pendingCollection.registerNameModel(nameModel, changeFlags.formula, changeFlags.label);
            startUpdateCycle();
        }

        /**
         * Handler for 'delete:table' events sent by the document model. Removes
         * all settings of a deleted defined name from this dependency manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the table range.
         *
         * @param {TableModel} tableModel
         *  The model of the table range that will be deleted.
         */
        function deleteTableHandler(event, sheet, tableModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: deleted table "' + tableModel.getName() + '"'); });

            // delete the table model from the collection of pending document actions
            pendingCollection.unregisterTableModel(tableModel);
            startUpdateCycle();
        }

        /**
         * Handler for 'change:table' events sent by the document model.
         * Handles a changed target range of the specified table model.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the table range.
         *
         * @param {TableModel} tableModel
         *  The model of a changed table range.
         *
         * @param {String} type
         *  The originating change event type sent by the table model. This
         *  method reacts on the 'change:range' event only.
         */
        function changeTableHandler(event, sheet, tableModel, type) {
            if (type === 'change:range') {
                DependencyUtils.withLogging(function () { DependencyUtils.log('Event: changed range of table "' + tableModel.getName() + '"'); });

                // register the model of the changed table range for lazy update
                pendingCollection.registerTableModel(tableModel);
                startUpdateCycle();
            }
        }

        /**
         * Handler for 'change:cells' events sent by the document model.
         * Registers changed values and formulas in the document.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the changed cells.
         *
         * @param {ChangeDescriptor} changeDesc
         *  The descriptor with the addresses of all changed cells.
         */
        function changeCellsHandler(event, sheet, changeDesc, external, results) {

            // ignore the change events that have been generated due to the own last update cycle
            if (results) { return; }

            var sheetModel = docModel.getSheetModel(sheet);

            DependencyUtils.withLogging(function () {
                if (!changeDesc.valueCells.empty()) {
                    DependencyUtils.log('Event: changed value cells in sheet "' + sheetModel.getName() + '": ' + RangeArray.mergeAddresses(changeDesc.valueCells));
                }
                if (changeDesc.formulaCells) {
                    DependencyUtils.log('Event: changed formula cells in sheet "' + sheetModel.getName() + '": ' + RangeArray.mergeAddresses(changeDesc.formulaCells));
                }
            });

            // register the addresses of all dirty value and formula cells
            if (pendingCollection.registerDirtyCells(sheetModel, changeDesc)) {
                startUpdateCycle();
            }
        }

        /**
         * Handler for 'move:cells' events sent by the document model. Updates
         * this dependency manager after cells have been moved in the document,
         * including inserted or deleted columns/rows.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the moved cells.
         *
         * @param {MoveDescriptor} moveDesc
         *  The descriptor with the positions of all moved cells.
         */
        function moveCellsHandler(event, sheet, moveDesc) {

            var sheetModel = docModel.getSheetModel(sheet);

            DependencyUtils.withLogging(function () {
                DependencyUtils.log('Event: moved cells in sheet "' + sheetModel.getName() + '": ' + moveDesc.dirtyRange);
            });

            // move the addresses of all dirty value and formula cells
            pendingCollection.moveDirtyCells(sheetModel, moveDesc);
            startUpdateCycle();
        }

        /**
         * Handler for 'change:rows' events sent by the document model. Updates
         * this dependency manager after rows have been shown or hidden in the
         * document. All volatile formulas will be recalculated.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the changed rows.
         */
        function changeRowsHandler(event, sheet, interval, attributes, styleId, changeFlags) {
            if (changeFlags.visibility) {
                DependencyUtils.withLogging(function () { DependencyUtils.log('Event: changed row visibility in sheet "' + docModel.getSheetName(sheet) + '"'); });

                // simply recalculate all volatile formulas, nothing more to do
                startUpdateCycle();
            }
        }

        /**
         * Handler for 'insert:rule' events sent by the document model. Inserts
         * all settings for a new formatting rule into this dependency manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the rule.
         *
         * @param {RuleModel} ruleModel
         *  The model of the new formatting rule.
         */
        function insertRuleHandler(event, sheet, ruleModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: inserted rule "' + DependencyUtils.getRuleLabel(ruleModel) + '"'); });

            registerRuleModel(ruleModel);
            startUpdateCycle();
        }

        /**
         * Handler for 'delete:rule' events sent by the document model. Removes
         * all settings of a deleted formatting rule from this dependency
         * manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the rule.
         *
         * @param {RuleModel} ruleModel
         *  The model of the deleted formatting rule.
         */
        function deleteRuleHandler(event, sheet, ruleModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: deleted rule "' + DependencyUtils.getRuleLabel(ruleModel) + '"'); });

            // remove all cached settings of the deleted rule
            unregisterRuleModel(ruleModel);
            startUpdateCycle();
        }

        /**
         * Handler for 'change:rule' events sent by the document model. Updates
         * all settings of a changed formatting rule in this dependency
         * manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the rule.
         *
         * @param {RuleModel} ruleModel
         *  The model of the changed formatting rule.
         */
        function changeRuleHandler(event, sheet, ruleModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: changed rule "' + DependencyUtils.getRuleLabel(ruleModel) + '"'); });

            registerRuleModel(ruleModel, { update: true });
            startUpdateCycle();
        }

        // public methods -----------------------------------------------------

        /**
         * Recalculates all dirty formulas in the document, and refreshes all
         * cells with outdated formula results, and all other document contents
         * that depend on formula results (e.g. charts, or conditional
         * formatting). The background task that calculates the formula results
         * will run with high priority (in difference to background tasks
         * started automatically due to change events of the document processed
         * by this dependency manager).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.all=false]
         *      If set to true, all availavle formulas in the document will be
         *      recalculated, regardless of their diryt state.
         *  @param {Boolean} [options.volatile=true]
         *      If set to false (default is true!), volatile formulas will NOT
         *      be recalculated (unless they depend on dirty cells). This
         *      option will be ignored, if the option 'all' has been set.
         *  @param {Number} [options.timeout]
         *      The maximum time allowed for the recalculation cycle, in
         *      milliseconds. If the time elapses before all formulas have been
         *      calculated, the recalculation cycle will be aborted, and the
         *      'calcOnLoad' document flag will be set.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all dirty formulas have been
         *  recalculated successfully.
         */
        this.recalcFormulas = function (options) {

            // initialize the 'recalcAll' flag in the internal data
            if (Utils.getBooleanOption(options, 'all', false)) {
                pendingCollection.recalcAll = true;
            }

            // start the update cycle, or continue a running update cycle
            var promise = startUpdateCycle({
                continue: !pendingCollection.recalcAll,
                priority: true,
                volatile: Utils.getBooleanOption(options, 'volatile', true)
            });

            // initialize a timer that aborts the returned promise after the specified delay
            var timeout = Utils.getIntegerOption(options, 'timeout', 0);
            if (timeout > 0) {

                // create a new deferred object that represents this method invocation
                // (this allows to call the method multiple times with individual timeouts)
                var deferred = DeferredUtils.createDeferred(app, 'DependencyManager: recalcFormulas');
                promise.done(deferred.resolve.bind(deferred));
                promise.fail(deferred.reject.bind(deferred));
                promise.progress(deferred.notify.bind(deferred));

                // create a promise with timeout from the new deferred object,
                // set the 'calcOnLoad' document flag if the timeout is reached
                promise = this.createAbortablePromise(deferred, setCalcOnLoadFlag, timeout);
            }

            return promise;
        };

        /**
         * Returns an up-to-date cell value for the specified cell in the
         * document. If the cell is a formula cell that this dependency manager
         * has marked as dirty, the formula will be recalculated, and all cells
         * referred by the formula that are dirty too will be recalculated
         * recursively. Otherwise, the current value of the cell as stored in
         * the cell model will be returned.
         *
         * @param {SheetModel} sheetModel
         *  The model of the sheet containing the cell to be resolved.
         *
         * @param {Address} address
         *  The address of the cell whose value will be resolved.
         *
         * @returns {Object}
         *  A descriptor object with information about the resolved cell value,
         *  with the following properties:
         *  - {Any} value
         *      The resolved scalar value of the specified cell.
         *  - {String} type
         *      How the cell value has been resolved. If set to 'calculated',
         *      the value has just been calculated by interpreting the dirty
         *      cell formula. If set to 'cached', the result has already been
         *      calculated by a preceding call of this method during the update
         *      cycle  currently running. If set to 'fixed', the value has been
         *      returned directly from the cell model (either because the cell
         *      is not a formula cell, or the formula was not dirty).
         *  - {Number} time
         *      The time in milliseconds it took to calculate the result of a
         *      dirty formula cell (type 'calculated'). Will be zero for all
         *      other value types ('cached' and 'fixed').
         */
        this.getCellValue = (function () {

            // stack with the addresses of all formulas currently calculated recursively
            var formulaAddressStack = new AddressArray();

            // calculates the result of a cell formula
            function calculateCellFormula(sheetModel, tokenArray, refAddress, targetAddress, matrixRange) {

                // take the time it needs to calculate the formula (and all formulas it depends on)
                var startTime = _.now();

                var sheetUid = sheetModel.getUid();
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('calculating formula in cell ' + sheetUid + '!' + targetAddress);
                });

                // put the target address with sheet UID onto the stack (used to detect circular references)
                var formulaAddress = targetAddress.clone();
                formulaAddress.sheetUid = sheetUid;
                formulaAddressStack.push(formulaAddress);

                // first, calculate the formula result
                var formulaResult = tokenArray.interpretFormula(matrixRange ? 'mat' : 'val', {
                    refAddress: refAddress,
                    targetAddress: matrixRange ? refAddress : targetAddress,
                    matrixRange: matrixRange,
                    recalcDirty: true
                });

                // detect circular reference errors, and other warnings and errors
                if (formulaResult.code === 'circular') {
                    DependencyUtils.withLogging(function () {
                        var addressNames = formulaAddressStack.map(function (address) { return address.sheetUid + '!' + address; });
                        DependencyUtils.warn('reference cycle found: ' + addressNames.join());
                    });
                    resultCollection.referenceCycles.push(formulaAddressStack.clone());
                } else if (formulaResult.type !== 'valid') {
                    resultCollection.hasErrors = true;
                }
                formulaAddressStack.pop();

                // keep old result in the map, if the formula has been recalculated already due to a circular reference
                var addressKey = targetAddress.key();
                var resultMap = resultCollection.resultMaps.getOrConstruct(sheetUid, SimpleMap);
                if (resultMap.has(addressKey)) { return null; }

                // immediately update the URL of the HYPERLINK function
                var cellCollection = sheetModel.getCellCollection();
                cellCollection.setFormulaURL(targetAddress, formulaResult.url || null);

                // add the timestamp to the result map, needed to detect timeouts during calculation
                var time = _.now() - startTime;

                // distribute the elements of a matrix result into the result map
                if (matrixRange) {
                    var matrix = formulaResult.value;
                    var col1 = matrixRange.start[0];
                    var row1 = matrixRange.start[1];
                    IteratorUtils.forEach(matrixRange.iterator(), function (address) {
                        resultMap.insert(address.key(), {
                            address: address,
                            value: Scalar.getCellValue(matrix.get(address[1] - row1, address[0] - col1)),
                            time: time
                        });
                    });
                    // return the cell result of the correct matrix element
                    return resultMap.get(addressKey, null);
                }

                // insert the new result into the result map
                return resultMap.insert(addressKey, {
                    address: targetAddress,
                    value: Scalar.getCellValue(formulaResult.value),
                    time: time
                });
            }

            // the implementation of the method getCellValue()
            function getCellValue(sheetModel, address) {

                // the result descriptor
                var valueDesc = { value: null, type: null, time: 0 };

                // first, try to find a result already calculated by a preceding call of this method
                var sheetUid = sheetModel.getUid();
                var addressKey = address.key();
                var resultDesc = resultCollection.resultMaps.with(sheetUid, function (resultMap) {
                    return resultMap.get(addressKey, null);
                });
                if (resultDesc) {
                    valueDesc.value = resultDesc.value;
                    valueDesc.type = 'cached';
                    return valueDesc;
                }

                // the cell collection of the current sheet
                var cellCollection = sheetModel.getCellCollection();

                // in 'recalculate all' mode, resolve the token array directly from the cell collection
                if (pendingCollection.recalcAll) {

                    var tokenDesc = cellCollection.getTokenArray(address, { fullMatrix: true });
                    resultDesc = tokenDesc ? calculateCellFormula(sheetModel, tokenDesc.tokenArray, tokenDesc.refAddress, address, tokenDesc.matrixRange) : null;

                } else {

                    // try to find a dirty formula, and calculate its result
                    var formulaKey = getCellKey(sheetUid, address);
                    resultCollection.cellFormulaMap.with(formulaKey, function (formulaDesc) {
                        // calculate the result of the cell formula
                        resultDesc = calculateCellFormula(sheetModel, formulaDesc.tokenArray, formulaDesc.refAddress, address, formulaDesc.matrixRange);
                        // remove the formula descriptor (MUST be done after calculation, needed to detect circular references!)
                        // if the formula has been removed already due to a circular reference, keep that result in the map
                        resultCollection.cellFormulaMap.remove(formulaKey);
                    });
                }

                // fill and return the descriptor if a formula has been calculated
                if (resultDesc) {
                    valueDesc.value = resultDesc.value;
                    valueDesc.type = 'calculated';
                    valueDesc.time = resultDesc.time;
                    return valueDesc;
                }

                // return the current cell value for all other formulas, and simple cells
                valueDesc.value = cellCollection.getValue(address);
                valueDesc.type = 'fixed';
                return valueDesc;
            }

            return getCellValue;
        }());

        /**
         * Resource keys of all unimplemented functions that occur in any
         * formula known by this dependency manager.
         *
         * @returns {Array<String>}
         *  The resource keys of all unimplemented functions that occur in any
         *  formula known by this dependency manager.
         */
        this.getMissingFunctionKeys = function () {

            // collect all function keys from all sheets in a map to prevent duplicates
            var missingFuncMap = new SimpleMap();
            sheetDescMap.forEach(function (sheetDesc) {
                missingFuncMap.merge(sheetDesc.missingFuncMap);
            });

            // return the function resource keys as array
            return missingFuncMap.keys();
        };

        // initialization -----------------------------------------------------

        // initialization after document import
        docModel.waitForImportSuccess(function () {

            // recalculate all formulas loaded from the document if specified
            calcOnLoad = docModel.getDocumentAttribute('calcOnLoad');

            // initialize dependency manager for all existing sheets after import
            IteratorUtils.forEach(docModel.createSheetIterator(), registerSheetModel);

            // keep track of all changes in the document
            this.listenTo(docModel, 'insert:sheet:after', insertSheetHandler);
            this.listenTo(docModel, 'delete:sheet:before', deleteSheetHandler);
            this.listenTo(docModel, 'move:sheet:before', moveSheetHandler);
            this.listenTo(docModel, 'rename:sheet', renameSheetHandler);
            this.listenTo(docModel, 'insert:name', insertNameHandler);
            this.listenTo(docModel, 'delete:name', deleteNameHandler);
            this.listenTo(docModel, 'change:name', changeNameHandler);
            this.listenTo(docModel, 'delete:table', deleteTableHandler);
            this.listenTo(docModel, 'change:table', changeTableHandler);
            this.listenTo(docModel, 'change:cells', changeCellsHandler);
            this.listenTo(docModel, 'move:cells', moveCellsHandler);
            this.listenTo(docModel, 'change:rows', changeRowsHandler);
            this.listenTo(docModel, 'insert:rule', insertRuleHandler);
            this.listenTo(docModel, 'delete:rule', deleteRuleHandler);
            this.listenTo(docModel, 'change:rule', changeRuleHandler);

            // initial update cycle for the imported sheets
            startUpdateCycle();
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            timerHelper.destroy();
            self = app = docModel = timerHelper = null;
            formulaDescMap = sheetDescMap = formulaDictionary = null;
            pendingCollection = resultCollection = null;
        });

    } }); // class DependencyManager

    // exports ================================================================

    return DependencyManager;

});
