/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chartmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/drawinglayer/model/drawingmodel',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'io.ox/office/drawinglayer/view/chartformatter',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/model/drawing/drawingmodelmixin',
    'io.ox/office/spreadsheet/model/drawing/dataseriesmodel',
    'io.ox/office/spreadsheet/model/drawing/titlemodel',
    'io.ox/office/spreadsheet/model/drawing/axismodel',
    'io.ox/office/spreadsheet/model/drawing/legendmodel',
    'io.ox/office/spreadsheet/model/drawing/manualcolorchart',
    'io.ox/office/drawinglayer/lib/canvasjs.min'
], function (Utils, LocaleData, Parser, Color, DrawingModel, ChartStyleUtil, ChartFormatter, SheetUtils, Operations, TokenArray, DrawingModelMixin, DataSeriesModel, TitleModel, AxisModel, LegendModel, ManualColorHandler, CanvasJS) {

    'use strict';

    var // convenience shortcuts
        Range3D = SheetUtils.Range3D,
        RangeArray = SheetUtils.RangeArray,
        Range3DArray = SheetUtils.Range3DArray,

        TYPES = {
            'column standard': 'column',
            'column clustered': 'column',
            'column stacked': 'stackedColumn',
            'column percentStacked': 'stackedColumn100',
            'bar standard': 'bar',
            'bar clustered': 'bar',
            'bar stacked': 'stackedBar',
            'bar percentStacked': 'stackedBar100',
            'area standard': 'area',
            'area stacked': 'stackedArea',
            'area percentStacked': 'stackedArea100',
            'line standard': 'line',
            'line clustered': 'line',
            'line stacked': 'line',
            'line percentStacked': 'spline',
            'line standard curved': 'spline',
            'line clustered curved': 'spline',
            'line stacked curved': 'spline',
            'line percentStacked curved': 'spline',
            'scatter standard': 'line',
            'scatter standard curved': 'spline',
            'bubble standard': 'bubble',
            'pie standard': 'pie',
            'donut standard': 'doughnut'
        },

        MARKER_TYPES = {
            circle: 'circle',
            dot: 'circle',
            square: 'square',
            triangle: 'triangle',
            x: 'cross',
            none: 'none'
        },

        MARKER_LIST = ['circle', 'square', 'triangle', 'cross'];

    // class SheetChartModel ==================================================

    /**
     * A class that implements a chart model to be inserted into a sheet of a
     * spreadsheet document.
     *
     * Additionally to the events triggered by the base class DrawingModel,
     * instances of this class trigger the following 'change:drawing' events:
     * - 'insert:series'
     *      After a new data series has been inserted into this chart. Event
     *      handlers receive the logical index of the new data series.
     * - 'delete:series'
     *      After a new data series has been deleted from this chart. Event
     *      handlers receive the last logical index of the data series.
     *
     * @constructor
     *
     * @extends DrawingModel
     * @extends DrawingModelMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this chart model.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the sheet that contains this chart object.
     *
     * @param {Object} [initAttributes]
     *  Initial formatting attribute set for this chart model.
     */
    function SheetChartModel(sheetModel, initAttributes) {

        var self = this;

        var docModel = sheetModel.getDocModel();

        var app = docModel.getApp();

        var fileFormat = docModel.getApp().getFileFormat();

        var controllerToken = null;

        var data = {
            animationEnabled: false,
            culture:  'en',
            bg: 'white',
            backgroundColor: 'transparent',
            axisX: { stripLines: [], labelAutoFit: true },
            axisY: { stripLines: [], labelAutoFit: true },
            axisZ: { stripLines: [] },
            creditHref: '',
            creditText: '',
            title: { text: '' },
            legend: {},
            data: [],
            series: [],
            toolTip: {
                backgroundColor: 'black',
                borderColor: 'black',
                fontColor: 'white',
                cornerRadius: 3,
                borderThickness: 4,
                fontStyle: 'normal',
                contentFormatter: function (e) {
                    var content = ' ';
                    for (var i = 0; i < e.entries.length; i++) {
                        if (e.entries[i].dataPoint.label) {
                            content += e.entries[i].dataPoint.label;
                        } else {
                            content += e.entries[i].dataPoint.x;
                        }
                        content += ': ';
                        if (e.entries[i].dataPoint.name) {
                            content += e.entries[i].dataPoint.name;
                        } else {
                            content += e.entries[i].dataPoint.y;
                        }
                    }

                    return Utils.escapeHTML(content);
                }
            }
        };

        var indexLabelAttrs = {
            fontSize: 10,
            color: { type: 'auto' },
            fontName: 'Arial'
        };

        var colorHandler = null;
        var chartFormatter = null;

        // base constructors --------------------------------------------------

        DrawingModel.call(this, docModel, 'chart', initAttributes, { families: 'chart fill line' });
        DrawingModelMixin.call(this, sheetModel, cloneConstructor, generateRestoreOperations, generateUpdateFormulaOperations);

        /**
         * this is only called by the Drawingframe to initialize the REAL CanvasJS-chart
         */
        this.getModelData = function () {
            return data;
        };

        var axisModelMap = {
            x: new AxisModel(this, 'x'),
            y: new AxisModel(this, 'y'),
            z: new AxisModel(this, 'z')
        };

        var mainTitleModel = new TitleModel(this, 'main', data.title, 'text');

        var legendModel = new LegendModel(this, {}, data.legend);

        // private methods ----------------------------------------------------

        /**
         * Returns a clone of this chart model for the specified sheet. Used by
         * the implementation of the public clone() method defined by the class
         * DrawingModelMixin.
         *
         * @param {SheetModel} targetModel
         *  The sheet model that will contain the cloned chart model.
         *
         * @returns {SheetChartModel}
         *  A clone of this chart model, initialized for ownership by the
         *  passed target sheet model.
         */
        function cloneConstructor(targetModel) {

            // new created copy of all attributemodels attributes
            var cloneData = self.getCloneData();

            // pass private data as hidden argument to the constructor
            return new SheetChartModel(targetModel, cloneData.attrs, cloneData);
        }

        /**
         * Generates additional undo operations needed to restore the contents
         * of this chart object, after it has been restored with an initial
         * 'insertDrawing' operation.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of this chart object in the sheet, as expected by the
         *  method SheetOperationsGenerator.generateDrawingOperation().
         */
        function generateRestoreOperations(generator, position) {

            // restore all data series of the chart
            data.series.forEach(function (seriesData, index) {
                seriesData.model.generateRestoreOperations(generator, position, index);
            });

            // restore all axes of the chart
            _.each(axisModelMap, function (axisModel) {
                axisModel.generateRestoreOperations(generator, position);
            });

            // restore the main title and legend
            mainTitleModel.generateRestoreOperations(generator, position);
            legendModel.generateRestoreOperations(generator, position);
        }

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the source links in this chart object.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of this drawing object in the sheet, as expected by
         *  the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         */
        function generateUpdateFormulaOperations(generator, position, changeDesc) {

            // generate the change operations for the source links in all data series
            // (in reverse order, to get the correct indexes when deleting entire data series)
            Utils.iterateArray(data.series, function (seriesData, index) {
                var generator2 = sheetModel.createOperationsGenerator();
                seriesData.model.generateUpdateFormulaOperations(generator2, position, index, changeDesc);
                generator.appendOperations(generator2);
                // prepend undo operations to restore the data series (needed for correct series array index)
                generator.prependOperations(generator2, { undo: true });
            }, { reverse: true });

            // update the source links of the axis titles
            _.each(axisModelMap, function (axisModel) {
                axisModel.generateUpdateFormulaOperations(generator, position, changeDesc);
            });

            // update the source link of the main title
            mainTitleModel.generateUpdateFormulaOperations(generator, position, changeDesc);
        }

        /**
         * Debounced version of the public method SheetChartModel.refresh()
         * with increased delay time.
         */
        var refreshChartDebounced = this.createDebouncedMethod(_.noop, innerRefresh, { delay: 500, infoString: 'SheetChartModel.refresh() for listeners', app: app });

        /**
         * Handles 'change:cells' events triggered by the document model (from
         * any sheet), and refreshes this chart model debounced, if it refers
         * to any of the changed cells.
         */
        function changeCellsHandler(event, sheet, changeData) {

            // create 3D ranges from changed cells (sheet index in the ranges required by series model)
            var ranges = RangeArray.mergeAddresses(changeData.allCells);
            ranges = Range3DArray.createFromRanges(ranges, sheet);

            // check passed ranges if they overlap with the source ranges of the data series
            var changed = false;
            _.each(data.series, function (seriesData) {
                if (seriesData.needUpdate) {
                    changed = true;
                } else if (seriesData.model.rangesOverlap(ranges)) {
                    seriesData.needUpdate = changed = true;
                }
            });
            if (mainTitleModel.rangesOverlap(ranges)) { changed = true; }

            if (changed) { refreshChartDebounced(); }
        }

        /**
         * Updates the visibility of data points, after columns or rows have
         * been shown or hidden in a sheet.
         */
        function updateColRowVisibility(sheet, interval, columns) {

            // the 3D cell range with changed visibility
            var range = Range3D.createFromRange(docModel.makeFullRange(interval, columns), sheet);

            // check passed ranges if they overlap with the source ranges of the data series
            var changed = false;
            data.series.forEach(function (seriesData) {
                if (seriesData.needUpdate) {
                    changed = true;
                } else if (seriesData.model.rangesOverlap(range)) {
                    changed = true;
                    seriesData.needUpdate = true;
                }
            });
            if (mainTitleModel.rangesOverlap(range)) { changed = true; }

            if (changed) { refreshChartDebounced(); }
        }

        function changeAttributes(evt, mergedAttributes, oldMergedAttributes) {
            var mc = mergedAttributes.chart;
            var oc = oldMergedAttributes.chart;
            var mf = mergedAttributes.fill;
            var of = oldMergedAttributes.fill;
            var ml = mergedAttributes.line;
            var ol = oldMergedAttributes.line;

            if (!_.isEqual(mc, oc) || !_.isEqual(mf, of) || !_.isEqual(ml, ol)) {

                updateChartType();

                if (mc.chartStyleId !== null && mc.chartStyleId !== undefined && mf.color.type === 'auto') {
                    self.updateBackgroundColor();
                }

                changeDrawing(evt, 'series');
            }
        }

        function isXYType() {
            var attributes = self.getMergedAttributeSet(true);
            return attributes.chart.type.indexOf('bubble') === 0 || attributes.chart.type.indexOf('scatter') === 0;
        }

        function isLineType() {
            var type = self.getMergedAttributeSet(true).chart.type;
            if (type.indexOf('line') >= 0 || type.indexOf('scatter') >= 0) {
                return true;
            }
            return false;
        }

        function changeDrawing(evt, subtype) {
            var lAttrs = legendModel.getMergedAttributeSet(true);
            var attributes = self.getMergedAttributeSet(true);

            var cjst = getCJSType();

            _(data.series).each(function (dataSeries, index) {
                if (dataSeries.needUpdate) {
                    return;
                }
                if (subtype === 'series') {
                    var attrs = dataSeries.model.getMergedAttributeSet(true);
                    colorHandler.handleColor(attrs.fill, dataSeries, dataSeries.seriesIndex, 'markerColor', data.series.length);
                    if (isLineType()) {
                        colorHandler.handleColor(attrs.line, dataSeries, dataSeries.seriesIndex, 'color', data.series.length);
                    } else {
                        if (Utils.getStringOption(attrs.fill, 'type', 'none') === 'none' && Utils.getStringOption(attrs.line, 'type', 'none') !== 'none') {
                            // use line color for fill, workaround for Bug 46848
                            colorHandler.handleColor(attrs.line, dataSeries, dataSeries.seriesIndex, 'color', data.series.length);
                        } else {
                            colorHandler.handleColor(attrs.fill, dataSeries, dataSeries.seriesIndex, 'color', data.series.length);
                        }
                    }
                }
                if (cjst === 'pie' || cjst === 'doughnut') {
                    dataSeries.startAngle = attributes.chart.rotation - 90;
                }
                if (lAttrs.legend.pos === 'off') {
                    dataSeries.showInLegend = false;
                } else {
                    dataSeries.showInLegend = true;
                    if (cjst === 'pie' || cjst === 'doughnut') {
                        if (index > 0) {
                            //we only see the first series
                            dataSeries.showInLegend = false;
                        } else if (dataSeries.dataPoints.length > 50) {
                            //canvasjs has rendering problems with to big legend-sizes
                            dataSeries.showInLegend = false;
                        }
                    }
                }

                if (dataSeries.showInLegend && (!dataSeries.dataPoints || !dataSeries.dataPoints.length)) {
                    dataSeries.showInLegend = false;
                }

                ChartStyleUtil.handleCharacterProps(self, indexLabelAttrs, dataSeries, 'indexLabel');

                if (attributes.chart.dataLabel) {
                    dataSeries.indexLabel = '{name}';
                    dataSeries.indexLabelLineColor = null;
                    dataSeries.indexLabelPlacement = 'inside';
                } else {
                    dataSeries.indexLabel = ' ';
                    dataSeries.indexLabelLineColor = 'transparent';
                    dataSeries.indexLabelPlacement = null;
                }

                if (isLineType()) {
                    if (!dataSeries.marker) {
                        var markerIndex = index % MARKER_LIST.length;
                        dataSeries.marker = MARKER_LIST[markerIndex];
                    }

                    dataSeries.legendMarkerType = dataSeries.marker;
                } else {
                    dataSeries.legendMarkerType = 'square';
                }

                _.each(dataSeries.dataPoints, function (dataPoint, pointIndex) {
                    if (isLineType()) {
                        if (data.series.length === 1 && self.isVaryColorEnabled()) {
                            var markerIndex = pointIndex % MARKER_LIST.length;
                            dataPoint.markerType = MARKER_LIST[markerIndex];
                        } else {
                            dataPoint.markerType = dataSeries.marker;
                        }
                        dataPoint.markerSize = sheetModel.getEffectiveZoom() * 7;
                    } else {
                        dataPoint.markerType = null;
                        dataPoint.markerSize = null;
                    }
                });
            });

            chartFormatter.format();

            legendModel.refreshInfo();

            mainTitleModel.refreshInfo();

            refreshAxis();

            updateMaxDataPointSize();
        }

        function getCJSType() {
            var type = self.getChartType();
            var res = TYPES[type];

            if (!res) {
                Utils.warn('no correct charttype found', type);
                if (!type) {
                    res = 'column';
                } else if (!type.indexOf('radar')) {
                    res = 'line';
                } else {
                    res = 'column';
                }
            }

            if (res === 'line' && self.getMergedAttributeSet(true).chart.curved === true) {
                res = 'spline';
            }
            return res;
        }

        function updateChartType() {
            var cjs = getCJSType();
            for (var i = 0; i < data.series.length; i++) {
                var series = data.series[i];
                series.type = cjs;
            }
            self.trigger('change:drawing');
        }

        /**
         * Inserts a new data series model into this chart model.
         */
        function insertDataSeries(index, attrs) {

            // workarounds for Bug 41988 (black charts from mac-excel)
            if (attrs.fill && !attrs.fill.color) {
                attrs.fill.color = { type: 'auto' };
            }
            if (!isLineType() && attrs.line && attrs.line.width) {
                delete attrs.line.width;
            }

            var dataSeries = {
                type: getCJSType(),
                fillOpacity: 1
            };
            data.series.splice(index, 0, dataSeries);

            dataSeries.marker = MARKER_TYPES[attrs.series.marker];

            var model = new DataSeriesModel(sheetModel, attrs);

            dataSeries.model = model;
            dataSeries.seriesIndex = index;

            dataSeries.needUpdate = true;

            dataSeries.dataPoints = [];
            dataSeries.name = '';
            dataSeries.markerColor = 'transparent';
            dataSeries.color = 'transparent';
            dataSeries.indexLabelFontColor = 'transparent';

            self.trigger('change:drawing');

            if (controllerToken) {
                updateTokenArrays();
            }

            return true;
        }

        function compareFormats(group, other) {

            if (group.format === other.format) {
                group.display = null;
                group.result = (group.result || group.value) + (other.result || other.value);
                return;
            }

            if (group.format.isAnyDateTime() && other.format.isAnyDateTime()) {
                group.display = null;
                group.result = (group.result || group.value) + (other.result || other.value);

                var groupDate = group.format.isAnyDate();
                var otherDate = other.format.isAnyDate();
                var groupTime = group.format.isAnyTime();
                var otherTime = other.format.isAnyTime();

                //take org format
                if ((groupDate && otherDate) || (groupTime && otherTime)) { return; }

                // use combined date/time format
                if ((groupDate && otherTime) || (groupTime && otherDate)) {
                    group.format = Parser.parseFormatCode(fileFormat, 'op', LocaleData.SHORT_DATE + ' ' + LocaleData.LONG_TIME);
                }
            }
        }

        function reduceBigRange(source, index, name) {

            var ranges = data.series[index].model.resolveRanges(name);
            var range = ranges ? ranges.first() : null;
            if (!range) { return source; }

            var cols = range.cols();
            var rows = range.rows();
            if (cols > rows) {
                Utils.warn('chart with 2d ranges as "' + name + '" reduceBigRange() must be implemented!');
                return source;
            } else if (rows > cols) {
                var newList = [];
                var numberFormatter = docModel.getNumberFormatter();
                for (var row = 0; row < rows; ++row) {
                    var group = source[row * cols];
                    for (var col = 1; col < cols; ++col) {
                        var other = source[row * cols + col];
                        compareFormats(group, other);
                    }
                    group.display = numberFormatter.formatValue(group.format, group.result || group.value);
                    newList.push(group);
                }
                return newList;
            }
        }

        function updateMaxDataPointSize() {
            if (self.getSeriesCount() === 0) { return; }
            var rect = self.getRectangle();
            if (!rect) { return; }

            delete data.dataPointMaxWidth;
            delete data.dataPointMinWidth;

            var attrs = self.getMergedAttributeSet(true);
            var stacking = attrs.chart.stacking;
            var chartType = attrs.chart.type;
            var dataPointMaxWidth = null;

            if (stacking !== 'percentStacked' && stacking !== 'stacked' && data.series[0].dataPoints.length === 1) {
                var multi = data.series.length;
                if (chartType.indexOf('bar') === 0) {
                    dataPointMaxWidth = Math.floor(rect.height / multi);
                } else if (chartType.indexOf('column') === 0) {
                    dataPointMaxWidth = Math.floor((rect.width * 0.9) / multi);
                }
                if (dataPointMaxWidth) {
                    data.dataPointMaxWidth = dataPointMaxWidth * 1.5;
                    data.dataPointMinWidth = dataPointMaxWidth * 0.5;
                }
            }

            ///////////////////////////////////

            var legendPos = legendModel.getMergedAttributeSet(true).legend.pos;
            var maxWidth = rect.width;

            if (legendPos === 'left' || legendPos === 'right' || legendPos === 'topRight') {
                maxWidth = rect.width / 4;
            }

            data.legend.maxWidth = maxWidth;
            data.legend.itemWrap = true;
            data.legend.maxHeight = rect.height;
        }

        function seriesAttrsToCellArray(array) {
            var result = [];
            if (array) {
                array.forEach(function (entry) {
                    var res = { display: entry };
                    if (isFinite(entry)) {
                        res.value = parseFloat(entry);
                    }
                    result.push(res);
                });
            }
            return result;
        }

        /**
         * Refreshes the source values of the data series of this chart.
         */
        function innerRefresh() {

            // nothing to do during import of the document
            if (!self.isImportFinished()) { return; }

            var // all ranges to be querried, as array of arrays of range addresses
                sourceRanges = null,
                // maps series indexes and source link types to array elements in 'sourceRanges'
                collectInfo = null;

            _(data.series).each(function (seriesData, series) {
                if (!seriesData.needUpdate) { return; }

                if (!sourceRanges) {
                    sourceRanges = [];
                    collectInfo = [];
                }

                var info = { series: series, indexes: {} };

                var tokenCount = 0;

                seriesData.model.iterateTokenArrays(function (tokenArray, attrName) {
                    tokenCount++;
                    var entryRanges = tokenArray.resolveRangeList({ resolveNames: true });
                    if (!entryRanges.empty()) {
                        info.indexes[attrName] = sourceRanges.length;
                        sourceRanges.push(entryRanges);
                    }
                });
                if (tokenCount) {
                    collectInfo.push(info);
                } else {

                    var sAttrs = seriesData.model.getExplicitAttributeSet().series;
                    chartFormatter.update(series, seriesAttrsToCellArray(sAttrs.values), seriesAttrsToCellArray(sAttrs.title), seriesAttrsToCellArray(sAttrs.names), seriesAttrsToCellArray(sAttrs.bubbles));
                }
                seriesData.needUpdate = false;
            });

            if (!sourceRanges || !sourceRanges.length) {
                if (mainTitleModel.refreshInfo()) {
                    // fix for Bug 48095
                    self.trigger('change:drawing', 'series');
                } else {
                    changeDrawing();
                }
                return;
            }

            if (sourceRanges.length > 0) {

                // query contents of visible cells for the data series (be nice and use a time-sliced loop)
                var allContents = [];
                var promise = self.iterateArraySliced(sourceRanges, function (ranges) {
                    allContents.push(docModel.getRangeContents(ranges, { blanks: true, visible: true, attributes: true, display: true, maxCount: 1000 }));
                }, { infoString: 'ChartModel: innerRefresh', app: app });

                // do not call the handler function, if the chart has been deleted in the meantime
                self.waitForSuccess(promise, function () {

                    _(collectInfo).each(function (info) {

                        // returns the cell contents for the specified source link type
                        function getCellContents(key) {
                            return _.isNumber(info.indexes[key]) ? allContents[info.indexes[key]] : [];
                        }

                        // series values must exist
                        var valuesSource = getCellContents('values');
                        var titleSource = getCellContents('title');
                        var namesSource = getCellContents('names');
                        var bubblesSource = getCellContents('bubbles');

                        if (namesSource.length && namesSource.length >= valuesSource.length * 2) {
                            // workaround for TOO BIG ranges (2d range)
                            namesSource = reduceBigRange(namesSource, info.series, 'names');
                        }
                        if (titleSource.length && titleSource.length >= valuesSource.length * 2) {
                            // workaround for TOO BIG ranges (2d range)
                            titleSource = reduceBigRange(titleSource, info.series, 'title');
                        }
                        chartFormatter.update(info.series, valuesSource, titleSource, namesSource, bubblesSource);

                    });
                    refreshAxis();
                    self.trigger('change:drawing', 'series');
                });
            } else {
                _(data.series).each(function (seriesData) {
                    seriesData.needUpdate = false;
                });
                self.trigger('change:drawing', 'series');
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Handler for the document operation 'insertChartDataSeries'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'insertChartDataSeries' operation.
         */
        this.applyInsertSeriesOperation = function (context) {
            var index = context.getOptInt('series', data.series.length);
            insertDataSeries(index, context.getOptObj('attrs'));
        };

        /**
         * Handler for the document operation 'deleteChartDataSeries'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'deleteChartDataSeries' operation.
         */
        this.applyDeleteSeriesOperation = function (context) {

            var index = context.getInt('series'),
                seriesData = data.series[index];
            context.ensure(seriesData, 'invalid series index');

            seriesData.model.destroy();

            //workaround for Bug 43958, changing chart type to Bubble chart, dataseries count is cut to half
            //so we make a copy of the array, that the array in CanvasJS has still the same length
            data.series = data.series.slice(0);

            data.series.splice(index, 1);
            this.trigger('change:drawing');

            if (controllerToken) {
                updateTokenArrays();
            }
        };

        /**
         * Handler for the document operation 'setChartDataSeriesAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartDataSeriesAttributes' operation.
         */
        this.applyChangeSeriesOperation = function (context) {

            var seriesData = data.series[context.getInt('series')];
            context.ensure(seriesData, 'invalid series index');

            seriesData.model.setAttributes(context.getObj('attrs'));
            seriesData.needUpdate = true;
            updateTokenArrays();
            this.trigger('change:drawing', 'series');
        };

        /**
         * Handler for the document operation 'setChartAxisAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartAxisAttributes' operation.
         */
        this.applyChangeAxisOperation = function (context) {

            var axisModel = axisModelMap[context.getStr('axis')];
            context.ensure(axisModel, 'invalid axis identifier');

            axisModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing');
        };

        /**
         * Handler for the document operation 'setChartGridlineAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartGridlineAttributes' operation.
         */
        this.applyChangeGridOperation = function (context) {

            var axisModel = axisModelMap[context.getStr('axis')];
            context.ensure(axisModel, 'invalid axis identifier');

            var gridModel = axisModel.getGrid();
            context.ensure(gridModel, 'missing grid line model');

            gridModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing');
        };

        /**
         * Handler for the document operation 'setChartTitleAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartTitleAttributes' operation.
         */
        this.applyChangeTitleOperation = function (context) {

            var axisId = context.getStr('axis');
            var titleModel = (axisId === 'main') ? mainTitleModel : (axisId in axisModelMap) ? axisModelMap[axisId].getTitle() : null;
            context.ensure(titleModel, 'invalid axis identifier');

            titleModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing');
        };

        /**
         * Handler for the document operation 'setChartLegendAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setChartLegendAttributes' operation.
         */
        this.applyChangeLegendOperation = function (context) {
            legendModel.setAttributes(context.getObj('attrs'));
            this.trigger('change:drawing', 'series');
        };

        // public methods -----------------------------------------------------

        this.setMainName = function (name) {
            mainTitleModel.setName(name);
        };

        /**
         * is called by the DrawingController for a selected chart
         */
        this.getChartType = function () {
            var attributes = self.getMergedAttributeSet(true);

            var type = attributes.chart.type;
            type = type.replace('2d', '').replace('3d', '');

            var stacking = attributes.chart.stacking;
            if (stacking.length) {
                type = type + ' ' + stacking;
            }
            var curved = attributes.chart.curved;
            if (curved && /(line|scatter)/.test(type)) {
                type += ' curved';
            }
            return type;
        };

        this.getChartTypeForGui = function () {
            var chartType = this.getChartType();
            return chartType.replace('clustered', 'standard');
        };

        /**
         * Refreshes the source values of the data series of this chart.
         */
        this.refresh = this.createDebouncedMethod(_.noop, innerRefresh, { delay: 50, maxDelay: 250, infoString: 'SheetChartModel.refresh()', app: app });

        this.getSeriesModel = function (index) {
            return data.series[index].model;
        };

        this.getLegendModel = function () {
            return legendModel;
        };

        this.getAxisModel = function (axisId) {
            return axisModelMap[axisId];
        };

        this.getTitleModel = function (axisId) {
            return (axisId === 'main') ? mainTitleModel : this.getAxisModel(axisId).getTitle();
        };

        /**
         *  Adds formula tokens to the given containing the source ranges of all data
         *  series of this chart object.
         *
         *  @param {Array} array where tokenarrays are pushed in
         */
        this.getTokenArrays = function (res) {
            updateTokenArrays();
            res.push(controllerToken);
        };

        /**
         * Invokes the passed callback function for all data series.
         *
         * @param {Function} callback
         *  (1) {Number} seriesIndex the index of the data series
         *  (2) {Object} tokens the object with the token name(key) and tokenArray(value)
         */
        this.iterateTokenArrays = function (callback) {
            data.series.forEach(function (seriesData) {
                var tokens = {};
                seriesData.model.iterateTokenArrays(function (tokenArray, attrName) {
                    tokens[attrName] = tokenArray;
                });
                callback(seriesData.seriesIndex, tokens);
            });
        };

        function updateTokenArrays() {
            if (!controllerToken) {
                controllerToken = new TokenArray(sheetModel, 'drawing');
            }

            var tmpList = [];
            _.each(data.series, function (seriesData) {
                seriesData.model.iterateTokenArrays(function (tokenArray) {
                    var formula = tokenArray.getFormula('op');
                    if (formula.length) {
                        tmpList.push(formula);
                    }
                });
            });

            controllerToken.parseFormula('op', tmpList.join(','));
        }

        function refreshAxis() {
            if (!data.series.length) {
                return;
            }
            axisModelMap.x.refreshInfo();
            axisModelMap.y.refreshInfo();
        }

        function getStyleId() {
            return self.getMergedAttributeSet(true).chart.chartStyleId;
        }

        function clearSeriesColor(generator, position) {
            _(data.series).each(function (dataSeries, index) {
                var at = dataSeries.model.getMergedAttributeSet(true);
                if (at.fill.color.type !== 'auto' || at.line.color.type !== 'auto') {
                    generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position, { series: index, attrs: { fill: { color: Color.AUTO }, line: { color: Color.AUTO } } });
                }
            });
        }

        function createBackgroundColorOp(styleId) {
            var drAttrs = { chart: { chartStyleId: styleId } };

            var bgColor = colorHandler.getBackgroundColor(styleId);
            if (bgColor) {
                drAttrs.fill = { color: bgColor, type: 'solid' };
            }
            return drAttrs;
        }

        function sourceExistsComplete(name) {
            return (data.series.length > 0) && data.series.every(function (dataSeries) {
                return !_.isEmpty(dataSeries.model.getMergedAttributeSet(true).series[name]);
            });
        }

        this.updateBackgroundColor = function () {
            //only fallback for broken background restore!!!
            var bgColor = colorHandler.getBackgroundColor(getStyleId());
            if (bgColor) {
                var drAttrs = {};
                drAttrs.fill = { color: bgColor, type: 'solid' };
                self.setAttributes(drAttrs);
            }
        };

        this.isXYType = isXYType;

        /**
         * return true if the chart has only one series or is type of bubble or donut (because there you see only the first series)
         * @returns {Boolean}
         */
        this.isVaryColorEnabled = function () {
            return colorHandler.isVaryColorEnabled();
        };

        this.isAxesEnabled = function () {
            return !self.isPieOrDonut();
        };

        this.isPieOrDonut = function () {
            var type = getCJSType();
            return type === 'pie' || type === 'doughnut';
        };

        /**
         * it generates a unique-color-ID out of the internal chartStyleId
         * for describing the color patterns
         * return null, if one or more series have a unique fill color
         * @returns {String|null}
         */
        this.getColorSet = function () {
            var styleId = getStyleId() - 1;
            var colorSet = styleId % 8;
            return 'cs' + colorSet;
        };

        /**
         * it generates a unique-style-ID out of the internal chartStyleId
         * for describing the background and the highlight of the chart
         * it ignores unique fill colors of the series
         * @returns {String}
         */
        this.getStyleSet = function () {
            var styleSet = Math.floor((getStyleId() - 1) / 8);
            return 'ss' + ((styleSet < 4) ? 0 : styleSet);
        };

        /**
         * updates the interal chartStyleId by the unique ID
         * deletes all series colors if existing
         *
         * @param {String} colorSet
         */
        this.changeColorSet = function (colorSet) {
            sheetModel.createAndApplyOperations(function (generator) {

                var styleId = getStyleId() - 1;
                var csId = colorSet.replace('cs', '') | 0;
                var styleSet = (styleId / 8) | 0;

                styleId = styleSet * 8 + csId + 1;

                var position = self.getSheetPosition(),
                    attrs = createBackgroundColorOp(styleId);

                clearSeriesColor(generator, position);
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: self.getUndoAttributeSet(attrs) }, { undo: true });
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: attrs });

                return $.when();
            }, { storeSelection: true });
        };

        /**
         * updates the interal chartStyleId by the unique ID
         * does not touch the series colors
         *
         * @param {String} colorSet
         */
        this.changeStyleSet = function (styleSet) {
            sheetModel.createAndApplyOperations(function (generator) {

                var styleId = getStyleId() - 1;
                var ssId = styleSet.replace('ss', '') | 0;
                var colorSet = styleId % 8;

                styleId = ssId * 8 + colorSet + 1;

                var position = self.getSheetPosition(),
                    attrs = createBackgroundColorOp(styleId);

                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: self.getUndoAttributeSet(attrs) }, { undo: true });
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: attrs });

                return $.when();
            }, { storeSelection: true });
        };

        /**
         * changes the varycolors flag
         * deletes all series colors if existing, so the colors only orient on the patterns
         *
         * @param {String} colorSet
         */
        this.changeVaryColors = function (state) {
            sheetModel.createAndApplyOperations(function (generator) {

                var position = self.getSheetPosition();
                var attrs = { chart: { varyColors: state } };

                clearSeriesColor(generator, position);
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: self.getUndoAttributeSet(attrs) }, { undo: true });
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: attrs });

                return $.when();
            }, { storeSelection: true });
        };

        this.getSeriesCount = function () {
            return data.series.length;
        };

        this.getFirstPointsCount = function () {
            try {
                if (data.series.length) {
                    return data.series[0].dataPoints.length;
                }
            } catch (e) {
                Utils.warn('error while calling chartmodel.getFirstPointsCount()', e);
            }
            return 0;
        };

        /**
         * must be called for having correct behavior mit "varyColors" in combination with more than one series
         */
        this.firstInit = function () {
            updateChartType();
        };

        /**
         * must be called for having correct behavior mit "new CanvasJS.Chart" is called, because of a crazy bug inside there
         */
        this.resetData = function () {
            data.axisX._oldOptions = null;
            data.axisY._oldOptions = null;
        };

        this.hasDataPoints = function () {
            for (var i = 0; i < data.series.length; i++) {
                if (!_.isEmpty(data.series[i].dataPoints)) { return true; }
            }
            return false;
        };

        /**
         * normally the assigned axisId (x or y) will be returned
         * except chart is a bar-type, then x & y are interchanged
         *
         * @param {String} axisId
         *
         * @returns {String}
         */
        this.getAxisIdForDrawing = function (axisId) {
            var chartType = self.getMergedAttributeSet(true).chart.type;
            if (chartType.indexOf('bar') === 0) {
                if (axisId === 'x') { return 'y'; }
                if (axisId === 'y') { return 'x'; }
            }
            return axisId;
        };

        /**
         * transfers model-data to CanvasJs-data
         */
        this.updateRenderInfo = function () {
            changeDrawing();
            self.refresh();
        };

        /**
         * iterates all sources and calculates min and max positions,
         * direction of the series-values and the sheetindex
         *
         * @returns {Object}
         *  A descriptor with several properties of the data source:
         *  - {String} [warn]
         *      If existing, a specific warning code for invalid source data:
         *      - 'nodata': Source ranges are not available at all.
         *      - 'sheets': Source ranges are not on the same sheet.
         *      - 'directions': Source ranges are not in the same direction.
         * - {Range} range
         *      The bounding range of all source ranges in the targeted sheet.
         * - {Number} sheet
         *      Sheet index of all source ranges.
         * - {Number} axis
         *      Identifier of the main axis of the source ranges.
         */
        this.getDataSourceInfo = function () {

            var ranges = new RangeArray();
            var mainAxis = null;
            var sheet = null;
            var warn = null;
            var lastValues = null;

            _.each(data.series, function (dataSeries) {
                dataSeries.model.iterateTokenArrays(function (tokenArray, attrName) {
                    var rangeList = tokenArray.resolveRangeList({ resolveNames: true });
                    if (rangeList.empty()) { return; }
                    var range = rangeList.first();
                    if (sheet === null) {
                        sheet = range.sheet1;
                    }
                    if (!range.isSheet(sheet)) {
                        warn = 'sheets';
                    }
                    if (attrName === 'values') {
                        var axis = null;
                        var col = range.cols();
                        var row = range.rows();
                        if (col > row) {
                            axis = 1;
                        } else if (row > 1) {
                            axis = 0;
                        } else if (lastValues) {
                            if (lastValues.start[0] !== range.start[0]) {
                                axis = 0;
                            } else {
                                axis = 1;
                            }
                        }
                        if (_.isNumber(axis)) {
                            if (mainAxis === null) {
                                mainAxis = axis;
                            } else if (mainAxis !== axis) {
                                warn = 'directions';
                            }
                        }
                        lastValues = range;
                    }
                    // convert the 3D range (with sheet indexes) to a 2D range (without sheet indexes)
                    ranges.push(range.toRange());
                });
            });

            if (warn) { return { warn: warn }; }
            if (sheet === null || ranges.empty()) { return { warn: 'nodata' }; }
            return { range: ranges.boundary(), axis: mainAxis, sheet: sheet };
        };

        /**
         * checks if all dataseries have a sourcelink 'title'
         * which is the first row in a normal chart
         */
        this.isTitleLabel = function () {
            return sourceExistsComplete('title');
        };

        /**
         * checks if all dataseries have a sourcelink 'names'
         * which is the first column in a normal chart
         */
        this.isNamesLabel = function () {
            return sourceExistsComplete('names');
        };

        /**
         *
         * @returns {Boolean}
         */
        this.isMarkerOnly = function () {
            var marker = false;
            _.find(data.series, function (dataSeries) {
                var att = dataSeries.model.getMergedAttributeSet(true);
                if (att.line.type === 'none') {
                    marker = true;
                }
                return true;
            });
            return marker;
        };

        /**
         * Creates an image replacement from this model with the given
         * extent and image mimeType (either image/png or image/jpeg).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.mimeType]
         *   If specified, the given image mime type will
         *   be used for image creation. Allowed values ATM
         *   are image/png (default) and image/jpeg
         *  @param {Object} [options.size]
         *   If specified, the chart replacement will be
         *   rendered with the given size, otherwise, its
         *   size is retrieved from the model's drawing
         *   attributes.
         *
         * @returns {jQuery|Null}
         *  If successful, a floating jquery img node, containg the image data
         *  as src dataUrl and with the requested width and height set
         *  accordingly.
         */
        this.getReplacementNode = function (options) {

            var drawingRectPx = this.getRectangle();
            if (!drawingRectPx) { return null; }

            var jqImgNode = null,
                modelData = this.getModelData(),
                chartId = 'io-ox-documents-chart-frame' + _.uniqueId(),
                widthHeightAttr = '"width="' + drawingRectPx.width + 'px" height="' + drawingRectPx.height + 'px"',
                jqDrawingDiv = $('<div class="chartnode chartholder" id="' + chartId + '" ' + widthHeightAttr + '>').css({
                    position: 'absolute',
                    left: -drawingRectPx.width,
                    top: -drawingRectPx.height,
                    width: drawingRectPx.width,
                    height: drawingRectPx.height
                });

            $(':root').append(jqDrawingDiv);

            var chartRenderer = new CanvasJS.Chart(chartId, _.extend(modelData, { backgroundColor: modelData.cssBackgroundColor })),
                jqCanvas = $('#' + chartId + ' .canvasjs-chart-canvas'),
                canvas = jqCanvas.get(0);

            try {
                // render chart
                chartRenderer.render();

                var dataUrl = canvas.toDataURL(Utils.getStringOption(options, 'mimeType', 'image/png'));

                if (dataUrl) {
                    jqImgNode = $('<img>').attr({
                        width: drawingRectPx.width,
                        height: drawingRectPx.height,
                        src: dataUrl
                    });

                    jqCanvas = dataUrl = chartRenderer = null;
                }
            } catch (ex) {
                Utils.exception(ex, 'replacement chart image rendering error');
            }

            jqDrawingDiv.remove();
            jqDrawingDiv = null;

            return jqImgNode;
        };

        /**
         * @param {Boolean} state
         */
        this.setMarkerOnly = function (state) {
            sheetModel.createAndApplyOperations(function (generator) {

                var noneShape = { type: 'none' };
                var solidShape = { type: 'solid' };

                var attrs = {};
                if (state) {
                    attrs = { line: noneShape, fill: solidShape };
                } else {
                    attrs = { line: solidShape, fill: solidShape };
                }
                var position = self.getSheetPosition();

                _(data.series).each(function (dataSeries, index) {
                    var undoAttrs = dataSeries.model.getExplicitAttributeSet(false);
                    generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position, { series: index, attrs: undoAttrs }, { undo: true });
                    generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position, { series: index, attrs: attrs });
                });

                return $.when();
            }, { storeSelection: true });
        };

        this.getCloneData = function () {

            // a data object passed as hidden parameter to the constructor of the clone
            var cloneData = { series: [], axes: {}, legend: legendModel.getExplicitAttributeSet(), title: mainTitleModel.getExplicitAttributeSet() };

            _.each(axisModelMap, function (axisModel, axisId) {
                // workaround for Bug 46966
                if (axisId === 'z') { return;  }

                var aClone = {
                    axis: axisModel.getExplicitAttributeSet()
                };
                if (axisModel.getGrid()) {
                    aClone.grid = axisModel.getGrid().getExplicitAttributeSet();
                }
                if (axisModel.getTitle()) {
                    aClone.title = axisModel.getTitle().getExplicitAttributeSet();
                }
                cloneData.axes[axisId] = aClone;
            });

            _.each(data.series, function (dataSeries) {
                var series = {};

                dataSeries.model.iterateTokenArrays(function (tokenArray, attrName) {
                    series[attrName] = tokenArray.getFormula('op');
                });
                var nsp = dataSeries.model.getExplicitAttributeSet();
                nsp.series = series;

                cloneData.series.push(nsp);
            });

            cloneData.attrs = this.getExplicitAttributeSet();

            return cloneData;
        };

        this.getCanvasJSData = function () {
            var cvsData = {
                axis: {},
                series: []
            };
            _.each(data.data, function (series) {
                var points = [];
                _.each(series.dataPoints, function (dataPoint) {
                    points.push({
                        color: dataPoint.color,
                        label: dataPoint.label,
                        x: dataPoint.x,
                        y: dataPoint.y,
                        z: dataPoint.z
                    });
                });

                cvsData.series.push({
                    bevelEnabled:     series.bevelEnabled,
                    color:            series.color,
                    fillOpacity:      series.fillOpacity,
                    indexLabelFont:   series.indexLabelFont,
                    legendMarkerType: series.legendMarkerType,
                    marker:           series.marker,
                    markerColor:      series.markerColor,
                    name:             series.name,
                    showInLegend:     series.showInLegend,
                    type:             series.type,
                    points:           points
                });

            });
            _.each(axisModelMap, function (axisModel, axisId) {
                var dataAxis = data['axis' + axisId.toUpperCase()];
                cvsData.axis[axisId] = {
                    label:     dataAxis.label,
                    grid:      dataAxis.grid,
                    labelFont: dataAxis.labelFont,
                    line:      dataAxis.line,
                    tick:      dataAxis.tick,
                    titleFont: dataAxis.titleFont
                };
            });
            cvsData.legend = data.legend.font;
            cvsData.title = data.title.textFont;

            return cvsData;
        };

        // initialization -----------------------------------------------------

        colorHandler = new ManualColorHandler(this, data);
        chartFormatter = new ChartFormatter(docModel, this);

        this.on('change:attributes', changeAttributes);
        this.on('change:drawing', changeDrawing);

        // additional processing and event handling after the document has been imported
        this.waitForImportSuccess(function (alreadyImported) {

            // update the chart model after changing the cell contents
            this.listenTo(docModel, 'change:cells', changeCellsHandler);

            // refresh the chart after the visibility of columns or rows has changed
            this.listenTo(docModel, 'change:columns', function (event, sheet, interval, attributes, styleId, changeFlags) {
                if (changeFlags.visibility) { updateColRowVisibility(sheet, interval, true); }
            });
            this.listenTo(docModel, 'change:rows', function (event, sheet, interval, attributes, styleId, changeFlags) {
                if (changeFlags.visibility) { updateColRowVisibility(sheet, interval, false); }
            });

            if (!alreadyImported) {
                self.executeDelayed(function () {
                    //small hack for fastcalc-loading
                    if (sheetModel.getIndex() === docModel.getActiveSheet()) {
                        self.refresh();
                    }
                }, { infoString: 'waitForImportSucess', app: docModel.getApp() });
            }

        }, this);

        // clone private data passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[SheetChartModel.length];
            if (_.isObject(cloneData)) {
                _.each(cloneData.axes, function (axisData, axisId) {
                    var targetAxisModel = axisModelMap[axisId];
                    targetAxisModel.setAttributes(axisData.axis);
                    if (!_.isEmpty(axisData.grid)) {
                        targetAxisModel.getGrid().setAttributes(axisData.grid);
                    }
                    if (!_.isEmpty(axisData.title)) {
                        targetAxisModel.getTitle().setAttributes(axisData.title);
                    }
                });
                _.each(cloneData.series, function (attrs, index) {
                    insertDataSeries(index, attrs);
                });
                mainTitleModel.setAttributes(cloneData.title);
                legendModel.setAttributes(cloneData.legend);
            }
        }(arguments));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(axisModelMap, 'destroy');
            mainTitleModel.destroy();
            legendModel.destroy();
            chartFormatter.destroy();

            for (var key in data.series) {
                var ds = data.series[key];
                ds.model.destroy();
                //must delete, otherwise cansjs has still reference on this
                delete ds.model;
            }

            self = docModel = sheetModel = mainTitleModel = legendModel = data = axisModelMap = colorHandler = chartFormatter = app = null;
        });

    } // class SheetChartModel

    // exports ================================================================

    // derive this class from class DrawingModel
    return DrawingModel.extend({ constructor: SheetChartModel });

});
