/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/celloperationsbuilder', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, Operations, SheetUtils) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var AddressArray = SheetUtils.AddressArray;
    var RangeArray = SheetUtils.RangeArray;

    // private global functions ===============================================

    /**
     * Returns whether the passed cell range addresses are equal, or are both
     * falsy.
     *
     * @param {Range|Null} range1
     *  The first cell range address.
     *
     * @param {Range|Null} range2
     *  The second cell range address.
     *
     * @returns {Boolean}
     *  Whether the passed cell range addresses are equal, or are both falsy.
     */
    function equalRanges(range1, range2) {
        return (!range1 && !range2) || (range1 && range2 && range1.equals(range2));
    }

    // class JSONBuffer =======================================================

    /**
     * A two-dimensional buffer of JSON cell content objects, as used in cell
     * operations.
     *
     * @constructor
     */
    function JSONBuffer(builder) {

        this._builder = builder;
        this._matrix = [];

        // create an initial empty row
        this.addRow();

    } // class JSONBuffer

    // public methods ---------------------------------------------------------

    /**
     * Appends a new empty row to this cell data buffer.
     *
     * @param {Number} [count=1]
     *  The initial repetition count for the new row.
     */
    JSONBuffer.prototype.addRow = function (count) {
        this._matrix.push({ c: [], r: count || 1 });
    };

    /**
     * Returns whether the passed JSON data objects for a cell operation have
     * equal properties (ignoring the 'r' property for repetition).
     *
     * @param {Object} jsonData1
     *  The first JSON cell data object, as used in cell operations.
     *
     * @param {Object} jsonData2
     *  The second JSON cell data object, as used in cell operations.
     *
     * @returns {Boolean}
     *  Whether the passed JSON data objects for a cell operation have equal
     *  properties (ignoring the 'r' property for repetition).
     */
    JSONBuffer.prototype.isEqualJSON = function (jsonData1, jsonData2) {
        return (jsonData1.u === jsonData2.u) &&
            (jsonData1.v === jsonData2.v) &&
            (jsonData1.e === jsonData2.e) &&
            (jsonData1.f === jsonData2.f) &&
            (jsonData1.si === jsonData2.si) &&
            (jsonData1.sr === jsonData2.sr) &&
            (jsonData1.mr === jsonData2.mr) &&
            this._builder.areEqualStyleIds(jsonData1.s, jsonData2.s);
    };

    /**
     * Appends the passed JSON cell data object to the end of this cell data
     * buffer. Tries to compress the buffer by updating the repetition count of
     * the last existing cell entry if possible.
     *
     * @param {Object} jsonData
     *  The JSON cell data object to be added to this buffer.
     *
     * @param {Number} [count=1]
     *  The repetition count for the passed JSON cell data.
     */
    JSONBuffer.prototype.addCell = function (jsonData, count) {
        var lastRow = _.last(this._matrix);
        var lastData = _.last(lastRow.c);
        count = count || 1;
        if (lastData && this.isEqualJSON(lastData, jsonData)) {
            lastData.r += count;
        } else {
            jsonData.r = count;
            lastRow.c.push(jsonData);
        }
    };

    /**
     * Generates the 'changeCells' operation for this cell data buffer.
     *
     * @param {SheetOperationsGenerator} generator
     *  The operations generator to be filled with the operation.
     *
     * @param {Address} address
     *  The address of the top-left cell of this data buffer.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      If set to true, an undo operation will be created.
     *  @param {Boolean} [options.filter=false]
     *      If set to true, an operation for export filters will be created
     *      (remote editor clients will ignore the operation).
     */
    JSONBuffer.prototype.generateOperation = function (generator, address, options) {

        // clone the start address to be able to modify it below
        address = address.clone();

        // number of leading skipped cells in all rows
        var leadingSkip = Number.MAX_VALUE;
        // the JSON keys of all cell data arrays in the matrix
        var cellKeys = [];

        // remove trailing garbage from the rows, remove useless repetition properties from cell entries
        this._matrix.forEach(function (currRow) {

            // the cell data array
            var cells = currRow.c;

            // remove trailing skip cell entries
            var lastData = _.last(cells);
            if (lastData && CellOperationsBuilder.isEmptyJSON(lastData)) {
                cells.pop();
            }

            // collect number of leading skipped cells (ignore empty rows completely)
            var firstData = cells[0];
            if (firstData) {
                leadingSkip = CellOperationsBuilder.isEmptyJSON(firstData) ? Math.min(leadingSkip, firstData.r) : 0;
            }

            // remove empty cell arrays completely, collect unique keys for cell arrays
            if (cells.length === 0) {
                delete currRow.c;
                cellKeys.push(null);
            } else {
                cellKeys.push(Utils.stringifyJSON(cells));
            }

        }, this);

        // remove leading skipped rows from the matrix
        for (var firstRow = this._matrix[0]; firstRow && !firstRow.c; firstRow = this._matrix[0]) {
            address[1] += firstRow.r;
            this._matrix.shift();
            cellKeys.shift();
        }

        // maps unique keys of cell arrays to array indexes of row data objects
        var cellKeyMap = {};

        // process all rows of the matrix
        this._matrix.forEach(function (currRow, index) {

            // collapse rows with the same cell entries (repeat as long as the next row can be merged
            // with the current row; the next row entry will always be removed immediately from the array)
            var cellKey = cellKeys[index];
            while (cellKey === cellKeys[index + 1]) {
                currRow.r += this._matrix[index + 1].r;
                this._matrix.splice(index + 1, 1);
                cellKeys.splice(index + 1, 1);
            }

            // remove r:1 property from the row entry
            if (currRow.r === 1) { delete currRow.r; }

            // add array index of an existing cell array in a preceding row data entry
            if (cellKey) {
                var oldIndex = cellKeyMap[cellKey];
                if (_.isNumber(oldIndex)) {
                    currRow.c = oldIndex;
                    return;
                }
                cellKeyMap[cellKey] = index;
            }

            // further processing of the cell array
            var cells = currRow.c;
            if (!cells) { return; }

            // remove leading skipped cells from the row
            if (leadingSkip > 0) {
                var firstData = cells[0];
                if (firstData.r === leadingSkip) {
                    cells.shift();
                } else {
                    firstData.r -= leadingSkip;
                }
            }

            // remove r:1 properties from all cell entries
            cells.forEach(function (cellData) {
                if (cellData.r === 1) { delete cellData.r; }
            });

        }, this);

        // update start address according to leading skipped cells
        address[0] += leadingSkip;

        // remove trailing skipped rows
        var lastRow = _.last(this._matrix);
        if (lastRow && !('c' in lastRow)) {
            this._matrix.pop();
        }

        // nothing to do without any changed cells
        if (this._matrix.length > 0) {
            var properties = { contents: this._matrix };
            if (Utils.getBooleanOption(options, 'filter', false)) { properties.scope = 'filter'; }
            generator.generateCellOperation(Operations.CHANGE_CELLS, address, properties, options);
        }
    };

    /**
     * Releases all properties of this instance.
     */
    JSONBuffer.prototype.destroy = function () {
        this._builder = this._matrix = null;
    };

    // class CellOperationsBuilder ============================================

    /**
     * A builder to generate cell operations and undo operations for multiple
     * cells in a cell collection.
     *
     * @constructor
     *
     * @param {SheetModel} sheetModel
     *  The model of the sheet this instance is associated to.
     *
     * @param {SheetOperationsGenerator} generator
     *  The operations generator that will be filled with the new document
     *  operations.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Address} [initOptions.startAddress=Address.A1]
     *      The address of the first cell to generate operations for, if known.
     *      By default, address A1 will be used to be sure to include all cells
     *      in the generated operations.
     *  @param {Address} [initOptions.skipShared=false]
     *      Whether to skip updating the shared formulas after generating the
     *      cell operations. Should be used by code that handles updating the
     *      shared formulas internally.
     */
    function CellOperationsBuilder(sheetModel, generator, initOptions) {

        // the collection of cell auto-styles of the document
        this._autoStyles = sheetModel.getDocModel().getCellAutoStyles();

        // the configuration for the formula grammar used in operations
        this._formulaGrammar = sheetModel.getDocModel().getFormulaGrammar('op');

        // the formula parser (used to generate range strings)
        this._formulaParser = sheetModel.getDocModel().getFormulaParser();

        // the cell collection this builder works on
        this._cellCollection = sheetModel.getCellCollection();

        // the operations generator
        this._generator = generator;

        // the JSON data for the cell operations
        this._cellBuffer = null;

        // the JSON data for the undo operations
        this._undoBuffer = null;

        // the cell address of the start cell for this builder
        this._startAddress = null;

        // the cell address of the first cell not yet manipulated by this builder
        this._nextAddress = null;

        // the addresses of all cells changed by this builder
        this._changedAddresses = null;

        // whether to update all shared formulas in the cell collection
        this._refreshShared = true;

        // whether shared formulas need to be updated after changing formula properties
        this._dirtyShared = false;

        // initialization -----------------------------------------------------

        this._initialize(initOptions);

    } // class CellOperationsBuilder

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the passed JSON data object for a cell operation will
     * not change a cell at all.
     *
     * @param {Object} jsonData
     *  The JSON cell data object, as used in cell operations.
     *
     * @returns {Boolean}
     *  Whether the JSON data object for a cell operation will not change a
     *  cell at all.
     */
    CellOperationsBuilder.isEmptyJSON = function (jsonData) {
        return !('u' in jsonData) && !('v' in jsonData) && !('e' in jsonData) && !('s' in jsonData) &&
            !('f' in jsonData) && !('si' in jsonData) && !('sr' in jsonData) && !('mr' in jsonData);
    };

    // private methods --------------------------------------------------------

    /**
     * Initializes the properties of this instance for a new processing cycle.
     *
     * @param {Object} [options]
     *  All optional parameters supported by the constructor of this class.
     */
    CellOperationsBuilder.prototype._initialize = function (options) {
        this._cellBuffer = new JSONBuffer(this);
        this._undoBuffer = new JSONBuffer(this);
        this._startAddress = Utils.getOption(options, 'startAddress', Address.A1).clone();
        this._nextAddress = this._startAddress.clone();
        this._changedAddresses = new AddressArray();
        this._refreshShared = !Utils.getBooleanOption(options, 'skipShared', false);
        this._dirtyShared = false;
    };

    /**
     * Assigns an error code as string to the 'e' operation property, otherwise
     * the passed value to the 'v' property of the passed JSON cell object, to
     * be able to distinguish plain strings looking like error codes.
     *
     * @param {Object} jsonData
     *  The JSON cell data object to be extended with a value property.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The cell value to be inserted into the passed JSON object. An error
     *  code will be inserted as 'e' string property, all other values will be
     *  inserted as 'v' property.
     */
    CellOperationsBuilder.prototype._putValue = function (jsonData, value) {
        if (value instanceof ErrorCode) {
            jsonData.e = this._formulaGrammar.getErrorName(value);
        } else {
            jsonData.v = value;
        }
    };

    /**
     * Converts the passed cell range address to its string representation as
     * used in document operations.
     *
     * @param {Range} range
     *  The cell range address to be converted.
     *
     * @returns {String}
     *  The string representation of the passed cell range address.
     */
    CellOperationsBuilder.prototype._formatRange = function (range) {
        return this._formulaParser.formatRange('op', range);
    };

    /**
     * Appends the JSON data for a single cell to the data buffers.
     */
    CellOperationsBuilder.prototype._createCell = function (contents) {

        // the address of teh cell to be modified
        var address = this._nextAddress;
        // the model of the cell to be manipulated
        var cellModel = this._cellCollection.getCellModel(address);
        // whether to delete the cell model explicitly
        var deleteCell = contents.u === true;
        // the resulting JSON contents object
        var cellJSON = {};
        // the resulting JSON contents object for the undo operation
        var undoJSON = {};

        // the original value or formula result of the cell
        var oldValue = cellModel ? cellModel.v : null;
        // the original formula expression of the cell
        var oldFormula = cellModel ? cellModel.f : null;
        // the original index of a shared formula
        var oldSharedIndex = cellModel ? cellModel.si : null;
        // the original bounding range of a shared formula
        var oldSharedRange = cellModel ? cellModel.sr : null;
        // the original bounding range of a matrix formula
        var oldMatrixRange = cellModel ? cellModel.mr : null;

        // create the properties in the JSON data that will really change the cell model
        if (!deleteCell) {

            // copy the cell value to the JSON data (unless it does not change)
            if (('v' in contents) && (contents.v !== oldValue)) {
                this._putValue(cellJSON, contents.v);
                this._putValue(undoJSON, oldValue);
            }

            // copy the cell formula to the JSON data (unless it does not change)
            if (('f' in contents) && (contents.f !== oldFormula)) {
                cellJSON.f = contents.f;
                undoJSON.f = oldFormula;
            }

            // copy the shared formula index to the JSON data (unless it does not change)
            if (('si' in contents) && (contents.si !== oldSharedIndex)) {
                cellJSON.si = contents.si;
                undoJSON.si = oldSharedIndex;
            }

            // copy the shared formula range to the JSON data (unless it does not change)
            if (('sr' in contents) && !equalRanges(contents.sr, oldSharedRange)) {
                cellJSON.sr = contents.sr ? this._formatRange(contents.sr) : null;
                undoJSON.sr = oldSharedRange ? this._formatRange(oldSharedRange) : null;
            }

            // copy the matrix formula range to the JSON data (unless it does not change)
            if (('mr' in contents) && !equalRanges(contents.mr, oldMatrixRange)) {
                cellJSON.mr = contents.mr ? this._formatRange(contents.mr) : null;
                undoJSON.mr = oldMatrixRange ? this._formatRange(oldMatrixRange) : null;
            }

            // add more cell properties to detach the cell from a shared formula or matrix formula
            if (this._refreshShared && ('f' in contents)) {
                // reset the shared index if it has not been changed explicitly
                if ((oldSharedIndex !== null) && !('si' in contents)) {
                    cellJSON.si = null;
                    undoJSON.si = oldSharedIndex;
                }
                // reset the bounding range if the current cell is the anchor cell of the shared formula
                if (oldSharedRange && !('sr' in contents)) {
                    cellJSON.sr = null;
                    undoJSON.sr = this._formatRange(oldSharedRange);
                }
                // reset the bounding range if the current cell is the anchor cell of the matrix formula
                if (oldMatrixRange && !('mr' in contents)) {
                    cellJSON.mr = null;
                    undoJSON.mr = this._formatRange(oldMatrixRange);
                }
            }

            // bounding ranges of shared formulas need to be updated when changing the shared index
            if ('si' in cellJSON) { this._dirtyShared = true; }

            // the default column/row auto-style identifier for undefined cells, with current column/row defaults
            var defStyleId = this._cellCollection.getDefaultStyleId(address);
            // range mode 'row:merge': ignore the row style that has been set during this operation cycle when creating the cell auto-style
            var ignoreRowStyle = contents.rangeMode === 'row:merge';
            // the identifier of the base auto-style used to create the new cell auto-style
            var baseStyleId = cellModel ? cellModel.s : ignoreRowStyle ? this._cellCollection.getDefaultStyleId(address, true) : defStyleId;
            // the resulting auto-style identifier (try to get a cached identifier from a previous call)
            var newStyleId = this._generator.generateAutoStyle(baseStyleId, contents);

            // When changing an existing cell model, add the auto-style if it changes, also if set from or to the default style.
            if (cellModel && !this.areEqualStyleIds(cellModel.s, newStyleId)) {
                cellJSON.s = newStyleId;
                undoJSON.s = cellModel.s;
            }

            // When the cell does not exist, the auto-style must be set if it differs from the default column/row
            // style (also set the default style to override an existing row style); or when creating the cell due
            // to a new value or formula, and the auto-style does not remain the default auto-style.
            if (!cellModel && (!this.areEqualStyleIds(defStyleId, newStyleId) || (!_.isEmpty(cellJSON) && !this.isDefaultStyleId(newStyleId)))) {
                cellJSON.s = newStyleId;
            }

            // Check whether to undefine (delete) an existing cell model implicitly: It must be (or become) blank, it must not
            // contain a formula, and its auto-style must be (or become) equal to the default auto-style of the row or column.
            deleteCell = cellModel &&
                (('v' in contents) ? (contents.v === null) : (oldValue === null)) &&
                (('f' in contents) ? (contents.f === null) : (oldFormula === null)) &&
                (('si' in cellJSON) ? (cellJSON.si === null) : (oldSharedIndex === null)) &&
                (('sr' in cellJSON) ? (cellJSON.sr === null) : (oldSharedRange === null)) &&
                (('mr' in cellJSON) ? (cellJSON.mr === null) : (oldMatrixRange === null)) &&
                this.areEqualStyleIds(defStyleId, newStyleId);

            // the cell simply needs to be deleted on undo, if it does not exist yet and will be created
            if (!cellModel && !_.isEmpty(cellJSON)) {
                undoJSON = { u: true };
            }
        }

        // create simple contents objects when an existing cell model will be deleted
        // (the flag 'deleteCell' may have been set in the previous if-block)
        if (deleteCell && cellModel) {

            // the property 'u' marks a cell to be deleted (undefined)
            cellJSON = { u: true };

            // restore the cell value, formula, and auto-style
            undoJSON = {};
            if (oldValue !== null) { this._putValue(undoJSON, oldValue); }
            if (oldFormula !== null) { undoJSON.f = oldFormula; }
            if (oldSharedIndex !== null) { undoJSON.si = oldSharedIndex; }
            if (oldSharedRange) { undoJSON.sr = this._formatRange(oldSharedRange); }
            if (oldMatrixRange) { undoJSON.mr = this._formatRange(oldMatrixRange); }
            if (!this.isDefaultStyleId(cellModel.s)) { undoJSON.s = cellModel.s; }

            // update shared formula after deleting one of its cells
            if (oldSharedIndex !== null) { this._dirtyShared = true; }
        }

        // whether the cell will be changed at all (store this information before calling the method
        // JSONBuffer.addCell(), which changes/extends the JSON object in-place)
        var changed = !_.isEmpty(cellJSON);

        // insert the JSON objects into the data buffers, update addresses
        this._cellBuffer.addCell(cellJSON);
        this._undoBuffer.addCell(undoJSON);

        return changed;
    };

    /**
     * Generates the document operation, and the undo operation, for all cells
     * collected by this builder instance.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options that are supported by the
     *  public method CellOperationsBuilder.finalizeOperations().
     *
     * @returns {AddressArray}
     *  The addresses of all changed cells.
     */
    CellOperationsBuilder.prototype._finalize = function (options) {

        // whether to generate the operation for the export filters only
        var filter = Utils.getBooleanOption(options, 'filter', false);

        // generate the undo operations
        if (!filter && Utils.getBooleanOption(options, 'createUndo', true)) {
            this._undoBuffer.generateOperation(this._generator, this._startAddress, { undo: true });
        }

        // generate operations for the collected cell data
        if (Utils.getBooleanOption(options, 'createCells', true)) {
            this._cellBuffer.generateOperation(this._generator, this._startAddress, { filter: filter });
        }

        // destroy the JSON data buffers
        this._cellBuffer.destroy();
        this._undoBuffer.destroy();
        this._cellBuffer = this._undoBuffer = null;

        // return the addresses of all changed cells, for convenience
        return this._changedAddresses;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the passed auto-style identifier is considered to be the
     * default auto-style. Convenience shortcut for the same method of the
     * auto-style collection.
     */
    CellOperationsBuilder.prototype.isDefaultStyleId = function (styleId) {
        return this._autoStyles.isDefaultStyleId(styleId);
    };

    /**
     * Returns whether the passed auto-style identifiers are considered to be
     * equal. Convenience shortcut for the same method of the auto-style
     * collection.
     */
    CellOperationsBuilder.prototype.areEqualStyleIds = function (styleId1, styleId2) {
        return this._autoStyles.areEqualStyleIds(styleId1, styleId2);
    };

    /**
     * Processes the passed cell properties, and stores all data needed to be
     * inserted into the resulting cell operations.
     *
     * @param {Address} address
     *  The address of the cell to be processed. The addresses passed to
     *  repeated invocations of this method MUST be increasing!
     *
     * @param {Object} contents
     *  The cell properties to be applied to the specified cell.
     *
     * @param {Object} count
     *  The repetition count for the cell data (consecutive cells in the
     *  current row).
     */
    CellOperationsBuilder.prototype.createCells = function (address, contents, count, options) {

        // jump to a new row, if the row index changes
        var incRows = address[1] - this._nextAddress[1];
        if (incRows > 0) {
            this._nextAddress[0] = this._startAddress[0]; // back to start column
            this._nextAddress[1] = address[1];

            // create separate entries for the skipped empty rows
            if (incRows > 1) {
                this._cellBuffer.addRow(incRows - 1);
                this._undoBuffer.addRow(incRows - 1);
            }
            this._cellBuffer.addRow();
            this._undoBuffer.addRow();
        }

        // skip cells in the current row
        var incCols = address[0] - this._nextAddress[0];
        if (incCols > 0) {
            this._cellBuffer.addCell({}, incCols);
            this._undoBuffer.addCell({}, incCols);
            this._nextAddress[0] = address[0];
        }

        // generate the JSON data for the passed cell properties repeatedly
        for (var index = 0; index < count; index += 1) {
            if (this._createCell(contents, options)) {
                this._changedAddresses.push(this._nextAddress.clone());
            }
            this._nextAddress[0] += 1;
        }
    };

    /**
     * Generates the document operations, and the undo operations, for all
     * cells collected by this builder instance, and invalidates the entire
     * builder instance.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.createCells=true]
     *      Whether to create the document operations for the changed cells.
     *  @param {Boolean} [options.createUndo=true]
     *      Whether to create the undo operations for the changed cells.
     *  @param {Boolean} [options.filter=false]
     *      If set to true, operations for export filters will be created
     *      (remote editor clients will ignore the operations).
     *
     * @returns {RangeArray}
     *  The addresses of all cell ranges containing changed cells.
     */
    CellOperationsBuilder.prototype.finalizeOperations = SheetUtils.profileMethod('CellOperationsBuilder.finalizeOperations()', function (options) {

        // create the operations for all cells collected by this instance
        var changedAddresses = this._finalize(options);

        // update the bounding ranges of all affected shared formulas
        if (this._refreshShared && this._dirtyShared) {

            // calculate all cell properties needed to update the cells of the shared formulas
            var contentsArray = this._cellCollection.generateUpdateSharedFormulaContents();

            // generate the document operations for all affected anchor cells
            if (contentsArray.length > 0) {
                this._initialize({ skipShared: true });
                contentsArray.forEach(function (entry) {
                    this.createCells(entry.address, entry.contents, 1);
                }, this);
                changedAddresses.append(this._finalize(options));
            }
        }

        // return the addresses of all changed ranges
        return RangeArray.mergeAddresses(changedAddresses);
    });

    // exports ================================================================

    return CellOperationsBuilder;

});
