/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/presentation/view/dialog/slidebackground', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/editframework/view/editcontrols',
    'io.ox/office/textframework/view/controls',
    'io.ox/office/textframework/utils/textutils',
    'gettext!io.ox/office/presentation/main',
    'less!io.ox/office/presentation/view/dialog/presentationdialogs'
], function (Utils, KeyCodes, Forms, Dialogs, Tracking, Parser, EditControls, Controls, TextUtils, gt) {

    'use strict';

    // Global constants and variables
    var slideBackgroundTitle = /*#. Change slide background from dialog menu*/ gt('Background'),
        dialogWidth = Utils.SMALL_DEVICE ? 300 : 360,
        // container node for first column (category drop-down, fields type list)
        $controlArea = null,
        modifyContainer = null,
        hideBgCheckbox = null,
        noBgRadioBtn = null,
        colorRadioBtn = null,
        imageRadioBtn = null;

    /**
     * A validator for percentage fields that restricts the allowed values to
     * integer numbers and optionally % sign. Entry must be integer value between 0 and 100.
     *
     * @constructor
     *
     * @extends SpinField.NumberValidator
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options that are supported by the
     *  base class NumberValidator.
     */
    var PercentValidator = EditControls.SpinField.NumberValidator.extend({ constructor: function (initOptions) {

        var // the regular expression for the percentage field
            regex = null;

        // base constructor ---------------------------------------------------

        EditControls.SpinField.NumberValidator.call(this, initOptions);

        // methods ------------------------------------------------------------

        /**
         * Converts the passed value to a display string with percentage in hundreds.
         *
         * @param {Number} value
         *  The percentage in number.
         *
         * @returns {String}
         *  The display string of a percentage with % sign.
         */
        this.valueToText = function (value) {
            if (_.isFinite(value)) {
                var precise = Utils.round(value, this.getPrecision());

                return Parser.numberToString(precise) + ' %';
            }
            return '';
        };

        /**
         * Converts the passed display string of a percentage to number value.
         *
         * @param {String} text
         *  The display string of a percentage with % sign.
         *
         * @returns {Number|Null}
         *  The percentage number in hundreds.
         */
        this.textToValue = function (text) {

            var matches = regex.exec(text) || [],
                value = matches[1] || '';

            value = Parser.stringToNumber(value);
            if (_.isNumber(value) && _.isFinite(value)) {
                return this.restrictValue(value);
            }

            return null;
        };

        /**
         * The validation functions that controls the input into the unit field
         * corresponding to the defined regular expression.
         *
         * @returns {Boolean}
         *  Whether the input into the unit field matches the defined regular
         *  expression.
         */
        this.validate = function (text) {
            return regex.test(text);
        };

        // initialization -----------------------------------------------------

        // the regular expression for the percentage field
        regex = new RegExp('^(\\d{1,2}|100)\\s*%?\\s*$');

    } }); // class PercentValidator

    // class SlideBackgroundDialog ===============================================

    /**
     * The slide background dialog.
     * Provides different types and options for slide backgrounds for Presentation document.
     *
     *
     * The changes will be applied to the current document after clicking the 'Ok' button.
     *
     * The dialog itself is shown by calling this.execute. The caller can react on successful
     * finalization of the dialog within an appropriate done-Handler, chained to the
     * this.execute function call.
     *
     * @constructor
     *
     * @extends Dialogs.ModalDialog
     *
     * @param {PresentationView} view
     *  The view instance containing this editor instance.
     * @param {Object} [dialogOptions]
     *  Options for filling the dialog
     * @param {Object} [dialogOptions.color]
     *  If background is already set as type solid color.
     * @param {String} [dialogOptions.type]
     *  Type of background.
     */
    function SlideBackgroundDialog(view, dialogOptions) {
        var
            self = this,
            model = view.getDocModel(),
            slideId = model.getActiveSlideId(),
            $slide = model.getSlideById(slideId),
            colorPicker = new Controls.FillColorPicker(view, { tooltip: null, title: null }),
            imagePicker = new Controls.CompactImagePicker(view, { label: gt('Choose') }),
            transpArea, transpContainer, transpSlider, transpSliderWidth, transpSliderOffset, transpRailOverlay, transpPointer, transpSpinner,
            colorArea, imageArea, $tempPreviewSlide,
            slidingActive = false,
            startOffset = 0,
            applyallBtn, headerNode;

        // workaround for Bug 49523
        model.trigger('change:activethemetarget', model.getPreviewTargetChain());

        // constants for pointer events
        var START_POINTER_EVENTS = Utils.IOS ? 'touchstart' : 'mousedown touchstart';
        var MOVE_POINTER_EVENTS = Utils.IOS ? 'touchmove' : 'mousemove touchmove';
        var END_POINTER_EVENTS = Utils.IOS ? 'touchend' : 'mouseup touchend';

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, { title: slideBackgroundTitle, width: dialogWidth });

        // private methods ----------------------------------------------------

        /**
         * Spinner wrapper.
         * @param {Number} min Value
         * @param {Number} max Value
         */
        function PercentSpin(min, max) {
            var spinnerValue = 0,
                spinnerNode;

            EditControls.SpinField.call(this, {
                width: Utils.IOS ? 75 : 60,
                min: min,
                max: max,
                validator: new PercentValidator({ min: min, max: max, percentageUnit: true }),
                keyboard: 'text', // #31765: force keyboard type to 'text', Android does not like characters in numeric text fields
                percentageUnit: true,
                smallerVersion: { hide: true }
            });

            this.setValue = function (value) {
                var attrs = $slide.data('previewAttrs');

                spinnerValue = value;
                this.setFieldValue(spinnerValue);
                // update slider
                updateSliderValue(spinnerValue);
                setAlphaPreview(attrs, spinnerValue);

                this.getInputNode().trigger('input');
            };

            this.getValue = function () {
                return spinnerValue;
            };

            this.triggerChange = function (value) {

                if (!_.isNull(value)) {
                    this.setValue(value);
                } else {
                    this.setValue(spinnerValue);
                }
            };

            /**
             * Returns the node of the spinner
             * @returns {jQuery} the spinner node.
             */
            this.node = function () {
                if (!spinnerNode) {
                    spinnerNode = $(this.getNode());
                }
                return spinnerNode;
            };
        }

        /**
         * Helper function for enabling or disabling the specified DOM elements.
         *
         * @param {jQuery} nodes
         *  The DOM elements to be manipulated. If this object is a jQuery
         *  collection, uses all nodes it contains.
         *
         * @param {Boolean} state
         *  Whether to enable (true) or disable (false) the DOM elements.
         */
        function enableNodes(nodes, state) {
            nodes.toggleClass(Forms.DISABLED_CLASS, !state);
            if (state) {
                nodes.removeAttr('aria-disabled');
                nodes.prop('tabindex', 0);
            } else {
                nodes.attr('aria-disabled', true);
                nodes.prop('tabindex', -1);
            }
        }

        /**
         * Generates and returns HTML markup string for color slider.
         *
         * @return {String}
         */
        function generateColorSlider() {
            var markup = '<div class="slider-container" tabindex="0"><div class="bg-color-slider"><div class="slider-rail rail-overlay"></div><div class="slider-rail"></div><div class="slider-pointer"></div></div></div>';

            return markup;
        }

        /**
         * Heleper to get client x coordinate from original event of jQuery's touch event
         * @param  {jQuery Event} event
         * @return {Number} clientX value
         */
        function getTouchClientX(event) {
            var originalEvent = event.originalEvent;
            var touch = originalEvent && originalEvent.touches && originalEvent.touches[0];
            return touch && touch.clientX;
        }

        /**
         * Set alpha with corresponding attributes to slide background as preview.
         *
         * @param {Object} attrs Fill attributes to set to slide background as preview
         * @param {Number} value Value of the transparency [0, 100]
         */
        function setAlphaPreview(attrs, value) {
            var inheritFromParent = false;
            var imageLoad = false;

            if (attrs && attrs.fill && (attrs.fill.type === null || attrs.fill.type === 'none')) { // inherit fill from parent
                var parentId = model.getParentSlideId(slideId);
                var parentFillAttrs = model.getSlideAttributesByFamily(parentId, 'fill');
                if (!parentFillAttrs || parentFillAttrs.type === 'none' || parentFillAttrs.type === null) {
                    parentId = model.getParentSlideId(parentId);
                    parentFillAttrs = model.getSlideAttributesByFamily(parentId, 'fill');
                }
                attrs = { fill: _.copy(parentFillAttrs, true) };
                if (parentFillAttrs && parentFillAttrs.color && !parentFillAttrs.bitmap) {
                    colorPicker.setValue(parentFillAttrs.color);
                }
                inheritFromParent = true;
                imageLoad = true;
            }
            if (attrs && attrs.fill && (attrs.fill.type === 'solid' || attrs.fill.type === 'scheme')) {
                attrs.fill.color = attrs.fill.color || {};
                attrs.fill.color.transformations = attrs.fill.color.transformations || [];
                var alphaT = _.findWhere(attrs.fill.color.transformations, { type: 'alpha' });
                if (alphaT) {
                    alphaT.value = _.isNumber(value) ? (100 - value) * 1000 : 0;
                } else {
                    attrs.fill.color.transformations.push({ type: 'alpha', value: (100 - value) * 1000 });
                }

                model.getSlideStyles().updateBackground($tempPreviewSlide, attrs, slideId, null, { preview: true });
                if (inheritFromParent) {
                    view.handleSlideBackgroundVisibility(slideId, attrs.fill);
                }
                // store values for apply
                $slide.data('previewAttrs', attrs);
            }
            if (attrs && attrs.fill && attrs.fill.type === 'bitmap' && attrs.fill.bitmap) {
                attrs.fill.bitmap.transparency = _.isNumber(value) ? value / 100 : 0;

                model.getSlideStyles().updateBackground($tempPreviewSlide, attrs, slideId, null, { imageLoad: imageLoad, preview: true });
                if (inheritFromParent) {
                    view.handleSlideBackgroundVisibility(slideId, attrs.fill);
                }
                // store values for apply
                $slide.data('previewAttrs', attrs);
            }
        }

        function updateSliderValue(value) {
            transpPointer.css('left', value + '%').data('percentage', value);
            transpRailOverlay.css('width', value + '%');
        }

        function updateSliderSpinner(value) {
            updateSliderValue(value);
            transpSpinner.setValue(value);
        }

        /**
         * Set value to the slider, corresponding percentage spinner, and make preview of transparency.
         *
         * @param  {Number} value
         */
        function updateTransparency(value) {
            var normalizedValue = Utils.minMax(Utils.round(value, 1), 0, 100);
            var attrs = $slide.data('previewAttrs');

            updateSliderSpinner(normalizedValue);
            setAlphaPreview(attrs, normalizedValue);
        }

        /**
         * Initializes mouse/touch tracking to move this menu around.
         */
        function trackingStartHandler(event) {
            startOffset = self.getBody().parent().offset();
            event.preventDefault();
        }

        /**
         * Moves the menu while mouse/touch tracking is active.
         */
        function trackingMoveHandler(event) {
            //console.warn(startOffset.left, event.offsetX, startOffset.top, event.offsetY);
            var appWindowNode = $(window);
            var dialogNode = self.getBody().parent();
            var marginBoundary = 5;
            var rightBoundary = appWindowNode.width() - dialogNode.width() - marginBoundary;
            var bottomBoundary = appWindowNode.height() - (Utils.SMALL_DEVICE ? self.getHeader().outerHeight() : dialogNode.height()) - marginBoundary;

            var leftValue = Utils.minMax(startOffset.left + event.offsetX, marginBoundary, rightBoundary);
            var topValue = Utils.minMax(startOffset.top + event.offsetY, marginBoundary, bottomBoundary);
            self.getBody().parent().css({ left: leftValue, top: topValue, margin: 0 });

            // set new value for transparency slider
            transpSliderOffset = transpSlider.offset().left;
            event.preventDefault();
        }

        /**
         * Initialize controls of the dialog.
         */
        function initControls() {
            var dialogOptionsType = (_.isObject(dialogOptions) && dialogOptions.type) || null;
            var isImageRadioEnabled = dialogOptionsType === 'bitmap';
            var isColorRadioEnabled = dialogOptionsType === 'solid' || dialogOptionsType === 'scheme';
            var transpValue = 0;

            var noBgRadio = Forms.createButtonMarkup({ attributes: { class: 'no-bg-radio', role: 'radio' }, label: gt('No background') });
            var colorRadio = Forms.createButtonMarkup({ attributes: { class: 'color-radio', role: 'radio' }, label: gt('Color:') });
            var imageRadio = Forms.createButtonMarkup({ attributes: { class: 'image-radio', role: 'radio' }, label: gt('Image:') });
            var noBgRadioArea = '<div class="dradio">' + noBgRadio + '</div>';
            var colorRadioArea = '<div class="dradio">' + colorRadio + '</div>';
            var imageRadioArea = '<div class="dradio">' + imageRadio + '</div>';
            var transpLabel = $('<div class="transparency-label">').append('Transparency:');

            var hideBgBtn = Forms.createButtonMarkup({ attributes: { class: 'hide-bg-btn', role: 'checkbox' }, label: gt('Hide background graphics') });
            var hideBgContainer = $('<div class="hidebg-container">').append(hideBgBtn);

            if (isColorRadioEnabled && dialogOptions.color) {
                colorPicker.setValue(dialogOptions.color);
            }
            colorArea = $('<div class="color-area">').append(colorPicker.getNode());
            imageArea = $('<div class="image-area">').append(imagePicker.getNode());
            transpArea = $('<div class="transparency-area">');

            transpSpinner = new PercentSpin(0, 100);
            transpArea.append(transpLabel, generateColorSlider(), transpSpinner.getNode());
            transpContainer = transpArea.children('.slider-container');
            transpSlider = transpContainer.children('.bg-color-slider');
            transpRailOverlay = transpSlider.children('.rail-overlay');
            transpPointer = transpSlider.children('.slider-pointer');

            if (Utils.SMALL_DEVICE) {
                transpPointer.css('transform', 'scale(2)');
            }

            modifyContainer = $('<div class="modify-container">').append(noBgRadioArea, colorRadioArea, colorArea, imageRadioArea, imageArea, transpArea);

            // append all containers to control area
            $controlArea.append(modifyContainer, hideBgContainer);

            // set value to slider/spinner
            if (_.isObject(dialogOptions) && dialogOptionsType !== 'none' && dialogOptionsType !== null) {
                if (isColorRadioEnabled && dialogOptions.color && dialogOptions.color.transformations) {
                    _.each(dialogOptions.color.transformations, function (transformation) {
                        if (transformation.type === 'alpha') {
                            transpValue = 100 - transformation.value / 1000;
                        }
                    });
                }
                if (isImageRadioEnabled && dialogOptions.bitmap && dialogOptions.bitmap.transparency) {
                    transpValue = dialogOptions.bitmap.transparency * 100;
                }
            } else { // check if bitmap is inherited from layout or master slide, and enable proper buttons
                var parentId = model.getParentSlideId(slideId);
                var parentFillAttrs = model.getSlideAttributesByFamily(parentId, 'fill');
                if (!parentFillAttrs || parentFillAttrs.type === 'none' || parentFillAttrs.type === null) {
                    parentId = model.getParentSlideId(parentId);
                    parentFillAttrs = model.getSlideAttributesByFamily(parentId, 'fill');
                }
                if (parentFillAttrs && parentFillAttrs.type === 'bitmap') {
                    isImageRadioEnabled = true;
                    transpValue = (parentFillAttrs.bitmap.transparency || 0) * 100;
                } else if (parentFillAttrs && (parentFillAttrs.type === 'solid' || parentFillAttrs.type === 'scheme')) {
                    isColorRadioEnabled = true;
                    var alphaT = _.findWhere(parentFillAttrs.color.transformations, { type: 'alpha' });
                    transpValue = (alphaT && _.isNumber(alphaT.value) && (100 - alphaT.value / 1000)) || 0;
                }
            }
            transpSpinner.setValue(transpValue);

            // enable/disable nodes
            enableNodes(colorArea.find('a'), isColorRadioEnabled);
            enableNodes(imageArea.find('a'), isImageRadioEnabled);
            enableNodes(transpArea.find('.slider-container, input'), isImageRadioEnabled || isColorRadioEnabled);
            imageArea.prop('disabled', !isImageRadioEnabled).toggleClass('disabled', !isImageRadioEnabled);
            colorArea.prop('disabled', !isColorRadioEnabled).toggleClass('disabled', !isColorRadioEnabled);
            transpArea.toggleClass('disabled', !isImageRadioEnabled && !isColorRadioEnabled);
            enableNodes(hideBgContainer.find('.hide-bg-btn'), !model.isMasterSlideId(model.getActiveSlideId()));

            // no-background radio
            noBgRadioBtn = $controlArea.find('.no-bg-radio');
            Forms.checkButtonNodes(noBgRadioBtn, !isImageRadioEnabled && !isColorRadioEnabled, { design: 'radio', ambiguous: false });
            Forms.setButtonKeyHandler(noBgRadioBtn);
            // color radio
            colorRadioBtn = $controlArea.find('.color-radio');
            Forms.checkButtonNodes(colorRadioBtn, isColorRadioEnabled, { design: 'radio', ambiguous: false });
            Forms.setButtonKeyHandler(colorRadioBtn);
            // image radio
            imageRadioBtn = $controlArea.find('.image-radio');
            Forms.checkButtonNodes(imageRadioBtn, isImageRadioEnabled, { design: 'radio', ambiguous: false });
            Forms.setButtonKeyHandler(imageRadioBtn);

            // hide background checkbox
            hideBgCheckbox = $controlArea.find('.hide-bg-btn');
            Forms.checkButtonNodes(hideBgCheckbox, !model.isFollowMasterShapeSlide(), { design: 'boxed', ambiguous: false });
            Forms.setButtonKeyHandler(hideBgCheckbox);
        }

        // public methods -----------------------------------------------------

        /**
         * Executes the dialog and performs all necessary steps, if the
         * user ended the dialog by clicking the Ok button
         *
         * @returns {jQuery} $.promise
         *  The resolved/rejected promise of the executed dialog
         */
        this.show = _.wrap(this.show, function (show) {
            return show.call(this).done(function (action) {
                var startOptions = _.extend({ fill: dialogOptions }, { slide: { followMasterShapes: model.isFollowMasterShapeSlide() } });
                var endOptions = $slide.data('previewAttrs') || {};
                endOptions = _.extend({ slide: { followMasterShapes: model.isFollowMasterShapeSlide() } }, endOptions);

                switch (action) {
                    case 'ok':
                        if (!_.isEqual(startOptions, endOptions)) {
                            model.applySlideBackgroundOperations($slide);
                        } else {
                            model.removeHelpBackgroundNode($slide);
                        }
                        break;
                    case 'applyall':
                        model.getSlideStyles().updateBackground($tempPreviewSlide, { fill: { type: null } }, slideId, null, { preview: true });
                        model.removeHelpBackgroundNode($slide);
                        model.applySlideBackgroundOperations($slide, { allSlides: true });
                }
            }).always(function () {
                // blocking keyboard input during applying of operations
                model.setBlockKeyboardEvent(false);
                self = model = view = colorPicker = imagePicker = null;
            });
        });

        // initialization -----------------------------------------------------

        // store deep cloned original values in node data
        $slide.data('previewAttrs', { fill: dialogOptions ? _.copy(dialogOptions, true) : TextUtils.getEmptySlideBackgroundAttributes().fill, slide: { followMasterShapes: model.isFollowMasterShapeSlide() } });

        // close dialog when losing edit rights
        view.closeDialogOnReadOnlyMode(this);

        $controlArea = $('<div class="control-area">');
        $tempPreviewSlide = model.getHelpBackgroundNode($slide);

        // initialize the body element of the dialog
        this.getBody()
            .addClass('io-ox-office-presentation-dialog-slide-background')
            .toggleClass('mobile', !!Utils.SMALL_DEVICE)
            .append($controlArea);

        // create the layout of the dialog
        initControls();

        if (dialogOptions && dialogOptions.type !== 'none' && dialogOptions.type !== null) {
            model.getSlideStyles().updateBackground($tempPreviewSlide, { fill: dialogOptions || { type: null } }, slideId, null, { preview: true, imageLoad: true, onDialogOpen: true });
        }

        // add additional button
        this.addButton('applyall', gt('Apply to all'));
        applyallBtn = this.getButton('applyall');
        applyallBtn.addClass('btn-apply-all btn-primary').detach();
        this.getFooter().append(applyallBtn);
        this.getOkButton().children().html(gt('Apply'));

        // ensure proper button order on small devices, because base dialog makes reorder
        if (Utils.SMALL_DEVICE) {
            this.on('beforeshow', function () {
                applyallBtn.parent().detach();
                this.getFooter().children('.row').append(applyallBtn.parent());
            });
        } else {
            this.getFooter().find('button').addClass('truncate-footer-btn');
        }

        // set initial control values and initial focus
        this.on('show', function () {
            // blocking keyboard input during applying of operations
            model.setBlockKeyboardEvent(true);

            transpSliderWidth = transpSlider.width();
            transpSliderOffset = transpSlider.offset().left;

            // us 102078426: dialogs on small devices should come up w/o keyboard,
            // so the focus shoud not be in an input element
            self.getOkButton().focus();
        });

        // listen on checkbox to change icon
        hideBgCheckbox.on('click', function () {
            var slideData = $slide.data('previewAttrs');
            Forms.checkButtonNodes(this, !Forms.isCheckedButtonNode(this), { design: 'boxed', ambiguous: false });
            $slide.data('previewAttrs', _.extend(slideData, { slide: { followMasterShapes: !Forms.isCheckedButtonNode(this) } }));
            view.previewFollowMasterPageChange({ followMasterShapes: !Forms.isCheckedButtonNode(this) });
        });

        noBgRadioBtn.on('click', function () {
            Forms.checkButtonNodes(colorRadioBtn.add(imageRadioBtn), false, { design: 'radio', ambiguous: false });
            Forms.checkButtonNodes(this, true, { design: 'radio', ambiguous: false });
            colorArea.prop('disabled', true).addClass('disabled');
            imageArea.prop('disabled', true).addClass('disabled');
            transpArea.addClass('disabled');
            enableNodes(transpArea.find('.slider-container, input'), false);
            enableNodes(colorArea.find('a'), false);
            enableNodes(imageArea.find('a'), false);

            var attrs = TextUtils.getEmptySlideBackgroundAttributes();
            transpSpinner.setValue(0);
            colorPicker.setValue({ type: 'auto' });
            view.handleSlideBackgroundVisibility(slideId, attrs.fill);
            model.getSlideStyles().updateBackground($tempPreviewSlide, attrs, slideId, null, { preview: true });
            $slide.data('previewAttrs', attrs);
        });

        colorRadioBtn.on('click', function () {
            Forms.checkButtonNodes(imageRadioBtn.add(noBgRadioBtn), false, { design: 'radio', ambiguous: false });
            Forms.checkButtonNodes(this, true, { design: 'radio', ambiguous: false });
            colorArea.prop('disabled', false).removeClass('disabled');
            imageArea.prop('disabled', true).addClass('disabled');
            transpArea.removeClass('disabled');
            enableNodes(transpArea.find('.slider-container, input'), true);
            enableNodes(colorArea.find('a'), true);
            enableNodes(imageArea.find('a'), false);
        });

        imageRadioBtn.on('click', function () {
            Forms.checkButtonNodes(colorRadioBtn.add(noBgRadioBtn), false, { design: 'radio', ambiguous: false });
            Forms.checkButtonNodes(this, true, { design: 'radio', ambiguous: false });
            imageArea.prop('disabled', false).removeClass('disabled');
            colorArea.prop('disabled', true).addClass('disabled');
            transpArea.removeClass('disabled');
            enableNodes(transpArea.find('.slider-container, input'), true);
            enableNodes(imageArea.find('a'), true);
            enableNodes(colorArea.find('a'), false);
        });

        // transparency slider events handling
        transpSlider.on(START_POINTER_EVENTS, function (event) {
            var clientX = event.type === 'mousedown' ? event.clientX : getTouchClientX(event);
            var leftPos = (clientX - transpSliderOffset) * 100 / transpSliderWidth;
            slidingActive = true;
            updateTransparency(leftPos);

            model.listenTo(self.getBody(), MOVE_POINTER_EVENTS, function (event) {
                var clientX = event.type === 'mousemove' ? event.clientX : getTouchClientX(event);
                var leftPos = (clientX - transpSliderOffset) * 100 / transpSliderWidth;
                updateTransparency(leftPos);
            });
        });

        this.getBody().on(END_POINTER_EVENTS, function () {
            if (slidingActive) {
                model.stopListeningTo(self.getBody(), MOVE_POINTER_EVENTS);
                slidingActive = false;
            }
        });

        transpContainer.on('keydown', function (event) {
            var perc = transpPointer.data('percentage') || 0;
            if (event.keyCode === KeyCodes.UP_ARROW || event.keyCode === KeyCodes.RIGHT_ARROW) {
                updateTransparency(perc + 1);
            }
            if (event.keyCode === KeyCodes.DOWN_ARROW || event.keyCode === KeyCodes.LEFT_ARROW) {
                updateTransparency(perc - 1);
            }
        });

        // color picker event handling
        colorPicker.on('group:change', function (event, data) {
            var attrs = data.type === 'auto' ? TextUtils.getEmptySlideBackgroundAttributes() : { fill: { type: 'solid', color: data, imageUrl: null } };

            view.handleSlideBackgroundVisibility(slideId, attrs.fill);
            setAlphaPreview(attrs, transpSpinner.getValue());
            this.hideMenu();
        });

        // on cancel restore original values
        this.on('cancel', function () {
            var attrs = dialogOptions || { type: null };

            model.getSlideStyles().updateBackground($tempPreviewSlide, { fill: attrs }, slideId, null, { preview: true });
            view.previewFollowMasterPageChange({ followMasterShapes: model.isFollowMasterShapeSlide() });
            model.removeHelpBackgroundNode($slide);
            view.handleSlideBackgroundVisibility(slideId);
        });

        // clean up after closing
        this.on('close', function () {
            $slide.removeData('previewAttrs');
            $controlArea = modifyContainer = hideBgCheckbox = noBgRadioBtn = colorRadioBtn = imageRadioBtn = null;
            self = model = slideId = $slide = colorPicker = imagePicker = colorArea = imageArea = null;
            transpArea = transpContainer = transpSlider = transpSliderWidth = transpSliderOffset = transpRailOverlay = transpPointer = null;
            transpSpinner = $tempPreviewSlide = applyallBtn = headerNode = null;
        });

        headerNode = this.getHeader().css('cursor', 'move');
        Tracking.enableTracking(headerNode);
        headerNode.on({ 'tracking:start': trackingStartHandler, 'tracking:move': trackingMoveHandler });

    } // class SlideBackgroundDialog

    // exports ================================================================

    // derive this class from class Dialogs.ModalDialog
    return Dialogs.ModalDialog.extend({ constructor: SlideBackgroundDialog });

});
