/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Peter Seliger <peter.seliger@open-xchange.com>
 */

define.async('io.ox/office/presentation/view/control/layoutslidepicker', [

    'io.ox/office/tk/utils',

    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/control/radiolist',

    'io.ox/office/presentation/view/layouttypetitlemap'

], function (Utils, LocaleData, RadioList, LayoutTypeTitleMap) {

    'use strict';

    // class LayoutSlidePicker ==================================================

    var
        global      = window,

        object_keys = global.Object.keys,
        math_round  = global.Math.round,

        THUMBNAIL_HEIGHT          = 50,
        VERTICAL_SPACING          = 45,

        css_value__thumb_height,
        css_value__thumb_width,

        css_value__button_height  = ((THUMBNAIL_HEIGHT + VERTICAL_SPACING) + 'px'),

        $thumbContainerBlueprint  = $('<div class="thumb-container" />'),
      //$thumbContainerBlueprint  = $('<div class="thumb-container io-ox-busy" />'),

        i18nLayoutTypeTitleDB     = {},

        // request the proper language data targeting layout type titles
        i18nDataRequest = LocaleData.loadResource('io.ox/office/presentation/resource/layouttypetitles').done(function (data) {
            i18nLayoutTypeTitleDB = data;
        });

    function ThumbBoxRegistry() {
        var
            registry = {};

        this.put = function (slideId, $thumbContainer) {
            registry[slideId] = $thumbContainer;
        };
        this.get = function (slideId) {
            return registry[slideId];
        };
        this.clear = function () {
            registry = {};
        };
        this.all = function () {
            return object_keys(registry).map(function (key) {
                return {
                    slideId:    key,
                    $container: registry[key]
                };
            });
        };
    }

    function recalculatePickerOffset(picker, pickerNodeParent/*, contentSize, availableSize*/) {
        var
          //elmTrigger    = picker.getNode()[0],
            elmPicker     = picker.getMenuNode()[0],

          //triggerCoords = elmTriggerMenu.getBoundingClientRect(),
            parentCoords  = pickerNodeParent.getBoundingClientRect();

        elmPicker.style.marginTop   = ((math_round(parentCoords.top)/* + 5*/) + 'px');
        elmPicker.style.marginLeft  = ((math_round(parentCoords.left)/* + 5*/) + 'px');

      //return contentSize;
    }

    /**
     * @constructor
     *
     * @extends RadioList
     */
    function LayoutSlidePicker(docView, initOptions) {
        var
            self                = this,                           // - self reference

            menu,                                                 // - the menu reference
            docModel            = docView.getDocModel(),          // - the document model

            pickerNodeParent    = docModel.getNode().parent()[0], // - alternative/custom parent-node the picker-node is going to be appended to.

            thumbBoxRegistry,

            promisedThumbList,
            promisedThumbCount;

          //isMenuContentExists;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({

            // hints
            tooltip: '+++ missing wording for this tooltip +++',
            label: '+++ missing wording for this label +++',

            // defaults
            icon: 'fa-clone',

            updateCaptionMode: 'none',
            toggle: true,

            itemDesign: 'grid',
            gridColumns: 5,

            smallerVersion: {
                hideLabel: true
            },
            updateLayoutHandler: function (/*contentSize, availableSize*/) {

                recalculatePickerOffset(self, pickerNodeParent);
              //return recalculatePickerOffset(self, pickerNodeParent/*, contentSize, availableSize*/);
            },
            rootContainerNode: pickerNodeParent

        }, initOptions));

        menu = this.getMenu();            // referable after having been extended.

        // private methods ----------------------------------------------------

        function getCustomSlideTitle(docModel, slideId) {

            return docModel.getSlideFamilyAttributeForSlide(slideId, 'slide', 'name');
        }

        function getMasterTypeTitle(/*key*/) {
            var
              //defaultTitle = LayoutTypeTitleMap['master'];  // - linter warning.
              //defaultTitle = LayoutTypeTitleMap[key];       // - does work, but too complicated.
                defaultTitle = LayoutTypeTitleMap.master;     // - go with what the linter did suggest.

            return (i18nLayoutTypeTitleDB[defaultTitle] || defaultTitle);
        }
        function getLayoutTypeTitle(slideId) {
          //return LayoutTypeTitleMap[docModel.getSlideFamilyAttributeForSlide(slideId, 'slide', 'type')];
            var
                defaultTitle = LayoutTypeTitleMap[docModel.getSlideFamilyAttributeForSlide(slideId, 'slide', 'type')];

            return (i18nLayoutTypeTitleDB[defaultTitle] || defaultTitle);
        }

        /**
         * hook on create items, only called once per type,
         * sets the button text and fill the chain-data for deferred rendering
         */
        function renderLayoutThumb($thumbContainer, slideId) {
          //Utils.info('+++ renderLayoutThumb :: $thumbContainer ', $thumbContainer);

            var
                $thumbPromise = docView.requestThumb(slideId),
                $layoutThumb;

            $thumbPromise.done(function (data) {
                $layoutThumb = $(data.thumb);

                // center align via math for it can't be done by css rules like `vertical-align: center`
                $layoutThumb.css('left', (Math.round(($thumbContainer.width() - $layoutThumb.data().width) / 2) + 'px'));

              //$thumbContainer.idle();
                $layoutThumb.appendTo($thumbContainer.empty());
            });

            if (promisedThumbList) {
                promisedThumbList.push($thumbPromise);

                if (promisedThumbList.length >= promisedThumbCount) {
                    $.when.apply($, promisedThumbList).then(handleEveryInitialPromiseStateResolved);
                }
            }
        }

        function rerenderEveryLayoutThumb() {
            if (thumbBoxRegistry) {
                thumbBoxRegistry.all().forEach(function (tuple) {
                    renderLayoutThumb(tuple.$container, tuple.slideId);
                });
            }
        }

        function updateLayoutThumb(slideId/*, idx, list*/) {
          //window.console.log('+++ updateLayoutThumb :: [slideId, $thumbContainer] +++ : ', slideId, thumbBoxRegistry.get(slideId));
            if (thumbBoxRegistry) { renderLayoutThumb(thumbBoxRegistry.get(slideId), slideId); }
        }

        function createThumbContainer(slideId) {
          //Utils.info('+++ createThumbContainer :: slideId ', slideId);
            var
                $thumbContainer = $thumbContainerBlueprint.clone();

            thumbBoxRegistry.put(slideId, $thumbContainer);

          //$thumbContainer.attr('data-container-id', slideId);

            $thumbContainer.css('width', css_value__thumb_width);
            $thumbContainer.css('height', css_value__thumb_height);

            return $thumbContainer;
        }

        function prependThumbContainer(event, $buttonNode) {
          //Utils.info('+++ prependThumbContainer :: event, $buttonNode ', event, $buttonNode);

            $buttonNode.css('height', css_value__button_height);

            var
                slideId         = $buttonNode.attr('data-value'),
                $thumbContainer = createThumbContainer(slideId);

          //$thumbContainer.busy();
            $thumbContainer.prependTo($buttonNode);

            renderLayoutThumb($thumbContainer, slideId);
        }

        function getSectionId(slideId) {
            return [

                'master_id_',
                docModel.isMasterSlideId(slideId) ? slideId : docModel.getMasterSlideId(slideId)

            ].join('');
        }

        function createMenuSection(slideId) {
            var
              //copy = getCustomSlideTitle(docModel, slideId) || getMasterTypeTitle('master') || ('master / ' + slideId);
                copy = getCustomSlideTitle(docModel, slideId) || getMasterTypeTitle() || ('master / ' + slideId);

            self.createMenuSection(getSectionId(slideId), { label: copy });
        }

        function createOptionButton(slideId) {
            // first parameter is the `data-value` payload value that will be passed along with all form control events.
            var
                copy = getCustomSlideTitle(docModel, slideId) || getLayoutTypeTitle(slideId) || ('layout / ' + slideId);

            self.createOptionButton(slideId, { label: copy, tooltip: copy, section: getSectionId(slideId) });
          //self.createOptionButton(slideId, { label: ('layout / ' + slideId), section: getSectionId(slideId) });
        }

        function createMenuItem(slideId/*, idx, list*/) {
          //Utils.info('+++ createMenuItem :: slideId ', slideId);
            if (docModel.isMasterSlideId(slideId)) {

                createMenuSection(slideId);

            } else if (docModel.isLayoutSlideId(slideId)) {

                createOptionButton(slideId);
            }
        }

        function handleLayoutThumbUpdate(evt, slideIdList) {
            slideIdList.forEach(updateLayoutThumb);
        }
        function handleEveryInitialPromiseStateResolved() {

            disableEveryInitialPromiseStateObservation();
            self.listenTo(docView, 'layoutthumb:change', handleLayoutThumbUpdate);

            rerenderEveryLayoutThumb();
        }

        function enableEveryInitialPromiseStateObservation(promiseCount) {
            promisedThumbList = [];
            promisedThumbCount = promiseCount;
        }
        function disableEveryInitialPromiseStateObservation() {
            promisedThumbList = null;
            promisedThumbCount = -1;
        }

        function createThumbRegistry() {
            thumbBoxRegistry = new ThumbBoxRegistry();
        }
        function deleteThumbRegistry() {
            if (thumbBoxRegistry) { // does not necessarily exist e.g. if there are no instances

                thumbBoxRegistry.clear();
                thumbBoxRegistry = null;
            }
        }

        function clearMenuContent() {
          //Utils.info('+++ clearMenuContent +++');
            self.stopListeningTo(docView, 'layoutthumb:change', handleLayoutThumbUpdate);

            menu.clearContents();
            deleteThumbRegistry();

            // bug 49404 & 50009  - flag for special handling for the layoutpicker because it has it's rootnode in the page, not outside it like normal.
            // on iOS would not be visible outside the content root node, and on iOS & Android the user could not select a slide in the layoutpicker
            if (Utils.IOS || Utils.CHROME_ON_ANDROID) { docView.getContentRootNode().removeClass('popup-open'); }

          //isMenuContentExists = false;
        }

        function createMenuContent() {
          //Utils.info('+++ createMenuContent +++');
            docView.forceLayoutMasterSlideFormatting();

            var
                masterLayoutSlideIdList = docModel.getMasterSlideOrder();

            createThumbRegistry();
            enableEveryInitialPromiseStateObservation(masterLayoutSlideIdList.filter(docModel.isLayoutSlideId).length);

            masterLayoutSlideIdList.forEach(createMenuItem);

            // bug 49404 & 50009  - flag for special handling for the layoutpicker because it has it's rootnode in the page, not outside it like normal.
            // on iOS would not be visible outside the content root node, and on iOS & Android the user could not select a slide in the layoutpicker
            if (Utils.IOS || Utils.CHROME_ON_ANDROID) { docView.getContentRootNode().addClass('popup-open'); }

          //isMenuContentExists = true;
        }

        // function handleBeforeShowPopup() {
        //     if (!isMenuContentExists) {
        //
        //         createMenuContent();
        //     }
        // }

        // initialization -----------------------------------------------------

        function initialize() {

            // destroy class members on destruction
            self.registerDestructor(function () {

                clearMenuContent();

                self.stopListeningTo(menu, 'popup:hide', clearMenuContent);
                self.stopListeningTo(menu, 'popup:beforeshow', createMenuContent);

              //self.stopListeningTo(menu, 'popup:beforeshow', handleBeforeShowPopup);
                self.stopListeningTo(menu, 'create:item', prependThumbContainer);

                self = initOptions = docModel = docView = null;
            });

            if (!css_value__thumb_height) {
                css_value__thumb_height = (THUMBNAIL_HEIGHT + 'px');
            }
            if (!css_value__thumb_width) {
                css_value__thumb_width = (Math.round(THUMBNAIL_HEIGHT * docModel.getSlideRatio()) + 'px');
            }

            self.getMenuNode().addClass('layout-slide-picker');

            self.listenTo(menu, 'popup:hide', clearMenuContent);
            self.listenTo(menu, 'popup:beforeshow', createMenuContent);

          //self.listenTo(menu, 'popup:beforeshow', handleBeforeShowPopup);
            self.listenTo(menu, 'create:item', prependThumbContainer);

            menu.clearContents();
          //isMenuContentExists = false;
        }

        initialize();

    } // class LayoutSlidePicker

    // exports ================================================================

    return i18nDataRequest.then(_.constant(RadioList.extend({ constructor: LayoutSlidePicker })));
});
