/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/updatelistsmixin', [
    'io.ox/office/tk/io',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/dom'
], function (IO, AttributeUtils, Utils, DOM) {

    'use strict';

    // mix-in class UpdateListsMixin ======================================

    /**
     * A mix-in class for the document model providing the handler functions
     * that keep lists up-to-date.
     *
     * Triggers the follwing events:
     * - 'listformatting:done': The debounced list formatting is done.
     *
     * @constructor
     *
     * @param {Application} app
     *  The application containing this instance.
     */
    function UpdateListsMixin(app) {

        var // self reference for local functions
            self = this,
            // a collector for all paragraph nodes that need to be updated
            paragraphs = $(),
            // whether only the current paragraph needs to be updated
            onlySingleParagraph = false,
            // the debounced list handler function
            updateListsDebounced = self.createDebouncedMethod(registerListUpdate, updateList, { delay: 10, infoString: 'Presentation: updateLists', app: app });

        // private methods ----------------------------------------------------

        /**
         * Direct callback for 'updateListsDebounced'. This is triggered every time, when a list update
         * is required. It allows registration of paragraphs, for that a list update must be done. Typically
         * this paragraph and all its neighbors need to be updated. Using the option 'singleParagraph' it
         * can be forced, that the neighbors will not be updated.
         *
         * @param {Object} options
         *  Optional parameters:
         *  @param {Boolean} [options.singleParagraph=false]
         *   If set to true, only the one specified paragraph is updated. Otherwise
         *   all paragraphs in the drawing that contains the specified paragraph are updated. 'singleParagraph'
         *   is useful, if a bullet type is modified or some character attributes in a paragraph are modified.
         */
        function registerListUpdate(paragraph, options) {

            // whether only the specified paragraph needs to be updated
            onlySingleParagraph = Utils.getBooleanOption(options, 'singleParagraph', false);

            // store the new paragraph in the collection (jQuery keeps the collection unique)
            if (paragraph) {
                paragraphs = paragraphs.add(paragraph);
            }
        }

        /**
         * The deferred callback for the list update.
         *
         * @returns {jQuery.Promise}
         *  A promise that is resolved, when all paragraphs are updated. Asynchronous updating of
         *  lists happens only during loading the document.
         */
        function updateList() {

            var // the collector for the paragraphs that need to be updated
                allParagraps = onlySingleParagraph ? paragraphs : paragraphs.parent().children(DOM.PARAGRAPH_NODE_SELECTOR),
                // a promise that is resolved, when all paragraphs are updated.
                // This happens only asynchronous, when the document is loaded
                updateListPromise = updateLists(null, allParagraps);

            paragraphs = $();
            onlySingleParagraph = false;

            return updateListPromise;
        }

        /**
         * Updates all paragraphs that are part of any bullet or numbering
         * list or update  all specified paragraphs.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.async=false]
         *      If set to true, all lists are updated asynchronously. This
         *      should happen only once when the document is loaded.
         *
         * @param {jQuery} [paragraphs]
         *  Optional set of paragraphs, that need to be updated. If not specified
         *  all paragraphs in the document need to be updated. This happens after
         *  the document was loaded successfully. In this case the option 'async'
         *  needs to be set to true.
         *
         * @returns {jQuery.Promise}
         *  A promise that is resolved, when all paragraphs are updated.
         */
        function updateLists(options, paragraphs) {

            var // a collector for all numbers for numbered list items
                listItemCounter = {},
                // check, if for the current level a start value is already defined
                isStartValueOnLevelSet = {},
                // list of all paragraphs in the document
                paragraphNodes = null,
                // whether the paragraphs are specified
                predefinedParagraphs = !_.isUndefined(paragraphs),
                // the deferred, whose promise will be returned
                def = null,
                // the target chain, that is required to resolve the correct theme
                target = null;

            function updateListInParagraph(para) {

                var // the attributes at the paragraph
                    elementAttributes = self.getParagraphStyles().getElementAttributes(para),
                    // the paragraph attributes at the paragraph
                    paraAttributes = elementAttributes.paragraph,
                    // the paragraph level
                    listLevel = paraAttributes.level,
                    // an existing list label at the paragraph
                    oldLabel = null,
                    // whether tab stops need to be recalculated at the paragraph
                    updateParaTabstops = false,
                    // during loading the document, the counter has to be resetted for every first paragraph in text frame
                    // -> Important: There can be paragraphs from several drawings in 'paragraphs', if a layout slide was modified(!)
                    resetListItemCounter = $(para).prevAll(DOM.PARAGRAPH_NODE_SELECTOR).length === 0,
                    // resetListItemCounter = !paragraphs && $(para).prevAll(DOM.PARAGRAPH_NODE_SELECTOR).length === 0,
                    // for some characters and fonts it is necessary to replace the values by more reliable values
                    replacementChar = null,
                    // the character and font name that are used for displaying
                    displayChar = null, displayFontName = null,
                    // the numbering type
                    numType = null,
                    // whether the paragraph has a list label
                    noListLabel = true,
                    // an url for a list item bitmap
                    imageUrl = null,
                    // the id of the numbered list
                    listCounterId =  null,
                    // the index of the sub levels
                    subLevelIdx = 0,
                    // a text string
                    textChar = '',
                    // the list label node
                    numberingElement = null,
                    // the first text span in the paragraph
                    span = null,
                    // the character attributes of the text span following the list label node
                    followingTextAttributes = null,
                    // the list span inside the list label node
                    listSpans = null,
                    // whether the paragraph is empty
                    isEmptyParagraph = false,
                    // must call more often the model
                    docModel = app.getModel();

                // helper function to count the number for the list items in numbered lists
                // (only used for numbered list paragraphs)
                function countListLevel() {

                    if (resetListItemCounter) {  // resetting for each new drawing
                        listItemCounter = {};
                        isStartValueOnLevelSet = {};
                    }

                    if (!listItemCounter[listCounterId]) {
                        listItemCounter[listCounterId] = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
                    }

                    if (!noListLabel) {
                        if (_.isNumber(paraAttributes.bullet.startAt)) {
                            if (isStartValueOnLevelSet[listLevel]) {
                                if (!isEmptyParagraph) { listItemCounter[listCounterId][listLevel]++; } // there is already a start value defined
                            } else {
                                listItemCounter[listCounterId][listLevel] = paraAttributes.bullet.startAt;  // this is the first paragraph with start value
                                isStartValueOnLevelSet[listLevel] = true;  // a new start value for this level is set
                            }
                        } else {
                            if (!isEmptyParagraph) { listItemCounter[listCounterId][listLevel]++; }
                            isStartValueOnLevelSet[listLevel] = false;  // no current start value for this level
                        }
                    }

                    // reset all sub-levels depending on their 'levelRestartValue' attribute
                    subLevelIdx = listLevel + 1;

                    _.each(_.keys(listItemCounter), function (counterId) {

                        var index = subLevelIdx;
                        for (; index < 10; index++) {
                            listItemCounter[counterId][index] = 0;
                            isStartValueOnLevelSet[index] = false;  // no current start value for all sub levels
                        }

                        // restart at the same level for all other list IDs
                        if (counterId !== listCounterId) {
                            listItemCounter[counterId][listLevel] = 0;
                        }
                    });

                }

                // handling the count list level for paragraphs, that are not numbered lists. This
                // can be bullet lists or no lists at all -> at that level and lower, all lists are
                // restarted.
                function restartCountListLevel(level) {

                    _.each(_.keys(listItemCounter), function (counterId) {

                        var index = level;
                        for (; index < 10; index++) {
                            listItemCounter[counterId][index] = 0;
                            isStartValueOnLevelSet[index] = false;
                        }
                    });
                }

                // updating the paragraph
                oldLabel = $(para).children(DOM.LIST_LABEL_NODE_SELECTOR);
                updateParaTabstops = oldLabel.length > 0;
                oldLabel.remove();

                // After removing the label, it can be checked, if the paragraph is empty
                // -> DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS might already be set by the 'selectionChangeHandler'
                isEmptyParagraph = $(para).hasClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS) || DOM.isEmptyParagraph(para);
                if (isEmptyParagraph) {
                    $(para).addClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS); // -> adding a class to the paragraph to mark it as empty
                    if (Utils.getDomNode(self.getSelectedParagraph()) === Utils.getDomNode(para)) {
                        $(para).addClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_SELECTED_CLASS); // -> adding a class to the paragraph to mark it as empty and selected
                    }
                }

                // resetting counter for non numbering lists
                if (paraAttributes.bullet.type !== 'numbering') { restartCountListLevel(_.isNumber(listLevel) ? listLevel : 0); }

                if (_.isNumber(listLevel) && listLevel >= 0) {

                    noListLabel = paraAttributes.bullet.type ? paraAttributes.bullet.type === 'none' : true;
                    imageUrl = paraAttributes.bullet.type === 'bitmap' ? paraAttributes.bullet.imageUrl : null;

                    // preparing counting of list item numbers
                    if (paraAttributes.bullet.type === 'numbering') {
                        listCounterId = paraAttributes.bullet.numType;  // different counter for every different numbering type
                    } else {
                        resetListItemCounter = true; // restart counting after every non-numbering paragraph (TODO: Only in same level?)
                    }

                    if (listLevel !== -1 && listLevel < 10) {

                        if (paraAttributes.bullet.type === 'numbering') { countListLevel(); }

                        updateParaTabstops = true;

                        if (paraAttributes.bullet.type === 'character') {

                            // check if a replacement is required for displaying the character correctly
                            replacementChar = self.getUnicodeReplacementForChar(paraAttributes.bullet.character, paraAttributes.bulletFont.name);

                            if (_.isObject(replacementChar)) {
                                displayChar = replacementChar.levelText;
                                displayFontName = replacementChar.fontFamily;
                            } else {
                                displayChar = paraAttributes.bullet.character;
                                displayFontName = paraAttributes.bulletFont.name ? paraAttributes.bulletFont.name : null;
                            }

                            textChar = paraAttributes.bullet.character ? displayChar : ''; // setting the character that is used inside the list label

                        } else if (paraAttributes.bullet.type === 'numbering') {
                            numType = paraAttributes.bullet.numType || 'arabicPeriod';
                            textChar = self.createNumberedListItem(isEmptyParagraph ? (listItemCounter[listCounterId][listLevel] + 1) : listItemCounter[listCounterId][listLevel], numType);
                            displayFontName = paraAttributes.bulletFont.name ? paraAttributes.bulletFont.name : null;
                        }

                        numberingElement = DOM.createListLabelNode(noListLabel ? '' : textChar);

                        span = DOM.findFirstPortionSpan(para);
                        followingTextAttributes = self.getCharacterStyles().getElementAttributes(span).character;
                        listSpans = numberingElement.children('span');

                        if (imageUrl) {

                            var absUrl = app.getServerModuleUrl(IO.FILTER_MODULE_NAME, { action: 'getfile', get_filename: imageUrl });
                            var image = $('<div>', { contenteditable: false });
                            var img = $('<img>', { src: absUrl });

                            var options = null;
                            if (!target && AttributeUtils.isCharacterFontThemed(followingTextAttributes)) { target = docModel.getTargetChainForNode(para); }
                            if (target) { options = { theme: docModel.getThemeCollection().getTheme('', target) }; }

                            var baseLine = 0.8 * docModel.getFontCollection().getBaseLineOffset(followingTextAttributes, options);

                            // fix for Bug 47859 & Bug 48025
                            image.css('height', Utils.convertToEM(baseLine, 'px'));
                            img.css('height', Utils.convertToEM(baseLine, 'px'));

                            image.addClass('drawing')
                            .addClass('bulletdrawing')
                            .data('url', imageUrl)
                            .append(
                                $('<div>').append(img)
                            );

                            numberingElement.prepend(image);

                        } else if (!noListLabel) {

                            if (paraAttributes.bulletSize.type === 'followText') {
                                listSpans.css('font-size', Utils.convertToEM(followingTextAttributes.fontSize, 'pt'));
                            } else if (paraAttributes.bulletSize.type === 'percent') {
                                listSpans.css('font-size', Utils.convertToEM(Utils.round((paraAttributes.bulletSize.size / 100) * followingTextAttributes.fontSize, 1), 'pt'));
                            } else {
                                listSpans.css('font-size', Utils.convertToEM(paraAttributes.bulletSize.size, 'pt')); // TODO: Is the size given in 'pt' ?
                            }

                            if (paraAttributes.bulletFont.followText) {
                                listSpans.css('font-family', self.getCssFontFamily(followingTextAttributes.fontName));
                            } else {
                                if (displayFontName) { listSpans.css('font-family', self.getCssFontFamily(displayFontName)); }
                            }

                            if (AttributeUtils.isColorThemed(followingTextAttributes.color) && (!target || !predefinedParagraphs)) {
                                target = docModel.getTargetChainForNode(para); // target chain should be identical for all predefined paragraphs
                            }

                            if (paraAttributes.bulletColor.followText) {
                                listSpans.css('color', self.getCssColor(followingTextAttributes.color, 'text', target));
                            } else {
                                listSpans.css('color', self.getCssColor(paraAttributes.bulletColor.color, 'text', target));
                            }

                            // in numbering also bold and italic are adapted from the following text span
                            if (paraAttributes.bullet.type === 'numbering' && paraAttributes.bulletFont.followText) {
                                listSpans.css('font-weight', $(span).css('font-weight'));
                                listSpans.css('font-style', $(span).css('font-style'));
                            }

                            self.getCharacterStyles().updateElementLineHeight(listSpans, paraAttributes.lineHeight, followingTextAttributes);
                        }

                        // setting distance between margin, numbering element and text (effects only the numbering element!)
                        if (!noListLabel && paraAttributes.indentFirstLine < 0) {
                            numberingElement.css('min-width', (-paraAttributes.indentFirstLine / 100) + 'mm');
                        }

                        if (-paraAttributes.indentFirstLine > paraAttributes.indentLeft) {
                            numberingElement.css('margin-left', (-paraAttributes.indentLeft - paraAttributes.indentFirstLine) / 100 + 'mm');
                        }

                        if (!noListLabel) { $(para).prepend(numberingElement); } // adding the list label

                    }
                }

                if (updateParaTabstops) {
                    self.getParagraphStyles().updateTabStops(para);
                }
            }

            // receiving list of all document paragraphs (excluding slides)
            paragraphNodes = paragraphs ? paragraphs : self.getCurrentRootNode().find(DOM.PARAGRAPH_NODE_SELECTOR).not('.slide');

            if (Utils.getBooleanOption(options, 'async', false)) {
                def = self.iterateArraySliced(paragraphNodes, updateListInParagraph, { delay: 'immediate', infoString: 'UpdateListMixin: updateListInParagraph', app: app });
            } else {
                paragraphNodes.each(function () { updateListInParagraph(this); });
                def = $.when();
            }

            return def.promise().always(function () {
                // informing the listeners, that the list formatting is done
                self.trigger('listformatting:done');
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Getting the debounced handler function that is used for updating the lists
         * in the document. This function must be used instead of 'updateLists', after
         * the document was loaded successfully.
         *
         * @returns {Function}
         *  The debounced handler function for updating the lists.
         */
        this.getDebouncedUpdateListsHandler = function () {
            return updateListsDebounced;
        };

        /**
         * Getting the not debounced handler function that is used for updating
         * the lists synchronously. This updateLists should only be called from
         * 'updateDocumentFormatting' during the loading phase. After the document
         * was loaded successfully, 'updateListsDebounced' need to be used.
         *
         * @returns {Function}
         *  The handler function for updating the lists.
         */
        this.getUpdateListsHandler = function () {
            return updateLists;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class UpdateListsMixin

    // constants --------------------------------------------------------------

    // exports ================================================================

    return UpdateListsMixin;

});
