/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/components/drawing/drawingresize', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/presentation/utils/presentationutils'
], function (Utils, Tracking, AttributeUtils, DrawingFrame, Operations, DOM, Position, PresentationUtils) {

    'use strict';

    // the names of all tracking events but 'tracking:start'
    var TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';

    var ACTIVE_DRAGGING = 'activedragging';

    var ACTIVE_RESIZING = 'activeresizing';

    // static class DrawingSlideResize ========================================

    var DrawingSlideResize = {};

    // static methods ---------------------------------------------------------

    /**
     * Draws a selection box for the specified drawing node and registers
     * mouse handlers for moving and resizing.
     *
     * @param {PresentationApplication} app
     *  The application instance containing the drawing.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node to be selected, as DOM node or jQuery object.
     */
    DrawingSlideResize.drawDrawingSelection = function (app, drawingNode) {

        var // the model object
            model = app.getModel(),
            // the selection object
            selection = model.getSelection(),
            // the current root node
            rootNode = model.getCurrentRootNode(),
            // the node containing the scroll bar
            scrollNode = app.getView().getContentRootNode(),
            // if it's text frame with table inside it
            containsTable = DrawingFrame.isTableDrawingFrame(drawingNode),
            // object containing information about movable and resizable
            options = { movable: true, resizable: true, rotatable: !containsTable },  // -> parameter?
            // the container element used to visualize the selection
            selectionBox = null,
            // saving the selected resize node
            resizeNode = null,
            //zoom factor in floating point notation
            zoomFactor,
            // target ID of currently active root node, if existing
            target = model.getActiveTarget(),
            // an object collecting the attributes for an operation
            operationProperties = {},
            // whether this is a multi drawing selection
            isMultiSelection = false,
            // a collector for all move boxes and drawing informations
            allDrawings = null,
            // constant for delaying shape formatting during resize for different browsers
            TEXTSHAPE_UPDATE_RESIZE_DELAY = _.browser.Chrome ? 5 : _.browser.Firefox ? 100 : 250,
            // the empty drawing node that contains the selection node
            selectionDrawing = null,
            // a debounced function for updating text frame shapes
            updateTextFrameShapesDebounced = model.createDebouncedMethod($.noop, updateTextFrameShapes, { delay: TEXTSHAPE_UPDATE_RESIZE_DELAY, infoString: 'Presentation: DrawingResize.updateTextFrameShapesDebounced', app: app });

        // debounced callback for updating text frames in allDrawings
        function updateTextFrameShapes() {
            _.each(allDrawings, function (oneDrawing) {
                if (!DrawingFrame.isTableDrawingFrame(oneDrawing.drawingNode)) {
                    updateTextFrameInShape(oneDrawing.drawingNode, oneDrawing.moveBox, { isResizeActive: true });
                }
            });
        }

        // helper function to collect all move box information in multiple drawing selections
        function collectAndActivateAllMoveBoxes(options) {
            _.each(selection.getMultiSelection(), function (drawingSelection) {
                var drawingNode = selection.getDrawingNodeFromMultiSelection(drawingSelection);
                var drawingPos = selection.getStartPositionFromMultiSelection(drawingSelection);

                activateMoveBox(drawingNode, drawingPos, options);
            });
        }

        function activateMoveBox(drawingNode, drawingPos, options) {
            var isResize = Utils.getBooleanOption(options, 'resize', false);
            var attrs = AttributeUtils.getExplicitAttributes(drawingNode);
            var hasNoFillNoLine = attrs && attrs.fill && attrs.fill.type === 'none' && attrs.line && attrs.line.type === 'none';
            var isNotEmptyPlaceholderDrawing = PresentationUtils.isPlaceHolderDrawing(drawingNode) && !PresentationUtils.isEmptyPlaceHolderDrawing(drawingNode);
            var isEmptyNoFillLineDrawing = hasNoFillNoLine && drawingNode.text().length === 0;
            var lockResizeAspectRatio = false;

            drawingNode.addClass(ACTIVE_DRAGGING);
            if (drawingNode.data('selection')) { drawingNode.data('selection').addClass(ACTIVE_DRAGGING); }

            if (isResize) { // don't evaluate lock ratio on move, just for resize
                drawingNode.addClass(ACTIVE_RESIZING);
                if (drawingNode.data('selection')) { drawingNode.data('selection').addClass(ACTIVE_RESIZING); }
                lockResizeAspectRatio = (attrs && attrs.drawing && attrs.drawing.noChangeAspect) || Utils.SMALL_DEVICE;
            }

            var oneMoveBox = cloneDiv(drawingNode, isResize);
            if ((isResize && hasNoFillNoLine) || isEmptyNoFillLineDrawing || isNotEmptyPlaceholderDrawing) {
                oneMoveBox.addClass(DrawingFrame.FRAME_WITH_TEMP_BORDER);
            }
            allDrawings.push({ drawingNode: drawingNode, moveBox: oneMoveBox, width: drawingNode.width(), height: drawingNode.height(), start: drawingPos, lockResizeAspectRatio: lockResizeAspectRatio });
        }

        function clearDrawing(drawing) {
            drawing.removeClass(ACTIVE_DRAGGING);
            drawing.removeClass(ACTIVE_RESIZING);
            drawing.children('.copy').remove();
            if (drawing.data('selection')) {
                drawing.data('selection').removeClass(ACTIVE_DRAGGING);
                drawing.data('selection').removeClass(ACTIVE_RESIZING);
            }
        }

        function cloneCanvas(canvas) {
            var img = document.createElement('img');
            img.src = canvas.toDataURL();
            img.width         = canvas.width;
            img.height        = canvas.height;
            img.className     = canvas.className;
            img.style.cssText = canvas.style.cssText;
            img.style.position = 'absolute';
            return img;
        }

        /**
         * Helper function to clone one drawing and the canvas elements
         * it contains. This function also handles a drawing of type 'group'
         * correctly.
         *
         * @param {jQuery} drawing
         *  The jQuerified drawing node.
         *
         * @param {Boolean} isResizeActive
         *  On resize don't replace cloned canvas, just prepend img, for better UX.
         *
         * @returns {jQuery}
         *  The jQueryfied content node, that is a direct child of the
         *  specified drawing node.
         */
        function cloneDiv(drawing, isResizeActive) {

            drawing.find('>.copy').remove();

            var content = drawing.children('.content:not(.copy)'); // only direct child (take care of groups)
            var clone = content.clone(DrawingFrame.isGroupContentNode(content));
            if (isResizeActive) {
                if (PresentationUtils.isEmptyPlaceHolderDrawing(drawing)) {
                    clone.find('.templatetext, .helper').remove();
                }
                drawing.prepend(clone);
            } else {
                drawing.append(clone);
            }

            var orgCanvases = content.find('canvas').get();  // the original canvas elements in the content node

            clone.find('canvas').get().forEach(function (canvas, index) {
                var img = cloneCanvas(orgCanvases[index]);
                if (!isResizeActive) {
                    $(canvas).replaceWith(img);
                } else {
                    if (DrawingFrame.isGroupContentNode(clone)) {
                        clone.find('.textframecontent').eq(index).prepend(img);
                    } else {
                        clone.prepend(img);
                    }
                }
            });

            clone.addClass('copy');

            return clone;
        }

        /**
         * Local method to update text frame box during resize of the shape,
         * so that text can reflow even before drawing resize is finalized.
         *
         * @param {jQuery} drawingNode
         *
         * @param {jQuery} moveBox
         *
         * @param {Object} [options]
         *  @param {Boolean} [options.isResizeActive=false]
         *     If this method is called during resizing of the shape.
         */
        function updateTextFrameInShape(drawingNode, moveBox, options) {
            if (DrawingFrame.isGroupDrawingFrame(drawingNode)) {
                moveBox.find('.textframecontent').children('img').remove();
                DrawingFrame.previewOnResizeDrawingsInGroup(drawingNode, options);
            } else {
                moveBox.children('img').remove();
                DrawingFrame.updateShapeFormatting(drawingNode, options);
            }
        }

        // helper function to collect and prepare all drawings for rotation in multiple drawing selections
        function collectAndPrepareDrawingsForRotation() {
            _.each(selection.getMultiSelection(), function (drawingSelection) {
                var drawingNode = selection.getDrawingNodeFromMultiSelection(drawingSelection);
                var drawingPos;
                var startRotation;

                if (!DrawingFrame.isTableDrawingFrame(drawingNode)) {
                    drawingPos = selection.getStartPositionFromMultiSelection(drawingSelection);
                    startRotation = DrawingFrame.getDrawingRotationAngle(model, drawingNode);
                    allDrawings.push({ drawingNode: drawingNode, start: drawingPos, rotation: startRotation });
                }
            });
        }

        // helper function to convert degrees to radians. If no valid degree is passed, returns null.
        function convertDegreeToRadian(deg) {
            return _.isNumber(deg) ? (deg * Math.PI / 180) : null;
        }

        // helper function to normalize rotated drawings pixels shift for move. Returns coordinates object
        function getNormalizedMoveCoordinates(drawing, shiftXo, shiftYo) {
            var tempX = null;
            var tempY = null;
            var angle = DrawingFrame.getDrawingRotationAngle(model, drawing);
            var rad = convertDegreeToRadian(angle);

            if (rad) {
                tempX = shiftXo * Math.cos(rad) + shiftYo * Math.sin(rad);
                tempY = shiftYo * Math.cos(rad) - shiftXo * Math.sin(rad);
                shiftXo = tempX;
                shiftYo = tempY;
            }

            return { x: shiftXo, y: shiftYo };
        }

        // helper function to normalize drawing pixels delta for resize. Returns coordinates object
        function getNormalizedResizeDeltas(drawing, deltaXo, deltaYo, useX, useY, scaleX, scaleY) {
            var localX = null;
            var localY = null;
            var angle = DrawingFrame.getDrawingRotationAngle(model, drawing);
            var rad = convertDegreeToRadian(angle);

            if (rad) {
                localX = useX ? (deltaXo * Math.cos(rad) + deltaYo * Math.sin(rad)) * scaleX : 0;
                localY = useY ? (deltaYo * Math.cos(rad) - deltaXo * Math.sin(rad)) * scaleY : 0;
            } else {
                localX = useX ? deltaXo * scaleX : 0;
                localY = useY ? deltaYo * scaleY : 0;
            }

            return { x: localX, y: localY };
        }

        // helper function to normalize offset of rotated drawings. Returns object of top and left offset properties.
        function normalizeOffset(drawingNode, moveBox) {
            var moveBoxOffsetLeft = moveBox.offset().left;
            var moveBoxOffsetTop = moveBox.offset().top;
            var angle = DrawingFrame.getDrawingRotationAngle(model, drawingNode);
            var tLeft, tTop, dLeft, dTop;
            if (_.isNumber(angle)) {
                $(drawingNode).css({ transform: '' }); // reset to def to get values
                tLeft = $(drawingNode).offset().left;
                tTop = $(drawingNode).offset().top;
                $(drawingNode).css({ transform: 'rotate(' + angle + 'deg)' }); // return property to it's value

                dLeft = tLeft - $(drawingNode).offset().left;
                dTop = tTop - $(drawingNode).offset().top;

                moveBoxOffsetLeft += dLeft;
                moveBoxOffsetTop += dTop;
            }
            return { left: moveBoxOffsetLeft, top: moveBoxOffsetTop };
        }

        // helper function to normalize resize offset of rotated drawings. Returns object of top and left offset properties,
        // and sendMoveOperation flag, to send appropriate move operations if drawing is rotated and needs to be translated to the position.
        function normalizeResizeOffset(drawingNode, moveBox) {
            var moveBoxOffsetTop, moveBoxOffsetLeft;
            var sendMoveOperation = false;
            var angle = DrawingFrame.getDrawingRotationAngle(model, drawingNode);

            if (_.isNumber(angle)) {
                moveBox.css({ transform: 'rotate(' + (-angle) + 'deg)' });
                moveBoxOffsetLeft = moveBox.offset().left;
                moveBoxOffsetTop = moveBox.offset().top;
                moveBox.css({ transform: '' });

                sendMoveOperation = true;
            } else {
                moveBoxOffsetLeft = moveBox.offset().left;
                moveBoxOffsetTop = moveBox.offset().top;
            }

            return { left: moveBoxOffsetLeft, top: moveBoxOffsetTop, sendMoveOperation: sendMoveOperation };
        }

        function extendAndGenerateOp(generator, drawingNode, operationProperties, target) {
            PresentationUtils.extendPlaceholderProp(app, drawingNode, operationProperties);
            model.extendPropertiesWithTarget(operationProperties, target);
            generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

            model.updateDynFontSizeDebounced(drawingNode);
        }

        function triggerDragging() {
            app.getView().getContentRootNode().trigger('activedragging');
        }

        /**
         * Handling all tracking events while moving a drawing.
         */
        var drawingMoveHandler = (function () {

            var // the current position change (in px)
                shiftX = 0, shiftY = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the start scroll position
                startScrollX = 0, startScrollY = 0,
                // whether the move box was at least once painted
                moveBoxPainted = false;

            // initialing the moving of the drawing according to the passed tracking event
            function startMoveDrawing() {

                zoomFactor = app.getView().getZoomFactor() / 100;

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();

                moveBoxPainted = false;

                isMultiSelection = selection.isMultiSelectionSupported() && selection.isMultiSelection();

                allDrawings = []; // 'tracking:start' event is triggered twice consecutively, so we clean collection to prevent doubling

                if (isMultiSelection) {
                    collectAndActivateAllMoveBoxes();
                } else {
                    var drawingPos = Position.getOxoPosition(rootNode, drawingNode, 0);
                    activateMoveBox(drawingNode, drawingPos);
                }
            }

            // updating the drawing position according to the passed tracking event
            function updateMove(event) {

                // make move box visible when tracking position has moved
                DrawingFrame.toggleTracking(drawingNode.data('selection'), true);

                // reading scrollPosition again. Maybe it was not updated or not updated completely.
                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;

                shiftX = (event.pageX - event.startX + scrollX) / zoomFactor;
                shiftY = (event.pageY - event.startY + scrollY) / zoomFactor;

                // only move the moveBox, if the mouse was really moved
                if (_.isNumber(shiftX) && _.isNumber(shiftY) && ((shiftX !== 0) || (shiftY !== 0))) {

                    var updateMoveBox = function (oneDrawing) {
                        var normalizedCoords = getNormalizedMoveCoordinates(oneDrawing.drawingNode, shiftX, shiftY);
                        oneDrawing.moveBox.css({ cursor: event.cursor, left: normalizedCoords.x, top: normalizedCoords.y, width: oneDrawing.width, height: oneDrawing.height });
                    };

                    moveBoxPainted = true;

                    _.each(allDrawings, updateMoveBox);
                }
            }

            // updates scroll position according to the passed tracking event
            function updateMoveScroll(event) {

                // update scrollPosition with suggestion from event
                if (event.scrollX) {
                    scrollNode.scrollLeft(scrollNode.scrollLeft() + event.scrollX);
                }

                if (event.scrollY) {
                    scrollNode.scrollTop(scrollNode.scrollTop() + event.scrollY);
                }

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // handling the drawing position, when moving is stopped according to the passed tracking event
            function stopMoveDrawing(event) {

                var // the operations generator
                    generator = model.createOperationsGenerator(),
                    //
                    availablePos = null,
                    //
                    newPoses = null,
                    //
                    dragToCopy = event.ctrlKey,
                    // whether at least one drawing is moved above the slide
                    isMovedAboveSlide = false,
                    //
                    finalizeDrawing = function (oneDrawing) {

                        var start = oneDrawing.start;

                        if (dragToCopy) {
                            if (!availablePos) {
                                availablePos = model.getNextAvailablePositionInActiveSlide();
                                newPoses = [];
                            } else {
                                availablePos[1]++;
                            }
                            start = _.clone(availablePos);
                            newPoses.push(start);

                            var node = oneDrawing.drawingNode;
                            if (DOM.isDrawingPlaceHolderNode(node)) { node = DOM.getDrawingPlaceHolderNode(node); }

                            // generate operations for the drawing (including its attributes)
                            generator.generateDrawingOperations(node, start, { ignoreFamily: 'changes presentation', target: target });
                        }

                        var normalizedOffset = normalizeOffset(oneDrawing.drawingNode, oneDrawing.moveBox);

                        // setting the new attributes of the drawing
                        operationProperties = {
                            start: start,
                            attrs: { drawing: {
                                left: Utils.convertLengthToHmm((normalizedOffset.left - rootNode.offset().left) / zoomFactor, 'px'),
                                top: Utils.convertLengthToHmm((normalizedOffset.top - rootNode.offset().top) / zoomFactor, 'px') }
                            }
                        };
                        extendAndGenerateOp(generator, oneDrawing.drawingNode, operationProperties, target);

                        // check whether a drawing is moved above the slide
                        if (operationProperties.attrs.drawing.top < 0) { isMovedAboveSlide = true; }
                    };

                if (moveBoxPainted && !($(drawingNode).parent().is('.notselectable'))) { // only move drawing, if it is selectable

                    _.each(allDrawings, finalizeDrawing);

                    // apply the operations (undo group is created automatically)
                    model.applyOperations(generator);

                    if (dragToCopy) {
                        model.getSelection().setMultiDrawingSelectionByPosition(newPoses);
                    }

                    // update of vertical scrollbar is required, if a drawing is moved above the slide
                    if (isMovedAboveSlide) { model.trigger('update:verticalscrollbar', { keepScrollPosition: true, pos: model.getActiveSlideIndex() }); }
                }
            }

            // finalizes the move tracking
            function finalizeMoveDrawing() {
                leaveTracking();
                drawingNode.off(TRACKING_EVENT_NAMES);

                // Resetting variables for new mouseup events without mousemove
                shiftX = shiftY = scrollX = scrollY = 0;

                isMultiSelection = false;
                _.each(allDrawings, function (oneDrawing) {
                    oneDrawing.moveBox.css({ left: 0, top: 0, width: 0, height: 0 }).toggle(false);
                    updateSelectionDrawingSize(oneDrawing.drawingNode);
                    clearDrawing(oneDrawing.drawingNode);
                });
                allDrawings = [];
            }

            function updateMoveAlways(event) {
                var dragToCopy = event.ctrlKey;
                event.cursor = dragToCopy ? 'copy' : 'move';
                // model.getNode().toggleClass('drag-to-copy', dragToCopy);
                triggerDragging();
            }

            // return the actual drawingMoveHandler() function
            return function (event) {

                // handling for text frame drawings, if the click happened inside the internal text frame element
                if ($(event.target).closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0) {

                    Tracking.disableTracking(drawingNode)
                        .off('tracking:start', trackingStartHandler)
                        .off(TRACKING_EVENT_NAMES);

                    return;
                }

                switch (event.type) {
                    case 'tracking:start':
                        startMoveDrawing(event);
                        break;
                    case 'tracking:move':
                        updateMove(event);
                        // prevent background scrolling in ios
                        if (model.getSlideTouchMode()) { event.preventDefault(); }
                        break;
                    case 'tracking:scroll':
                        updateMoveScroll(event);
                        break;
                    case 'tracking:end':
                        stopMoveDrawing(event);
                        finalizeMoveDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeMoveDrawing(event);
                        break;
                }

                updateMoveAlways(event);
            };

        }()); // end of drawingMoveHandler() local scope

        /**
         * Handles all tracking events while resizing a drawing.
         */
        var drawingResizeHandler = (function () {

            var // the size of the resized drawing (in px)
                finalWidth = 0, finalHeight = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the initial scroll position
                startScrollX = 0, startScrollY = 0,
                // whether resizing is available in horizontal/vertical direction
                useX = false, useY = false,
                // whether resizing is available in left or top direction
                useLeft = false, useTop = false,
                // correction factor for resizing to the left/top
                scaleX = 0, scaleY = 0,
                // a collector for the rows of a table drawing
                allRows = null;

            // initializes resizing the drawing according to the passed tracking event
            function startResizeDrawing(event) {

                var // evaluating which resize element is active
                    pos = $(event.target).attr('data-pos');

                // helper function to set position properties at move box(es)
                function setPositionAtMoveBox(oneMoveBox) {

                    if (useLeft) {
                        scaleX = -1;
                        oneMoveBox.css({ left: 'auto', right: 0 });
                    } else {
                        scaleX = 1;
                        oneMoveBox.css({ left: 0, right: 'auto' });
                    }

                    if (useTop) {
                        scaleY = -1;
                        oneMoveBox.css({ top: 'auto', bottom: 0 });
                    } else {
                        scaleY = 1;
                        oneMoveBox.css({ top: 0, bottom: 'auto' });
                    }

                }

                zoomFactor = app.getView().getZoomFactor() / 100;

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();

                // collecting information about the handle node
                useX = /[lr]/.test(pos);
                useY = /[tb]/.test(pos);

                useLeft = /[l]/.test(pos);
                useTop = /[t]/.test(pos);

                isMultiSelection = selection.isMultiSelectionSupported() && selection.isMultiSelection();
                allDrawings = [];

                if (isMultiSelection) {
                    collectAndActivateAllMoveBoxes({ resize: true });
                } else {
                    var drawingPos = Position.getOxoPosition(rootNode, drawingNode, 0);
                    activateMoveBox(drawingNode, drawingPos, { resize: true });
                }
                _.each(allDrawings, function (oneDrawing) {
                    setPositionAtMoveBox(oneDrawing.moveBox);
                    oneDrawing.drawingNode.addClass(ACTIVE_DRAGGING);
                    oneDrawing.drawingNode.addClass(ACTIVE_RESIZING);
                    if (oneDrawing.drawingNode.data('selection')) {
                        oneDrawing.drawingNode.data('selection').addClass(ACTIVE_DRAGGING);
                        oneDrawing.drawingNode.data('selection').addClass(ACTIVE_RESIZING);
                    }
                });
            }

            // updates scroll position according to the passed tracking event
            function updateResizeScroll(event) {

                // update scrollPosition with suggestion from event
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // updates resizing the drawing according to the passed tracking event
            function updateResize(event) {

                var // the horizontal shift
                    deltaX = (event.pageX / zoomFactor - event.startX / zoomFactor + scrollX),
                    // the vertical shift
                    deltaY = (event.pageY / zoomFactor - event.startY / zoomFactor + scrollY),
                    // the scaling factor for the width
                    scaleWidth = 1,
                    // the scaling factor for the height
                    scaleHeight = 1,
                    // normalize and scale deltas of the dimensions
                    normalizedDeltas = getNormalizedResizeDeltas(drawingNode, deltaX, deltaY, useX, useY, scaleX, scaleY);

                // helper function to set the height of rows and cells in table drawing nodes
                function updateVerticalTableHeight(drawingNode, moveBox, oldHeight) {

                    var firstRun = !allRows;
                    allRows = allRows || moveBox.find('tr');
                    if (firstRun) {
                        _.each(allRows, function (row) {
                            $(row).data('startrowheight', $(row).height()); // saving row height at start of resize
                        });
                    }

                    oldHeight = oldHeight || drawingNode.height;
                    var currentRatio = moveBox.height() / oldHeight; // ratio of drawing height
                    _.each(allRows, function (row) {
                        var newRowHeight = $(row).data('startrowheight') * currentRatio;
                        if (_.browser.WebKit) {
                            $(row).children('td').css('height', newRowHeight);
                        } else if (_.browser.IE) {
                            $(row).children('td').css('min-height', newRowHeight);
                        } else {
                            $(row).css('height', newRowHeight);
                        }
                    });
                }

                // helper function to set the current width and height of each selected drawing
                function updateDrawingDimension(drawingNode, moveBox, oldWidth, oldHeight, lockAspectRatio) {

                    var canvasExpansion = 0; // an optional expansion of the canvas, for example for arrows

                    // update drawing size
                    finalWidth = Math.max(0, oldWidth + normalizedDeltas.x);
                    finalHeight = Math.max(0, oldHeight + normalizedDeltas.y);

                    // use the same scaling factor for vertical and horizontal resizing, if both are enabled
                    // -> the larger number wins
                    if (useX && useY && (lockAspectRatio || event.shiftKey)) {
                        scaleWidth = finalWidth / oldWidth;
                        scaleHeight = finalHeight / oldHeight;

                        if (scaleWidth > scaleHeight) {
                            finalHeight = scaleWidth * oldHeight;
                        } else {
                            finalWidth = scaleHeight * oldWidth;
                        }
                    }

                    // make move box visible when tracking position has moved
                    DrawingFrame.toggleTracking(drawingNode.data('selection'), true);

                    moveBox.css({ width: finalWidth, height: finalHeight });

                    // simulating height changes of tables (makes only sense for single drawing selections because of performance)
                    if (useY && !isMultiSelection && DrawingFrame.isTableDrawingFrame(drawingNode)) { updateVerticalTableHeight(drawingNode, moveBox, oldHeight); }

                    if (!DrawingFrame.isGroupContentNode(moveBox)) {
                        canvasExpansion = drawingNode.data('canvasexpansion') || 0; // canvas expansion was saved at the drawing node
                        canvasExpansion *= 2;
                        moveBox.find('canvas, .cropping-frame img').css({ width: finalWidth + canvasExpansion + 2, height: finalHeight + canvasExpansion + 2 }); // make smooth transition, until canvas is updated
                    }
                }

                _.each(allDrawings, function (oneDrawing) {
                    updateDrawingDimension(oneDrawing.drawingNode, oneDrawing.moveBox, oneDrawing.width, oneDrawing.height, oneDrawing.lockResizeAspectRatio);
                });

                // debounce updating of text frames in allDrawings on slower browsers & devices
                updateTextFrameShapesDebounced();
            }

            // stop resizing of drawing
            function stopResizeDrawing() {

                var // the operations generator
                    generator = model.createOperationsGenerator();

                // helper function to generate the operation for resizing one drawing
                function generateResizeOperation(drawingNode, moveBox, start) {

                    var normalizedOffset = normalizeResizeOffset(drawingNode, moveBox);

                    if (moveBox.css('display') !== 'none' && moveBox.width() && moveBox.height()) { // generate operations only if moveBox has real w/h value, measured without outer
                        // setting the new attributes of the drawing
                        operationProperties = {
                            start: start,
                            attrs: { drawing: {
                                width: Utils.convertLengthToHmm(moveBox.outerWidth(true), 'px'), // 'outer' is required to avoid shrinking in unused dimension
                                height: Utils.convertLengthToHmm(moveBox.outerHeight(true), 'px') }
                            }
                        };

                        // evaluating useTop and useLeft to avoid sending of superfluous drawing information
                        if (useTop || normalizedOffset.sendMoveOperation) { operationProperties.attrs.drawing.top = Utils.convertLengthToHmm((normalizedOffset.top - rootNode.offset().top) / zoomFactor, 'px'); }
                        if (useLeft || normalizedOffset.sendMoveOperation) { operationProperties.attrs.drawing.left = Utils.convertLengthToHmm((normalizedOffset.left - rootNode.offset().left) / zoomFactor, 'px'); }

                        if (useX && DrawingFrame.isNoWordWrapDrawingFrame(drawingNode) && DrawingFrame.isAutoResizeHeightDrawingFrame(drawingNode)) {
                            operationProperties.attrs.shape = { wordWrap: true };
                        }

                        extendAndGenerateOp(generator, drawingNode, operationProperties, target);
                    }
                }

                // helper function to generate the operation for resizing one drawing of type table
                function generateTableDrawingResizeOperations(drawingNode, moveBox, start, height) {

                    if (useX || useTop) {
                        // generating the drawing operation for the width or the movement upwards
                        generateResizeOperation(drawingNode, moveBox, start);
                    }

                    if (useY) {
                        // increase the height of all table rows, if the drawing height has changed
                        // -> height must not be set explicitely for table drawings, because this is
                        //    determined by the height of all rows.
                        var tableNode = drawingNode.children('.content:not(.copy)').find('table');  // the table node, that is NOT inside the copy content
                        var oldHeight = height || tableNode.outerHeight(true);
                        var newHeight = moveBox.outerHeight(true);
                        var factor = newHeight / oldHeight;
                        var allRows = DOM.getTableRows(tableNode);  // using specific table node, otherwise there might be too many rows because of 'copy' node
                        var newRowHeight = 0;
                        var rowCounter = 0;

                        _.each(allRows, function (row) {

                            newRowHeight = Utils.round(Utils.convertLengthToHmm($(row).outerHeight(true), 'px') * factor, 1);
                            // -> not using the saved height in the explicit attributes, because this value might be lower
                            //    than the 'real' height, if the text content forces the row to be higher.

                            operationProperties = {
                                start: Position.appendNewIndex(start, rowCounter),
                                attrs: { row: { height: newRowHeight } }
                            };

                            extendAndGenerateOp(generator, drawingNode, operationProperties, target);

                            rowCounter++;
                        });
                    }
                }

                _.each(allDrawings, function (oneDrawing) {

                    if (DrawingFrame.isTableDrawingFrame(oneDrawing.drawingNode)) {
                        generateTableDrawingResizeOperations(oneDrawing.drawingNode, oneDrawing.moveBox, oneDrawing.start, oneDrawing.height);
                    } else {
                        generateResizeOperation(oneDrawing.drawingNode, oneDrawing.moveBox, oneDrawing.start);
                    }

                    // prevent flickering on finalize resizing by replacing original canvas with repainted one
                    var copyCanvas = oneDrawing.moveBox.children('canvas');
                    oneDrawing.drawingNode.children('.textframecontent').not('.copy').children('canvas').replaceWith(copyCanvas);
                });

                // Applying the operations (undo group is created automatically)
                model.applyOperations(generator);
            }

            // finalizes the resize tracking
            function finalizeResizeDrawing() {
                leaveTracking();
                resizeNode.off(TRACKING_EVENT_NAMES);

                isMultiSelection = false;
                _.each(allDrawings, function (oneDrawing) {
                    DrawingFrame.toggleTracking(oneDrawing.drawingNode.data('selection'), false); // leaving tracking for all selected drawings
                    oneDrawing.moveBox.css({ left: 0, top: 0, width: 0, height: 0 }).toggle(false);
                    updateSelectionDrawingSize(oneDrawing.drawingNode);
                    // oneDrawing.drawingNode.data('selection').css({ transform: oneDrawing.drawingNode.css('transform'), left: oneDrawing.drawingNode.css('left'), top: oneDrawing.drawingNode.css('top'), width: oneDrawing.drawingNode.width(), height: oneDrawing.drawingNode.height() });
                    clearDrawing(oneDrawing.drawingNode);
                });
                allDrawings = [];
                allRows = null;
            }

            // return the actual drawingResizeHandler() function
            return function (event) {

                event.preventDefault();

                switch (event.type) {
                    case 'tracking:start':
                        startResizeDrawing(event);
                        break;
                    case 'tracking:move':
                        updateResize(event);
                        break;
                    case 'tracking:scroll':
                        updateResizeScroll(event);
                        break;
                    case 'tracking:end':
                        stopResizeDrawing(event);
                        finalizeResizeDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeResizeDrawing(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();

                triggerDragging();
            };

        }()); // end of drawingResizeHandler() local scope

        var drawingRotateHandler = (function () {
            // dimensions of the drawing
            var drawingWidth = 0, drawingHeight = 0;
            // position of the center rotation of the drawingWidth
            var centerX, centerY;
            // the current scroll position
            var scrollX = 0, scrollY = 0;
            // the initial scroll position
            var startScrollX = 0, startScrollY = 0;
            // starting angle before rotation
            var startAngle = 0;
            // final angle on finish of the rotation
            var rotationAngle = null;
            // flag for marking that the operation is generated
            var generatedOp = false;

            function startRotateDrawing() {
                var drawingOffset;

                zoomFactor = app.getView().getZoomFactor() / 100;
                startAngle = DrawingFrame.getDrawingRotationAngle(model, drawingNode);
                if (_.isNumber(startAngle)) {
                    $(drawingNode).css({ transform: '' }); // reset to def to get values
                    drawingOffset = drawingNode.offset();
                    $(drawingNode).css({ transform: 'rotate(' + startAngle + 'deg)' }); // return property to it's value
                } else {
                    startAngle = 0;
                    drawingOffset = drawingNode.offset();
                }

                drawingHeight = drawingNode.height() * zoomFactor;
                drawingWidth = drawingNode.width() * zoomFactor;
                centerX = drawingOffset.left + (drawingWidth / 2);
                centerY = drawingOffset.top + (drawingHeight / 2);

                isMultiSelection = selection.isMultiSelectionSupported() && selection.isMultiSelection();

                allDrawings = []; // 'tracking:start' event is triggered twice consecutively, so we clean collection to prevent doubling

                if (isMultiSelection) {
                    collectAndPrepareDrawingsForRotation();
                } else {
                    var drawingPos = Position.getOxoPosition(rootNode, drawingNode, 0);
                    allDrawings.push({ drawingNode: drawingNode, start: drawingPos, rotation: startAngle });

                }
                _.each(allDrawings, function (obj) {
                    obj.drawingNode.addClass('rotating-state');
                    obj.drawingNode.data('selection').addClass('rotating-state');
                });
            }

            function updateRotate(event) {
                var radians = Math.atan2((event.pageX + scrollX - centerX), (event.pageY + scrollY - centerY));
                rotationAngle = (radians * (180 / Math.PI) * -1) + 180;
                // round to full degrees
                rotationAngle = parseInt(rotationAngle, 10);

                if (event.shiftKey) {
                    rotationAngle = Math.round(rotationAngle / 15) * 15; // make 15 deg step with shift key
                }
                if (isMultiSelection) {
                    _.each(allDrawings, function (obj) {
                        var startAngleTemp = obj.rotation;
                        obj.drawingNode.css('transform', 'rotate(' + (rotationAngle - startAngle + startAngleTemp) + 'deg)');
                    });
                } else {
                    drawingNode.css('transform', 'rotate(' + (rotationAngle) + 'deg)');
                }
            }

            function updateRotateScroll(event) {
                // update scrollPosition with suggestion from event
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            function stopRotateDrawing() {
                var generator = model.createOperationsGenerator();
                var operationProperties;

                if (_.isNumber(rotationAngle)) {
                    if (isMultiSelection) {
                        _.each(allDrawings, function (obj) {
                            var startAngleTemp = obj.rotation;

                            var tempRotationAngle = rotationAngle - startAngle + startAngleTemp;
                            // for 360 degrees apply 0 rotation
                            tempRotationAngle = tempRotationAngle === 360 ? 0 : tempRotationAngle;

                            // normalize negative and angles greater than 360
                            if (tempRotationAngle < 0) {
                                tempRotationAngle += 360;
                            }
                            if (tempRotationAngle > 360) {
                                tempRotationAngle -= 360;
                            }

                            operationProperties = {
                                start: obj.start,
                                attrs: { drawing: { rotation: tempRotationAngle } }
                            };
                            extendAndGenerateOp(generator, obj.drawingNode, operationProperties, target);
                        });
                    } else {
                        // for 360 degrees apply 0 rotation
                        rotationAngle = rotationAngle === 360 ? 0 : rotationAngle;

                        operationProperties = {
                            start: Position.getOxoPosition(rootNode, drawingNode, 0),
                            attrs: { drawing: { rotation: rotationAngle } }
                        };
                        extendAndGenerateOp(generator, drawingNode, operationProperties, target);
                    }

                    // Applying the operations (undo group is created automatically)
                    model.applyOperations(generator);
                    generatedOp = true;
                }
            }

            function finalizeRotateDrawing() {
                if (isMultiSelection) {
                    _.each(allDrawings, function (obj) {
                        obj.drawingNode.removeClass('rotating-state');
                        obj.drawingNode.data('selection').removeClass('rotating-state');
                    });
                } else {
                    drawingNode.removeClass('rotating-state');
                    drawingNode.data('selection').removeClass('rotating-state');
                }

                if (!generatedOp) { // reset to start angle if tracking was canceled
                    if (isMultiSelection) {
                        _.each(allDrawings, function (obj) {
                            var startAngleTemp = obj.rotation || 0;

                            obj.drawingNode.css('transform', 'rotate(' + startAngleTemp + 'deg)');
                        });
                    } else {
                        drawingNode.css('transform', 'rotate(' + startAngle + 'deg)');
                    }
                } else {
                    if (isMultiSelection) {
                        _.each(allDrawings, function (obj) {
                            DrawingFrame.updateResizersMousePointers(obj.drawingNode, DrawingFrame.getDrawingRotationAngle(model, obj.drawingNode));
                            obj.drawingNode.data('selection').css({ transform: obj.drawingNode.css('transform') });
                        });
                    } else {
                        DrawingFrame.updateResizersMousePointers(drawingNode, DrawingFrame.getDrawingRotationAngle(model, drawingNode));
                        drawingNode.data('selection').css({ transform: drawingNode.css('transform') });
                    }
                }

                leaveTracking();

                if (isMultiSelection) {
                    isMultiSelection = false;
                    allDrawings = [];
                }
                generatedOp = false;
            }

            // return the actual drawingRotateHandler() function
            return function (event) {

                event.preventDefault();

                switch (event.type) {
                    case 'tracking:start':
                        startRotateDrawing(event);
                        break;
                    case 'tracking:move':
                        updateRotate(event);
                        break;
                    case 'tracking:scroll':
                        updateRotateScroll(event);
                        break;
                    case 'tracking:end':
                        stopRotateDrawing(event);
                        finalizeRotateDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeRotateDrawing(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();

                triggerDragging();
            };
        }()); // end of drawingRotateHandler local scope

        /**
         * Leaving the tracking mode.
         */
        function leaveTracking() {
            DrawingFrame.toggleTracking(drawingNode.data('selection'), false);
        }

        /**
         * Handler for 'tracking:start' events for 'div.handle' and 'div.move'
         * nodes. This handler has to decide, which node got the event. For this
         * node all other tracking events have to be registered. Additionally the
         * handler is called with the 'tracking:start' event, so that the drawing
         * initialization happens.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event, that starts the moving or the resizing
         *  of a drawing.
         */
        function trackingStartHandler(event) {

            var pos = DrawingFrame.getResizerHandleType(event.target);
            if (_.isString(pos)) {
                resizeNode = $(event.target);
                resizeNode.off(TRACKING_EVENT_NAMES);
                resizeNode.on(TRACKING_EVENT_NAMES, drawingResizeHandler);
                drawingResizeHandler.call(this, event);
            } else if ($(event.target).hasClass('rotate-handle')) {
                selectionDrawing.off(TRACKING_EVENT_NAMES); // <- should not be necessary, but it is (more and more setAttributes OPs)
                selectionDrawing.on(TRACKING_EVENT_NAMES, drawingRotateHandler);
                drawingRotateHandler.call(this, event);
            } else {
                selectionDrawing.off(TRACKING_EVENT_NAMES); // <- should not be necessary, but it is (more and more setAttributes OPs)
                selectionDrawing.on(TRACKING_EVENT_NAMES, drawingMoveHandler);
                drawingNode.off(TRACKING_EVENT_NAMES); // <- should not be necessary, but it is (more and more setAttributes OPs)
                drawingNode.on(TRACKING_EVENT_NAMES, drawingMoveHandler);
                drawingMoveHandler.call(this, event);
            }
        }

        function toggleTracking(node, state) {
            if (state) {
                Tracking.enableTracking(node, {
                    autoScroll: true,
                    borderNode: scrollNode,
                    borderMargin: -30,
                    borderSize: 60,
                    minSpeed: 10,
                    maxSpeed: 250,
                    trackModifiers: true
                });
            } else {
                Tracking.disableTracking(node);
            }
        }

        // helper function to set the size, position and rotation of a selection drawing node
        function updateSelectionDrawingSize(drawing) {
            if (drawing.data('selection')) {
                drawing.data('selection').css({ transform: drawing.css('transform'), left: drawing.css('left'), top: drawing.css('top'), width: drawing.width(), height: drawing.height() });
            }
        }

        // shifting the selection into an overlay node
        function shiftSelectionToOverlayNode() {

            var // the overlay node that contains the selection drawings
                overlayNode = model.getNode().children('.drawingselection-overlay'),
                // the empty drawing node that contains the selection
                selDrawing =  $('<div>').addClass('drawing selectiondrawing').css({ position: 'absolute' }),
                // a helper string for class names
                classesString = drawingNode.get(0).className;

            if (classesString) { selDrawing.addClass(classesString); }  // transferring drawing class names to the selection drawing node
            drawingNode.data('selection', selDrawing); // saving the drawing selection node at the drawing
            selectionBox.css('position', 'static');
            updateSelectionDrawingSize(drawingNode);
            selDrawing.append(selectionBox); // shifting the selection to the new empty drawing node
            overlayNode.append(selDrawing);

            return selDrawing;
        }

        // starting code of static method drawDrawingSelection()
        drawingNode = $(drawingNode);

        if (drawingNode.length !== 1) {
            Utils.error('DrawingResize.drawDrawingSelection(): single drawing node expected');
            drawingNode = drawingNode.first();
        }

        if (drawingNode.hasClass('horizontal-line')) {
            options.movable = false;
        } else if (drawingNode.hasClass('grouped')) {
            // options.movable = options.resizable = false;
            drawingNode = drawingNode.closest(DrawingFrame.GROUPCONTENT_SELECTOR).parent();
        }

        if (!model.getEditMode()) {
            options.movable = options.resizable = options.rotatable = false;
        }

        options.scaleHandles = 100 / app.getView().getZoomFactor();

        // removing an existing selection (possible for example in additional text frame selections)
        if (drawingNode.data('selection')) { DrawingFrame.clearSelection(drawingNode); }

        // the container element used to visualize the selection
        selectionBox = DrawingFrame.drawSelection(drawingNode, options);

        // shifting the selection into an overlay node
        selectionDrawing = shiftSelectionToOverlayNode();

        DrawingFrame.updateResizersMousePointers(drawingNode, DrawingFrame.getDrawingRotationAngle(model, drawingNode));

        // set visible drawing anchor
        app.getView().setVisibleDrawingAnchor(drawingNode, AttributeUtils.getExplicitAttributes(drawingNode));

        toggleTracking(drawingNode, options.movable);
        toggleTracking(selectionDrawing, options.movable);
        // initialize resize tracking
        toggleTracking(selectionBox.find('>.resizers>[data-pos]'), options.resizable);
        // initialize rotate tracking
        toggleTracking(selectionBox.find('.rotate-handle'), options.rotatable);

        if (options.movable || options.resizeable || options.rotatable) {
            selectionDrawing.on('tracking:start', trackingStartHandler);
            drawingNode.on('tracking:start', trackingStartHandler);
        } else {
            selectionDrawing.off('tracking:start', trackingStartHandler);
            drawingNode.on('tracking:start', trackingStartHandler);
        }
    };

    // exports ================================================================

    return DrawingSlideResize;

});
