/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/portal/presentation/fulltour', [
    'io.ox/office/portal/portaltour',
    'io.ox/office/baseframework/utils/baseconfig',
    'gettext!io.ox/office/portal/main'
], function (PortalTour, BaseConfig, gt) {

    'use strict';

    var PRESENTER_WINDOW_SELECTOR = '.io-ox-presenter-window:visible';

    var START_PRESENTATION_SELECTOR = PRESENTER_WINDOW_SELECTOR + ' .io-ox-action-link[data-ref="io.ox/presenter/toolbar/dropdown/start-presentation"]:visible';

    // class PresentationFullTour =============================================

    /**
     * The guided tour for the Presentation Portal application.
     *
     * @constructor
     *
     * @extends PortalTour
     */
    var PresentationFullTour = PortalTour.extend({ constructor: function (initOptions) {

        // self reference
        var self = this;

        // the presenter application launched from the presentation created by this tour
        var presenterApp = null;

        // base constructor ---------------------------------------------------

        PortalTour.call(this, 'presentation', initOptions);

        // public methods -----------------------------------------------------

        /**
         * Launches the OX Presenter application for the presentation created
         * by this tour.
         */
        this.launchPresenterApp = function () {
            if (presenterApp) {
                presenterApp.launch();
            } else {
                this.executeItem('document/present').done(function () {
                    presenterApp = this.getApp();
                }.bind(this));
            }
        };

        // initialization -----------------------------------------------------

        // the initial welcome message
        this.welcomeStep()
            .title(gt.pgettext('tour', 'The Presentation app'))
            .content(gt.pgettext('tour', 'Welcome to your new Presentation app. This Guided Tour will introduce you to the most important functions of creating and editing presentations online.'));

        // do not show the following steps in the short version of the tour
        if (!this.isShortTour()) {

            // highlight the header of the template area
            this.templatesAreaStep()
                .title(gt.pgettext('tour', 'Templates'))
                .content(gt.pgettext('tour', 'Clicking on the Presentation app in the navigation bar displays an overview of all the templates available to you in Presentation.'));

            // describe the 'Blank document' button
            this.blankDocumentStep()
                .title(gt.pgettext('tour', 'Blank presentation'))
                .content(gt.pgettext('tour', 'You can choose from a variety of templates to help you create professional presentations.'))
                .content($('<br>'))
                .content(gt.pgettext('tour', 'Let\'s start with a blank presentation.'));
        }

        // launch the editor application with a new empty document
        this.launchDocumentStep();

        // describe the Slide and Layout buttons
        this.step()
            .title(gt.pgettext('tour', 'New slides'))
            .content(gt.pgettext('tour', 'With a click on "Slide" you can add new slides to your presentation. Choose "Layout" to change the design of the selected slide. Depending on your template, many different layouts and slide types are available to you.'))
            // open and highlight the 'Save As' drop-down menu
            .showDropDownMenu(this.EDITOR_TOOL_PANE_SELECTOR, 'layoutslidepicker/insertslide')
            .waitForAndReferToPopup({ position: 'right', hotspot: true });

        // do not show the following steps in the short version of the tour
        if (!this.isShortTour()) {

            // insert document contents, show the 'Saving changes' label
            this.autoSaveStep(function () {
                this.getDocModel().insertText('Lorem ipsum dolor sit amet.', [0, 0, 0, 0]);
            });
        }

        // describe the "Edit master" button
        this.step()
            //#. "master" means a master slide in a presentation
            .title(gt.pgettext('tour', 'Edit master'))
            .content(gt.pgettext('tour', 'You can create and edit the layout of your slides by clicking on "Edit master". While editing your master slides, you are able to change back to edit your normal slides by clicking on "Edit normal".'))
            .on('before:show', function () { this.triggerOptionButton(self.EDITOR_TOP_PANE_SELECTOR, 'view/toolbars/tab', 'slide'); })
            .waitForAndReferToGroup(this.EDITOR_TOOL_PANE_SELECTOR, 'slide/masternormalslide', { position: 'bottom', hotspot: true });

        // do not show the following steps in the short version of the tour
        if (!this.isShortTour()) {

            // describe the 'File' toolbar
            this.fileTabStep();
            this.saveAsStep();
            if (BaseConfig.MAIL_AVAILABLE) { this.sendAsMailStep(); }
        }

        // do not show presenter steps without OX Presenter application
        if (BaseConfig.PRESENTER_AVAILABLE) {

            // describe the "Present" button starting a presentation
            this.step()
                .title(gt.pgettext('tour', 'Present slides'))
                .content(gt.pgettext('tour', 'With a click on "Present" you can start a presentation of your slides right away.'))
                .waitFor(this.EDITOR_TOP_PANE_SELECTOR)
                .referToGroup(this.EDITOR_TOP_PANE_SELECTOR, 'document/present', { position: 'bottom', hotspot: 'i' });

            // start the Presenter application
            var presStep = this.step();
            // different texts according to availability of remote presentations
            if (BaseConfig.REMOTE_PRESENTER_AVAILABLE) {
                presStep
                    .title(gt.pgettext('tour', 'Presentation mode'))
                    .content(gt.pgettext('tour', 'You can choose whether you would like to give a full-screen presentation or present remotely and provide your audience with a link to your slides.'));
            } else {
                presStep
                    .title(gt.pgettext('tour', 'Start presentation'))
                    .content(gt.pgettext('tour', 'Click here to give a full-screen presentation to your audience.'));
            }
            presStep
                // launch the presenter aaplication
                .on('before:show', this.launchPresenterApp.bind(this))
                // wait for the drop-down menu button
                .waitFor(START_PRESENTATION_SELECTOR, 10)
                .on('ready before:hide', function () { $(START_PRESENTATION_SELECTOR).dropdown('toggle'); })
                .referTo(START_PRESENTATION_SELECTOR + '~ul.dropdown-menu', { hotspot: true })
                // launch the editor application when leaving this step
                .on('back next', this.launchEditorApp.bind(this));
        }

        // describe the 'Close' button, switch to 'Documents' folder in Drive
        if (!this.isShortTour()) {
            this.closeButtonStep();
            this.documentsFolderStep();
        }

        // final tour step describing how to restart the tour, return to Drive application with the Back button
        this.restartTourStep().on('back', function () {
            // 'this' is the tour step instance (not the tour)!
            if (!self.isShortTour()) {
                self.launchDriveApp();
            } else if (BaseConfig.PRESENTER_AVAILABLE) {
                self.launchPresenterApp();
            } else {
                self.launchEditorApp();
            }
        });

        // destroy the presenter application when closing the tour
        this.on('stop', function () {
            if (presenterApp) { presenterApp.quit(); }
        });

    } }); // class PresentationFullTour

    // exports ================================================================

    return PresentationFullTour;

});
