/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/searchpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/baseframework/view/toolpane',
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/baseframework/view/basecontrols',
    'io.ox/office/editframework/view/editlabels'
], function (Utils, KeyCodes, ToolPane, ToolBar, Controls, EditLabels) {

    'use strict';

    // class SearchPane =======================================================

    /**
     * Represents the search pane in OX Document edit applications.
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {EditView} docView
     *  The document view instance containing this search pane.
     */
    var SearchPane = ToolPane.extend({ constructor: function (docView) {

        var self = this,

            lastLayoutState = null,

            searchBar = new ToolBar(docView);

        // base constructor ---------------------------------------------------

        ToolPane.call(this, docView, { position: 'top', classes: 'standard-design' });

        // private methods ----------------------------------------------------

        function paneLayoutHandler() {

            if (lastLayoutState) {
                //clear all states
                searchBar.iterateGroups(function (group) {
                    var node = group.getNode().children().first();
                    node.removeAttr('style');
                });
                lastLayoutState = null;
            }

            var container = self.getNode().find('.center .area-container');
            var availW = self.getNode().width();
            var currW = container.width();

            var layoutState = availW < currW;

            if (layoutState !== lastLayoutState) {

                lastLayoutState = layoutState;

                if (layoutState) {
                    searchBar.iterateGroups(function (group) {
                        var options = group.getOptions();
                        if (options.smallerVersion && layoutState) {
                            var node = group.getNode().children().first();
                            node.css(options.smallerVersion.css);

                            currW = container.width();
                            layoutState = availW < currW;
                        }
                    });
                }
            }
        }

        /**
         * Key handler that hides this search pane on ESCAPE key.
         */
        function keyDownHandler(event) {
            if ((event.keyCode === KeyCodes.ESCAPE) && self.isVisible()) {
                docView.grabFocus();
                self.hide();
                return false;
            }
        }

        // initialization -----------------------------------------------------

        this.hide();

        searchBar
            .addGroup('document/search/text',  new Controls.TextField(EditLabels.SEARCH_TEXT))
            .addGroup('document/search/start', new Controls.Button(EditLabels.SEARCH_START))
            .addGroup('document/search/prev',  new Controls.Button(EditLabels.SEARCH_PREV))
            .addGroup('document/search/next',  new Controls.Button(EditLabels.SEARCH_NEXT))
            .addGroup('document/replace/text', new Controls.TextField(EditLabels.REPLACE_TEXT))
            .addGroup('document/replace/next', new Controls.Button(EditLabels.REPLACE_NEXT))
            .addGroup('document/replace/all',  new Controls.Button(EditLabels.REPLACE_ALL));

        this.addViewComponent(searchBar);

        // this fn is only for small devices
        this.handleTabListButtons = Utils.NOOP;

        this.on('pane:layout', paneLayoutHandler);

        // when ESCAPE bubbles up to the window body from everywhere, hide the search pane too
        this.listenTo(docView.getApp().getWindowNode(), 'keydown', keyDownHandler);

        // hide the search pane explicitly on ESCAPE key while focused
        // (base class Pane stops propagation of the ESCAPE key event)
        this.getNode().on('keydown', keyDownHandler);

        // execute the 'search:end' search command after closing the search pane
        this.on('pane:hide', function () {
            docView.executeControllerItem('document/search/end');
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = null;
        });

    } }); // class SearchPane

    // exports ================================================================

    return SearchPane;

});
