/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Jonas Regier <jonas.regier@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/colorslider', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/control/group'
], function (Utils, Tracking, Group) {

    'use strict';

    // Width of a color field. Must have the same value as the width in the less definition (.colorSliderField.width).
    var colorFieldWidth = 28;

    // Margin of the color slider. Must have the same value as the width in the less definition (.colorSlider.margin-left).
    var marginLeft = 12;

    // Width of the handle border. Must have the same value as the width in the less definition (.colorSliderHandle.border)
    var handleBorderWidth = 10;

    // class colorSlider ======================================================

    /**
     * A color slider element with markup and event handling.
     *
     * @constructor
     *
     * @extends Group
     *
     * @param {Array<String>} hexColors
     *  The scheme colors to be shown in this color slider element, as 6-digit
     *  hexadecimal color values (RRGGBB).
     *
     * @param {Number} defaultIndex
     *  The default position (zero-based button index) of the color slider
     *  handle element.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Group base class.
     */
    var ColorSlider = Group.extend({ constructor: function (hexColors, defaultIndex, initOptions) {

        // self reference
        var self = this;

        // colorSlider handle
        var handleNode = $('<div class="color-slider-handle">');

        // the index of the active color (hovered by the handle element)
        var currentIndex = defaultIndex;

        // range definitions to detect to color under the handle. Same order as in colorpicker.js SCHEME_COLOR_DEFINITIONS.
        var colorFieldRanges = _.times(hexColors.length, function (index) {
            return { first: colorFieldWidth * index + marginLeft, last: colorFieldWidth * (index + 1) + marginLeft };
        });

        // base constructor ---------------------------------------------------

        Group.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         *  Sets the current color according to the passed position, and
         *  tiggers a 'change:color' event.
         */
        function setColorAtPosition(posX) {

            var newIndex = Utils.findFirstIndex(colorFieldRanges, function (range) {
                return (posX >= range.first) && (posX < range.last);
            });

            // show the corresponding scheme color row in the mobile color picker
            if ((newIndex >= 0) && (newIndex !== currentIndex)) {
                currentIndex = newIndex;
                self.trigger('change:color', currentIndex);
            }
        }

        /**
         *  Moves the handle to a given color number. The handleBorderWidth is
         *  included in the calculation as a offset.
         */
        function moveHandleToColor(index) {
            var range = colorFieldRanges[index];
            handleNode.css('left', (range.first - handleBorderWidth) + 'px');
        }

        /**
         *  Event handling for drag&drop and tapping.
         */
        var trackingHandler = (function () {

            // needed to correct an offset when marinLeft and handleBorderWidth are not equal
            var adjustHandleOffset = marginLeft - handleBorderWidth;
            // limit the handle leftwards
            var handleStartPos = adjustHandleOffset;
            // limit the handle rightwards
            var handleEndPos = colorFieldWidth * (hexColors.length - 1) + adjustHandleOffset;

            // used to show that the handle is moved via tap on the slider
            var tapMode = false;
            // offset in the dragged element to prevent a jumping handle
            var xOffset = null;

            function trackingStartHandler(event) {
                // to prevent a delayed click after touchstart - necessary due to prevent.default in tracking:move
                event.preventDefault();
                // case: drag&drop - when start target is the handle
                if (handleNode.is(event.target)) {
                    // offset in the handle: pointerposition in the handle remains the same while dragged
                    xOffset = event.pageX - handleNode[0].offsetLeft;
                    // flag that the slider can be dragged
                    tapMode = false;
                    // remove the transition for a delay free drag&drop
                    handleNode.removeClass('transition');
                } else {
                    // case: tap on the slider to select a color
                    tapMode = true;
                }
            }

            function trackingMoveHandler(event) {
                // to prevent events on iOS which triggered under the menu
                event.preventDefault();
                // only move the handle in case of drag&drop
                if (!tapMode) {
                    // set the handle to new position on the DOM
                    handleNode.css('left', (event.pageX - xOffset) + 'px');
                    // restrict the handle to the start position
                    if (handleNode[0].offsetLeft < handleStartPos) {
                        handleNode.css('left', handleStartPos + 'px');
                    }
                    // restrict the handle to the end position
                    if (handleNode[0].offsetLeft > handleEndPos) {
                        handleNode.css('left', handleEndPos + 'px');
                    }
                    // set the color which is under the center from the handle
                    setColorAtPosition(handleNode[0].offsetLeft + (handleNode[0].offsetWidth / 2));
                }
            }

            function trackingCancelHandler() {
                // add a transition to animate the handle movment on tapping and quantizise
                handleNode.addClass('transition');
            }

            function trackingEndHandler(event) {
                trackingCancelHandler(event);
                if (tapMode) { setColorAtPosition(event.pageX - self.getNode().offset().left); }
                moveHandleToColor(currentIndex);
            }

            return function (event) {
                switch (event.type) {
                    case 'tracking:start':
                        trackingStartHandler(event);
                        break;
                    case 'tracking:move':
                        trackingMoveHandler(event);
                        break;
                    case 'tracking:end':
                        trackingEndHandler(event);
                        break;
                    case 'tracking:cancel':
                        trackingCancelHandler(event);
                        break;
                }
            };
        }());

        // public methods -----------------------------------------------------

        /**
         * Moves the handle to the specified color, and triggers a
         * 'change:color' event.
         *
         * @param {Number} index
         *  The zero-based index of the color to be activated.
         *
         * @returns {ColorSlider}
         *  A reference to this instance.
         */
        this.moveHandle = function (index) {
            if (index !== currentIndex) {
                // no animation is needed here
                handleNode.removeClass('transition');
                moveHandleToColor(index);
                // updates the currentColor
                currentIndex = index;
                this.trigger('change:color', currentIndex);
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // create the HTML mark-up of this control
        (function () {
            var markup = '<div class="color-slider">';
            hexColors.forEach(function (hexRGB) {
                markup += '<div class="color-slider-field" style="background:#' + hexRGB + ';"></div>';
            });
            markup += '</div>';

            self.getNode().addClass('color-slider-container').append(markup, handleNode);
            moveHandleToColor(currentIndex);
        }());

        // create the event listeners for drag&drop and tapping interaction
        Tracking.enableTracking(this.getNode());
        this.getNode().on('tracking:start tracking:move tracking:end', trackingHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = self = handleNode = null;
        });

    } }); // class ColorSlider

    // exports ================================================================

    return ColorSlider;

});
