/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/bordermodepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/radiolist',
    'gettext!io.ox/office/editframework/main'
], function (Utils, RadioList, gt) {

    'use strict';

    // all available entries for the border picker control
    var BORDER_ENTRIES = [

        { name: 'left',    value: 'left',    icon: 'docs-border-l', label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Left border') },
        { name: 'right',   value: 'right',   icon: 'docs-border-r', label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Right border') },
        { name: 'top',     value: 'top',     icon: 'docs-border-t', label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Top border') },
        { name: 'bottom',  value: 'bottom',  icon: 'docs-border-b', label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Bottom border') },
        { name: 'insidev', value: 'insidev', icon: 'docs-border-v', label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Inner vertical borders') },
        { name: 'insideh', value: 'insideh', icon: 'docs-border-h', label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Inner horizontal borders') },

        { name: 'none',               value: Utils.makeSet(['left', 'right', 'insidev', 'top', 'bottom', 'insideh'], false), icon: 'docs-border',        label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'No borders') },

        { name: 'outer',              value: Utils.makeSet(['left', 'right', 'top', 'bottom']),                              icon: 'docs-border-tblr',   label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Outer borders') },
        { name: 'inner',              value: Utils.makeSet(['insidev', 'insideh']),                                          icon: 'docs-border-hv',     label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Inner borders') },
        { name: 'all',                value: Utils.makeSet(['left', 'right', 'insidev', 'top', 'bottom', 'insideh']),        icon: 'docs-border-tbhlrv', label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'All borders') },
        { name: 'left-right',         value: Utils.makeSet(['left', 'right']),                                               icon: 'docs-border-lr',     label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Left and right borders') },
        { name: 'top-bottom',         value: Utils.makeSet(['top', 'bottom']),                                               icon: 'docs-border-tb',     label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Top and bottom borders') },
        { name: 'left-right-insidev', value: Utils.makeSet(['left', 'right', 'insidev']),                                    icon: 'docs-border-lrv',    label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'All vertical borders') },
        { name: 'top-bottom-insideh', value: Utils.makeSet(['top', 'bottom', 'insideh']),                                    icon: 'docs-border-tbh',    label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'All horizontal borders') },
        { name: 'outer-insidev',      value: Utils.makeSet(['left', 'right', 'top', 'bottom', 'insidev']),                   icon: 'docs-border-tblrv',  label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Inner vertical and outer borders') },
        { name: 'outer-insideh',      value: Utils.makeSet(['left', 'right', 'top', 'bottom', 'insideh']),                   icon: 'docs-border-tbhlr',  label: /*#. in paragraphs and table cells */ gt.pgettext('borders', 'Inner horizontal and outer borders') }
    ];

    // class BorderModePicker =================================================

    /**
     * A drop-down list control with different settings for single or multiple
     * components supporting outer and inner border lines. The list entries are
     * divided into two sections. The first section contains entries for single
     * border lines that can be toggled. The second section contains list
     * entries for complex border settings for multiple component borders.
     *
     * The method 'BorderModePicker.setValue()' expects object values with the
     * optional properties 'top', 'bottom', 'left', 'right', insideh', and
     * 'insidev' of type Boolean or Null specifying the visibility state of the
     * respective border line. The property value null represents an ambiguous
     * state for the border line. Additionally, the method accepts an options
     * map as second parameter. The two options 'showInsideHor' and
     * 'showInsideVert' will override the respective options passed to the
     * constructor of this instance.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class RadioList.
     *  Additionally, the following options are supported:
     *  @param {Boolean|Function} [initOptions.showInsideHor=false]
     *      If set to true, the drop-down list will provide list entries for
     *      horizontal inner borders between multiple components that are
     *      stacked vertically. If set to a function, the visibility of the
     *      affected list items will be evaluated dynamically every time this
     *      control will be updated, according to the return value.
     *  @param {Boolean|Function} [initOptions.showInsideVert=false]
     *      If set to true, the drop-down list will provide list entries for
     *      vertical inner borders between multiple components that are located
     *      side-by-side. If set to a function, the visibility of the affected
     *      list items will be evaluated dynamically every time this control
     *      will be updated, according to the return value.
     */
    function BorderModePicker(initOptions) {

        var // self reference
            self = this,

            // whether list entries with inner horizontal borders will be shown
            showInsideHor = Utils.getFunctionOption(initOptions, 'showInsideHor', Utils.getBooleanOption(initOptions, 'showInsideHor', false)),

            // whether list entries with inner vertical borders will be shown
            showInsideVert = Utils.getFunctionOption(initOptions, 'showInsideVert', Utils.getBooleanOption(initOptions, 'showInsideVert', false));

        // base constructor ---------------------------------------------------

        initOptions = Utils.extendOptions({
            icon: 'docs-border-tblr',
            tooltip: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Borders')
        }, initOptions);
        RadioList.call(this, Utils.extendOptions({
            dropDownVersion: { label: Utils.getStringOption(initOptions, 'tooltip') }
        }, initOptions, {
            itemMatcher: matchBorder,
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        function isInsideHor(value) {
            return (value === 'insideh') || (value.insideh === true);
        }

        function isInsideVert(value) {
            return (value === 'insidev') || (value.insidev === true);
        }

        /**
         * Returns whether to show list items with horizontal inner borders.
         */
        function showInsideHorItems(value) {
            return _.isFunction(showInsideHor) ? showInsideHor.call(self, value) : showInsideHor;
        }

        /**
         * Returns whether to show list items with vertical inner borders.
         */
        function showInsideVertItems(value) {
            return _.isFunction(showInsideVert) ? showInsideVert.call(self, value) : showInsideVert;
        }

        /**
         * Returns whether to show list items with both inner borders.
         */
        function showInsideBothItems(value) {
            return showInsideHorItems(value) && showInsideVertItems(value);
        }

        /**
         * Returns whether the passed borders matches one of the drop-down list
         * entries, used for updating the selection in the drop-down list.
         *
         * @param {Object} value
         *  The border value to be matched against a list entry.
         *
         * @param {Object|String} entryValue
         *  The value of one of the list entries of this control.
         *
         * @returns {Boolean|Null}
         *  Whether the border value matches the list entry. All properties
         *  contained in the list entry must be equal in the border value.
         *  Other existing properties of the border value are ignored. If the
         *  passed border value contains a single property only (single border
         *  selectors from the upper part of the list), this function returns
         *  null for all list items representing other border lines than the
         *  passed border line.
         */
        function matchBorder(value, entryValue) {

            // skip single border line values
            if (_.isString(value)) { return null; }

            // return false for non-object values (deselect all entries in ambiguous state)
            if (!_.isObject(value)) { return false; }

            // update single-border list items
            if (_.isString(entryValue)) {
                // return false for null values
                return (entryValue in value) && (value[entryValue] === true);
            }

            // update compound-border list items
            return _.all(entryValue, function (visible, propName) {
                // use false for missing properties in value, but do not match existing null values
                return visible === ((propName in value) ? value[propName] : false);
            });
        }

        /**
         * Updates the icon in the drop-down menu button according to the
         * current value of this control.
         */
        function updateHandler(value) {

            // reset caption to default settings in undetermined state
            if (!_.isObject(value) && !_.isString(value)) {
                self.setCaption(self.getOptions());
                return;
            }

            // ignore single border values (after activating a single-border list item)
            if (_.isObject(value)) {
                var icon = '';
                if (value.top) { icon += 't'; }
                if (value.bottom) { icon += 'b'; }
                if (value.insideh) { icon += 'h'; }
                if (value.left) { icon += 'l'; }
                if (value.right) { icon += 'r'; }
                if (value.insidev) { icon += 'v'; }
                if (icon.length > 0) { icon = '-' + icon; }
                icon = 'docs-border' + icon;
                self.setIcon(icon);
            }
        }

        // initialization -----------------------------------------------------

        // prepare the sections for the toggle buttons and the complex border buttons
        self.createMenuSection('single').createMenuSection('none').createMenuSection('compound');

        // insert all supported list items
        BORDER_ENTRIES.forEach(function (entry) {
            var section = _.isString(entry.value) ? 'single' : (entry.name === 'none') ? 'none' : 'compound';
            var visible = isInsideHor(entry.value) ? (isInsideVert(entry.value) ? showInsideBothItems : showInsideHorItems) : (isInsideVert(entry.value) ? showInsideVertItems : true);
            this.createOptionButton(entry.value, { section: section, icon: entry.icon, label: entry.label, dataValue: entry.name, visible: visible });
        }, this);

        // register custom update handler to select the correct border icon
        this.registerUpdateHandler(updateHandler);

        // destroy all members on destruction
        this.registerDestructor(function () {
            self = showInsideHor = showInsideVert = null;
        });

    } // class BorderModePicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: BorderModePicker });

});
