/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 */

define('io.ox/office/editframework/model/themecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/simplemap',
    'io.ox/office/baseframework/model/modelobject'
], function (Utils, SimpleMap, ModelObject) {

    'use strict';

    // the attributes of the built-in default theme
    var BUILTIN_THEME_ATTRS = {
        colorScheme: {
            accent1: '4f81bd',
            accent2: 'c0504d',
            accent3: '9bbb59',
            accent4: '8064a2',
            accent5: '4bacc6',
            accent6: 'f79646',
            text1: '000000',
            text2: '1f497d',
            light1: 'ffffff',
            light2: 'eeece1',
            dark1: '000000',
            dark2: '1f497d',
            background1: 'ffffff',
            background2: 'eeece1',
            hyperlink: '0000ff',
            followedHyperlink: '800080'
        },
        fontScheme: {
            '+mj-lt': 'Arial',
            '+mn-lt': 'Arial',
            'Arial Narrow': 'Calibri' //fix for Bug 46782
        }
    };

    // the name of the default theme
    var DEFAULT_THEME_NAME = 'Larissa';

    // private global functions ===============================================

    /**
     * Generates a unique map key from a theme's name and target.
     *
     * @param {String} name
     *  The name of the theme.
     *
     * @param {String} [target]
     *  The target of the theme. If omitted, an empty string is used.
     *
     * @returns {String}
     *  A unique map key from a theme's name and target.
     */
    function generateKey(name, target) {
        return JSON.stringify([name, target || '']);
    }

    // class Theme ============================================================

    /**
     * Contains the definitions of a single theme in a document.
     *
     * @constructor
     *
     * @param {Object} attributes
     *  The theme attributes as received from the 'insertTheme' operation.
     *
     * @param {String} name
     *  The name of the theme.
     *
     * @param {String} [target]
     *  The optional target of the theme.
     */
    function Theme(attributes, name, target) {

        // the color scheme of this theme
        var colorScheme = _.copy(Utils.getObjectOption(attributes, 'colorScheme', {}), true);
        var fontScheme  = _.copy(Utils.getObjectOption(attributes, 'fontScheme',  {}), true);

        // methods ------------------------------------------------------------

        /**
         * Getting the theme name.
         *
         * @returns {String}
         *  The name of the theme.
         */
        this.getName = function () {
            return name;
        };

        /**
         * Getting the theme target.
         *
         * @returns {String}
         *  The name of the target for the theme. If not specified, an
         *  empty string is returned.
         */
        this.getTarget = function () {
            return target || '';
        };

        /**
         * Generates a unique map key for this theme's name and target.
         *
         * @returns {String}
         *  A unique map key for this theme's name and target.
         */
        this.getKey = function () {
            return generateKey(this.getName(), this.getTarget());
        };

        /**
         * Returns whether this theme contains scheme color definitions.
         */
        this.hasSchemeColors = function () {
            return !_.isEmpty(colorScheme);
        };

        /**
         * Returns the RGB color value of the specified scheme color.
         *
         * @param {String} name
         *  The name of the scheme color.
         *
         * @param {String} [defRGB]
         *  The default RGB value, if the specified scheme color does not
         *  exist. If omitted, null will be returned in that case.
         *
         * @returns {String|Null}
         *  The RGB value of the scheme color as hexadecimal string, if
         *  existing, otherwise null.
         */
        this.getSchemeColor = function (name, defRGB) {
            var color = colorScheme[name];
            return (typeof color === 'string') ? color : (defRGB || null);
        };

        /**
         * Returns the font names value of the specified font key.
         *
         * @param {String} name
         *  The name of the font.
         *
         * @returns {String|Null}
         *  The real fontname of the assigned font,
         *  otherwise the assigned fontname will be returned.
         */
        this.getFontName = function (name) {
            var fontName = fontScheme[name];
            return (typeof fontName === 'string') ? fontName : name;
        };

    } // class Theme

    // class ThemeCollection ==================================================

    /**
     * Contains the definitions of all themes in a document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     */
    var ThemeCollection = ModelObject.extend({ constructor: function (docModel) {

        // themes, mapped by identifier
        var themeMap = new SimpleMap();

        // the default theme names for each target
        var targetDefaults = {};

        // default theme (first inserted theme)
        var defaultTheme = new Theme(BUILTIN_THEME_ATTRS, DEFAULT_THEME_NAME);

        // whether the default theme is still the built-in theme
        var isBuiltInTheme = true;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Adds a new theme to this container. An existing theme with the
         * specified name will be replaced.
         *
         * @param {String} name
         *  The name of of the new theme.
         *
         * @param {Object} attributes
         *  The formatting attributes of the theme.
         *
         * @param {String} [target]
         *  The optional name of a target for the new theme.
         *
         * @returns {ThemeCollection}
         *  A reference to this instance.
         */
        this.insertTheme = function (name, attributes, target) {

            // create a theme object with all passed attributes
            var theme = new Theme(attributes, name, target);

            // insert the new theme into the map
            themeMap.insert(theme.getKey(), theme);

            // get resulting target name (specific default value for missing targets)
            target = theme.getTarget();

            // set first inserted theme as new default theme
            if (isBuiltInTheme && !target) {
                defaultTheme = theme;
                isBuiltInTheme = false;
            }

            // saving the first theme for a specified target as default theme
            if (target && !targetDefaults[target]) {
                targetDefaults[target] = name;
            }

            // notify listeners
            this.trigger('insert:theme', name, target);
            return this;
        };

        /**
         * Returns, whether the target defaults collection contains an entry
         * for the specified target.
         *
         * @param {String} target
         *  A target string to specify the theme's target.
         *
         * @returns {Boolean}
         *  Whether there is a registered default theme for the specified target.
         */
        this.hasDefaultTheme = function (target) {
            return !!targetDefaults[target];
        };

        /**
         * Returns a single theme from this collection.
         *
         * @param {String} [name]
         *  The name of the theme to be returned. If omitted or not existing,
         *  the 'current' theme will be returned.
         *
         * @param {String|String[]} [target]
         *  The optional name of the target of the theme to be returned. Or an
         *  array of targets, that can be used to find a valid theme. This
         *  array is required to specify the slide inheritance chain in
         *  presentation application.
         *
         * @returns {Theme}
         *  The specified theme, or the current theme. If no theme has been
         *  inserted, returns the built-in default theme.
         */
        this.getTheme = function (name, target) {

            var // the theme object
                theme = null;

            // resolve missing theme name by target
            if (!name && target) {
                if (_.isString(target)) {
                    name = targetDefaults[target];
                } else if (_.isArray(target)) {
                    Utils.iterateArray(target, function (oneTarget) {
                        if (oneTarget === '' && !isBuiltInTheme) { // this can be a valid value in the target array
                            return Utils.BREAK; // forcing usage of default theme  (that is not the built in theme)
                        } else {
                            if (targetDefaults[oneTarget]) {
                                target = oneTarget; // using string instead of array
                                name = targetDefaults[oneTarget];
                                return Utils.BREAK;
                            }
                        }
                    });
                }
            }

            // return global default theme, if no name is known for the theme
            if (!name) { return defaultTheme; }

            // handling a specified name
            if (!target || _.isString(target)) {
                // get a theme from the collection, fall-back to default theme
                theme = themeMap.get(generateKey(name, target), defaultTheme);
            } else if (_.isArray(target)) {
                // getting the first existing theme
                Utils.iterateArray(target, function (oneTarget) {
                    var oneTheme = themeMap.get(generateKey(name, oneTarget), null);
                    if (oneTheme) {
                        theme = oneTheme; // found a theme with valid name and target
                        return Utils.BREAK;
                    }
                });

                // theme not found -> using default theme
                if (!theme) { theme = defaultTheme; }
            }

            return theme;
        };

        // initialization -----------------------------------------------------

        // register the built-in default theme
        themeMap.insert(defaultTheme.getKey(), defaultTheme);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = themeMap = defaultTheme = null;
        });

    } }); // class ThemeCollection

    // exports ================================================================

    return ThemeCollection;

});
