/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

/* eslint no-new: 0 */

define('io.ox/office/editframework/app/fileactions', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/baseframework/app/extensionregistry',
    'io.ox/office/baseframework/app/applicationlauncher',
    'io.ox/office/tk/utils/deferredutils',
    'gettext!io.ox/office/editframework/actions'
], function (ext, links, DriveUtils, Utils, IO, ExtensionRegistry, ApplicationLauncher, DeferredUtils, gt) {

    'use strict';

    // static private functions ===============================================

    /**
     * Returns whether the files in the folder described by the passed baton
     * object are considered to be editable.
     * The trash folder for example will be treated as read-only folder.
     *
     * @param {Object} baton
     *  The baton object providing access to a folder.
     *
     * @returns {jQuery.Promise}
     *  A Promise that will resolve with a Boolean value,
     *  specifying whether the folder is considered editable.
     */
    function isEditableFolder(baton) {
        // no baton.data, no trash folder (e.g. main portal)
        if (!baton.data.id) { return $.when(true); }

        return DriveUtils.getTrashState(baton.data.folder_id).then(function (state) {
            return $.when(!state);
        });
    }

    /**
     * Returns whether the files in the folder described by the passed baton
     * object are considered to be editable and the current version.
     *
     * @param {Object} baton
     *  The baton object providing access to a folder.
     *
     * @returns {jQuery.Promise}
     *  A Promise that will resolve with a Boolean value,
     *  specifying whether the folder is considered editable and
     *  the file represents the current version.
     */
    function isEditableFolderAndCurrentVersion(baton) {

        // check if file is editable
        var promise = isEditableFolder(baton);

        // check if file is the current version
        return promise.then(function (canEdit) {
            return canEdit && DriveUtils.getCurrentFileVersionState(baton.data, { cache: true });
        });
    }

    /**
     * Disables the passed extension point for invalid files (all files names
     * with a specific error suffix in their extension).
     *
     * @param {String} actionId
     *  The identifier of the action to be disabled.
     */
    function disableActionForInvalidFiles(actionId, capabilities) {
        var opt = {
            id: 'disable_action',
            index: 'first',
            requires: function (data) {
                var fileName = data.baton.data.filename;
                if (fileName && ExtensionRegistry.isError(fileName)) {
                    data.stopPropagation();
                    return false;
                }
            }
        };
        if (capabilities) {
            //workaround for Bug 47678
            opt.capabilities = capabilities;
        }

        new links.Action(actionId, opt);
    }

    /**
     *  e.collection.has('one', 'read', 'modify') is the normal appsuite code
     *  e.baton.models[0].hasWritePermissions() is the new code for sharing
     *  Bug 40846
     */
    function hasWritePermissions(e) {
        if (e.collection.has('one', 'read')) {
            if (e.collection.has('modify')) {
                return true;
            }
            if (e.baton.models && e.baton.models.length) {
                return e.baton.models[0].hasWritePermissions();
            }
        }
        return false;
    }

    function getRunningApp(fileDesc, moduleName) {
        if (!fileDesc.filename || !ExtensionRegistry.isEditable(fileDesc.filename, moduleName)) { return null; }
        var app = ApplicationLauncher.getRunningApp(moduleName, fileDesc);
        return (app && app.isImportSucceeded()) ? app : null;
    }

    function overwriteExtPointForRunningApp(ref, moduleName, id, index, action) {
        new links.Action(ref, {
            id: moduleName + '/' + id,
            index: index,
            requires: function (e) {
                var filename = e.context.filename;
                return $.when(!!filename && ExtensionRegistry.isEditable(filename, moduleName));
            },
            filter: function (file) {
                return !!getRunningApp(file, moduleName);
            },
            action: function (baton) {
                var runningApp = getRunningApp(baton.data, moduleName);
                if (runningApp) { action.call(this, runningApp); }
            }
        });
    }

    // static class FileActions ===============================================

    var FileActions = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates all required actions and links in the OX Drive application for a
     * specific OX Documents application.
     *
     * @param {String} appBaseName
     *  The base name of the application (name of the application source code
     *  directory).
     *
     * @param {Object} options
     *  Optional parameters:
     *  @param {String} options.newDocumentLabel
     *      The string shown for the 'Create new document' action link inserted
     *      into the tool bar of OX Drive.
     *  @param {Number} options.newDocumentIndex
     *      The index of the 'Create new document' action link that will be
     *      inserted into the tool bar of OX Drive. The 'Create new document'
     *      action links of the various OX Document applications will be
     *      ordered increasing by this index.
     */
    FileActions.create = function (appBaseName, options) {

        var // root for extension points in OX Drive application
            ACTION_POINT = appBaseName,
            // the module name of the edit application
            MODULE_NAME = 'io.ox/office/' + appBaseName;

        // private methods ----------------------------------------------------

        /**
         * Launches a new OX Documents application with the passed options.
         */
        function launchApplication(launchOptions) {
            ox.launch(MODULE_NAME + '/main', launchOptions);
        }

        // initialization -----------------------------------------------------

        // Creates a new empty document in the current folder of the OX Drive application.
        if (ExtensionRegistry.supportsEditMode(appBaseName)) {
            new links.Action(ACTION_POINT + '-newblank', {
                requires: function (e) {
                    //Bug 40479
                    //it is important to call getData before of "can" or "is" calls
                    //DriveUtils does this already, but in this situation we dont have the id of the folder yet
                    return e.baton.app.folder.getData().then(function (data) {
                        if (_.isObject(data) && _.isString(data.id)) {
                            return DriveUtils.getCreateState(data.id).then(function (create) {
                                if (create) {
                                    return DriveUtils.getTrashState(data.id).then(function (state) {
                                        return !state;
                                    });
                                } else {
                                    return false;
                                }
                            });
                        } else {
                            // Bug 46390
                            // In case we cannot retrieve any data from the folder
                            // we cannot support the edit mode.
                            return false;
                        }
                    });
                },
                action: function (baton) {
                    launchApplication({ action: 'new', target_folder_id: baton.app.folder.get() });
                }
            });
        }

        var editAction = function (baton) {
            var promise = DriveUtils.getFile(baton.data);
            promise.done(function (data) {
                var fileName = data.filename;
                if (fileName && ExtensionRegistry.isNative(fileName)) {
                    launchApplication({ action: 'load', file: data });
                } else {
                    launchApplication({ action: 'convert', templateFile: data, preserveFileName: true });
                }
            });
            promise.fail(function (error) {
                require(['io.ox/core/notifications', 'io.ox/office/baseframework/utils/errorcode', 'io.ox/office/baseframework/utils/errormessages']).done(function (CoreNotification, ErrorCode, ErrorMessages) {
                    CoreNotification.yell('error', ErrorMessages.getMessageData(new ErrorCode(error)).message);
                });
            });
        };
        var templAction = function (baton) {
            launchApplication({ action: 'convert', template: true, target_folder_id: DriveUtils.getStandardDocumentsFolderId(), templateFile: baton.data, target_filename: gt('unnamed') });
        };

        // Loads an existing document from the current file in the OX Drive application.
        // If the document requires conversion to a supported file format, creates the
        // converted file and loads it.
        new links.Action(ACTION_POINT + '-edit', {
            requires: function (e) {
                var filename = e.context.filename;
                if (filename && hasWritePermissions(e) && ExtensionRegistry.isEditable(filename, MODULE_NAME) && !ExtensionRegistry.isTemplate(filename, MODULE_NAME) && ExtensionRegistry.isNative(filename, MODULE_NAME)) {
                    // prefetch application source code
                    IO.prefetchModuleSource(MODULE_NAME);
                    return isEditableFolderAndCurrentVersion(e.baton);
                }
                return $.when(false);
            },
            filter: function (obj) {
                return !!obj.filename && ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
            },
            action: editAction
        });

        new links.Action(ACTION_POINT + '-new-fromtemplate', {
            requires: function (e) {

                // workaround for Guest/anonymous
                // guest -> no new-from-template button!
                // TODO: make param newfolderid, this could happen in current folder, if there are create rights for the guest
                if (DriveUtils.isGuest()) {
                    return $.when(false);
                }

                var filename = e.context.filename;
                if (filename && e.collection.has('one', 'read') && !ExtensionRegistry.isError(filename) && ExtensionRegistry.isTemplate(filename, MODULE_NAME)) {
                    return isEditableFolderAndCurrentVersion(e.baton);
                }
                return $.when(false);
            },
            filter: function (obj) {
                return ExtensionRegistry.isTemplate(obj.filename, MODULE_NAME);
            },
            action: templAction
        });

        //double click or enter in Drive
        new links.Action('io.ox/files/actions/default', {
            id: MODULE_NAME,
            //requires is not called for 'io.ox/files/actions/default' so we must use filter function!
            filter: function (obj) {
                var filename = obj.filename;
                if (!filename || DriveUtils.isTrash(obj.folder_id)) { return false; }

                if (ExtensionRegistry.isTemplate(filename, MODULE_NAME)) { return true; }

                return DriveUtils.canModify(obj) && ExtensionRegistry.isNative(filename) && ExtensionRegistry.isEditable(filename, MODULE_NAME);
            },
            action: function (baton) {
                var filename = baton.data.filename;
                if (filename && ExtensionRegistry.isTemplate(filename, MODULE_NAME)) {
                    templAction(baton);
                } else {
                    editAction(baton);
                }
            }
        });

        // Creates a new document as copy of the current file in the OX Drive application.
        if (ExtensionRegistry.supportsEditMode(appBaseName)) {

            ////////////////////////////////////////////////////////////////////////

            var editAsNew = function (ref, requires) {
                new links.Action(ACTION_POINT + ref, {
                    requires: requires,
                    filter: function (obj) {
                        return obj.filename && ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
                    },
                    action: function (baton) {
                        var filename = baton.data.filename;
                        var convert = filename && ExtensionRegistry.isConvertible(filename);
                        launchApplication({ action: convert ? 'convert' : 'new', target_folder_id: baton.data.folder_id, templateFile: baton.data, preserveFileName: convert });
                    }
                });
            };

            var isExtraEditAsNew = function (e) {
                // workaround for Guest/anonymous
                // guest -> no my files -> no edit-as-new button!
                var filename = e.context.filename;
                if (!filename || DriveUtils.isGuest()) { return false; }
                return e.collection.has('one', 'read') && !ExtensionRegistry.isError(filename) && !ExtensionRegistry.isGuardEncrypted(filename) && !ExtensionRegistry.isTemplate(filename, MODULE_NAME) && ExtensionRegistry.isEditable(filename, MODULE_NAME);
            };

            var isNormalEditAsNew = function (e) {
                var filename = e.context.filename;
                return !!filename && !ExtensionRegistry.isGuardEncrypted(filename) && hasWritePermissions(e) && ExtensionRegistry.isNative(filename, MODULE_NAME);
            };

            editAsNew('-edit-asnew', function (e) {
                if (isExtraEditAsNew(e) && isNormalEditAsNew(e)) {
                    return isEditableFolderAndCurrentVersion(e.baton);
                }
            });

            editAsNew('-edit-asnew-hi', function (e) {
                if (isExtraEditAsNew(e) && !isNormalEditAsNew(e)) {
                    return isEditableFolderAndCurrentVersion(e.baton);
                }
            });

            ////////////////////////////////////////////////////////////////////////

            new links.Action(ACTION_POINT + '-edit-template', {
                requires: function (e) {
                    var filename = e.context.filename;
                    // template files cannot be edited directly, if they would need to be converted to another file format
                    if (filename && hasWritePermissions(e) && ExtensionRegistry.isTemplate(filename, MODULE_NAME) && ExtensionRegistry.isNative(filename, MODULE_NAME)) {
                        return isEditableFolderAndCurrentVersion(e.baton);
                    }
                    return $.when(false);
                },
                filter: function (obj) {
                    return obj.filename && ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
                },
                action: function (baton) {
                    var filename = baton.data.filename;
                    if (filename && ExtensionRegistry.isNative(filename)) {
                        launchApplication({ action: 'load', file: baton.data });
                    } else {
                        launchApplication({ action: 'convert', target_folder_id: baton.data.folder_id, templateFile: baton.data, preserveFileName: true });
                    }
                }
            });
        }

        // An action link in the main tool bar of the OX Drive application used
        // to create a new empty document.
        new links.ActionLink('io.ox/files/links/toolbar/default', {
            index: 200 + Utils.getIntegerOption(options, 'newDocumentIndex', 0),
            id: appBaseName + '-newblank',
            label: Utils.getStringOption(options, 'newDocumentLabel', ''),
            ref: ACTION_POINT + '-newblank'
        });

        function extend(id, label, prio, prioMobile, index, ref, section, customize) {
            var link = {
                id: appBaseName + id,
                index: index,
                prio: prio,
                mobile: prioMobile,
                label: label,
                title: label,
                ref: ACTION_POINT + ref
            };
            if (section) {
                link.section = section;
            }
            if (customize) {
                link.customize = customize;
            }

            //@Deprecated old halo view
            ext.point('io.ox/files/links/inline').extend(new links.Link(link));

            // define links for document actions in the OX Viewer toolbar.
            ext.point('io.ox/core/viewer/toolbar/links/drive').extend(new links.Link(link));
            ext.point('io.ox/core/viewer/toolbar/links/guardDrive').extend(new links.Link(link));

            // define links for document actions in the OX Drive toolbar.
            ext.point('io.ox/files/classic-toolbar/links').extend(new links.Link(link));

            // define links for document actions in the OX Viewer version drop-down.
            ext.point('io.ox/files/versions/links/inline').extend(new links.Link(link));
        }

        //      id                  label                       prio    prioMobile    index     ref                 section
        extend('-edit',            gt('Edit'),                 'hi',   'lo',          150,      '-edit',            'edit');
        extend('-newfromtemplate', gt('New from template'),    'hi',   'lo',          150,      '-new-fromtemplate');
        extend('-edittemplate',    gt('Edit template'),        'lo',   'lo',          160,      '-edit-template',   'edit');

        extend('-editasnew',       gt('Edit as new'),          'lo',   'lo',          160,      '-edit-asnew',      'edit');
        extend('-editasnewhi',     gt('Edit as new'),          'hi',   'lo',          150,      '-edit-asnew-hi',   'edit');

        if (!Utils.IOS) {
            overwriteExtPointForRunningApp('io.ox/files/actions/download', MODULE_NAME, '/download', 50, function (app) {
                app.download();
            });
        }

    };

    // static initialization ==================================================

    if (!DriveUtils.isGuest()) {
        //isguest check is a workaround for Bug 40480
        disableActionForInvalidFiles('io.ox/files/actions/add-to-portal', 'portal');
        disableActionForInvalidFiles('io.ox/files/actions/publish');
    }

    // exports ================================================================

    return FileActions;

});
