/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/imageutil', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/utils/driveutils',
    'settings!io.ox/office',
    'io.ox/office/tk/utils/deferredutils',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, IO, Dialogs, DriveUtils, Settings, DeferredUtils, gt) {

    'use strict';

    var CRC32_TABLE = [
        0x00000000, 0x77073096, 0xEE0E612C, 0x990951BA, 0x076DC419, 0x706AF48F, 0xE963A535, 0x9E6495A3,
        0x0EDB8832, 0x79DCB8A4, 0xE0D5E91E, 0x97D2D988, 0x09B64C2B, 0x7EB17CBD, 0xE7B82D07, 0x90BF1D91,
        0x1DB71064, 0x6AB020F2, 0xF3B97148, 0x84BE41DE, 0x1ADAD47D, 0x6DDDE4EB, 0xF4D4B551, 0x83D385C7,
        0x136C9856, 0x646BA8C0, 0xFD62F97A, 0x8A65C9EC, 0x14015C4F, 0x63066CD9, 0xFA0F3D63, 0x8D080DF5,
        0x3B6E20C8, 0x4C69105E, 0xD56041E4, 0xA2677172, 0x3C03E4D1, 0x4B04D447, 0xD20D85FD, 0xA50AB56B,
        0x35B5A8FA, 0x42B2986C, 0xDBBBC9D6, 0xACBCF940, 0x32D86CE3, 0x45DF5C75, 0xDCD60DCF, 0xABD13D59,
        0x26D930AC, 0x51DE003A, 0xC8D75180, 0xBFD06116, 0x21B4F4B5, 0x56B3C423, 0xCFBA9599, 0xB8BDA50F,
        0x2802B89E, 0x5F058808, 0xC60CD9B2, 0xB10BE924, 0x2F6F7C87, 0x58684C11, 0xC1611DAB, 0xB6662D3D,
        0x76DC4190, 0x01DB7106, 0x98D220BC, 0xEFD5102A, 0x71B18589, 0x06B6B51F, 0x9FBFE4A5, 0xE8B8D433,
        0x7807C9A2, 0x0F00F934, 0x9609A88E, 0xE10E9818, 0x7F6A0DBB, 0x086D3D2D, 0x91646C97, 0xE6635C01,
        0x6B6B51F4, 0x1C6C6162, 0x856530D8, 0xF262004E, 0x6C0695ED, 0x1B01A57B, 0x8208F4C1, 0xF50FC457,
        0x65B0D9C6, 0x12B7E950, 0x8BBEB8EA, 0xFCB9887C, 0x62DD1DDF, 0x15DA2D49, 0x8CD37CF3, 0xFBD44C65,
        0x4DB26158, 0x3AB551CE, 0xA3BC0074, 0xD4BB30E2, 0x4ADFA541, 0x3DD895D7, 0xA4D1C46D, 0xD3D6F4FB,
        0x4369E96A, 0x346ED9FC, 0xAD678846, 0xDA60B8D0, 0x44042D73, 0x33031DE5, 0xAA0A4C5F, 0xDD0D7CC9,
        0x5005713C, 0x270241AA, 0xBE0B1010, 0xC90C2086, 0x5768B525, 0x206F85B3, 0xB966D409, 0xCE61E49F,
        0x5EDEF90E, 0x29D9C998, 0xB0D09822, 0xC7D7A8B4, 0x59B33D17, 0x2EB40D81, 0xB7BD5C3B, 0xC0BA6CAD,
        0xEDB88320, 0x9ABFB3B6, 0x03B6E20C, 0x74B1D29A, 0xEAD54739, 0x9DD277AF, 0x04DB2615, 0x73DC1683,
        0xE3630B12, 0x94643B84, 0x0D6D6A3E, 0x7A6A5AA8, 0xE40ECF0B, 0x9309FF9D, 0x0A00AE27, 0x7D079EB1,
        0xF00F9344, 0x8708A3D2, 0x1E01F268, 0x6906C2FE, 0xF762575D, 0x806567CB, 0x196C3671, 0x6E6B06E7,
        0xFED41B76, 0x89D32BE0, 0x10DA7A5A, 0x67DD4ACC, 0xF9B9DF6F, 0x8EBEEFF9, 0x17B7BE43, 0x60B08ED5,
        0xD6D6A3E8, 0xA1D1937E, 0x38D8C2C4, 0x4FDFF252, 0xD1BB67F1, 0xA6BC5767, 0x3FB506DD, 0x48B2364B,
        0xD80D2BDA, 0xAF0A1B4C, 0x36034AF6, 0x41047A60, 0xDF60EFC3, 0xA867DF55, 0x316E8EEF, 0x4669BE79,
        0xCB61B38C, 0xBC66831A, 0x256FD2A0, 0x5268E236, 0xCC0C7795, 0xBB0B4703, 0x220216B9, 0x5505262F,
        0xC5BA3BBE, 0xB2BD0B28, 0x2BB45A92, 0x5CB36A04, 0xC2D7FFA7, 0xB5D0CF31, 0x2CD99E8B, 0x5BDEAE1D,
        0x9B64C2B0, 0xEC63F226, 0x756AA39C, 0x026D930A, 0x9C0906A9, 0xEB0E363F, 0x72076785, 0x05005713,
        0x95BF4A82, 0xE2B87A14, 0x7BB12BAE, 0x0CB61B38, 0x92D28E9B, 0xE5D5BE0D, 0x7CDCEFB7, 0x0BDBDF21,
        0x86D3D2D4, 0xF1D4E242, 0x68DDB3F8, 0x1FDA836E, 0x81BE16CD, 0xF6B9265B, 0x6FB077E1, 0x18B74777,
        0x88085AE6, 0xFF0F6A70, 0x66063BCA, 0x11010B5C, 0x8F659EFF, 0xF862AE69, 0x616BFFD3, 0x166CCF45,
        0xA00AE278, 0xD70DD2EE, 0x4E048354, 0x3903B3C2, 0xA7672661, 0xD06016F7, 0x4969474D, 0x3E6E77DB,
        0xAED16A4A, 0xD9D65ADC, 0x40DF0B66, 0x37D83BF0, 0xA9BCAE53, 0xDEBB9EC5, 0x47B2CF7F, 0x30B5FFE9,
        0xBDBDF21C, 0xCABAC28A, 0x53B39330, 0x24B4A3A6, 0xBAD03605, 0xCDD70693, 0x54DE5729, 0x23D967BF,
        0xB3667A2E, 0xC4614AB8, 0x5D681B02, 0x2A6F2B94, 0xB40BBE37, 0xC30C8EA1, 0x5A05DF1B, 0x2D02EF8D
    ];

    var BASE64_CHAR_TABLE = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=';

    var IMAGE_EXTENSIONS = {
        jpg:  'image/jpeg',
        jpe:  'image/jpeg',
        jpeg: 'image/jpeg',
        png:  'image/png',
        gif:  'image/gif',
        bmp:  'image/bmp'
    };

    var RE_IMAGE_URL = /^(?:([a-z]+):)?(\/{0,3})([0-9.\-a-z]+)(?::(\d+))?(?:\/([^?#]*))?(?:\?([^#]*))?(?:#(.*))?$/i;

    // private static functions ===============================================

    function toHex32(n) {
        if (n < 0) { n += 0x100000000; }
        return ('0000000' + n.toString(16).toUpperCase()).substr(-8);
    }

    function crc32(n, iCRC) {
        return (iCRC >>> 8) ^ CRC32_TABLE[(iCRC & 0xFF) ^ n];
    }

    /**
     * calculates crc32 value of base64 encoded data (crc32 is then calculated from the decoded data)
     *
     * @param {String} data
     *  The String must contain base64 encoded data. The data length is determined by data.length - offset;
     *  It is not necessary to include '=' padding bytes at the end, the String size is sufficient.
     *
     * @param {Number} offset
     *  Specifies offset to the beginning of base64 encoded data.
     *
     * @returns {Number|Null}
     *  Null is returned, if no base64 encoded data could be found in the data
     *  string. Otherwise the CRC32 value is returned.
     *
     */
    function getCRC32fromBase64(data, offset) {

        var dataLength = data.length - offset;

        // data length must be at least two bytes to encode one character (if no pad bytes are used)
        if (dataLength < 2) { return null; }

        // taking care of padbytes
        if (data[offset + dataLength - 1] === '=') { dataLength--; }
        if (data[offset + dataLength - 1] === '=') { dataLength--; }

        var groups = dataLength >> 2;
        var hangover = dataLength - (groups << 2);

        var bits = 0, n1, n2, n3, n4, iCRC = 0xffffffff, i = offset;

        while (groups--) {
            n1 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n2 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n3 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n4 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            bits = (n1 << 18) | (n2 << 12) | (n3 << 6) | n4;
            iCRC = crc32((bits >> 16) & 0xff, iCRC);
            iCRC = crc32((bits >> 8) & 0xff, iCRC);
            iCRC = crc32(bits & 0xff, iCRC);
        }
        // taking care of pad bytes
        if (hangover === 3) {
            n1 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n2 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n3 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            bits = (n1 << 12) | (n2 << 6) | n3;
            iCRC = crc32((bits >> 10) & 0xff, iCRC);
            iCRC = crc32((bits >> 2) & 0xff, iCRC);
        } else if (hangover === 2) {
            n1 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n2 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            bits = (n1 << 6) | n2;
            iCRC = crc32((bits >> 4) & 0xff, iCRC);

        // shouldn't be 1, we are padding the least significant two bits with zero
        } else if (hangover === 1) {
            n1 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            iCRC = crc32((bits << 2) & 0xff, iCRC);
        }
        return iCRC ^ 0xFFFFFFFF;
    }

    function insertDataUrl(app, dataUrl, fileName) {
        var offset = dataUrl && dataUrl.indexOf('base64,') + 7,
            crc32 = null,
            extension = null;

        if (!_.isString(fileName) || (fileName.length < 1)) {
            fileName = 'image.img';
        }

        crc32 = getCRC32fromBase64(dataUrl, offset);

        // if no crc32 value could be determined, the loading process must be stopped now.
        if (!crc32) { return $.Deferred().reject(); }

        crc32 = toHex32(crc32);
        extension = fileName.substring(fileName.lastIndexOf('.') + 1);

        var params = {
            add_crc32: crc32,
            add_ext: extension,
            alt_filename: fileName,
            add_filedata: dataUrl,
            requestdata: 'hallo'
        };

        return app.sendActionRequest('addfile', params);
    }

    function insertFile(app, file) {

        return IO.readClientFileAsDataUrl(file).then(function (dataUrl) {
            return insertDataUrl(app, dataUrl, file.name);
        });
    }

    /**
     * Inserts a file from OX Drive.
     *
     * @param {ox.ui.App} app
     *  The application object representing the edited document.
     *
     * @param {Number} fileDescriptor
     *  The infostore descriptor of the file to be inserted
     *
     * @returns {jQuery.Promise}
     *  a Promise which be resolved when the insert action is successful
     */
    function insertDriveImage(app, fileDescriptor) {

        if (!fileDescriptor) { return $.Deferred().reject().promise(); }

        var params = {
            add_ext: fileDescriptor.filename.substring(fileDescriptor.filename.lastIndexOf('.') + 1),
            add_fileid: fileDescriptor.id,
            add_folderid: fileDescriptor.folder_id,
            alt_filename: fileDescriptor.filename
        };

        return app.sendActionRequest('addfile', params);
    }

    // static class Image =====================================================

    /**
     * Provides static helper methods for manipulation and calculation
     * of image nodes.
     */
    var Image = {};

    // static methods ---------------------------------------------------------

    /**
     * Checks if a file name has an image type extension.
     *
     * @param {String} file
     *
     * @returns {Boolean}
     *  Returns true if there is an extension and if has been detected
     *  as a image extension, otherwise false.
     */
    Image.hasFileImageExtension = function (file) {

        var ext = _.isString(file) ? file.split('.').pop().toLowerCase() : '';
        return ext in IMAGE_EXTENSIONS;
    };

    /**
     * Extracts the fileName from a hierarchical URI and checks if the extension
     * defines an supported image format.
     *
     * @param {String} uri
     *  A hierarchical uri
     *
     * @returns {Boolean}
     *  TRUE if the uri references a file which has a supported image extensions,
     *  otherwise FALSE.
     */
    Image.hasUrlImageExtension = function (uri) {
        return Image.hasFileImageExtension(Image.getFileNameFromImageUri(uri));
    };

    /**
     * Extracts the fileName from a hierarchical URI.
     *
     * @param {String} uri
     *  A hierarchical uri
     */
    Image.getFileNameFromImageUri = function (uri) {

        // Removes the fragment part (#), the query part (?) and chooses the last
        // segment
        var fileName = uri.substring(0, (uri.indexOf('#') === -1) ? uri.length : uri.indexOf('#'));
        fileName = fileName.substring(0, (fileName.indexOf('?') === -1) ? fileName.length : fileName.indexOf('?'));
        fileName = fileName.substring(fileName.lastIndexOf('/') + 1, fileName.length);
        return fileName;
    };

    Image.getFileNameWithoutExt = function (url) {
        var filename = Image.getFileNameFromImageUri(url);
        var pIndex = filename.lastIndexOf('.');
        return filename.substring(0, pIndex);
    };

    /**
     * checks if url is a 'bas64 data url' or an 'absolute / external url'
     * if not, it returns the documenturl of the cacheserver
     *
     *  @param {BaseApplication} app
     *
     *  @param {String} url
     */
    Image.getFileUrl = function (app, url) {

        // base64 data URL, or absolute / external URL
        if (url.substring(0, 10) === 'data:image' || (/:\/\//.test(url))) { return url; }

        // document URL
        return app.getServerModuleUrl(IO.FILTER_MODULE_NAME, { action: 'getfile', get_filename: url });
    };

    /**
     * Returns the MIME type for the given image uri according to the image file name extension.
     *
     * @param {String} uri
     *  A hierarchical uri
     *
     * @returns {String}
     *  The MIME type or an empty String.
     */
    Image.getMimeTypeFromImageUri = function (uri) {
        var fileName = Image.getFileNameFromImageUri(uri),
            ext = fileName ? fileName.split('.').pop().toLowerCase() : '';
        return IMAGE_EXTENSIONS[ext] || '';
    };

    /**
     * Post-processes the attributes of insert drawing operations. Based on the
     * drawing attributes and the editor instance the operation is applied to:
     *  - An image URL to an external location is always used.
     *  - The image media URL is used if the operation is applied within the
     *      same editor instance.
     *  - The BASE64 image data is used if the operation is applied in another
     *      editor instance.
     * Finally, the attributes will contain either the imageUrl or the
     * imageData attribute.
     *
     * @param {Object} drawingAttrs
     *  The (incomplete) drawing attributes of the operation.
     *
     * @param {String} fileId
     *  The identifier of the file where the operation is applied to.
     */
    Image.postProcessOperationAttributes = function (imageAttrs, fileId) {

        //imageAttrs.fileId from the clipboard is encoded because of Drive-file-ids (66/66 -> 66%2F66)
        if ((imageAttrs.sessionId === ox.session) && _.isString(imageAttrs.imageUrl) && (imageAttrs.fileId === fileId || imageAttrs.fileId === encodeURIComponent(fileId))) {

            // target is the same editor instance: use the image URL and remove the image BASE64 data
            delete imageAttrs.imageData;

        } else if (_.isString(imageAttrs.imageUrl) && /:\/\//.test(imageAttrs.imageUrl)) {

            // target is another editor instance but an external image URL is used: remove the image BASE64 data
            delete imageAttrs.imageData;

        } else if (_.isString(imageAttrs.imageData) && /^data:image/.test(imageAttrs.imageData)) {

            // target is another editor instance, BASE64 data is available: remove the image URL
            delete imageAttrs.imageUrl;

        } else {

            // remove invalid image data attribute
            delete imageAttrs.imageData;
        }

        delete imageAttrs.fileId;
        delete imageAttrs.sessionId;
    };

    /**
     * Inserts the image from the specified data URL into a document.
     *
     * @param {ox.ui.App} app
     *  The application object representing the edited document.
     *
     * @param {String} dataURL
     *  The image, Base64 encoded as data URL.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved if the image
     *  has been uploaded to the server; or rejected, in case of an error.
     */
    Image.insertDataUrl = function (app, dataUrl, fileName) {
        // the callback methods for insertFile
        return insertDataUrl(app, dataUrl, fileName)
        .then(function (data) {
            return {
                url: Utils.getStringOption(data, 'added_filename', ''),
                name: Image.getFileNameWithoutExt(fileName || 'image.img')
            };
        }, function () {
            app.getView().yell({ type: 'warning', message: gt('Image could not be loaded.') });
        });
    };

    /**
     * Inserts the image from the specified file into a document.
     *
     * @param {ox.ui.App} app
     *  The application object representing the edited document.
     *
     * @param {File} file
     *  The file object describing the image file to be inserted.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved if the image
     *  has been uploaded to the server; or rejected, in case of an error.
     */
    Image.insertFile = function (app, file) {
        // the callback methods for insertFile
        return insertFile(app, file)
        .then(function (data) {
            return {
                url: Utils.getStringOption(data, 'added_filename', ''),
                name: Image.getFileNameWithoutExt(file.name)
            };
        }, function () {
            app.getView().yell({ type: 'warning', message: gt('Image could not be loaded.') });
        });
    };

    /**
     * Inserts the image file of the passed infostore descriptor into a document.
     *
     * @param {ox.ui.App} app
     *  The application object representing the edited document.
     *
     * @param {Object} file
     *  The infostore descriptor of the file being inserted.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved if the image
     *  has been uploaded to the server; or rejected, in case of an error.
     */
    Image.insertDriveImage = function (app, file) {
        return insertDriveImage(app, file).then(function (data) {
            return {
                url: Utils.getStringOption(data, 'added_filename', ''),
                name: Image.getFileNameWithoutExt(file.filename)
            };
        });
    };

    /**
     * Inserts the image specified by a URL into a document.
     *
     * @param {ox.ui.App} app
     *  The application object representing the edited document.
     *
     * @param {String} url
     *  The full URL of the image to be inserted.
     */
    Image.insertURL = function (app, url) {
        var def = DeferredUtils.createDeferred(app, 'ImageUtil: insertURL');
        if (_.isString(url) && RE_IMAGE_URL.test(url)) {
            def.resolve({ url: url, name: Image.getFileNameWithoutExt(url) });
        } else {
            def.reject();
        }
        return def;
    };

     /**
     * Shows an insert image dialog from drive, local or URL dialog
     *
     * @param {TextSpplication} app
     *  The current application.
     *
     * @param {String} dialogType
     *  The dialog type. Types: 'drive', 'local' or 'url'. Default is 'drive'.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved if the dialog
     *  has been closed with the default action; or rejected, if the dialog has
     *  been canceled.
     */
    Image.showInsertImageDialogByType = function (app, dialogType) {
        var def;
        if (dialogType === 'url') {
            def = Image.showInsertImageUrlDialog(app);
        } else if (dialogType === 'local') {
            def = Image.showInsertImageLocalFileDialog(app);
        } else {
            def = Image.showInsertImageDriveFileDialog(app);
        }

        return def;
    };

    /**
     * Shows an insert image drive dialog
     *
     * @param {TextSpplication} app
     *  The current application.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved if the dialog
     *  has been closed with the default action; or rejected, if the dialog has
     *  been canceled.
     */
    Image.showInsertImageDriveFileDialog = function (app) {
        var deferred = DeferredUtils.createDeferred(app, 'ImageUtil: showInsertImageFileDialog');
        require(['io.ox/files/filepicker']).done(function (FilePicker) {

            var recentImageFolderKey = 'recentImageFolder' + app.getDocumentType();
            var recentImageFolder = Settings.get(recentImageFolderKey) || DriveUtils.getStandardPicturesFolderId();

            var dialogPromise = new FilePicker({
                // filter files of disabled applications (capabilities)
                filter: function (file) {
                    return Image.hasFileImageExtension(file.filename);
                },
                sorter: function (file) {
                    return (file.filename || file.title).toLowerCase();
                },
                primaryButtonText: gt('OK'),
                header: gt('Insert Image'),
                folder: recentImageFolder,
                uploadButton: true,
                multiselect: false,
                width: Dialogs.getBestDialogWidth(800),
                //uploadFolder: DriveUtils.getStandardDocumentsFolderId(),
                hideTrashfolder: true,
                acceptLocalFileType: 'image/*',
                cancel: function () {
                    deferred.reject();
                },
                initialize: function (dialog) {
                    app.getView().closeDialogOnReadOnlyMode(dialog, function () {
                        deferred.reject();
                    });
                }
            });

            dialogPromise.done(function (selectedFiles) {
                if (app.getModel().getEditMode()) {
                    var selectedFile = selectedFiles[0];
                    if (selectedFile) {
                        Image.insertDriveImage(app, selectedFile).done(function (imageFragment) {
                            deferred.resolve(imageFragment);
                        });

                        Settings.set(recentImageFolderKey, selectedFile.folder_id).save();
                    } else {
                        deferred.reject();
                    }
                } else {
                    deferred.reject();
                }

            });

        });

        return deferred.promise();
    };

    /**
     * Shows an insert image URL dialog
     *
     * @param {TextSpplication} app
     *  The current application.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved if the dialog
     *  has been closed with the default action; or rejected, if the dialog has
     *  been canceled.
     */
    Image.showInsertImageUrlDialog = function (app) {

        var
        // the dialog instance, and the tab panel nodes
            dialog = null,

        // the 'Enter URL' widgets
            urlInput = $('<input>', {
                tabindex: 0,
                role: 'textbox',
                'aria-label': gt('enter image URL')
            }).addClass('form-control'),

        // the resulting promise
            promise = null;

        // Bug 43868: ios auto correction should be of for input fields
        // TODO: use Forms.createInputMarkup() to create inputs here, it already has this fix
        if (Utils.IOS) {
            urlInput.attr({ autocorrect: 'off', autocapitalize: 'none' });
        }

        // create the dialog
        dialog = new Dialogs.ModalDialog({ title: gt('Insert Image URL'), async: true, width: Dialogs.getBestDialogWidth(800) });

        app.getView().closeDialogOnReadOnlyMode(dialog);

        dialog.append(urlInput);

        // register a handler at the url input field
        urlInput.on('input',  function () {
            dialog.enableOkButton(app.getModel().getEditMode() && (urlInput.val().length > 0));
        });

        // show the dialog
        promise = dialog.show(function () {
            // us 102078426: dialogs on small devices should come up w/o keyboard,
            // so the focus should not be in an input element
            if (Utils.SMALL_DEVICE) {
                Utils.setFocus(dialog.getOkButton());
                dialog.enableOkButton(false); // If the Ok button is not enabled on focus, the 'Insert Image' ImagePicker is not closed
            } else {
                dialog.enableOkButton(false);
                Utils.setFocus(urlInput);
            }
        });

        // process the result after the OK button has been pressed
        promise = promise.then(function () {

            var file = null,
                def = DeferredUtils.createDeferred(app, 'ImageUtil: showInsertImageDialog');

            // show progress indicator
            dialog.getHeader().find('h4').first().busy();

            file = urlInput.val();

            if (_.isObject(file) || _.isString(file)) {
                def.resolve(file);
            } else {
                def.reject();
            }

            return def.promise();
        });

        promise = promise.then(function (file) {
            return Image.insertURL(app, file.trim());
        });

        return promise.always(function () {
            //close dialog manually in asynchronous mode
            dialog.close();
            dialog = null;
        });
    };

    /**
     * Shows an insert image from local dialog
     *
     * @param {TextSpplication} app
     *  The current application.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved if the dialog
     *  has been closed with the default action; or rejected, if the dialog has
     *  been canceled.
     */
    Image.showInsertImageLocalFileDialog = function (app) {

        var
        // the dialog instance, and the tab panel nodes
            dialog = null,

        // the file picker widget
            filePicker = filePickerWidget(),
            fileInput = filePicker.find('input[type="file"]'),

        // the file descriptor of the file currently selected
            uploadFile = null,

        // the resulting promise
            promise = null;

        // create a file picker widget node
        function filePickerWidget(options) {
            options = _.extend({
                buttontext: gt('Select file'),
                placeholder: gt('No file selected'),
                multi: false,
                accept: 'image/*'
            }, options);

            var guid = _.uniqueId('form-control-'),

                node = $('<div class="file-picker">').addClass((options.wrapperClass ? options.wrapperClass : 'form-group')),

                buttonFile = $('<span class="btn btn-default btn-file">')
                    .text(options.buttontext),

                labelFile = $('<label class="sr-only">')
                    .attr('for', guid)
                    .text(options.buttontext),

                inputFile = $('<input name="file" type="file" style="margin-top: -20px; margin-bottom: -20px;">')
                    .prop({ multiple: options.multi })
                    .attr({ accept: options.accept, id: guid }),

                buttonGroupFile = $('<span class="input-group-btn btn-file" role="button">')
                    .append(buttonFile, labelFile, inputFile),

                inputText = $('<input type="text" class="form-control" readonly="" aria-readonly="true" style="pointer-events: none;">')
                    .attr({ placeholder: options.placeholder, 'aria-label': gt('file name') });

            // set tabindex depending on browser
            ((_.browser.IE) ? buttonFile : buttonGroupFile).attr('tabindex', '1');

            node.append(($('<div class="input-group">').append(buttonGroupFile, inputText)));

            node.on('keypress', function (e) {
                // press SPACE (ENTER isn't active, cause of the special 'send'-occupancy)
                if (e.which === 32) {
                    Utils.setFocus(inputFile); // BUG #34034: FF needs to focus the input-element first
                    inputFile.trigger('click');
                }
            });

            inputFile.on('change', function (event) {
                var name = (event.target && event.target.files && event.target.files[0] && event.target.files[0].name) || (fileInput[0].value && fileInput[0].value.substring(fileInput[0].value.lastIndexOf('\\') + 1)) || '';
                inputText.val(name);
            });

            return node;
        }

        // create the dialog
        dialog = new Dialogs.ModalDialog({ title: gt('Insert Image'), async: true, width: Dialogs.getBestDialogWidth(800) });
        dialog.enableOkButton(false);

        // close dialog automatically after losing edit rights
        app.getView().closeDialogOnReadOnlyMode(dialog);

        // prepare the 'Upload File' tab panel
        var uploadForm = $('<form>');

        dialog.append(uploadForm.append(filePicker.css('margin-bottom', 0)));

        // IE has a special handling for single inputs inside a form.
        // Pressing enter directly submits the form. Adding a hidden input solves this.
        if (_.browser.IE) {
            uploadForm.append($('<input>', { type: 'text', value: '' }).css('display', 'none'));
        }

        // show the dialog
        promise = dialog.show(function () {
            Utils.setFocus(fileInput);
        });
        // register a change handler at the file input that extracts the file descriptor

        fileInput.on('change', function (event) {
            uploadFile = (event.target && event.target.files && event.target.files[0]) || null;  // requires IE 10+
            dialog.enableOkButton(false);
            if (uploadFile) {
                if (Image.hasFileImageExtension(uploadFile.name)) {
                    dialog.enableOkButton(app.getModel().getEditMode());
                } else {
                    app.rejectEditAttempt('image');
                    uploadFile = null;
                }
            }
        });

        // process the result after the OK button has been pressed
        promise = promise.then(function () {

            var file = null,
                form = null,
                def = DeferredUtils.createDeferred(app, 'ImageUtil: showInsertImageLocalFileDialog');

            // show progress indicator
            dialog.getHeader().find('h4').first().busy();

            if (uploadFile) {
                file = uploadFile;
            }

            if (_.isObject(file) || _.isString(file) || _.isObject(form)) {
                def.resolve(file, form);
            } else {
                def.reject();
            }

            return def.promise();
        });

        promise = promise.then(function (file, form) {
            return Image.insertFile(app, file, form);
        });

        return promise.always(function () {
            //close dialog manually in asynchronous mode
            dialog.close();
            // remove all event listeners
            fileInput.off();
            filePicker.off();

            dialog = null;
        });
    };

    // exports ================================================================
    return Image;
});
