/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/drawinglayer/view/drawingframe', [

    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/io',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/render/canvas',

    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/gradient',
    'io.ox/office/editframework/utils/pattern',
    'io.ox/office/editframework/utils/texture',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/lib/canvasjs.min',
    'less!io.ox/office/drawinglayer/view/drawingstyle'

], function (Utils, Forms, IO, LocaleData, Canvas, AttributeUtils, Border, Gradient, Pattern, Texture, Labels, DrawingUtils, CanvasJS) {

    'use strict';

    var
        Math        = window.Math,

        math_sin    = Math.sin,
        math_cos    = Math.cos,
        math_tan    = Math.tan,
        math_atan2  = Math.atan2,
        math_sqrt   = Math.sqrt,

        math_abs    = Math.abs,
        math_floor  = Math.floor,
        math_ceil   = Math.ceil,
        math_round  = Math.round,

        math_min    = Math.min,
        math_max    = Math.max,

        MATH_PI     = Math.PI,

        // the CSS class name of the drawing content node
        CONTENT_CLASS = 'content',

        // the CSS class used to crop images
        CROPPING_CLASS = 'cropping-frame',

        // the CSS class name of the selection root node
        SELECTION_CLASS = 'selection',

        // the CSS class name for active tracking mode
        TRACKING_CLASS = 'tracking-active',

        CULTURE_INFO = null,

        PRESET_GEOMETRIES = null,

        MAXIMUM_CANVAS_SIZE = (_.browser.iOS || _.browser.Android || _.browser.Safari || _.browser.IE <= 10) ? 2156 : 4096,

        // cache during resize for texture bitmap used as shape backgound
        textureContextCache = null;

    // private global functions ===============================================

    /**
     * Returns the selection root node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The selection root node of the specified drawing frame. Will be an
     *  empty collection, if the drawing frame is not selected currently.
     */
    function getSelectionNode(drawingFrame) {
        return $(drawingFrame).first().children('.' + SELECTION_CLASS);
    }

    /**
     * Calculates the offset and size of the bitmap in an image frame for one
     * dimension (either horizontally or vertically), according to the passed
     * cropping settings.
     *
     * @param {Number} frameSize
     *  Visible size (with or height) of the drawing frame containing the image
     *  bitmap, in 1/100 of millimeters.
     *
     * @param {Number} leadingCrop
     *  The leading cropping value (left/top).
     *
     * @param {Number} trailingCrop
     *  The trailing cropping value (right/bottom).
     *
     * @returns {Object}
     *  An object containing 'offset' and 'size' CSS attributes specifying how
     *  to format the bitmap node (in pixels with 'px' unit name).
     */
    function calculateBitmapSettings(app, frameSize, leadingCrop, trailingCrop) {

        // ODF: absolute cropping length (in 1/100mm), OOXML: percentage
        if (app.isODF()) {
            leadingCrop /= frameSize;
            trailingCrop /= frameSize;
        } else {
            leadingCrop /= 100;
            trailingCrop /= 100;
        }

        var // sum of leading and trailing cropping (must not exceed a specific amount)
            totalCrop = leadingCrop + trailingCrop,
            // resulting settings for the bitmap
            size = 0, offset = 0;

        // do not crop away more than 90% of the bitmap
        if (totalCrop > 0.9) {
            leadingCrop *= (0.9 / totalCrop);
            trailingCrop *= (0.9 / totalCrop);
        }

        // bitmap size and offset, according to object size and cropping
        size = frameSize / (1 - leadingCrop - trailingCrop);
        offset = size * leadingCrop;

        // convert to CSS pixels
        return {
            offset: Utils.convertHmmToCssLength(-offset, 'px', 1),
            size: Utils.convertHmmToCssLength(size, 'px', 1)
        };
    }

    /**
     * Returns the path rendering mode for the specified flags, as expected by
     * the rendering methods of a canvas context.
     *
     * @param {Boolean} hasFill
     *  Whether the shape area will be filled.
     *
     * @param {Boolean} hasLine
     *  Whether the shape outline will be rendered.
     *
     * @returns {String|Null}
     *  The path rendering mode as string, if either of the passed flags is
     *  true, otherwise null to indicate that nothing needs to be drawn.
     */
    function getRenderMode(hasFill, hasLine) {
        return hasFill ? (hasLine ? 'all' : 'fill') : (hasLine ? 'stroke' : null);
    }

    function handleContentEditable(drawingFrame) {
        if (!_.browser.IE && !_.browser.Firefox) { $(drawingFrame).attr('contenteditable', true); }
    }

    /**
     * Calculating an optional increase of a canvas node inside a drawing node. This
     * is especially required for arrows at line ends.
     *
     * @param {Object} lineAttrs
     *  The line attributes of the drawing.
     *
     * @param {Object} geometry
     *  The geometry attribute of a drawing.
     *
     * @returns {Number}
     *  The expansion (in pixel) required for a canvas node, so that the path together
     *  with the line ends can be displayed completely.
     */
    function getCanvasExpansion(lineAttrs, geometry) {

        var minExpansion = 10; // the minimum expansion in pixel

        if (lineAttrs && geometry) {
            if ((lineAttrs.headEndType && lineAttrs.headEndType !== 'none') ||
                (lineAttrs.tailEndType && lineAttrs.tailEndType !== 'none')) {

                if (('pathList' in geometry) && (geometry.pathList.length > 0)) {
                    return Math.max(Utils.round(2.5 * Utils.convertHmmToLength((lineAttrs.width  || 100), 'px'), 1), minExpansion);
                }
            }
        }

        return 0;
    }

    /**
     * Checking whether a line specified by two points p1 (x1, y1) and p2 (x2, y2)
     * goes from p2 to p1 to the right direction. If it does not go from left to
     * right, it is checked, if it goes from top to down.
     * This function can be used to check the direction of an arrowhead. This can
     * typically only be used for 'simple' arrows, that are drawn in a canvas from
     * top left to bottom right.
     *
     * @param {Number} x1
     *  The x-position of point 1.
     *
     * @param {Number} y1
     *  The y-position of point 1.
     *
     * @param {Number} x2
     *  The x-position of point 2.
     *
     * @param {Number} x2
     *  The y-position of point 2.
     *
     * @returns {Boolean}
     *  Whether the specified two points describe a line from left-to-right or if
     *  the line is exact vertical from top-to-bottom.
     */
    function isRightOrDownDirection(x1, y1, x2, y2) {
        return (x2 < x1) || ((x2 === x1) && (y2 < y1));
    }

    /**
     * Analyzing a path to get a start or an end point together with a reference
     * point. The reference point can be used to calculate an angle of an arrowhead
     * at the head or the tail of the path.
     *
     * @param {Object[]} commands
     *  The command to draw the path. There must be at least two commands in the path.
     *  For the head the first two commands are evaluated, for the tail the final two
     *  pathes.
     *
     * @param {Object} [options]
     *  Optional parameter:
     *  @param {Boolean} [options.first=false]
     *      If set to true, the head of the path is evaluated, otherwise the tail.
     *  @param {Object} [snapRect]
     *     The rectangle with the properties left, top, width and height
     *     for setting the size of the canvas.
     *  @param {Object} [avList]
     *     The used adjustment values.
     *  @param {Object[]} [gdList]
     *     An array with formulas that can be used.
     *  @param {Object} [guideCache]
     *     The cache with already calculated positioning values.
     *
     * @returns {Object}
     *  An object containing the properties 'start' and 'end'. The values of these
     *  properties are again objects with the properties 'x' and 'y' that are
     *  calculated from the commands.
     */
    function getReferencePointsFromPath(commands, options) {

        // whether the first visible line of the path is searched
        var first = Utils.getBooleanOption(options, 'first', false);
        // the first visible line of the path
        var length = 0;
        // the point of the first or the second last command
        var start = {};
        // the point of the second or the last command
        var end = {};
        // the first command of the command list
        var firstCommand = null;
        // the last command of the command list
        var secondCommand = null;
        // the end point of a curved line
        var endPoint = null;
        // the reference point of a curved line
        var refPoint = null;
        // the number of point of a specific path command
        var pointCount = 0;

        // helper function, that calculates, if a specified reference point is left
        // of the start point or right from the end point at bezier curves.
        // pointA is the start or end point, pointB the corresponding reference point.
        function isOutside(pointA, pointB, isStart) {

            var pointAX = 0;
            var pointBX = 0;
            var isOutside = false;

            if (options.snapRect && options.avList && options.gdList && options.guideCache) {
                pointAX = DrawingFrame.getGeometryValue(pointA.x, options.snapRect, options.avList, options.gdList, options.guideCache, 1000);
                pointBX = DrawingFrame.getGeometryValue(pointB.x, options.snapRect, options.avList, options.gdList, options.guideCache, 1000);
                isOutside = isStart ? (pointBX < pointAX) : (pointBX > pointAX);
            }

            return isOutside;
        }

        if (first) {

            firstCommand = commands[0];
            secondCommand = commands[1];

            start.x = firstCommand.x;
            start.y = firstCommand.y;

            end.x = secondCommand.x;
            end.y = secondCommand.y;

            if (!end.x && start.x &&  secondCommand.pts) { // handling of non-linear curves (cubicBezierTo)
                end.x = start.x;
                end.y = start.y;
                // comparing start point with first reference point to check direction of curve
                end.xOffset = isOutside(secondCommand.pts[0], secondCommand.pts[1], true) ? -5 : +5; // must be negative, if the bezier point is outside the shape -> enabling valid direction of arrow at line start
            }

        } else {

            length = commands.length;
            firstCommand = commands[length - 2];
            secondCommand = commands[length - 1];

            start.x = firstCommand.x;
            start.y = firstCommand.y;

            end.x = secondCommand.x;
            end.y = secondCommand.y;

            if (!start.x &&  secondCommand.pts) { // handling of non-linear curves (cubicBezierTo)

                pointCount = secondCommand.pts.length;

                endPoint = secondCommand.pts[pointCount - 1];
                refPoint = secondCommand.pts[pointCount - 2];

                start.x = endPoint.x;
                start.y = endPoint.y;

                end.x = endPoint.x;
                end.y = endPoint.y;

                // comparing end point with last reference point to check direction of curve
                start.xOffset = isOutside(endPoint, refPoint, false) ? 5 : -5; // must be positive, if the bezier point is outside the shape -> enabling valid direction of arrow at line end
            }
        }

        return { start: start, end: end };
    }

    /**
     * Handling the line ends for a specified path. The line end is handled by the line attributes
     * "headEndType", "headEndLength", "headEndWidth", "tailEndType", "tailEndLength" and "tailEndWidth".
     *
     * @param {Object[]|[]|Null} lineEndPathes
     *  The collector for the pathes that are necessary to draw the line endings. This can be an
     *  array, that already contains pathes, or an empty array, or null. In the latter case a new
     *  array is generated (and returned). This parameter is used as return value.
     *
     * @param {ContextWrapper} context
     *  Canvas context wrapper.
     *
     * @param {Object} lineAttrs
     *  Object containing line attributes for the canvas
     *
     * @param {Number} lineWidth
     *  The line width in pixel.
     *
     * @param {Object} color
     *  The line color object.
     *
     * @param {Object} path
     *  The line path element.
     *
     * @param {String} arrowType
     *  The type of the line ending. Supported values are:
     *  'oval', 'diamond', 'triangle', 'stealth' and 'arrow'.
     *
     * @param {String} endType
     *  The end type for the specified path. Supported values are:
     *  'head' and 'tail'.
     *
     * @param {Object} [options]
     *  Optional parameter:
     *  @param {Object} [snapRect]
     *     The rectangle with the properties left, top, width and height
     *     for setting the size of the canvas.
     *  @param {Object} [avList]
     *     The used adjustment values.
     *  @param {Object[]} [gdList]
     *     An array with formulas that can be used.
     *  @param {Object} [guideCache]
     *     The cache with already calculated positioning values.
     *
     * @returns {Object[]|[]|Null}
     *  The collector for the pathes that are necessary to draw the line endings. This can be an
     *  array, that already contains pathes, or an empty array, or null.
     */
    function handleLineEnd(lineEndPathes, context, lineAttrs, lineWidth, color, path, arrowType, endType, options) {

        // an additional line end path that is collected inside lineEndPathes
        var lineEndPath = null;
        // the length of the arrow (in pixel)
        var length = 0;
        // the width of the arrow (in pixel)
        var width = 0;
        // whether the arrow head required a 'fill' additionally to the 'stroke' for the path
        var fillRequired = false;
        // a collector for all commands of the path
        var allCommands = null;
        // an object containing a start and an end point received from the commands of the path
        var refPoints = null;
        // the start point object received from the specified path
        var start = null;
        // the end point object received from the specified path
        var end = null;

        // helper function to get display factors for different line widths
        // values 2, 4, 6 for lineWidth < 2
        // values 1, 2, 3 for lineWidth 2 < 8
        // values 0.5, 1, 1.5 for lineWidth > 8
        function getFactor(type, wid, aType) {

            var factor = 1;
            var arrowFactor = 1;
            var ovalFactor = 1.3;

            if (wid <= 2) {  // larger values for thickness below 3 px
                factor = 3;
                if (type === 'small') {
                    factor = 2;
                } else if (type === 'large') {
                    factor = 5;
                }
            } else {
                factor = 0.75; //  1.5;
                if (type === 'small') {
                    factor = 0.5; // 1;
                } else if (type === 'large') {
                    factor = 1.25; // 2.5;
                }
            }

            // additionally increasing length and width for arrows
            if (aType === 'arrow') {
                arrowFactor = 1.5;
                if (type === 'small') {
                    arrowFactor = 1.8;
                } else if (type === 'large') {
                    arrowFactor = 1.2;
                }

                factor *= arrowFactor;
            } else if (aType === 'oval') {
                factor *= ovalFactor; // small increasement for circles and ellipses
            }

            return factor;
        }

        // helper function to get line property values for width and height in pixel
        function getLineEndAttribute(attrs, property, type) {

            var value = getFactor('medium', lineWidth, type) * lineWidth;
            if (attrs && attrs[property] === 'small') {
                value = getFactor('small', lineWidth, type) * lineWidth;
            } else if (attrs && attrs[property] === 'large') {
                value = getFactor('large', lineWidth, type) * lineWidth;
            }

            return value;
        }

        if (endType === 'head') {

            lineEndPath = _.clone(path, true);
            lineEndPath.commands = [];
            allCommands = path.commands; // the array with the commands

            if (allCommands.length < 2) { return lineEndPathes; }

            options = options || {};
            options.first = true;
            refPoints = getReferencePointsFromPath(allCommands, options);

            start = refPoints.start;
            end = refPoints.end;

            length = getLineEndAttribute(lineAttrs, 'headEndLength', arrowType);
            width = getLineEndAttribute(lineAttrs, 'headEndWidth', arrowType);

            if (arrowType === 'oval') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'oval', x: start.x, y: start.y, r1: length, r2: width, rpx: end.x, rpy: end.y, rpxOffset: (end.xOffset || 0) });
            } else if (arrowType === 'diamond') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'diamond', x: start.x, y: start.y, r1: length, r2: width, rpx: end.x, rpy: end.y, rpxOffset: (end.xOffset || 0) });
            } else if (arrowType === 'triangle') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'triangle', x: start.x, y: start.y, r1: length, r2: width, rpx: end.x, rpy: end.y, rpxOffset: (end.xOffset || 0) });
            } else if (arrowType === 'stealth') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'stealth', x: start.x, y: start.y, r1: length, r2: width, rpx: end.x, rpy: end.y, rpxOffset: (end.xOffset || 0) });
            } else if (arrowType === 'arrow') {
                lineEndPath.commands.push({ c: 'arrow', x: start.x, y: start.y, r1: length, r2: width, rpx: end.x, rpy: end.y, rpxOffset: (end.xOffset || 0) });
            }

        } else if (endType === 'tail') {

            lineEndPath = _.clone(path, true);
            lineEndPath.commands = [];
            allCommands = path.commands; // the array with the commands

            if (allCommands.length < 2) { return lineEndPathes; }

            options = options || {};
            options.first = false;
            refPoints = getReferencePointsFromPath(allCommands, options);

            start = refPoints.start;
            end = refPoints.end;

            length = getLineEndAttribute(lineAttrs, 'tailEndLength', arrowType);
            width = getLineEndAttribute(lineAttrs, 'tailEndWidth', arrowType);

            if (arrowType === 'oval') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'oval', x: end.x, y: end.y, r1: length, r2: width, rpx: start.x, rpy: start.y, rpxOffset: (start.xOffset || 0) });
            } else if (arrowType === 'diamond') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'diamond', x: end.x, y: end.y, r1: length, r2: width, rpx: start.x, rpy: start.y, rpxOffset: (start.xOffset || 0) });
            } else if (arrowType === 'triangle') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'triangle', x: end.x, y: end.y, r1: length, r2: width, rpx: start.x, rpy: start.y, rpxOffset: (start.xOffset || 0) });
            } else if (arrowType === 'stealth') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'stealth', x: end.x, y: end.y, r1: length, r2: width, rpx: start.x, rpy: start.y, rpxOffset: (start.xOffset || 0) });
            } else if (arrowType === 'arrow') {
                lineEndPath.commands.push({ c: 'arrow', x: end.x, y: end.y, r1: length, r2: width, rpx: start.x, rpy: start.y, rpxOffset: (start.xOffset || 0) });
            }

        }

        if (lineEndPath && lineEndPath.commands && lineEndPath.commands.length > 0) {
            lineEndPath.drawMode = fillRequired ? 'all' : 'stroke';
            lineEndPathes = lineEndPathes || [];
            lineEndPathes.push(lineEndPath);
        }

        return lineEndPathes;
    }

    /**
     * Sets the line attributes to the given canvas context.
     *
     * @param {Object} docModel
     *  The application model object
     *
     * @param {ContextWrapper} context
     *  Canvas context wrapper.
     *
     * @param {Object} lineAttrs
     *  Object containing line attributes for the canvas
     *
     * @param {Number} [lineWidth]
     *  Optional parameter for line width in px.
     *
     * @param {Object} [path]
     *  Optional parameter for the line path.
     *
     * @param {Object} [options]
     *  Optional parameter for the line handling:
     *  @param {Boolean} [options.isFirstPath=false]
     *      If set to true, this is the first path element of a specified path.
     *  @param {Boolean} [options.isLastPath=false]
     *      If set to true, this is the last path element of a specified path.
     *  @param {Object} [snapRect]
     *     The rectangle with the properties left, top, width and height
     *     for setting the size of the canvas.
     *  @param {Object} [avList]
     *     The used adjustment values.
     *  @param {Object[]} [gdList]
     *     An array with formulas that can be used.
     *  @param {Object} [guideCache]
     *     The cache with already calculated positioning values.
     *
     * @returns {Objects[]|Null}
     *  An array containing additional pathes that are necessary to draw the line
     *  endings. Or null, if no line endings are required.
     */
    function setLineAttributes(docModel, context, lineAttrs, lineWidth, path, options) {

        var
            lineColor = docModel.getCssColor(lineAttrs.color, 'line', lineAttrs[DrawingFrame.TARGET_CHAIN_PROPERTY]),
            locLineWidth = lineWidth || math_max(Utils.convertHmmToLength(lineAttrs.width, 'px', 1), 1),
            pattern   = Border.getBorderPattern(lineAttrs.style, locLineWidth),
            isFirstPath = Utils.getBooleanOption(options, 'isFirstPath', false),
            isLastPath = Utils.getBooleanOption(options, 'isLastPath', false),
            newPathes = null;

        context.setLineStyle({ style: lineColor, width: locLineWidth, pattern: pattern });

        if (path && lineAttrs) {
            if (isFirstPath && lineAttrs.headEndType && lineAttrs.headEndType !== 'none') { newPathes = handleLineEnd(newPathes, context, lineAttrs, locLineWidth, lineColor, path, lineAttrs.headEndType, 'head', options); }
            if (isLastPath && lineAttrs.tailEndType && lineAttrs.tailEndType !== 'none') { newPathes = handleLineEnd(newPathes, context, lineAttrs, locLineWidth, lineColor, path, lineAttrs.tailEndType, 'tail', options); }
        }

        return newPathes;
    }

    /**
     * Sets the bitmap texture fill style to the given canvas context, according to different properties.
     *
     * @param {DocumentModel}  docModel
     *  Current document model.
     * @param {ContextWrapper}  context
     *  Canvas context wrapper.
     * @param {function}  drawMethod
     *   Method which should be called for drawing canvas
     * @param {CanvasPattern}  fillStyle
     * @param {Number}  widthPx
     *  Width of shape in pixels.
     * @param {Number}  heightPx
     *  Height of shape in pixels.
     * @param {Boolean} hasLine
     *  If shape has line or not.
     * @param {Object}  lineAttrs
     *  Attribute set for the line.
     * @param {Boolean} isTiling
     *  If bitmap is tiling.
     * @param {Boolean}  notRotatedWithShape
     *  If bitmap should contain its own rotation angle and not rotate with shape.
     * @param {Number|null}  rotationAngle
     *  If exists and shape shouldn't rotate with shape, use it as counter angle for texture.
     */
    function setBitmapTextureFillStyle(docModel, context, drawMethod, fillStyle, widthPx, heightPx, hasLine, lineAttrs, isTiling, notRotatedWithShape, rotationAngle) {
        var nWidth, nHeight;

        context.clearRect(0, 0, widthPx, heightPx);
        if (notRotatedWithShape && rotationAngle) {
            var rectPos = widthPx > heightPx ? widthPx : heightPx;
            // initialize context with fill to mask path with pattern
            context.setFillStyle('#000000');
            drawMethod('fill');

            context.translate(widthPx / 2, heightPx / 2);
            context.rotate(rotationAngle);
            context.setFillStyle(fillStyle);

            context.valueOf().globalCompositeOperation = 'source-in'; // The new shape is drawn only where both the new shape and the destination canvas overlap. Everything else is made transparent.
            if (notRotatedWithShape && !isTiling) {
                nWidth = widthPx * Math.abs(Math.cos(rotationAngle)) + heightPx * Math.abs(Math.sin(rotationAngle));
                nHeight = heightPx * Math.abs(Math.cos(rotationAngle)) + widthPx * Math.abs(Math.sin(rotationAngle));
                context.translate(-nWidth / 2, -nHeight / 2);
                context.drawRect(0, 0, nWidth, nHeight, 'fill'); // cover whole canvas area with pattern fill
            } else {
                context.drawRect(2 * -rectPos, 2 * -rectPos, 4 * rectPos, 4 * rectPos, 'fill'); // cover whole canvas area with pattern fill
            }
            context.valueOf().globalCompositeOperation = 'source-over'; // reset to original values
            context.valueOf().setTransform(1, 0, 0, 1, 0, 0); // reset transformations
            if (hasLine) {
                setLineAttributes(docModel, context, lineAttrs);
            }
            drawMethod('stroke'); // stroke needs to be painted again after overlap
        } else {
            context.setFillStyle(fillStyle);
            // initialize context with proper line attributes for the next path
            if (hasLine) {
                setLineAttributes(docModel, context, lineAttrs);
            }
            drawMethod();
        }
    }

    /**
     *
     * fix for Bug 48477 explicit style-entries contentNode got lost,
     * so we changed to css-classes
     * html-attributes (interpredet by css would be also a nice solution)
     *
     */
    function handleFlipping(docModel, drawingFrame, drawingAttributes) {

        var contentNode = DrawingFrame.getContentNode(drawingFrame);
        var textFrameNode = DrawingFrame.getTextFrameNode(drawingFrame);
        var flipH = false;
        var flipV = false;

        // used for fix for Bugs 45323, 48187
        // special behavior of presentation files
        // Shapes do ignore their horizontal flip state if parent group has standard transform
        // or completely inverted transform
        // textDirection info is set always but only checked in presentation if horizontal flip is true
        function isOddFlipHVCount(textFrameNode) {
            var flipHCount = textFrameNode.parentsUntil('.slide', '.flipH').length; // must be separate queries, because if element has flipH and flipV,
            var flipVCount = textFrameNode.parentsUntil('.slide', '.flipV').length; // both need to be counted
            return (flipHCount + flipVCount) % 2 !== 0;
        }

        if (drawingAttributes) {
            // convert to single jQuery object
            drawingFrame = $(drawingFrame).first();

            flipH = drawingAttributes.flipH;
            flipV = drawingAttributes.flipV;
        }

        contentNode.toggleClass('flipH', flipH);
        contentNode.toggleClass('flipV', flipV);

        textFrameNode.toggleClass('flipH', isOddFlipHVCount(textFrameNode));
    }

    // static class DrawingFrame ==============================================

    /**
     * Contains common static helper functions to display and handle drawing
     * frames in any document.
     */
    var DrawingFrame = {};

    // constants --------------------------------------------------------------

    /**
     * The CSS class used to mark drawing frame nodes.
     *
     * @constant
     */
    DrawingFrame.NODE_CLASS = 'drawing';

    /**
     * A jQuery selector that matches nodes representing a drawing frame.
     *
     * @constant
     */
    DrawingFrame.NODE_SELECTOR = '.' + DrawingFrame.NODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing the content nodes in a drawing frame.
     *
     * @constant
     */
    DrawingFrame.CONTENT_NODE_SELECTOR = '.' + CONTENT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes as place holder.
     *
     * @constant
     */
    DrawingFrame.PLACEHOLDER_CLASS = 'placeholder';

    /**
     * A jQuery selector that matches nodes representing a place holder element
     * of type 'div'.
     *
     * @constant
     */
    DrawingFrame.PLACEHOLDER_SELECTOR = 'div.' + DrawingFrame.PLACEHOLDER_CLASS;

    /**
     * A jQuery selector that matches content nodes of group drawing objects.
     *
     * @constant
     */
    DrawingFrame.GROUPCONTENT_SELECTOR = DrawingFrame.NODE_SELECTOR + '[data-type="group"]>.' + CONTENT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.AUTORESIZEHEIGHT_CLASS = 'autoresizeheight';

    /**
     * The CSS class used to mark drawing content node for no wraping of words.
     *
     * @constant
     */
    DrawingFrame.NO_WORDWRAP_CLASS = 'no-wordwrap';

    /**
     * A jQuery selector that matches nodes representing a content node inside a
     * test frame that automatically resizes its height.
     *
     * @constant
     */
    DrawingFrame.AUTORESIZEHEIGHT_SELECTOR = 'div.' + DrawingFrame.AUTORESIZEHEIGHT_CLASS;

    DrawingFrame.AUTORESIZETEXT_CLASS = 'autoresizetext';

    DrawingFrame.AUTORESIZETEXT_SELECTOR = 'div.' + DrawingFrame.AUTORESIZETEXT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE = 'verticalalign';

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.ODFTEXTFRAME_CLASS = 'odftextframe';

    /**
     * A jQuery selector that matches nodes representing a text frame node that
     * are 'classical' text frames in odf format.
     *
     * @constant
     */
    DrawingFrame.ODFTEXTFRAME_SELECTOR = 'div.' + DrawingFrame.ODFTEXTFRAME_CLASS;

    /**
     * The CSS class used to mark empty text frames.
     *
     * @constant
     */
    DrawingFrame.EMPTYTEXTFRAME_CLASS = 'emptytextframe';

    /**
     * The name of the jQuery data attribute that stores the unique node
     * identifier of a drawing frame node.
     *
     * @constant
     */
    DrawingFrame.DATA_UID = 'uid';

    /**
     * The name of the jQuery data attribute that stores the drawing model
     * instance for a drawing frame node.
     *
     * @constant
     */
    DrawingFrame.DATA_MODEL = 'model';

    /**
     * A jQuery selector that matches elements representing a table text frame element.
     *
     * @constant
     */
    DrawingFrame.TABLE_TEXTFRAME_NODE_SELECTOR = '.drawing[data-type="table"]';

    /**
     * The CSS class used to mark table nodes inside drawings of type 'table'.
     *
     * @constant
     */
    DrawingFrame.TABLE_NODE_IN_TABLE_DRAWING_CLASS = 'isdrawingtablenode';

    /**
     * A jQuery selector that matches elements representing a text frame element
     * inside a drawing frame of type shape.
     */
    DrawingFrame.TEXTFRAME_NODE_SELECTOR = 'div.textframe';

    /**
     * The CSS class used to mark text frame content nodes inside a
     * drawing node.
     *
     * @constant
     */
    DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS = 'textframecontent';

    /**
     * A jQuery selector that matches nodes representing a text frame content
     * element inside a drawing node.
     *
     * @constant
     */
    DrawingFrame.TEXTFRAMECONTENT_NODE_SELECTOR = 'div.' + DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS;

    /**
     * The CSS class used to mark drawing nodes inside a group container.
     *
     * @constant
     */
    DrawingFrame.GROUPED_NODE_CLASS = 'grouped';

    /**
     * A jQuery selector that matches nodes representing a grouped element.
     *
     * @constant
     */
    DrawingFrame.GROUPED_NODE_SELECTOR = '.' + DrawingFrame.GROUPED_NODE_CLASS;

    /**
     * The CSS class used to mark drawing nodes that cannot be grouped.
     *
     * @constant
     */
    DrawingFrame.NO_GROUP_CLASS = 'nogroup';

    /**
     * The CSS class used to mark canvas nodes of a drawing.
     *
     * @constant
     */
    DrawingFrame.CANVAS_CLASS = 'canvasgeometry';

    /**
     * A jQuery selector that matches nodes representing a canvas element.
     *
     * @constant
     */
    DrawingFrame.CANVAS_NODE_SELECTOR = 'canvas.' + DrawingFrame.CANVAS_CLASS;

    /**
     * The CSS class used to mark unsupported drawing nodes.
     *
     * @constant
     */
    DrawingFrame.UNSUPPORTED_CLASS = 'unsupported';

    /**
     * A jQuery selector that matches nodes representing unsupported drawings.
     *
     * @constant
     */
    DrawingFrame.UNSUPPORTED_SELECTOR = '.' + DrawingFrame.UNSUPPORTED_CLASS;

    /**
     * A class name that matches elements representing a rotated drawing.
     */
    DrawingFrame.ROTATED_DRAWING_CLASSNAME = 'rotated-drawing';

    /**
     * A class name that matches elements representing a rotated text in drawing.
     */
    DrawingFrame.ROTATED_TEXT_IN_DRAWING_CLASSNAME = 'rotated-text-drawing';

    /**
     * The name of the property for the target chain that needs to be added to
     * the 'fill' and the 'line' family.
     */
    DrawingFrame.TARGET_CHAIN_PROPERTY = 'targetchain';

    /**
     * Class name that maches elements that need temporary border drawn around them,
     * usually on resize or move.
     */
    DrawingFrame.FRAME_WITH_TEMP_BORDER = 'temp-border-frame';

    // methods ----------------------------------------------------------------

    /**
     * Returns a new drawing frame node without specific contents.
     *
     * @param {String} type
     *  The type of the drawing.
     *
     * @param {DrawingModel|String} model
     *  The drawing model the new drawing frame will be based on. Alternatively
     *  (for applications not supporting drawing models), this value can be a
     *  string with the type of the drawing object.
     *
     * @returns {jQuery}
     *  A new drawing frame with empty content node, as jQuery object.
     */
    DrawingFrame.createDrawingFrame = function (model) {

        var // the drawing type identifier
            type = _.isObject(model) ? model.getType() : model,
            // whether the drawing object is a group
            isGroup = type === 'group',
            // the content node
            contentNode = $('<div class="' + CONTENT_CLASS + '">'),
            // the resulting drawing frame node
            drawingFrame = $('<div class="' + DrawingFrame.NODE_CLASS + '" data-type="' + type + '" contenteditable="' + (isGroup && !_.browser.IE && !_.browser.Firefox) + '">');

        // add data attributes, event handlers, and the content node
        drawingFrame
            .data(DrawingFrame.DATA_UID, 'frame' + _.uniqueId())
            .data(DrawingFrame.DATA_MODEL, _.isObject(model) ? model : null)
            .on('dragstart', false)
            .append(contentNode);

        return drawingFrame;
    };

    /**
     * Returns whether the passed node is the root node of a drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the root node of a drawing frame.
     */
    DrawingFrame.isDrawingFrame = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR);
    };

    /**
     * Returns the type of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The type of the specified drawing frame, as specified while the frame
     *  has been created.
     */
    DrawingFrame.getDrawingType = function (drawingFrame) {
        return $(drawingFrame).first().attr('data-type');
    };

    /**
     * Returns whether the passed node is a content node inside the drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a content node inside a drawing frame.
     */
    DrawingFrame.isDrawingContentNode = function (node) {
        return $(node).is('.' + CONTENT_CLASS);
    };

    /**
     * Returns whether the passed node is a content node inside the drawing frame
     * that has no content.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a content node inside a drawing frame that has
     *  no content.
     */
    DrawingFrame.isEmptyDrawingContentNode = function (node) {
        return DrawingFrame.isDrawingContentNode(node) && $(node).children().not(DrawingFrame.CANVAS_NODE_SELECTOR).length === 0;
    };

    /**
     * Returns whether the passed node is a placeholder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a place holder node (of a drawing frame).
     */
    DrawingFrame.isPlaceHolderNode = function (node) {
        return $(node).is(DrawingFrame.PLACEHOLDER_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'shape'.
     */
    DrawingFrame.isShapeDrawingFrame = function (node) { // TODO: type 'table' should be handled differently
        var nodeType = DrawingFrame.getDrawingType(node);
        return DrawingFrame.isDrawingFrame(node) && (nodeType === 'shape' || nodeType === 'table' || nodeType === 'connector');
        // TODO This should be activated asap: return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'shape';
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'shape'.
     */
    DrawingFrame.isTableDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'table';
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape' AND
     * that is used as text frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'shape' and it is used
     *  as text frame.
     */
    DrawingFrame.isTextFrameShapeDrawingFrame = function (node) {
        return DrawingFrame.isShapeDrawingFrame(node) && DrawingFrame.getTextFrameNode(node).length > 0;
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'group'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'group'.
     */
    DrawingFrame.isGroupDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'group';
    };

    /**
     * Returns whether the passed node is a drawing frame that is grouped inside a
     * group.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a grouped drawing frame.
     */
    DrawingFrame.isGroupedDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && $(node).is(DrawingFrame.GROUPED_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that can be moved only
     * in the region near the border. In its center, it can contain text frames.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that cannot be moved by clicking
     *  in its center.
     */
    DrawingFrame.isOnlyBorderMoveableDrawing = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) || DrawingFrame.isGroupDrawingFrame(node);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'
     * and that contains text content. Furthermore the drawing is NOT marked as
     * 'classical' odf text frame, so that there is only reduced text functionality
     * available.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that is of type 'shape'
     *  and that contains text content. Furthermore the drawing is NOT marked as
     *  'classical' odf text frame, so that there is only reduced text functionality
     *  available.
     */
    DrawingFrame.isReducedOdfTextframeNode = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && !$(node).is(DrawingFrame.ODFTEXTFRAME_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'
     * and that contains text content. Furthermore the drawing is marked as
     * 'classical' odf text frame, so that there is no text functionality
     * available.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that is of type 'shape'
     *  and that contains text content. Furthermore the drawing is marked as
     *  'classical' odf text frame, so that there is no reduced text functionality
     *  available.
     */
    DrawingFrame.isFullOdfTextframeNode = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && $(node).is(DrawingFrame.ODFTEXTFRAME_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node that's rotated by some angle.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing node that has rotation applied.
     */
    DrawingFrame.isRotatedDrawingNode = function (node) {
        return $(node).hasClass(DrawingFrame.ROTATED_DRAWING_CLASSNAME);
    };

    /**
     * Returns whether the passed node is text node in drawing, that is rotated by some angle.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text node in drawing that has rotation applied.
     */
    DrawingFrame.isRotatedTextInDrawingNode = function (node) {
        return $(node).hasClass(DrawingFrame.ROTATED_TEXT_IN_DRAWING_CLASSNAME);
    };

    /**
     * Returns the number of drawing frames inside a specified drawing group. Only
     * the direct children are counted, not the children in sub groups.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Number|Null}
     *  The number of drawing frames inside a specified drawing group
     */
    DrawingFrame.getGroupDrawingCount = function (node) {
        if (!DrawingFrame.isGroupDrawingFrame(node)) { return null; }
        return $(node).children().first().children().length;
    };

    /**
     * Returns the child drawing frame of a drawing group at a specified
     * position. Or null, if it cannot be determined.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @param {Number} [number]
     *  The zero based number of the child to be returned.
     *
     * @returns {Node|Null}
     *  The child drawing frame of a drawing group at a specified position.
     *  Or null, if it cannot be determined.
     */
    DrawingFrame.getGroupDrawingChildren = function (node, number) {
        if (!DrawingFrame.isGroupDrawingFrame(node) || !_.isNumber(number)) { return null; }
        return $(node).children().first()[0].childNodes[number];
    };

    /**
     * Returns all drawing frames inside a specified drawing group. This does
     * not only contain direct children, but also the children of further groups
     * inside the group. If the option 'onlyDirectChildren' is set to true,
     * only the direct children are returned.
     * Returns null, if the specified node is not a group drawing node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @param {Object} [options]
     *  A map with options to control the selection. The following options are
     *  supported:
     *  @param {Boolean} [options.onlyDirectChildren=false]
     *      If set to true, only those drawing nodes are returned, that are
     *      direct children of the group.
     *
     * @returns {Node|Null}
     *  The child drawing frames of a drawing group at a specified position.
     *  Or null, if it cannot be determined.
     */
    DrawingFrame.getAllGroupDrawingChildren = function (node, options) {

        if (!DrawingFrame.isGroupDrawingFrame(node)) { return null; }

        if (Utils.getBooleanOption(options, 'onlyDirectChildren', false)) {
            return $(node).children().children(DrawingFrame.NODE_SELECTOR);
        } else {
            return $(node).find(DrawingFrame.NODE_SELECTOR);
        }
    };

    /**
     * Returns the closest drawing node of a given node (for example a canvas.border node).
     * If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} node
     *  If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery|Null}
     *  The drawing group node, or null, if it cannot be determined.
     */
    DrawingFrame.getDrawingNode = function (node) {

        var // the group content node is the direct child of the group node itself
            drawingNode = $(node).closest(DrawingFrame.NODE_SELECTOR);

        return drawingNode ? drawingNode : null;
    };

    /**
     * Returns the closest drawing node of a given node that is a text frame shape
     * drawing node. If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} node
     *  If this object is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {jQuery|Null}
     *  The text frame drawing node, or null, if it cannot be determined.
     */
    DrawingFrame.getClosestTextFrameDrawingNode = function (node) {

        var // the closest drawing node that is ancestor of the specified node
            drawingNode = DrawingFrame.getDrawingNode(node);

        if (!drawingNode) { return null; }

        return DrawingFrame.isTextFrameShapeDrawingFrame(drawingNode) ? drawingNode : null;
    };

    /**
     * Returns the closest drawing group node of a drawing node that is grouped.
     * If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} drawingNode
     *  The grouped drawing node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to determine the group node. The following options are
     *  supported:
     *  @param {Boolean} [options.farthest=false]
     *      If set to true, that group node is returned, that is the highest in
     *      the dom. This is necessary, because groups can be included into groups.
     *      Otherwise the closest group node is returned.
     *  @param {Node} [options.rootNode]
     *      If the option 'farthest' is set to true, the root node must be determined,
     *      until which the ancestors will be searched.
     *
     * @returns {jQuery|Null}
     *  The closest or farthest drawing group node, or null, if it cannot be determined.
     */
    DrawingFrame.getGroupNode = function (node, options) {

        var // the group content node is the direct child of the group node itself
            groupContentNode = null,
            // the root node, until which the ancestors are searched. This is only
            // used, if the option 'farthest' is set to true
            rootNode = null;

        if (Utils.getBooleanOption(options, 'farthest', false)) {
            rootNode = Utils.getOption(options, 'rootNode');
            if (rootNode) { groupContentNode = $(Utils.findFarthest(rootNode, node, DrawingFrame.GROUPCONTENT_SELECTOR)); }
        } else {
            groupContentNode = $(node).closest(DrawingFrame.GROUPCONTENT_SELECTOR);
        }

        return (groupContentNode && groupContentNode.length) > 0 ? groupContentNode.parent() : null;
    };

    /**
     * Returns the container node of a drawing frame, that contains all
     * top-level content nodes (paragraphs and tables).
     *
     * @param {HTMLDivElement|jQuery} drawingNode
     *  The drawing frame DOM node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node from the passed table cell that contains all
     *  top-level content nodes (paragraphs and tables).
     */
    DrawingFrame.getTextFrameNode = function (drawingNode) {
        return $(drawingNode).find('> * > ' + DrawingFrame.TEXTFRAME_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame <div> element, that is
     * used inside drawing frames
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div textframe element.
     */
    DrawingFrame.isTextFrameNode = function (node) {
        return $(node).is(DrawingFrame.TEXTFRAME_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame content <div> element, that is
     * used inside drawing frames
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div textframecontent element.
     */
    DrawingFrame.isTextFrameContentNode = function (node) {
        return $(node).is(DrawingFrame.TEXTFRAMECONTENT_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * automatically resizes its height. In this case the content node contains a
     * specific class.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an automatically resizing text frame drawing node.
     */
    DrawingFrame.isAutoResizeHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has class 'autoresizeheight'
        return DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * automatically resizes its font size. In this case the content node contains a
     * specific class.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an text frame drawing node which automatically resizes its text.
     */
    DrawingFrame.isAutoTextHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has class 'autotextheight'
        return DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZETEXT_SELECTOR);
    };

    /**
     * Returns whether the passed shape attributes "autoResizeHeight" is true
     * but only if "noAutoResize" is false
     *
     * @param {Object} [shapeAttrs]
     *  shape attributes
     *
     * @returns {Boolean}
     *  Whether the attributes automatically resizing height.
     */
    DrawingFrame.isAutoResizeHeightAttributes = function (shapeAttrs) {
        return shapeAttrs && (shapeAttrs.autoResizeHeight && !shapeAttrs.noAutoResize);
    };

    /**
     * Returns whether the passed shape attributes "autoResizeText" is true
     * but only if "noAutoResize" is false
     *
     * @param {Object} [shapeAttrs]
     *  shape attributes
     *
     * @returns {Boolean}
     *  Whether the attributes automatically resizing font size.
     */
    DrawingFrame.isAutoTextHeightAttributes = function (shapeAttrs) {
        return shapeAttrs && (shapeAttrs.autoResizeText && !shapeAttrs.noAutoResize);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * has word wrap property set to false
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node has no word wrap property.
     */
    DrawingFrame.isNoWordWrapDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has class 'no-wordwrap'
        return DrawingFrame.getContentNode(node).hasClass(DrawingFrame.NO_WORDWRAP_CLASS);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * has a fixed height.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text frame drawing node with fixed height.
     */
    DrawingFrame.isFixedHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has NOT the class 'autoresizeheight'
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && !DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame <div> element, that is
     * used inside drawing frames as container for additional drawings.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div groupcontent element.
     */
    DrawingFrame.isGroupContentNode = function (node) {
        return $(node).is(DrawingFrame.GROUPCONTENT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a unsupported drawing.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a unsupported drawing element.
     */
    DrawingFrame.isUnsupportedDrawing = function (drawingFrame) {
        return $(drawingFrame).hasClass(DrawingFrame.UNSUPPORTED_CLASS);
    };

    /**
     * Returns the model of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Object|Null}
     *  The model object of the specified drawing frame, as specified while the
     *  frame has been created.
     */
    DrawingFrame.getModel = function (drawingFrame) {
        return $(drawingFrame).first().data(DrawingFrame.DATA_MODEL);
    };

    /**
     * Returns the unique identifier of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The unique identifier of the specified drawing frame, as specified
     *  while the frame has been created.
     */
    DrawingFrame.getUid = function (drawingFrame) {
        return $(drawingFrame).first().data(DrawingFrame.DATA_UID);
    };

    /**
     * Returns the root content node of the specified drawing frame containing
     * all type specific contents of the drawing.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with additional options. The following options are supported:
     *  @param {Boolean} [options.isResizeActive=false]
     *     If this method is called during resizing of the shape.
     *
     * @returns {jQuery}
     *  The root content node of the specified drawing frame.
     */
    DrawingFrame.getContentNode = function (drawingFrame, options) {
        var selector = '.' + CONTENT_CLASS + (Utils.getBooleanOption(options, 'isResizeActive', false) ? '.copy' : ':not(.copy)');
        return $(drawingFrame).first().children(selector);
    };

    /**
     * Returns the Canvas node (canvas) of the specified drawing frame
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The canvas node of the specified drawing frame.
     */
    DrawingFrame.getCanvasNode = function (drawingFrame) {
        return $(drawingFrame).find(DrawingFrame.CANVAS_NODE_SELECTOR);
    };

    /**
     * Returns whether the specified node is a canvas node (canvas)
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  Whether the specified node is a canvas node.
     */
    DrawingFrame.isCanvasNode = function (node) {
        return $(node).is(DrawingFrame.CANVAS_NODE_SELECTOR);
    };

    /**
     * Returns whether the specified node is currently modified by move, resize or rotate operations.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node element.
     *
     * @returns {Boolean}
     */
    DrawingFrame.isModifyingDrawingActive = function (drawingNode) {
        var $node = $(drawingNode);

        return $node.hasClass('activedragging') || $node.hasClass('rotating-state');
    };

    /**
     * Clears and returns the root content node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The empty root content node of the specified drawing frame.
     */
    DrawingFrame.getAndClearContentNode = function (drawingFrame) {
        return DrawingFrame.getContentNode(drawingFrame).empty().removeClass(DrawingFrame.PLACEHOLDER_CLASS).attr('style', '');
    };

    /**
     * Generating the target inheritance chain for a specified drawing node. This inheritance
     * chain need to be defined, if a color scheme is used in the merged attributes. Because
     * the drawing is painted in 'DrawingFrame.drawShape', where the drawing node is no longer
     * available, it is necessary to add the target chain into the merged attributes. For
     * performance reasons this is only done, is the attributes contain a scheme color for
     * fill color or line color.
     *
     * @param {Object} docModel
     *  The application model.
     *
     * @param {HTMLElement|jQuery} node
     *  The drawing node.
     *
     * @param {Object} mergedAttrs
     *  The complete set of drawing attributes. It can happen, that this object is modified
     *  within this function!
     *  This can also be an incomplete set of attributes, if it is called for grouped
     *  drawings (48096).
     */
    DrawingFrame.handleTargetChainInheritance = function (docModel, node, mergedAttrs) {

        if (!docModel.useSlideMode()) { return; }

        var targetChain = null;

        if (mergedAttrs) {
            if (AttributeUtils.isFillThemed(mergedAttrs.fill) || AttributeUtils.isLineThemed(mergedAttrs.line)) {
                // setting theme information for colors at 'line' and 'fill' family
                // -> setting the target chain always for gradients, because the theme color might be hidden in
                //    color stops (47924)
                targetChain = docModel.getTargetChainForNode(node);
                mergedAttrs.fill = mergedAttrs.fill || {};
                mergedAttrs.line = mergedAttrs.line || {};
                mergedAttrs.fill[DrawingFrame.TARGET_CHAIN_PROPERTY] = targetChain;
                mergedAttrs.line[DrawingFrame.TARGET_CHAIN_PROPERTY] = targetChain;
            }
        }
    };

    /**
     * Increasing the height of a group drawing frame, that contains text frames
     * with auto resize height functionality. Those text frames expand automatically
     * the height of the drawing, if required.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The text frame drawing node with auto resize height functionality.
     */
    DrawingFrame.updateDrawingGroupHeight = function (app, drawingNode) {

        // comparing lower border of auto resize text frame node with the group

        var // the offset of the bottom border in pixel
            drawingBottom = 0,
            // an optional group node surrounding the specified text frame drawing node
            groupNode = null,
            // the offset of the group border in pixel
            groupBottom = 0,
            // the new height of the group drawing node
            newHeight = 0,
            // whether the group iteration need to be continued
            doContinue = true,
            // zoom factor of the view
            zoomFactor = app.getView().getZoomFactor() / 100,
            // collection of all parent group drawing nodes of current drawing node
            rotatedGroupParentNodes = [];

        // helper function to find the pixel positions of all bottom borders inside a group
        function getLowestChildDrawingBottom(node) {

            var // the lowest bottom pixel position of all child drawings inside the specified node
                lowestBottom = null;

            // iterating over all children of the specified (group) node
            _.each(node.children(), function (oneDrawing) {

                var // the bottom pixel position of one child node
                    localBottom = Utils.round(($(oneDrawing).offset().top / zoomFactor) + $(oneDrawing).height(), 1);

                // the larger the value, the deeper the drawing
                if (lowestBottom === null || localBottom > lowestBottom) { lowestBottom = localBottom; }
            });

            return lowestBottom;
        }

        // helper function to unrotate all rotated group parent nodes of current node,
        // so that calling offset on that node will return correct value.
        function unrotateAllGroupParentNodes(drawingNode) {
            var rotatedGroupParentNodes = [];
            var groupParentNodes = $(drawingNode).parents(DrawingFrame.NODE_SELECTOR);

            _.each(groupParentNodes, function (parentNode) {
                var rotationAngle = DrawingFrame.getDrawingRotationAngle(app.getModel(), parentNode);
                if (_.isNumber(rotationAngle) && rotationAngle !== 0) {
                    $(parentNode).css({ transform: '' }); // reset to def to get values
                    rotatedGroupParentNodes.push({ node: parentNode, angle: rotationAngle });
                }
            });
            return rotatedGroupParentNodes;
        }

        var rotationAngle = null;

        // iterating over all groups
        while (doContinue && DrawingFrame.isGroupedDrawingFrame(drawingNode)) {
            rotatedGroupParentNodes = unrotateAllGroupParentNodes(drawingNode);
            rotationAngle = DrawingFrame.getDrawingRotationAngle(app.getModel(), groupNode);
            if (_.isNumber(rotationAngle) && rotationAngle !== 0) {
                $(groupNode).css({ transform: '' }); // reset to def to get values
                rotatedGroupParentNodes.push({ node: groupNode, angle: rotationAngle });
            }

            groupNode = DrawingFrame.getGroupNode(drawingNode);
            groupBottom = groupNode ? Utils.round((groupNode.offset().top / zoomFactor) + groupNode.height(), 1) : null;
            drawingBottom = getLowestChildDrawingBottom($(drawingNode).parent());

            // comparing the bottom borders of group and specified text frame
            if (_.isNumber(groupBottom) && _.isNumber(drawingBottom) && (groupBottom !== drawingBottom)) {
                // increasing or decreasing the height of the group without operation
                newHeight = Utils.round(groupNode.height() + (drawingBottom - groupBottom), 1);
                groupNode.height(newHeight);
                drawingNode = groupNode;
            } else {
                doContinue = false;
            }
        }
        _.each(rotatedGroupParentNodes, function (obj) { // all collected rotated group root nodes need to be returned to original rotation
            $(obj.node).css({ transform: 'rotate(' + obj.angle + 'deg)' });
        });
    };

    /**
     * Prepares a specified drawing frame for text insertion. This requires
     * that the content node inside drawing frame is cleared and then a new
     * text frame child is appended. This new text frame element is returned
     * and can be used as parent for paragraphs, tables, ... .
     * Only drawingFrames of type 'shape' allow this conversion.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The new text frame 'div' element, that can be used as container for
     *  the text content.
     */
    DrawingFrame.prepareDrawingFrameForTextInsertion = function (drawingFrame) {

        // only drawing frames of type 'shape' can be converted
        if (!DrawingFrame.isShapeDrawingFrame(drawingFrame)) { return $(); }

        var // the content node of the drawing
            contentNode = DrawingFrame.getAndClearContentNode(drawingFrame),
            // the text frame node inside the content node, parent of paragraphs and tables
            // -> required for setting cursor type, distance to content node and click event
            textFrameNode = $('<div>').addClass('textframe').attr('contenteditable', true);

        // making content node visible by assigning a border via class 'textframecontent'
        contentNode.addClass(DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS).append(textFrameNode);
        // in IE the drawing node itself must have contenteditable 'false', otherwise the
        // grabbers will be visible. Other browsers have to use the value 'true'.
        handleContentEditable(drawingFrame);

        // returning the new created text frame node
        return textFrameNode;
    };

    /**
     * Inserts replacement layout nodes for unsupported drawing types. Inserts
     * the passed name and description of the drawing as text.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {String} [name]
     *  The name of the drawing frame.
     *
     * @param {String} [description]
     *  The description text for the drawing frame.
     */
    DrawingFrame.insertReplacementNodes = function (drawingFrame, name, description) {

        var // the type of the drawing frame
            type = DrawingFrame.getDrawingType(drawingFrame),
            // the empty content node
            contentNode = DrawingFrame.getAndClearContentNode(drawingFrame),
            // the inner width and height available in the content node
            innerWidth = math_max(0, drawingFrame.outerWidth() - 2),
            innerHeight = math_max(0, drawingFrame.outerHeight() - 2),
            // the vertical padding in the content node
            verticalPadding = Utils.minMax(math_min(innerWidth, innerHeight) / 24, 1, 6),
            // the font size of the picture icon
            pictureIconSize = Utils.minMax(math_min(innerWidth - 16, innerHeight - 28), 8, 72),
            // the base font size of the text
            fontSize = Utils.minMax(math_min(innerWidth, innerHeight) / 4, 9, 13);

        // set border width at the content node, insert the picture icon
        contentNode
            .addClass(DrawingFrame.PLACEHOLDER_CLASS)
            .css({
                padding: math_floor(verticalPadding) + 'px ' + math_floor(verticalPadding * 2) + 'px',
                fontSize: fontSize + 'px'
            })
            .append(
                $('<div>')
                    .addClass('abs background-icon')
                    .css('line-height', innerHeight + 'px')
                    .append(Forms.createIconMarkup(Labels.getDrawingTypeIcon(type), { style: 'font-size:' + pictureIconSize + 'px' })),
                $('<p>').text(_.noI18n(name) || Labels.getDrawingTypeLabel(type))
            );

        // insert description if there is a reasonable amount of space available
        if ((innerWidth >= 20) && (innerHeight >= 20)) {
            contentNode.append($('<p>').text(_.noI18n(description)));
        }
    };

    // formatting -------------------------------------------------------------

    DrawingFrame.drawGradient = function (/*fillAttrs, canvas, context*/) {

    };

    /**
     * Updates the CSS formatting of the passed drawing frame, according to the
     * passed generic formatting attributes.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame to be updated. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {Object} mergedAttributes
     *  The drawing attribute set (a map of attribute maps as name/value pairs,
     *  keyed by the 'drawing' attribute family), containing the effective
     *  attribute values merged from style sheets and explicit attributes.
     */
    DrawingFrame.updateFormatting = function (app, drawingFrame, mergedAttributes) {

        var // the document model
            docModel = app.getModel(),
            // the drawing attributes of the passed attribute set
            drawingAttributes = mergedAttributes.drawing,
            // type of the drawing object: 'image', ...
            type = DrawingFrame.getDrawingType(drawingFrame),
            // the content node inside the drawing frame
            contentNode = DrawingFrame.getContentNode(drawingFrame),
            // the group node belonging to a grouped drawing node
            drawingGroupNode = null,
            // the replacement data for unsupported drawings
            replacementData = Utils.getStringOption(drawingAttributes, 'replacementData', ''),
            // rendering result (false to fall-back to placeholder graphic)
            rendered = false;

        // first remove the class which marks this drawing as unsupported
        drawingFrame.removeClass(DrawingFrame.UNSUPPORTED_CLASS);

        // helper function to get all node attributes (TODO: This needs to be valid for all applications)
        function getAllDrawingAttributes(node, family) {

            var // the drawing style collection
                styleCollection = docModel.getStyleCollection('drawing'),
                // the element specific attributes
                elementAttributes = styleCollection ? styleCollection.getElementAttributes(node) : null,
                // the family specific attributes of the element
                familyAttributes = elementAttributes ? elementAttributes[family] : null;

            return familyAttributes;
        }

        // creates an image node in the content node with the specified source URL
        function createImageNodeFromUrl(srcUrl) {
            return app.createImageNode(srcUrl, { timeout: 15000 }).done(function (imgNode) {
                var cropNode = $('<div class="' + CROPPING_CLASS + '">').append(imgNode);
                DrawingFrame.getAndClearContentNode(drawingFrame).append(cropNode);
            });
        }

        // creates an image node or an SVG image according to the passed image data
        function createImageNodeFromData(imageData) {
            if (/^data:/.test(imageData)) {
                return createImageNodeFromUrl(imageData);
            }
            if (/^<svg/.test(imageData)) {
                DrawingFrame.getAndClearContentNode(drawingFrame)[0].innerHTML = imageData;
                return $.when();
            }
            Utils.warn('DrawingFrame.createImageNodeFromData() - unsupported source data for the image object');
            return $.Deferred().reject();
        }

        // updates all formatting for unsupported objects
        function updateUnsupportedFormatting() {
            if (replacementData.length === 0) {
                // insert default replacement nodes, if no replacement data is available
                DrawingFrame.insertReplacementNodes(drawingFrame, drawingAttributes.name, drawingAttributes.description);
            } else if (!drawingFrame.data('has-replacement')) {
                // replacement data MUST NOT change at runtime
                createImageNodeFromData(replacementData);
                drawingFrame.data('has-replacement', true);
            }
            // add class to mark this drawing as a unsupported drawing.
            drawingFrame.addClass(DrawingFrame.UNSUPPORTED_CLASS);
        }

        // updating the size attributes at an image inside a drawing (also supported inside a drawing group)
        function setImageCssSizeAttributes(imageParent, drawingWidth, drawingHeight, drawingAttrs, imageAttrs) {

            var // IE requires to swap cropping settings if image is flipped
                swapHorCrop = _.browser.IE && drawingAttrs.flipH,
                swapVertCrop = _.browser.IE && drawingAttrs.flipV,
                // effective cropping values (checking for 0, because explicit attributes might be used)
                cropLeft = swapHorCrop ? (imageAttrs.cropRight || 0) : (imageAttrs.cropLeft || 0),
                cropRight = swapHorCrop ? (imageAttrs.cropLeft || 0) : (imageAttrs.cropRight || 0),
                cropTop = swapVertCrop ? (imageAttrs.cropBottom || 0) : (imageAttrs.cropTop || 0),
                cropBottom = swapVertCrop ? (imageAttrs.cropTop || 0) : (imageAttrs.cropBottom || 0),
                // horizontal offset/size of cropped bitmaps, as CSS attributes
                horizontalSettings = calculateBitmapSettings(app, drawingWidth, cropLeft, cropRight),
                // vertical offset/size of cropped bitmaps, as CSS attributes
                verticalSettings = calculateBitmapSettings(app, drawingHeight, cropTop, cropBottom);

            // workaround for Bug 48435
            if (cropLeft < 0 && docModel.useSlideMode()) { horizontalSettings.offset = Utils.convertHmmToCssLength(cropLeft, 'px', 1); }

            imageParent.find('img').css({
                left: horizontalSettings.offset,
                top: verticalSettings.offset,
                width: horizontalSettings.size,
                height: verticalSettings.size
            });
        }

        // updates all formatting for image objects
        function updateImageFormatting() {

            var // special attributes for images
                imageAttributes = mergedAttributes.image,
                // the promise that handles the loading of the image
                imageLoadPromise = null;

            // updates attributes of the image node after loading
            // -> but not for grouped drawings. Those drawings are handled by the group container
            function updateImageAttributes(options) {

                var // current width of the drawing frame, in 1/100 mm
                    drawingWidth = Utils.convertLengthToHmm(drawingFrame.width(), 'px'),
                    // current height of the drawing frame, in 1/100 mm
                    drawingHeight = Utils.convertLengthToHmm(drawingFrame.height(), 'px'),
                    // whether the call of this function is asynchronous
                    isAsync = Utils.getBooleanOption(options, 'isAsync', true),
                    // whether the visibility was changed (sometimes required in async mode)
                    visibilityChanged = false;

                if ((drawingWidth > 0) && (drawingHeight > 0)) {

                    if (DrawingFrame.isGroupedDrawingFrame(drawingFrame)) {
                        // For grouped drawings, it is useless to set the image attributes directly,
                        // because these attributes are dependent from the group itself.
                        // -> after the image is loaded (deferred!), the image attributes need to be set again
                        // to be correct for the complete group.
                        // The call of resizeDrawingsInGroup() is only required after loading the document for
                        // grouped drawings when loading happens without fast load and without
                        // local storage.
                        if (isAsync) {
                            if (docModel.useSlideMode()) { visibilityChanged = docModel.handleContainerVisibility(drawingFrame, { makeVisible: true }); }
                            resizeDrawingsInGroup(DrawingFrame.getGroupNode(drawingFrame));
                            if (visibilityChanged) { docModel.handleContainerVisibility(drawingFrame, { makeVisible: false }); }
                        }
                    } else {
                        setImageCssSizeAttributes(contentNode, drawingWidth, drawingHeight, drawingAttributes, imageAttributes);
                    }
                    // remove class which marks this drawing as unsupported
                    drawingFrame.removeClass(DrawingFrame.UNSUPPORTED_CLASS);
                    // update the border
                    updateLineFormatting();
                }
            }

            // create the image or SVG node (considered to be constant at runtime)
            if (contentNode.find('>.' + CROPPING_CLASS).children().length === 0) {
                // start with replacement graphics (image data will be loaded asynchronously)
                updateUnsupportedFormatting();
                if (imageAttributes.imageData.length > 0) {
                    imageLoadPromise = createImageNodeFromData(imageAttributes.imageData).done(updateImageAttributes);
                } else if (imageAttributes.imageUrl.length > 0) {
                    // convert relative URL to an application-specific absolute URL
                    imageLoadPromise = createImageNodeFromUrl(/:\/\//.test(imageAttributes.imageUrl) ? imageAttributes.imageUrl : app.getServerModuleUrl(IO.FILTER_MODULE_NAME, { action: 'getfile', get_filename: imageAttributes.imageUrl })).done(updateImageAttributes);
                } else {
                    Utils.warn('DrawingFrame.updateFormatting() - missing source data for the image object');
                    imageLoadPromise = $.when();
                }
            } else {
                // update attributes of an existing image element
                updateImageAttributes({ isAsync: false });
                imageLoadPromise = $.when();
            }

            imageLoadPromise.then(function () {
                if (docModel.useSlideMode()) { docModel.trigger('image:loaded', drawingFrame); } // this might be interesting for other apps, too
            });

            // image successfully rendered
            return true;
        }

        // updates chart formatting
        function updateChartFormatting() {

            var // the drawing object model
                model = DrawingFrame.getModel(drawingFrame),
                // the chart rendering engine (lazy initialization, see below)
                renderer = drawingFrame.data('chart-renderer'),
                // unique DOM element identifier for the renderer
                chartId = null,
                drawingNode = null;

            // drawing object model must exist for chart objects
            if (!model) { return false; }

            // lazy creation of the renderer
            if (!renderer) {
                chartId = 'io-ox-documents-chart-' + DrawingFrame.getUid(drawingFrame);
                contentNode.addClass('chartholder');

                // localization for CanvasJS
                if (!CULTURE_INFO) {
                    CULTURE_INFO = {
                        decimalSeparator: LocaleData.DEC,
                        digitGroupSeparator: LocaleData.GROUP,
                        shortDays: LocaleData.SHORT_WEEKDAYS.slice(),
                        days: LocaleData.LONG_WEEKDAYS.slice(),
                        shortMonths: LocaleData.SHORT_MONTHS.slice(),
                        months: LocaleData.LONG_MONTHS.slice()
                    };
                    CanvasJS.addCultureInfo('en', CULTURE_INFO);
                }

                drawingNode = $('<div>');
                drawingNode.addClass('chartnode');
                drawingNode.attr('id', chartId);
                contentNode.append(drawingNode);

                // really important call, because of a bug in canvasJS
                model.resetData();

                renderer = new CanvasJS.Chart(chartId, model.getModelData());
                drawingFrame.data('chart-renderer', renderer);
                model.firstInit();
            }

            // render the chart object
            model.updateRenderInfo();
            if (model.hasDataPoints()) {
                try {
                    renderer.render();
                } catch (ex) {
                    Utils.exception(ex, 'chart rendering error');
                }
            }
            // background color of model data is already prepared for css
            contentNode.css('backgroundColor', model.getModelData().cssBackgroundColor);

            // chart successfully rendered
            return true;
        }

        // updates drawings of type 'table'
        function updateTableFormatting() {

            // adding auto resize height to the table
            contentNode.addClass(DrawingFrame.AUTORESIZEHEIGHT_CLASS);

            // table successfully rendered
            return true;
        }

        /**
         * Updates the fill formatting of the drawing frame.
         *
         * @param {jQuery} [node]
         *  An optional jQuery node, to which the css properties are
         *  assigned. If it is not set, the content node inside the
         *  drawing frame is used. Setting this node, is only required
         *  for drawing groups, where all children need to be modified.
         */
        function updateFillFormatting() {

            var // the document model
                docModel = app.getModel(),
                // the CSS properties for the drawing frame
                cssProps = {},
                // the node, whose properties will be set
                cssNode = contentNode,
                // the fill attributes
                fillAttrs = mergedAttributes.fill;

            if (fillAttrs) {
                // calculate the CSS fill attributes
                switch (fillAttrs.type) {
                    case 'none':
                        // clear everything: color and bitmaps
                        cssProps.background = '';
                        break;
                    case 'solid':
                        // use 'background' compound attribute to clear bitmaps
                        cssProps.background = docModel.getCssColor(fillAttrs.color, 'fill', fillAttrs[DrawingFrame.TARGET_CHAIN_PROPERTY]);
                        break;
                    default:
                        Utils.warn('DrawingFrame.updateFillFormatting(): unknown fill type "' + fillAttrs.type + '"');
                }

                // apply the fill attributes
                cssNode.css(cssProps);
            }
        }

        /**
         * Updates the border formatting of the drawing frame.
         *
         * @param {jQuery} [node]
         *  An optional jQuery node, to which the css properties are
         *  assigned. If it is not set, the content node inside the
         *  drawing frame is used. Setting this node, is only required
         *  for drawing groups, where all children need to be modified.
         */
        function updateLineFormatting(node) {

            var // gets the correct node to work on (parent() used at grouped drawings)
                currentNode = node ? DrawingFrame.getDrawingNode(node) : drawingFrame,
                // the line attributes
                lineAttrs = node ? AttributeUtils.getExplicitAttributes(currentNode).line : mergedAttributes.line;

            // Explicit attributes are not sufficient, if default line attributes are used (36892)
            if (node && (!lineAttrs || !lineAttrs.type)) { lineAttrs = getAllDrawingAttributes(node, 'line'); }

            if (lineAttrs && lineAttrs.type) {
                // calculate the border properties
                switch (lineAttrs.type) {
                    case 'none':
                        // clear everything
                        DrawingFrame.clearCanvas(currentNode);
                        break;
                    case 'solid':
                        DrawingFrame.drawBorder(app, currentNode, lineAttrs);
                        break;
                    default:
                        Utils.warn('DrawingFrame.updateLineFormatting(): unknown line type "' + lineAttrs.type + '"');
                }
            }
        }

        /**
         * Public method for updating shape formatting.
         *
         * @param {jQuery} node
         *  Shape node that is going to be formatted.
         *
         * @param {Object} [options]
         *  @param {Boolean} [options.isResizeActive=false]
         *      If this method is called during resizing of the shape.
         */
        this.updateShapeFormatting = function (node, options) {
            updateShapeFormatting(node, options);
        };

        /**
         * Updating the 'shape' attributes at the drawing.
         *
         * @param {jQuery} drawing
         *  The jQueryfied drawing node.
         * @param {Object} [options]
         *  @param {Boolean} [options.isResizeActive=false]
         *      If this method is called during resizing of the shape.
         */
        function updateShapeFormatting(node, options) {

            var // gets the correct node to work on (parent() used at grouped drawings)
                currentNode = node ? DrawingFrame.getDrawingNode(node) : drawingFrame,
                // the content node inside the drawing frame
                contentNode = DrawingFrame.getContentNode(currentNode, options), // needs to be fetched each time, because of grouped drawings, see #47962
                // the explicit attributes of the handled node
                explicitAttributes = AttributeUtils.getExplicitAttributes(currentNode),
                // the shape attributes
                shapeAttrs = node ? explicitAttributes.shape : mergedAttributes.shape,
                // the line attributes
                lineAttrs = node ? explicitAttributes.line : mergedAttributes.line,
                // the fill attributes
                fillAttrs = node ? explicitAttributes.fill : mergedAttributes.fill,
                // shape geometry
                geometryAttrs = explicitAttributes.geometry ? explicitAttributes.geometry : mergedAttributes.geometry,
                // the canvas wrapper
                canvas = null,
                // if formatting is called during resize
                isResizeActive = Utils.getBooleanOption(options, 'isResizeActive', false),
                // copied node of textframecontent
                textFrameContentCopy = currentNode.children('.copy'),
                // the text frame node of the drawing
                textFrame = isResizeActive ? textFrameContentCopy.children('.textframe') : DrawingFrame.getTextFrameNode(currentNode),
                // if no word wrap property is set
                noWordWrap = shapeAttrs && shapeAttrs.wordWrap === false,
                // variables for storing current width and height
                currentHeight = null,
                currentWidth = null;

            function orientateTextInShape(angle) {
                var zoomFactor = app.getView().getZoomFactor() / 100;
                // width and height of shape node
                var nodeWidth = null;
                var nodeHeight = null;
                var maxHeight = null;
                var preBoundingRect, postBoundingRect, topDelta, leftDelta;
                var parentAngle = DrawingFrame.getDrawingRotationAngle(docModel, currentNode);
                var notDefaultAngle = angle !== 0;

                currentNode.data({ widthHeightRev: notDefaultAngle });
                if (notDefaultAngle) {
                    nodeWidth = isResizeActive ? textFrameContentCopy.width() : currentNode.width();
                    nodeHeight = isResizeActive ? textFrameContentCopy.height() : currentNode.height();
                    maxHeight = textFrame[0].style.maxHeight;
                    if (_.isNumber(parentAngle)) {
                        currentNode.css({ transform: '' }); // reset to def to get values
                    }

                    textFrame.css({ transform: 'rotate(0deg)' }); // reset previously set rotation
                    preBoundingRect = textFrame[0].getBoundingClientRect();

                    textFrame.css({ width: nodeHeight, height: nodeWidth, maxHeight: '', maxWidth: maxHeight, transform: 'rotate(' + angle + 'deg)' }).addClass(DrawingFrame.ROTATED_TEXT_IN_DRAWING_CLASSNAME);
                    postBoundingRect = textFrame[0].getBoundingClientRect();
                    topDelta = (preBoundingRect.top - postBoundingRect.top) / zoomFactor;
                    leftDelta = (preBoundingRect.left - postBoundingRect.left) / zoomFactor;
                    textFrame.css({ top: topDelta, left: leftDelta });

                    if (_.isNumber(parentAngle)) {
                        currentNode.css({ transform: 'rotate(' + parentAngle + 'deg)' }); // return property to it's value
                    }
                } else {
                    textFrame.css({ transform: 'rotate(0deg)' }).removeClass(DrawingFrame.ROTATED_TEXT_IN_DRAWING_CLASSNAME);
                }
            }

            function getSnapRect() {
                if (isResizeActive) {
                    return { left: 0, top: 0, width: textFrameContentCopy.outerWidth(true), height: textFrameContentCopy.outerHeight(true) };
                } else {
                    var width = noWordWrap ? textFrame.outerWidth(true) : currentNode.first().width();
                    var height = currentNode.first().height();
                    return { left: 0, top: 0, width: width, height: height };
                }
            }

            function drawGeometry(geometry, snapRect, boundRect, guideCache) {
                if (!docModel || docModel.destroyed) { return; } // the model might have been destroyed in the meantime

                function getTextRectValue(value, defaultValue) {
                    return value ? Utils.round(DrawingFrame.getGeometryValue(value, snapRect, geometry.avList, geometry.gdList, guideCache, 1000), 1) : defaultValue;
                }

                var visibilityChanged = false;
                var lineWidth = 1;
                var canvasExpansion = 0; // an optional expansion of the canvas, required for arrows

                if (lineAttrs && lineAttrs.width) { lineWidth = math_max(Utils.convertHmmToLength(lineAttrs.width, 'px', 1), 1); }

                // taking care of special text rects
                if (geometry.textRect && textFrame && textFrame.length > 0) {
                    var left = getTextRectValue(geometry.textRect.l, snapRect.left) + Utils.convertHmmToLength(mergedAttributes.shape.paddingLeft, 'px');
                    var top = getTextRectValue(geometry.textRect.t, snapRect.top) + Utils.convertHmmToLength(mergedAttributes.shape.paddingTop, 'px');
                    var right = getTextRectValue(geometry.textRect.r, snapRect.left + snapRect.width) - Utils.convertHmmToLength(mergedAttributes.shape.paddingRight, 'px');

                    var paddingBottom = (snapRect.height - getTextRectValue(geometry.textRect.b, (snapRect.top + snapRect.height) - 1)) + top
                                        + Utils.convertHmmToLength(mergedAttributes.shape.paddingTop, 'px')
                                        + Utils.convertHmmToLength(mergedAttributes.shape.paddingBottom, 'px');

                    if (docModel.useSlideMode()) {
                        // workaround for Bug 48476, drawings in powerpoint are 2px bigger than they save in document, reduce padding is easier than changing all sizes
                        left -= 1; right += 2;
                    }

                    left = Math.round(left);
                    top = Math.round(top);
                    right = Math.round(right);
                    paddingBottom = Math.round(paddingBottom);

                    // IE workarounds for vertical aligment @see style.less div.textframe
                    // Bug 46803, Bug 42893 & Bug 44965
                    if (_.browser.IE) {
                        textFrame.css({ left: left + 'px', top: top + 'px', width: (right - left) + 'px', paddingBottom: paddingBottom + 'px' });
                        if (shapeAttrs && shapeAttrs.anchor !== 'top') {
                            var offset = shapeAttrs.anchor === 'bottom' ? '100%' : '50%';
                            textFrame.css('top', 'calc(' + offset + ' + ' + top + 'px)');
                        }
                        if (docModel.useSlideMode()) {
                            if (noWordWrap) {
                                textFrame.css({ width: 'auto', whiteSpace: 'nowrap' });
                                contentNode.addClass(DrawingFrame.NO_WORDWRAP_CLASS);
                                if (DrawingFrame.isAutoResizeHeightAttributes(shapeAttrs)) {
                                    currentNode.css('width', 'auto');
                                }
                            } else {
                                textFrame.css('white-space', '');
                            }
                        }
                    } else {
                        // changed from "paddingBottom" to "height - padding" & "marginBottom", so textframe does not gets bigger than its parent drawing (needed for selection)
                        textFrame.css({ left: left + 'px', top: top + 'px', width: noWordWrap ? 'auto' : ((right - left) + 'px'), height: 'calc(100% - ' + paddingBottom + 'px)', marginBottom: paddingBottom + 'px' });
                        if (noWordWrap) {
                            textFrame.css({ left: '', marginLeft: left, marginRight: left, whiteSpace: 'nowrap' });
                        }
                    }

                    // the textframe been changed so also the snaprect, we have to reset the guideCache and also
                    // reset the new snaprect
                    guideCache = {};

                    if (docModel.useSlideMode()) { visibilityChanged = docModel.handleContainerVisibility(currentNode.first(), { makeVisible: true }); }
                    snapRect = getSnapRect();
                    if (visibilityChanged) { docModel.handleContainerVisibility(currentNode.first(), { makeVisible: false }); }

                }

                // checking, if the canvas needs an increased size, because of arrows at line ends
                if (lineAttrs && geometry) {
                    canvasExpansion = getCanvasExpansion(lineAttrs, geometry);
                    if (canvasExpansion > 0) { currentNode.data('canvasexpansion', canvasExpansion); }
                }

                boundRect = DrawingFrame.calcBoundRectPixel(snapRect, canvasExpansion);

                canvas = DrawingFrame.initializeCanvas(currentNode, boundRect, lineWidth, isResizeActive ? { resizeIsActive: true } : null);

                canvas.render(function (context) {
                    if (('pathList' in geometry) && (geometry.pathList.length > 0)) {
                        DrawingFrame.drawShape(docModel, context, snapRect, geometry, fillAttrs, lineAttrs, guideCache, { isResizeActive: isResizeActive, currentNode: currentNode });
                        return;
                    }

                    var hasFill = fillAttrs && (fillAttrs.type !== 'none');
                    var hasLine = lineAttrs && (lineAttrs.type !== 'none');
                    var mode = getRenderMode(hasFill, hasLine);
                    var rotationAngle = null;
                    var notRotatedWithShape = null;

                    function drawRect(forcedMode) {
                        var mode = forcedMode || getRenderMode(hasFill, hasLine);
                        context.drawRect(boundRect, mode);
                    }

                    if (hasFill) {
                        var
                            target = fillAttrs[DrawingFrame.TARGET_CHAIN_PROPERTY];

                        if (fillAttrs.type === 'gradient') {
                            var
                                gradient = Gradient.create(Gradient.getDescriptorFromAttributes(fillAttrs));

                            context.setFillStyle(docModel.getGradientFill(gradient, boundRect, context, target));
                        } else if (fillAttrs.type === 'pattern') {
                            context.setFillStyle(Pattern.createCanvasPattern(fillAttrs.pattern, fillAttrs.backgroundColor, fillAttrs.color, docModel.getTheme(target)));
                        } else if (fillAttrs.type === 'bitmap') {
                            notRotatedWithShape = fillAttrs.bitmap.rotateWithShape === false;
                            if (notRotatedWithShape && currentNode) {
                                rotationAngle = DrawingFrame.getDrawingRotationAngle(docModel, currentNode) || 0;
                                rotationAngle = ((360 - rotationAngle) % 360) * (Math.PI / 180); // normalize and convert to radians
                            }
                            Texture.getTextureFill(docModel, context, fillAttrs, snapRect.width, snapRect.height, rotationAngle).then(function (textureFillStyle) {
                                setBitmapTextureFillStyle(docModel, context, drawRect, textureFillStyle, snapRect.width, snapRect.height, hasLine, lineAttrs, fillAttrs.bitmap.isTiling, notRotatedWithShape, rotationAngle);
                            });
                            return;
                        } else {
                            context.setFillStyle(docModel.getCssColor(fillAttrs.color, 'fill', target));
                        }
                    }

                    if (hasLine) {
                        setLineAttributes(docModel, context, lineAttrs, lineWidth);
                    }
                    if (mode) { context.drawRect(boundRect, mode); }
                });
            }

            // Handling the target chain inheritance also for grouped drawings (48096)
            // Or during active resizing of shape to fetch correct color from themes
            if (DrawingFrame.isGroupedDrawingFrame(node) || isResizeActive) {
                DrawingFrame.handleTargetChainInheritance(docModel, node, explicitAttributes);
            }

            // applying text padding & vertical alignment
            if (textFrame && textFrame.length > 0) {
                if (shapeAttrs) {
                    if (shapeAttrs.anchor) {
                        textFrame.attr(DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE, shapeAttrs.anchor);
                    }
                    if (!geometryAttrs) {
                        textFrame.css({
                            paddingTop: Utils.convertHmmToCssLength(mergedAttributes.shape.paddingTop, 'px', 1),
                            paddingLeft: Utils.convertHmmToCssLength(mergedAttributes.shape.paddingLeft, 'px', 1),
                            paddingBottom: Utils.convertHmmToCssLength(mergedAttributes.shape.paddingBottom, 'px', 1),
                            paddingRight: Utils.convertHmmToCssLength(mergedAttributes.shape.paddingRight, 'px', 1)
                        });
                    }
                }
            }

            // setting the value for automatic vertical resizing of the shape.
            if (DrawingFrame.isAutoResizeHeightAttributes(shapeAttrs)) {
                if (currentNode.data('widthHeightRev')) {
                    textFrame.css('height', 'auto');
                    currentNode.width(textFrame.outerHeight(true));
                    currentNode.height(textFrame.outerWidth(true));
                }
                if (!currentNode.data('drawingHeight') && !DrawingFrame.isGroupContentNode(currentNode.parent())) {
                    currentHeight = (explicitAttributes && explicitAttributes.drawing && Utils.convertHmmToLength(explicitAttributes.drawing.height, 'px', 1)) || currentNode.height();
                    currentNode.data('drawingHeight', currentHeight);
                }
                contentNode.addClass(DrawingFrame.AUTORESIZEHEIGHT_CLASS);
                // updating the height of a surrounding group node
                if (DrawingFrame.isGroupedDrawingFrame(contentNode.parent())) { DrawingFrame.updateDrawingGroupHeight(app, contentNode.parent()); }
            } else {
                contentNode.removeClass(DrawingFrame.AUTORESIZEHEIGHT_CLASS);
                currentNode.data('drawingHeight', null);
            }

            if (docModel.useSlideMode()) {
                // setting the value for automatic text height in the shape.
                if (DrawingFrame.isAutoTextHeightAttributes(shapeAttrs)) {
                    contentNode.css('font-size', (shapeAttrs.fontScale * 16) + 'px');
                    contentNode.attr('font-scale', shapeAttrs.fontScale);
                    contentNode.attr('lineReduction', math_round(math_min(0.2, shapeAttrs.lineReduction) * 100));
                    contentNode.addClass(DrawingFrame.AUTORESIZETEXT_CLASS);
                } else {
                    contentNode.css('font-scale', null);
                    contentNode.css('font-size', null);
                    contentNode.attr('lineReduction', null);
                    contentNode.removeClass(DrawingFrame.AUTORESIZETEXT_CLASS);
                }
                if (noWordWrap) {
                    textFrame.css('width', 'auto');
                    if (DrawingFrame.isAutoResizeHeightAttributes(shapeAttrs)) { // #47690, #48153
                        currentNode.css('width', 'auto');
                    }
                    contentNode.addClass(DrawingFrame.NO_WORDWRAP_CLASS);
                    if (!currentNode.data('drawingWidth') && !DrawingFrame.isGroupContentNode(currentNode.parent())) {
                        currentWidth = (explicitAttributes && explicitAttributes.drawing && Utils.convertHmmToLength(explicitAttributes.drawing.width, 'px', 1)) || currentNode.width();
                        currentNode.data('drawingWidth', currentWidth);
                    }
                } else {
                    contentNode.removeClass(DrawingFrame.NO_WORDWRAP_CLASS);
                    currentNode.data('drawingWidth', null);
                    textFrame.css('white-space', '');
                }
            }

            if (!geometryAttrs) {
                updateFillFormatting();
                updateLineFormatting(node);
            } else {
                // bugfix :: BUG#45141 :: https://bugs.open-xchange.com/show_bug.cgi?id=45141 :: Vertical drawing object is shown diagonally
                if (geometryAttrs.presetShape === 'line') {
                    if (currentNode.width() === 0) {
                        currentNode.width(1);
                    }
                    if (currentNode.height() === 0) {
                        currentNode.height(1);
                    }
                }
                // the canvas wrapper of the border node
                var snapRect = getSnapRect();
                var boundRect = DrawingFrame.calcBoundRectPixel(snapRect);
                var guideCache = {};

                if (('presetShape' in geometryAttrs) && PRESET_GEOMETRIES[geometryAttrs.presetShape]) {
                    var presetGeo = PRESET_GEOMETRIES[geometryAttrs.presetShape],
                        geo = _.defaults(_.clone(geometryAttrs), presetGeo);
                    if (presetGeo.avList && geo.avList) {
                        _.defaults(geo.avList, presetGeo.avList);
                    }
                    drawGeometry(geo, snapRect, boundRect, guideCache);
                } else {
                    drawGeometry(geometryAttrs, snapRect, boundRect, guideCache);
                }
            }
            // applying text direction and stacking
            if (shapeAttrs && shapeAttrs.vert && textFrame.length) {
                switch (shapeAttrs.vert) {
                    case 'vert':
                    case 'eaVert':
                        orientateTextInShape(90); // vertical (90deg)
                        break;
                    case 'vert270':
                        orientateTextInShape(270); // vertical reversed (270deg)
                        break;
                    case 'horz':
                        if (DrawingFrame.isRotatedTextInDrawingNode(textFrame)) {
                            orientateTextInShape(0); // revert from vertical to horizontal
                        }
                        break;
                }
            }
        }

        // updates the class at the drawing, that is used to find standard odt text frames (and not shapes with text)
        function updateTextFrameState() {

            var // the explicit drawing attributes (not using merge attributes, because they always contain a style id)
                attrs = AttributeUtils.getExplicitAttributes(drawingFrame);

            if (attrs && attrs.styleId && _.isString(attrs.styleId)) {
                drawingFrame.addClass(DrawingFrame.ODFTEXTFRAME_CLASS);
            } else {
                drawingFrame.removeClass(DrawingFrame.ODFTEXTFRAME_CLASS);
            }

        }

        /**
         * Updates the size and position of all children of a drawing group,
         * but only temporary, during resize of the group.
         * INFO: Use private method for final resize!
         *
         * @param {jQuery} drawing
         */
        DrawingFrame.previewOnResizeDrawingsInGroup = function (drawing) {
            resizeDrawingsInGroup(drawing, { resizeActive: true });
        };

        /**
         * Updates the size and position of all children of a drawing group.
         */
        function resizeDrawingsInGroup(drawing, options) {

            var // if resize is active, calculate live dimensions, not from attributes
                isResizeActive = Utils.getBooleanOption(options, 'resizeActive', false),
                // copied content of the drawing
                $content = DrawingFrame.isGroupedDrawingFrame(drawing) ? drawing.children('.content') : drawing.children('.copy'),
                // the direct drawing children of the drawing group
                allChildren = isResizeActive ? $content.children(DrawingFrame.NODE_SELECTOR) : DrawingFrame.getAllGroupDrawingChildren(drawing, { onlyDirectChildren: true }),
                // the maximum width of the children
                maxWidth = 0,
                // the maximum height of the children
                maxHeight = 0,
                // width of the group
                groupWidth = 0,
                // height of the group
                groupHeight = 0,
                // all horizontal offsets
                allHorzOffsets = [],
                // all vertical offsets
                allVertOffsets = [],
                // the horizontal offset of one drawing
                offsetHorz = 0,
                // the vertical offset of one drawing
                offsetVert = 0,
                // the horizontal stretch factor for the children
                horzFactor = 1,
                // the vertical stretch factor for the children
                vertFactor = 1,
                // zoom factor of the application
                zoomFactor = app.getView().getZoomFactor() / 100,
                // rotation angle of the drawing group element
                rotationAngle = DrawingFrame.getDrawingRotationAngle(docModel, drawing),
                // collection of drawing attributes for each drawing in group
                drawingAttrsCollection = [],
                // if rotation angle is set
                isRotationAngle = _.isNumber(rotationAngle) && rotationAngle !== 0;

            function getGroupWidthHeight() {
                var groupWidth, groupHeight;

                if (isResizeActive) {
                    var contentBoundingRect = $content[0].getBoundingClientRect();
                    groupWidth = Utils.convertLengthToHmm(contentBoundingRect.width, 'px');
                    groupHeight = Utils.convertLengthToHmm(contentBoundingRect.height, 'px');
                } else {
                    // whether the explicit drawing attributes can be used (this is not the case for groups in groups, where
                    // the explicit attributes do not contain values in hmm, but in an arbitrary unit).
                    var isToplevelGroup = !DrawingFrame.isDrawingContentNode(drawing.parent());  // Info: 'grouped' not set for groups in groups in first run (46784)
                    // the explicit attributes at the group node (should be used for precision)
                    var explicitGroupAttributes = isToplevelGroup ? AttributeUtils.getExplicitAttributes(drawing) : null;
                    // the explicit drawing attributes
                    var explicitDrawingAttributes = explicitGroupAttributes && explicitGroupAttributes.drawing;
                    // the width of the drawing group element in 1/100 mm
                    groupWidth = (explicitDrawingAttributes && _.isNumber(explicitDrawingAttributes.width)) ? explicitDrawingAttributes.width : Utils.convertLengthToHmm(drawing.children().first().width(), 'px');
                    // the height of the drawing group element in  1/100 mm
                    groupHeight = (explicitDrawingAttributes && _.isNumber(explicitDrawingAttributes.height)) ? explicitDrawingAttributes.height : Utils.convertLengthToHmm(drawing.children().first().height(), 'px');
                }
                return { width: groupWidth, height: groupHeight };
            }

            function getUnrotatedDimensions(oneDrawing) {
                var boundRect;
                var rotationAngle = DrawingFrame.getDrawingRotationAngle(docModel, oneDrawing);
                var isRotated = _.isNumber(rotationAngle) && rotationAngle !== 0;

                if (isRotated) {
                    $(oneDrawing).css({ transform: '' }); // reset to def to get values
                }
                boundRect = oneDrawing.getBoundingClientRect();
                if (isRotated) {
                    $(oneDrawing).css({ transform: 'rotate(' + rotationAngle + 'deg)' }); // return property to it's value
                }

                return { width: Utils.convertLengthToHmm(boundRect.width, 'px'), height: Utils.convertLengthToHmm(boundRect.height, 'px'), top: Utils.convertLengthToHmm(boundRect.top, 'px'), left: Utils.convertLengthToHmm(boundRect.left, 'px') };
            }

            // updates the size and position of one drawing child inside a drawing group
            function handleDrawingInsideDrawingGroup(drawing, attrs, horzFactor, vertFactor, offsetX, offsetY, index) {

                var // the attributes at the drawing property
                    drawingAttrs = isResizeActive ? drawingAttrsCollection[index] : attrs.drawing,
                    // the required css attributes
                    cssAttrs = {},
                    // an optional horizontal resize factor
                    horzResizeFactor = horzFactor || 1,
                    // an optional vertical resize factor
                    vertResizeFactor = vertFactor || 1,
                    // type of the drawing object inside the group: 'image', ...
                    type = DrawingFrame.getDrawingType(drawing);

                // updates attributes of the image node of a drawing frame inside a drawing group container
                function updateImageAttributesInGroup() {

                    var // current width of the image, in 1/100 mm
                        drawingWidth = Math.max(Utils.round(drawingAttrs.width * horzResizeFactor, 1), 100),
                        // current height of the image, in 1/100 mm
                        drawingHeight = Math.max(Utils.round(drawingAttrs.height * vertResizeFactor, 1), 100);

                    setImageCssSizeAttributes(drawing.children().first(), drawingWidth, drawingHeight, attrs.drawing, attrs.image);
                }

                if (!attrs.drawing) { return; } // drawing attributes must be defined (47633)

                // set the CSS attributes for the drawing
                if (_.isNumber(drawingAttrs.left))  { cssAttrs.left     = Utils.convertHmmToCssLength((drawingAttrs.left - offsetX) * horzResizeFactor, 'px', 1); }
                if (_.isNumber(drawingAttrs.top))   { cssAttrs.top      = Utils.convertHmmToCssLength((drawingAttrs.top - offsetY) * vertResizeFactor, 'px', 1); }
                if (_.isNumber(drawingAttrs.width)) { cssAttrs.width    = Math.max(Utils.convertHmmToLength(drawingAttrs.width * horzResizeFactor, 'px', 1), 1) + 'px'; }
                if (_.isNumber(drawingAttrs.height) && !DrawingFrame.isAutoResizeHeightDrawingFrame(drawing)) { // not setting height for auto resize drawings
                    cssAttrs.height = Math.max(Utils.convertHmmToLength(drawingAttrs.height * vertResizeFactor, 'px', 1), 1) + 'px';
                }

                drawing.css(cssAttrs);

                // updating the size of the images inside the drawing
                if (type === 'image') { updateImageAttributesInGroup(drawing); }

                // updating the canvas (border and background)
                if (type !== 'group') { updateShapeFormatting(drawing); }

                // handling groups in groups
                if (DrawingFrame.isGroupDrawingFrame(drawing)) {
                    resizeDrawingsInGroup(drawing, options);
                }
            }

            if (drawing.hasClass('activedragging') && !isResizeActive && !drawing.hasClass('activeresizing')) {
                return; // if updateShape was triggered by move operation, resizing inside groups is unnecessary
            }

            if (isResizeActive && isRotationAngle) {
                drawing.css({ transform: '' }); // reset to def to get values
            }
            var groupWidthHeight = getGroupWidthHeight();
            groupWidth = groupWidthHeight.width;
            groupHeight = groupWidthHeight.height;

            // iterating over all drawing children of the group
            if (allChildren.length > 0) {

                // comparing width and height of the group with all values of the children
                // -> the aim is, that all children fit optimal into the group
                // -> this needs to be set after loading and after resizing

                // getting the maximum value of the children for 'left + width' and 'top + height')
                _.each(allChildren, function (drawing) {

                    var // the explicit drawing attributes
                        drawingAttrs = isResizeActive ? getUnrotatedDimensions(drawing) : AttributeUtils.getExplicitAttributes(drawing),
                        // the horizontal expansion of one drawing
                        currentHorz = 0,
                        // the vertical expansion of one drawing
                        currentVert = 0;

                    if (drawingAttrs) {
                        if (!isResizeActive) {
                            if (!drawingAttrs.drawing) { return; }

                            drawingAttrs = drawingAttrs.drawing;
                            if ('rotation' in drawingAttrs) {
                                drawingAttrs = _.extend(drawingAttrs, DrawingUtils.getRotatedDrawingPoints(drawingAttrs, drawingAttrs.rotation));
                            }
                        } else {
                            drawingAttrsCollection.push(drawingAttrs); // cache drawing attrs in collection for later reading
                        }

                        if (_.isNumber(drawingAttrs.left)) { currentHorz = drawingAttrs.left; }
                        if (_.isNumber(drawingAttrs.width)) { currentHorz += drawingAttrs.width; }
                        if (_.isNumber(drawingAttrs.top)) { currentVert = drawingAttrs.top; }
                        if (_.isNumber(drawingAttrs.height)) { currentVert += drawingAttrs.height; }

                        if (currentHorz > maxWidth) { maxWidth = currentHorz; }
                        if (currentVert > maxHeight) { maxHeight = currentVert; }

                        // collecting all horizontal and vertical offset positions
                        allHorzOffsets.push(_.isNumber(drawingAttrs.left) ? drawingAttrs.left : 0);
                        allVertOffsets.push(_.isNumber(drawingAttrs.top) ? drawingAttrs.top : 0);
                    }
                });

                if (maxWidth > 0 || maxHeight > 0) {

                    offsetHorz = math_min.apply(null, allHorzOffsets);
                    offsetVert = math_min.apply(null, allVertOffsets);

                    // now checking values for maxWidth and maxHeight
                    // -> no rounding, because values for maxWidth can be very large (no hmm)
                    horzFactor = groupWidth / (maxWidth - offsetHorz + 1);
                    vertFactor = groupHeight / (maxHeight - offsetVert + 1);

                    if (isResizeActive) {
                        horzFactor /= zoomFactor;
                        vertFactor /= zoomFactor;
                    }

                    // getting the maximum value of the children for 'left + width' and 'top + height')
                    _.each(allChildren, function (drawing, index) {
                        handleDrawingInsideDrawingGroup($(drawing), AttributeUtils.getExplicitAttributes(drawing), horzFactor, vertFactor, offsetHorz, offsetVert, index);
                    });
                }
            }
            if (isResizeActive && isRotationAngle) {
                drawing.css({ transform: 'rotate(' + rotationAngle + 'deg)' }); // return property to it's value
            }
        }

        // handle theme color inheritance chain (the merged attributes object might be modified)
        DrawingFrame.handleTargetChainInheritance(docModel, drawingFrame, mergedAttributes);

        handleFlipping(docModel, drawingFrame, drawingAttributes);

        // update drawing frame specific to the drawing type
        switch (type) {
            case 'image':
                updateLineFormatting();
                rendered = updateImageFormatting();
                if (rendered) { handleContentEditable(drawingFrame); }  // TODO: Not good here, but required for fast load (Bug #48389)
                break;
            case 'ole':
                rendered = updateImageFormatting();
                break;
            case 'chart':
                rendered = updateChartFormatting();
                break;
            case 'table':
                rendered = updateTableFormatting();
                if (app.isODF()) { updateTextFrameState(); }
                handleContentEditable(drawingFrame); // TODO: Not good here, but required for fast load (36246)
                break;
            case 'connector':
             // falls through
            case 'shape':
                updateShapeFormatting();
                if (app.isODF()) { updateTextFrameState(); }
                // TODO: The following line has to be removed to activate shape geometry in spreadsheet documents
                rendered = app.getName() === 'io.ox/office/text' || app.getName() === 'io.ox/office/presentation' || DrawingFrame.isTextFrameShapeDrawingFrame(drawingFrame);
                if (rendered) { handleContentEditable(drawingFrame); }  // TODO: Not good here, but required for fast load (36246)
                break;
            case 'group':
                // the group needs to take care of the size of all children
                resizeDrawingsInGroup(drawingFrame);
                handleContentEditable(drawingFrame); // TODO: Not good here, but required for fast load
                rendered = true;
                break;
        }

        // an additional resize of drawings in groups is required after all children are added to the group.
        // -> during working with document, operations are only generated for the group itself, so that
        //    'DrawingFrame.isGroupedDrawingFrame()' should always be false.
        // -> but during working with the document, this step is also necessary after undo (47204)
        // -> Performance: Setting the data attribute 'grouphandling' to the parent drawing makes it
        //                 possible that 'resizeDrawingsInGroup' is not called after every child drawing is
        //                 added to the group. In this case it is necessary, that the drawing of type
        //                 'group' is formatted, after all children are inserted.
        if (DrawingFrame.isGroupedDrawingFrame(drawingFrame)) {
            drawingGroupNode = DrawingFrame.getGroupNode(drawingFrame);
            if (drawingGroupNode && drawingGroupNode.length > 0 && !drawingGroupNode.data('grouphandling')) {
                resizeDrawingsInGroup(drawingGroupNode);
            }
        }

        // add replacement in case the drawing type is not supported, or if rendering was not successful
        if (!rendered) {
            updateUnsupportedFormatting();
        }

    };

    /**
     * returns the wrapped canvas for the current node.
     *
     * @param {jQuery} [node]
     *
     * @param {Object} [options]
     *  @param {Boolean} [options.resizeIsActive=false]
     *      If this method is called during resizing of the shape.
     */
    DrawingFrame.getWrappedCanvas = function (currentNode, options) {

        var // if process of resizing of drawing is currently active
            resizeIsActive  = Utils.getBooleanOption(options, 'resizeIsActive', false),
            // the canvasNode inside the drawing frame
            canvasNode = DrawingFrame.getCanvasNode(currentNode),
            // the canvas wrapper
            canvas = null;

        // while resizing of the drawing is active, there are cloned and original canvas,
        // only first has to be fetched
        if (resizeIsActive) {
            canvasNode = canvasNode.first();
        } else if (canvasNode.length > 1) {
            canvasNode = canvasNode.last();
        }

        if (canvasNode.length > 0) {
            canvas = new Canvas({ classes: DrawingFrame.CANVAS_CLASS, node: canvasNode });
        } else {
            canvas = new Canvas({ classes: DrawingFrame.CANVAS_CLASS });
            DrawingFrame.getContentNode(currentNode).prepend(canvas.getNode());
        }
        return canvas;
    };

    /**
     * a initialized canvas is returned for the current node
     *
     * @param {jQuery} [node]
     */
    DrawingFrame.initializeCanvas = function (currentNode, rectangle, borderSize, options) {

        var // drawing-type
            type = DrawingFrame.getDrawingType(currentNode),
            // the canvas wrapper of the border node
            canvas          = DrawingFrame.getWrappedCanvas(currentNode, options),
            // border size
            halfBorderSize  = math_ceil(borderSize * 0.5),
            // size to calculate with (different for images and shapes)
            calcSize        = (type === 'image') ? borderSize : halfBorderSize,
            // canvas start (left, top)
            left            = rectangle.left - halfBorderSize,
            top             = rectangle.top - halfBorderSize,
            // width and height of both (canvas and canvas-node)
            width           = rectangle.width + 2 * calcSize,
            height          = rectangle.height + 2 * calcSize,
            // canvasnode start (left, top)
            nodeLeft        = rectangle.left - calcSize,
            nodeTop         = rectangle.top - calcSize,
            whRatio         = 1;

        // see bug #48101
        if (width > height && width > MAXIMUM_CANVAS_SIZE) {
            whRatio = height / width;
            width = MAXIMUM_CANVAS_SIZE;
            height = Utils.round(width * whRatio, 1);
        } else if (height > MAXIMUM_CANVAS_SIZE) {
            whRatio = width / height;
            height = MAXIMUM_CANVAS_SIZE;
            width = Utils.round(height * whRatio, 1);
        }

        // initialize canvas
        canvas.initialize({ left: left, top: top, width: width, height: height });

        // set position of the canvas element
        canvas.getNode().css({
            left: nodeLeft, top: nodeTop,
            width: width, height: height
        });
        return canvas;
    };

    /*
     * Returns the bounding rectangle in pixel of the drawing. This is
     * necessary because the shape might exceed the initial node size
     *
     * @param {Object} snapRect
     *  The rectangle with the properties left, top, width and height
     *  for setting the size of the canvas.
     *
     * @param {Number} [canvasExpansion]
     *  An optional number, that describes a required expansion of the
     *  canvas in pixel. If not specified, the specified parameter
     *  snapRect is used without modification.
     *
     * @returns {Object}
     *  An rectangle with the properties left, top, width and height
     *  for setting the size of the canvas.
     */
    DrawingFrame.calcBoundRectPixel = function (snapRect, canvasExpansion) {
        var expansion = canvasExpansion || 0; // an optional expansion of the canvas, caused by an arrow at line end
        return { left: snapRect.left - expansion, top: snapRect.top - expansion, width: snapRect.width + 2 * expansion, height: snapRect.height + 2 * expansion };
    };

    /**
     *
     * @param collector
     * @param command
     * @returns {*}
     */
    DrawingFrame.collectPathCoordinates = function (collector, command) {
        var
            pathCoords  = collector.pathCoords,

            cx          = collector.cx || 0,
            cy          = collector.cy || 0,
            fXScale     = collector.fXScale || 1,
            fYScale     = collector.fYScale || 1,

            getValue    = collector.getGeometryValue,

            i           = 0,
            pts         = null,
            length      = null,

            commandType = command.c;

        if (commandType === 'moveTo') {

            // pathCoords.push({
            //     type: 'point',
            //
            //     x1:   getValue(command.x) * fXScale,
            //     y1:   getValue(command.y) * fYScale
            // });
            pathCoords.xValues.push(math_round(getValue(command.x) * fXScale));
            pathCoords.yValues.push(math_round(getValue(command.y) * fYScale));

        } else if (commandType === 'lineTo') {

            // pathCoords.push({
            //     type: 'point',
            //
            //     x1:   getValue(command.x) * fXScale,
            //     y1:   getValue(command.y) * fYScale
            // });
            pathCoords.xValues.push(math_round(getValue(command.x) * fXScale));
            pathCoords.yValues.push(math_round(getValue(command.y) * fYScale));

        } else if (commandType === 'arcTo') {
            var
                rx    = getValue(command.wr),
                ry    = getValue(command.hr),
                stAng = (getValue(command.stAng) / 60000) * MATH_PI / 180,
                swAng = getValue(command.swAng),

                /*
                 * calculates the point on an arc that corresponds to the given angle (ms like)
                 */
                calcArcPoint = function (rx, ry, angle) {
                    var wt1 = rx * math_sin(angle),
                        ht1 = ry * math_cos(angle),
                        dx1 = rx * math_cos(math_atan2(wt1, ht1)) * fXScale,
                        dy1 = ry * math_sin(math_atan2(wt1, ht1)) * fYScale;

                    return { x: dx1, y: dy1 };
                },
                // calculating new currentPoint
                startPoint  = calcArcPoint(rx, ry, stAng),
                endPoint    = calcArcPoint(rx, ry, stAng + swAng / 10800000 * MATH_PI);

            //     calcAngle   = function (x, y) {
            //         var
            //             x1 = rx * fXScale,
            //             y1 = ry * fYScale;
            //
            //         return (x1 > y1) ? math_atan2(y * x1 / y1, x) : math_atan2(y, x * y1 / x1);
            //     },
            //     eStAng  = calcAngle(startPoint.x, startPoint.y),
            //     eEndAng = calcAngle(endPoint.x, endPoint.y);
            //
            // // check if a full circle has to be drawn
            // if (math_abs(swAng) >= 21600000) {
            //     swAng   = -1;
            //     eEndAng = eStAng + 0.0001;
            // }

            // pathCoords.push({
            //     type: 'arc',
            //
            //     x1:     cx,
            //     y1:     cy,
            //     x2:     (cx = cx + endPoint.x - startPoint.x),
            //     y2:     (cy = cy + endPoint.y - startPoint.y),
            //     mx:     rx * fXScale,
            //     my:     ry * fYScale,
            //     ang1:   eStAng,
            //     ang2:   eEndAng,
            //     isFull: (swAng < 0)
            // });
            pathCoords.xValues.push(cx);
            pathCoords.yValues.push(cy);
            pathCoords.xValues.push(cx = math_round(cx + endPoint.x - startPoint.x));
            pathCoords.yValues.push(cy = math_round(cy + endPoint.y - startPoint.y));

        } else if (commandType === 'quadBezierTo') {

            if (command.pts) {
                pts     = command.pts;
                length  = pts.length;

                if (length % 2 === 0) {
                    for (i = 0; i < length; i += 2) {

                        // pathCoords.push({
                        //     type: 'quad',
                        //
                        //     x1: getValue(pts[i].x) * fXScale,
                        //     y1: getValue(pts[i].y) * fYScale,
                        //     x2: (cx = getValue(pts[i + 1].x) * fXScale),
                        //     y2: (cy = getValue(pts[i + 1].y) * fYScale)
                        // });
                        pathCoords.xValues.push(math_round(getValue(pts[i].x) * fXScale));
                        pathCoords.yValues.push(math_round(getValue(pts[i].y) * fYScale));
                        pathCoords.xValues.push(cx = math_round(getValue(pts[i + 1].x) * fXScale));
                        pathCoords.yValues.push(cy = math_round(getValue(pts[i + 1].y) * fYScale));
                    }
                }

            }
        } else if (commandType === 'cubicBezierTo') {

            if (command.pts) {
                pts     = command.pts;
                length  = pts.length;

                if (length % 3 === 0) {
                    for (i = 0; i < length; i += 3) {

                        // pathCoords.push({
                        //     type: 'cubic',
                        //
                        //     x1: getValue(pts[i].x) * fXScale,
                        //     y1: getValue(pts[i].y) * fYScale,
                        //     x2: getValue(pts[i + 1].x) * fXScale,
                        //     y2: getValue(pts[i + 1].y) * fYScale,
                        //     x3: (cx = getValue(pts[i + 2].x) * fXScale),
                        //     y3: (cy = getValue(pts[i + 2].y) * fYScale)
                        // });
                        pathCoords.xValues.push(math_round(getValue(pts[i].x) * fXScale));
                        pathCoords.yValues.push(math_round(getValue(pts[i].y) * fYScale));
                        pathCoords.xValues.push(math_round(getValue(pts[i + 1].x) * fXScale));
                        pathCoords.yValues.push(math_round(getValue(pts[i + 1].y) * fYScale));
                        pathCoords.xValues.push(cx = math_round(getValue(pts[i + 2].x) * fXScale));
                        pathCoords.yValues.push(cy = math_round(getValue(pts[i + 2].y) * fYScale));
                    }
                }
            }
        }
        collector.cx = cx;
        collector.cy = cy;

        return collector;
    };

    /**
     *
     * @param collector
     * @param command
     * @returns {*}
     */
    DrawingFrame.executePathCommands = function (collector, command) {

        var canvasPath  = collector.canvasPath,

            cx          = collector.cx || 0,
            cy          = collector.cy || 0,
            fXScale     = collector.fXScale || 1,
            fYScale     = collector.fYScale || 1,

            getValue    = collector.getGeometryValue,

            i           = 0,
            pts         = null,
            length      = null,

            r           = 0,
            r1          = 0,
            r2          = 0,
            rpx         = 0,
            rpy         = 0,

            angle       = 0,
            rightDirection = false; // the direction of triangle, arrow, ...

        switch (command.c) {
            case 'moveTo':
                cx = getValue(command.x) * fXScale;
                cy = getValue(command.y) * fYScale;
                canvasPath.moveTo(cx, cy);
                break;

            case 'lineTo':
                cx = getValue(command.x) * fXScale;
                cy = getValue(command.y) * fYScale;
                canvasPath.lineTo(cx, cy);
                break;

            case 'circle':
                r = getValue(command.r) * fXScale;
                cx = getValue(command.x) * fXScale;
                cy = getValue(command.y) * fYScale;
                canvasPath.pushCircle(cx, cy, r);
                break;

            case 'oval':
            case 'diamond':
            case 'triangle':
            case 'stealth':
            case 'arrow':
                r1 = getValue(command.r1) * fXScale;
                r2 = getValue(command.r2) * fYScale;

                cx = getValue(command.x) * fXScale;
                cy = getValue(command.y) * fYScale;

                // reference point for calculating the angle (of the line)
                rpx = getValue(command.rpx) * fXScale;
                rpy = getValue(command.rpy) * fYScale;

                // an optional offset, for calculation arrow direction at bezier curves
                if (command.rpxOffset) { rpx += command.rpxOffset; }

                if (rpy !== cy) { angle = math_atan2(math_abs(rpy - cy), math_abs(rpx - cx)); }

                switch (command.c) {
                    case 'oval':
                        canvasPath.pushEllipse(cx, cy, r1, r2, angle);
                        break;
                    case 'diamond':
                        canvasPath.pushDiamond(cx, cy, r1, r2, angle);
                        break;
                    case 'triangle':
                        if (isRightOrDownDirection(cx, cy, rpx, rpy)) {
                            rightDirection = true; // setting the direction of the triangle
                        }
                        canvasPath.pushTriangle(cx, cy, r1, r2, angle, rightDirection);
                        break;
                    case 'stealth':
                        if (isRightOrDownDirection(cx, cy, rpx, rpy)) {
                            rightDirection = true;  // setting the direction of the stealth arrow head
                        }
                        canvasPath.pushStealth(cx, cy, r1, r2, angle, rightDirection);
                        break;
                    case 'arrow':
                        if (isRightOrDownDirection(cx, cy, rpx, rpy)) {
                            rightDirection = true; // setting the direction of the arrow head
                        }
                        canvasPath.pushArrow(cx, cy, r1, r2, angle, rightDirection);
                        break;
                }
                break;

            case 'arcTo':
                var rx    = getValue(command.wr),
                    ry    = getValue(command.hr),
                    stAng = (getValue(command.stAng) / 60000) * MATH_PI / 180,
                    swAng = getValue(command.swAng),

                    /*
                     * calculates the point on an arc that corresponds to the given angle (ms like)
                     */
                    calcArcPoint = function (rx, ry, angle) {
                        var wt1 = rx * math_sin(angle),
                            ht1 = ry * math_cos(angle),
                            dx1 = rx * math_cos(math_atan2(wt1, ht1)) * fXScale,
                            dy1 = ry * math_sin(math_atan2(wt1, ht1)) * fYScale;

                        return { x: dx1, y: dy1 };
                    },
                    // calculating new currentPoint
                    startPoint  = calcArcPoint(rx, ry, stAng),
                    endPoint    = calcArcPoint(rx, ry, stAng + swAng / 10800000 * MATH_PI),

                    calcAngle   = function (x, y) {
                        var
                            x1 = rx * fXScale,
                            y1 = ry * fYScale;

                        return (x1 > y1) ? math_atan2(y * x1 / y1, x) : math_atan2(y, x * y1 / x1);
                    },
                    eStAng  = calcAngle(startPoint.x, startPoint.y),
                    eEndAng = calcAngle(endPoint.x, endPoint.y);

                // check if a full circle has to be drawn
                if (math_abs(swAng) >= 21600000) {
                    swAng   = -1;
                    eEndAng = eStAng + 0.0001;
                }
                canvasPath.ellipseTo(cx, cy, rx * fXScale, ry * fYScale, eStAng, eEndAng, swAng < 0);

                cx = cx + endPoint.x - startPoint.x;
                cy = cy + endPoint.y - startPoint.y;
                break;

            case 'quadBezierTo':
                if (command.pts) {
                    pts     = command.pts;
                    length  = pts.length;

                    if (length % 2 === 0) {
                        for (i = 0; i < length; i += 2) {

                            canvasPath.quadraticCurveTo(
                                getValue(pts[i].x) * fXScale,
                                getValue(pts[i].y) * fYScale,
                                cx = getValue(pts[i + 1].x) * fXScale,
                                cy = getValue(pts[i + 1].y) * fYScale
                            );
                        }
                    }
                }
                break;

            case 'cubicBezierTo':
                if (command.pts) {
                    pts     = command.pts;
                    length  = pts.length;

                    if (length % 3 === 0) {
                        for (i = 0; i < length; i += 3) {

                            canvasPath.bezierCurveTo(
                                getValue(pts[i].x) * fXScale,
                                getValue(pts[i].y) * fYScale,
                                getValue(pts[i + 1].x) * fXScale,
                                getValue(pts[i + 1].y) * fYScale,
                                cx = getValue(pts[i + 2].x) * fXScale,
                                cy = getValue(pts[i + 2].y) * fYScale
                            );
                        }
                    }
                }
                break;

            case 'close':
                canvasPath.close();
                break;
        }

        collector.cx = cx;
        collector.cy = cy;

        return collector;
    };

    DrawingFrame.drawShape = function (docModel, context, snapRect, geometryAttrs, fillAttrs, lineAttrs, guideCache, options) {

        if (!geometryAttrs.pathList) {
            return;
        }
        var
            avList = geometryAttrs.avList,
            gdList = geometryAttrs.gdList,

            executePathCommands     = DrawingFrame.executePathCommands,
            collectPathCoordinates  = DrawingFrame.collectPathCoordinates,
            isResizeActive          = Utils.getBooleanOption(options, 'isResizeActive', false),
            currentNode             = Utils.getObjectOption(options, 'currentNode', null),
            rotationAngle           = null,
            pathListLength          = geometryAttrs.pathList ? geometryAttrs.pathList.length : 0,
            pathCounter             = 1,
            isFirstPath             = false,
            isLastPath              = false;

        function getGeometryValue(value) {
            return DrawingFrame.getGeometryValue(value, snapRect, avList, gdList, guideCache, 1000);
        }

        function shadeColor(color, percent) {
            var
                f = parseInt(color.slice(1), 16),
                t = percent < 0 ? 0 : 255,
                p = math_abs(percent),
                R = f >> 16,
                G = (f >> 8) & 0xFF,
                B = f & 0xFF;

            return '#' + (0x1000000 + (math_round((t - R) * p) + R) * 0x10000 + (math_round((t - G) * p) + G) * 0x100 + (math_round((t - B) * p) + B)).toString(16).slice(1);
        }

        geometryAttrs.pathList.forEach(function (onePath) {

            // initialize context with proper fill attributes for the next path
            var
                path        = _.copy(onePath, true), // creating a deep copy of the path, so that it can be expanded

                widthPx     = snapRect.width,
                heightPx    = snapRect.height,

                logWidth    = ('width' in path) ? path.width : widthPx,
                logHeight   = ('height' in path) ? path.height : heightPx,

                fXScale     = (widthPx / logWidth),
                fYScale     = (heightPx / logHeight),

                hasFill     = fillAttrs && (fillAttrs.type !== 'none') && (path.fillMode !== 'none'),
                hasLine     = lineAttrs && (lineAttrs.type !== 'none') && (path.isStroke !== false),
                notRotatedWithShape = null;

            /**
             * Local function to draw final path of canvas.
             *
             * @param {String} [forcedMode]
             *  This optional parameter forces 'fill' or 'stroke' mode.
             *  Otherwise, if hasFill and hasLine, mode will be 'all'.
             *
             * @param {Object} [forcedPath]
             *  An optional path, that will be drawn, instead of the
             *  path specified in the global path object.
             */
            function drawFinalPath(forcedMode, forcedPath) {

                var // the mode for 'fill', 'stroke' or 'all'
                    mode = forcedMode || getRenderMode(hasFill, hasLine),
                    // the path to be drawn
                    localPath = forcedPath ? forcedPath : path;

                if (mode && localPath.commands) {

                    if (logWidth > 0 && logHeight > 0) {

                        var // the new converted path with resolved path commands
                            newPath = localPath.commands.reduce(executePathCommands, {

                                canvasPath: context.createPath(),

                                cx: 0,
                                cy: 0,
                                fXScale: fXScale,
                                fYScale: fYScale,

                                getGeometryValue: getGeometryValue

                            }).canvasPath;

                        context.drawPath(newPath, mode);
                    }
                }
            }

            isFirstPath = (pathCounter === 1);
            isLastPath = (pathCounter === pathListLength);
            pathCounter++;

            if (hasFill) {
                var
                    fillStyle,
                    target          = fillAttrs[DrawingFrame.TARGET_CHAIN_PROPERTY];

                if (fillAttrs.type === 'gradient') {
                    var angle = !fillAttrs.gradient.isRotateWithShape ? DrawingFrame.getDrawingRotationAngle(docModel, currentNode) || 0 : 0;
                    if (angle) {
                        // rotate the triangle (rotate the gradient not with shape) to get the new with and height
                        var rotatedAttrs = DrawingUtils.getRotatedDrawingPoints(AttributeUtils.getExplicitAttributes(currentNode).drawing, angle);
                        logWidth = Utils.convertHmmToLength(rotatedAttrs.width, 'px', 1);
                        logHeight = Utils.convertHmmToLength(rotatedAttrs.height, 'px', 1);
                    }
                    var
                        gradient    = Gradient.create(Gradient.getDescriptorFromAttributes(fillAttrs)),

                        halfWidth   = math_round(logWidth / 2),
                        halfHeight  = math_round(logHeight / 2),

                        pathCoords  = path.commands.reduce(collectPathCoordinates, {
                            pathCoords: {
                                xValues: [halfWidth, logWidth, halfWidth, 0],
                                yValues: [0, halfHeight, logHeight, halfHeight],
                                mx: halfWidth,
                                my: halfHeight
                            },
                            cx: 0,
                            cy: 0,
                            fXScale: fXScale,
                            fYScale: fYScale,

                            getGeometryValue: getGeometryValue

                        }).pathCoords;

                    if (angle) {
                        // rotate the triangle (rotate the gradient not with shape) cords
                        if (pathCoords.xValues.length === pathCoords.yValues.length) {
                            for (var i = 0; i < pathCoords.xValues.length; i++) {
                                var x = pathCoords.xValues[i];
                                var y = pathCoords.yValues[i];
                                var rotPoint = DrawingUtils.rotatePointWithAngle(pathCoords.mx, pathCoords.my, x, y, angle);
                                pathCoords.xValues[i] = rotPoint.left;
                                pathCoords.yValues[i] = rotPoint.top;
                            }
                        }

                    }
                    fillStyle = docModel.getGradientFill(gradient, pathCoords, context, target);
                } else if (fillAttrs.type === 'pattern') {
                    fillStyle = Pattern.createCanvasPattern(fillAttrs.pattern, fillAttrs.backgroundColor, fillAttrs.color, docModel.getTheme(target));
                } else if (fillAttrs.type === 'bitmap' && fillAttrs.bitmap) {
                    notRotatedWithShape = fillAttrs.bitmap.rotateWithShape === false;
                    if (notRotatedWithShape && currentNode) {
                        rotationAngle = DrawingFrame.getDrawingRotationAngle(docModel, currentNode) || 0;
                        rotationAngle = ((360 - rotationAngle) % 360) * (Math.PI / 180); // normalize and convert to radians
                    }
                    context.clearRect(0, 0, widthPx, heightPx);
                    if (isResizeActive && textureContextCache) { // Performance: during resize, try to fetch texture pattern from cache
                        setBitmapTextureFillStyle(docModel, context, drawFinalPath, textureContextCache, widthPx, heightPx, hasLine, lineAttrs, fillAttrs.bitmap.isTiling, notRotatedWithShape, rotationAngle);
                    } else {
                        Texture.getTextureFill(docModel, context, fillAttrs, widthPx, heightPx, rotationAngle).then(function (textureFillStyle) {
                            if (isResizeActive) {
                                // store texture for performance
                                textureContextCache = textureFillStyle;
                            } else {
                                textureContextCache = null;
                            }
                            setBitmapTextureFillStyle(docModel, context, drawFinalPath, textureFillStyle, widthPx, heightPx, hasLine, lineAttrs, fillAttrs.bitmap.isTiling, notRotatedWithShape, rotationAngle);
                        });
                    }
                    return;
                } else {
                    fillStyle = docModel.getCssColor(fillAttrs.color, 'fill', target);

                    switch (path.fillMode) {
                        case 'lighten':
                            fillStyle = shadeColor(fillStyle, 0.40);
                            break;
                        case 'lightenLess':
                            fillStyle = shadeColor(fillStyle, 0.20);
                            break;
                        case 'darken':
                            fillStyle = shadeColor(fillStyle, -0.40);
                            break;
                        case 'darkenLess':
                            fillStyle = shadeColor(fillStyle, -0.20);
                            break;
                    }
                }
                context.setFillStyle(fillStyle);
            }

            // initialize context with proper line attributes for the next path
            var additionalPathes = null;
            if (hasLine) {
                additionalPathes = setLineAttributes(docModel, context, lineAttrs, null, path, { isFirstPath: isFirstPath, isLastPath: isLastPath, snapRect: snapRect, avList: avList, gdList: gdList, guideCache: guideCache });
            }

            drawFinalPath();

            if (additionalPathes) {
                _.each(additionalPathes, function (onePath) {
                    drawFinalPath(onePath.drawMode || null, onePath);
                });
            }
        });
    };

    DrawingFrame.getGeometryValue = function (value, snapRect, avList, gdList, guideCache, maxGdNumber) {
        if (typeof value === 'number') {
            return value;
        }
        if (value in guideCache) {
            return guideCache[value];
        }
        switch (value) {
            // 3 x 360deg / 4 = 270deg
            case '3cd4':
                return 270 * 60000;
            // 3 x 360deg / 8 = 135deg
            case '3cd8':
                return 135 * 60000;
            // 5 x 360deg / 8 = 225deg
            case '5cd8':
                return 225 * 60000;
            // 7 x 360deg / 8 = 315deg
            case '7cd8':
                return 315 * 60000;
            // bottom
            case 'b':
                return snapRect.top + (snapRect.height - 1);
            // 360deg / 2 = 180deg
            case 'cd2':
                return 180 * 60000;
            // 360deg / 4 = 90deg
            case 'cd4':
                return 90 * 60000;
            // 360deg / 8 = 45deg
            case 'cd8':
                return 45 * 60000;
            // horizontal center
            case 'hc':
                return snapRect.left + snapRect.width * 0.5;
            // height
            case 'h':
                return snapRect.height;
            // height / 2
            case 'hd2':
                return snapRect.height / 2;
            // height / 3
            case 'hd3':
                return snapRect.height / 3;
            // height / 4
            case 'hd4':
                return snapRect.height / 4;
            // height / 5
            case 'hd5':
                return snapRect.height / 5;
            // height / 6
            case 'hd6':
                return snapRect.height / 6;
            // height / 8
            case 'hd8':
                return snapRect.height / 8;
            // left
            case 'l':
                return snapRect.left;
            // long side
            case 'ls':
                return math_max(snapRect.width, snapRect.height);
            // right
            case 'r':
                return snapRect.left + (snapRect.width - 1);
            // short side
            case 'ss':
                return math_min(snapRect.width, snapRect.height);
            // short side / 2
            case 'ssd2':
                return math_min(snapRect.width, snapRect.height) / 2;
            // short side / 4
            case 'ssd4':
                return math_min(snapRect.width, snapRect.height) / 4;
            // short side / 6
            case 'ssd6':
                return math_min(snapRect.width, snapRect.height) / 6;
            // short side / 8
            case 'ssd8':
                return math_min(snapRect.width, snapRect.height) / 8;
            // short side / 16
            case 'ssd16':
                return math_min(snapRect.width, snapRect.height) / 16;
            // short side / 32
            case 'ssd32':
                return math_min(snapRect.width, snapRect.height) / 32;
            // top
            case 't':
                return snapRect.top;
            // vertical center
            case 'vc':
                return snapRect.top + snapRect.height * 0.5;
            // width
            case 'w':
                return snapRect.width;
            // width / 2
            case 'wd2':
                return snapRect.width / 2;
            // width / 4
            case 'wd4':
                return snapRect.width / 4;
            // width / 5
            case 'wd5':
                return snapRect.width / 5;
            // width / 6
            case 'wd6':
                return snapRect.width / 6;
            // width / 8
            case 'wd8':
                return snapRect.width / 8;
            // width / 10
            case 'wd10':
                return snapRect.width / 10;
            // width / 32
            case 'wd32':
                return snapRect.width / 32;
        }
        // the value is not a constant value, next check if this is a guide name
        if (avList && value in avList) {
            return avList[value];
        }
        if (gdList) {
            for (var i = 0; i < gdList.length; i++) {
                if (gdList[i].name === value) {
                    if (i > maxGdNumber) {
                        return 0.0;
                    }
                    var result = DrawingFrame.calcGeometryValue(gdList[i], snapRect, avList, gdList, guideCache, i - 1);
                    guideCache[value] = result;
                    return result;
                }
            }
        }
        // we never should come here ...
        if (!$.isNumeric(value)) {
            return 0.0;
        }
        // this should be normal float/int value
        return parseFloat(value);
    };

    DrawingFrame.calcGeometryValue = function (gd, snapRect, avList, gdList, guideCache, maxGdNumber) {

        function p(p) {
            return parseFloat(DrawingFrame.getGeometryValue(gd[p], snapRect, avList, gdList, guideCache, maxGdNumber));
        }

        var p0, p1, p2;

        switch (gd.op) {
            // Multiply Divide Formula
            // Arguments: 3 (fmla="*/ x y z")
            // Usage: "*/ x y z" = ((x * y) / z) = value of this guide
            case '*/':
                return (p('p0') * p('p1')) / p('p2');

            // Add Subtract Formula
            // Arguments: 3 (fmla="+- x y z")
            // Usage: "+- x y z" = ((x + y) - z) = value of this guide
            case '+-':
                return (p('p0') + p('p1')) - p('p2');

            //  Add Divide Formula
            // Arguments: 3 (fmla="+/ x y z")
            // Usage: "+/ x y z" = ((x + y) / z) = value of this guide
            case '+/':
                return (p('p0') + p('p1')) / p('p2');

            // If Else Formula
            // Arguments: 3 (fmla="?: x y z")
            // Usage: "?: x y z" = if (x > 0), then y = value of this guide,
            // else z = value of this guide
            case '?:':
                return (p('p0') > 0) ? p('p1') : p('p2');

            // Absolute Value Formula
            // Arguments: 1 (fmla="abs x")
            // Usage: "abs x" = if (x < 0), then (-1) * x = value of this guide
            // else x = value of this guide
            case 'abs':
                return math_abs(p('p0'));

            // ArcTan Formula
            // Arguments: 2 (fmla="at2 x y")
            // Usage: "at2 x y" = arctan(y / x) = value of this guide
            case 'at2':
                return (10800000 * math_atan2(p('p1'), p('p0'))) / MATH_PI;

            // Cosine ArcTan Formula
            // Arguments: 3 (fmla="cat2 x y z")
            // Usage: "cat2 x y z" = (x*(cos(arctan(z / y))) = value of this guide
            case 'cat2':
                return p('p0') * (math_cos(math_atan2(p('p2'), p('p1'))));

            // Cosine Formula
            // Arguments: 2 (fmla="cos x y")
            // Usage: "cos x y" = (x * cos( y )) = value of this guide
            case 'cos':
                return p('p0') * math_cos(MATH_PI * p('p1') / 10800000);

            // Maximum Value Formula
            // Arguments: 2 (fmla="max x y")
            // Usage: "max x y" = if (x > y), then x = value of this guide
            // else y = value of this guide
            case 'max':
                return math_max(p('p0'), p('p1'));

            // Minimum Value Formula
            // Arguments: 2 (fmla="min x y")
            // Usage: "min x y" = if (x < y), then x = value of this guide
            // else y = value of this guide
            case 'min':
                return math_min(p('p0'), p('p1'));

            // Modulo Formula
            // Arguments: 3 (fmla="mod x y z")
            // Usage: "mod x y z" = sqrt(x^2 + b^2 + c^2) = value of this guide
            case 'mod':
                p0 = p('p0');
                p1 = p('p1');
                p2 = p('p2');
                return math_sqrt(p0 * p0 + p1 * p1 + p2 * p2);

            // Pin To Formula
            // Arguments: 3 (fmla="pin x y z")
            // Usage: "pin x y z" = if (y < x), then x = value of this guide
            // else if (y > z), then z = value of this guide
            // else y = value of this guide
            case 'pin':
                p0 = p('p0');
                p1 = p('p1');
                p2 = p('p2');
                return (p1 < p0) ? p0 : (p1 > p2) ? p2 : p1;

            // Sine ArcTan Formula
            // Arguments: 3 (fmla="sat2 x y z")
            // Usage: "sat2 x y z" = (x*sin(arctan(z / y))) = value of this guide
            case 'sat2':
                return p('p0') * (math_sin(math_atan2(p('p2'), p('p1'))));

            // Sine Formula
            // Arguments: 2 (fmla="sin x y")
            // Usage: "sin x y" = (x * sin( y )) = value of this guide
            case 'sin':
                return p('p0') * math_sin(MATH_PI * p('p1') / 10800000);

            // Square Root Formula
            // Arguments: 1 (fmla="sqrt x")
            // Usage: "sqrt x" = sqrt(x) = value of this guide
            case 'sqrt':
                return math_sqrt(p('p0'));

            // Tangent Formula
            // Arguments: 2 (fmla="tan x y")
            // Usage: "tan x y" = (x * tan( y )) = value of this guide
            case 'tan':
                return p('p0') * math_tan(MATH_PI * p('p1') / 10800000);

            // Literal Value Formula
            // Arguments: 1 (fmla="val x")
            // Usage: "val x" = x = value of this guide
            case 'val':
                return p('p0');
        }
        return 0.0;
    };

    // draws the border to the canvas
    DrawingFrame.drawBorder = function (app, currentNode, lineAttrs) {

        var // the document model
            docModel = app.getModel(),
            // border width
            lineWidth = math_max(Utils.convertHmmToLength(lineAttrs.width, 'px', 1), 1),
            halfBorderSize = math_ceil(lineWidth * 0.5),
            // dimensions of the canvas element, special behavior for text frames. Their border covers their boundary
            cWidth = currentNode.first().width(),
            cHeight = currentNode.first().height(),
            // the canvas wrapper of the border node
            canvas = DrawingFrame.initializeCanvas(currentNode, { left: 0, top: 0, width: cWidth, height: cHeight }, lineWidth);

        canvas.render(function (context, width, height) {
            var color = docModel.getCssColor(lineAttrs.color, 'line', lineAttrs[DrawingFrame.TARGET_CHAIN_PROPERTY]),
                pattern = Border.getBorderPattern(lineAttrs.style, lineWidth);

            context.setLineStyle({ style: color, width: lineWidth, pattern: pattern });

            var drawX = 0,
                drawY = 0,
                drawW = (width - 2 * halfBorderSize),
                drawH = (height - 2 * halfBorderSize);

            context.drawRect(drawX, drawY, drawW, drawH, 'stroke');
        });

        if (app && app.attributes && app.attributes.name === 'io.ox/office/spreadsheet') {
            //TODO: should be checked from outside!
            DrawingFrame.setCanvasDynHeight(currentNode);
        }
    };

    // clears the canvas
    DrawingFrame.clearCanvas = function (currentNode) {

        var // the border canvas node inside the drawing frame
            canvasNode = DrawingFrame.getCanvasNode(currentNode),
            // the canvas wrapper of the border node
            canvas = null;

        if (canvasNode.length > 0) {
            canvas = new Canvas({ classes: DrawingFrame.CANVAS_CLASS, node: canvasNode });
            canvas.initialize({ width: canvasNode.width(), height: canvasNode.height() });
        }
    };

    /**
     * Returns rotation angle of given drawing node.
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Number|null}
     *  Angle of rotation, or null if no rotation is set.
     */
    DrawingFrame.getDrawingRotationAngle = function (docModel, drawingNode) {
        var attrs = null;
        var rotation = null;
        if (DrawingFrame.isRotatedDrawingNode(drawingNode)) {
            // first trying to fetch direct attributes of the element
            attrs = AttributeUtils.getExplicitAttributes(drawingNode);
            rotation = (attrs && attrs.drawing && attrs.drawing.rotation) || null;
            if (!rotation) {
                // if not fetched, get from inherited attributes
                attrs = docModel.getDrawingStyles().getElementAttributes(drawingNode);
                rotation = (attrs && attrs.drawing && attrs.drawing.rotation) || null;
            }
        }
        return rotation;
    };

    DrawingFrame.setCanvasDynHeight = function (drawingFrame) {
        var canvasNode = DrawingFrame.getCanvasNode(drawingFrame);
        if (!canvasNode.length) { return; }

        var position = canvasNode.position();
        // css calc is used to keep canvas in its frame while zooming
        canvasNode.css({ width: 'calc(100% - ' + (2 * position.left) + 'px)',  height: 'calc(100% - ' + (2 * position.top) + 'px)' });
    };

    // selection --------------------------------------------------------------

    /**
     * Returns whether the specified drawing frame is currently selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the specified drawing frame is selected.
     */
    DrawingFrame.isSelected = function (drawingFrame) {
        return Forms.isSelectedNode(drawingFrame);
    };

    /**
     * Creates or updates additional nodes displayed while a drawing frame is
     * selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *  @param {Number} [options.scaleHandles=1]
     *      the usable handles are scaled or themselves
     *      (without changing the size of the whole selection)
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.drawSelection = function (drawingFrame, options) {

        var // HTML mark-up for the selection
            markup = null;

        // create new selection nodes if missing
        if (!DrawingFrame.isSelected(drawingFrame)) {

            // create a new selection node
            markup = '<div class="' + SELECTION_CLASS + '">';

            // create the border nodes
            markup += '<div class="borders">';
            _.each('tblr', function (pos) {
                markup += '<div data-pos="' + pos + '"></div>';
            });
            markup += '</div>';

            // create resizer handles
            markup += '<div class="resizers">';
            _.each(['t', 'tr', 'r', 'br', 'b', 'bl', 'l', 'tl'], function (pos) {
                markup += '<div data-pos="' + pos + '"></div>';
            });
            markup += '</div>';

            if (options.rotatable) {
                // create rotate handle bar
                markup += '<div class="rotate-handle"></div>';
            }

            // create the tracker node
            markup += '<div class="tracker"></div>';

            // close the selection node
            markup += '</div>';

            $(drawingFrame).first().append(markup);
        }

        // update state classes at the drawing frame
        $(drawingFrame).first()
            .addClass(Forms.SELECTED_CLASS)
            .toggleClass('movable', Utils.getBooleanOption(options, 'movable', false))
            .toggleClass('resizable', Utils.getBooleanOption(options, 'resizable', false));

        DrawingFrame.updateScaleHandles(drawingFrame, options);

        return getSelectionNode(drawingFrame);
    };

    /**
     * updates the additional nodes displayed while a drawing frame is selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Number} [options.scaleHandles=1]
     *      the usable handles are scaled or themselves
     *      (without changing the size of the whole selection)
     *      range is 1 - 3
     */
    DrawingFrame.updateScaleHandles = function (drawingFrame, options) {

        var // scaleHandles for setting fontsize to scale inner elements
            scaleHandles = Utils.getNumberOption(options, 'scaleHandles', 1, 1, 3);

        getSelectionNode(drawingFrame).css('font-size', scaleHandles + 'rem');
    };

    /**
     * Updates mouse pointers for the resizers of the drawing frame, according to the rotation angle of the drawing.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Number} angle
     *  Rotation angle of the drawing, in degrees.
     */
    DrawingFrame.updateResizersMousePointers = function (drawingFrame, angle) {
        var resizeHandles, shift;
        var pointers = ['n-resize', 'ne-resize', 'e-resize', 'se-resize', 's-resize', 'sw-resize', 'w-resize', 'nw-resize'];

        if (_.isNumber(angle)) {
            angle %= 360;
            angle += 22.5; // start offset
            shift = ~~(angle / 45); // 8 handles makes 8 areas with 45 degrees each
            resizeHandles = drawingFrame.find('.resizers').children();
            _.each(resizeHandles, function (handle, index) {
                var pointerInd = (index + shift) % pointers.length;
                $(handle).css('cursor', pointers[pointerInd]);
            });
        }
    };

    /**
     * Removes the selection node from the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.clearSelection = function (drawingFrame) {
        $(drawingFrame).first().removeClass(Forms.SELECTED_CLASS + ' movable resizable');
        getSelectionNode(drawingFrame).remove();
        DrawingFrame.removeDrawingSelection(drawingFrame);
    };

    /**
     * Helper function to set the size, position and rotation of a selection of a drawing node.
     * This handles selection objects that are located in the selection overlay node.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node, whose selection size, position and rotation shall be updated.
     */
    DrawingFrame.updateSelectionDrawingSize = function (drawingNode) {
        var drawing = $(drawingNode);
        if (drawing.data('selection')) {
            drawing.data('selection').css({ transform: drawing.css('transform'), left: drawing.css('left'), top: drawing.css('top'), width: drawing.width(), height: drawing.height() });
        }
    };

    /**
     * Helper function to remove the selection of a drawing node. This handles selection
     * objects that are located in the selection overlay node.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node, whose selection shall be removed.
     */
    DrawingFrame.removeDrawingSelection = function (drawingNode) {
        var drawing = $(drawingNode);
        if (drawing.data('selection')) {
            drawing.data('selection').remove();
            drawing.removeData('selection');
        }
    };

    /**
     * Sets or resets the CSS class representing the active tracking mode for
     * the specified drawing frames.
     *
     * @param {HTMLElement|jQuery} drawingFrames
     *  The root node of a single drawing frame, or multiple drawing frames in
     *  a jQuery collection. This method changes the tracking mode of all
     *  drawing frames at once.
     *
     * @param {Boolean} active
     *  Whether tracking mode is active.
     */
    DrawingFrame.toggleTracking = function (drawingFrames, active) {
        $(drawingFrames).find('>.' + SELECTION_CLASS).toggleClass(TRACKING_CLASS, active === true);
    };

    /**
     * Returns the tracking direction stored in a resizer handle node of a
     * drawing frame selection box.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String|Null}
     *  If the passed node is a resizer handle of a drawing frame selection
     *  box, returns its direction identifier (one of 'l', 'r', 't', 'b', 'tl',
     *  'tr', 'bl', or 'br'). Otherwise, returns null.
     */
    DrawingFrame.getResizerHandleType = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR + '>.' + SELECTION_CLASS + '>.resizers>[data-pos]') ? $(node).attr('data-pos') : null;
    };

    // exports ================================================================

    var promise = IO.loadJSON('io.ox/office/drawinglayer/view/presetGeometries');
    promise.done(function (preset) { PRESET_GEOMETRIES = preset; });

    return promise.then(_.constant(DrawingFrame));

});
