/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/model/drawingmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/model/attributedmodel'
], function (Utils, TimerMixin, AttributedModel) {

    'use strict';

    // class DrawingModel =====================================================

    /**
     * The model of a drawing object.
     *
     * @constructor
     *
     * @extends AttributedModel
     * @extends TimerMixin
     *
     * @param {EditModel} docModel
     *  The document model containing this drawing object.
     *
     * @param {String} drawingType
     *  The type of this drawing object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the drawing
     *  object.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.families]
     *      Additional explicit attribute families supported by this drawing
     *      model object (space-separated). The main attribute family 'drawing'
     *      supported by all drawing objects will be registered implicitly, and
     *      does not have to be added here.
     */
    function DrawingModel(docModel, drawingType, initAttributes, initOptions) {

        // base constructors --------------------------------------------------

        AttributedModel.call(this, docModel, initAttributes, {
            styleFamily: 'drawing',
            families: Utils.getStringOption(initOptions, 'families')
        });

        TimerMixin.call(this);

        // protected methods --------------------------------------------------

        /**
         * Handler for the document operation 'setDrawingAttributes'.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setDrawingAttributes' operation.
         */
        this.applySetAttributesOperation = function (context) {
            this.setAttributes(context.getObj('attrs'));
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the type of this drawing model, as specified by the document
         * operation that has created the drawing model.
         *
         * @returns {String}
         *  The type of this drawing model.
         */
        this.getType = function () {
            return drawingType;
        };

        /**
         * Returns whether this drawing model is in visible state (the 'hidden'
         * attribute is not set to true).
         *
         * @returns {Boolean}
         *  Whether this drawing model is in visible state.
         */
        this.isVisible = function () {
            return !this.getMergedAttributeSet(true).drawing.hidden;
        };

        /**
         * Returns the collection of child drawing objects contained in this
         * drawing object model. Intended to be overwritten by sub classes that
         * represent drawing models with an internal child drawing collection
         * (e.g. group objects).
         *
         * @returns {DrawingCollection|Null}
         *  The child drawing collection contained in this instance; or null,
         *  if the drawing object does not contain a child collection.
         */
        this.getCollection = function () {
            return null;
        };

        /**
         * Returns whether this drawing model is set to the specified type, or
         * contains an embedded drawing collection with at least one drawing
         * model of that type (e.g. in group objects).
         *
         * @param {String} type
         *  The drawing model type to be checked.
         *
         * @returns {Boolean}
         *  Whether this drawing model has the specified type, or contains an
         *  embedded drawing model of that type.
         */
        this.isOrContainsType = function (type) {

            // first check the own type
            if (drawingType === type) { return true; }

            // check whether an embedded drawing model exists
            var collection = this.getCollection();
            return !!collection && (collection.getModelCount({ deep: true, type: type }) > 0);
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = initAttributes = initOptions = null;
        });

    } // class DrawingModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: DrawingModel });

});
