/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/baseframework/utils/infomessages', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/utils/infostate',
    'io.ox/office/baseframework/utils/errorcontextdata',
    'gettext!io.ox/office/baseframework/main'
], function (Utils, InfoState, ErrorContextData, gt) {

    'use strict';

    // static class InfoMessages ==============================================

    var // the exported ErrorContext class
        InfoMessages = {},

        // client-side mapping error code to error messages
        // ATTENTION: only define non-context data dependent messages here
        // There are the following possible entries for actions available:
        // 'acquireedit' - Acquire edit rights
        INFO_MESSAGES = {
            INFO_SYNCHRONIZATION_SUCCESSFUL:      { headline: gt('Synchronization'), message: gt('Synchronization between client and server was successful. You can edit the document again.') },
            INFO_EDITRIGHTS_ARE_TRANSFERED:       { message: gt('Edit rights are being transferred to you. Please wait a moment.') },
            INFO_EDITRIGHTS_SAVE_IN_PROGRESS:     { type: 'warning', message: gt('Edit rights cannot be transferred. The server is busy to save the document. Please try again later.'), action: 'acquireedit' },
            INFO_EDITRIGHTS_PLEASE_WAIT:          { message: gt('Edit rights are still being transferred to you. The system is busy and will need more time. Please be patient.'), duration: 30000 },
            INFO_ALREADY_TRANSFERING_EDIT_RIGHTS: { type: 'warning', message: gt('Someone else already wants to receive the edit rights for this document. Please try again later.'), action: 'acquireedit' },
            INFO_CONNECTION_WAS_LOST_TRYING_SYNC: { message: gt('Connection to server was lost. Please wait, trying to synchronize to enable edit mode.') },
            INFO_CONNECTION_WAS_LOST:             { message: gt('Connection to server was lost. Enable client to acquire edit rights.') },
            INFO_CONNECTION_LOST_RELOAD_DOC:      { message: gt('Connection to server was lost. Please reopen the document to acquire edit rights.') },
            INFO_NO_PERMISSION_TO_CHANGE_DOC:     { message: gt('You do not have permissions to change this document.') },
            INFO_EDITRIGHTS_RECEIVED:             { type: 'success', message: gt('You have edit rights.') },
            INFO_LOADING_IN_PROGRESS:             { message: gt('Please wait until document is loaded completely.'), headline: gt('Read-Only Mode') },
            INFO_DOC_SAVE_FAILED:                 { message: gt('Saving your document failed, we can try to restore your document using backup data.'), headline: gt('Error') },
            INFO_SYNCHRONIZATION_FAILED:          { type: 'warning', headline: gt('Synchronization'), message: gt('The synchronization process between client and server was not successful.') }
        };

    // static methods ---------------------------------------------------------

    /**
     * Provides the information message data to be shown to the user.
     *
     * @param {String} infoState
     *  The information state as constant string defined by InfoState.
     *
     * @param {Object} options
     *  Optional data for the information
     */

    InfoMessages.getMessageData = function (infoState, options) {
        var // information message
            infoMessage = null,
            // context data from options
            contextData, contextData2;

        if (_.isString(infoState)) {
            infoMessage = (infoState in INFO_MESSAGES) ? INFO_MESSAGES[infoState] : infoMessage;
        }

        if (!_.isObject(infoMessage) && _.isObject(options)) {
            // special handling of information messages which need external data - use options to retrieve
            // this data to call gt()
            infoMessage = {};
            switch (infoState) {
                case 'INFO_PREPARE_LOSING_EDIT_RIGHTS':
                    contextData = Utils.getStringOption(options, 'wantsToEditUser', null);
                    contextData2 = Utils.getStringOption(options, 'oldWantsToEditUser', null);
                    // in case of a long edit-rights switch, we have to use 'oldWantsToEditUser'
                    contextData = (_.isEmpty(contextData)) ? contextData2 : contextData;
                    if (!_.isEmpty(contextData)) {
                        infoMessage.message =
                            //#. %1$s is the name of the user who will get the edit rights
                            //#, c-format
                            gt('Transferring edit rights to %1$s. Your latest changes will be saved now.', gt.noI18n(contextData));
                    } else {
                        infoMessage.message = gt('Transferring edit rights to another user. Your latest changes will be saved now.');
                    }
                    infoMessage.headline = gt('Read-Only Mode');
                    break;
                case 'INFO_USER_IS_CURRENTLY_EDIT_DOC':
                    contextData = Utils.getStringOption(options, 'oldEditUser', null);
                    // message if we lost the edit rights
                    if (!_.isEmpty(contextData)) {
                        infoMessage.message =
                            //#. %1$s is the name of the user who is currently editing the document
                            //#, c-format
                            gt('%1$s is currently editing this document.', _.noI18n(contextData));
                    } else {
                        infoMessage.message = gt('Another user is currently editing this document.');
                    }
                    infoMessage.headline = gt('Read-Only Mode');
                    infoMessage.action = 'acquireedit';
                    break;
                case 'INFO_DOC_CONVERT_STORED_IN_DEFFOLDER':
                    contextData = Utils.getStringOption(options, 'fullFileName', null);
                    infoMessage.message =
                        //#. %1$s is the new file name
                        //#, c-format
                        gt('Document was converted and stored in your documents folder as "%1$s".', gt.noI18n(contextData));
                    break;
                case 'INFO_DOC_CONVERTED_AND_STORED':
                    contextData = Utils.getStringOption(options, 'fullFileName', null);
                    infoMessage.message =
                        //#. %1$s is the new file name
                        //#, c-format
                        gt('Document was converted and stored as "%1$s".', gt.noI18n(contextData));
                    break;
                case 'INFO_DOC_CREATED_IN_DEFAULTFOLDER':
                    contextData = Utils.getStringOption(options, 'fullFileName', null);
                    infoMessage.message =
                        //#. %1$s is the file name
                        //#, c-format
                        gt('Document "%1$s" was created in your documents folder to allow editing.', gt.noI18n(contextData));
                    break;
                case 'INFO_DOC_STORED_IN_DEFAULTFOLDER_AS':
                    contextData = Utils.getStringOption(options, 'fullFileName', null);
                    infoMessage.message =
                        //#. %1$s is the file name
                        //#, c-format
                        gt('Document has been stored in your documents folder as "%1$s".', gt.noI18n(contextData));
                    break;
                case 'INFO_DOC_SAVED_AS_TEMPLATE':
                    contextData = Utils.getStringOption(options, 'fullFileName', null);
                    contextData2 = Utils.getStringOption(options, 'fullPathName', null);
                    infoMessage.message =
                        //#. %1$s is the file name of the copied document
                        //#, c-format
                        gt('The document was saved as a template "%1$s" in folder "%2$s".', gt.noI18n(contextData), gt.noI18n(contextData2));
                    break;
                case 'INFO_DOC_SAVED_IN_FOLDER':
                    contextData = Utils.getStringOption(options, 'fullFileName', null);
                    contextData2 = Utils.getStringOption(options, 'fullPathName', null);
                    infoMessage.message =
                        //#. %1$s is the file name and %2$s is the entire folder path
                        //#, c-format
                        gt('Document "%1$s" was created in folder "%2$s".', gt.noI18n(contextData), gt.noI18n(contextData2));
                    break;
                case 'INFO_DOC_RESCUED_SUCCESSFULLY':
                    contextData = Utils.getStringOption(options, ErrorContextData.RESTORED_FILENAME, null);
                    infoMessage.message =
                        //#. %1$s is the file name of the restored document
                        //#, c-format
                        gt('Your document could be restored as "%1$s" successfully.', gt.noI18n(contextData));
                    infoMessage.headline = gt('Restore Document');
                    break;
            }
        }

        if (_.isObject(infoMessage)) {
            // ensure that we have a defined headline and type for information messages
            infoMessage.type = infoMessage.type || 'info';

            //type is important
            infoMessage.type = Utils.getStringOption(options, 'type', infoMessage.type);
            infoMessage.duration = Utils.getIntegerOption(options, 'duration', infoMessage.duration);
        }

        return infoMessage;
    };

    // exports ================================================================

    return InfoMessages;

});
