/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/guidedtour', [
    'io.ox/core/tk/wizard',
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/app/baseapplication'
], function (Wizard, Config, Utils, BaseApplication) {

    'use strict';

    // debug mode: whether to show all tours repeatedly
    var DEBUG_SHOW_TOURS = Config.getDebugUrlFlag('office:show-tours');

    // the RequireJS paths of all registered application tours
    var appTourRegistry = {};

    // whether a tour is currently running, for fail-safety
    var isTourRunning = false;

    // mix-in class ApplicationMixin ==========================================

    /**
     * Mix-in class for tour classes to add convenience methods for accessing
     * the components of an application instance.
     *
     * @constructor
     */
    function ApplicationMixin() {

        // public methods -----------------------------------------------------

        /**
         * Convenience method to get the application that is currently active.
         *
         * @returns {ox.ui.App|Null}
         *  The application that is currently active, if available. Note that
         *  the returned application may not be a Document application (i.e. an
         *  instance of BaseApplication).
         */
        this.getApp = ox.ui.App.getCurrentApp;

        /**
         * Convenience method to get the document model of the Documents
         * application that is currently active.
         *
         * @returns {BaseModel|Null}
         *  The document model of the Documents application that is currently
         *  active, if available.
         */
        this.getDocModel = function () {
            var app = this.getApp();
            return BaseApplication.isInstance(app) ? app.getModel() : null;
        };

        /**
         * Convenience method to get the document view of the Documents
         * application that is currently active.
         *
         * @returns {BaseView|Null}
         *  The document view of the Documents application that is currently
         *  active, if available.
         */
        this.getDocView = function () {
            var app = this.getApp();
            return BaseApplication.isInstance(app) ? app.getView() : null;
        };

        /**
         * Convenience method to get the document controller of the Documents
         * application that is currently active.
         *
         * @returns {BaseController|Null}
         *  The document controller of the Documents application that is
         *  currently active, if available.
         */
        this.getDocController = function () {
            var app = this.getApp();
            return BaseApplication.isInstance(app) ? app.getController() : null;
        };

        /**
         * Convenience method to execute a controller item of the Documents
         * application that is currently active. Does nothing, if the active
         * application is not a Documents application (i.e. not an instance of
         * BaseApplication).
         *
         * @attention
         *  The method invokes the controller item immediately. It is intended
         *  to be used in an event handler, NOT when defining this tour step.
         *
         * @param {String} key
         *  The key of the controller item to be executed.
         *
         * @param {Any} [value]
         *  The new value of the item.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected according to the result
         *  of the item setter function. See BaseController.executeItem() for
         *  more details.
         */
        this.executeItem = function (key, value) {
            var controller = this.getDocController();
            return controller ? controller.executeItem(key, value) : $.Deferred().reject();
        };

    } // class ApplicationMixin

    // mix-in class StepMixin =================================================

    /**
     * Mix-in class for a tour step of instances of GuidedTour. Provided to add
     * some convenience and special behavior to generic AppSuite tours.
     *
     * @constructor
     *
     * @extends ApplicationMixin
     */
    function StepMixin() {

        // base constructors --------------------------------------------------

        ApplicationMixin.call(this);

        // public methods -----------------------------------------------------

        /**
         * Replacement for the core method Step.referTo() with additional
         * functionality.
         *
         * @param {String} selector
         *  A CSS selector for the DOM element this step will refer to.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.position='right']
         *      The preferred position of the pop-up box, relative to the DOM
         *      element referred by this step. Must be one of 'left', 'right',
         *      'top', or 'bottom'. If the box does not fit at the specified
         *      position, it will be placed at one of the other borders.
         *  @param {Boolean|String} [options.hotspot]
         *      Specifies where to place a hotspot in the element referred by
         *      this step. If omitted, no hotspot will be shown. If set to
         *      true, the DOM element selected with the passed CSS selector
         *      will show a hotspot at its top-left corner. A string will be
         *      used as CSS selector for a descendant element of the DOM
         *      element selected by the parameter 'selector'.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.referTo = _.wrap(this.referTo, function (referTo, selector, options) {

            // call base class method
            referTo.call(this, selector, options);

            // initialize a hotspot
            var hotspot = _.isObject(options) ? options.hotspot : null;
            if (hotspot === true) {
                this.hotspot(selector, options);
            } else if (_.isString(hotspot)) {
                this.hotspot(selector + ' ' + hotspot, options);
            }

            return this;
        });

        /**
         * Lets this step wait for and refer to a specific DOM element. Can be
         * used as shortcut for the two public methods waitFor() and referTo()
         * with the same CSS selector.
         *
         * @param {String} selector
         *  The CSS selector of the element to wait for and to refer to.
         *
         * @param {Object} [options]
         *  Optional parameters. See method StepMixin.referTo() for details.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.waitForAndReferTo = function (selector, options) {
            return this.waitFor(selector).referTo(selector, options);
        };

        /**
         * Lets this step wait for and refer to a pop-up node.
         *
         * @param {Object} [options]
         *  Optional parameters. See method StepMixin.referTo() for details.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.waitForAndReferToPopup = function (options) {
            return this.waitForAndReferTo('.io-ox-office-main.popup-container', options);
        };

        /**
         * Lets this step refer to the root node of a specific control group.
         *
         * @param {String} selector
         *  A CSS selector for an ancestor element of the control group.
         *
         * @param {String} key
         *  The controller key the control group is bound to.
         *
         * @param {Object} [options]
         *  Optional parameters. See method StepMixin.referTo() for details.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.referToGroup = function (selector, key, options) {
            return this.referTo(GuidedTour.getGroupSelector(selector, key), options);
        };

        /**
         * Lets this step wait for and refer to the root node of a specific
         * control group.
         *
         * @param {String} selector
         *  A CSS selector for an ancestor element of the control group.
         *
         * @param {String} key
         *  The controller key the control group is bound to.
         *
         * @param {Object} [options]
         *  Optional parameters. See method StepMixin.referTo() for details.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.waitForAndReferToGroup = function (selector, key, options) {
            return this.waitForAndReferTo(GuidedTour.getGroupSelector(selector, key), options);
        };

        /**
         * Lets this step refer to a specific option button element in a
         * control group, e.g. a radio button group.
         *
         * @param {String} selector
         *  A CSS selector for an ancestor element of the control group.
         *
         * @param {String} key
         *  The controller key the control group is bound to.
         *
         * @param {Any} value
         *  The value of the option button to be highlighted.
         *
         * @param {Object} [options]
         *  Optional parameters. See method StepMixin.referTo() for details.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.referToOptionButton = function (selector, key, value, options) {
            return this.referTo(GuidedTour.getOptionButtonSelector(selector, key, value), options);
        };

        /**
         * Lets this step wait for and refer to a specific option button
         * element in a control group, e.g. a radio button group.
         *
         * @param {String} selector
         *  A CSS selector for an ancestor element of the control group.
         *
         * @param {String} key
         *  The controller key the control group is bound to.
         *
         * @param {Any} value
         *  The value of the option button to be highlighted.
         *
         * @param {Object} [options]
         *  Optional parameters. See method StepMixin.referTo() for details.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.waitForAndReferToOptionButton = function (selector, key, value, options) {
            return this.waitForAndReferTo(GuidedTour.getOptionButtonSelector(selector, key, value), options);
        };

        /**
         * Immediately triggers a 'remote' event at the root node of the
         * specified control group.
         *
         * @attention
         *  The method triggers the event immediately. It is intended to be
         *  used in an event handler, NOT when defining this tour step.
         *
         * @param {String} selector
         *  A CSS selector for an ancestor element of the control group.
         *
         * @param {String} key
         *  The controller key the control group is bound to.
         *
         * @param {String} command
         *  The command to be passed with the 'remote' event.
         *
         * @param {Any} value
         *  The value for the command to be passed with the 'remote' event.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.triggerRemoteEvent = function (selector, key, command, value) {
            $(GuidedTour.getGroupSelector(selector, key)).first().trigger('remote', [command, value]);
            return this;
        };

        /**
         * Immediately triggers a remote click event at the root node of the
         * specified button control.
         *
         * @attention
         *  The method triggers the click event immediately. It is intended to
         *  be used in an event handler, NOT when defining this tour step.
         *
         * @param {String} selector
         *  A CSS selector for an ancestor element of the control group.
         *
         * @param {String} key
         *  The controller key the button control is bound to.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.triggerButton = function (selector, key) {
            return this.triggerRemoteEvent(selector, key, 'click');
        };

        /**
         * Immediately triggers a remote click event at the option button
         * element of the specified control group, e.g. a radio button group.
         *
         * @attention
         *  The method triggers the click event immediately. It is intended to
         *  be used in an event handler, NOT when defining this tour step.
         *
         * @param {String} selector
         *  A CSS selector for an ancestor element of the control group.
         *
         * @param {String} key
         *  The controller key the button control is bound to.
         *
         * @param {Any} value
         *  The value of the option button to be clicked.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.triggerOptionButton = function (selector, key, value) {
            return this.triggerRemoteEvent(selector, key, 'click', value);
        };

        /**
         * Opens the drop-down menu attached to the specified control group
         * when this step will be activated, and closes the drop-down menu when
         * leaving this step.
         *
         * @returns {StepMixin}
         *  A reference to this instance.
         */
        this.showDropDownMenu = function (selector, key) {

            // open the drop-down menu via 'remote' event
            this.on('before:show', this.triggerRemoteEvent.bind(this, selector, key, 'show'));

            // close the drop-down menu via 'remote' event
            this.on('before:hide', this.triggerRemoteEvent.bind(this, selector, key, 'hide'));

            return this;
        };

    } // class StepMixin

    // class GuidedTour =======================================================

    /**
     * A guided tour through an OX Documents application. Provided to add some
     * convenience and special behavior to generic AppSuite tours.
     *
     * @constructor
     *
     * @extends Wizard
     * @extends ApplicationMixin
     */
    var GuidedTour = _.makeExtendable(Wizard).extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        Wizard.call(this);
        ApplicationMixin.call(this);

        // public methods -----------------------------------------------------

        /**
         * Starts this tour, except if another tour is currently running.
         */
        this.start = _.wrap(this.start, function (startFunc) {
            if (isTourRunning) { return this; }
            isTourRunning = true;
            return startFunc.call(this);
        });

        /**
         * Creates a new tour step that has been extended with the StepMixin
         * mix-in class.
         *
         * @returns {Step}
         *  A new tour step.
         */
        this.step = _.wrap(this.step, function (stepFunc) {
            var step = stepFunc.apply(this, _.toArray(arguments).slice(1));
            StepMixin.call(step);
            return step;
        });

        // initialization -----------------------------------------------------

        // when closing the tour, move browser focus into Documents applications
        this.on('stop', function () {
            var app = this.getApp();
            if (BaseApplication.isInstance(app)) {
                app.getView().grabFocus();
            }
            isTourRunning = false;
        });

    } }); // class GuidedTour

    // static methods ---------------------------------------------------------

    /**
     * Returns the configuration key used to store the 'shown' flag for a tour.
     */
    function getShownConfigKey(requirePath) {
        return requirePath + '/shown';
    }

    /**
     * Returns the CSS selector for the root node of a specific control group.
     *
     * @param {String} selector
     *  A CSS selector for an ancestor element of the control group.
     *
     * @param {String} key
     *  The controller key the control group is bound to.
     *
     * @returns {String}
     *  The CSS selector for the root node of the specified control group.
     */
    GuidedTour.getGroupSelector = function (selector, key) {
        return selector + ' .group[data-key="' + key + '"]:visible';
    };

    /**
     * Returns the CSS selector for a specific option button element in a
     * control group, e.g. a radio button group.
     *
     * @param {String} selector
     *  A CSS selector for an ancestor element of the control group.
     *
     * @param {String} key
     *  The controller key the control group is bound to.
     *
     * @param {Any} value
     *  The value of the option button to be selected.
     *
     * @returns {String}
     *  The CSS selector for the option button in the control group.
     */
    GuidedTour.getOptionButtonSelector = function (selector, key, value) {
        return GuidedTour.getGroupSelector(selector, key) + ' .button[data-value="' + value + '"]';
    };

    /**
     * Returns whether the specified tour has been shown at least once.
     *
     * @param {String} requirePath
     *  The path to the module that exports the tour class constructor.
     *
     * @returns {Boolean}
     *  Whether the specified tour has been shown at least once.
     */
    GuidedTour.isShown = function (requirePath) {
        return Config.getFlag(getShownConfigKey(requirePath));
    };

    /**
     * Starts the specified tour, and sets a server configuration item that
     * specifies that the tour has been shown.
     *
     * @param {String} requirePath
     *  The path to the module that exports the tour class constructor. The
     *  options passed to this method will be forwarded to the tour
     *  constructor.
     *
     * @param {Object} [options]
     *  Optional parameters (will also be passed to the tour constructor):
     *  @param {Boolean} [options.once=false]
     *      If set to true, the tour will only be started, if it has not been
     *      shown ever (if the configuration item mentioned above has not been
     *      set yet).
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the running tour instance; or that
     *  will be rejected, if the tour has not been started (e.g. when using the
     *  option 'once').
     */
    GuidedTour.run = function (requirePath, options) {

        // do not run a tour twice if specified (override the once flag with debug URL hash flag)
        var once = Utils.getBooleanOption(options, 'once', false);
        if (once && !DEBUG_SHOW_TOURS && GuidedTour.isShown(requirePath)) {
            return $.Deferred().reject();
        }

        // require the tour implementation dynamically
        return require([requirePath]).then(function (TourClass) {
            Config.set(getShownConfigKey(requirePath), true);
            return new TourClass(options).start();
        });
    };

    /**
     * Registers a welcome tour for the specified application type. A menu
     * entry starting the tour will be added to the global 'Settings' drop-down
     * menu of the AppSuite.
     *
     * @param {String} appType
     *  The application type identifier.
     *
     * @param {String} requirePath
     *  The path to the module that exports the tour class constructor.
     */
    GuidedTour.registerAppTour = function (appType, requirePath) {
        appTourRegistry[appType] = requirePath;
        Wizard.registry.add({ id: 'default/' + appType }, function () { GuidedTour.run(requirePath); });
    };

    /**
     * Returns whether a welcome tour has been registered for the specified
     * application type with the method GuidedTour.registerAppTour().
     *
     * @param {String} appType
     *  The application type identifier.
     *
     * @returns {Boolean}
     *  Whether a welcome tour has been registered for the specified
     *  application type.
     */
    GuidedTour.hasAppTour = function (appType) {
        return appType in appTourRegistry;
    };

    /**
     * Returns whether the welcome tour for the specified application type has
     * been shown at least once.
     *
     * @param {String} appType
     *  The application type identifier.
     *
     * @returns {Boolean}
     *  Whether the specified welcome tour has been shown at least once.
     */
    GuidedTour.isAppTourShown = function (appType) {
        var requirePath = appTourRegistry[appType];
        return !!requirePath && GuidedTour.isShown(requirePath);
    };

    /**
     * Starts the welcome tour for the specified application type.
     *
     * @param {String} appType
     *  The application type identifier.
     *
     * @param {Object} [options]
     *  Optional parameters. See method GuidedTour.run() for details.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the running tour instance; or that
     *  will be rejected, if the tour has not been started (e.g. when using the
     *  option 'once').
     */
    GuidedTour.runAppTour = function (appType, options) {
        var requirePath = appTourRegistry[appType];
        return requirePath ? GuidedTour.run(requirePath, options) : $.Deferred().reject();
    };

    // exports ================================================================

    return GuidedTour;

});
