/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/extensionregistry', [
    'io.ox/core/capabilities',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/config'
], function (Capabilities, Utils, Config) {

    'use strict';

    // which file formats are editable (maps format identifier to booleans)
    var EDITABLE_FILE_FORMATS = {
        ooxml: true,
        odf: Config.getFlag('odfsupport', true)
    };

    // the extension suffix for files with an internal error
    var INTERNAL_ERROR_SUFFIX = '_ox';

    // File extension configuration
    //
    // Each extension configuration entry supports the following properties:
    // - {String} [format]
    //      The file format used by the server-side filter component to load
    //      and save the contents of the document. If omitted, documents with
    //      the current file extension cannot be edited. The following file
    //      formats are supported:
    //      - 'ooxml': Office Open XML (Microsoft Office)
    //      - 'odf' OpenDocument Format (OpenOffice/LibreOffice)
    // - {String} [template]
    //      The file extension of the template file used to create a new
    //      document with the current file extension. If omitted, there is no
    //      template file extension available for the current file format (i.e.
    //      it is not possible to create a template for the edited document).
    // - {Boolean} [macros=false]
    //      If set to true, the document may contain scripting macros.
    // - {Boolean} [convert=false]
    //      If set to true, the server needs to convert the original file to
    //      the file format specified with the property 'format'.
    //
    var FILE_EXTENSION_CONFIGURATION = {

        text: {
            module: 'io.ox/office/text',
            requires: 'text',
            editable: !_.browser.Android || Utils.supportedChromeVersionOnAndroidForText(),
            extensions: {
                docx: { format: 'ooxml', template: 'dotx' },
                docm: { format: 'ooxml', template: 'dotm', macros: true },
                doc:  { format: 'ooxml', template: 'dot', macros: true, convert: true },
                odt:  { format: 'odf', template: 'ott',  macros: true },
                rtf:  { format: 'ooxml', convert: true }
            }
        },

        spreadsheet: {
            module: 'io.ox/office/spreadsheet',
            requires: 'spreadsheet',
            editable: true,
            extensions: {
                xlsx: { format: 'ooxml', template: 'xltx' },
                xlsm: { format: 'ooxml', template: 'xltm', macros: true },
                xlsb: { format: 'ooxml', macros: true, convert: true },
                xlam: { macros: true }, // add-in
                xls:  { format: 'ooxml', template: 'xlt', macros: true, convert: true },
                xla:  { macros: true }, // add-in
                ods:  { format: 'odf', template: 'ots', macros: true }
            }
        },

        presentation: {
            module: 'io.ox/office/presentation',
            requires: 'presentation',
            editable: !_.browser.Android || Utils.supportedChromeVersionOnAndroidForText(),
            extensions: {
                pptx: { format: 'ooxml', template: 'potx' },
                pptm: { format: 'ooxml', template: 'potm', macros: true },
                ppsx: {}, // slide show
                ppsm: { macros: true }, // slide show
                ppam: { macros: true }, // add-in
                // disabled for US DOCS-618, no direct odf support, and no doc-converter support
                // odp:  { format: 'odf', template: 'otp', macros: true },
                ppt:  { format: 'ooxml', template: 'pot', macros: true, convert: true },
                pps:  { format: 'ooxml', macros: true, convert: true }, // slide show
                ppa:  { format: 'ooxml', macros: true, convert: true } // add-in
            }
        }
    };

    // extension to template extension map
    var TEMPLATE_EXT_MAP = {};

    // all application configurations, mapped by application identifier
    var applicationMap = {};

    // all configurations, mapped by lower-case extension
    var fileExtensionMap = {};

    // can we edit guard protected documents?
    var editGuardDocs = Capabilities.has('guard-docs');

    // private global functions ===============================================

    function registerFileExtension(extension, settings) {

        if (extension in fileExtensionMap) {
            Utils.warn('ExtensionRegistry: extension "' + extension + '" already registered');
        }

        fileExtensionMap[extension] = settings;

        // add extension with internal error suffix (native only)
        if (settings.native) {
            fileExtensionMap[extension + INTERNAL_ERROR_SUFFIX] = settings;
        }
    }

    // static class ExtensionRegistry =========================================

    /**
     * Configuration settings for all supported file extensions of all OX
     * Documents application.
     */
    var ExtensionRegistry = {};

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the passed module name specifies an OX Documents
     * application that can edit documents on the current platform.
     *
     * @param {String} editModule
     *  The module name of the application.
     *
     * @returns {Boolean}
     *  Whether the application is able to edit documents.
     */
    ExtensionRegistry.supportsEditMode = function (editModule) {
        return Utils.getBooleanOption(applicationMap[editModule], 'editable', false);
    };

    /**
     * Returns the configuration settings of the file extension contained by
     * the passed file name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     *  @param {Boolean} ignoreGuardExt
     *  Ignore/skip the Guard extension
     *
     * @returns {Object|Null}
     *  The file extension settings; or null, if the extension is not known.
     */
    ExtensionRegistry.getExtensionSettings = function (fileName, editModule, ignoreGuardExt) {

        // the lower-case extension of the passed file name
        var extension = Utils.getFileExtension(fileName, ignoreGuardExt);
        // the extension settings object
        var extensionSettings = (extension in fileExtensionMap) ? fileExtensionMap[extension] : null;
        // the module name of the edit application for the extension
        var module = Utils.getStringOption(extensionSettings, 'module');

        // passed module name must match if specified
        return (_.isString(editModule) && (editModule !== module)) ? null : extensionSettings;
    };

    /**
     * Returns the type category of the file with the passed file name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {String}
     *  The type category of the file with the passed file name; or an empty
     *  string, if the file extension is not registered.
     *  - 'text': A text document.
     *  - 'spreadsheet': A spreadsheet document.
     *  - 'presentation': A presentation document.
     */
    ExtensionRegistry.getFileCategory = function (fileName) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName);
        return Utils.getStringOption(extensionSettings, 'type', '');
    };

    /**
     * Returns an array containing all file extensions that can be edited.
     *
     * @returns {Array<String>}
     *  An array containing all file extensions that can be edited.
     */
    ExtensionRegistry.getEditableExtensions = function () {
        var extensions = [];
        _.each(fileExtensionMap, function (settings, extension) {
            if (settings.editable) {
                extensions.push(extension);
            }
        });
        return extensions;
    };

    /**
     * Returns the module name of the edit application that is able to modify
     * the file with the passed file name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {String|Null}
     *  The module name of the edit application that is able to modify the file
     *  with the passed file name; or null, if no edit module exists for the
     *  file extension.
     */
    ExtensionRegistry.getEditModule = function (fileName) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, null, editGuardDocs);
        return Utils.getStringOption(extensionSettings, 'module', null);
    };

    /**
     * Returns whether the file with the passed name is editable by any of the
     * OX Documents edit applications.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file is editable by one of the OX Documents edit
     *  applications.
     */
    ExtensionRegistry.isEditable = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule, editGuardDocs);
        return Utils.getBooleanOption(extensionSettings, 'editable', false);
    };

    /**
     * Returns whether the file with the passed name is editable natively with
     * any of the OX Documents edit applications, according to the file format.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file is editable natively with one of the OX
     *  Documents edit applications.
     */
    ExtensionRegistry.isNative = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule, editGuardDocs);
        return Utils.getBooleanOption(extensionSettings, 'editable', false) && Utils.getBooleanOption(extensionSettings, 'native', false);
    };

    /**
     * Returns whether the file with the passed name is editable with any of
     * the OX Documents edit applications by conversion to a natively supported
     * file format, according to the file format.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file is editable with one of the OX Documents
     *  edit applications by conversion to a native file format.
     */
    ExtensionRegistry.isConvertible = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule);
        return Utils.getBooleanOption(extensionSettings, 'editable', false) && !Utils.getBooleanOption(extensionSettings, 'native', false);
    };

    /**
     * Returns the file format identifier of the file with the passed name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {String}
     *  The file format identifier of the file with the passed name:
     *  - 'ooxml' for a file that will be edited in the OfficeOpenXML file
     *      format (either natively, or by conversion from a file format not
     *      supported natively).
     *  - 'odf' for a file that will be edited in the OpenDocument file format.
     *  - An empty string for all other file names.
     */
    ExtensionRegistry.getFileFormat = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule, true);
        return Utils.getStringOption(extensionSettings, 'format', '');
    };

    /**
     * Returns whether the file with the passed name is a template file.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file is a document template file.
     */
    ExtensionRegistry.isTemplate = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule);
        return Utils.getBooleanOption(extensionSettings, 'template', false);
    };

    /**
     * Provides the template file extension for a given file name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {String|Null}
     *  The template extension for the file name; or null if the format cannot
     *  be written to a template format. If the passed file name refers to a
     *  template file by itself, its own extension will be returned.
     */
    ExtensionRegistry.getTemplateExtension = function (fileName) {
        var extension = Utils.getFileExtension(fileName);
        return TEMPLATE_EXT_MAP[extension] || null;
    };

    /**
     * Returns whether the file with the passed name may contain macro scripts.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @param {String} [editModule]
     *  If specified, must match the module name of the edit application
     *  registered for the extension of the passed file name. If omitted, the
     *  edit application module name for the extension will be ignored.
     *
     * @returns {Boolean}
     *  Whether the specified file may contain macro scripts.
     */
    ExtensionRegistry.isScriptable = function (fileName, editModule) {
        var extensionSettings = ExtensionRegistry.getExtensionSettings(fileName, editModule);
        return Utils.getBooleanOption(extensionSettings, 'macros', false);
    };

    /**
     * Returns whether the file with the passed name contains the 'internal
     * error' extension suffix.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {Boolean}
     *  Whether the file with the passed name contains the 'internal error'
     *  extension suffix.
     */
    ExtensionRegistry.isGuardEncrypted = function (fileName) {
        return Utils.isGuardEncrypted(fileName);
    };

    /**
     * Returns whether the file with the passed name contains the 'internal
     * error' extension suffix.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {Boolean}
     *  Whether the file with the passed name contains the 'internal error'
     *  extension suffix.
     */
    ExtensionRegistry.isError = function (fileName) {
        var extension = Utils.getFileExtension(fileName);
        return extension.substr(-INTERNAL_ERROR_SUFFIX.length) === INTERNAL_ERROR_SUFFIX;
    };

    /**
     * Extends and returns the passed file name with the 'internal error'
     * extension suffix. If the file name already contains this suffix, it will
     * be returned as is.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {String}
     *  The passed passed file name with the 'internal error' extension suffix.
     */
    ExtensionRegistry.createErrorFileName = function (fileName) {
        return ExtensionRegistry.isError(fileName) ? fileName : (fileName + INTERNAL_ERROR_SUFFIX);
    };

    // static initialization ==================================================

    // process the configuration of all modules
    _.each(FILE_EXTENSION_CONFIGURATION, function (moduleConfiguration, documentType) {

        // the capability required to edit the file
        var editable = Utils.getBooleanOption(moduleConfiguration, 'editable', false);
        // the capability required to edit the file
        var requires = Utils.getStringOption(moduleConfiguration, 'requires', null);
        // the module name of the edit application
        var module = Utils.getStringOption(moduleConfiguration, 'module', null);
        // whether the edit module is available at all
        var editAvailable = editable && _.isString(requires) && Capabilities.has(requires) && _.isString(module);

        // application configuration entry
        applicationMap[documentType] = { editable: editAvailable };

        // process all extensions registered for the module
        _.each(moduleConfiguration.extensions, function (extensionSettings, extension) {

            // the file format of files with the current extension
            var format = Utils.getStringOption(extensionSettings, 'format', '');
            // whether the file extension is supported natively
            var native = !Utils.getBooleanOption(extensionSettings, 'convert', false);
            // the file extension of template files
            var template = Utils.getStringOption(extensionSettings, 'template', null);

            // initialize all properties of the current extension
            registerFileExtension(extension, extensionSettings = {
                module: module,
                type: documentType,
                format: format,
                template: false,
                editable: editAvailable && Utils.getBooleanOption(EDITABLE_FILE_FORMATS, format, false),
                native: native,
                macros: Utils.getBooleanOption(extensionSettings, 'macros', false)
            });

            // register the template extension, store the template file extension for the
            // original extension, AND for the template extension
            if (template) {
                registerFileExtension(template, _.extend({}, extensionSettings, { template: true }));
                TEMPLATE_EXT_MAP[extension] = TEMPLATE_EXT_MAP[template] = template;
            }
        });
    });

    // exports ================================================================

    return ExtensionRegistry;

});
