/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

(function () {

    'use strict';

    // helper function that does nothing but returns the calling context
    function noop() { return this; }

    // mocked prototype methods
    if (!_.isFunction(String.prototype.normalize)) {
        String.prototype.normalize = noop;
    }

    // class CoreApplication ==================================================

    /**
     * Replacement for a core application for unit tests. Mimics the public API
     * of an application object as returned by the method ox.ui.createApp().
     *
     * @constructor
     */
    function CoreApplication(options) {

        var appWindow = null,
            launcher = null,
            self = this;

        this.on = this.one = this.off = this.listenTo = noop;
        this.registerGlobalEventHandler = this.registerWindowResizeHandler = noop;

        this.getName = _.constant(options.name);
        this.setTitle = _.noop;

        this.getWindow = function () { return appWindow; };
        this.getWindowNode = function () { return appWindow.nodes.main; };
        this.setWindow = function (appWin) { appWindow = appWin; };

        this.setLauncher = function (callback) { launcher = callback; };
        this.launch = function () { return $.when(launcher()); };

        this.getInstance = function () { return self; };

        this.setQuit = _.noop;
        this.quit = _.noop;

        this.isInQuit = _.noop;

    } // class CoreApplication

    // static methods ---------------------------------------------------------

    /**
     * Returns a new instance of the class CoreApplication.
     */
    CoreApplication.createApp = function (options) {
        return new CoreApplication(options);
    };

    /**
     * Returns a replacement for a core application window for unit tests.
     * Mimics the public API of an application window object as returned by the
     * method ox.ui.createWindow().
     */
    CoreApplication.createWindow = function () {
        var appWin = $({});
        appWin.nodes = { main: $('<div>'), outer: $('<div>') };
        appWin.state = { visible: true };
        appWin.setTitle = _.constant(appWin);
        appWin.busy = _.constant(appWin);
        appWin.idle = _.constant(appWin);
        return appWin;
    };

    /**
     * Launches an application instance of the specified module type (one of
     * the Documents editor application types).
     *
     * @returns {CoreApplication}
     *  An application instance. The passed Application class constructor has
     *  been mixed into this instance.
     */
    CoreApplication.launchApp = function (module, Application) {
        var app = new CoreApplication(module);
        Application.call(app, { action: 'new' });
        app.launch();
        return app;
    };

    // window.ox Replacement ==================================================

    /**
     * Replacement for the global object 'window.ox' for unit tests.
     */
    window.ox = {

        // standard properties
        base: 'v=7.x.x.20150511.070854',
        language: 'de_DE',

        // event interface
        on: noop,
        one: noop,
        off: noop,

        // standard contents of the 'ox.ui' object
        ui: {
            createApp: CoreApplication.createApp,
            createWindow: CoreApplication.createWindow,

            App: {
                getCurrentApp: _.constant(null),
                getCurrentWindow: _.constant(null)
            }
        }
    };

    // ========================================================================
}());
