/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils/simplemap'
], function (SimpleMap) {

    'use strict';

    // class SimpleMap ========================================================

    describe('Toolkit class SimpleMap', function () {

        it('should exist', function () {
            expect(SimpleMap).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "empty"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('empty');
            });
            it('should return whether the map is empty', function () {
                var map = new SimpleMap();
                expect(map.empty()).to.equal(true);
                map.insert('a', 1);
                expect(map.empty()).to.equal(false);
            });
        });

        describe('method "clear"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('clear');
            });
            it('should clear the map', function () {
                var map = new SimpleMap();
                map.insert('a', 1);
                expect(map.empty()).to.equal(false);
                expect(map.clear()).to.equal(map);
                expect(map.empty()).to.equal(true);
            });
        });

        describe('method "has"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('has');
            });
            it('should return whether the element exists', function () {
                var map = new SimpleMap();
                map.insert('a', 1);
                expect(map.has('a')).to.equal(true);
                expect(map.has('b')).to.equal(false);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('get');
            });
            it('should return an element', function () {
                var map = new SimpleMap();
                map.insert('a', 1);
                expect(map.get('a')).to.equal(1);
                expect(map.get('b')).to.equal(undefined);
                expect(map.get('a', null)).to.equal(1);
                expect(map.get('b', null)).to.equal(null);
            });
        });

        describe('method "getAny"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('getAny');
            });
            it('should return an element', function () {
                var map = new SimpleMap();
                expect(map.getAny()).to.equal(undefined);
                expect(map.getAny(null)).to.equal(null);
                map.insert('a', 1);
                map.insert('b', 2);
                map.insert('c', 3);
                var v1 = map.getAny();
                expect(v1).to.be.a('number');
                var v2 = map.getAny();
                expect(v2).to.equal(v1);
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('clone');
            });
            var map = new SimpleMap();
            map.insert('a', Object);
            it('should create a shallow clone', function () {
                var clone = map.clone();
                expect(clone.get('a')).to.equal(Object);
                expect(clone.get('b')).to.equal(undefined);
            });
            it('should create a deep clone', function () {
                var spy = sinon.spy(_.constant(42));
                var context = {};
                var clone = map.clone(spy, context);
                expect(clone.get('a')).to.equal(42);
                expect(clone.get('b')).to.equal(undefined);
                sinon.assert.alwaysCalledOn(spy, context);
            });
        });

        describe('method "insert"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('insert');
            });
            it('should insert map elements', function () {
                var map = new SimpleMap();
                expect(map.has('a')).to.equal(false);
                expect(map.insert('a', Object)).to.equal(Object);
                expect(map.has('a')).to.equal(true);
                expect(map.get('a')).to.equal(Object);
                expect(map.insert('a', Array)).to.equal(Array);
                expect(map.get('a')).to.equal(Array);
                expect(map.insert('b', 42)).to.equal(42);
                expect(map.get('a')).to.equal(Array);
                expect(map.get('b')).to.equal(42);
            });
        });

        describe('method "remove"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('remove');
            });
            it('should remove map elements', function () {
                var map = new SimpleMap();
                map.insert('a', 42);
                map.insert('b', 'abc');
                expect(map.has('a')).to.equal(true);
                expect(map.remove('a')).to.equal(42);
                expect(map.has('a')).to.equal(false);
                expect(map.remove('c')).to.equal(undefined);
                expect(map.remove('c', null)).to.equal(null);
            });
        });

        describe('method "move"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('move');
            });
            it('should move map elements', function () {
                var map = new SimpleMap();
                map.insert('a', 42);
                map.insert('b', 'abc');
                expect(map.has('a')).to.equal(true);
                expect(map.move('a', 'c')).to.equal(map);
                expect(map.has('a')).to.equal(false);
                expect(map.get('c')).to.equal(42);
                expect(map.move('c', 'b')).to.equal(map);
                expect(map.get('b')).to.equal(42);
            });
        });

        describe('method "merge"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('remove');
            });
            it('should merge the maps', function () {
                var map1 = new SimpleMap();
                map1.insert('a', 42);
                map1.insert('b', 'abc');
                var map2 = new SimpleMap();
                map2.insert('b', false);
                map2.insert('c', null);
                expect(map1.has('c')).to.equal(false);
                expect(map1.merge(map2)).to.equal(map1);
                expect(map1.get('a')).to.equal(42);
                expect(map1.get('b')).to.equal(false);
                expect(map1.get('c')).to.equal(null);
            });
        });

        describe('method "getOrCreate"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('getOrCreate');
            });
            it('should return existing or create new elements', function () {
                var map = new SimpleMap();
                var spy = sinon.spy(_.constant('abc'));
                var context = {};
                map.insert('a', 42);
                expect(map.getOrCreate('a', spy, context)).to.equal(42);
                sinon.assert.callCount(spy, 0);
                expect(map.getOrCreate('b', spy, context)).to.equal('abc');
                sinon.assert.callCount(spy, 1);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 'b');
            });
        });

        describe('method "getOrConstruct"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('getOrConstruct');
            });
            it('should return existing or create new elements', function () {
                var map = new SimpleMap();
                map.insert('a', 42);
                expect(map.getOrConstruct('a', Object)).to.equal(42);
                expect(map.getOrConstruct('b', Object)).to.deep.equal({});
                expect(map.getOrConstruct('c', RegExp, 'a')).to.deep.equal(/a/);
                expect(map.getOrConstruct('d', Array, 1, 2)).to.deep.equal([1, 2]);
                expect(map.getOrConstruct('e', Array, 1, 2, 3)).to.deep.equal([1, 2, 3]);
                expect(map.getOrConstruct('f', Array, 1, 2, 3, 4)).to.deep.equal([1, 2, 3, 4]);
                expect(map.getOrConstruct('g', Array, 1, 2, 3, 4, 5, 6, 7, 8)).to.deep.equal([1, 2, 3, 4, 5, 6, 7, 8]);
            });
        });

        describe('method "with"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('with');
            });
            it('should invoke callback for existing elements', function () {
                var map = new SimpleMap();
                map.insert('a', 42);
                var spy = sinon.spy(_.constant('abc'));
                var context = {};
                expect(map.with('b', spy, context)).to.equal(undefined);
                sinon.assert.callCount(spy, 0);
                expect(map.with('a', spy, context)).to.equal('abc');
                sinon.assert.callCount(spy, 1);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 42);
            });
        });

        describe('method "keys"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('keys');
            });
            it('should return all keys', function () {
                var map = new SimpleMap();
                map.insert('a', 42);
                map.insert('b', 'abc');
                var keys = map.keys();
                expect(keys).to.be.an('array');
                expect(keys.sort()).to.deep.equal(['a', 'b']);
            });
            it('should return an empty array for an empty map', function () {
                expect(new SimpleMap().keys()).to.deep.equal([]);
            });
        });

        describe('method "values"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('values');
            });
            it('should return all values', function () {
                var map = new SimpleMap();
                map.insert('a', 42);
                map.insert('b', 'abc');
                var values = map.values();
                expect(values).to.be.an('array');
                expect(values.indexOf(42)).to.be.at.least(0);
                expect(values.indexOf('abc')).to.be.at.least(0);
            });
            it('should return an empty array for an empty map', function () {
                expect(new SimpleMap().values()).to.deep.equal([]);
            });
        });

        describe('method "iterator"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('iterator');
            });
            it('should return a property iterator', function () {
                var map = new SimpleMap();
                map.insert('a', 'c');
                map.insert('b', 'd');
                function matcher(result) { return !result.done && /^(ac|bd)$/.test(result.key + result.value); }
                var it = map.iterator();
                expect(it).to.respondTo('next');
                expect(it.next()).to.satisfy(matcher);
                expect(it.next()).to.satisfy(matcher);
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('forEach');
            });
            it('should visit all elements', function () {
                var map = new SimpleMap();
                map.insert('a', 42);
                map.insert('b', 'abc');
                var spy = sinon.spy(), context = {};
                expect(map.forEach(spy, context)).to.equal(map);
                sinon.assert.callCount(spy, 2);
                sinon.assert.alwaysCalledOn(spy, context);
            });
        });

        describe('method "some"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('some');
            });
        });

        describe('method "every"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('every');
            });
        });

        describe('method "map"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('map');
            });
        });

        describe('method "filter"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('filter');
            });
        });

        describe('method "reject"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('reject');
            });
        });

        describe('method "pluck"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('pluck');
            });
        });

        describe('method "find"', function () {
            it('should exist', function () {
                expect(SimpleMap).to.respondTo('find');
            });
        });
    });

    // ========================================================================
});
