/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils/dateutils'
], function (DateUtils) {

    'use strict';

    // static class DateUtils =================================================

    describe('Toolkit module DateUtils', function () {

        it('should exist', function () {
            expect(DateUtils).to.be.an('object');
        });

        // constants ----------------------------------------------------------

        describe('constant "SEC_PER_DAY"', function () {
            it('should exist', function () {
                expect(DateUtils).to.have.a.property('SEC_PER_DAY').that.equals(24 * 60 * 60);
            });
        });

        describe('constant "MSEC_PER_DAY"', function () {
            it('should exist', function () {
                expect(DateUtils).to.have.a.property('MSEC_PER_DAY').that.equals(24 * 60 * 60 * 1000);
            });
        });

        describe('constant "MSEC_PER_WEEK"', function () {
            it('should exist', function () {
                expect(DateUtils).to.have.a.property('MSEC_PER_WEEK').that.equals(7 * 24 * 60 * 60 * 1000);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "getUTCDateComponents"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getUTCDateComponents');
            });
            it('should return the UTC date components', function () {
                expect(DateUtils.getUTCDateComponents(new Date(Date.UTC(2015, 11, 31, 0, 0, 0, 0)))).to.deep.equal({ Y: 2015, M: 11, D: 31 });
                expect(DateUtils.getUTCDateComponents(new Date(Date.UTC(2015, 11, 31, 23, 59, 59, 999)))).to.deep.equal({ Y: 2015, M: 11, D: 31 });
            });
        });

        describe('method "getLocalDateComponents"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getLocalDateComponents');
            });
            it('should return the local date components', function () {
                expect(DateUtils.getLocalDateComponents(new Date(2015, 11, 31, 0, 0, 0, 0))).to.deep.equal({ Y: 2015, M: 11, D: 31 });
                expect(DateUtils.getLocalDateComponents(new Date(2015, 11, 31, 23, 59, 59, 999))).to.deep.equal({ Y: 2015, M: 11, D: 31 });
            });
        });

        describe('method "getUTCDateTimeComponents"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getUTCDateTimeComponents');
            });
            it('should return the UTC date/time components', function () {
                expect(DateUtils.getUTCDateTimeComponents(new Date(Date.UTC(2015, 11, 31, 0, 0, 0, 0)))).to.deep.equal({ Y: 2015, M: 11, D: 31, h: 0, m: 0, s: 0, ms: 0 });
                expect(DateUtils.getUTCDateTimeComponents(new Date(Date.UTC(2015, 11, 31, 23, 59, 59, 999)))).to.deep.equal({ Y: 2015, M: 11, D: 31, h: 23, m: 59, s: 59, ms: 999 });
            });
        });

        describe('method "getLocalDateTimeComponents"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getLocalDateTimeComponents');
            });
            it('should return the local date/time components', function () {
                expect(DateUtils.getLocalDateTimeComponents(new Date(2015, 11, 31, 0, 0, 0, 0))).to.deep.equal({ Y: 2015, M: 11, D: 31, h: 0, m: 0, s: 0, ms: 0 });
                expect(DateUtils.getLocalDateTimeComponents(new Date(2015, 11, 31, 23, 59, 59, 999))).to.deep.equal({ Y: 2015, M: 11, D: 31, h: 23, m: 59, s: 59, ms: 999 });
            });
        });

        describe('method "makeUTCDate"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('makeUTCDate');
            });
            it('should return the UTC date', function () {
                expect(DateUtils.makeUTCDate({ Y: 2015, M: 11, D: 31 }).getTime()).to.equal(new Date(Date.UTC(2015, 11, 31)).getTime());
            });
        });

        describe('method "makeUTCDateTime"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('makeUTCDateTime');
            });
            it('should return the UTC date/time', function () {
                expect(DateUtils.makeUTCDateTime({ Y: 2015, M: 11, D: 31, h: 0, m: 0, s: 0, ms: 0 }).getTime()).to.equal(new Date(Date.UTC(2015, 11, 31, 0, 0, 0, 0)).getTime());
                expect(DateUtils.makeUTCDateTime({ Y: 2015, M: 11, D: 31, h: 23, m: 59, s: 59, ms: 999 }).getTime()).to.equal(new Date(Date.UTC(2015, 11, 31, 23, 59, 59, 999)).getTime());
            });
        });

        describe('method "makeUTCToday"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('makeUTCToday');
            });
            it('should return the UTC date for today', function () {
                var today1 = DateUtils.makeUTCDate(DateUtils.getLocalDateComponents(new Date()));
                var today2 = DateUtils.makeUTCToday();
                var today3 = DateUtils.makeUTCDate(DateUtils.getLocalDateComponents(new Date()));
                expect(today2).to.be.a('date');
                expect((today1 <= today2) && (today2 <= today3)).to.equal(true);
            });
        });

        describe('method "makeUTCNow"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('makeUTCNow');
            });
            it('should return the UTC date for now', function () {
                var now1 = DateUtils.makeUTCDateTime(DateUtils.getLocalDateTimeComponents(new Date()));
                var now2 = DateUtils.makeUTCNow();
                var now3 = DateUtils.makeUTCDateTime(DateUtils.getLocalDateTimeComponents(new Date()));
                expect(now2).to.be.a('date');
                expect((now1 <= now2) && (now2 <= now3)).to.equal(true);
            });
        });

        describe('method "isLeapYear"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('isLeapYear');
            });
            it('should return the correct result', function () {
                expect(DateUtils.isLeapYear(1500)).to.equal(false);
                expect(DateUtils.isLeapYear(1600)).to.equal(true);
                expect(DateUtils.isLeapYear(1700)).to.equal(false);
                expect(DateUtils.isLeapYear(1800)).to.equal(false);
                expect(DateUtils.isLeapYear(1900)).to.equal(false);
                expect(DateUtils.isLeapYear(1995)).to.equal(false);
                expect(DateUtils.isLeapYear(1996)).to.equal(true);
                expect(DateUtils.isLeapYear(1997)).to.equal(false);
                expect(DateUtils.isLeapYear(1998)).to.equal(false);
                expect(DateUtils.isLeapYear(1999)).to.equal(false);
                expect(DateUtils.isLeapYear(2000)).to.equal(true);
                expect(DateUtils.isLeapYear(2001)).to.equal(false);
                expect(DateUtils.isLeapYear(2100)).to.equal(false);
                expect(DateUtils.isLeapYear(2200)).to.equal(false);
                expect(DateUtils.isLeapYear(2300)).to.equal(false);
                expect(DateUtils.isLeapYear(2400)).to.equal(true);
                expect(DateUtils.isLeapYear(2500)).to.equal(false);
            });
        });

        describe('method "getDaysInYear"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getDaysInYear');
            });
            it('should return the correct result', function () {
                expect(DateUtils.getDaysInYear(1500)).to.equal(365);
                expect(DateUtils.getDaysInYear(1600)).to.equal(366);
                expect(DateUtils.getDaysInYear(1700)).to.equal(365);
                expect(DateUtils.getDaysInYear(1800)).to.equal(365);
                expect(DateUtils.getDaysInYear(1900)).to.equal(365);
                expect(DateUtils.getDaysInYear(1995)).to.equal(365);
                expect(DateUtils.getDaysInYear(1996)).to.equal(366);
                expect(DateUtils.getDaysInYear(1997)).to.equal(365);
                expect(DateUtils.getDaysInYear(1998)).to.equal(365);
                expect(DateUtils.getDaysInYear(1999)).to.equal(365);
                expect(DateUtils.getDaysInYear(2000)).to.equal(366);
                expect(DateUtils.getDaysInYear(2001)).to.equal(365);
                expect(DateUtils.getDaysInYear(2100)).to.equal(365);
                expect(DateUtils.getDaysInYear(2200)).to.equal(365);
                expect(DateUtils.getDaysInYear(2300)).to.equal(365);
                expect(DateUtils.getDaysInYear(2400)).to.equal(366);
                expect(DateUtils.getDaysInYear(2500)).to.equal(365);
            });
        });

        describe('method "getDaysInFebruary"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getDaysInFebruary');
            });
            it('should return the correct result', function () {
                expect(DateUtils.getDaysInFebruary(1500)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(1600)).to.equal(29);
                expect(DateUtils.getDaysInFebruary(1700)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(1800)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(1900)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(1995)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(1996)).to.equal(29);
                expect(DateUtils.getDaysInFebruary(1997)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(1998)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(1999)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(2000)).to.equal(29);
                expect(DateUtils.getDaysInFebruary(2001)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(2100)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(2200)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(2300)).to.equal(28);
                expect(DateUtils.getDaysInFebruary(2400)).to.equal(29);
                expect(DateUtils.getDaysInFebruary(2500)).to.equal(28);
            });
        });

        describe('method "getDaysInMonth"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getDaysInMonth');
            });
            it('should return the correct result', function () {
                expect(DateUtils.getDaysInMonth(1999, 0)).to.equal(31);
                expect(DateUtils.getDaysInMonth(1999, 1)).to.equal(28);
                expect(DateUtils.getDaysInMonth(1999, 2)).to.equal(31);
                expect(DateUtils.getDaysInMonth(1999, 3)).to.equal(30);
                expect(DateUtils.getDaysInMonth(1999, 4)).to.equal(31);
                expect(DateUtils.getDaysInMonth(1999, 5)).to.equal(30);
                expect(DateUtils.getDaysInMonth(1999, 6)).to.equal(31);
                expect(DateUtils.getDaysInMonth(1999, 7)).to.equal(31);
                expect(DateUtils.getDaysInMonth(1999, 8)).to.equal(30);
                expect(DateUtils.getDaysInMonth(1999, 9)).to.equal(31);
                expect(DateUtils.getDaysInMonth(1999, 10)).to.equal(30);
                expect(DateUtils.getDaysInMonth(1999, 11)).to.equal(31);
                expect(DateUtils.getDaysInMonth(2000, 1)).to.equal(29);
            });
        });

        describe('method "convertUTCDateToLocalDate"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('convertUTCDateToLocalDate');
            });
            it('should convert UTC date to local date', function () {
                expect(DateUtils.convertUTCDateToLocalDate('2007-08-31T16:47+00:00')).to.equal('2007-08-31T18:47:00Z');
                expect(DateUtils.convertUTCDateToLocalDate('2007-08-31T16:47+00:00', { useMilliSeconds: true })).to.equal('2007-08-31T18:47:00.000Z');
            });
        });

        describe('method "getIsoDateString"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getIsoDateString');
            });
            it('should create an ISO conform date string', function () {
                expect(DateUtils.getIsoDateString()).to.be.a('string');
                expect(DateUtils.getIsoDateString().length).to.equal(24);
                expect(DateUtils.getIsoDateString({ useSeconds: true }).length).to.equal(24);
                expect(DateUtils.getIsoDateString({ useSeconds: false })).to.be.a('string');
                expect(DateUtils.getIsoDateString({ useSeconds: false }).length).to.equal(20);
                expect(DateUtils.getIsoDateString().match(/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d{3}Z$/)[0]).to.be.a('string');
                expect(DateUtils.getIsoDateString({ useSeconds: false }).match(/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:00Z$/)[0]).to.be.a('string');
            });
        });

        describe('method "getMSFormatIsoDateString"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getMSFormatIsoDateString');
            });
            it('should create an iso conform date string', function () {
                expect(DateUtils.getMSFormatIsoDateString()).to.be.a('string');
                expect(DateUtils.getMSFormatIsoDateString({ useSeconds: false })).to.be.a('string');
            });
        });

        describe('method "getUTCDate"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getUTCDate');
            });
            it('should create an UTC conform date string', function () {
                expect(DateUtils.getUTCDate()).to.be.a('date');
                // expect(DateUtils.getUTCDate('1442393575460').toString()).to.be.equal('Wed Sep 16 2015 08:52:55 GMT+0200 (CEST)');
            });
        });

        describe('method "getWeekDayFormat"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getWeekDayFormat');
            });
            it('should create an week-day format', function () {
                expect(DateUtils.getWeekDayFormat()).to.be.a('string');
                expect(DateUtils.getWeekDayFormat()).to.be.equal('dddd');
            });
        });

        describe('method "getLocaleMonths"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getLocaleMonths');
            });
            it('should create an array with month-names', function () {
                expect(DateUtils.getLocaleMonths()).to.be.a('array');
                expect(DateUtils.getLocaleMonths()).to.have.length(12);
                // expect(DateUtils.getLocaleMonths()).to.be.deep.equal(['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December']);
            });
        });

        describe('method "getLocaleMonthsShort"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getLocaleMonthsShort');
            });
            it('should create an array with short month-names', function () {
                expect(DateUtils.getLocaleMonthsShort()).to.be.a('array');
                expect(DateUtils.getLocaleMonthsShort()).to.have.length(12);
                // expect(DateUtils.getLocaleMonthsShort()).to.be.deep.equal(['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']);
            });
        });

        describe('method "getLocaleDays"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getLocaleDays');
            });
            it('should create an array with day-names', function () {
                expect(DateUtils.getLocaleDays()).to.be.a('array');
                expect(DateUtils.getLocaleDays()).to.have.length(7);
                // expect(DateUtils.getLocaleDays()).to.be.deep.equal(['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday']);
            });
        });

        describe('method "getLocaleDaysShort"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getLocaleDaysShort');
            });
            it('should create an array with short day-names', function () {
                expect(DateUtils.getLocaleDaysShort()).to.be.a('array');
                expect(DateUtils.getLocaleDaysShort()).to.have.length(7);
                // expect(DateUtils.getLocaleDaysShort()).to.be.deep.equal(['Su', 'Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa']);
            });
        });

        describe('method "getLocaleMonth"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getLocaleMonth');
            });
            it('should return a string with day-name of given number', function () {
                expect(DateUtils.getLocaleMonth(0)).to.be.a('string');
                // expect(DateUtils.getLocaleMonth(0)).to.be.deep.equal('January');
            });
        });

        describe('method "getLocaleMonthShort"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getLocaleMonthShort');
            });
            it('should return a string with short day-name of given number', function () {
                expect(DateUtils.getLocaleMonthShort(0)).to.be.a('string');
                // expect(DateUtils.getLocaleMonthShort(0)).to.be.deep.equal('Jan');
            });
        });

        describe('method "getLocaleDay"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getLocaleDay');
            });
            it('should return a string with short day-name of given number', function () {
                expect(DateUtils.getLocaleDay(0)).to.be.a('string');
                // expect(DateUtils.getLocaleDay(0)).to.be.deep.equal('Sunday');
            });
        });

        describe('method "getLocaleDayShort"', function () {
            it('should exist', function () {
                expect(DateUtils).itself.to.respondTo('getLocaleDayShort');
            });
            it('should return string with short day-name of the gicen number', function () {
                expect(DateUtils.getLocaleDayShort(0)).to.be.a('string');
                // expect(DateUtils.getLocaleDayShort(0)).to.be.deep.equal('Su');
            });
        });
    });

    // ========================================================================
});
