/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/selection/multiselectionmixin'
], function (AppHelper, DrawingFrame, DOM, MultiSelectionMixin) {

    'use strict';

    // class MultiSelectionMixin ==============================================

    describe('Textframework class MultiSelectionMixin', function () {

        // private helpers ----------------------------------------------------

        var model = null,
            selection = null,
            layoutId_1 = 'layout1',
            layoutId_2 = 'layout2',
            masterId_1 = 'master1',
            ctrTitleLeft = 1900,
            subTitleLeft = 3800,
            drawing1Left = 2000,
            ctrTitleTop = 5900,
            subTitleTop = 10800,
            drawing1Top = 6000,
            ctrTitleHeight = 4000,
            subTitleHeight = 5000,
            drawing1Height = 3600,
            ctrTitleWidth = 21600,
            subTitleWidth = 17800,
            drawing1Width = 6000,
            drawingDelta = 100, // shift in 1/100 mm
            drawingSizePercentage = 0.05, // 5% of increase or decrease
            activeSlide = null,
            drawingsOnSlide = null,

            // the operations to be applied by the document model
            OPERATIONS = [
                {
                    name: 'setDocumentAttributes',
                    attrs: { page: { width: 33866, height: 19050, orientation: 'landscape' } }
                },
                { name: 'insertMasterSlide', id: masterId_1 },

                { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
                { name: 'insertDrawing', start: [0, 0], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Titel 1', left: ctrTitleLeft, top: ctrTitleTop, width: ctrTitleWidth, height: ctrTitleHeight } } },
                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 0, 0, 0], target: layoutId_1, text: 'Mastertitelformat bearbeiten' },
                { name: 'insertDrawing', start: [0, 1], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'subTitle', phIndex: 1 }, drawing: { name: 'Untertitel 2', left: subTitleLeft, top: subTitleTop, width: subTitleWidth, height: subTitleHeight } } },
                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 1, 0, 0], target: layoutId_1, text: 'Master-Untertitelformat bearbeiten' },

                { name: 'insertLayoutSlide', id: layoutId_2, target: masterId_1 },
                { name: 'insertDrawing', start: [0, 0], target: layoutId_2, type: 'shape', attrs: { presentation: { phType: 'title' }, drawing: { name: 'Titel 1' } } },
                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_2 },
                { name: 'insertText', start: [0, 0, 0, 0], target: layoutId_2, text: 'Mastertitelformat bearbeiten' },
                { name: 'insertDrawing', start: [0, 1], target: layoutId_2, type: 'shape', attrs: { presentation: { phIndex: 1 }, drawing: { name: 'Drawing 2', left: 1270, top: 4445, width: 11218, height: 12572 } } },
                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_2, attrs: { paragraph: { level: 0 } } },
                { name: 'insertText', start: [0, 1, 0, 0], target: layoutId_2, text: 'Mastertextformat bearbeiten' },
                { name: 'insertDrawing', start: [0, 2], target: layoutId_2, type: 'shape', attrs: { presentation: { phIndex: 2 }, drawing: { name: 'Drawing 3', left: 12912, top: 4445, width: 11218, height: 12572 } } },
                { name: 'insertParagraph', start: [0, 2, 0], target: layoutId_2, attrs: { paragraph: { level: 0 } } },
                { name: 'insertText', start: [0, 2, 0, 0], target: layoutId_2, text: 'Mastertextformat·bearbeiten' },

                { name: 'insertSlide', start: [0], target: layoutId_1 },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Titel 1' } } },
                { name: 'insertParagraph', start: [0, 0, 0] },
                { name: 'insertText', start: [0, 0, 0, 0], text: 'Hello paragraph 1 in drawing 1' },
                { name: 'insertParagraph', start: [0, 0, 1] },
                { name: 'insertText', start: [0, 0, 1, 0], text: 'Hello paragraph 2 in drawing 1' },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { presentation: { phType: 'subTitle', phIndex: 1 }, drawing: { name: 'Untertitel 2' } } },
                { name: 'insertParagraph', start: [0, 1, 0] },
                { name: 'insertText', start: [0, 1, 0, 0], text: 'Hello paragraph 1 in drawing 2' },
                { name: 'insertDrawing', start: [0, 2], type: 'shape', attrs: { presentation: { phType: 'body', phIndex: 1 }, drawing: { name: 'Text body', left: drawing1Left, top: drawing1Top, width: drawing1Width, height: drawing1Height } } },
                { name: 'insertParagraph', start: [0, 2, 0] },
                { name: 'insertText', start: [0, 2, 0, 0], text: 'Hello paragraph 1 in drawing 3' }
            ];

        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            selection = model.getSelection();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(MultiSelectionMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "isMultiSelection"', function () {
            it('should exist', function () {
                expect(selection).to.respondTo('isMultiSelection');
            });
        });

        describe('method "getMultiSelection"', function () {
            it('should exist', function () {
                expect(selection).to.respondTo('getMultiSelection');
            });
        });

        describe('method "selectAllDrawingsOnSlide"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('selectAllDrawingsOnSlide');
            });

            it('should select all drawings on the slide', function () {

                selection.setTextSelection([0, 0, 0, 0]);  // selecting the first slide
                activeSlide = model.getSlideById(model.getActiveSlideId());
                drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                expect(selection.isMultiSelection()).to.equal(false);

                expect(drawingsOnSlide.length).to.equal(3);

                selection.selectAllDrawingsOnSlide();

                expect(selection.isMultiSelection()).to.equal(true);

                expect(selection.getMultiSelection().length).to.equal(3);

                selection.setTextSelection([0, 0, 0, 0]);  // selecting the text position again -> no more multiple selection

                expect(selection.isMultiSelection()).to.equal(false);

                expect(selection.getMultiSelection().length).to.equal(0);

                selection.selectAllDrawingsOnSlide();  // and switching back to multiple selection

                expect(selection.isMultiSelection()).to.equal(true);

                expect(selection.getMultiSelection().length).to.equal(3);
            });
        });

        describe('method "getListOfPositions"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('getListOfPositions');
            });

            it('should return the list of all start positions of the selected drawings as string', function () {
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,1),(0,2)');
            });
        });

        describe('method "addOneDrawingIntoMultipleSelection"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('addOneDrawingIntoMultipleSelection');
            });

            it('should add one drawing into the list of multiple selections', function () {

                selection.setTextSelection([0, 0, 0, 0]);  // selecting the text position again -> no more multiple selection
                expect(selection.isMultiSelection()).to.equal(false);
                expect(selection.getMultiSelection().length).to.equal(0);

                selection.addOneDrawingIntoMultipleSelection(drawingsOnSlide[0]);
                selection.addOneDrawingIntoMultipleSelection(drawingsOnSlide[2]);

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(2);
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,2)');
            });
        });

        describe('method "removeOneDrawingFromMultipleSelection"', function () {

            it('should exist', function () {
                expect(selection).to.respondTo('removeOneDrawingFromMultipleSelection');
            });

            it('should remove one drawing from the list of multiple selections', function () {

                selection.removeOneDrawingFromMultipleSelection(drawingsOnSlide[0]);
                selection.addOneDrawingIntoMultipleSelection(drawingsOnSlide[1]);

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(2);
                // expect(selection.getListOfPositions()).to.equal('(0,1),(0,2)'); // TODO
            });
        });

        // setting character attributes to all selected drawings
        describe('method "model.setAttribute"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('setAttribute');
            });

            it('should assign character attributes to all selected drawings in a multiple drawing selection', function () {

                var spansInDrawing1 = $(drawingsOnSlide[0]).find('div.p > span'),
                    spansInDrawing2 = $(drawingsOnSlide[1]).find('div.p > span'),
                    spansInDrawing3 = $(drawingsOnSlide[2]).find('div.p > span');

                selection.setTextSelection([0, 0, 0, 0]);  // selecting the text position again -> no more multiple selection
                expect(selection.isMultiSelection()).to.equal(false);
                expect(selection.getMultiSelection().length).to.equal(0);

                expect(spansInDrawing1.length).to.equal(2);
                expect(spansInDrawing2.length).to.equal(1);
                expect(spansInDrawing3.length).to.equal(1);

                expect($(spansInDrawing1[0]).data('attributes')).to.equal(undefined);
                expect($(spansInDrawing1[1]).data('attributes')).to.equal(undefined);
                expect($(spansInDrawing2[0]).data('attributes')).to.equal(undefined);
                expect($(spansInDrawing3[0]).data('attributes')).to.equal(undefined);

                selection.selectAllDrawingsOnSlide();
                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(3);

                model.setAttribute('character', 'bold', true); // setting character attribute 'bold' to all selected drawings

                expect($(spansInDrawing1[0]).data('attributes').character.bold).to.equal(true);
                expect($(spansInDrawing1[1]).data('attributes').character.bold).to.equal(true);
                expect($(spansInDrawing2[0]).data('attributes').character.bold).to.equal(true);
                expect($(spansInDrawing3[0]).data('attributes').character.bold).to.equal(true);
            });

        });

        // delete all selected drawings
        describe('method "model.deleteSelected"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('deleteSelected');
            });

            it('should delete all selected drawings in a multiple drawing selection', function (done) {

                selection.setTextSelection([0, 0, 0, 0]);  // selecting the text position again -> no more multiple selection
                expect(selection.isMultiSelection()).to.equal(false);
                expect(selection.getMultiSelection().length).to.equal(0);

                selection.selectAllDrawingsOnSlide();

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(3);

                model.deleteSelected({ deleteKey: true }); // deleting all selected drawings with 'delete' or 'backspace'

                activeSlide = model.getSlideById(model.getActiveSlideId());
                drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                expect(selection.isMultiSelection()).to.equal(false);
                expect(drawingsOnSlide.length).to.equal(0);

                // restoring the drawings again
                var promise = model.getUndoManager().undo();
                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                    selection.selectAllDrawingsOnSlide();

                    expect(drawingsOnSlide.length).to.equal(3);
                    expect(selection.isMultiSelection()).to.equal(true);
                    expect(selection.getMultiSelection().length).to.equal(3);

                    done();
                });
            });
        });

        // moving all selected drawings
        describe('method "model.handleDrawingOperations"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('handleDrawingOperations');
            });

            it('should move all selected drawings in a multiple drawing selection', function () {

                selection.setTextSelection([0, 0, 0, 0]);  // selecting the text position again -> no more multiple selection
                expect(selection.isMultiSelection()).to.equal(false);
                expect(selection.getMultiSelection().length).to.equal(0);

                selection.selectAllDrawingsOnSlide();

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(3);

                expect(drawingsOnSlide.length).to.equal(3);

                // reading the drawing attributes for 'left' -> no explicit attributes set at drawing node (excepot for one drawing)
                expect($(drawingsOnSlide[0]).data('attributes').drawing.left).to.equal(undefined);
                expect($(drawingsOnSlide[1]).data('attributes').drawing.left).to.equal(undefined);
                expect($(drawingsOnSlide[2]).data('attributes').drawing.left).to.equal(drawing1Left);

                model.handleDrawingOperations({ ctrlKey: false, shiftKey: false, metaKey: false, keyCode: 39 }); // cursor right

                // reading the drawing attributes for 'left' from the explicit drawing attributes
                expect($(drawingsOnSlide[0]).data('attributes').drawing.left).to.equal(ctrTitleLeft + drawingDelta); // new values calculated with merged attributes from 'ctrTitle'
                expect($(drawingsOnSlide[1]).data('attributes').drawing.left).to.equal(subTitleLeft + drawingDelta); // new values calculated with merged attributes from 'subTitle'
                expect($(drawingsOnSlide[2]).data('attributes').drawing.left).to.equal(drawing1Left + drawingDelta);

                model.handleDrawingOperations({ ctrlKey: false, shiftKey: false, metaKey: false, keyCode: 39 }); // cursor right

                expect($(drawingsOnSlide[0]).data('attributes').drawing.left).to.equal(ctrTitleLeft + 2 * drawingDelta); // reading explicit attributes at drawing
                expect($(drawingsOnSlide[1]).data('attributes').drawing.left).to.equal(subTitleLeft + 2 * drawingDelta); // reading explicit attributes at drawing
                expect($(drawingsOnSlide[2]).data('attributes').drawing.left).to.equal(drawing1Left + 2 * drawingDelta);

                // double velocity with 'ctrl'-key pressed
                model.handleDrawingOperations({ ctrlKey: true, shiftKey: false, metaKey: false, keyCode: 39 }); // cursor right + ctrl key

                expect($(drawingsOnSlide[0]).data('attributes').drawing.left).to.equal(ctrTitleLeft + 4 * drawingDelta); // reading explicit attributes at drawing
                expect($(drawingsOnSlide[1]).data('attributes').drawing.left).to.equal(subTitleLeft + 4 * drawingDelta); // reading explicit attributes at drawing
                expect($(drawingsOnSlide[2]).data('attributes').drawing.left).to.equal(drawing1Left + 4 * drawingDelta);

                expect(selection.isMultiSelection()).to.equal(true);
                expect(drawingsOnSlide.length).to.equal(3);
            });

            it('should resize all selected drawings in a multiple drawing selection', function () {

                selection.setTextSelection([0, 0, 0, 0]);  // selecting the text position again -> no more multiple selection
                expect(selection.isMultiSelection()).to.equal(false);
                expect(selection.getMultiSelection().length).to.equal(0);

                selection.selectAllDrawingsOnSlide();

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(3);

                expect(drawingsOnSlide.length).to.equal(3);

                // reading the drawing attributes for 'top' and 'height' -> no explicit attributes set at drawing node (except for one drawing)
                expect($(drawingsOnSlide[0]).data('attributes').drawing.top).to.equal(undefined);
                expect($(drawingsOnSlide[1]).data('attributes').drawing.top).to.equal(undefined);
                expect($(drawingsOnSlide[2]).data('attributes').drawing.top).to.equal(drawing1Top);

                model.handleDrawingOperations({ ctrlKey: false, shiftKey: true, metaKey: false, keyCode: 38 }); // cursor top + shift key

                // reading the drawing attributes for 'top' and 'height' from the explicit drawing attributes
                expect($(drawingsOnSlide[0]).data('attributes').drawing.top).to.equal(ctrTitleTop - 0.5 * drawingSizePercentage * ctrTitleHeight); // half of 5% of ctrTitleHeight
                expect($(drawingsOnSlide[0]).data('attributes').drawing.height).to.equal(ctrTitleHeight + drawingSizePercentage * ctrTitleHeight); // 5% of ctrTitleHeight
                expect($(drawingsOnSlide[1]).data('attributes').drawing.top).to.equal(subTitleTop - 0.5 * drawingSizePercentage * subTitleHeight); // half of 5% of subTitleHeight
                expect($(drawingsOnSlide[1]).data('attributes').drawing.height).to.equal(subTitleHeight + drawingSizePercentage * subTitleHeight); // 5% of subTitleHeight
                expect($(drawingsOnSlide[2]).data('attributes').drawing.top).to.equal(drawing1Top - 0.5 * drawingSizePercentage * drawing1Height); // half of 5% of drawing1Height
                expect($(drawingsOnSlide[2]).data('attributes').drawing.height).to.equal(drawing1Height + drawingSizePercentage * drawing1Height); // 5% of drawing1Height

                // refresh global variables for following test
                ctrTitleLeft = $(drawingsOnSlide[0]).data('attributes').drawing.left;
                subTitleLeft = $(drawingsOnSlide[1]).data('attributes').drawing.left;
                drawing1Left = $(drawingsOnSlide[2]).data('attributes').drawing.left;

                model.handleDrawingOperations({ ctrlKey: false, shiftKey: true, metaKey: false, keyCode: 39 }); // cursor right + shift key

                // reading the drawing attributes for 'left' and 'width' from the explicit drawing attributes
                expect($(drawingsOnSlide[0]).data('attributes').drawing.left).to.equal(ctrTitleLeft - 0.5 * drawingSizePercentage * ctrTitleWidth); // half of 5% of ctrTitleWidth
                expect($(drawingsOnSlide[0]).data('attributes').drawing.width).to.equal(ctrTitleWidth + drawingSizePercentage * ctrTitleWidth); // 5% of ctrTitleWidth
                expect($(drawingsOnSlide[1]).data('attributes').drawing.left).to.equal(subTitleLeft - 0.5 * drawingSizePercentage * subTitleWidth); // half of 5% of subTitleWidth
                expect($(drawingsOnSlide[1]).data('attributes').drawing.width).to.equal(subTitleWidth + drawingSizePercentage * subTitleWidth); // 5% of subTitleWidth
                expect($(drawingsOnSlide[2]).data('attributes').drawing.left).to.equal(drawing1Left - 0.5 * drawingSizePercentage * drawing1Width); // half of 5% of drawing1Width
                expect($(drawingsOnSlide[2]).data('attributes').drawing.width).to.equal(drawing1Width + drawingSizePercentage * drawing1Width); // 5% of drawing1Width

                expect(selection.isMultiSelection()).to.equal(true);
                expect(drawingsOnSlide.length).to.equal(3);
            });

            it('should rotate all selected drawings in a multiple drawing selection', function () {

                selection.setTextSelection([0, 0, 0, 0]);  // selecting the text position again -> no more multiple selection
                expect(selection.isMultiSelection()).to.equal(false);
                expect(selection.getMultiSelection().length).to.equal(0);

                selection.selectAllDrawingsOnSlide();

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(3);

                expect(drawingsOnSlide.length).to.equal(3);

                expect($(drawingsOnSlide[0]).data('attributes').drawing.rotation).to.equal(undefined);
                expect($(drawingsOnSlide[1]).data('attributes').drawing.rotation).to.equal(undefined);
                expect($(drawingsOnSlide[2]).data('attributes').drawing.rotation).to.equal(undefined);

                model.handleDrawingOperations({ ctrlKey: false, shiftKey: false, metaKey: false, altKey: true, keyCode: 39 }); // alt + right

                // reading the drawing attributes for 'rotate' from the explicit drawing attributes
                expect($(drawingsOnSlide[0]).data('attributes').drawing.rotation).to.equal(15);
                expect($(drawingsOnSlide[1]).data('attributes').drawing.rotation).to.equal(15);
                expect($(drawingsOnSlide[2]).data('attributes').drawing.rotation).to.equal(15);

                model.handleDrawingOperations({ ctrlKey: false, shiftKey: false, metaKey: false, altKey: true, keyCode: 39 }); // alt + right

                // test also the getter function for rotation angle
                expect(DrawingFrame.getDrawingRotationAngle(model, drawingsOnSlide[0])).to.equal(30);
                expect(DrawingFrame.getDrawingRotationAngle(model, drawingsOnSlide[1])).to.equal(30);
                expect(DrawingFrame.getDrawingRotationAngle(model, drawingsOnSlide[2])).to.equal(30);

                model.handleDrawingOperations({ ctrlKey: false, shiftKey: false, metaKey: false, altKey: true, keyCode: 37 }); // alt + left

                // reading the drawing attributes for 'rotate' from the explicit drawing attributes
                expect($(drawingsOnSlide[0]).data('attributes').drawing.rotation).to.equal(15);
                expect($(drawingsOnSlide[1]).data('attributes').drawing.rotation).to.equal(15);
                expect($(drawingsOnSlide[2]).data('attributes').drawing.rotation).to.equal(15);

                model.handleDrawingOperations({ ctrlKey: false, shiftKey: false, metaKey: false, altKey: true, keyCode: 37 }); // alt + left

                // test also the getter function for rotation angle
                expect(DrawingFrame.getDrawingRotationAngle(model, drawingsOnSlide[0])).to.equal(null);
                expect(DrawingFrame.getDrawingRotationAngle(model, drawingsOnSlide[1])).to.equal(null);
                expect(DrawingFrame.getDrawingRotationAngle(model, drawingsOnSlide[2])).to.equal(null);

                expect(selection.isMultiSelection()).to.equal(true);
                expect(drawingsOnSlide.length).to.equal(3);
            });

        });

    });

    // ========================================================================
});
