/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/utils/paragraphspacing',
    'io.ox/office/textframework/model/attributeoperationmixin',
    'io.ox/office/textframework/utils/position'
], function (AppHelper, Utils, LineHeight, ParagraphSpacing, AttributeOperationMixin, Position) {

    'use strict';

    // class AttributeOperationMixin ==========================================

    describe('TextFramework class AttributeOperationMixin', function () {

        var LINEHEIGHT__NORMAL       = LineHeight.NORMAL,
            LINEHEIGHT__SINGLE       = LineHeight.SINGLE,
            LINEHEIGHT___115         = LineHeight._115,
            LINEHEIGHT__ONE_HALF     = LineHeight.ONE_HALF,
            LINEHEIGHT__DOUBLE       = LineHeight.DOUBLE,

            PARAGRAPHSPACING__NONE   = ParagraphSpacing.setting.multiplier.NONE,
            PARAGRAPHSPACING__NORMAL = ParagraphSpacing.setting.multiplier.NORMAL,
            PARAGRAPHSPACING__WIDE   = ParagraphSpacing.setting.multiplier.WIDE,

            FALLBACK__MARGIN_BOTTOM  = ParagraphSpacing.setting.fallbackValue.NORMAL,

            defaultMarginBottom;

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes',
                attrs: {
                    document:  { defaultTabStop: 1270, zoom: { value: 100 } },
                    page:      { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 },
                    character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' },
                    paragraph: { lineHeight: { type: 'percent', value: 115 }, marginBottom: 352 }
                }
            },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading1', styleName: 'heading·1',
                attrs: {
                    character: { bold: true, fontName: 'Times·New·Roman', fontSize: 14, color: { transformations: [{ type: 'shade', value: 74902 }], type: 'scheme', value: 'accent1' } },
                    paragraph: { marginTop: 846, outlineLevel: 0, nextStyleId: 'Normal' }
                },
                parent: 'Normal',
                uiPriority: 9
            },
            { name: 'insertParagraph', start: [0] },
            { name: 'insertText', text: 'Hello World', start: [0, 0] },

            { name: 'splitParagraph', start: [0, 11] },
            { name: 'insertText', start: [1, 0], text: [
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet.'
            ].join('') },

            { name: 'splitParagraph', start: [1, 240] },
            { name: 'insertText', start: [2, 0], text: [
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet.'
            ].join('') },

            { name: 'splitParagraph', start: [2, 240] },
            { name: 'insertText', start: [3, 0], text: [
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet',
                'lorem ipsum dolor sit amet lorem ipsum dolor sit amet lorem ipsum dolor sit amet.'
            ].join('') },

            { name: 'splitParagraph', start: [3, 240] }
        ];

        var model, firstParagraph, secondParagraph, thirdParagraph, fourthParagraph;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            var node = model.getNode();
            firstParagraph  = Position.getParagraphElement(node, [0]);
            secondParagraph = Position.getParagraphElement(node, [1]);
            thirdParagraph  = Position.getParagraphElement(node, [2]);
            fourthParagraph = Position.getParagraphElement(node, [3]);

            var defaultAttributes = model.getDefaultAttributes('paragraph');
            defaultMarginBottom = defaultAttributes.marginBottom || FALLBACK__MARGIN_BOTTOM;
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(AttributeOperationMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "setAttributes"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('setAttributes');
            });
            it('should set the specified paragraph style', function () {
                var
                    attributes,
                    multiplier;

                function compareEMWithPT(elementEM, comparePT) {
                    var compare = Utils.convertToEMValue(comparePT, 'pt');
                    expect(parseFloat(elementEM, 10)).to.be.closeTo(compare, 2);
                }

                expect($(firstParagraph)).to.have.property('hasClass');
                expect($(firstParagraph).hasClass('p')).to.equal(true);
                compareEMWithPT($(firstParagraph).children().first().css('font-size'), 11);

                // setting styleId 'Heading1' to one and only paragraph
                model.getSelection().setTextSelection([0, 2]); // not necessary
                model.setAttributes('paragraph', { styleId: 'Heading1' }, { clear: true });

                compareEMWithPT($(firstParagraph).children().first().css('font-size'), 14);

                /**
                 *  'Paragraph and LineHeight spacing' test - User Story #99693312:
                 *  https://www.pivotaltracker.com/projects/534975/stories/99693312
                 */
                // setting both the paragraph spacing and the line height, covering the whole selection ...
                //
                // ... that should be equal to the following operations ...
                // { name: 'setAttributes', start: [1], attrs: { paragraph: { marginBottom: 500 }} },
                // { name: 'setAttributes', start: [2], attrs: { paragraph: { marginBottom: 500 }} },
                // { name: 'setAttributes', start: [3], attrs: { paragraph: { marginBottom: 500 }} }

                //
                // 1st COMBINED ROUND (lineHeight + marginBottom)
                //
                model.getSelection().setTextSelection([1, 0], [3, 240]);
                model.setAttribute('paragraph', 'lineHeight', LINEHEIGHT___115);
                model.setAttribute('paragraph', 'marginBottom', (PARAGRAPHSPACING__WIDE * defaultMarginBottom));

                expect(parseInt($(secondParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(115);
                expect(parseInt($(thirdParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(115);
                expect(parseInt($(fourthParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(115);

                model.getSelection().setTextSelection([1, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__WIDE);

                model.getSelection().setTextSelection([2, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__WIDE);

                model.getSelection().setTextSelection([3, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__WIDE);

                //
                // 2nd COMBINED ROUND (lineHeight + marginBottom)
                //
                model.getSelection().setTextSelection([1, 0], [2, 240]);
                model.setAttribute('paragraph', 'lineHeight', LINEHEIGHT__ONE_HALF);
                model.setAttribute('paragraph', 'marginBottom', (PARAGRAPHSPACING__NONE * defaultMarginBottom));

                expect(parseInt($(secondParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(150);
                expect(parseInt($(thirdParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(150);
                expect(parseInt($(fourthParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(115);

                model.getSelection().setTextSelection([1, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__NONE);

                model.getSelection().setTextSelection([2, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__NONE);

                model.getSelection().setTextSelection([3, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__WIDE);

                //
                // 3rd COMBINED ROUND (lineHeight + marginBottom)
                //
                model.getSelection().setTextSelection([2, 0], [3, 240]);
                model.setAttribute('paragraph', 'lineHeight', LINEHEIGHT__SINGLE);
                model.setAttribute('paragraph', 'marginBottom', (PARAGRAPHSPACING__NORMAL * defaultMarginBottom));

                expect(parseInt($(secondParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(150);
                expect(parseInt($(thirdParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(100);
                expect(parseInt($(fourthParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(100);

                model.getSelection().setTextSelection([1, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__NONE);

                model.getSelection().setTextSelection([2, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__NORMAL);

                model.getSelection().setTextSelection([3, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__NORMAL);

                //
                // 4th COMBINED ROUND (lineHeight + marginBottom)
                //
                model.getSelection().setTextSelection([1, 5]);
                model.setAttribute('paragraph', 'lineHeight', LINEHEIGHT__DOUBLE);
                model.setAttribute('paragraph', 'marginBottom', (PARAGRAPHSPACING__WIDE * defaultMarginBottom));

                expect(parseInt($(secondParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(200);
                expect(parseInt($(thirdParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(100);
                expect(parseInt($(fourthParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(100);

                model.getSelection().setTextSelection([1, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__WIDE);

                model.getSelection().setTextSelection([2, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__NORMAL);

                model.getSelection().setTextSelection([3, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__NORMAL);

                //
                // 5th COMBINED ROUND (lineHeight + marginBottom)
                //
                model.getSelection().setTextSelection([1, 5], [2, 12]);
                model.setAttribute('paragraph', 'lineHeight', LINEHEIGHT__NORMAL);
                model.setAttribute('paragraph', 'marginBottom', (PARAGRAPHSPACING__NONE * defaultMarginBottom));

                expect(parseInt($(secondParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(100);
                expect(parseInt($(thirdParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(100);
                expect(parseInt($(fourthParagraph).find('span')[0].style.lineHeight, 10)).to.be.equal(100);

                model.getSelection().setTextSelection([1, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__NONE);

                model.getSelection().setTextSelection([2, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__NONE);

                model.getSelection().setTextSelection([3, 0]);
                attributes = ((model.getAttributes('paragraph', { maxIterations: 10 }) || {}).paragraph || {});
                multiplier = ParagraphSpacing.helper.computeMultiplier(attributes, model);
                expect(multiplier).to.equal(PARAGRAPHSPACING__NORMAL);
                //
                // END OF 'Paragraph and LineHeight spacing' test
            });
        });
    });

    // ========================================================================
});
