/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/addressarray',
    'io.ox/office/spreadsheet/utils/intervalarray',
    'io.ox/office/spreadsheet/utils/rangearray'
], function (SheetHelper, AddressArray, IntervalArray, RangeArray) {

    'use strict';

    // class RangeArray =======================================================

    describe('Spreadsheet class RangeArray', function () {

        it('should exist', function () {
            expect(RangeArray).to.be.a('function');
        });
        it('should be a subclass of Array', function () {
            expect(new RangeArray()).to.be.an.instanceof(Array);
        });

        // private helpers ----------------------------------------------------

        var i = SheetHelper.i;
        var a = SheetHelper.a;
        var r = SheetHelper.r;
        var ia = SheetHelper.ia;
        var aa = SheetHelper.aa;
        var ra = SheetHelper.ra;
        var rangesMatcher = SheetHelper.unorderedRangesMatcher;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a range array', function () {
                var ra1 = new RangeArray();
                expect(ra1).to.be.an.instanceof(RangeArray);
                expect(ra1).to.be.empty;
            });
            var r1 = r('A2:C4'), r2 = r('B3:D5');
            it('should insert single ranges', function () {
                var ra1 = new RangeArray(r1);
                expect(ra1).to.have.length(1);
                expect(ra1[0]).to.equal(r1);
                var ra2 = new RangeArray(r1, r2, r1);
                expect(ra2).to.have.length(3);
                expect(ra2[0]).to.equal(r1);
                expect(ra2[1]).to.equal(r2);
                expect(ra2[2]).to.equal(r1);
            });
            it('should insert plain arrays of ranges', function () {
                var ra1 = new RangeArray([r1]);
                expect(ra1).to.have.length(1);
                expect(ra1[0]).to.equal(r1);
                var ra2 = new RangeArray([r1, r2], [], [r1]);
                expect(ra2).to.have.length(3);
                expect(ra2[0]).to.equal(r1);
                expect(ra2[1]).to.equal(r2);
                expect(ra2[2]).to.equal(r1);
            });
            it('should copy-construct range arrays', function () {
                var ra0 = new RangeArray(r1, r2),
                    ra1 = new RangeArray(ra0);
                expect(ra1).to.have.length(2);
                expect(ra1[0]).to.equal(r1);
                expect(ra1[1]).to.equal(r2);
                var ra2 = new RangeArray(ra0, new RangeArray(), ra1);
                expect(ra2).to.have.length(4);
                expect(ra2[0]).to.equal(r1);
                expect(ra2[1]).to.equal(r2);
                expect(ra2[2]).to.equal(r1);
                expect(ra2[3]).to.equal(r2);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "createFromIntervals"', function () {
            it('should exist', function () {
                expect(RangeArray).itself.to.respondTo('createFromIntervals');
            });
            var ia1 = ia('B:C C:D'), ia2 = ia('3:4 4:5');
            it('should create range addresses from column and row intervals', function () {
                expect(RangeArray.createFromIntervals(ia1, ia2)).to.deep.equal(ra('B3:C4 C3:D4 B4:C5 C4:D5'));
            });
            it('should accept single intervals', function () {
                var i1 = i('B:D'), i2 = i('3:5');
                expect(RangeArray.createFromIntervals(i1, ia2)).to.deep.equal(ra('B3:D4 B4:D5'));
                expect(RangeArray.createFromIntervals(ia1, i2)).to.deep.equal(ra('B3:C5 C3:D5'));
                expect(RangeArray.createFromIntervals(i1, i2)).to.deep.equal(ra('B3:D5'));
            });
            it('should accept empty arrays', function () {
                var ia0 = new IntervalArray();
                expect(RangeArray.createFromIntervals(ia1, ia0)).to.have.length(0);
                expect(RangeArray.createFromIntervals(ia0, ia2)).to.have.length(0);
                expect(RangeArray.createFromIntervals(ia0, ia0)).to.have.length(0);
            });
        });

        describe('method "createFromColIntervals"', function () {
            it('should exist', function () {
                expect(RangeArray).itself.to.respondTo('createFromColIntervals');
            });
            var ia1 = ia('B:C C:D');
            it('should create range addresses from column intervals', function () {
                expect(RangeArray.createFromColIntervals(ia1, 2, 3)).to.deep.equal(ra('B3:C4 C3:D4'));
                expect(RangeArray.createFromColIntervals(ia1, 2)).to.deep.equal(ra('B3:C3 C3:D3'));
            });
        });

        describe('method "createFromRowIntervals"', function () {
            it('should exist', function () {
                expect(RangeArray).itself.to.respondTo('createFromRowIntervals');
            });
            var ia1 = ia('3:4 4:5');
            it('should create range addresses from row intervals', function () {
                expect(RangeArray.createFromRowIntervals(ia1, 1, 2)).to.deep.equal(ra('B3:C4 B4:C5'));
                expect(RangeArray.createFromRowIntervals(ia1, 1)).to.deep.equal(ra('B3:B4 B4:B5'));
            });
        });

        describe('method "mergeAddresses"', function () {
            it('should exist', function () {
                expect(RangeArray).itself.to.respondTo('mergeAddresses');
            });
            it('should merge cell addresses to range', function () {
                expect(RangeArray.mergeAddresses(aa('A1 A3 A4 A2 A1'))).to.deep.equal(ra('A1:A4'));
                expect(RangeArray.mergeAddresses(aa('A1 B2 B1 A2'))).to.deep.equal(ra('A1:B2'));
            });
            it('should accept an empty array', function () {
                expect(RangeArray.mergeAddresses(new AddressArray())).to.be.empty;
            });
        });

        // public methods -----------------------------------------------------

        describe('method "key"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('key');
            });
            it('should return unique key for the array', function () {
                expect(ra('A2:C4 B3:D5').key()).to.equal('0,1:2,3 1,2:3,4');
                expect(ra('').key()).to.equal('');
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('clone');
            });
            it('should return a shallow clone', function () {
                var ra1 = ra('A2:C4 B3:D5'), ra2 = ra1.clone();
                expect(ra2).to.be.an.instanceof(RangeArray);
                expect(ra2).to.not.equal(ra1);
                expect(ra2[0]).to.equal(ra1[0]);
                expect(ra2[1]).to.equal(ra1[1]);
                expect(ra2).to.deep.equal(ra1);
            });
            it('should return a deep clone', function () {
                var ra1 = ra('A2:C4 B3:D5'), ra2 = ra1.clone(true);
                expect(ra2).to.be.an.instanceof(RangeArray);
                expect(ra2).to.not.equal(ra1);
                expect(ra2[0]).to.not.equal(ra1[0]);
                expect(ra2[1]).to.not.equal(ra1[1]);
                expect(ra2).to.deep.equal(ra1);
            });
        });

        describe('method "cells"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('cells');
            });
            it('should count cells in ranges', function () {
                expect(ra('A1:C3 B2:D4').cells()).to.equal(18);
                expect(new RangeArray().cells()).to.equal(0);
            });
        });

        describe('method "containsCol"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('containsCol');
            });
            var ra1 = ra('B4:D6 G9:H10 D6:E7');
            it('should return false for columns outside the ranges', function () {
                expect(ra1.containsCol(0)).to.equal(false);
                expect(ra1.containsCol(5)).to.equal(false);
                expect(ra1.containsCol(8)).to.equal(false);
                expect(ra1.containsCol(9)).to.equal(false);
                expect(ra1.containsCol(10)).to.equal(false);
                expect(ra1.containsCol(11)).to.equal(false);
                expect(ra1.containsCol(12)).to.equal(false);
            });
            it('should return true for columns inside the ranges', function () {
                expect(ra1.containsCol(1)).to.equal(true);
                expect(ra1.containsCol(2)).to.equal(true);
                expect(ra1.containsCol(3)).to.equal(true);
                expect(ra1.containsCol(4)).to.equal(true);
                expect(ra1.containsCol(6)).to.equal(true);
                expect(ra1.containsCol(7)).to.equal(true);
            });
        });

        describe('method "containsRow"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('containsRow');
            });
            var ra1 = ra('B4:D6 G9:H10 D6:E7');
            it('should return false for rows outside the ranges', function () {
                expect(ra1.containsRow(0)).to.equal(false);
                expect(ra1.containsRow(1)).to.equal(false);
                expect(ra1.containsRow(2)).to.equal(false);
                expect(ra1.containsRow(7)).to.equal(false);
                expect(ra1.containsRow(10)).to.equal(false);
                expect(ra1.containsRow(11)).to.equal(false);
                expect(ra1.containsRow(12)).to.equal(false);
            });
            it('should return true for rows inside the ranges', function () {
                expect(ra1.containsRow(3)).to.equal(true);
                expect(ra1.containsRow(4)).to.equal(true);
                expect(ra1.containsRow(5)).to.equal(true);
                expect(ra1.containsRow(6)).to.equal(true);
                expect(ra1.containsRow(8)).to.equal(true);
                expect(ra1.containsRow(9)).to.equal(true);
            });
        });

        describe('method "containsIndex"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('containsIndex');
            });
            var ra1 = ra('C2:C2 E4:E4');
            it('should check column indexes', function () {
                expect(ra1.containsIndex(1, true)).to.equal(false);
                expect(ra1.containsIndex(2, true)).to.equal(true);
                expect(ra1.containsIndex(3, true)).to.equal(false);
                expect(ra1.containsIndex(4, true)).to.equal(true);
                expect(ra1.containsIndex(5, true)).to.equal(false);
            });
            it('should check row indexes', function () {
                expect(ra1.containsIndex(0, false)).to.equal(false);
                expect(ra1.containsIndex(1, false)).to.equal(true);
                expect(ra1.containsIndex(2, false)).to.equal(false);
                expect(ra1.containsIndex(3, false)).to.equal(true);
                expect(ra1.containsIndex(4, false)).to.equal(false);
            });
        });

        describe('method "containsAddress"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('containsAddress');
            });
            var ra1 = ra('B4:C5 C5:D6');
            it('should return false for cells outside the ranges', function () {
                'A3 B3 C3 D3 E3 A4 D4 E4 A5 E5 A6 B6 E6 A7 B7 C7 D7 E7'.split(' ').forEach(function (str) {
                    expect(ra1.containsAddress(a(str))).to.equal(false);
                });
            });
            it('should return true for cells inside the ranges', function () {
                expect(ra1.containsAddress(a('B4'))).to.equal(true);
                expect(ra1.containsAddress(a('C4'))).to.equal(true);
                expect(ra1.containsAddress(a('B5'))).to.equal(true);
                expect(ra1.containsAddress(a('C5'))).to.equal(true);
                expect(ra1.containsAddress(a('D5'))).to.equal(true);
                expect(ra1.containsAddress(a('C6'))).to.equal(true);
                expect(ra1.containsAddress(a('D6'))).to.equal(true);
            });
        });

        describe('method "contains"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('contains');
            });
            var ra1 = ra('A1:B2 B2:C3 C3:D4');
            it('should return false for ranges outside the ranges', function () {
                expect(ra1.contains(ra('C1:D1 B2:C3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 C1:D1 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C3 C1:D1'))).to.equal(false);
            });
            it('should return false for overlapping ranges', function () {
                expect(ra1.contains(ra('A1:B3 B2:C3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:C2 B2:C3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 A2:C3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B1:C3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:D3 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C4 C3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C3 B3:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C3 C2:D4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C3 C3:E4'))).to.equal(false);
                expect(ra1.contains(ra('A1:B2 B2:C3 C3:D5'))).to.equal(false);
            });
            it('should return true for contained ranges', function () {
                expect(ra1.contains(ra('A1:B1 A1:A2 A1:B2'))).to.equal(true);
                expect(ra1.contains(ra('A1:B1 B2:C2 C3:D3'))).to.equal(true);
                expect(ra1.contains(ra1)).to.equal(true);
            });
            it('should accept empty array', function () {
                var ra0 = new RangeArray();
                expect(ra0.contains(ra1)).to.equal(false);
                expect(ra1.contains(ra0)).to.equal(true);
                expect(ra0.contains(ra0)).to.equal(true);
            });
            it('should accept single range address', function () {
                expect(ra1.contains(r('C1:D1'))).to.equal(false);
                expect(ra1.contains(r('A1:B3'))).to.equal(false);
                expect(ra1.contains(r('B2:C3'))).to.equal(true);
            });
        });

        describe('method "overlaps"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('overlaps');
            });
            var ra1 = ra('A1:B2 B2:C3 C3:D4');
            it('should return false for distinct ranges', function () {
                expect(ra1.overlaps(ra('C1:D1 A3:A4 D1:D2 A4:B4'))).to.equal(false);
            });
            it('should return true for overlapping ranges', function () {
                expect(ra1.overlaps(ra('C1:D1 A3:B4 D1:D2'))).to.equal(true);
            });
            it('should accept empty array', function () {
                var ra0 = new RangeArray();
                expect(ra0.overlaps(ra1)).to.equal(false);
                expect(ra1.overlaps(ra0)).to.equal(false);
                expect(ra0.overlaps(ra0)).to.equal(false);
            });
            it('should accept single range address', function () {
                expect(ra1.overlaps(r('C1:D1'))).to.equal(false);
                expect(ra1.overlaps(r('A3:B4'))).to.equal(true);
            });
        });

        describe('method "overlapsSelf"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('overlapsSelf');
            });
            it('should return false for distinct ranges', function () {
                expect(ra('B3:C4 D3:E4 B5:C6 D5:E6').overlapsSelf()).to.equal(false);
            });
            it('should return true for overlapping ranges', function () {
                expect(ra('B3:C4 D3:E4 B5:D6 D5:E6 B3:C4').overlapsSelf()).to.equal(true);
                expect(ra('B3:C4 D3:E4 B5:D6 D5:E6 C4:D5').overlapsSelf()).to.equal(true);
            });
            it('should return false for an empty array', function () {
                expect(new RangeArray().overlapsSelf()).to.equal(false);
            });
        });

        describe('method "findByCol"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('findByCol');
            });
            var ra1 = ra('B4:D6 G9:H10 D6:E7');
            it('should return null for columns outside the ranges', function () {
                expect(ra1.findByCol(0)).to.equal(null);
                expect(ra1.findByCol(5)).to.equal(null);
                expect(ra1.findByCol(8)).to.equal(null);
                expect(ra1.findByCol(9)).to.equal(null);
                expect(ra1.findByCol(10)).to.equal(null);
                expect(ra1.findByCol(11)).to.equal(null);
                expect(ra1.findByCol(12)).to.equal(null);
            });
            it('should return first existing range', function () {
                expect(ra1.findByCol(1)).to.equal(ra1[0]);
                expect(ra1.findByCol(2)).to.equal(ra1[0]);
                expect(ra1.findByCol(3)).to.equal(ra1[0]);
                expect(ra1.findByCol(4)).to.equal(ra1[2]);
                expect(ra1.findByCol(6)).to.equal(ra1[1]);
                expect(ra1.findByCol(7)).to.equal(ra1[1]);
            });
        });

        describe('method "findByRow"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('findByRow');
            });
            var ra1 = ra('B4:D6 G9:H10 D6:E7');
            it('should return null for rows outside the ranges', function () {
                expect(ra1.findByRow(0)).to.equal(null);
                expect(ra1.findByRow(1)).to.equal(null);
                expect(ra1.findByRow(2)).to.equal(null);
                expect(ra1.findByRow(7)).to.equal(null);
                expect(ra1.findByRow(10)).to.equal(null);
                expect(ra1.findByRow(11)).to.equal(null);
                expect(ra1.findByRow(12)).to.equal(null);
            });
            it('should return first existing range', function () {
                expect(ra1.findByRow(3)).to.equal(ra1[0]);
                expect(ra1.findByRow(4)).to.equal(ra1[0]);
                expect(ra1.findByRow(5)).to.equal(ra1[0]);
                expect(ra1.findByRow(6)).to.equal(ra1[2]);
                expect(ra1.findByRow(8)).to.equal(ra1[1]);
                expect(ra1.findByRow(9)).to.equal(ra1[1]);
            });
        });

        describe('method "findByIndex"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('findByIndex');
            });
            var ra1 = ra('C2:C2 E4:E4');
            it('should check column indexes', function () {
                expect(ra1.findByIndex(1, true)).to.equal(null);
                expect(ra1.findByIndex(2, true)).to.equal(ra1[0]);
                expect(ra1.findByIndex(3, true)).to.equal(null);
                expect(ra1.findByIndex(4, true)).to.equal(ra1[1]);
                expect(ra1.findByIndex(5, true)).to.equal(null);
            });
            it('should check row indexes', function () {
                expect(ra1.findByIndex(0, false)).to.equal(null);
                expect(ra1.findByIndex(1, false)).to.equal(ra1[0]);
                expect(ra1.findByIndex(2, false)).to.equal(null);
                expect(ra1.findByIndex(3, false)).to.equal(ra1[1]);
                expect(ra1.findByIndex(4, false)).to.equal(null);
            });
        });

        describe('method "findByAddress"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('findByAddress');
            });
            var ra1 = ra('B4:C5 C5:D6');
            it('should return null for cells outside the ranges', function () {
                'A3 B3 C3 D3 E3 A4 D4 E4 A5 E5 A6 B6 E6 A7 B7 C7 D7 E7'.split(' ').forEach(function (str) {
                    expect(ra1.findByAddress(a(str))).to.equal(null);
                });
            });
            it('should return first existing range', function () {
                expect(ra1.findByAddress(a('B4'))).to.equal(ra1[0]);
                expect(ra1.findByAddress(a('C4'))).to.equal(ra1[0]);
                expect(ra1.findByAddress(a('B5'))).to.equal(ra1[0]);
                expect(ra1.findByAddress(a('C5'))).to.equal(ra1[0]);
                expect(ra1.findByAddress(a('D5'))).to.equal(ra1[1]);
                expect(ra1.findByAddress(a('C6'))).to.equal(ra1[1]);
                expect(ra1.findByAddress(a('D6'))).to.equal(ra1[1]);
            });
        });

        describe('method "colIntervals"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('colIntervals');
            });
            it('should return the column intervals', function () {
                expect(ra('B3:C4 D5:E6 C4:D5').colIntervals()).to.deep.equal(ia('B:C D:E C:D'));
                expect(new RangeArray().colIntervals()).to.be.empty;
            });
        });

        describe('method "rowIntervals"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('rowIntervals');
            });
            it('should return the row intervals', function () {
                expect(ra('B3:C4 D5:E6 C4:D5').rowIntervals()).to.deep.equal(ia('3:4 5:6 4:5'));
                expect(new RangeArray().rowIntervals()).to.be.empty;
            });
        });

        describe('method "intervals"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('intervals');
            });
            it('should return the column intervals', function () {
                expect(ra('B3:C4 D5:E6 C4:D5').intervals(true)).to.deep.equal(ia('B:C D:E C:D'));
            });
            it('should return the row intervals', function () {
                expect(ra('B3:C4 D5:E6 C4:D5').intervals(false)).to.deep.equal(ia('3:4 5:6 4:5'));
            });
        });

        describe('method "boundary"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('boundary');
            });
            it('should return the bounding range', function () {
                expect(ra('B3:C4 C4:D5').boundary()).to.deep.equal(r('B3:D5'));
                expect(ra('C4:D5 B3:C4 A2:B3').boundary()).to.deep.equal(r('A2:D5'));
            });
            it('should return a clone of a single range', function () {
                var ra1 = ra('B3:C4'), r1 = ra1.boundary();
                expect(r1).to.deep.equal(ra1[0]);
                expect(r1).to.not.equal(ra1[0]);
            });
            it('should return null for an empty array', function () {
                expect(new RangeArray().boundary()).to.equal(null);
            });
        });

        describe('method "unify"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('unify');
            });
            it('should return a shallow copy', function () {
                var ra1 = ra('A2:C4 B3:D5'), ra2 = ra1.unify();
                expect(ra2).to.not.equal(ra1);
                expect(ra2[0]).to.equal(ra1[0]);
                expect(ra2[1]).to.equal(ra1[1]);
            });
            it('should remove all duplicates', function () {
                var ra1 = ra('A2:C4 B3:D5 A2:C4 B3:C4 B3:D5 B3:C4'), ra2 = ra1.unify();
                expect(ra2).to.have.length(3);
                expect(ra2[0]).to.equal(ra1[0]);
                expect(ra2[1]).to.equal(ra1[1]);
                expect(ra2[2]).to.equal(ra1[3]);
            });
            it('should accept empty array', function () {
                var ra0 = new RangeArray(), ra1 = ra0.unify();
                expect(ra1).to.be.an.instanceof(RangeArray);
                expect(ra1).to.be.empty;
                expect(ra1).to.not.equal(ra0);
            });
        });

        describe('method "filterCovered"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('filterCovered');
            });
            it('should return a shallow copy', function () {
                var ra1 = ra('A2:C4 B3:D5'), ra2 = ra1.filterCovered();
                expect(ra2).to.not.equal(ra1);
                expect(ra2[0]).to.equal(ra1[0]);
                expect(ra2[1]).to.equal(ra1[1]);
            });
            it('should remove all covered ranges', function () {
                var ra1 = ra('B2:D4 D4:F6 D4:E5 A1:A1 A1:B2 B2:D4'), ra2 = ra1.filterCovered();
                expect(ra2).to.have.length(3);
                expect(ra2[0]).to.equal(ra1[0]);
                expect(ra2[1]).to.equal(ra1[1]);
                expect(ra2[2]).to.equal(ra1[4]);
            });
            it('should accept an empty array', function () {
                var ra0 = new RangeArray(), ra1 = ra0.filterCovered();
                expect(ra1).to.be.an.instanceof(RangeArray);
                expect(ra1).to.be.empty;
                expect(ra1).to.not.equal(ra0);
            });
        });

        describe('method "getColBands"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('getColBands');
            });
            it('should return the band intervals', function () {
                expect(ra('A2:C4 B3:D5').getColBands()).to.deep.equal(ia('A:A B:C D:D'));
                expect(ra('A2:D5 B3:C4').getColBands()).to.deep.equal(ia('A:A B:C D:D'));
            });
            it('should return the original ranges', function () {
                var colBands = ra('A2:C4 B3:D5').getColBands({ ranges: true });
                expect(colBands).to.be.an.instanceof(IntervalArray);
                colBands.forEach(function (colBand) {
                    expect(colBand).to.have.a.property('ranges').that.is.an.instanceof(RangeArray);
                    expect(colBand).to.not.have.a.property('intervals');
                });
            });
            it('should return the merged row intervals', function () {
                var colBands = ra('A2:C4 B3:D5').getColBands({ intervals: true });
                expect(colBands).to.be.an.instanceof(IntervalArray);
                colBands.forEach(function (colBand) {
                    expect(colBand).to.not.have.a.property('ranges');
                    expect(colBand).to.have.a.property('intervals').that.is.an.instanceof(IntervalArray);
                });
            });
            it('should return the original ranges and merged row intervals', function () {
                var colBands = ra('A2:C4 B3:D5').getColBands({ ranges: true, intervals: true });
                expect(colBands).to.be.an.instanceof(IntervalArray);
                colBands.forEach(function (colBand) {
                    expect(colBand).to.have.a.property('ranges').that.is.an.instanceof(RangeArray);
                    expect(colBand).to.have.a.property('intervals').that.is.an.instanceof(IntervalArray);
                });
            });
        });

        describe('method "getRowBands"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('getRowBands');
            });
            it('should return the band intervals', function () {
                expect(ra('A2:C4 B3:D5').getRowBands()).to.deep.equal(ia('2:2 3:4 5:5'));
                expect(ra('A2:D5 B3:C4').getRowBands()).to.deep.equal(ia('2:2 3:4 5:5'));
            });
            it('should return the original ranges', function () {
                var rowBands = ra('A2:C4 B3:D5').getRowBands({ ranges: true });
                expect(rowBands).to.be.an.instanceof(IntervalArray);
                rowBands.forEach(function (rowBand) {
                    expect(rowBand).to.have.a.property('ranges').that.is.an.instanceof(RangeArray);
                    expect(rowBand).to.not.have.a.property('intervals');
                });
            });
            it('should return the merged column intervals', function () {
                var rowBands = ra('A2:C4 B3:D5').getRowBands({ intervals: true });
                expect(rowBands).to.be.an.instanceof(IntervalArray);
                rowBands.forEach(function (rowBand) {
                    expect(rowBand).to.not.have.a.property('ranges');
                    expect(rowBand).to.have.a.property('intervals').that.is.an.instanceof(IntervalArray);
                });
            });
            it('should return the original ranges and merged column intervals', function () {
                var rowBands = ra('A2:C4 B3:D5').getRowBands({ ranges: true, intervals: true });
                expect(rowBands).to.be.an.instanceof(IntervalArray);
                rowBands.forEach(function (rowBand) {
                    expect(rowBand).to.have.a.property('ranges').that.is.an.instanceof(RangeArray);
                    expect(rowBand).to.have.a.property('intervals').that.is.an.instanceof(IntervalArray);
                });
            });
        });

        describe('method "merge"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('merge');
            });
            it('should not merge distinct ranges', function () {
                expect(ra('C6:D7 B3:C4 E4:F5').merge()).to.satisfy(rangesMatcher('B3:C4 E4:F5 C6:D7'));
            });
            it('should merge adjacent ranges', function () {
                expect(ra('B5:C6 B3:C4').merge()).to.deep.equal(ra('B3:C6'));
                expect(ra('D3:E4 B3:C4').merge()).to.deep.equal(ra('B3:E4'));
                expect(ra('D3:E4 B5:C6 B3:C4').merge()).to.satisfy(rangesMatcher('B3:E4 B5:C6'));
                expect(ra('D3:E4 B5:C6 B3:C4 D5:E6').merge()).to.deep.equal(ra('B3:E6'));
                expect(ra('B3:B5 B6:D6 E4:E6 C3:E3 C4:D5').merge()).to.deep.equal(ra('B3:E6'));
            });
            it('should merge overlapping ranges', function () {
                expect(ra('B4:C6 B3:C5').merge()).to.deep.equal(ra('B3:C6'));
                expect(ra('C3:E4 B3:D4').merge()).to.deep.equal(ra('B3:E4'));
                expect(ra('C3:E5 E5:G7').merge()).to.satisfy(rangesMatcher('C3:E4 C5:G5 E6:G7'));
                expect(ra('C3:E4 B4:C6 B3:D4').merge()).to.satisfy(rangesMatcher('B3:E4 B5:C6'));
                expect(ra('C3:E4 B5:D6 B3:D4 D4:E6').merge()).to.deep.equal(ra('B3:E6'));
                expect(ra('B3:B6 B6:E6 E3:E6 B3:E3 C3:C5 C4:E4 D4:D6 B5:D5').merge()).to.deep.equal(ra('B3:E6'));
            });
            it('should split ranges to prevent covering a cell multiple times', function () {
                expect(ra('C3:C5 B3:D3').merge()).to.satisfy(rangesMatcher('B3:D3 C4:C5'));
                expect(ra('C3:C5 B4:D4').merge()).to.satisfy(rangesMatcher('C3:C3 B4:D4 C5:C5'));
                expect(ra('C3:C5 B5:D5').merge()).to.satisfy(rangesMatcher('C3:C4 B5:D5'));
            });
            it('should accept empty array', function () {
                var ra1 = new RangeArray(), ra2 = ra1.merge();
                expect(ra2).to.be.empty;
                expect(ra2).to.not.equal(ra1);
            });
        });

        describe('method "partition"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('partition');
            });
            it('should not partition distinct ranges', function () {
                expect(ra('C6:D7 B3:C4 E4:F5').partition()).to.satisfy(rangesMatcher('B3:C4 E4:F5 C6:D7'));
            });
            it('should not partition adjacent ranges', function () {
                expect(ra('B5:C6 B3:C4').partition()).to.satisfy(rangesMatcher('B3:C4 B5:C6'));
                expect(ra('C5:D6 B3:C4').partition()).to.satisfy(rangesMatcher('B3:C4 C5:D6'));
                expect(ra('D3:E4 B3:C4').partition()).to.satisfy(rangesMatcher('B3:C4 D3:E4'));
                expect(ra('D4:E5 B3:C4').partition()).to.satisfy(rangesMatcher('B3:C4 D4:E5'));
                expect(ra('D3:E4 B5:C6 B3:C4').partition()).to.satisfy(rangesMatcher('B3:C4 D3:E4 B5:C6'));
                expect(ra('D3:E4 B5:C6 B3:C4 D5:E6').partition()).to.satisfy(rangesMatcher('B3:C4 D3:E4 B5:C6 D5:E6'));
            });
            it('should partition overlapping ranges', function () {
                expect(ra('B3:C5 B4:C6').partition()).to.satisfy(rangesMatcher('B3:C3 B4:C5 B6:C6'));
                expect(ra('B4:C6 B3:C5').partition()).to.satisfy(rangesMatcher('B3:C3 B4:C5 B6:C6'));
                expect(ra('B3:D4 C3:E4').partition()).to.satisfy(rangesMatcher('B3:B4 C3:D4 E3:E4'));
                expect(ra('C3:E4 B3:D4').partition()).to.satisfy(rangesMatcher('B3:B4 C3:D4 E3:E4'));
                expect(ra('C3:E5 E5:G7').partition()).to.satisfy(rangesMatcher('C3:E4 C5:D5 E5:E5 F5:G5 E6:G7'));
                expect(ra('E5:G7 C3:E5').partition()).to.satisfy(rangesMatcher('C3:E4 C5:D5 E5:E5 F5:G5 E6:G7'));
                expect(ra('C3:E5 D4:D4').partition()).to.satisfy(rangesMatcher('C3:E3 C4:C4 D4:D4 E4:E4 C5:E5'));
                expect(ra('D4:D4 C3:E5').partition()).to.satisfy(rangesMatcher('C3:E3 C4:C4 D4:D4 E4:E4 C5:E5'));
                expect(ra('B3:C3 B4:C4 B3:B4 C3:C4').partition()).to.satisfy(rangesMatcher('B3:B3 C3:C3 B4:B4 C4:C4'));
                expect(ra('B3:E3 B6:E6 B3:B6 E3:E6').partition()).to.satisfy(rangesMatcher('B3:B3 C3:D3 E3:E3 B4:B5 E4:E5 B6:B6 C6:D6 E6:E6'));
            });
            it('should contain the original ranges', function () {
                var ranges = ra('C3:E5 E5:G7');
                ranges.partition().forEach(function (partRange) {
                    expect(partRange).to.have.a.property('coveredBy').that.is.an.instanceof(RangeArray);
                    ranges.forEach(function (range) {
                        var index = partRange.coveredBy.indexOf(range);
                        if (range.contains(partRange)) {
                            expect(index).to.be.at.least(0);
                        } else {
                            expect(index).to.be.below(0);
                        }
                    });
                });
            });
            it('should clone all ranges', function () {
                var r1 = r('C3:E5'), ranges = new RangeArray(r1).partition();
                expect(ranges).to.satisfy(rangesMatcher('C3:E5'));
                expect(ranges[0]).to.not.equal(r1);
            });
            it('should accept empty array', function () {
                var ra1 = new RangeArray(), ra2 = ra1.partition();
                expect(ra2).to.be.empty;
                expect(ra2).to.not.equal(ra1);
            });
        });

        describe('method "intersect"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('intersect');
            });
            var ra1 = ra('B3:D5 D5:F7');
            it('should return empty array for distinct ranges', function () {
                expect(ra1.intersect(ra('E3:F4 B6:C7'))).to.be.empty;
            });
            it('should return intersection ranges', function () {
                expect(ra1.intersect(ra1)).to.deep.equal(ra('B3:D5 D5:D5 D5:D5 D5:F7'));
                expect(ra1.intersect(ra('B3:D5'))).to.deep.equal(ra('B3:D5 D5:D5'));
                expect(ra1.intersect(ra('D5:E6'))).to.deep.equal(ra('D5:D5 D5:E6'));
                expect(ra1.intersect(ra('A1:C9'))).to.deep.equal(ra('B3:C5'));
                expect(ra1.intersect(ra('A5:I6'))).to.deep.equal(ra('B5:D5 D5:F6'));
                expect(ra1.intersect(ra('D1:D9 A5:I5'))).to.deep.equal(ra('D3:D5 B5:D5 D5:D7 D5:F5'));
            });
            it('should accept empty arrays', function () {
                var ra0 = new RangeArray();
                expect(ra0.intersect(ra1)).to.be.empty;
                expect(ra1.intersect(ra0)).to.be.empty;
                expect(ra0.intersect(ra0)).to.be.empty;
            });
        });

        describe('method "difference"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('difference');
            });
            var ra1 = ra('B3:D5 D5:F7');
            it('should return remaining ranges', function () {
                expect(ra1.difference(ra('E3:F4 B6:C7'))).to.deep.equal(ra1);
                expect(ra1.difference(ra('E3:F4 B6:C7'))).to.not.equal(ra1);
                expect(ra1.difference(ra('D3:F5 B5:D7'))).to.deep.equal(ra('B3:C4 E6:F7'));
                expect(ra1.difference(ra('C4:E6'))).to.deep.equal(ra('B3:D3 B4:B5 F5:F6 D7:F7'));
                expect(ra1.difference(ra('C4:C4 E6:E6'))).to.deep.equal(ra('B3:D3 B4:B4 D4:D4 B5:D5 D5:F5 D6:D6 F6:F6 D7:F7'));
                expect(ra1.difference(ra('C4:C4 E6:E6 D5:D5'))).to.deep.equal(ra('B3:D3 B4:B4 D4:D4 B5:C5 E5:F5 D6:D6 F6:F6 D7:F7'));
                expect(ra1.difference(ra('B3:F7'))).to.be.empty;
            });
            it('should accept empty array', function () {
                var ra0 = new RangeArray();
                expect(ra1.difference(ra0)).to.deep.equal(ra1);
                expect(ra1.difference(ra0)).to.not.equal(ra1);
                expect(ra0.difference(ra1)).to.be.empty;
                expect(ra0.difference(ra0)).to.be.empty;
            });
            it('should clone all ranges', function () {
                var r1 = r('C3:E5'), ranges = new RangeArray(r1).difference(new RangeArray());
                expect(ranges[0]).to.deep.equal(r1);
                expect(ranges[0]).to.not.equal(r1);
            });
            it('should accept single range object instead of array', function () {
                expect(ra('B3:D5 D5:F7').difference(r('C4:E6'))).to.deep.equal(ra('B3:D3 B4:B5 F5:F6 D7:F7'));
            });
        });

        describe('method "addressIterator"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('addressIterator');
            });
            it('should visit the addresses in a range array', function () {
                var ra1 = ra('B3:C4 C4:D5'), it = ra1.addressIterator();
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: a('B3'), range: ra1[0], index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: a('C3'), range: ra1[0], index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: a('B4'), range: ra1[0], index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: a('C4'), range: ra1[0], index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: a('C4'), range: ra1[1], index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: a('D4'), range: ra1[1], index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: a('C5'), range: ra1[1], index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: a('D5'), range: ra1[1], index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit the addresses in a range array in vertical reversed order', function () {
                var ra1 = ra('B3:C4 C4:D5 D5:E6'), it = ra1.addressIterator({ reverse: true, columns: true, begin: 1 });
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: a('D5'), range: ra1[1], index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: a('D4'), range: ra1[1], index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: a('C5'), range: ra1[1], index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: a('C4'), range: ra1[1], index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: a('C4'), range: ra1[0], index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: a('C3'), range: ra1[0], index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: a('B4'), range: ra1[0], index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: a('B3'), range: ra1[0], index: 0 });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('toString');
            });
            var ra1 = ra('A2:C4 B3:D5');
            it('should stringify the ranges', function () {
                expect(ra1.toString()).to.equal('A2:C4,B3:D5');
            });
            it('should use the separator string', function () {
                expect(ra1.toString(' + ')).to.equal('A2:C4 + B3:D5');
            });
            it('should stringify implicitly', function () {
                expect('<' + ra1 + '>').to.equal('<A2:C4,B3:D5>');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(RangeArray).to.respondTo('toJSON');
            });
            var ra1 = ra('A2:C4 B3:D5'), ra2 = [{ start: [0, 1], end: [2, 3] }, { start: [1, 2], end: [3, 4] }];
            it('should convert to JSON data', function () {
                expect(ra1.toJSON()).to.deep.equal(ra2);
            });
            it('should stringify to JSON', function () {
                expect(JSON.parse(JSON.stringify(ra1))).to.deep.equal(ra2);
            });
        });
    });

    // ========================================================================
});
