/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/editframework/utils/operationerror',
    'io.ox/office/editframework/utils/operationcontext',
    'io.ox/office/spreadsheet/utils/operationcontext'
], function (AppHelper, OperationError, OperationContext, SheetOperationContext) {

    'use strict';

    // class SheetOperationContext ============================================

    describe('Spreadsheet class SheetOperationContext', function () {

        it('should exist', function () {
            expect(SheetOperationContext).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // base properties for the test operation that are not convertible to addresses, ranges, etc.
        var operation = { name: 'test', num: 1.5, obj: { a: 1 }, arr: [1] };

        // initialize test document
        var docModel = null, intContext = null, addrContext = null, rangeContext = null, arrayContext = null;
        AppHelper.createSpreadsheetApp('ooxml').done(function (app) {
            docModel = app.getModel();

            // test context for column/row intervals
            intContext = new SheetOperationContext(docModel, _.extend({
                interval1: { start: 1, end: 999999 },
                interval2: { start: 1 },
                interval3: { start: 'a' },
                interval4: { start: Number.NaN },
                start: 2
            }, operation), false);
            intContext.bindGet = function (name, columns) { return this.getInterval.bind(this, name, columns); };
            intContext.bindOpt = function (name, columns) { return this.getOptInterval.bind(this, name, columns); };

            // test context for cell addresses
            addrContext = new SheetOperationContext(docModel, _.extend({
                address1: [1, 2],
                address2: [999999, 2],
                address3: [null, 2],
                start: [2, 3]
            }, operation), false);
            addrContext.bindGet = function (name) { return this.getAddress.bind(this, name); };
            addrContext.bindOpt = function (name) { return this.getOptAddress.bind(this, name); };

            // test context for cell range addresses
            rangeContext = new SheetOperationContext(docModel, _.extend({
                range1: { start: [1, 2], end: [3, 4] },
                range2: { start: [1, 2] },
                range3: { start: [1, 2], end: [3, null] },
                start: [2, 3]
            }, operation), false);
            rangeContext.bindGet = function (name) { return this.getRange.bind(this, name); };
            rangeContext.bindOpt = function (name) { return this.getOptRange.bind(this, name); };

            // test context for cell range address arrays
            arrayContext = new SheetOperationContext(docModel, _.extend({
                ranges: [{ start: [1, 2] }],
                ranges2: [{ start: [1, 2] }, { start: [2, 3], end: [3, 4] }],
                ranges3: [{ start: [2, 3], end: [3, null] }],
                ranges4: []
            }, operation), false);
            arrayContext.bindGet = function (name) { return this.getRangeArray.bind(this, name); };
            arrayContext.bindOpt = function (name) { return this.getOptRangeArray.bind(this, name); };
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a context', function () {
                var context = new SheetOperationContext(docModel, operation, false);
                expect(context).to.be.an.instanceof(SheetOperationContext);
                expect(context).to.be.an.instanceof(OperationContext);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "getInterval"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getInterval');
                expect(OperationContext).to.not.respondTo('getInterval');
            });
            it('should return existing properties', function () {
                expect(intContext.getInterval('interval1', false)).to.stringifyTo('2:1000000');
                expect(intContext.getInterval('interval2', false)).to.stringifyTo('2:2');
                expect(intContext.getInterval('interval2', true)).to.stringifyTo('2:2');
                expect(intContext.getInterval(true)).to.stringifyTo('3:3');
            });
            it('should throw on missing properties', function () {
                expect(intContext.bindGet('wrong', true)).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(intContext.bindGet('num', true)).to.throw(OperationError);
                expect(intContext.bindGet('obj', true)).to.throw(OperationError);
                expect(intContext.bindGet('arr', true)).to.throw(OperationError);
                expect(intContext.bindGet('interval1', true)).to.throw(OperationError);
                expect(intContext.bindGet('interval3', true)).to.throw(OperationError);
                expect(intContext.bindGet('interval4', true)).to.throw(OperationError);
            });
        });

        describe('method "getOptInterval"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptInterval');
                expect(OperationContext).to.not.respondTo('getOptInterval');
            });
            it('should return existing properties', function () {
                expect(intContext.getOptInterval('interval1', false)).to.stringifyTo('2:1000000');
                expect(intContext.getOptInterval('interval2', false)).to.stringifyTo('2:2');
                expect(intContext.getOptInterval('interval2', true)).to.stringifyTo('2:2');
                expect(intContext.getOptInterval(true)).to.stringifyTo('3:3');
            });
            it('should return null for missing properties', function () {
                expect(intContext.getOptInterval('wrong', true)).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(intContext.bindOpt('num', true)).to.throw(OperationError);
                expect(intContext.bindOpt('obj', true)).to.throw(OperationError);
                expect(intContext.bindOpt('arr', true)).to.throw(OperationError);
                expect(intContext.bindOpt('interval1', true)).to.throw(OperationError);
                expect(intContext.bindOpt('interval3', true)).to.throw(OperationError);
                expect(intContext.bindOpt('interval4', true)).to.throw(OperationError);
            });
        });

        describe('method "getAddress"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getAddress');
                expect(OperationContext).to.not.respondTo('getAddress');
            });
            it('should return existing properties', function () {
                expect(addrContext.getAddress('address1')).to.stringifyTo('B3');
                expect(addrContext.getAddress('start')).to.stringifyTo('C4');
                expect(addrContext.getAddress()).to.stringifyTo('C4');
            });
            it('should throw on missing properties', function () {
                expect(addrContext.bindGet('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(addrContext.bindGet('num')).to.throw(OperationError);
                expect(addrContext.bindGet('obj')).to.throw(OperationError);
                expect(addrContext.bindGet('arr')).to.throw(OperationError);
                expect(addrContext.bindGet('address2')).to.throw(OperationError);
                expect(addrContext.bindGet('address3')).to.throw(OperationError);
            });
        });

        describe('method "getOptAddress"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptAddress');
                expect(OperationContext).to.not.respondTo('getOptAddress');
            });
            it('should return existing properties', function () {
                expect(addrContext.getOptAddress('address1')).to.stringifyTo('B3');
                expect(addrContext.getOptAddress('start')).to.stringifyTo('C4');
                expect(addrContext.getOptAddress()).to.stringifyTo('C4');
            });
            it('should return null for missing properties', function () {
                expect(addrContext.getOptAddress('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(addrContext.bindOpt('num')).to.throw(OperationError);
                expect(addrContext.bindOpt('obj')).to.throw(OperationError);
                expect(addrContext.bindOpt('arr')).to.throw(OperationError);
                expect(addrContext.bindOpt('address2')).to.throw(OperationError);
                expect(addrContext.bindOpt('address3')).to.throw(OperationError);
            });
        });

        describe('method "getRange"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getRange');
                expect(OperationContext).to.not.respondTo('getRange');
            });
            it('should return existing properties', function () {
                expect(rangeContext.getRange('range1')).to.stringifyTo('B3:D5');
                expect(rangeContext.getRange('range2')).to.stringifyTo('B3:B3');
                expect(rangeContext.getRange()).to.stringifyTo('C4:C4');
            });
            it('should throw on missing properties', function () {
                expect(rangeContext.bindGet('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(rangeContext.bindGet('num')).to.throw(OperationError);
                expect(rangeContext.bindGet('obj')).to.throw(OperationError);
                expect(rangeContext.bindGet('arr')).to.throw(OperationError);
                expect(rangeContext.bindGet('range3')).to.throw(OperationError);
            });
        });

        describe('method "getOptRange"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptRange');
                expect(OperationContext).to.not.respondTo('getOptRange');
            });
            it('should return existing properties', function () {
                expect(rangeContext.getOptRange('range1')).to.stringifyTo('B3:D5');
                expect(rangeContext.getOptRange('range2')).to.stringifyTo('B3:B3');
                expect(rangeContext.getOptRange()).to.stringifyTo('C4:C4');
            });
            it('should return null for missing properties', function () {
                expect(rangeContext.getOptRange('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(rangeContext.bindOpt('num')).to.throw(OperationError);
                expect(rangeContext.bindOpt('obj')).to.throw(OperationError);
                expect(rangeContext.bindOpt('arr')).to.throw(OperationError);
                expect(rangeContext.bindOpt('range3')).to.throw(OperationError);
            });
        });

        describe('method "getRangeArray"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getRangeArray');
                expect(OperationContext).to.not.respondTo('getRangeArray');
            });
            it('should return existing properties', function () {
                expect(arrayContext.getRangeArray('ranges')).to.stringifyTo('B3:B3');
                expect(arrayContext.getRangeArray('ranges2')).to.stringifyTo('B3:B3,C4:D5');
                expect(arrayContext.getRangeArray()).to.stringifyTo('B3:B3');
            });
            it('should throw on missing properties', function () {
                expect(arrayContext.bindGet('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(arrayContext.bindGet('num')).to.throw(OperationError);
                expect(arrayContext.bindGet('obj')).to.throw(OperationError);
                expect(arrayContext.bindGet('arr')).to.throw(OperationError);
                expect(arrayContext.bindGet('ranges3')).to.throw(OperationError);
                expect(arrayContext.bindGet('ranges4')).to.throw(OperationError);
            });
        });

        describe('method "getOptRangeArray"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptRangeArray');
                expect(OperationContext).to.not.respondTo('getOptRangeArray');
            });
            it('should return existing properties', function () {
                expect(arrayContext.getOptRangeArray('ranges')).to.stringifyTo('B3:B3');
                expect(arrayContext.getOptRangeArray('ranges2')).to.stringifyTo('B3:B3,C4:D5');
                expect(arrayContext.getOptRangeArray()).to.stringifyTo('B3:B3');
            });
            it('should return null for missing properties', function () {
                expect(arrayContext.getOptRangeArray('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(arrayContext.bindOpt('num')).to.throw(OperationError);
                expect(arrayContext.bindOpt('obj')).to.throw(OperationError);
                expect(arrayContext.bindOpt('arr')).to.throw(OperationError);
                expect(arrayContext.bindOpt('ranges3')).to.throw(OperationError);
                expect(arrayContext.bindOpt('ranges4')).to.throw(OperationError);
            });
        });
    });

    // ========================================================================
});
