/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/errorcode'
], function (ErrorCode) {

    'use strict';

    // class ErrorCode ========================================================

    describe('Spreadsheet class ErrorCode', function () {

        it('should exist', function () {
            expect(ErrorCode).to.be.a('function');
        });

        // constants ----------------------------------------------------------

        describe('constant "NATIVE_CODES"', function () {
            it('should exist', function () {
                expect(ErrorCode).to.have.a.property('NATIVE_CODES').that.is.an('object');
            });
            it('should provide built-in error code #NULL!', function () {
                expect(ErrorCode.NATIVE_CODES).to.have.a.property('NULL').that.is.an.instanceof(ErrorCode);
                expect(ErrorCode.NATIVE_CODES.NULL).to.have.a.property('key', 'NULL');
                expect(ErrorCode.NATIVE_CODES.NULL).to.have.a.property('num', 1);
            });
            it('should provide built-in error code #DIV/0!', function () {
                expect(ErrorCode.NATIVE_CODES).to.have.a.property('DIV0').that.is.an.instanceof(ErrorCode);
                expect(ErrorCode.NATIVE_CODES.DIV0).to.have.a.property('key', 'DIV0');
                expect(ErrorCode.NATIVE_CODES.DIV0).to.have.a.property('num', 2);
            });
            it('should provide built-in error code #VALUE!', function () {
                expect(ErrorCode.NATIVE_CODES).to.have.a.property('VALUE').that.is.an.instanceof(ErrorCode);
                expect(ErrorCode.NATIVE_CODES.VALUE).to.have.a.property('key', 'VALUE');
                expect(ErrorCode.NATIVE_CODES.VALUE).to.have.a.property('num', 3);
            });
            it('should provide built-in error code #REF!', function () {
                expect(ErrorCode.NATIVE_CODES).to.have.a.property('REF').that.is.an.instanceof(ErrorCode);
                expect(ErrorCode.NATIVE_CODES.REF).to.have.a.property('key', 'REF');
                expect(ErrorCode.NATIVE_CODES.REF).to.have.a.property('num', 4);
            });
            it('should provide built-in error code #NAME?', function () {
                expect(ErrorCode.NATIVE_CODES).to.have.a.property('NAME').that.is.an.instanceof(ErrorCode);
                expect(ErrorCode.NATIVE_CODES.NAME).to.have.a.property('key', 'NAME');
                expect(ErrorCode.NATIVE_CODES.NAME).to.have.a.property('num', 5);
            });
            it('should provide built-in error code #NUM!', function () {
                expect(ErrorCode.NATIVE_CODES).to.have.a.property('NUM').that.is.an.instanceof(ErrorCode);
                expect(ErrorCode.NATIVE_CODES.NUM).to.have.a.property('key', 'NUM');
                expect(ErrorCode.NATIVE_CODES.NUM).to.have.a.property('num', 6);
            });
            it('should provide built-in error code #N/A', function () {
                expect(ErrorCode.NATIVE_CODES).to.have.a.property('NA').that.is.an.instanceof(ErrorCode);
                expect(ErrorCode.NATIVE_CODES.NA).to.have.a.property('key', 'NA');
                expect(ErrorCode.NATIVE_CODES.NA).to.have.a.property('num', 7);
            });
        });

        describe('constant', function () {
            it('"NULL" should exist', function () {
                expect(ErrorCode).to.have.a.property('NULL', ErrorCode.NATIVE_CODES.NULL);
            });
            it('"DIV0" should exist', function () {
                expect(ErrorCode).to.have.a.property('DIV0', ErrorCode.NATIVE_CODES.DIV0);
            });
            it('"VALUE" should exist', function () {
                expect(ErrorCode).to.have.a.property('VALUE', ErrorCode.NATIVE_CODES.VALUE);
            });
            it('"REF" should exist', function () {
                expect(ErrorCode).to.have.a.property('REF', ErrorCode.NATIVE_CODES.REF);
            });
            it('"NAME" should exist', function () {
                expect(ErrorCode).to.have.a.property('NAME', ErrorCode.NATIVE_CODES.NAME);
            });
            it('"NUM" should exist', function () {
                expect(ErrorCode).to.have.a.property('NUM', ErrorCode.NATIVE_CODES.NUM);
            });
            it('"NA" should exist', function () {
                expect(ErrorCode).to.have.a.property('NA', ErrorCode.NATIVE_CODES.NA);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(ErrorCode).itself.to.respondTo('create');
            });
            it('should return existing built-in error codes', function () {
                expect(ErrorCode.create('NUM')).to.equal(ErrorCode.NUM);
                expect(ErrorCode.create('DIV0')).to.equal(ErrorCode.DIV0);
            });
            it('should create other error codes', function () {
                var test1 = ErrorCode.create('test');
                expect(test1).to.be.an.instanceof(ErrorCode);
                expect(test1).to.have.a.property('key').that.equals('test');
                var test2 = ErrorCode.create('test');
                expect(test1).to.equal(test2);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(ErrorCode).to.respondTo('toJSON');
            });
            it('should convert error code to the string representation', function () {
                expect(JSON.stringify(ErrorCode.NUM)).to.equal('"#NUM"');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(ErrorCode).to.respondTo('toString');
            });
            it('should convert error code to the string representation', function () {
                expect(ErrorCode.NUM.toString()).to.equal('#NUM');
                expect('<' + ErrorCode.NUM + '>').to.equal('<#NUM>');
            });
        });
    });

    // ========================================================================
});
