/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/model/presetformattable'
], function (PresetFormatTable) {

    'use strict';

    // class PresetFormatTable ================================================

    describe('Spreadsheet class PresetFormatTable', function () {

        it('should exist', function () {
            expect(PresetFormatTable).to.be.a('function');
        });

        // static methods -----------------------------------------------------

        describe('static method "create"', function () {
            it('should exist', function () {
                expect(PresetFormatTable).itself.to.respondTo('create');
            });
            it('should create a preset table once for each locale', function () {
                var presetTableDe = PresetFormatTable.create('de_DE');
                expect(presetTableDe).to.be.an.instanceof(PresetFormatTable);
                expect(PresetFormatTable.create('de_DE')).to.equal(presetTableDe);
                var presetTableEn = PresetFormatTable.create('en_US');
                expect(presetTableEn).to.be.an.instanceof(PresetFormatTable);
                expect(presetTableEn).to.not.equal(presetTableDe);
            });
        });

        describe('static method "getDecimalId"', function () {
            it('should exist', function () {
                expect(PresetFormatTable).itself.to.respondTo('getDecimalId');
            });
            it('should return the identifiers of decimal number formats', function () {
                expect(PresetFormatTable.getDecimalId(true, false)).to.equal(1);
                expect(PresetFormatTable.getDecimalId(false, false)).to.equal(2);
                expect(PresetFormatTable.getDecimalId(true, true)).to.equal(3);
                expect(PresetFormatTable.getDecimalId(false, true)).to.equal(4);
            });
        });

        describe('static method "getScientificId"', function () {
            it('should exist', function () {
                expect(PresetFormatTable).itself.to.respondTo('getScientificId');
            });
            it('should return the identifiers of scientific number formats', function () {
                expect(PresetFormatTable.getScientificId(true)).to.equal(11);
                expect(PresetFormatTable.getScientificId(false)).to.equal(48);
            });
        });

        describe('static method "getPercentId"', function () {
            it('should exist', function () {
                expect(PresetFormatTable).itself.to.respondTo('getPercentId');
            });
            it('should return the identifiers of percentage formats', function () {
                expect(PresetFormatTable.getPercentId(true)).to.equal(9);
                expect(PresetFormatTable.getPercentId(false)).to.equal(10);
            });
        });

        describe('static method "getFractionId"', function () {
            it('should exist', function () {
                expect(PresetFormatTable).itself.to.respondTo('getFractionId');
            });
            it('should return the identifiers of fraction formats', function () {
                expect(PresetFormatTable.getFractionId(false)).to.equal(12);
                expect(PresetFormatTable.getFractionId(true)).to.equal(13);
            });
        });

        describe('static method "getCurrencyId"', function () {
            it('should exist', function () {
                expect(PresetFormatTable).itself.to.respondTo('getCurrencyId');
            });
            it('should return the identifiers of currency formats', function () {
                expect(PresetFormatTable.getCurrencyId(true, false, false)).to.equal(5);
                expect(PresetFormatTable.getCurrencyId(true, true, false)).to.equal(6);
                expect(PresetFormatTable.getCurrencyId(false, false, false)).to.equal(7);
                expect(PresetFormatTable.getCurrencyId(false, true, false)).to.equal(8);
                expect(PresetFormatTable.getCurrencyId(true, false, true)).to.equal(37);
                expect(PresetFormatTable.getCurrencyId(true, true, true)).to.equal(38);
                expect(PresetFormatTable.getCurrencyId(false, false, true)).to.equal(39);
                expect(PresetFormatTable.getCurrencyId(false, true, true)).to.equal(40);
            });
        });

        describe('static method "getTimeId"', function () {
            it('should exist', function () {
                expect(PresetFormatTable).itself.to.respondTo('getTimeId');
            });
            it('should return the identifiers of time formats', function () {
                expect(PresetFormatTable.getTimeId(false, false)).to.equal(18);
                expect(PresetFormatTable.getTimeId(false, true)).to.equal(19);
                expect(PresetFormatTable.getTimeId(true, false)).to.equal(20);
                expect(PresetFormatTable.getTimeId(true, true)).to.equal(21);
                expect(PresetFormatTable.getTimeId(null, false)).to.equal(20);
                expect(PresetFormatTable.getTimeId(null, true)).to.equal(21);
            });
        });

        describe('static method "getCurrencyCode"', function () {
            it('should exist', function () {
                expect(PresetFormatTable).itself.to.respondTo('getCurrencyCode');
            });
            it('should return the format codes for currencies', function () {
                expect(PresetFormatTable.getCurrencyCode('-0$', 'EUR')).to.equal('#,##0.00"EUR";-#,##0.00"EUR"');
                expect(PresetFormatTable.getCurrencyCode('-0$', 'EUR', { int: true })).to.equal('#,##0"EUR";-#,##0"EUR"');
                expect(PresetFormatTable.getCurrencyCode('-0$', 'EUR', { space: true })).to.equal('#,##0.00 "EUR";-#,##0.00 "EUR"');
                expect(PresetFormatTable.getCurrencyCode('-0$', 'EUR', { red: true })).to.equal('#,##0.00"EUR";[Red]-#,##0.00"EUR"');
            });
        });

        describe('static method "getFractionCode"', function () {
            it('should exist', function () {
                expect(PresetFormatTable).itself.to.respondTo('getFractionCode');
            });
            it('should return the format codes for fractions', function () {
                expect(PresetFormatTable.getFractionCode(1)).to.equal('# ?/?');
                expect(PresetFormatTable.getFractionCode(2)).to.equal('# ??/??');
                expect(PresetFormatTable.getFractionCode(3)).to.equal('# ???/???');
            });
        });

        // public methods -----------------------------------------------------

        describe('method "getFormatCode"', function () {
            it('should exist', function () {
                expect(PresetFormatTable).to.respondTo('getFormatCode');
            });
            it('should return the preset format codes', function () {
                var presetTable = PresetFormatTable.create('de_DE');
                expect(presetTable.getFormatCode(0)).to.equal('General');
                expect(presetTable.getFormatCode(1)).to.equal('0');
                expect(presetTable.getFormatCode(2)).to.equal('0.00');
                expect(presetTable.getFormatCode(3)).to.equal('#,##0');
                expect(presetTable.getFormatCode(4)).to.equal('#,##0.00');
                expect(presetTable.getFormatCode(5)).to.equal('#,##0 \u20ac;-#,##0 \u20ac');
                expect(presetTable.getFormatCode(6)).to.equal('#,##0 \u20ac;[Red]-#,##0 \u20ac');
                expect(presetTable.getFormatCode(7)).to.equal('#,##0.00 \u20ac;-#,##0.00 \u20ac');
                expect(presetTable.getFormatCode(8)).to.equal('#,##0.00 \u20ac;[Red]-#,##0.00 \u20ac');
                expect(presetTable.getFormatCode(9)).to.equal('0%');
                expect(presetTable.getFormatCode(10)).to.equal('0.00%');
                expect(presetTable.getFormatCode(11)).to.equal('0.00E+00');
                expect(presetTable.getFormatCode(12)).to.equal('# ?/?');
                expect(presetTable.getFormatCode(13)).to.equal('# ??/??');
                expect(presetTable.getFormatCode(14)).to.equal('DD.MM.YYYY');
                expect(presetTable.getFormatCode(15)).to.equal('DD. MMM YY');
                expect(presetTable.getFormatCode(16)).to.equal('DD. MMM');
                expect(presetTable.getFormatCode(17)).to.equal('MMM YY');
                expect(presetTable.getFormatCode(18)).to.equal('h:mm AM/PM');
                expect(presetTable.getFormatCode(19)).to.equal('h:mm:ss AM/PM');
                expect(presetTable.getFormatCode(20)).to.equal('hh:mm');
                expect(presetTable.getFormatCode(21)).to.equal('hh:mm:ss');
                expect(presetTable.getFormatCode(22)).to.equal('DD.MM.YYYY hh:mm');
                expect(presetTable.getFormatCode(37)).to.equal('#,##0 _\u20ac;-#,##0 _\u20ac');
                expect(presetTable.getFormatCode(38)).to.equal('#,##0 _\u20ac;[Red]-#,##0 _\u20ac');
                expect(presetTable.getFormatCode(39)).to.equal('#,##0.00 _\u20ac;-#,##0.00 _\u20ac');
                expect(presetTable.getFormatCode(40)).to.equal('#,##0.00 _\u20ac;[Red]-#,##0.00 _\u20ac');
                expect(presetTable.getFormatCode(41)).to.equal('_-* #,##0 _\u20ac_-;-* #,##0 _\u20ac_-;_-* "-" _\u20ac_-;_-@_-');
                expect(presetTable.getFormatCode(42)).to.equal('_-* #,##0 \u20ac_-;-* #,##0 \u20ac_-;_-* "-" \u20ac_-;_-@_-');
                expect(presetTable.getFormatCode(43)).to.equal('_-* #,##0.00 _\u20ac_-;-* #,##0.00 _\u20ac_-;_-* "-"?? _\u20ac_-;_-@_-');
                expect(presetTable.getFormatCode(44)).to.equal('_-* #,##0.00 \u20ac_-;-* #,##0.00 \u20ac_-;_-* "-"?? \u20ac_-;_-@_-');
                expect(presetTable.getFormatCode(45)).to.equal('mm:ss');
                expect(presetTable.getFormatCode(46)).to.equal('[h]:mm:ss');
                expect(presetTable.getFormatCode(47)).to.equal('mm:ss.0');
                expect(presetTable.getFormatCode(48)).to.equal('0.0E+0');
                expect(presetTable.getFormatCode(49)).to.equal('@');
            });
        });
    });

    // ========================================================================
});
