/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/model/formula/utils/sheetref'
], function (SheetRef) {

    'use strict';

    // class SheetRef =========================================================

    describe('Spreadsheet class SheetRef', function () {

        it('should exist', function () {
            expect(SheetRef).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // dummy spreadsheet document model
        var docModel = {
            getSheetIndex: function (sheetName) {
                var sheet = { Sheet1: 0, 'Sheet 2': 1, 'Sheet\'3': 2 }[sheetName];
                return _.isNumber(sheet) ? sheet : -1;
            }
        };

        // convenience: create a SheetRef instance on-the-fly
        function ref(sheet, abs) {
            return new SheetRef(sheet, abs);
        }

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a sheet reference', function () {
                var r1 = new SheetRef(1, true);
                expect(r1).to.have.property('sheet', 1);
                expect(r1).to.have.property('abs', true);
                var r2 = new SheetRef('Sheet0', false);
                expect(r2).to.have.property('sheet', 'Sheet0');
                expect(r2).to.have.property('abs', false);
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(SheetRef).to.respondTo('toString');
            });
            it('should return an internal string representation', function () {
                expect(ref(1, true).toString()).to.equal('$1');
                expect(ref(2, false).toString()).to.equal('2');
                expect(ref(-1, false).toString()).to.equal('#REF');
                expect(ref('Sheet1', true).toString()).to.equal('$\'Sheet1\'');
                expect(ref('Sheet\'2', false).toString()).to.equal('\'Sheet\'\'2\'');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(SheetRef).itself.to.respondTo('create');
            });
            it('should create a sheet reference', function () {
                var r1 = SheetRef.create(docModel, 'Sheet1', null, null, true);
                expect(r1).to.be.an.instanceof(SheetRef);
                expect(r1).to.stringifyTo('$0');
                var r2 = SheetRef.create(docModel, null, 'Sheet 2', null, false);
                expect(r2).to.be.an.instanceof(SheetRef);
                expect(r2).to.stringifyTo('1');
                var r3 = SheetRef.create(docModel, null, 'Sheet\'\'3', null, true);
                expect(r3).to.be.an.instanceof(SheetRef);
                expect(r3).to.stringifyTo('$2');
                var r4 = SheetRef.create(docModel, null, null, '#BEZUG!', false);
                expect(r4).to.be.an.instanceof(SheetRef);
                expect(r4).to.stringifyTo('#REF');
                var r5 = SheetRef.create(docModel, 'Sheet0', null, null, true);
                expect(r5).to.be.an.instanceof(SheetRef);
                expect(r5).to.stringifyTo('$\'Sheet0\'');
                expect(SheetRef.create(docModel, null, null, null, false)).to.equal(null);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(SheetRef).to.respondTo('clone');
            });
            it('should create a clone', function () {
                var r1 = ref(2, true), r2 = r1.clone();
                expect(r2).to.be.an.instanceof(SheetRef);
                expect(r2).to.not.equal(r1);
                expect(r2).to.stringifyTo('$2');
            });
        });

        describe('method "valid"', function () {
            it('should exist', function () {
                expect(SheetRef).to.respondTo('valid');
            });
            it('should return true for valid sheet references', function () {
                expect(ref(0, true).valid()).to.equal(true);
                expect(ref(2, false).valid()).to.equal(true);
            });
            it('should return false for invalid sheet references', function () {
                expect(ref(-1, true).valid()).to.equal(false);
                expect(ref('Sheet0', false).valid()).to.equal(false);
            });
        });

        describe('method "equals"', function () {
            it('should exist', function () {
                expect(SheetRef).to.respondTo('equals');
            });
            it('should return true for equal sheet references', function () {
                expect(ref(0, true).equals(ref(0, true))).to.equal(true);
                expect(ref('Sheet0', false).equals(ref('Sheet0', false))).to.equal(true);
            });
            it('should return false for different sheet references', function () {
                expect(ref(1, true).equals(ref(0, true))).to.equal(false);
                expect(ref(1, true).equals(ref(1, false))).to.equal(false);
                expect(ref(1, true).equals(ref('Sheet1', true))).to.equal(false);
                expect(ref('Sheet0', true).equals(ref('Sheet1', true))).to.equal(false);
                expect(ref('Sheet0', true).equals(ref('Sheet0', false))).to.equal(false);
            });
        });

        describe('method "transform"', function () {
            it('should exist', function () {
                expect(SheetRef).to.respondTo('transform');
            });
            it('should adjust sheet index when inserting a sheet', function () {
                var r1 = ref(2, true);
                expect(r1.transform(3, null)).to.equal(false);
                expect(r1).to.stringifyTo('$2');
                expect(r1.transform(2, null)).to.equal(true);
                expect(r1).to.stringifyTo('$3');
                expect(r1.transform(1, null)).to.equal(true);
                expect(r1).to.stringifyTo('$4');
            });
            it('should adjust sheet index when deleting a sheet', function () {
                var r1 = ref(2, true);
                expect(r1.transform(null, 3)).to.equal(false);
                expect(r1).to.stringifyTo('$2');
                expect(r1.transform(null, 1)).to.equal(true);
                expect(r1).to.stringifyTo('$1');
                expect(r1.transform(null, 1)).to.equal(true);
                expect(r1).to.stringifyTo('#REF');
            });
            it('should adjust sheet index when moving a sheet', function () {
                var r1 = ref(2, true);
                expect(r1.transform(1, 0)).to.equal(false);
                expect(r1).to.stringifyTo('$2');
                expect(r1.transform(4, 3)).to.equal(false);
                expect(r1).to.stringifyTo('$2');
                expect(r1.transform(3, 2)).to.equal(true);
                expect(r1).to.stringifyTo('$3');
                expect(r1.transform(1, 3)).to.equal(true);
                expect(r1).to.stringifyTo('$1');
                expect(r1.transform(0, 2)).to.equal(true);
                expect(r1).to.stringifyTo('$2');
                expect(r1.transform(2, 4)).to.equal(true);
                expect(r1).to.stringifyTo('$3');
                expect(r1.transform(3, 0)).to.equal(true);
                expect(r1).to.stringifyTo('$2');
            });
            it('should not do anything for invalid sheet references', function () {
                expect(ref(-1, true).transform(1, 0)).to.equal(false);
                expect(ref('Sheet1', true).transform(1, 0)).to.equal(false);
            });
        });
    });

    // ========================================================================
});
