/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/utils/complex'
], function (ErrorCode, Complex) {

    'use strict';

    // class Complex ==========================================================

    describe('Spreadsheet class Complex', function () {

        it('should exist', function () {
            expect(Complex).to.be.a('function');
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a complex numner', function () {
                var c1 = new Complex(2, -1, 'i');
                expect(c1).to.have.property('real', 2);
                expect(c1).to.have.property('imag', -1);
                expect(c1).to.have.property('unit', 'i');
                var c2 = new Complex(-1, 0, 'j');
                expect(c2).to.have.property('real', -1);
                expect(c2).to.have.property('imag', 0);
                expect(c2).to.have.property('unit', 'j');
                var c3 = new Complex(0, 0, null);
                expect(c3).to.have.property('real', 0);
                expect(c3).to.have.property('imag', 0);
                expect(c3).to.have.property('unit', null);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "abs"', function () {
            it('should exist', function () {
                expect(Complex).to.respondTo('abs');
            });
            it('should return the absolute value', function () {
                expect(new Complex(3, 4, null).abs()).to.equal(5);
                expect(new Complex(-3, 4, null).abs()).to.equal(5);
                expect(new Complex(3, -4, null).abs()).to.equal(5);
                expect(new Complex(-3, -4, null).abs()).to.equal(5);
                expect(new Complex(3, 0, null).abs()).to.equal(3);
                expect(new Complex(-3, 0, null).abs()).to.equal(3);
                expect(new Complex(0, 4, null).abs()).to.equal(4);
                expect(new Complex(0, -4, null).abs()).to.equal(4);
                expect(new Complex(0, 0, null).abs()).to.equal(0);
            });
        });

        describe('method "arg"', function () {
            it('should exist', function () {
                expect(Complex).to.respondTo('arg');
            });
            it('should return the argument', function () {
                expect(new Complex(3, 0, null).arg()).to.equal(0);
                expect(new Complex(3, 3, null).arg()).to.almostEqual(Math.PI * 0.25);
                expect(new Complex(0, 3, null).arg()).to.almostEqual(Math.PI * 0.5);
                expect(new Complex(-3, 3, null).arg()).to.almostEqual(Math.PI * 0.75);
                expect(new Complex(-3, 0, null).arg()).to.almostEqual(Math.PI);
                expect(new Complex(-3, -3, null).arg()).to.almostEqual(Math.PI * -0.75);
                expect(new Complex(0, -3, null).arg()).to.almostEqual(Math.PI * -0.5);
                expect(new Complex(3, -3, null).arg()).to.almostEqual(Math.PI * -0.25);
            });
            it('should throw #DIV/0! error code for complex zero', function () {
                var c1 = new Complex(0, 0, null);
                expect(c1.arg.bind(c1)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "exp"', function () {
            it('should exist', function () {
                expect(Complex).to.respondTo('exp');
            });
            it('should return the complex exponential', function () {
                var c1 = new Complex(0, 0, 'i'), c2 = c1.exp();
                expect(c2).to.be.an.instanceof(Complex);
                expect(c2.real).to.equal(1);
                expect(c2.imag).to.equal(0);
                expect(c2.unit).to.equal('i');
                expect(c2).to.not.equal(c1);
                var c3 = new Complex(4, 3, 'j').exp();
                expect(c3.real).to.almostEqual(Math.exp(4) * Math.cos(3));
                expect(c3.imag).to.almostEqual(Math.exp(4) * Math.sin(3));
                expect(c3.unit).to.equal('j');
            });
        });

        describe('method "pow"', function () {
            it('should exist', function () {
                expect(Complex).to.respondTo('pow');
            });
            it('should return the complex power', function () {
                var c1 = new Complex(0, 0, 'i'), c2 = c1.pow(1);
                expect(c2).to.be.an.instanceof(Complex);
                expect(c2.real).to.equal(0);
                expect(c2.imag).to.equal(0);
                expect(c2.unit).to.equal('i');
                expect(c2).to.not.equal(c1);
                var c3 = new Complex(4, -3, 'j').pow(2);
                expect(c3.real).to.almostEqual(7);
                expect(c3.imag).to.almostEqual(-24);
                expect(c3.unit).to.equal('j');
                var c4 = new Complex(-7, 24, null).pow(0.5);
                expect(c4.real).to.almostEqual(3);
                expect(c4.imag).to.almostEqual(4);
                expect(c4.unit).to.equal(null);
                var c5 = new Complex(3, 4, null).pow(-1);
                expect(c5.real).to.almostEqual(0.12);
                expect(c5.imag).to.almostEqual(-0.16);
            });
            it('should throw #NUM! error code for complex zero and non-positive exponent', function () {
                var c1 = new Complex(0, 0, null);
                expect(c1.pow.bind(c1, 0)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
                expect(c1.pow.bind(c1, -1)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('method "log"', function () {
            it('should exist', function () {
                expect(Complex).to.respondTo('log');
            });
            it('should return the complex logarithm', function () {
                var c1 = new Complex(3, 4, 'i'), c2 = c1.log(1);
                expect(c2).to.be.an.instanceof(Complex);
                expect(c2.real).to.almostEqual(Math.log(c1.abs()));
                expect(c2.imag).to.almostEqual(c1.arg());
                expect(c2.unit).to.equal('i');
                expect(c2).to.not.equal(c1);
                var c3 = new Complex(3, 4, 'j').log(2);
                expect(c3.real).to.almostEqual(Math.log(c1.abs()) / 2);
                expect(c3.imag).to.almostEqual(c1.arg() / 2);
                expect(c3.unit).to.equal('j');
            });
            it('should throw #DIV/0! error code for complex zero', function () {
                var c1 = new Complex(0, 0, null);
                expect(c1.log.bind(c1, 1)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Complex).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                expect(new Complex(3, 4, 'i').toString()).to.equal('3+4i');
                expect(new Complex(-3, 4, 'j').toString()).to.equal('-3+4j');
                expect(new Complex(3, -4, 'i').toString()).to.equal('3-4i');
                expect(new Complex(-3, -4, 'j').toString()).to.equal('-3-4j');
                expect(new Complex(3, 4, null).toString()).to.equal('3+4i');
                expect(new Complex(3, 0, 'i').toString()).to.equal('3+0i');
                expect(new Complex(-3, 0, 'j').toString()).to.equal('-3+0j');
                expect(new Complex(0, 4, 'i').toString()).to.equal('0+4i');
                expect(new Complex(0, -4, 'j').toString()).to.equal('0-4j');
                expect(new Complex(0, 0, null).toString()).to.equal('0+0i');
            });
        });
    });

    // ========================================================================
});
