/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/movedescriptor',
    'io.ox/office/spreadsheet/model/formula/utils/matrix',
    'io.ox/office/spreadsheet/model/formula/utils/cellref',
    'io.ox/office/spreadsheet/model/formula/utils/sheetref',
    'io.ox/office/spreadsheet/model/formula/parser/tokens'
], function (AppHelper, SheetHelper, MoveDescriptor, Matrix, CellRef, SheetRef, Tokens) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var i = SheetHelper.i;
    var a = SheetHelper.a;
    var r = SheetHelper.r;

    // create a CellRef instance on-the-fly from $1.$1 notation (column/row offsets)
    function cell(text) {
        var m = /^(\$)?(-?\d+)\.(\$)?(-?\d+)$/i.exec(text);
        return new CellRef(parseInt(m[2], 10), parseInt(m[4], 10), !!m[1], !!m[3]);
    }

    // create a SheetRef instance on-the-fly
    function sh(sheet, abs) {
        return new SheetRef(sheet, !_.isBoolean(abs) || abs);
    }

    // module Tokens ==========================================================

    describe('Spreadsheet module Tokens', function () {

        it('should exist', function () {
            expect(Tokens).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet 4' }, // Sheet name with white-space
            { name: 'insertName', label: 'name', formula: 'Sheet1!$A$1' }, // global name
            { name: 'insertName', sheet: 1, label: 'name', formula: 'Sheet1!$A$1' }, // sheet-local name
            { name: 'insertTable', sheet: 0, table: 'Table1', start: [1, 1], end: [3, 4], attrs: { table: { headerRow: true, footerRow: true } } },
            { name: 'changeCells', sheet: 0, start: 'B2', contents: [['Col1', 'Col2', 'Col3']] }
        ];

        // initialize test document
        var docModel = null, opGrammar = null, uiGrammar = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            opGrammar = docModel.getFormulaGrammar('op');
            uiGrammar = docModel.getFormulaGrammar('ui');
        });

        // class FixedToken ===================================================

        describe('Spreadsheet class FixedToken', function () {
            it('should exist', function () {
                expect(Tokens.FixedToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var token = new Tokens.FixedToken('type', 'text');

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.FixedToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(token.getType()).to.equal('type');
                });
            });

            describe('method "isType"', function () {
                it('should exist', function () {
                    expect(Tokens.FixedToken).to.respondTo('isType');
                });
                it('should test the token type', function () {
                    expect(token.isType('type')).to.equal(true);
                    expect(token.isType('other')).to.equal(false);
                });
            });

            describe('method "matchesType"', function () {
                it('should exist', function () {
                    expect(Tokens.FixedToken).to.respondTo('matchesType');
                });
                it('should match the token type', function () {
                    expect(token.matchesType(/^t/)).to.equal(true);
                    expect(token.matchesType(/^o/)).to.equal(false);
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.FixedToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(token.getText(opGrammar)).to.equal('text');
                    expect(token.getText(uiGrammar)).to.equal('text');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.FixedToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(token.toString()).to.equal('type[text]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(Tokens.FixedToken).to.respondTo('getValue');
                });
                it('should return the value', function () {
                    expect(token.getValue()).to.equal('text');
                });
            });

            describe('method "setValue"', function () {
                it('should exist', function () {
                    expect(Tokens.FixedToken).to.respondTo('setValue');
                });
                it('should return whether the value changes', function () {
                    var token1 = new Tokens.FixedToken('type', 'text');
                    expect(token1.setValue('text')).to.equal(false);
                    expect(token1.getValue()).to.equal('text');
                    expect(token1.setValue('other')).to.equal(true);
                    expect(token1.getValue()).to.equal('other');
                    expect(token1.getText(opGrammar)).to.equal('other');
                    expect(token1.getText(uiGrammar)).to.equal('other');
                });
            });

            describe('method "appendValue"', function () {
                it('should exist', function () {
                    expect(Tokens.FixedToken).to.respondTo('appendValue');
                });
                it('should return whether the value changes', function () {
                    var token1 = new Tokens.FixedToken('type', 'other');
                    expect(token1.appendValue('')).to.equal(false);
                    expect(token1.getValue()).to.equal('other');
                    expect(token1.appendValue('text')).to.equal(true);
                    expect(token1.getValue()).to.equal('othertext');
                    expect(token1.getText(opGrammar)).to.equal('othertext');
                    expect(token1.getText(uiGrammar)).to.equal('othertext');
                });
            });
        });

        // class OperatorToken ================================================

        describe('Spreadsheet class OperatorToken', function () {
            it('should exist', function () {
                expect(Tokens.OperatorToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var token = new Tokens.OperatorToken('list');

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.OperatorToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(token.getType()).to.equal('op');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.OperatorToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(token.getText(opGrammar)).to.equal(',');
                    expect(token.getText(uiGrammar)).to.equal(';');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.OperatorToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(token.toString()).to.equal('op[list]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(Tokens.OperatorToken).to.respondTo('getValue');
                });
                it('should return the value', function () {
                    expect(token.getValue()).to.equal('list');
                });
            });
        });

        // class SeparatorToken ===============================================

        describe('Spreadsheet class SeparatorToken', function () {
            it('should exist', function () {
                expect(Tokens.SeparatorToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var token = new Tokens.SeparatorToken();

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.SeparatorToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(token.getType()).to.equal('sep');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.SeparatorToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(token.getText(opGrammar)).to.equal(',');
                    expect(token.getText(uiGrammar)).to.equal(';');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.SeparatorToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(token.toString()).to.equal('sep');
                });
            });
        });

        // class ParenthesisToken =============================================

        describe('Spreadsheet class ParenthesisToken', function () {
            it('should exist', function () {
                expect(Tokens.ParenthesisToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var openToken = new Tokens.ParenthesisToken(true);
            var closeToken = new Tokens.ParenthesisToken(false);

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.ParenthesisToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(openToken.getType()).to.equal('open');
                    expect(closeToken.getType()).to.equal('close');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.ParenthesisToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(openToken.getText(opGrammar)).to.equal('(');
                    expect(openToken.getText(uiGrammar)).to.equal('(');
                    expect(closeToken.getText(opGrammar)).to.equal(')');
                    expect(closeToken.getText(uiGrammar)).to.equal(')');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.ParenthesisToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(openToken.toString()).to.equal('open');
                    expect(closeToken.toString()).to.equal('close');
                });
            });
        });

        // class LiteralToken =================================================

        describe('Spreadsheet class LiteralToken', function () {
            it('should exist', function () {
                expect(Tokens.LiteralToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var numToken = new Tokens.LiteralToken(12.5);
            var strToken = new Tokens.LiteralToken('a"b"c');
            var falseToken = new Tokens.LiteralToken(false);
            var trueToken = new Tokens.LiteralToken(true);
            var errToken = new Tokens.LiteralToken(ErrorCode.REF);
            var nullToken = new Tokens.LiteralToken(null);

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.LiteralToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(numToken.getType()).to.equal('lit');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.LiteralToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(numToken.getText(opGrammar)).to.equal('12.5');
                    expect(numToken.getText(uiGrammar)).to.equal('12,5');
                    expect(strToken.getText(opGrammar)).to.equal('"a""b""c"');
                    expect(strToken.getText(uiGrammar)).to.equal('"a""b""c"');
                    expect(falseToken.getText(opGrammar)).to.equal('FALSE');
                    expect(falseToken.getText(uiGrammar)).to.equal('FALSCH');
                    expect(trueToken.getText(opGrammar)).to.equal('TRUE');
                    expect(trueToken.getText(uiGrammar)).to.equal('WAHR');
                    expect(errToken.getText(opGrammar)).to.equal('#REF!');
                    expect(errToken.getText(uiGrammar)).to.equal('#BEZUG!');
                    expect(nullToken.getText(opGrammar)).to.equal('');
                    expect(nullToken.getText(uiGrammar)).to.equal('');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.LiteralToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(numToken.toString()).to.equal('lit[12.5]');
                    expect(strToken.toString()).to.equal('lit["a""b""c"]');
                    expect(falseToken.toString()).to.equal('lit[FALSE]');
                    expect(trueToken.toString()).to.equal('lit[TRUE]');
                    expect(errToken.toString()).to.equal('lit[#REF]');
                    expect(nullToken.toString()).to.equal('lit[null]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(Tokens.LiteralToken).to.respondTo('getValue');
                });
                it('should return the value', function () {
                    expect(numToken.getValue()).to.equal(12.5);
                    expect(strToken.getValue()).to.equal('a"b"c');
                    expect(falseToken.getValue()).to.equal(false);
                    expect(trueToken.getValue()).to.equal(true);
                    expect(errToken.getValue()).to.equal(ErrorCode.REF);
                    expect(nullToken.getValue()).to.equal(null);
                });
            });

            describe('method "setValue"', function () {
                it('should exist', function () {
                    expect(Tokens.LiteralToken).to.respondTo('setValue');
                });
                it('should return whether the value changes', function () {
                    var token = new Tokens.LiteralToken(12.5);
                    expect(token.setValue(12.5)).to.equal(false);
                    expect(token.getValue()).to.equal(12.5);
                    expect(token.getText(uiGrammar)).to.equal('12,5');
                    expect(token.setValue(1)).to.equal(true);
                    expect(token.getValue()).to.equal(1);
                    expect(token.getText(uiGrammar)).to.equal('1');
                    expect(token.setValue('a"b"c')).to.equal(true);
                    expect(token.getValue()).to.equal('a"b"c');
                    expect(token.getText(uiGrammar)).to.equal('"a""b""c"');
                    expect(token.setValue(true)).to.equal(true);
                    expect(token.getValue()).to.equal(true);
                    expect(token.getText(uiGrammar)).to.equal('WAHR');
                    expect(token.setValue(ErrorCode.REF)).to.equal(true);
                    expect(token.getValue()).to.equal(ErrorCode.REF);
                    expect(token.getText(uiGrammar)).to.equal('#BEZUG!');
                    expect(token.setValue(null)).to.equal(true);
                    expect(token.getValue()).to.equal(null);
                    expect(token.getText(uiGrammar)).to.equal('');
                });
            });
        });

        // class MatrixToken ==================================================

        describe('Spreadsheet class MatrixToken', function () {
            it('should exist', function () {
                expect(Tokens.MatrixToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var matrix = new Matrix([[12.5, 'a"b"c'], [true, ErrorCode.REF]]);
            var token = new Tokens.MatrixToken(matrix);

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.MatrixToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(token.getType()).to.equal('mat');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.MatrixToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(token.getText(opGrammar)).to.equal('{12.5,"a""b""c";TRUE,#REF!}');
                    expect(token.getText(uiGrammar)).to.equal('{12,5;"a""b""c"|WAHR;#BEZUG!}');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.MatrixToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(token.toString()).to.equal('mat[{12.5;"a""b""c"|TRUE;#REF}]');
                });
            });

            describe('method "getMatrix"', function () {
                it('should exist', function () {
                    expect(Tokens.MatrixToken).to.respondTo('getMatrix');
                });
                it('should return the matrix', function () {
                    expect(token.getMatrix()).to.equal(matrix);
                });
            });
        });

        // class FunctionToken ================================================

        describe('Spreadsheet class FunctionToken', function () {
            it('should exist', function () {
                expect(Tokens.FunctionToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var token = new Tokens.FunctionToken('SUM');

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.FunctionToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(token.getType()).to.equal('func');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.FunctionToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(token.getText(opGrammar)).to.equal('SUM');
                    expect(token.getText(uiGrammar)).to.equal('SUMME');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.FunctionToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(token.toString()).to.equal('func[SUM]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(Tokens.FunctionToken).to.respondTo('getValue');
                });
                it('should return the value of the token', function () {
                    expect(token.getValue()).to.equal('SUM');
                });
            });
        });

        // class ReferenceToken ===============================================

        describe('Spreadsheet class ReferenceToken', function () {
            it('should exist', function () {
                expect(Tokens.ReferenceToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var cellToken, rangeToken, errToken, sheet1Token, sheet2Token, sheetErrToken;
            before(function () {
                cellToken = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, null, null);
                rangeToken = new Tokens.ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), null, null);
                errToken = new Tokens.ReferenceToken(docModel, null, null, null, null);
                sheet1Token = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sh(1), null);
                sheet2Token = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sh(1), sh(3));
                sheetErrToken = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sh(-1), null);
            });

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.ReferenceToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(cellToken.getType()).to.equal('ref');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.ReferenceToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(cellToken.getText(opGrammar)).to.equal('B$3');
                    expect(cellToken.getText(uiGrammar)).to.equal('B$3');
                    expect(rangeToken.getText(opGrammar)).to.equal('B$3:$D5');
                    expect(rangeToken.getText(uiGrammar)).to.equal('B$3:$D5');
                    expect(errToken.getText(opGrammar)).to.equal('#REF!');
                    expect(errToken.getText(uiGrammar)).to.equal('#BEZUG!');
                    expect(sheet1Token.getText(opGrammar)).to.equal('Sheet2!B$3');
                    expect(sheet1Token.getText(uiGrammar)).to.equal('Sheet2!B$3');
                    expect(sheet2Token.getText(opGrammar)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                    expect(sheet2Token.getText(uiGrammar)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                    expect(sheetErrToken.getText(opGrammar)).to.equal('#REF!!B$3');
                    expect(sheetErrToken.getText(uiGrammar)).to.equal('#BEZUG!!B$3');
                });
                it('should handle simple/complex sheet names correctly', function () {
                    var token1 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('Sheet1'), null);
                    expect(token1.getText(opGrammar)).to.equal('Sheet1!A1');
                    expect(token1.getText(uiGrammar)).to.equal('Sheet1!A1');
                    var token2 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('Sheet 2'), null);
                    expect(token2.getText(opGrammar)).to.equal('\'Sheet 2\'!A1');
                    expect(token2.getText(uiGrammar)).to.equal('\'Sheet 2\'!A1');
                    var token3 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('1Sheet'), null);
                    expect(token3.getText(opGrammar)).to.equal('\'1Sheet\'!A1');
                    expect(token3.getText(uiGrammar)).to.equal('\'1Sheet\'!A1');
                    var token4 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('Sheet+1'), null);
                    expect(token4.getText(opGrammar)).to.equal('\'Sheet+1\'!A1');
                    expect(token4.getText(uiGrammar)).to.equal('\'Sheet+1\'!A1');
                    var token5 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('Sheet1'), sh('Sheet2'));
                    expect(token5.getText(opGrammar)).to.equal('Sheet1:Sheet2!A1');
                    expect(token5.getText(uiGrammar)).to.equal('Sheet1:Sheet2!A1');
                    var token6 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('Sheet1'), sh('Sheet 2'));
                    expect(token6.getText(opGrammar)).to.equal('\'Sheet1:Sheet 2\'!A1');
                    expect(token6.getText(uiGrammar)).to.equal('\'Sheet1:Sheet 2\'!A1');
                    var token7 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('Sheet 1'), sh('Sheet2'));
                    expect(token7.getText(opGrammar)).to.equal('\'Sheet 1:Sheet2\'!A1');
                    expect(token7.getText(uiGrammar)).to.equal('\'Sheet 1:Sheet2\'!A1');
                });
                it('should treat booleans as complex sheet names', function () {
                    var token1 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('TRUE'));
                    expect(token1.getText(opGrammar)).to.equal('\'TRUE\'!A1');
                    expect(token1.getText(uiGrammar)).to.equal('TRUE!A1');
                    var token2 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('WAHR'));
                    expect(token2.getText(opGrammar)).to.equal('WAHR!A1');
                    expect(token2.getText(uiGrammar)).to.equal('\'WAHR\'!A1');
                });
                it('should treat cell addresses as complex sheet names', function () {
                    var token1 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('A1'));
                    expect(token1.getText(opGrammar)).to.equal('\'A1\'!A1');
                    expect(token1.getText(uiGrammar)).to.equal('\'A1\'!A1');
                    var token2 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('ZZZZ1'));
                    expect(token2.getText(opGrammar)).to.equal('ZZZZ1!A1');
                    expect(token2.getText(uiGrammar)).to.equal('ZZZZ1!A1');
                    var token3 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('R1C1'));
                    expect(token3.getText(opGrammar)).to.equal('\'R1C1\'!A1');
                    expect(token3.getText(uiGrammar)).to.equal('\'R1C1\'!A1');
                    var token4 = new Tokens.ReferenceToken(docModel, cell('0.0'), null, sh('Z1S1'));
                    expect(token4.getText(opGrammar)).to.equal('Z1S1!A1');
                    expect(token4.getText(uiGrammar)).to.equal('\'Z1S1\'!A1');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.ReferenceToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(cellToken.toString()).to.equal('ref[B$3]');
                    expect(rangeToken.toString()).to.equal('ref[B$3:$D5]');
                    expect(errToken.toString()).to.equal('ref[#REF]');
                    expect(sheet1Token.toString()).to.equal('ref[$1!B$3]');
                    expect(sheet2Token.toString()).to.equal('ref[$1:$3!B$3]');
                    expect(sheetErrToken.toString()).to.equal('ref[#REF!B$3]');
                });
            });

            describe('method "hasSheetRef"', function () {
                it('should exist', function () {
                    expect(Tokens.ReferenceToken).to.respondTo('hasSheetRef');
                });
                it('should return the correct result', function () {
                    expect(cellToken.hasSheetRef()).to.equal(false);
                    expect(rangeToken.hasSheetRef()).to.equal(false);
                    expect(errToken.hasSheetRef()).to.equal(false);
                    expect(sheet1Token.hasSheetRef()).to.equal(true);
                    expect(sheet2Token.hasSheetRef()).to.equal(true);
                    expect(sheetErrToken.hasSheetRef()).to.equal(true);
                });
            });

            describe('method "hasSheetError"', function () {
                it('should exist', function () {
                    expect(Tokens.ReferenceToken).to.respondTo('hasSheetError');
                });
                it('should return the correct result', function () {
                    expect(cellToken.hasSheetError()).to.equal(false);
                    expect(rangeToken.hasSheetError()).to.equal(false);
                    expect(errToken.hasSheetError()).to.equal(false);
                    expect(sheet1Token.hasSheetError()).to.equal(false);
                    expect(sheet2Token.hasSheetError()).to.equal(false);
                    expect(sheetErrToken.hasSheetError()).to.equal(true);
                });
            });

            describe('method "getRange3D"', function () {
                it('should exist', function () {
                    expect(Tokens.ReferenceToken).to.respondTo('getRange3D');
                });
                it('should return the correct range address', function () {
                    expect(cellToken.getRange3D({ refSheet: 0 })).to.stringifyTo('0:0!B3:B3');
                    expect(cellToken.getRange3D({ refSheet: 1 })).to.stringifyTo('1:1!B3:B3');
                    expect(rangeToken.getRange3D({ refSheet: 0 })).to.stringifyTo('0:0!B3:D5');
                    expect(rangeToken.getRange3D({ refSheet: 1 })).to.stringifyTo('1:1!B3:D5');
                    expect(errToken.getRange3D({ refSheet: 0 })).to.equal(null);
                    expect(errToken.getRange3D({ refSheet: 1 })).to.equal(null);
                    expect(sheet1Token.getRange3D({ refSheet: 0 })).to.stringifyTo('1:1!B3:B3');
                    expect(sheet1Token.getRange3D({ refSheet: 1 })).to.stringifyTo('1:1!B3:B3');
                    expect(sheet2Token.getRange3D()).to.stringifyTo('1:3!B3:B3');
                    expect(sheet2Token.getRange3D({ refSheet: 0 })).to.stringifyTo('1:3!B3:B3');
                    expect(sheet2Token.getRange3D({ refSheet: 1 })).to.stringifyTo('1:3!B3:B3');
                    expect(sheetErrToken.getRange3D({ refSheet: 0 })).to.equal(null);
                    expect(sheetErrToken.getRange3D({ refSheet: 1 })).to.equal(null);
                });
                it('should return the correct range address without reference sheet', function () {
                    expect(cellToken.getRange3D()).to.equal(null);
                    expect(rangeToken.getRange3D()).to.equal(null);
                    expect(errToken.getRange3D()).to.equal(null);
                    expect(sheet1Token.getRange3D()).to.stringifyTo('1:1!B3:B3');
                    expect(sheetErrToken.getRange3D()).to.equal(null);
                });
                it('should relocate cell range without wrapping', function () {
                    expect(cellToken.getRange3D({ refSheet: 0, targetAddress: a('E6') })).to.stringifyTo('0:0!F3:F3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2') })).to.stringifyTo('0:0!A3:A3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('C2') })).to.equal(null);
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9') })).to.stringifyTo('0:0!F3:F3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2') })).to.equal(null);
                    expect(rangeToken.getRange3D({ refSheet: 0, targetAddress: a('E6') })).to.stringifyTo('0:0!D3:F10');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B5') })).to.stringifyTo('0:0!A1:D3');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('C5') })).to.equal(null);
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B6') })).to.equal(null);
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9') })).to.stringifyTo('0:0!D3:F12');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2') })).to.equal(null);
                    expect(errToken.getRange3D({ refSheet: 0, targetAddress: a('E6') })).to.equal(null);
                });
                it('should relocate cell range with wrapping', function () {
                    expect(cellToken.getRange3D({ refSheet: 0, targetAddress: a('E6'), wrapReferences: true })).to.stringifyTo('0:0!F3:F3');
                    expect(cellToken.getRange3D({ refSheet: 0, targetAddress: a('XFD1'), wrapReferences: true })).to.stringifyTo('0:0!A3:A3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2'), wrapReferences: true })).to.stringifyTo('0:0!A3:A3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('C2'), wrapReferences: true })).to.stringifyTo('0:0!XFD3:XFD3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9'), wrapReferences: true })).to.stringifyTo('0:0!F3:F3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2'), wrapReferences: true })).to.stringifyTo('0:0!XFB3:XFB3');
                    expect(rangeToken.getRange3D({ refSheet: 0, targetAddress: a('E6'), wrapReferences: true })).to.stringifyTo('0:0!D3:F10');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B5'), wrapReferences: true })).to.stringifyTo('0:0!A1:D3');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('C5'), wrapReferences: true })).to.stringifyTo('0:0!D1:XFD3');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B6'), wrapReferences: true })).to.stringifyTo('0:0!A3:D1048576');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9'), wrapReferences: true })).to.stringifyTo('0:0!D3:F12');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2'), wrapReferences: true })).to.stringifyTo('0:0!D3:XFB1048574');
                    expect(errToken.getRange3D({ refSheet: 0, targetAddress: a('E6'), wrapReferences: true })).to.equal(null);
                });
            });

            describe('method "setRange"', function () {
                it('should exist', function () {
                    expect(Tokens.ReferenceToken).to.respondTo('setRange');
                });
                it('should change range address', function () {
                    var token1 = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, null, null);
                    expect(token1).to.stringifyTo('ref[B$3]');
                    expect(token1.setRange(r('B3:B3'))).to.equal(false);
                    expect(token1).to.stringifyTo('ref[B$3]');
                    expect(token1.setRange(r('D5:D5'))).to.equal(true);
                    expect(token1).to.stringifyTo('ref[D$5]');
                    expect(token1.setRange(r('B3:D5'))).to.equal(true);
                    expect(token1).to.stringifyTo('ref[B$3:D$5]');
                    var token2 = new Tokens.ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), null, null);
                    expect(token2.setRange(r('B3:D5'))).to.equal(false);
                    expect(token2).to.stringifyTo('ref[B$3:$D5]');
                    expect(token2.setRange(r('C4:D5'))).to.equal(true);
                    expect(token2).to.stringifyTo('ref[C$4:$D5]');
                    expect(token2.setRange(r('C4:E6'))).to.equal(true);
                    expect(token2).to.stringifyTo('ref[C$4:$E6]');
                    expect(token2.setRange(r('C4:C4'))).to.equal(true);
                    expect(token2).to.stringifyTo('ref[C$4:$C4]');
                });
                it('should convert reference error to valid reference', function () {
                    var token = new Tokens.ReferenceToken(docModel, null, null, null, null);
                    expect(token).to.stringifyTo('ref[#REF]');
                    expect(token.setRange(r('B3:D5'))).to.equal(true);
                    expect(token).to.stringifyTo('ref[$B$3:$D$5]');
                });
            });

            describe('method "relocateRange"', function () {
                it('should exist', function () {
                    expect(Tokens.ReferenceToken).to.respondTo('relocateRange');
                });
                it('should relocate cell range without wrapping', function () {
                    var token = new Tokens.ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), sh(1), null);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('B2'), a('B2'), false)).to.equal(false);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('A2'), a('B4'), false)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!C$3:$D7]');
                    expect(token.relocateRange(a('B4'), a('A2'), false)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('C1'), a('A1'), false)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!#REF]');
                });
                it('should relocate cell range with wrapping', function () {
                    var token = new Tokens.ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), sh(1), null);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('B2'), a('B2'), true)).to.equal(false);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('A2'), a('B4'), true)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!C$3:$D7]');
                    expect(token.relocateRange(a('B4'), a('A2'), true)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('C1'), a('A1'), true)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!XFD$3:$D5]');
                    expect(token.relocateRange(a('A6'), a('A1'), true)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!XFD$3:$D1048576]');
                });
            });

            describe('method "transformSheet"', function () {
                it('should exist', function () {
                    expect(Tokens.ReferenceToken).to.respondTo('transformSheet');
                });
                it('should do nothing for tokens without sheet', function () {
                    expect(cellToken.transformSheet(0, null)).to.equal(false);
                    expect(cellToken).to.stringifyTo('ref[B$3]');
                    expect(rangeToken.transformSheet(0, null)).to.equal(false);
                    expect(rangeToken).to.stringifyTo('ref[B$3:$D5]');
                });
                it('should transform sheet index', function () {
                    var token1 = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sh(1), null);
                    expect(token1.transformSheet(1, null)).to.equal(true);
                    expect(token1).to.stringifyTo('ref[$2!B$3]');
                    expect(token1.transformSheet(3, null)).to.equal(false);
                    expect(token1).to.stringifyTo('ref[$2!B$3]');
                    expect(token1.transformSheet(null, 1)).to.equal(true);
                    expect(token1).to.stringifyTo('ref[$1!B$3]');
                    expect(token1.transformSheet(null, 2)).to.equal(false);
                    expect(token1).to.stringifyTo('ref[$1!B$3]');
                    var token2 = new Tokens.ReferenceToken(docModel, cell('1.$2'), null, sh(1), sh(3));
                    expect(token2.transformSheet(1, null)).to.equal(true);
                    expect(token2).to.stringifyTo('ref[$2:$4!B$3]');
                    expect(token2.transformSheet(4, null)).to.equal(true);
                    expect(token2).to.stringifyTo('ref[$2:$5!B$3]');
                    expect(token2.transformSheet(6, null)).to.equal(false);
                    expect(token2).to.stringifyTo('ref[$2:$5!B$3]');
                    expect(token2.transformSheet(null, 4)).to.equal(true);
                    expect(token2).to.stringifyTo('ref[$2:$4!B$3]');
                    expect(token2.transformSheet(null, 1)).to.equal(true);
                    expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
                    expect(token2.transformSheet(null, 4)).to.equal(false);
                    expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
                });
                it('should do nothing for token with sheet error', function () {
                    expect(sheetErrToken.transformSheet(0, null)).to.equal(false);
                    expect(sheetErrToken).to.stringifyTo('ref[#REF!B$3]');
                });
            });

            describe('method "resolveMovedCells"', function () {
                it('should exist', function () {
                    expect(Tokens.ReferenceToken).to.respondTo('resolveMovedCells');
                });

                function makeMoveDescs(interval, columns, insert) {
                    return [new MoveDescriptor(docModel, columns ? i('1:1048576') : i('A:XFD'), i(interval), columns, insert)];
                }

                it('should transform single cell address', function () {
                    var token = new Tokens.ReferenceToken(docModel, cell('2.$2'), null, null, null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, true), { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, true))).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, true), { refSheet: 1 })).to.deep.equal({ text: 'D$3', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('B:C', true, true), { refSheet: 1 })).to.deep.equal({ text: 'E$3', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('D:D', true, true), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, true), { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, true))).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, true), { refSheet: 1 })).to.deep.equal({ text: 'C$4', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('2:3', false, true), { refSheet: 1 })).to.deep.equal({ text: 'C$5', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('4:4', false, true), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, false), { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, false))).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, false), { refSheet: 1 })).to.deep.equal({ text: 'B$3', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('B:C', true, false), { refSheet: 1 })).to.deep.equal({ text: '#REF!', refError: true });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('D:D', true, false), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, false), { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, false))).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, false), { refSheet: 1 })).to.deep.equal({ text: 'C$2', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('2:3', false, false), { refSheet: 1 })).to.deep.equal({ text: '#REF!', refError: true });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('4:4', false, false), { refSheet: 1 })).to.equal(null);
                });
                it('should transform range address', function () {
                    var token = new Tokens.ReferenceToken(docModel, cell('2.$2'), cell('$4.4'), null, null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, true), { refSheet: 1 })).to.deep.equal({ text: 'D$3:$F5', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('D:F', true, true), { refSheet: 1 })).to.deep.equal({ text: 'C$3:$H5', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('F:F', true, true), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, true), { refSheet: 1 })).to.deep.equal({ text: 'C$4:$E6', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('4:6', false, true), { refSheet: 1 })).to.deep.equal({ text: 'C$3:$E8', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('6:6', false, true), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, false), { refSheet: 1 })).to.deep.equal({ text: 'B$3:$D5', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('D:F', true, false), { refSheet: 1 })).to.deep.equal({ text: 'C$3:$C5', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('C:E', true, false), { refSheet: 1 })).to.deep.equal({ text: '#REF!', refError: true });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('F:F', true, false), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, false), { refSheet: 1 })).to.deep.equal({ text: 'C$2:$E4', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('4:6', false, false), { refSheet: 1 })).to.deep.equal({ text: 'C$3:$E3', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('3:5', false, false), { refSheet: 1 })).to.deep.equal({ text: '#REF!', refError: true });
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('6:6', false, false), { refSheet: 1 })).to.equal(null);
                });
                it('should transform references with sheet correctly', function () {
                    var token = new Tokens.ReferenceToken(docModel, cell('2.$2'), null, sh(1), null);
                    var moveDescs = makeMoveDescs('A:A', true, true);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs)).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs, { refSheet: 0 })).to.deep.equal({ text: 'Sheet2!D$3', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs, { refSheet: 1 })).to.deep.equal({ text: 'Sheet2!D$3', refError: false });
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs)).to.deep.equal({ text: 'Sheet2!D$3', refError: false });
                });
                it('should not transform references with sheet range', function () {
                    var token = new Tokens.ReferenceToken(docModel, cell('2.$2'), null, sh(1), sh(3));
                    var moveDescs = makeMoveDescs('A:A', true, true);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs)).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs, { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs, { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs)).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 2, moveDescs, { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 2, moveDescs, { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 2, moveDescs)).to.equal(null);
                });
                it('should not transform error references', function () {
                    var moveDescs = makeMoveDescs('A:A', true, true);
                    expect(errToken.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 0 })).to.equal(null);
                    expect(sheetErrToken.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 0 })).to.equal(null);
                });
            });
        });

        // class NameToken ====================================================

        describe('Spreadsheet class NameToken', function () {
            it('should exist', function () {
                expect(Tokens.NameToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var nameToken, sheet2NameToken, errorNameToken, tableToken, sheet1TableToken, sheet2TableToken, errorTableToken;
            before(function () {
                nameToken = new Tokens.NameToken(docModel, 'name', null);
                sheet2NameToken = new Tokens.NameToken(docModel, 'Name', sh(1));
                errorNameToken = new Tokens.NameToken(docModel, 'NAME', sh(-1));
                tableToken = new Tokens.NameToken(docModel, 'table1', null);
                sheet1TableToken = new Tokens.NameToken(docModel, 'Table1', sh(0));
                sheet2TableToken = new Tokens.NameToken(docModel, 'TABLE1', sh(1));
                errorTableToken = new Tokens.NameToken(docModel, 'Table1', sh(-1));
            });

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.NameToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(nameToken.getType()).to.equal('name');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.NameToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(nameToken.getText(opGrammar)).to.equal('name');
                    expect(nameToken.getText(uiGrammar)).to.equal('name');
                    expect(sheet2NameToken.getText(opGrammar)).to.equal('Sheet2!Name');
                    expect(sheet2NameToken.getText(uiGrammar)).to.equal('Sheet2!Name');
                    expect(errorNameToken.getText(opGrammar)).to.equal('#REF!!NAME');
                    expect(errorNameToken.getText(uiGrammar)).to.equal('#BEZUG!!NAME');
                    expect(tableToken.getText(opGrammar)).to.equal('table1');
                    expect(tableToken.getText(uiGrammar)).to.equal('table1');
                    expect(sheet1TableToken.getText(opGrammar)).to.equal('Sheet1!Table1');
                    expect(sheet1TableToken.getText(uiGrammar)).to.equal('Sheet1!Table1');
                    expect(sheet2TableToken.getText(opGrammar)).to.equal('Sheet2!TABLE1');
                    expect(sheet2TableToken.getText(uiGrammar)).to.equal('Sheet2!TABLE1');
                    expect(errorTableToken.getText(opGrammar)).to.equal('#REF!!Table1');
                    expect(errorTableToken.getText(uiGrammar)).to.equal('#BEZUG!!Table1');
                });
                it('should use a custom label', function () {
                    expect(nameToken.getText(opGrammar, { label: 'new' })).to.equal('new');
                    expect(nameToken.getText(uiGrammar, { label: 'new' })).to.equal('new');
                    expect(sheet2NameToken.getText(opGrammar, { label: 'new' })).to.equal('Sheet2!new');
                    expect(sheet2NameToken.getText(uiGrammar, { label: 'new' })).to.equal('Sheet2!new');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.NameToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(nameToken.toString()).to.equal('name[name]');
                    expect(sheet2NameToken.toString()).to.equal('name[$1!Name]');
                    expect(errorNameToken.toString()).to.equal('name[#REF!NAME]');
                    expect(tableToken.toString()).to.equal('name[table1]');
                    expect(sheet1TableToken.toString()).to.equal('name[$0!Table1]');
                    expect(sheet2TableToken.toString()).to.equal('name[$1!TABLE1]');
                    expect(errorTableToken.toString()).to.equal('name[#REF!Table1]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(Tokens.NameToken).to.respondTo('getValue');
                });
                it('should return the value of the token', function () {
                    expect(nameToken.getValue()).to.equal('name');
                    expect(sheet2NameToken.getValue()).to.equal('Name');
                    expect(errorNameToken.getValue()).to.equal('NAME');
                    expect(tableToken.getValue()).to.equal('table1');
                    expect(sheet1TableToken.getValue()).to.equal('Table1');
                    expect(sheet2TableToken.getValue()).to.equal('TABLE1');
                    expect(errorTableToken.getValue()).to.equal('Table1');
                });
            });

            describe('method "hasSheetRef"', function () {
                it('should exist', function () {
                    expect(Tokens.NameToken).to.respondTo('hasSheetRef');
                });
                it('should return the correct result', function () {
                    expect(nameToken.hasSheetRef()).to.equal(false);
                    expect(sheet2NameToken.hasSheetRef()).to.equal(true);
                    expect(errorNameToken.hasSheetRef()).to.equal(true);
                    expect(tableToken.hasSheetRef()).to.equal(false);
                    expect(sheet1TableToken.hasSheetRef()).to.equal(true);
                    expect(sheet2TableToken.hasSheetRef()).to.equal(true);
                    expect(errorTableToken.hasSheetRef()).to.equal(true);
                });
            });

            describe('method "hasSheetError"', function () {
                it('should exist', function () {
                    expect(Tokens.NameToken).to.respondTo('hasSheetError');
                });
                it('should return the correct result', function () {
                    expect(nameToken.hasSheetError()).to.equal(false);
                    expect(sheet2NameToken.hasSheetError()).to.equal(false);
                    expect(errorNameToken.hasSheetError()).to.equal(true);
                    expect(tableToken.hasSheetError()).to.equal(false);
                    expect(sheet1TableToken.hasSheetError()).to.equal(false);
                    expect(sheet2TableToken.hasSheetError()).to.equal(false);
                    expect(errorTableToken.hasSheetError()).to.equal(true);
                });
            });

            describe('method "resolveNameModel"', function () {
                it('should exist', function () {
                    expect(Tokens.NameToken).to.respondTo('resolveNameModel');
                });
                it('should return globally defined name', function () {
                    var globalName = docModel.getNameCollection().getName('name');
                    expect(nameToken.resolveNameModel(null)).to.equal(globalName);
                    expect(nameToken.resolveNameModel(0)).to.equal(globalName);
                });
                it('should return sheet-local defined name', function () {
                    var localName = docModel.getSheetModel(1).getNameCollection().getName('name');
                    expect(nameToken.resolveNameModel(1)).to.equal(localName);
                    expect(sheet2NameToken.resolveNameModel(null)).to.equal(localName);
                    expect(sheet2NameToken.resolveNameModel(0)).to.equal(localName);
                    expect(sheet2NameToken.resolveNameModel(1)).to.equal(localName);
                });
                it('should return null for unknown names', function () {
                    expect(new Tokens.NameToken(docModel, 'invalid', null).resolveNameModel(0)).to.equal(null);
                    expect(new Tokens.NameToken(docModel, 'invalid', sh(1)).resolveNameModel(0)).to.equal(null);
                    expect(new Tokens.NameToken(docModel, 'name', sh(0)).resolveNameModel(null)).to.equal(null);
                    expect(new Tokens.NameToken(docModel, 'name', sh(0)).resolveNameModel(0)).to.equal(null);
                    expect(new Tokens.NameToken(docModel, 'name', sh(0)).resolveNameModel(1)).to.equal(null);
                });
                it('should return null for token with sheet error', function () {
                    expect(errorNameToken.resolveNameModel(null)).to.equal(null);
                    expect(errorNameToken.resolveNameModel(0)).to.equal(null);
                });
                it('should return null for table tokens', function () {
                    expect(tableToken.resolveNameModel(null)).to.equal(null);
                    expect(sheet1TableToken.resolveNameModel(null)).to.equal(null);
                    expect(sheet2TableToken.resolveNameModel(null)).to.equal(null);
                    expect(errorTableToken.resolveNameModel(null)).to.equal(null);
                });
            });

            describe('method "resolveTableModel"', function () {
                it('should exist', function () {
                    expect(Tokens.NameToken).to.respondTo('resolveTableModel');
                });
                it('should return the table model', function () {
                    var tableModel = docModel.getTable('Table1');
                    expect(tableToken.resolveTableModel()).to.equal(tableModel);
                    expect(sheet1TableToken.resolveTableModel()).to.equal(tableModel);
                    expect(sheet2TableToken.resolveTableModel()).to.equal(null);
                    expect(errorTableToken.resolveTableModel()).to.equal(null);
                });
                it('should return null for name tokens', function () {
                    expect(nameToken.resolveTableModel()).to.equal(null);
                    expect(sheet2NameToken.resolveTableModel()).to.equal(null);
                    expect(errorNameToken.resolveTableModel()).to.equal(null);
                });
            });

            describe('method "transformSheet"', function () {
                it('should exist', function () {
                    expect(Tokens.NameToken).to.respondTo('transformSheet');
                });
                it('should do nothing for tokens without sheet', function () {
                    expect(nameToken.transformSheet(0, null)).to.equal(false);
                    expect(nameToken).to.stringifyTo('name[name]');
                });
                it('should transform sheet index', function () {
                    var token = new Tokens.NameToken(docModel, 'name', sh(1));
                    expect(token).to.stringifyTo('name[$1!name]');
                    expect(token.transformSheet(1, null)).to.equal(true);
                    expect(token).to.stringifyTo('name[$2!name]');
                    expect(token.transformSheet(3, null)).to.equal(false);
                    expect(token).to.stringifyTo('name[$2!name]');
                    expect(token.transformSheet(null, 1)).to.equal(true);
                    expect(token).to.stringifyTo('name[$1!name]');
                    expect(token.transformSheet(null, 2)).to.equal(false);
                    expect(token).to.stringifyTo('name[$1!name]');
                    expect(token.transformSheet(null, 1)).to.equal(true);
                    expect(token).to.stringifyTo('name[#REF!name]');
                });
                it('should do nothing for token with sheet error', function () {
                    expect(errorNameToken.transformSheet(0, null)).to.equal(false);
                    expect(errorNameToken).to.stringifyTo('name[#REF!NAME]');
                });
            });
        });

        // class MacroToken ===================================================

        describe('Spreadsheet class MacroToken', function () {
            it('should exist', function () {
                expect(Tokens.MacroToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var macroToken, sheetToken, errorToken;
            before(function () {
                macroToken = new Tokens.MacroToken(docModel, 'Sum', null);
                sheetToken = new Tokens.MacroToken(docModel, 'Macro', sh(3));
                errorToken = new Tokens.MacroToken(docModel, 'Macro', sh(-1));
            });

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.MacroToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(macroToken.getType()).to.equal('macro');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.MacroToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(macroToken.getText(opGrammar)).to.equal('_xludf.Sum');
                    expect(macroToken.getText(uiGrammar)).to.equal('Sum');
                    expect(sheetToken.getText(opGrammar)).to.equal('\'Sheet 4\'!Macro');
                    expect(sheetToken.getText(uiGrammar)).to.equal('\'Sheet 4\'!Macro');
                    expect(errorToken.getText(opGrammar)).to.equal('#REF!!Macro');
                    expect(errorToken.getText(uiGrammar)).to.equal('#BEZUG!!Macro');
                    var udfToken = new Tokens.MacroToken(docModel, '_xludf.SUM', null);
                    expect(udfToken.getText(opGrammar)).to.equal('_xludf.SUM');
                    expect(udfToken.getText(uiGrammar)).to.equal('SUM');
                });
                it('should use a custom label', function () {
                    expect(macroToken.getText(opGrammar, { label: 'new' })).to.equal('new');
                    expect(macroToken.getText(uiGrammar, { label: 'new' })).to.equal('new');
                    expect(sheetToken.getText(opGrammar, { label: 'new' })).to.equal('\'Sheet 4\'!new');
                    expect(sheetToken.getText(uiGrammar, { label: 'new' })).to.equal('\'Sheet 4\'!new');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.MacroToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(macroToken.toString()).to.equal('macro[Sum]');
                    expect(sheetToken.toString()).to.equal('macro[$3!Macro]');
                    expect(errorToken.toString()).to.equal('macro[#REF!Macro]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(Tokens.MacroToken).to.respondTo('getValue');
                });
                it('should return the value of the token', function () {
                    expect(macroToken.getValue()).to.equal('Sum');
                    expect(sheetToken.getValue()).to.equal('Macro');
                    expect(errorToken.getValue()).to.equal('Macro');
                });
            });

            describe('method "hasSheetRef"', function () {
                it('should exist', function () {
                    expect(Tokens.MacroToken).to.respondTo('hasSheetRef');
                });
                it('should return the correct result', function () {
                    expect(macroToken.hasSheetRef()).to.equal(false);
                    expect(sheetToken.hasSheetRef()).to.equal(true);
                    expect(errorToken.hasSheetRef()).to.equal(true);
                });
            });

            describe('method "hasSheetError"', function () {
                it('should exist', function () {
                    expect(Tokens.MacroToken).to.respondTo('hasSheetError');
                });
                it('should return the correct result', function () {
                    expect(macroToken.hasSheetError()).to.equal(false);
                    expect(sheetToken.hasSheetError()).to.equal(false);
                    expect(errorToken.hasSheetError()).to.equal(true);
                });
            });

            describe('method "transformSheet"', function () {
                it('should exist', function () {
                    expect(Tokens.MacroToken).to.respondTo('transformSheet');
                });
                it('should do nothing for tokens without sheet', function () {
                    expect(macroToken.transformSheet(0, null)).to.equal(false);
                    expect(macroToken).to.stringifyTo('macro[Sum]');
                });
                it('should transform sheet index', function () {
                    var token = new Tokens.MacroToken(docModel, 'Sum', sh(1));
                    expect(token).to.stringifyTo('macro[$1!Sum]');
                    expect(token.transformSheet(1, null)).to.equal(true);
                    expect(token).to.stringifyTo('macro[$2!Sum]');
                    expect(token.transformSheet(3, null)).to.equal(false);
                    expect(token).to.stringifyTo('macro[$2!Sum]');
                    expect(token.transformSheet(null, 1)).to.equal(true);
                    expect(token).to.stringifyTo('macro[$1!Sum]');
                    expect(token.transformSheet(null, 2)).to.equal(false);
                    expect(token).to.stringifyTo('macro[$1!Sum]');
                    expect(token.transformSheet(null, 1)).to.equal(true);
                    expect(token).to.stringifyTo('macro[#REF!Sum]');
                });
                it('should do nothing for token with sheet error', function () {
                    expect(errorToken.transformSheet(0, null)).to.equal(false);
                    expect(errorToken).to.stringifyTo('macro[#REF!Macro]');
                });
            });
        });

        // class TableToken ===================================================

        describe('Spreadsheet class TableToken', function () {
            it('should exist', function () {
                expect(Tokens.TableToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var tableToken;
            before(function () {
                tableToken = new Tokens.TableToken(docModel, 'Table1');
            });

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(Tokens.TableToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(tableToken.getType()).to.equal('table');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(Tokens.TableToken).to.respondTo('getText');
                });
                function testToken(tableName, tableOptions, textOptions, expOp, expUi) {
                    var token = new Tokens.TableToken(docModel, tableName, tableOptions);
                    expect(token.getText(opGrammar, textOptions)).to.equal(expOp);
                    expect(token.getText(uiGrammar, textOptions)).to.equal(expUi);
                }
                it('should return the string representation', function () {
                    testToken('Table1', null, null, 'Table1[]', 'Table1');
                    testToken('Table1', { regionKey: 'ALL' }, null, 'Table1[#All]', 'Table1[#Alle]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA' }, null, 'Table1[[#Headers],[#Data]]', 'Table1[[#Kopfzeilen];[#Daten]]');
                    testToken('Table1', { regionKey: 'DATA,TOTALS' }, null, 'Table1[[#Data],[#Totals]]', 'Table1[[#Daten];[#Ergebnisse]]');
                    testToken('Table1', { col1Name: 'Col 1' }, null, 'Table1[Col 1]', 'Table1[Col 1]');
                    testToken('Table1', { col1Name: ' Col ' }, null, 'Table1[[ Col ]]', 'Table1[[ Col ]]');
                    testToken('Table1', { col1Name: ' ' }, null, 'Table1[[ ]]', 'Table1[[ ]]');
                    testToken('Table1', { col1Name: 'Col1', col2Name: 'Col2' }, null, 'Table1[[Col1]:[Col2]]', 'Table1[[Col1]:[Col2]]');
                    testToken('Table1', { regionKey: 'ALL', col1Name: 'Col1' }, null, 'Table1[[#All],[Col1]]', 'Table1[[#Alle];[Col1]]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA', col1Name: 'Col1', col2Name: 'Col2' }, null, 'Table1[[#Headers],[#Data],[Col1]:[Col2]]', 'Table1[[#Kopfzeilen];[#Daten];[Col1]:[Col2]]');
                    testToken('Table1', { regionKey: 'ROW' }, null, 'Table1[#This Row]', 'Table1[@]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col1' }, null, 'Table1[[#This Row],[Col1]]', 'Table1[@Col1]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col 1' }, null, 'Table1[[#This Row],[Col 1]]', 'Table1[@[Col 1]]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col:1' }, null, 'Table1[[#This Row],[Col:1]]', 'Table1[@[Col:1]]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col1', col2Name: 'Col2' }, null, 'Table1[[#This Row],[Col1]:[Col2]]', 'Table1[@[Col1]:[Col2]]');
                });
                it('should add arbitrary whitespace characters', function () {
                    testToken('Table1', { openWs: true }, null, 'Table1[ ]', 'Table1');
                    testToken('Table1', { regionKey: 'ALL', openWs: true, closeWs: true }, null, 'Table1[ [#All] ]', 'Table1[ [#Alle] ]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA', col1Name: 'Col1', openWs: true }, null, 'Table1[ [#Headers],[#Data],[Col1]]', 'Table1[ [#Kopfzeilen];[#Daten];[Col1]]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA', col1Name: 'Col1', closeWs: true }, null, 'Table1[[#Headers],[#Data],[Col1] ]', 'Table1[[#Kopfzeilen];[#Daten];[Col1] ]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA', col1Name: 'Col1', sepWs: true }, null, 'Table1[[#Headers], [#Data], [Col1]]', 'Table1[[#Kopfzeilen]; [#Daten]; [Col1]]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA', col1Name: 'Col1', openWs: true, closeWs: true, sepWs: true }, null, 'Table1[ [#Headers], [#Data], [Col1] ]', 'Table1[ [#Kopfzeilen]; [#Daten]; [Col1] ]');
                    testToken('Table1', { col1Name: 'Col 1', openWs: true, closeWs: true }, null, 'Table1[ [Col 1] ]', 'Table1[ [Col 1] ]');
                    testToken('Table1', { col1Name: ' Col ', openWs: true, closeWs: true }, null, 'Table1[ [ Col ] ]', 'Table1[ [ Col ] ]');
                    testToken('Table1', { col1Name: ' ', openWs: true, closeWs: true }, null, 'Table1[ [ ] ]', 'Table1[ [ ] ]');
                    testToken('Table1', { col1Name: 'Col1', col2Name: 'Col2', openWs: true, closeWs: true }, null, 'Table1[ [Col1]:[Col2] ]', 'Table1[ [Col1]:[Col2] ]');
                    testToken('Table1', { regionKey: 'ROW', openWs: true, closeWs: true }, null, 'Table1[ [#This Row] ]', 'Table1[ @ ]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col1', openWs: true, closeWs: true, sepWs: true }, null, 'Table1[ [#This Row], [Col1] ]', 'Table1[ @Col1 ]');
                });
                it('should omit table name in specific situations', function () {
                    testToken('Table1', { col1Name: 'Col1' }, { refAddress: a('B3'), targetAddress: a('B3'), refSheet: 0 }, 'Table1[Col1]', 'Table1[Col1]');
                    testToken('Table1', { col1Name: 'Col1' }, { refAddress: a('B3'), targetAddress: a('B3'), refSheet: 0, unqualifiedTables: true }, 'Table1[Col1]', '[Col1]');
                    testToken('Table1', { col1Name: 'Col1' }, { refAddress: a('A1'), targetAddress: a('A1'), refSheet: 0, unqualifiedTables: true }, 'Table1[Col1]', 'Table1[Col1]');
                    testToken('Table1', { col1Name: 'Col1' }, { refAddress: a('B3'), targetAddress: a('B3'), refSheet: 1, unqualifiedTables: true }, 'Table1[Col1]', 'Table1[Col1]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col1' }, { refAddress: a('B3'), targetAddress: a('B3'), refSheet: 0, unqualifiedTables: true }, 'Table1[[#This Row],[Col1]]', '[@Col1]');
                });
                it('should use a custom table name', function () {
                    testToken('Table1', { regionKey: 'ALL' }, { tableName: 'New' }, 'New[#All]', 'New[#Alle]');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(Tokens.TableToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(tableToken.toString()).to.equal('table[Table1]');
                });
            });

            describe('method "getTableName"', function () {
                it('should exist', function () {
                    expect(Tokens.TableToken).to.respondTo('getTableName');
                });
                it('should return the table name', function () {
                    expect(tableToken.getTableName()).to.equal('Table1');
                });
            });

            describe('method "resolveTableModel"', function () {
                it('should exist', function () {
                    expect(Tokens.TableToken).to.respondTo('resolveTableModel');
                });
                it('should return the table model', function () {
                    var tableModel = docModel.getTable('Table1');
                    expect(tableToken.resolveTableModel()).to.equal(tableModel);
                });
                it('should return null for invalid table name', function () {
                    expect(new Tokens.TableToken(docModel, '__invalid__').resolveTableModel()).to.equal(null);
                });
            });

            describe('method "getRange3D"', function () {
                it('should exist', function () {
                    expect(Tokens.TableToken).to.respondTo('getRange3D');
                });
                it('should return the range', function () {
                    expect(tableToken.getRange3D(a('A1'))).to.stringifyTo('0:0!B3:D4');
                    expect(tableToken.getRange3D(a('B2'))).to.stringifyTo('0:0!B3:D4');
                });
                it('should return the correct table region', function () {
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'ALL' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B2:D5');
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'HEADERS' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B2:D2');
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'DATA' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B3:D4');
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'TOTALS' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B5:D5');
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'HEADERS,DATA' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B2:D4');
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'DATA,TOTALS' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B3:D5');
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B1'))).to.equal(null);
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B2'))).to.equal(null);
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B3'))).to.stringifyTo('0:0!B3:D3');
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B4'))).to.stringifyTo('0:0!B4:D4');
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B5'))).to.equal(null);
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B6'))).to.equal(null);
                });
                it('should return the correct table columns', function () {
                    expect(new Tokens.TableToken(docModel, 'Table1', { col1Name: 'Col1' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B3:B4');
                    expect(new Tokens.TableToken(docModel, 'Table1', { col1Name: 'Col2', col2Name: 'Col3' }).getRange3D(a('B2'))).to.stringifyTo('0:0!C3:D4');
                    expect(new Tokens.TableToken(docModel, 'Table1', { col1Name: 'Col3', col2Name: 'Col1' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B3:D4');
                });
                it('should return the correct combined ranges', function () {
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'ALL', col1Name: 'Col1' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B2:B5');
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'ROW', col1Name: 'Col1', col2Name: 'Col2' }).getRange3D(a('B2'))).to.equal(null);
                    expect(new Tokens.TableToken(docModel, 'Table1', { regionKey: 'ROW', col1Name: 'Col1', col2Name: 'Col2' }).getRange3D(a('B4'))).to.stringifyTo('0:0!B4:C4');
                });
                it('should return null for invalid table name', function () {
                    expect(new Tokens.TableToken(docModel, '__invalid__').getRange3D(a('B2'))).to.equal(null);
                });
            });
        });
    });

    // ========================================================================
});
