/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/model/formula/parser/tokens',
    'io.ox/office/spreadsheet/model/formula/parser/formulaparser'
], function (AppHelper, SheetHelper, RangeArray, Tokens, FormulaParser) {

    'use strict';

    // convenience shortcuts
    var ra = SheetHelper.ra;

    // private global functions ===============================================

    function expectResult(result, expected) {
        expect(result).to.be.an('array');
        expect(_.pluck(result, 'token').join(' ')).to.equal(expected);
    }

    // class FormulaParser ====================================================

    describe('Spreadsheet class FormulaParser', function () {

        it('should exist', function () {
            expect(FormulaParser).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the OOXML document model
        var OOX_OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet 4' }, // Sheet name with white-space
            { name: 'insertName', label: 'name', formula: 'Sheet1!$A$1' }, // global name
            { name: 'insertName', sheet: 1, label: 'name', formula: 'Sheet1!$A$1' } // sheet-local name
        ];

        // the operations to be applied by the ODF document model
        var ODF_OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 1024, rows: 1048576 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet 4' }, // Sheet name with white-space
            { name: 'insertName', label: 'name', formula: '[Sheet1.$A$1]', ref: '[Sheet1.$A$1]' } // global name
        ];

        // prepare OOXML test document
        var ooxDocModel = null, ooxParser = null;
        AppHelper.createSpreadsheetApp('ooxml', OOX_OPERATIONS).done(function (app) {
            ooxDocModel = app.getModel();
            ooxParser = ooxDocModel.getFormulaParser();
        });

        // prepare ODF test document
        var odfDocModel = null, odfParser = null;
        AppHelper.createSpreadsheetApp('odf', ODF_OPERATIONS).done(function (app) {
            odfDocModel = app.getModel();
            odfParser = odfDocModel.getFormulaParser();
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a formula parser', function () {
                expect(ooxParser).to.be.an.instanceof(FormulaParser);
                expect(odfParser).to.be.an.instanceof(FormulaParser);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "parseFormula"', function () {
            it('should exist', function () {
                expect(ooxParser).to.respondTo('parseFormula');
            });
            it('should return an array of token descriptors', function () {
                var result = ooxParser.parseFormula('op', '42+012.50 *-.01');
                expect(result).to.be.an('array').and.to.have.length(6);
                expect(result[0]).to.have.a.property('token').that.is.an.instanceof(Tokens.LiteralToken);
                expect(result[0].token.getValue()).to.equal(42);
                expect(result[0]).to.have.a.property('index', 0);
                expect(result[0]).to.have.a.property('text', '42');
                expect(result[0]).to.have.a.property('start', 0);
                expect(result[0]).to.have.a.property('end', 2);
                expect(result[1]).to.have.a.property('token').that.is.an.instanceof(Tokens.OperatorToken);
                expect(result[1].token.getValue()).to.equal('add');
                expect(result[1]).to.have.a.property('index', 1);
                expect(result[1]).to.have.a.property('text', '+');
                expect(result[1]).to.have.a.property('start', 2);
                expect(result[1]).to.have.a.property('end', 3);
                expect(result[2]).to.have.a.property('token').that.is.an.instanceof(Tokens.LiteralToken);
                expect(result[2].token.getValue()).to.equal(12.5);
                expect(result[2]).to.have.a.property('index', 2);
                expect(result[2]).to.have.a.property('text', '012.50');
                expect(result[2]).to.have.a.property('start', 3);
                expect(result[2]).to.have.a.property('end', 9);
                expect(result[3]).to.have.a.property('token').that.is.an.instanceof(Tokens.FixedToken);
                expect(result[3].token.getType()).to.equal('ws');
                expect(result[3].token.getValue()).to.equal(' ');
                expect(result[3]).to.have.a.property('index', 3);
                expect(result[3]).to.have.a.property('text', ' ');
                expect(result[3]).to.have.a.property('start', 9);
                expect(result[3]).to.have.a.property('end', 10);
                expect(result[4]).to.have.a.property('token').that.is.an.instanceof(Tokens.OperatorToken);
                expect(result[4].token.getValue()).to.equal('mul');
                expect(result[4]).to.have.a.property('index', 4);
                expect(result[4]).to.have.a.property('text', '*');
                expect(result[4]).to.have.a.property('start', 10);
                expect(result[4]).to.have.a.property('end', 11);
                expect(result[5]).to.have.a.property('token').that.is.an.instanceof(Tokens.LiteralToken);
                expect(result[5].token.getValue()).to.equal(-0.01);
                expect(result[5]).to.have.a.property('index', 5);
                expect(result[5]).to.have.a.property('text', '-.01');
                expect(result[5]).to.have.a.property('start', 11);
                expect(result[5]).to.have.a.property('end', 15);
                expectResult(result, 'lit[42] op[add] lit[12.5] ws[ ] op[mul] lit[-0.01]');
            });

            it('should parse number literals and numeric operators', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', '1+2%%'), 'lit[1] op[add] lit[2] op[pct] op[pct]');
                expectResult(ooxParser.parseFormula('op', '0042-4200'), 'lit[42] op[sub] lit[4200]');
                expectResult(ooxParser.parseFormula('op', '.010*010.'), 'lit[0.01] op[mul] lit[10]');
                expectResult(ooxParser.parseFormula('op', '077.880/0.0'), 'lit[77.88] op[div] lit[0]');
                expectResult(ooxParser.parseFormula('op', '1e30^001.100E-0030^0e0'), 'lit[1e+30] op[pow] lit[1.1e-30] op[pow] lit[0]');
                expectResult(ooxParser.parseFormula('op', '.01e+30+10.E-30-10.01e30'), 'lit[1e+28] op[add] lit[1e-29] op[sub] lit[1.001e+31]');
                expectResult(ooxParser.parseFormula('op', '1 2/4+0 0/2+02 03/04'), 'lit[1.5] op[add] lit[0] op[add] lit[2.75]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', '1+2%%'), 'lit[1] op[add] lit[2] op[pct] op[pct]');
                expectResult(ooxParser.parseFormula('ui', '0042-4200'), 'lit[42] op[sub] lit[4200]');
                expectResult(ooxParser.parseFormula('ui', ',010*010,'), 'lit[0.01] op[mul] lit[10]');
                expectResult(ooxParser.parseFormula('ui', '077,880/0,0'), 'lit[77.88] op[div] lit[0]');
                expectResult(ooxParser.parseFormula('ui', '1e30^001,100E-0030^0e0'), 'lit[1e+30] op[pow] lit[1.1e-30] op[pow] lit[0]');
                expectResult(ooxParser.parseFormula('ui', ',01e+30+10,E-30-10,01e30'), 'lit[1e+28] op[add] lit[1e-29] op[sub] lit[1.001e+31]');
                expectResult(ooxParser.parseFormula('ui', '1 2/4+0 0/2+02 03/04'), 'lit[1.5] op[add] lit[0] op[add] lit[2.75]');
                // native ODF
                expectResult(odfParser.parseFormula('op', '1+2%%'), 'lit[1] op[add] lit[2] op[pct] op[pct]');
                expectResult(odfParser.parseFormula('op', '0042-4200'), 'lit[42] op[sub] lit[4200]');
                expectResult(odfParser.parseFormula('op', '.010*010.'), 'lit[0.01] op[mul] lit[10]');
                expectResult(odfParser.parseFormula('op', '077.880/0.0'), 'lit[77.88] op[div] lit[0]');
                expectResult(odfParser.parseFormula('op', '1e30^001.100E-0030^0e0'), 'lit[1e+30] op[pow] lit[1.1e-30] op[pow] lit[0]');
                expectResult(odfParser.parseFormula('op', '.01e+30+10.E-30-10.01e30'), 'lit[1e+28] op[add] lit[1e-29] op[sub] lit[1.001e+31]');
                expectResult(odfParser.parseFormula('op', '1 2/4+0 0/2+02 03/04'), 'lit[1.5] op[add] lit[0] op[add] lit[2.75]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', '1+2%%'), 'lit[1] op[add] lit[2] op[pct] op[pct]');
                expectResult(odfParser.parseFormula('ui', '0042-4200'), 'lit[42] op[sub] lit[4200]');
                expectResult(odfParser.parseFormula('ui', ',010*010,'), 'lit[0.01] op[mul] lit[10]');
                expectResult(odfParser.parseFormula('ui', '077,880/0,0'), 'lit[77.88] op[div] lit[0]');
                expectResult(odfParser.parseFormula('ui', '1e30^001,100E-0030^0e0'), 'lit[1e+30] op[pow] lit[1.1e-30] op[pow] lit[0]');
                expectResult(odfParser.parseFormula('ui', ',01e+30+10,E-30-10,01e30'), 'lit[1e+28] op[add] lit[1e-29] op[sub] lit[1.001e+31]');
                expectResult(odfParser.parseFormula('ui', '1 2/4+0 0/2+02 03/04'), 'lit[1.5] op[add] lit[0] op[add] lit[2.75]');
            });

            it('should parse boolean literals and comparison operators', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', 'true<false>True<>False'), 'lit[TRUE] op[lt] lit[FALSE] op[gt] lit[TRUE] op[ne] lit[FALSE]');
                expectResult(ooxParser.parseFormula('op', 'tRUE<=fALSE>=TRUE=FALSE'), 'lit[TRUE] op[le] lit[FALSE] op[ge] lit[TRUE] op[eq] lit[FALSE]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', 'wahr<falsch>Wahr<>Falsch'), 'lit[TRUE] op[lt] lit[FALSE] op[gt] lit[TRUE] op[ne] lit[FALSE]');
                expectResult(ooxParser.parseFormula('ui', 'wAHR<=fALSCH>=WAHR=FALSCH'), 'lit[TRUE] op[le] lit[FALSE] op[ge] lit[TRUE] op[eq] lit[FALSE]');
                // native ODF
                expectResult(ooxParser.parseFormula('op', 'true<false>True<>False'), 'lit[TRUE] op[lt] lit[FALSE] op[gt] lit[TRUE] op[ne] lit[FALSE]');
                expectResult(ooxParser.parseFormula('op', 'tRUE<=fALSE>=TRUE=FALSE'), 'lit[TRUE] op[le] lit[FALSE] op[ge] lit[TRUE] op[eq] lit[FALSE]');
                // localized ODF
                expectResult(ooxParser.parseFormula('ui', 'wahr<falsch>Wahr<>Falsch'), 'lit[TRUE] op[lt] lit[FALSE] op[gt] lit[TRUE] op[ne] lit[FALSE]');
                expectResult(ooxParser.parseFormula('ui', 'wAHR<=fALSCH>=WAHR=FALSCH'), 'lit[TRUE] op[le] lit[FALSE] op[ge] lit[TRUE] op[eq] lit[FALSE]');
            });

            it('should parse string literals and string operators', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', '"abc"&"D""E""F"&"&""&"&""'), 'lit["abc"] op[con] lit["D""E""F"] op[con] lit["&""&"] op[con] lit[""]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', '"abc"&"D""E""F"&"&""&"&""'), 'lit["abc"] op[con] lit["D""E""F"] op[con] lit["&""&"] op[con] lit[""]');
                // native ODF
                expectResult(odfParser.parseFormula('op', '"abc"&"D""E""F"&"&""&"&""'), 'lit["abc"] op[con] lit["D""E""F"] op[con] lit["&""&"] op[con] lit[""]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', '"abc"&"D""E""F"&"&""&"&""'), 'lit["abc"] op[con] lit["D""E""F"] op[con] lit["&""&"] op[con] lit[""]');
            });

            it('should parse error code literals', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', '#NULL!#DIV/0!#ref!#NAME?'), 'lit[#NULL] lit[#DIV0] lit[#REF] lit[#NAME]');
                expectResult(ooxParser.parseFormula('op', '#VALUE!#NUM!#num!#N/A'), 'lit[#VALUE] lit[#NUM] lit[#NUM] lit[#NA]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', '#NULL!#DIV/0!#bezug!#NAME?'), 'lit[#NULL] lit[#DIV0] lit[#REF] lit[#NAME]');
                expectResult(ooxParser.parseFormula('ui', '#WERT!#ZAHL!#zahl!#NV'), 'lit[#VALUE] lit[#NUM] lit[#NUM] lit[#NA]');
                // native ODF
                expectResult(odfParser.parseFormula('op', '#NULL!#DIV/0!#ref!#NAME?'), 'lit[#NULL] lit[#DIV0] lit[#REF] lit[#NAME]');
                expectResult(odfParser.parseFormula('op', '#VALUE!#NUM!#num!#N/A'), 'lit[#VALUE] lit[#NUM] lit[#NUM] lit[#NA]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', '#NULL!#DIV/0!#bezug!#NAME?'), 'lit[#NULL] lit[#DIV0] lit[#REF] lit[#NAME]');
                expectResult(odfParser.parseFormula('ui', '#WERT!#ZAHL!#zahl!#NV'), 'lit[#VALUE] lit[#NUM] lit[#NUM] lit[#NA]');
            });

            it('should parse matrix literals', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', '{1,0042;+1,-0042}'), 'mat[{1;42|1;-42}]');
                expectResult(ooxParser.parseFormula('op', '{.010,-010.;077.880,+0.0}'), 'mat[{0.01;-10|77.88;0}]');
                expectResult(ooxParser.parseFormula('op', '{1e30,+001.100E-0030;-0e0,-.01e+30}'), 'mat[{1e+30;1.1e-30|0;-1e+28}]');
                expectResult(ooxParser.parseFormula('op', '{true,fAlSe;"abc","D""E";","";""|",""}'), 'mat[{TRUE;FALSE|"abc";"D""E"|","";""|";""}]');
                expectResult(ooxParser.parseFormula('op', '{#VALUE!,#num!}'), 'mat[{#VALUE;#NUM}]');
                expectResult(ooxParser.parseFormula('op', '{ 1 , 2 ; 3 , 4 }'), 'mat[{1;2|3;4}]');
                expectResult(ooxParser.parseFormula('op', '{1,2;3,4'), 'bad[{1,2;3,4]');
                expectResult(ooxParser.parseFormula('op', '{1,2;3,4', { autoCorrect: true }), 'mat[{1;2|3;4}]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', '{1;0042|+1;-0042}'), 'mat[{1;42|1;-42}]');
                expectResult(ooxParser.parseFormula('ui', '{,010;-010,|077,880;+0,0}'), 'mat[{0.01;-10|77.88;0}]');
                expectResult(ooxParser.parseFormula('ui', '{1e30;+001,100E-0030|-0e0;-,01e+30}'), 'mat[{1e+30;1.1e-30|0;-1e+28}]');
                expectResult(ooxParser.parseFormula('ui', '{wahr;fAlScH|"abc";"D""E"|","";""|";""}'), 'mat[{TRUE;FALSE|"abc";"D""E"|","";""|";""}]');
                expectResult(ooxParser.parseFormula('ui', '{#WERT!;#zahl!}'), 'mat[{#VALUE;#NUM}]');
                expectResult(ooxParser.parseFormula('ui', '{ 1 ; 2 | 3 ; 4 }'), 'mat[{1;2|3;4}]');
                expectResult(ooxParser.parseFormula('ui', '{1;2|3;4'), 'bad[{1;2|3;4]');
                expectResult(ooxParser.parseFormula('ui', '{1;2|3;4', { autoCorrect: true }), 'mat[{1;2|3;4}]');
                // native ODF
                expectResult(odfParser.parseFormula('op', '{1;0042|+1;-0042}'), 'mat[{1;42|1;-42}]');
                expectResult(odfParser.parseFormula('op', '{.010;-010.|077.880;+0.0}'), 'mat[{0.01;-10|77.88;0}]');
                expectResult(odfParser.parseFormula('op', '{1e30;+001.100E-0030|-0e0;-.01e+30}'), 'mat[{1e+30;1.1e-30|0;-1e+28}]');
                expectResult(odfParser.parseFormula('op', '{true;fAlSe|"abc";"D""E"|","";""|";""}'), 'mat[{TRUE;FALSE|"abc";"D""E"|","";""|";""}]');
                expectResult(odfParser.parseFormula('op', '{#VALUE!;#num!}'), 'mat[{#VALUE;#NUM}]');
                expectResult(odfParser.parseFormula('op', '{ 1 ; 2 | 3 ; 4 }'), 'mat[{1;2|3;4}]');
                expectResult(odfParser.parseFormula('op', '{1;2|3;4'), 'bad[{1;2|3;4]');
                expectResult(odfParser.parseFormula('op', '{1;2|3;4', { autoCorrect: true }), 'mat[{1;2|3;4}]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', '{1;0042|+1;-0042}'), 'mat[{1;42|1;-42}]');
                expectResult(odfParser.parseFormula('ui', '{,010;-010,|077,880;+0,0}'), 'mat[{0.01;-10|77.88;0}]');
                expectResult(odfParser.parseFormula('ui', '{1e30;+001,100E-0030|-0e0;-,01e+30}'), 'mat[{1e+30;1.1e-30|0;-1e+28}]');
                expectResult(odfParser.parseFormula('ui', '{wahr;fAlScH|"abc";"D""E"|","";""|";""}'), 'mat[{TRUE;FALSE|"abc";"D""E"|","";""|";""}]');
                expectResult(odfParser.parseFormula('ui', '{#WERT!;#zahl!}'), 'mat[{#VALUE;#NUM}]');
                expectResult(odfParser.parseFormula('ui', '{ 1 ; 2 | 3 ; 4 }'), 'mat[{1;2|3;4}]');
                expectResult(odfParser.parseFormula('ui', '{1;2|3;4'), 'bad[{1;2|3;4]');
                expectResult(odfParser.parseFormula('ui', '{1;2|3;4', { autoCorrect: true }), 'mat[{1;2|3;4}]');
            });

            it('should parse local cell references and separators', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', 'A1,xfd001048576'), 'ref[A1] sep ref[XFD1048576]');
                expectResult(ooxParser.parseFormula('op', 'B3,B$3 $B3\n  $B$3'), 'ref[B3] sep ref[B$3] op[isect] ref[$B3] ws[\n] op[isect] ws[ ] ref[$B$3]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', 'A1;xfd001048576'), 'ref[A1] sep ref[XFD1048576]');
                expectResult(ooxParser.parseFormula('ui', 'B3;B$3;$B3 $B$3'), 'ref[B3] sep ref[B$3] sep ref[$B3] op[isect] ref[$B$3]');
                // native ODF
                expectResult(odfParser.parseFormula('op', '[.A1];[.amj001048576]'), 'ref[A1] sep ref[AMJ1048576]');
                expectResult(odfParser.parseFormula('op', '[.B3];[.B$3]~[.$B3]![.$B$3]'), 'ref[B3] sep ref[B$3] op[list] ref[$B3] op[isect] ref[$B$3]');
                expectResult(odfParser.parseFormula('op', '[.B#REF!];[.#REF!$3];[.$B3:.$#REF!$3]'), 'ref[#REF] sep ref[#REF] sep ref[#REF]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', 'A1;amj001048576'), 'ref[A1] sep ref[AMJ1048576]');
                expectResult(odfParser.parseFormula('ui', 'B3;B$3;$B3 $B$3'), 'ref[B3] sep ref[B$3] sep ref[$B3] op[isect] ref[$B$3]');
            });

            it('should parse local range references and range operators', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', 'B3:D7:B7:D3:D3:B7:D7:B3'), 'ref[B3:D7] op[range] ref[B3:D7] op[range] ref[B3:D7] op[range] ref[B3:D7]');
                expectResult(ooxParser.parseFormula('op', 'B$3:$D7:d7:$b$3:$B$3:$D$7'), 'ref[B$3:$D7] op[range] ref[$B$3:D7] op[range] ref[$B$3:$D$7]');
                expectResult(ooxParser.parseFormula('op', 'B3:B3:$B3:B3'), 'ref[B3:B3] op[range] ref[$B3:B3]');
                expectResult(ooxParser.parseFormula('op', 'B3:B3:B3,B3'), 'ref[B3:B3] op[range] ref[B3] sep ref[B3]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', 'B3:D7:B7:D3:D3:B7:D7:B3'), 'ref[B3:D7] op[range] ref[B3:D7] op[range] ref[B3:D7] op[range] ref[B3:D7]');
                expectResult(ooxParser.parseFormula('ui', 'B$3:$D7:d7:$b$3:$B$3:$D$7'), 'ref[B$3:$D7] op[range] ref[$B$3:D7] op[range] ref[$B$3:$D$7]');
                expectResult(ooxParser.parseFormula('ui', 'B3:B3:$B3:B3'), 'ref[B3:B3] op[range] ref[$B3:B3]');
                expectResult(ooxParser.parseFormula('ui', 'B3:B3:B3;B3'), 'ref[B3:B3] op[range] ref[B3] sep ref[B3]');
                // native ODF
                expectResult(odfParser.parseFormula('op', '[.B3:.D7]:[.B7:.D3]:[.D3:.B7]:[.D7:.B3]'), 'ref[B3:D7] op[range] ref[B3:D7] op[range] ref[B3:D7] op[range] ref[B3:D7]');
                expectResult(odfParser.parseFormula('op', '[.B$3:.$D7]:[.d7:.$b$3]:[.$B$3:.$D$7]'), 'ref[B$3:$D7] op[range] ref[$B$3:D7] op[range] ref[$B$3:$D$7]');
                expectResult(odfParser.parseFormula('op', '[.B3:.B3]:[.$B3:.B3]'), 'ref[B3:B3] op[range] ref[$B3:B3]');
                expectResult(odfParser.parseFormula('op', '[.B3:.B3]:[.B3];[.B3]'), 'ref[B3:B3] op[range] ref[B3] sep ref[B3]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', 'B3:D7:B7:D3:D3:B7:D7:B3'), 'ref[B3:D7] op[range] ref[B3:D7] op[range] ref[B3:D7] op[range] ref[B3:D7]');
                expectResult(odfParser.parseFormula('ui', 'B$3:$D7:d7:$b$3:$B$3:$D$7'), 'ref[B$3:$D7] op[range] ref[$B$3:D7] op[range] ref[$B$3:$D$7]');
                expectResult(odfParser.parseFormula('ui', 'B3:B3:$B3:B3'), 'ref[B3:B3] op[range] ref[$B3:B3]');
                expectResult(odfParser.parseFormula('ui', 'B3:B3:B3;B3'), 'ref[B3:B3] op[range] ref[B3] sep ref[B3]');
            });

            it('should parse local column/row references', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', 'A:xfd:A:$B:$A:B:$b:$a'), 'ref[A$1:XFD$1048576] op[range] ref[A$1:$B$1048576] op[range] ref[$A$1:B$1048576] op[range] ref[$A$1:$B$1048576]');
                expectResult(ooxParser.parseFormula('op', '1:001048576:1:$2:$1:2:$002:$001'), 'ref[$A1:$XFD1048576] op[range] ref[$A1:$XFD$2] op[range] ref[$A$1:$XFD2] op[range] ref[$A$1:$XFD$2]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', 'A:xfd:A:$B:$A:B:$b:$a'), 'ref[A$1:XFD$1048576] op[range] ref[A$1:$B$1048576] op[range] ref[$A$1:B$1048576] op[range] ref[$A$1:$B$1048576]');
                expectResult(ooxParser.parseFormula('ui', '1:001048576:1:$2:$1:2:$002:$001'), 'ref[$A1:$XFD1048576] op[range] ref[$A1:$XFD$2] op[range] ref[$A$1:$XFD2] op[range] ref[$A$1:$XFD$2]');
                // native ODF
                // ... not available
                // localized ODF
                expectResult(odfParser.parseFormula('ui', 'A:amj:A:$B:$A:B:$b:$a'), 'ref[A$1:AMJ$1048576] op[range] ref[A$1:$B$1048576] op[range] ref[$A$1:B$1048576] op[range] ref[$A$1:$B$1048576]');
                expectResult(odfParser.parseFormula('ui', '1:001048576:1:$2:$1:2:$002:$001'), 'ref[$A1:$AMJ1048576] op[range] ref[$A1:$AMJ$2] op[range] ref[$A$1:$AMJ2] op[range] ref[$A$1:$AMJ$2]');
            });

            it('should parse cell and range references with sheets', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', 'A1:Sheet1!A1:\'Sheet 4\'!xfd001048576'), 'ref[A1] op[range] ref[$0!A1] op[range] ref[$3!XFD1048576]');
                expectResult(ooxParser.parseFormula('op', 'A1:Sheet1:Sheet3!A$1:$C2:\'Sheet 4:Sheet1\'!$a$1:$c$2'), 'ref[A1] op[range] ref[$0:$2!A$1:$C2] op[range] ref[$0:$3!$A$1:$C$2]');
                expectResult(ooxParser.parseFormula('op', 'A1:A1!A1:Sheet1:A1!A1:A1:Sheet1:A1!A1'), 'ref[A1] op[range] ref[$\'A1\'!A1] op[range] ref[$0:$\'A1\'!A1:A1] op[range] ref[$0:$\'A1\'!A1]');
                expectResult(ooxParser.parseFormula('op', 'TRUE!A1:WAHR!A1'), 'ref[$\'TRUE\'!A1] op[range] ref[$\'WAHR\'!A1]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', 'A1:Sheet1!A1:\'Sheet 4\'!xfd001048576'), 'ref[A1] op[range] ref[$0!A1] op[range] ref[$3!XFD1048576]');
                expectResult(ooxParser.parseFormula('ui', 'A1:Sheet1:Sheet3!A$1:$C2:\'Sheet 4:Sheet1\'!$a$1:$c$2'), 'ref[A1] op[range] ref[$0:$2!A$1:$C2] op[range] ref[$0:$3!$A$1:$C$2]');
                expectResult(ooxParser.parseFormula('ui', 'A1:A1!A1:Sheet1:A1!A1:A1:Sheet1:A1!A1'), 'ref[A1] op[range] ref[$\'A1\'!A1] op[range] ref[$0:$\'A1\'!A1:A1] op[range] ref[$0:$\'A1\'!A1]');
                expectResult(ooxParser.parseFormula('ui', 'TRUE!A1:WAHR!A1'), 'ref[$\'TRUE\'!A1] op[range] ref[$\'WAHR\'!A1]');
                // native ODF
                expectResult(odfParser.parseFormula('op', '[.A1]:[Sheet1.A1]:[$\'Sheet 4\'.amj001048576]'), 'ref[A1] op[range] ref[0!A1] op[range] ref[$3!AMJ1048576]');
                expectResult(odfParser.parseFormula('op', '[.A1]:[$Sheet1.A$1:Sheet3.$C2]:[\'Sheet 4\'.$a$1:$Sheet1.$c$2]'), 'ref[A1] op[range] ref[$0:2!A$1:$C2] op[range] ref[$0:3!$A$1:$C$2]');
                expectResult(odfParser.parseFormula('op', '[.A1]:[A1.A1]:[Sheet1.A1:A1.A1]:[Sheet1.A1:A1.A1]'), 'ref[A1] op[range] ref[\'A1\'!A1] op[range] ref[0:\'A1\'!A1:A1] op[range] ref[0:\'A1\'!A1:A1]');
                expectResult(odfParser.parseFormula('op', '[TRUE.A1]:[WAHR.A1]'), 'ref[\'TRUE\'!A1] op[range] ref[\'WAHR\'!A1]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', 'A1:Sheet1!A1:$\'Sheet 4\'!amj001048576'), 'ref[A1] op[range] ref[0!A1] op[range] ref[$3!AMJ1048576]');
                expectResult(odfParser.parseFormula('ui', 'A1:$Sheet1:Sheet3!A$1:$C2:\'Sheet 4\':$Sheet1!$a$1:$c$2'), 'ref[A1] op[range] ref[$0:2!A$1:$C2] op[range] ref[$0:3!$A$1:$C$2]');
                expectResult(odfParser.parseFormula('ui', 'A1:A1!A1:Sheet1:A1!A1:A1:Sheet1:A1!A1'), 'ref[A1] op[range] ref[\'A1\'!A1] op[range] ref[0:\'A1\'!A1:A1] op[range] ref[0:\'A1\'!A1]');
                expectResult(odfParser.parseFormula('ui', 'TRUE!A1:WAHR!A1'), 'ref[\'TRUE\'!A1] op[range] ref[\'WAHR\'!A1]');
            });

            it('should parse column/row references with sheets', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', 'A:XFD:Sheet1!A:$B:\'Sheet 4:Sheet1\'!$b:$a'), 'ref[A$1:XFD$1048576] op[range] ref[$0!A$1:$B$1048576] op[range] ref[$0:$3!$A$1:$B$1048576]');
                expectResult(ooxParser.parseFormula('op', '1:001048576+1:$2+$1:2+$002:$001'), 'ref[$A1:$XFD1048576] op[add] ref[$A1:$XFD$2] op[add] ref[$A$1:$XFD2] op[add] ref[$A$1:$XFD$2]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', 'A:XFD:Sheet1!A:$B:\'Sheet 4:Sheet1\'!$b:$a'), 'ref[A$1:XFD$1048576] op[range] ref[$0!A$1:$B$1048576] op[range] ref[$0:$3!$A$1:$B$1048576]');
                expectResult(ooxParser.parseFormula('ui', '1:001048576+1:$2+$1:2+$002:$001'), 'ref[$A1:$XFD1048576] op[add] ref[$A1:$XFD$2] op[add] ref[$A$1:$XFD2] op[add] ref[$A$1:$XFD$2]');
                // native ODF
                // ... not available
                // localized ODF
                expectResult(odfParser.parseFormula('ui', 'A:AMJ:Sheet1!A:$B:$\'Sheet 4\':Sheet1!$b:$a'), 'ref[A$1:AMJ$1048576] op[range] ref[0!A$1:$B$1048576] op[range] ref[0:$3!$A$1:$B$1048576]');
                expectResult(odfParser.parseFormula('ui', '1:001048576+1:$2+$1:2+$002:$001'), 'ref[$A1:$AMJ1048576] op[add] ref[$A1:$AMJ$2] op[add] ref[$A$1:$AMJ2] op[add] ref[$A$1:$AMJ$2]');
            });

            it('should not accept oversized cell addresses', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', 'A1048757+XFE1+Sheet1!A1:XFE1'), 'name[A1048757] op[add] name[XFE1] op[add] ref[$0!A1] op[range] name[XFE1]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', 'A1048757+XFE1+Sheet1!A1:XFE1'), 'name[A1048757] op[add] name[XFE1] op[add] ref[$0!A1] op[range] name[XFE1]');
                // native ODF
                expectResult(odfParser.parseFormula('op', '[.A1048757]+[.AMK1]+[Sheet1.A1:.XFE1]'), 'ref[#REF] op[add] ref[#REF] op[add] ref[0!#REF]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', 'A1048757+AMK1+Sheet1!A1:AMK1'), 'name[A1048757] op[add] name[AMK1] op[add] ref[0!A1] op[range] name[AMK1]');
            });

            it('should parse function names', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', 'true()+wahr()+my_func()'), 'func[TRUE] open close op[add] macro[wahr] open close op[add] macro[my_func] open close');
                expectResult(ooxParser.parseFormula('op', 'Sheet1!true()+Sheet1!wahr()'), 'macro[$0!true] open close op[add] macro[$0!wahr] open close');
                expectResult(ooxParser.parseFormula('op', 'Sheet1:Sheet2!my_func()+Sheet1!A1()'), 'name[Sheet1] op[range] macro[$1!my_func] open close op[add] macro[$0!A1] open close');
                expectResult(ooxParser.parseFormula('op', '_xlfn.FORMULATEXT()+FORMULATEXT()+FORMULA()'), 'func[FORMULATEXT] open close op[add] macro[FORMULATEXT] open close op[add] macro[FORMULA] open close');
                expectResult(ooxParser.parseFormula('op', 'FORMELTEXT()+FORMEL()'), 'macro[FORMELTEXT] open close op[add] macro[FORMEL] open close');
                expectResult(ooxParser.parseFormula('op', 'SUM (A1)'), 'name[SUM] op[isect] open ref[A1] close');
                expectResult(ooxParser.parseFormula('op', 'SUMME()+_xludf.SUM()'), 'macro[SUMME] open close op[add] macro[_xludf.SUM] open close');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', 'true()+wahr()+my_func()'), 'macro[true] open close op[add] func[TRUE] open close op[add] macro[my_func] open close');
                expectResult(ooxParser.parseFormula('ui', 'Sheet1!true()+Sheet1!wahr()'), 'macro[$0!true] open close op[add] macro[$0!wahr] open close');
                expectResult(ooxParser.parseFormula('ui', 'Sheet1:Sheet2!my_func()+Sheet1!A1()'), 'name[Sheet1] op[range] macro[$1!my_func] open close op[add] macro[$0!A1] open close');
                expectResult(ooxParser.parseFormula('ui', '_xlfn.FORMULATEXT()+FORMULATEXT()+FORMULA()'), 'macro[_xlfn.FORMULATEXT] open close op[add] macro[FORMULATEXT] open close op[add] macro[FORMULA] open close');
                expectResult(ooxParser.parseFormula('ui', 'FORMELTEXT()+FORMEL()'), 'func[FORMULATEXT] open close op[add] macro[FORMEL] open close');
                expectResult(ooxParser.parseFormula('ui', 'SUMME (A1)'), 'name[SUMME] op[isect] open ref[A1] close');
                // native ODF
                expectResult(odfParser.parseFormula('op', 'true()+wahr()+my_func()'), 'func[TRUE] open close op[add] macro[wahr] open close op[add] macro[my_func] open close');
                expectResult(odfParser.parseFormula('op', '_xlfn.FORMULATEXT()+FORMULATEXT()+FORMULA()'), 'macro[_xlfn.FORMULATEXT] open close op[add] macro[FORMULATEXT] open close op[add] func[FORMULATEXT] open close');
                expectResult(odfParser.parseFormula('op', 'FORMELTEXT()+FORMEL()'), 'macro[FORMELTEXT] open close op[add] macro[FORMEL] open close');
                expectResult(odfParser.parseFormula('op', 'SUM ([.A1])'), 'func[SUM] open ref[A1] close');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', 'true()+wahr()+my_func()'), 'macro[true] open close op[add] func[TRUE] open close op[add] macro[my_func] open close');
                expectResult(odfParser.parseFormula('ui', '_xlfn.FORMULATEXT()+FORMULATEXT()+FORMULA()'), 'macro[_xlfn.FORMULATEXT] open close op[add] macro[FORMULATEXT] open close op[add] macro[FORMULA] open close');
                expectResult(odfParser.parseFormula('ui', 'FORMELTEXT()+FORMEL()'), 'macro[FORMELTEXT] open close op[add] func[FORMULATEXT] open close');
                expectResult(odfParser.parseFormula('ui', 'SUMME (A1)'), 'name[SUMME] op[isect] open ref[A1] close');
            });

            it('should parse defined names', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', 'true1+wahr1+a1a'), 'name[true1] op[add] name[wahr1] op[add] name[a1a]');
                expectResult(ooxParser.parseFormula('op', 'Sheet1!name1+\'Sheet 4\'!name2+Sheet1:Sheet2!name3'), 'name[$0!name1] op[add] name[$3!name2] op[add] name[Sheet1] op[range] name[$1!name3]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', 'true1+wahr1+a1a'), 'name[true1] op[add] name[wahr1] op[add] name[a1a]');
                expectResult(ooxParser.parseFormula('ui', 'Sheet1!name1+\'Sheet 4\'!name2+Sheet1:Sheet2!name3'), 'name[$0!name1] op[add] name[$3!name2] op[add] name[Sheet1] op[range] name[$1!name3]');
                // native ODF
                expectResult(odfParser.parseFormula('op', 'true1+wahr1+a1a'), 'name[true1] op[add] name[wahr1] op[add] name[a1a]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', 'true1+wahr1+a1a'), 'name[true1] op[add] name[wahr1] op[add] name[a1a]');
                expectResult(odfParser.parseFormula('ui', 'Sheet1!name1'), 'bad[Sheet1!name1]');
            });

            it('should recognize negative numbers after operators', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', '-1-1--1-(-1)-1,-1'), 'lit[-1] op[sub] lit[1] op[sub] lit[-1] op[sub] open lit[-1] close op[sub] lit[1] sep lit[-1]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', '-1-1--1-(-1)-1;-1'), 'lit[-1] op[sub] lit[1] op[sub] lit[-1] op[sub] open lit[-1] close op[sub] lit[1] sep lit[-1]');
                // native ODF
                expectResult(odfParser.parseFormula('op', '-1-1--1-(-1)-1;-1'), 'lit[-1] op[sub] lit[1] op[sub] lit[-1] op[sub] open lit[-1] close op[sub] lit[1] sep lit[-1]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', '-1-1--1-(-1)-1;-1'), 'lit[-1] op[sub] lit[1] op[sub] lit[-1] op[sub] open lit[-1] close op[sub] lit[1] sep lit[-1]');
            });

            it('should fail for invalid matrix literals', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', '{1,{2}}'), 'bad[{1,{2}}]');
                expectResult(ooxParser.parseFormula('op', '1+2}'), 'lit[1] op[add] lit[2] bad[}]');
                expectResult(ooxParser.parseFormula('op', '{1*1}'), 'bad[{1*1}]');
                expectResult(ooxParser.parseFormula('op', '{1,A1}'), 'bad[{1,A1}]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', '{1;{2}}'), 'bad[{1;{2}}]');
                expectResult(ooxParser.parseFormula('ui', '1+2}'), 'lit[1] op[add] lit[2] bad[}]');
                expectResult(ooxParser.parseFormula('ui', '{1*1}'), 'bad[{1*1}]');
                expectResult(ooxParser.parseFormula('ui', '{1;A1}'), 'bad[{1;A1}]');
                // native ODF
                expectResult(odfParser.parseFormula('op', '{1;{2}}'), 'bad[{1;{2}}]');
                expectResult(odfParser.parseFormula('op', '1+2}'), 'lit[1] op[add] lit[2] bad[}]');
                expectResult(odfParser.parseFormula('op', '{1*1}'), 'bad[{1*1}]');
                expectResult(odfParser.parseFormula('op', '{1;A1}'), 'bad[{1;A1}]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', '{1;{2}}'), 'bad[{1;{2}}]');
                expectResult(odfParser.parseFormula('ui', '1+2}'), 'lit[1] op[add] lit[2] bad[}]');
                expectResult(odfParser.parseFormula('ui', '{1*1}'), 'bad[{1*1}]');
                expectResult(odfParser.parseFormula('ui', '{1;A1}'), 'bad[{1;A1}]');
            });

            it('should fail for tokens of other grammars', function () {
                // native OOXML
                expectResult(ooxParser.parseFormula('op', '1.2+3,4'), 'lit[1.2] op[add] lit[3] sep lit[4]');
                expectResult(ooxParser.parseFormula('op', 'True=Wahr=False=Falsch'), 'lit[TRUE] op[eq] name[Wahr] op[eq] lit[FALSE] op[eq] name[Falsch]');
                expectResult(ooxParser.parseFormula('op', '#Ref!<>#Bezug!+1'), 'lit[#REF] op[ne] bad[#Bezug!+1]');
                expectResult(ooxParser.parseFormula('op', '{1,2;3,4}+{1;2|3;4}'), 'mat[{1;2|3;4}] op[add] bad[{1;2|3;4}]');
                expectResult(ooxParser.parseFormula('op', 'SUMME(1,2;3)'), 'macro[SUMME] open lit[1] sep lit[2] bad[;3)]');
                // localized OOXML
                expectResult(ooxParser.parseFormula('ui', '1.2+3,4'), 'lit[1] bad[.2+3,4]');
                expectResult(ooxParser.parseFormula('ui', 'True=Wahr=False=Falsch'), 'name[True] op[eq] lit[TRUE] op[eq] name[False] op[eq] lit[FALSE]');
                expectResult(ooxParser.parseFormula('ui', '{1,2;3,4}+{1;2|3;4}'), 'mat[{1.2;3.4}] op[add] mat[{1;2|3;4}]');
                expectResult(ooxParser.parseFormula('ui', 'SUMME(1,2;3)'), 'func[SUM] open lit[1.2] sep lit[3] close');
                // native ODF
                expectResult(odfParser.parseFormula('op', '1.2+3,4'), 'lit[1.2] op[add] lit[3] bad[,4]');
                expectResult(odfParser.parseFormula('op', 'True=Wahr=False=Falsch'), 'lit[TRUE] op[eq] name[Wahr] op[eq] lit[FALSE] op[eq] name[Falsch]');
                expectResult(odfParser.parseFormula('op', '#Ref!<>#Bezug!+1'), 'lit[#REF] op[ne] bad[#Bezug!+1]');
                expectResult(odfParser.parseFormula('op', '{1,2;3,4}+{1;2|3;4}'), 'bad[{1,2;3,4}+{1;2|3;4}]');
                expectResult(odfParser.parseFormula('op', 'SUMME(1,2;3)'), 'macro[SUMME] open lit[1] bad[,2;3)]');
                // localized ODF
                expectResult(odfParser.parseFormula('ui', '1.2+3,4'), 'lit[1] bad[.2+3,4]');
                expectResult(odfParser.parseFormula('ui', 'True=Wahr=False=Falsch'), 'name[True] op[eq] lit[TRUE] op[eq] name[False] op[eq] lit[FALSE]');
                expectResult(odfParser.parseFormula('ui', '{1,2;3,4}+{1;2|3;4}'), 'mat[{1.2;3.4}] op[add] mat[{1;2|3;4}]');
                expectResult(odfParser.parseFormula('ui', 'SUMME(1,2;3)'), 'func[SUM] open lit[1.2] sep lit[3] close');
            });
        });

        describe('method "parseRangeList"', function () {
            it('should exist', function () {
                expect(ooxParser).to.respondTo('parseRangeList');
            });

            it('should parse simple range lists', function () {
                // native OOXML
                expect(ooxParser.parseRangeList('op', 'B3:C4')).to.be.an.instanceof(RangeArray);
                expect(ooxParser.parseRangeList('op', 'B3:C4')).to.stringifyTo('B3:C4');
                expect(ooxParser.parseRangeList('op', 'B3:C4 c4:d5  \tD5:E6')).to.stringifyTo('B3:C4,C4:D5,D5:E6');
                expect(ooxParser.parseRangeList('op', 'B3 c4')).to.stringifyTo('B3:B3,C4:C4');
                expect(ooxParser.parseRangeList('op', 'B:c 3:4')).to.stringifyTo('B1:C1048576,A3:XFD4');
                // localized OOXML
                expect(ooxParser.parseRangeList('ui', 'B3:C4')).to.be.an.instanceof(RangeArray);
                expect(ooxParser.parseRangeList('ui', 'B3:C4')).to.stringifyTo('B3:C4');
                expect(ooxParser.parseRangeList('ui', 'B3:C4 c4:d5  \tD5:E6')).to.stringifyTo('B3:C4,C4:D5,D5:E6');
                expect(ooxParser.parseRangeList('ui', 'B3 c4')).to.stringifyTo('B3:B3,C4:C4');
                expect(ooxParser.parseRangeList('ui', 'B:c 3:4')).to.stringifyTo('B1:C1048576,A3:XFD4');
                // native ODF
                expect(odfParser.parseRangeList('op', 'B3:C4')).to.be.an.instanceof(RangeArray);
                expect(odfParser.parseRangeList('op', 'B3:C4')).to.stringifyTo('B3:C4');
                expect(odfParser.parseRangeList('op', 'B3:C4 c4:d5  \tD5:E6')).to.stringifyTo('B3:C4,C4:D5,D5:E6');
                expect(odfParser.parseRangeList('op', 'B3 c4')).to.stringifyTo('B3:B3,C4:C4');
                // localized ODF
                expect(odfParser.parseRangeList('ui', 'B3:C4')).to.be.an.instanceof(RangeArray);
                expect(odfParser.parseRangeList('ui', 'B3:C4')).to.stringifyTo('B3:C4');
                expect(odfParser.parseRangeList('ui', 'B3:C4 c4:d5  \tD5:E6')).to.stringifyTo('B3:C4,C4:D5,D5:E6');
                expect(odfParser.parseRangeList('ui', 'B3 c4')).to.stringifyTo('B3:B3,C4:C4');
                expect(odfParser.parseRangeList('ui', 'B:c 3:4')).to.stringifyTo('B1:C1048576,A3:AMJ4');
            });

            it('should fail with invalid range lists', function () {
                // native OOXML
                expect(ooxParser.parseRangeList('op', 'abc')).to.equal(null);
                expect(ooxParser.parseRangeList('op', 'B3:ZZZZ4')).to.equal(null);
                expect(ooxParser.parseRangeList('op', 'B:3')).to.equal(null);
                // localized OOXML
                expect(ooxParser.parseRangeList('ui', 'abc')).to.equal(null);
                expect(ooxParser.parseRangeList('ui', 'B3:ZZZZ4')).to.equal(null);
                expect(ooxParser.parseRangeList('ui', 'B:3')).to.equal(null);
                // native ODF
                expect(odfParser.parseRangeList('op', 'abc')).to.equal(null);
                expect(odfParser.parseRangeList('op', 'B3:ZZZZ4')).to.equal(null);
                expect(odfParser.parseRangeList('op', 'B:3')).to.equal(null);
                expect(odfParser.parseRangeList('op', 'B:C')).to.equal(null);
                expect(odfParser.parseRangeList('op', '3:4')).to.equal(null);
                // localized ODF
                expect(odfParser.parseRangeList('ui', 'abc')).to.equal(null);
                expect(odfParser.parseRangeList('ui', 'B3:ZZZZ4')).to.equal(null);
                expect(odfParser.parseRangeList('ui', 'B:3')).to.equal(null);
            });

            it('should handle special separators', function () {
                expect(ooxParser.parseRangeList('op', 'B3:C4 C4:D5 D5:E6')).to.stringifyTo('B3:C4,C4:D5,D5:E6');
                expect(ooxParser.parseRangeList('op', 'B3:C4,C4:D5,D5:E6')).to.equal(null);
                expect(ooxParser.parseRangeList('op', 'B3:C4;C4:D5;D5:E6')).to.equal(null);
                expect(ooxParser.parseRangeList('op', 'B3:C4 C4:D5 D5:E6', { extSep: true })).to.stringifyTo('B3:C4,C4:D5,D5:E6');
                expect(ooxParser.parseRangeList('op', 'B3:C4,C4:D5,D5:E6', { extSep: true })).to.stringifyTo('B3:C4,C4:D5,D5:E6');
                expect(ooxParser.parseRangeList('op', 'B3:C4;C4:D5;D5:E6', { extSep: true })).to.stringifyTo('B3:C4,C4:D5,D5:E6');
                expect(ooxParser.parseRangeList('op', 'B3:C4,C4:D5;D5:E6', { extSep: true })).to.equal(null); // no mixed separators
                expect(ooxParser.parseRangeList('op', 'B3:C4 C4:D5;D5:E6', { extSep: true })).to.equal(null); // no mixed separators
            });

            it('should ignore sheet names if specified', function () {
                // native OOXML
                expect(ooxParser.parseRangeList('op', 'A2 Sheet1!B3 Sheet1:Sheet2!C4')).to.equal(null);
                expect(ooxParser.parseRangeList('op', 'A2:B3 Sheet1!B3:C4 Sheet1:Sheet2!C4:D5')).to.equal(null);
                expect(ooxParser.parseRangeList('op', 'A:B Sheet1!B:C Sheet1:Sheet2!C:D')).to.equal(null);
                expect(ooxParser.parseRangeList('op', '2:3 Sheet1!3:4 Sheet1:Sheet2!4:5')).to.equal(null);
                expect(ooxParser.parseRangeList('op', 'A2 Sheet1!B3 Sheet1:Sheet2!C4', { skipSheets: true })).to.stringifyTo('A2:A2,B3:B3,C4:C4');
                expect(ooxParser.parseRangeList('op', 'A2:B3 Sheet1!B3:C4 Sheet1:Sheet2!C4:D5', { skipSheets: true })).to.stringifyTo('A2:B3,B3:C4,C4:D5');
                expect(ooxParser.parseRangeList('op', 'A:B Sheet1!B:C Sheet1:Sheet2!C:D', { skipSheets: true })).to.stringifyTo('A1:B1048576,B1:C1048576,C1:D1048576');
                expect(ooxParser.parseRangeList('op', '2:3 Sheet1!3:4 Sheet1:Sheet2!4:5', { skipSheets: true })).to.stringifyTo('A2:XFD3,A3:XFD4,A4:XFD5');
                // localized OOXML
                expect(ooxParser.parseRangeList('ui', 'A2 Sheet1!B3 Sheet1:Sheet2!C4')).to.equal(null);
                expect(ooxParser.parseRangeList('ui', 'A2:B3 Sheet1!B3:C4 Sheet1:Sheet2!C4:D5')).to.equal(null);
                expect(ooxParser.parseRangeList('ui', 'A:B Sheet1!B:C Sheet1:Sheet2!C:D')).to.equal(null);
                expect(ooxParser.parseRangeList('ui', '2:3 Sheet1!3:4 Sheet1:Sheet2!4:5')).to.equal(null);
                expect(ooxParser.parseRangeList('ui', 'A2 Sheet1!B3 Sheet1:Sheet2!C4', { skipSheets: true })).to.stringifyTo('A2:A2,B3:B3,C4:C4');
                expect(ooxParser.parseRangeList('ui', 'A2:B3 Sheet1!B3:C4 Sheet1:Sheet2!C4:D5', { skipSheets: true })).to.stringifyTo('A2:B3,B3:C4,C4:D5');
                expect(ooxParser.parseRangeList('ui', 'A:B Sheet1!B:C Sheet1:Sheet2!C:D', { skipSheets: true })).to.stringifyTo('A1:B1048576,B1:C1048576,C1:D1048576');
                expect(ooxParser.parseRangeList('ui', '2:3 Sheet1!3:4 Sheet1:Sheet2!4:5', { skipSheets: true })).to.stringifyTo('A2:XFD3,A3:XFD4,A4:XFD5');
                // native ODF (no support for column/row intervals)
                expect(odfParser.parseRangeList('op', 'A2 Sheet1.B3')).to.equal(null);
                expect(odfParser.parseRangeList('op', 'A2:B3 Sheet1.B3:C4 Sheet1.C4:Sheet2.D5')).to.equal(null);
                expect(odfParser.parseRangeList('op', 'A:B Sheet1.B:C Sheet1:Sheet2.C:D')).to.equal(null);
                expect(odfParser.parseRangeList('op', '2:3 Sheet1.3:4 Sheet1:Sheet2.4:5')).to.equal(null);
                expect(odfParser.parseRangeList('op', 'A2 Sheet1.B3', { skipSheets: true })).to.stringifyTo('A2:A2,B3:B3');
                expect(odfParser.parseRangeList('op', 'A2:B3 Sheet1.B3:C4 Sheet1.C4:Sheet2.D5', { skipSheets: true })).to.stringifyTo('A2:B3,B3:C4,C4:D5');
                expect(odfParser.parseRangeList('op', 'A:B Sheet1.B:C Sheet1.C:Sheet2.D', { skipSheets: true })).to.equal(null);
                expect(odfParser.parseRangeList('op', '2:3 Sheet1.3:4 Sheet1.4:Sheet2.5', { skipSheets: true })).to.equal(null);
                // localized ODF
                expect(odfParser.parseRangeList('ui', 'A2 Sheet1!B3 Sheet1:Sheet2!C4')).to.equal(null);
                expect(odfParser.parseRangeList('ui', 'A2:B3 Sheet1!B3:C4 Sheet1:Sheet2!C4:D5')).to.equal(null);
                expect(odfParser.parseRangeList('ui', 'A:B Sheet1!B:C Sheet1:Sheet2!C:D')).to.equal(null);
                expect(odfParser.parseRangeList('ui', '2:3 Sheet1!3:4 Sheet1:Sheet2!4:5')).to.equal(null);
                expect(odfParser.parseRangeList('ui', 'A2 Sheet1!B3 Sheet1:Sheet2!C4', { skipSheets: true })).to.stringifyTo('A2:A2,B3:B3,C4:C4');
                expect(odfParser.parseRangeList('ui', 'A2:B3 Sheet1!B3:C4 Sheet1:Sheet2!C4:D5', { skipSheets: true })).to.stringifyTo('A2:B3,B3:C4,C4:D5');
                expect(odfParser.parseRangeList('ui', 'A:B Sheet1!B:C Sheet1:Sheet2!C:D', { skipSheets: true })).to.stringifyTo('A1:B1048576,B1:C1048576,C1:D1048576');
                expect(odfParser.parseRangeList('ui', '2:3 Sheet1!3:4 Sheet1:Sheet2!4:5', { skipSheets: true })).to.stringifyTo('A2:AMJ3,A3:AMJ4,A4:AMJ5');
            });
        });

        describe('method "formatRangeList"', function () {
            it('should exist', function () {
                expect(ooxParser).to.respondTo('formatRangeList');
            });

            it('should format range lists', function () {
                // native OOXML
                expect(ooxParser.formatRangeList('op', ra('B3:C4 C4:D5 D5:E6'))).to.equal('B3:C4 C4:D5 D5:E6');
                expect(ooxParser.formatRangeList('op', ra('B3:B3 C4:C4'))).to.equal('B3 C4');
                expect(ooxParser.formatRangeList('op', ra('B1:C1048576 A3:XFD4'))).to.equal('B:C 3:4');
                // localized OOXML
                expect(ooxParser.formatRangeList('ui', ra('B3:C4 C4:D5 D5:E6'))).to.equal('B3:C4 C4:D5 D5:E6');
                expect(ooxParser.formatRangeList('ui', ra('B3:B3 C4:C4'))).to.equal('B3 C4');
                expect(ooxParser.formatRangeList('ui', ra('B1:C1048576 A3:XFD4'))).to.equal('B:C 3:4');
                // native ODF
                expect(odfParser.formatRangeList('op', ra('B3:C4 C4:D5 D5:E6'))).to.equal('B3:C4 C4:D5 D5:E6');
                expect(odfParser.formatRangeList('op', ra('B3:B3 C4:C4'))).to.equal('B3 C4');
                expect(odfParser.formatRangeList('op', ra('B1:C1048576 A3:AMJ4'))).to.equal('B1:C1048576 A3:AMJ4');
                // localized ODF
                expect(odfParser.formatRangeList('ui', ra('B3:C4 C4:D5 D5:E6'))).to.equal('B3:C4 C4:D5 D5:E6');
                expect(odfParser.formatRangeList('ui', ra('B3:B3 C4:C4'))).to.equal('B3 C4');
                expect(odfParser.formatRangeList('ui', ra('B1:C1048576 A3:AMJ4'))).to.equal('B:C 3:4');
            });

            it('should use custom separators', function () {
                expect(ooxParser.formatRangeList('op', ra('B3:C4 C4:D5 D5:E6'), { sep: ',' })).to.equal('B3:C4,C4:D5,D5:E6');
                expect(ooxParser.formatRangeList('op', ra('B3:C4 C4:D5 D5:E6'), { sep: ';' })).to.equal('B3:C4;C4:D5;D5:E6');
            });

            it('should add sheet names', function () {
                // native OOXML
                expect(ooxParser.formatRangeList('op', ra('B3:B3 C4:D5 B1:C1048576 A3:XFD4'), { sheetName: 'Sheet1' })).to.equal('Sheet1!B3 Sheet1!C4:D5 Sheet1!B:C Sheet1!3:4');
                expect(ooxParser.formatRangeList('op', ra('C4:D5'), { sheetName: 'Sheet 1' })).to.equal('\'Sheet 1\'!C4:D5');
                // localized OOXML
                expect(ooxParser.formatRangeList('ui', ra('B3:B3 C4:D5 B1:C1048576 A3:XFD4'), { sheetName: 'Sheet1' })).to.equal('Sheet1!B3 Sheet1!C4:D5 Sheet1!B:C Sheet1!3:4');
                expect(ooxParser.formatRangeList('ui', ra('C4:D5'), { sheetName: 'Sheet 1' })).to.equal('\'Sheet 1\'!C4:D5');
                // native ODF
                expect(odfParser.formatRangeList('op', ra('B3:B3 C4:D5 B1:C1048576 A3:AMJ4'), { sheetName: 'Sheet1' })).to.equal('Sheet1.B3 Sheet1.C4:Sheet1.D5 Sheet1.B1:Sheet1.C1048576 Sheet1.A3:Sheet1.AMJ4');
                expect(odfParser.formatRangeList('op', ra('C4:D5'), { sheetName: 'Sheet 1' })).to.equal('\'Sheet 1\'.C4:\'Sheet 1\'.D5');
                // localized ODF
                expect(odfParser.formatRangeList('ui', ra('B3:B3 C4:D5 B1:C1048576 A3:AMJ4'), { sheetName: 'Sheet1' })).to.equal('Sheet1!B3 Sheet1!C4:D5 Sheet1!B:C Sheet1!3:4');
                expect(odfParser.formatRangeList('ui', ra('C4:D5'), { sheetName: 'Sheet 1' })).to.equal('\'Sheet 1\'!C4:D5');
            });
        });

        describe('method "parseRange3DList"', function () {
            it('should exist', function () {
                expect(ooxParser).to.respondTo('parseRange3DList');
            });
        });

        describe('method "formatRange3DList"', function () {
            it('should exist', function () {
                expect(ooxParser).to.respondTo('formatRange3DList');
            });
        });
    });

    // ========================================================================
});
