/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/utils/cellref',
    'io.ox/office/spreadsheet/model/formula/utils/sheetref',
    'io.ox/office/spreadsheet/model/formula/parser/formulagrammar'
], function (SheetUtils, CellRef, SheetRef, FormulaGrammar) {

    'use strict';

    // class FormulaGrammar ===================================================

    describe('Spreadsheet class FormulaGrammar', function () {

        it('should exist', function () {
            expect(FormulaGrammar).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // convenience shortcuts
        var ErrorCode = SheetUtils.ErrorCode;

        // create a CellRef instance on-the-fly from $1.$1 notation (column/row offsets)
        function cell(text) {
            var m = /^(\$)?(-?\d+)\.(\$)?(-?\d+)$/i.exec(text);
            return new CellRef(parseInt(m[2], 10), parseInt(m[4], 10), !!m[1], !!m[3]);
        }

        // create a SheetRef instance on-the-fly
        function sh(sheet, abs) {
            return new SheetRef(sheet, !_.isBoolean(abs) || abs);
        }

        var ooxOpGrammar = null;
        var ooxUiGrammar = null;
        var ooxEnGrammar = null;
        var odfOpGrammar = null;
        var odfUiGrammar = null;
        var odfEnGrammar = null;

        // maximum column/row index for this test
        var MAXCOL = 255;
        var MAXROW = 65535;

        // simple mock of a document model, used for sheet names, and maximum column/row checks
        var docModel = {
            getSheetName: function (index) {
                return ['Sheet1', 'Sheet2', 'Sheet3', 'Sheet 4'][index] || null;
            },
            isValidAddress: function (address) {
                return (address[0] >= 0) && (address[0] <= MAXCOL) && (address[1] >= 0) && (address[1] <= MAXROW);
            },
            isColRange: function (range) {
                return (range.start[1] === 0) && (range.end[1] === MAXROW);
            },
            isRowRange: function (range) {
                return (range.start[0] === 0) && (range.end[0] === MAXCOL);
            }
        };

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a formula grammar instance', function () {
                ooxOpGrammar = new FormulaGrammar('ooxml', 'op');
                expect(ooxOpGrammar).to.be.an('object');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(FormulaGrammar).itself.to.respondTo('create');
            });
            it('should create a new configuration', function () {
                ooxUiGrammar = FormulaGrammar.create('ooxml', 'ui');
                expect(ooxUiGrammar).to.be.an.instanceof(FormulaGrammar);
                expect(ooxUiGrammar).to.not.equal(ooxOpGrammar);
                ooxEnGrammar = FormulaGrammar.create('ooxml', 'en');
                expect(ooxEnGrammar).to.be.an.instanceof(FormulaGrammar);
                expect(ooxEnGrammar).to.not.equal(ooxOpGrammar);
                expect(ooxEnGrammar).to.not.equal(ooxUiGrammar);
                odfOpGrammar = FormulaGrammar.create('odf', 'op');
                expect(odfOpGrammar).to.be.an.instanceof(FormulaGrammar);
                expect(odfOpGrammar).to.not.equal(ooxOpGrammar);
                expect(odfOpGrammar).to.not.equal(ooxUiGrammar);
                expect(odfOpGrammar).to.not.equal(ooxEnGrammar);
                odfUiGrammar = FormulaGrammar.create('odf', 'ui');
                expect(odfUiGrammar).to.be.an.instanceof(FormulaGrammar);
                expect(odfUiGrammar).to.not.equal(ooxOpGrammar);
                expect(odfUiGrammar).to.not.equal(ooxUiGrammar);
                expect(odfUiGrammar).to.not.equal(ooxEnGrammar);
                expect(odfUiGrammar).to.not.equal(odfOpGrammar);
                odfEnGrammar = FormulaGrammar.create('odf', 'en');
                expect(odfEnGrammar).to.be.an.instanceof(FormulaGrammar);
                expect(odfEnGrammar).to.not.equal(ooxOpGrammar);
                expect(odfEnGrammar).to.not.equal(ooxUiGrammar);
                expect(odfEnGrammar).to.not.equal(ooxEnGrammar);
                expect(odfEnGrammar).to.not.equal(odfOpGrammar);
                expect(odfEnGrammar).to.not.equal(odfUiGrammar);
            });
            it('should return an existing configuration', function () {
                var config = FormulaGrammar.create('ooxml', 'ui');
                expect(config).to.equal(ooxUiGrammar);
            });
        });

        // constants ----------------------------------------------------------

        describe('constant "GRAMMAR"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.have.a.property('GRAMMAR', 'op');
                expect(ooxUiGrammar).to.have.a.property('GRAMMAR', 'ui');
                expect(ooxEnGrammar).to.have.a.property('GRAMMAR', 'en');
                expect(odfOpGrammar).to.have.a.property('GRAMMAR', 'op');
                expect(odfUiGrammar).to.have.a.property('GRAMMAR', 'ui');
                expect(odfEnGrammar).to.have.a.property('GRAMMAR', 'en');
            });
        });

        describe('constant "UI"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.have.a.property('UI', false);
                expect(ooxUiGrammar).to.have.a.property('UI', true);
                expect(ooxEnGrammar).to.have.a.property('UI', true);
                expect(odfOpGrammar).to.have.a.property('UI', false);
                expect(odfUiGrammar).to.have.a.property('UI', true);
                expect(odfEnGrammar).to.have.a.property('UI', true);
            });
        });

        describe('constant "REF_SYNTAX"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.have.a.property('REF_SYNTAX', 'ooxml');
                expect(ooxUiGrammar).to.have.a.property('REF_SYNTAX', 'ooxml');
                expect(ooxEnGrammar).to.have.a.property('REF_SYNTAX', 'ooxml');
                expect(odfOpGrammar).to.have.a.property('REF_SYNTAX', 'of');
                expect(odfUiGrammar).to.have.a.property('REF_SYNTAX', 'odfui');
                expect(odfEnGrammar).to.have.a.property('REF_SYNTAX', 'odfui');
            });
        });

        describe('constant "DEC"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.have.a.property('DEC', '.');
                expect(ooxUiGrammar).to.have.a.property('DEC', ',');
                expect(ooxEnGrammar).to.have.a.property('DEC', '.');
                expect(odfOpGrammar).to.have.a.property('DEC', '.');
                expect(odfUiGrammar).to.have.a.property('DEC', ',');
                expect(odfEnGrammar).to.have.a.property('DEC', '.');
            });
        });

        describe('constant "SEP"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.have.a.property('SEP', ',');
                expect(ooxUiGrammar).to.have.a.property('SEP', ';');
                expect(ooxEnGrammar).to.have.a.property('SEP', ',');
                expect(odfOpGrammar).to.have.a.property('SEP', ';');
                expect(odfUiGrammar).to.have.a.property('SEP', ';');
                expect(odfEnGrammar).to.have.a.property('SEP', ',');
            });
        });

        describe('constant "MAT_OPEN"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.have.a.property('MAT_OPEN', '{');
                expect(ooxUiGrammar).to.have.a.property('MAT_OPEN', '{');
                expect(ooxEnGrammar).to.have.a.property('MAT_OPEN', '{');
                expect(odfOpGrammar).to.have.a.property('MAT_OPEN', '{');
                expect(odfUiGrammar).to.have.a.property('MAT_OPEN', '{');
                expect(odfEnGrammar).to.have.a.property('MAT_OPEN', '{');
            });
        });

        describe('constant "MAT_ROW"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.have.a.property('MAT_ROW', ';');
                expect(ooxUiGrammar).to.have.a.property('MAT_ROW', '|');
                expect(ooxEnGrammar).to.have.a.property('MAT_ROW', '|');
                expect(odfOpGrammar).to.have.a.property('MAT_ROW', '|');
                expect(odfUiGrammar).to.have.a.property('MAT_ROW', '|');
                expect(odfEnGrammar).to.have.a.property('MAT_ROW', '|');
            });
        });

        describe('constant "MAT_COL"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.have.a.property('MAT_COL', ',');
                expect(ooxUiGrammar).to.have.a.property('MAT_COL', ';');
                expect(ooxEnGrammar).to.have.a.property('MAT_COL', ';');
                expect(odfOpGrammar).to.have.a.property('MAT_COL', ';');
                expect(odfUiGrammar).to.have.a.property('MAT_COL', ';');
                expect(odfEnGrammar).to.have.a.property('MAT_COL', ';');
            });
        });

        describe('constant "MAT_CLOSE"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.have.a.property('MAT_CLOSE', '}');
                expect(ooxUiGrammar).to.have.a.property('MAT_CLOSE', '}');
                expect(ooxEnGrammar).to.have.a.property('MAT_CLOSE', '}');
                expect(odfOpGrammar).to.have.a.property('MAT_CLOSE', '}');
                expect(odfUiGrammar).to.have.a.property('MAT_CLOSE', '}');
                expect(odfEnGrammar).to.have.a.property('MAT_CLOSE', '}');
            });
        });

        describe('constant "RE"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.have.a.property('RE').that.is.an('object');
                expect(ooxUiGrammar).to.have.a.property('RE').that.is.an('object');
                expect(ooxEnGrammar).to.have.a.property('RE').that.is.an('object');
                expect(odfOpGrammar).to.have.a.property('RE').that.is.an('object');
                expect(odfUiGrammar).to.have.a.property('RE').that.is.an('object');
                expect(odfEnGrammar).to.have.a.property('RE').that.is.an('object');
            });
            it('should contain regular expressions', function () {
                expect(_.every(ooxOpGrammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(ooxUiGrammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(ooxEnGrammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(odfOpGrammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(odfUiGrammar.RE, _.isRegExp)).to.equal(true);
                expect(_.every(odfEnGrammar.RE, _.isRegExp)).to.equal(true);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "isReservedSymbol"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('isReservedSymbol');
                expect(ooxUiGrammar).to.respondTo('isReservedSymbol');
            });
            it('should recognize boolean literals', function () {
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'FALSE')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'FALSE')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'FALSE')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'TRUE')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'TRUE')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'TRUE')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'FALSCH')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'FALSCH')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'FALSCH')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'WAHR')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'WAHR')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'WAHR')).to.equal(false);
            });
            it('should recgonize A1 references', function () {
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'A1')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'A1')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'A1')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'iv65536')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'iv65536')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'iv65536')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'IW65536')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'IW65536')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'IW65536')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'IV65537')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'IV65537')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'IV65537')).to.equal(false);
            });
            it('should recgonize R1C1 references', function () {
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'R1C1')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'R1C1')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'R1C1')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'R65536C256')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'R65536C256')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'R65536C256')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'R65537C256')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'R65537C256')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'R65537C256')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'R65536C257')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'R65536C257')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'R65536C257')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'R1C')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'R1C')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'R1C')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'RC1')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'RC1')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'RC1')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'R1')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'R1')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'R1')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'C1')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'C1')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'C1')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'R')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'R')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'R')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'C')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'C')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'C')).to.equal(true);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'RC')).to.equal(true);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'RC')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'RC')).to.equal(true);
                // localized R1C1 references (not used in ODF)
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(true);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
                expect(odfOpGrammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
                expect(odfUiGrammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
                expect(odfEnGrammar.isReservedSymbol(docModel, 'Z1S1')).to.equal(false);
            });
            it('should return false for other strings', function () {
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'A')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'A')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'A')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, '1')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, '1')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, '1')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, ' A1')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, ' A1')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, ' A1')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'A 1')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'A 1')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'A 1')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'R1 C1')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'R1 C1')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'R1 C1')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'C1R1')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'C1R1')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'C1R1')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, 'S1Z1')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, 'S1Z1')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, 'S1Z1')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, '')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, '')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, '')).to.equal(false);
                expect(ooxOpGrammar.isReservedSymbol(docModel, ' TRUE')).to.equal(false);
                expect(ooxUiGrammar.isReservedSymbol(docModel, ' WAHR')).to.equal(false);
                expect(ooxEnGrammar.isReservedSymbol(docModel, ' WAHR')).to.equal(false);
            });
        });

        describe('method "isSimpleSheetName"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('isSimpleSheetName');
                expect(ooxUiGrammar).to.respondTo('isSimpleSheetName');
            });
            it('should recognize simple sheet names', function () {
                // simple sheet name
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                expect(odfOpGrammar.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                expect(odfUiGrammar.isSimpleSheetName(docModel, 'Sheet1')).to.equal(true);
                // whitespace
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                expect(odfOpGrammar.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                expect(odfUiGrammar.isSimpleSheetName(docModel, 'Sheet 1')).to.equal(false);
                // sheet range
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                expect(odfOpGrammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                expect(odfUiGrammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2')).to.equal(false);
                // booleans
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'TRUE')).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'TRUE')).to.equal(true);
                expect(ooxEnGrammar.isSimpleSheetName(docModel, 'TRUE')).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'WAHR')).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'WAHR')).to.equal(false);
                expect(ooxEnGrammar.isSimpleSheetName(docModel, 'WAHR')).to.equal(true);
                // A1 notation
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                expect(odfOpGrammar.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                expect(odfUiGrammar.isSimpleSheetName(docModel, 'A1')).to.equal(false);
                // native R1C1 notation
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(ooxEnGrammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(odfOpGrammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(odfUiGrammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                expect(odfEnGrammar.isSimpleSheetName(docModel, 'R1C1')).to.equal(false);
                // localized R1C1 notation
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(false);
                expect(ooxEnGrammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                expect(odfOpGrammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                expect(odfUiGrammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                expect(odfEnGrammar.isSimpleSheetName(docModel, 'Z1S1')).to.equal(true);
                // external path
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
                expect(odfOpGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
                expect(odfUiGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1')).to.equal(false);
            });
            it('should recognize simple sheet ranges', function () {
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1', { range: true })).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1', { range: true })).to.equal(true);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet 1', { range: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet 1', { range: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true })).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true })).to.equal(true);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1:Sheet 2', { range: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1:Sheet 2', { range: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1:TRUE', { range: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1:TRUE', { range: true })).to.equal(true);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1:WAHR', { range: true })).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1:WAHR', { range: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1:A1', { range: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1:A1', { range: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1:R1C1', { range: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1:R1C1', { range: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1:Z1S1', { range: true })).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1:Z1S1', { range: true })).to.equal(false);
            });
            it('should recognize simple sheet names with document name', function () {
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1', { external: true })).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1', { external: true })).to.equal(true);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet 1', { external: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet 1', { external: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { external: true })).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { external: true })).to.equal(true);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { external: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { external: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { external: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { external: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1', { external: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1', { external: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path/doc1.xlsx]Sheet1', { external: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path/doc1.xlsx]Sheet1', { external: true })).to.equal(false);
            });
            it('should recognize simple sheet ranges with document name', function () {
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1', { range: true, external: true })).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1', { range: true, external: true })).to.equal(true);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet 1', { range: true, external: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet 1', { range: true, external: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, 'Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { range: true, external: true })).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1', { range: true, external: true })).to.equal(true);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { range: true, external: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet 1', { range: true, external: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(true);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet 2', { range: true, external: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path\\doc.xlsx]Sheet1:Sheet 2', { range: true, external: true })).to.equal(false);
                expect(ooxOpGrammar.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(false);
                expect(ooxUiGrammar.isSimpleSheetName(docModel, '[path\\doc 1.xlsx]Sheet1:Sheet2', { range: true, external: true })).to.equal(false);
            });
        });

        describe('method "validateNameLabel"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('validateNameLabel');
                expect(ooxUiGrammar).to.respondTo('validateNameLabel');
            });
            it('should reject invalid strings', function () {
                expect(ooxOpGrammar.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(ooxUiGrammar.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(odfOpGrammar.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(odfUiGrammar.validateNameLabel(docModel, '')).to.equal('name:empty');
                expect(ooxOpGrammar.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(ooxUiGrammar.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(odfOpGrammar.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(odfUiGrammar.validateNameLabel(docModel, ' name')).to.equal('name:invalid');
                expect(ooxOpGrammar.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(ooxUiGrammar.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(odfOpGrammar.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(odfUiGrammar.validateNameLabel(docModel, '1name')).to.equal('name:invalid');
                expect(ooxOpGrammar.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(ooxUiGrammar.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(odfOpGrammar.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(odfUiGrammar.validateNameLabel(docModel, '/name')).to.equal('name:invalid');
                expect(ooxOpGrammar.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(ooxUiGrammar.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(odfOpGrammar.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(odfUiGrammar.validateNameLabel(docModel, '?name')).to.equal('name:invalid');
                expect(ooxOpGrammar.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
                expect(ooxUiGrammar.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
                expect(odfOpGrammar.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
                expect(odfUiGrammar.validateNameLabel(docModel, '.name')).to.equal('name:invalid');
            });
            it('should handle boolean literals', function () {
                expect(ooxOpGrammar.validateNameLabel(docModel, 'true')).to.equal('name:invalid');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'true')).to.equal('');
                expect(ooxEnGrammar.validateNameLabel(docModel, 'true')).to.equal('name:invalid');
                expect(odfOpGrammar.validateNameLabel(docModel, 'true')).to.equal('');
                expect(odfUiGrammar.validateNameLabel(docModel, 'true')).to.equal('');
                expect(odfEnGrammar.validateNameLabel(docModel, 'true')).to.equal('');
                expect(ooxOpGrammar.validateNameLabel(docModel, 'wahr')).to.equal('');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'wahr')).to.equal('name:invalid');
                expect(ooxEnGrammar.validateNameLabel(docModel, 'wahr')).to.equal('');
                expect(odfOpGrammar.validateNameLabel(docModel, 'wahr')).to.equal('');
                expect(odfUiGrammar.validateNameLabel(docModel, 'wahr')).to.equal('');
                expect(odfEnGrammar.validateNameLabel(docModel, 'wahr')).to.equal('');
            });
            it('should reject A1 cell references', function () {
                // first cell in sheet
                expect(ooxOpGrammar.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                expect(odfOpGrammar.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                expect(odfUiGrammar.validateNameLabel(docModel, 'a1')).to.equal('name:address');
                // last cell in sheet
                expect(ooxOpGrammar.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                expect(odfOpGrammar.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                expect(odfUiGrammar.validateNameLabel(docModel, 'iv65536')).to.equal('name:address');
                // outside of sheet range
                expect(ooxOpGrammar.validateNameLabel(docModel, 'iw65536')).to.equal('');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'iw65536')).to.equal('');
                expect(odfOpGrammar.validateNameLabel(docModel, 'iw65536')).to.equal('');
                expect(odfUiGrammar.validateNameLabel(docModel, 'iw65536')).to.equal('');
            });
            it('should reject R1C1 cell references', function () {
                // native absolute R1C1
                expect(ooxOpGrammar.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                expect(odfOpGrammar.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                expect(odfUiGrammar.validateNameLabel(docModel, 'r1c1')).to.equal('name:address');
                // native relative R1C1
                expect(ooxOpGrammar.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                expect(odfOpGrammar.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                expect(odfUiGrammar.validateNameLabel(docModel, 'rc')).to.equal('name:address');
                // localized absolute R1C1
                expect(ooxOpGrammar.validateNameLabel(docModel, 'z1s1')).to.equal('');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'z1s1')).to.equal('name:address');
                expect(odfOpGrammar.validateNameLabel(docModel, 'z1s1')).to.equal('');
                expect(odfUiGrammar.validateNameLabel(docModel, 'z1s1')).to.equal('');
                // localized relative R1C1
                expect(ooxOpGrammar.validateNameLabel(docModel, 'zs')).to.equal('');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'zs')).to.equal('name:address');
                expect(odfOpGrammar.validateNameLabel(docModel, 'zs')).to.equal('');
                expect(odfUiGrammar.validateNameLabel(docModel, 'zs')).to.equal('');
            });
            it('should accept valid names', function () {
                expect(ooxOpGrammar.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(odfOpGrammar.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(odfUiGrammar.validateNameLabel(docModel, 'my_name')).to.equal('');
                expect(ooxOpGrammar.validateNameLabel(docModel, 'true1')).to.equal('');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'wahr1')).to.equal('');
                expect(odfOpGrammar.validateNameLabel(docModel, 'true1')).to.equal('');
                expect(odfUiGrammar.validateNameLabel(docModel, 'wahr1')).to.equal('');
                expect(ooxOpGrammar.validateNameLabel(docModel, 'c1r1')).to.equal('');
                expect(ooxUiGrammar.validateNameLabel(docModel, 'c1r1')).to.equal('');
                expect(odfOpGrammar.validateNameLabel(docModel, 'c1r1')).to.equal('');
                expect(odfUiGrammar.validateNameLabel(docModel, 'c1r1')).to.equal('');
            });
            it('should handle backslash and question mark', function () {
                expect(ooxOpGrammar.validateNameLabel(docModel, '\\name?name.name\\name')).to.equal('');
                expect(ooxUiGrammar.validateNameLabel(docModel, '\\name?name.name\\name')).to.equal('');
                expect(odfOpGrammar.validateNameLabel(docModel, '\\name')).to.equal('name:invalid');
                expect(odfUiGrammar.validateNameLabel(docModel, '\\name')).to.equal('name:invalid');
                expect(odfOpGrammar.validateNameLabel(docModel, 'name?a')).to.equal('');
                expect(odfUiGrammar.validateNameLabel(docModel, 'name?a')).to.equal('');
                expect(odfOpGrammar.validateNameLabel(docModel, 'name.a')).to.equal('name:invalid');
                expect(odfUiGrammar.validateNameLabel(docModel, 'name.a')).to.equal('name:invalid');
            });
        });

        describe('method "formatScalar"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('formatScalar');
                expect(ooxUiGrammar).to.respondTo('formatScalar');
            });
            it('should convert values to native text', function () {
                expect(ooxOpGrammar.formatScalar(42)).to.equal('42');
                expect(ooxOpGrammar.formatScalar(-12.5)).to.equal('-12.5');
                expect(ooxOpGrammar.formatScalar(1e300)).to.equal('1E+300');
                expect(ooxOpGrammar.formatScalar(1e-300)).to.equal('1E-300');
                expect(ooxOpGrammar.formatScalar(1e-310)).to.equal('0');
                expect(ooxOpGrammar.formatScalar(Number.POSITIVE_INFINITY)).to.equal('#NUM!');
                expect(ooxOpGrammar.formatScalar(Number.NaN)).to.equal('#NUM!');
                expect(ooxOpGrammar.formatScalar('abc')).to.equal('"abc"');
                expect(ooxOpGrammar.formatScalar('a"b"c')).to.equal('"a""b""c"');
                expect(ooxOpGrammar.formatScalar('')).to.equal('""');
                expect(ooxOpGrammar.formatScalar(false)).to.equal('FALSE');
                expect(ooxOpGrammar.formatScalar(true)).to.equal('TRUE');
                expect(ooxOpGrammar.formatScalar(ErrorCode.NULL)).to.equal('#NULL!');
                expect(ooxOpGrammar.formatScalar(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(ooxOpGrammar.formatScalar(ErrorCode.VALUE)).to.equal('#VALUE!');
                expect(ooxOpGrammar.formatScalar(ErrorCode.REF)).to.equal('#REF!');
                expect(ooxOpGrammar.formatScalar(ErrorCode.NAME)).to.equal('#NAME?');
                expect(ooxOpGrammar.formatScalar(ErrorCode.NUM)).to.equal('#NUM!');
                expect(ooxOpGrammar.formatScalar(ErrorCode.NA)).to.equal('#N/A');
                expect(ooxOpGrammar.formatScalar(ErrorCode.DATA)).to.equal('#GETTING_DATA');
                expect(ooxOpGrammar.formatScalar(null)).to.equal('');
            });
            it('should convert values to localized text', function () {
                expect(ooxUiGrammar.formatScalar(42)).to.equal('42');
                expect(ooxUiGrammar.formatScalar(-12.5)).to.equal('-12,5');
                expect(ooxUiGrammar.formatScalar(1e300)).to.equal('1E+300');
                expect(ooxUiGrammar.formatScalar(1e-300)).to.equal('1E-300');
                expect(ooxUiGrammar.formatScalar(1e-310)).to.equal('0');
                expect(ooxUiGrammar.formatScalar(Number.POSITIVE_INFINITY)).to.equal('#ZAHL!');
                expect(ooxUiGrammar.formatScalar(Number.NaN)).to.equal('#ZAHL!');
                expect(ooxUiGrammar.formatScalar('abc')).to.equal('"abc"');
                expect(ooxUiGrammar.formatScalar('a"b"c')).to.equal('"a""b""c"');
                expect(ooxUiGrammar.formatScalar('')).to.equal('""');
                expect(ooxUiGrammar.formatScalar(ErrorCode.NULL)).to.equal('#NULL!');
                expect(ooxUiGrammar.formatScalar(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(ooxUiGrammar.formatScalar(ErrorCode.VALUE)).to.equal('#WERT!');
                expect(ooxUiGrammar.formatScalar(ErrorCode.REF)).to.equal('#BEZUG!');
                expect(ooxUiGrammar.formatScalar(ErrorCode.NAME)).to.equal('#NAME?');
                expect(ooxUiGrammar.formatScalar(ErrorCode.NUM)).to.equal('#ZAHL!');
                expect(ooxUiGrammar.formatScalar(ErrorCode.NA)).to.equal('#NV');
                expect(ooxUiGrammar.formatScalar(ErrorCode.DATA)).to.equal('#GETTING_DATA');
                expect(ooxUiGrammar.formatScalar(null)).to.equal('');
            });
            it('should convert values to English text', function () {
                expect(ooxEnGrammar.formatScalar(42)).to.equal('42');
                expect(ooxEnGrammar.formatScalar(-12.5)).to.equal('-12.5');
                expect(ooxEnGrammar.formatScalar(1e300)).to.equal('1E+300');
                expect(ooxEnGrammar.formatScalar(1e-300)).to.equal('1E-300');
                expect(ooxEnGrammar.formatScalar(1e-310)).to.equal('0');
                expect(ooxEnGrammar.formatScalar(Number.POSITIVE_INFINITY)).to.equal('#NUM!');
                expect(ooxEnGrammar.formatScalar(Number.NaN)).to.equal('#NUM!');
                expect(ooxEnGrammar.formatScalar('abc')).to.equal('"abc"');
                expect(ooxEnGrammar.formatScalar('a"b"c')).to.equal('"a""b""c"');
                expect(ooxEnGrammar.formatScalar('')).to.equal('""');
                expect(ooxEnGrammar.formatScalar(ErrorCode.NULL)).to.equal('#NULL!');
                expect(ooxEnGrammar.formatScalar(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(ooxEnGrammar.formatScalar(ErrorCode.VALUE)).to.equal('#VALUE!');
                expect(ooxEnGrammar.formatScalar(ErrorCode.REF)).to.equal('#REF!');
                expect(ooxEnGrammar.formatScalar(ErrorCode.NAME)).to.equal('#NAME?');
                expect(ooxEnGrammar.formatScalar(ErrorCode.NUM)).to.equal('#NUM!');
                expect(ooxEnGrammar.formatScalar(ErrorCode.NA)).to.equal('#N/A');
                expect(ooxEnGrammar.formatScalar(ErrorCode.DATA)).to.equal('#GETTING_DATA');
                expect(ooxEnGrammar.formatScalar(null)).to.equal('');
            });
        });

        describe('method "formatGenericRange"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('formatGenericRange');
                expect(ooxUiGrammar).to.respondTo('formatGenericRange');
            });
            var s1 = sh(1), s2 = sh(3), s3 = sh(-1);
            it('should return the string representation of a cell address', function () {
                expect(ooxOpGrammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
                expect(ooxEnGrammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
                expect(odfOpGrammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
                expect(odfUiGrammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
                expect(odfEnGrammar.formatGenericRange(docModel, null, null, 'B3')).to.equal('B3');
            });
            it('should return the string representation of a cell range address', function () {
                expect(ooxOpGrammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
                expect(ooxUiGrammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
                expect(ooxEnGrammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
                expect(odfOpGrammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
                expect(odfUiGrammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
                expect(odfEnGrammar.formatGenericRange(docModel, null, null, 'B3', 'D5')).to.equal('B3:D5');
            });
            it('should return the string representation of a cell address with sheet', function () {
                expect(ooxOpGrammar.formatGenericRange(docModel, s1, null, 'B3')).to.equal('Sheet2!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, s1, null, 'B3')).to.equal('Sheet2!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, s1, null, 'B3')).to.equal('$Sheet2.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, s1, null, 'B3')).to.equal('$Sheet2!B3');
            });
            it('should return the string representation of a cell range address with sheet', function () {
                expect(ooxOpGrammar.formatGenericRange(docModel, s1, null, 'B3', 'D5')).to.equal('Sheet2!B3:D5');
                expect(ooxUiGrammar.formatGenericRange(docModel, s1, null, 'B3', 'D5')).to.equal('Sheet2!B3:D5');
                expect(odfOpGrammar.formatGenericRange(docModel, s1, null, 'B3', 'D5')).to.equal('$Sheet2.B3:D5');
                expect(odfUiGrammar.formatGenericRange(docModel, s1, null, 'B3', 'D5')).to.equal('$Sheet2!B3:D5');
            });
            it('should return the string representation of a reference with sheet range', function () {
                // cell address
                expect(ooxOpGrammar.formatGenericRange(docModel, s1, s2, 'B3')).to.equal('\'Sheet2:Sheet 4\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, s1, s2, 'B3')).to.equal('\'Sheet2:Sheet 4\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, s1, s2, 'B3')).to.equal('$Sheet2.B3:$\'Sheet 4\'.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, s1, s2, 'B3')).to.equal('$Sheet2:$\'Sheet 4\'!B3');
                // cell range address
                expect(ooxOpGrammar.formatGenericRange(docModel, s1, s2, 'B3', 'D5')).to.equal('\'Sheet2:Sheet 4\'!B3:D5');
                expect(ooxUiGrammar.formatGenericRange(docModel, s1, s2, 'B3', 'D5')).to.equal('\'Sheet2:Sheet 4\'!B3:D5');
                expect(odfOpGrammar.formatGenericRange(docModel, s1, s2, 'B3', 'D5')).to.equal('$Sheet2.B3:$\'Sheet 4\'.D5');
                expect(odfUiGrammar.formatGenericRange(docModel, s1, s2, 'B3', 'D5')).to.equal('$Sheet2:$\'Sheet 4\'!B3:D5');
            });
            it('should return the string representation of a sheet reference error', function () {
                // cell address
                expect(ooxOpGrammar.formatGenericRange(docModel, s3, null, 'B3')).to.equal('#REF!!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, s3, null, 'B3')).to.equal('#BEZUG!!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, s3, null, 'B3')).to.equal('$#REF!.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, s3, null, 'B3')).to.equal('#BEZUG!!B3');
                // cell range address
                expect(ooxOpGrammar.formatGenericRange(docModel, s3, null, 'B3', 'D5')).to.equal('#REF!!B3:D5');
                expect(ooxUiGrammar.formatGenericRange(docModel, s3, null, 'B3', 'D5')).to.equal('#BEZUG!!B3:D5');
                expect(odfOpGrammar.formatGenericRange(docModel, s3, null, 'B3', 'D5')).to.equal('$#REF!.B3:D5');
                expect(odfUiGrammar.formatGenericRange(docModel, s3, null, 'B3', 'D5')).to.equal('#BEZUG!!B3:D5');
            });
            it('should handle simple/complex sheet names correctly', function () {
                // simple name
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('Sheet1'), null, 'B3')).to.equal('Sheet1!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('Sheet1'), null, 'B3')).to.equal('Sheet1!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet1'), null, 'B3')).to.equal('$Sheet1.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet1'), null, 'B3')).to.equal('$Sheet1!B3');
                // whitespace
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('Sheet 2'), null, 'B3')).to.equal('\'Sheet 2\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('Sheet 2'), null, 'B3')).to.equal('\'Sheet 2\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet 2'), null, 'B3')).to.equal('$\'Sheet 2\'.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet 2'), null, 'B3')).to.equal('$\'Sheet 2\'!B3');
                // leading digit
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('1Sheet'), null, 'B3')).to.equal('\'1Sheet\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('1Sheet'), null, 'B3')).to.equal('\'1Sheet\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('1Sheet'), null, 'B3')).to.equal('$\'1Sheet\'.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('1Sheet'), null, 'B3')).to.equal('$\'1Sheet\'!B3');
                // operator
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('Sheet+1'), null, 'B3')).to.equal('\'Sheet+1\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('Sheet+1'), null, 'B3')).to.equal('\'Sheet+1\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet+1'), null, 'B3')).to.equal('$\'Sheet+1\'.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet+1'), null, 'B3')).to.equal('$\'Sheet+1\'!B3');
                // simple sheet range
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2'), 'B3')).to.equal('Sheet1:Sheet2!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2'), 'B3')).to.equal('Sheet1:Sheet2!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2'), 'B3')).to.equal('$Sheet1.B3:$Sheet2.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2'), 'B3')).to.equal('$Sheet1:$Sheet2!B3');
                // complex first sheet
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('Sheet 1'), sh('Sheet2'), 'B3')).to.equal('\'Sheet 1:Sheet2\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('Sheet 1'), sh('Sheet2'), 'B3')).to.equal('\'Sheet 1:Sheet2\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet 1'), sh('Sheet2'), 'B3')).to.equal('$\'Sheet 1\'.B3:$Sheet2.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet 1'), sh('Sheet2'), 'B3')).to.equal('$\'Sheet 1\':$Sheet2!B3');
                // complex second sheet
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet 2'), 'B3')).to.equal('\'Sheet1:Sheet 2\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet 2'), 'B3')).to.equal('\'Sheet1:Sheet 2\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet 2'), 'B3')).to.equal('$Sheet1.B3:$\'Sheet 2\'.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet 2'), 'B3')).to.equal('$Sheet1:$\'Sheet 2\'!B3');
                // ODF: relative sheets
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet1', false), null, 'B3')).to.equal('Sheet1.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet1', false), null, 'B3')).to.equal('Sheet1!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet 2', false), null, 'B3')).to.equal('\'Sheet 2\'.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet 2', false), null, 'B3')).to.equal('\'Sheet 2\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet2'), 'B3')).to.equal('Sheet1.B3:$Sheet2.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet2'), 'B3')).to.equal('Sheet1:$Sheet2!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2', false), 'B3')).to.equal('$Sheet1.B3:Sheet2.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet1'), sh('Sheet2', false), 'B3')).to.equal('$Sheet1:Sheet2!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet2', false), 'B3')).to.equal('Sheet1.B3:Sheet2.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet2', false), 'B3')).to.equal('Sheet1:Sheet2!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet 1', false), sh('Sheet2', false), 'B3')).to.equal('\'Sheet 1\'.B3:Sheet2.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet 1', false), sh('Sheet2', false), 'B3')).to.equal('\'Sheet 1\':Sheet2!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet 2', false), 'B3')).to.equal('Sheet1.B3:\'Sheet 2\'.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Sheet1', false), sh('Sheet 2', false), 'B3')).to.equal('Sheet1:\'Sheet 2\'!B3');
            });
            it('should treat booleans as complex sheet names', function () {
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('TRUE'), null, 'B3')).to.equal('\'TRUE\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('TRUE'), null, 'B3')).to.equal('TRUE!B3');
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('WAHR'), null, 'B3')).to.equal('WAHR!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('WAHR'), null, 'B3')).to.equal('\'WAHR\'!B3');
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('TRUE'), sh('WAHR'), 'B3')).to.equal('\'TRUE:WAHR\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('TRUE'), sh('WAHR'), 'B3')).to.equal('\'TRUE:WAHR\'!B3');
            });
            it('should treat cell addresses as complex sheet names', function () {
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('A1'), null, 'B3')).to.equal('\'A1\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('A1'), null, 'B3')).to.equal('\'A1\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('A1'), null, 'B3')).to.equal('$\'A1\'.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('A1'), null, 'B3')).to.equal('$\'A1\'!B3');
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('IW1'), null, 'B3')).to.equal('IW1!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('IW1'), null, 'B3')).to.equal('IW1!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('IW1'), null, 'B3')).to.equal('$IW1.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('IW1'), null, 'B3')).to.equal('$IW1!B3');
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('R1C1'), null, 'B3')).to.equal('\'R1C1\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('R1C1'), null, 'B3')).to.equal('\'R1C1\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('R1C1'), null, 'B3')).to.equal('$\'R1C1\'.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('R1C1'), null, 'B3')).to.equal('$\'R1C1\'!B3');
                expect(ooxOpGrammar.formatGenericRange(docModel, sh('Z1S1'), null, 'B3')).to.equal('Z1S1!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh('Z1S1'), null, 'B3')).to.equal('\'Z1S1\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh('Z1S1'), null, 'B3')).to.equal('$Z1S1.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh('Z1S1'), null, 'B3')).to.equal('$Z1S1!B3');
                expect(ooxOpGrammar.formatGenericRange(docModel, sh(1), sh('A1'), 'B3')).to.equal('\'Sheet2:A1\'!B3');
                expect(ooxUiGrammar.formatGenericRange(docModel, sh(1), sh('A1'), 'B3')).to.equal('\'Sheet2:A1\'!B3');
                expect(odfOpGrammar.formatGenericRange(docModel, sh(1), sh('A1'), 'B3')).to.equal('$Sheet2.B3:$\'A1\'.B3');
                expect(odfUiGrammar.formatGenericRange(docModel, sh(1), sh('A1'), 'B3')).to.equal('$Sheet2:$\'A1\'!B3');
            });
        });

        describe('method "formatReference"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('formatReference');
                expect(ooxUiGrammar).to.respondTo('formatReference');
            });
            var c1 = cell('1.$2'), c2 = cell('$3.4'), s1 = sh(1), s2 = sh(3), s3 = sh(-1);
            it('should return the string representation of a cell address', function () {
                expect(ooxOpGrammar.formatReference(docModel, null, null, c1, null)).to.equal('B$3');
                expect(ooxUiGrammar.formatReference(docModel, null, null, c1, null)).to.equal('B$3');
                expect(ooxEnGrammar.formatReference(docModel, null, null, c1, null)).to.equal('B$3');
                expect(odfOpGrammar.formatReference(docModel, null, null, c1, null)).to.equal('[.B$3]');
                expect(odfUiGrammar.formatReference(docModel, null, null, c1, null)).to.equal('B$3');
                expect(odfEnGrammar.formatReference(docModel, null, null, c1, null)).to.equal('B$3');
                expect(ooxOpGrammar.formatReference(docModel, null, null, c1, null, true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, c1, null, true)).to.equal('Z3S[1]');
                expect(ooxEnGrammar.formatReference(docModel, null, null, c1, null, true)).to.equal('R3C[1]');
                expect(odfOpGrammar.formatReference(docModel, null, null, c1, null, true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, c1, null, true)).to.equal('R3C[1]');
                expect(odfEnGrammar.formatReference(docModel, null, null, c1, null, true)).to.equal('R3C[1]');
            });
            it('should return the string representation of a cell range address', function () {
                expect(ooxOpGrammar.formatReference(docModel, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(ooxUiGrammar.formatReference(docModel, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(ooxEnGrammar.formatReference(docModel, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(odfOpGrammar.formatReference(docModel, null, null, c1, c2)).to.equal('[.B$3:.$D5]');
                expect(odfUiGrammar.formatReference(docModel, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(odfEnGrammar.formatReference(docModel, null, null, c1, c2)).to.equal('B$3:$D5');
                expect(ooxOpGrammar.formatReference(docModel, null, null, c1, c2, true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, c1, c2, true)).to.equal('Z3S[1]:Z[4]S4');
                expect(ooxEnGrammar.formatReference(docModel, null, null, c1, c2, true)).to.equal('R3C[1]:R[4]C4');
                expect(odfOpGrammar.formatReference(docModel, null, null, c1, c2, true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, c1, c2, true)).to.equal('R3C[1]:R[4]C4');
                expect(odfEnGrammar.formatReference(docModel, null, null, c1, c2, true)).to.equal('R3C[1]:R[4]C4');
            });
            it('should return the string representation of a reference error', function () {
                expect(ooxOpGrammar.formatReference(docModel, null, null, null, null)).to.equal('#REF!');
                expect(ooxUiGrammar.formatReference(docModel, null, null, null, null)).to.equal('#BEZUG!');
                expect(ooxEnGrammar.formatReference(docModel, null, null, null, null)).to.equal('#REF!');
                expect(odfOpGrammar.formatReference(docModel, null, null, null, null)).to.equal('[.#REF!#REF!]');
                expect(odfUiGrammar.formatReference(docModel, null, null, null, null)).to.equal('#BEZUG!');
                expect(odfEnGrammar.formatReference(docModel, null, null, null, null)).to.equal('#REF!');
                expect(ooxOpGrammar.formatReference(docModel, null, null, null, null, true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, null, null, true)).to.equal('#BEZUG!');
                expect(ooxEnGrammar.formatReference(docModel, null, null, null, null, true)).to.equal('#REF!');
                expect(odfOpGrammar.formatReference(docModel, null, null, null, null, true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, null, null, true)).to.equal('#BEZUG!');
                expect(odfEnGrammar.formatReference(docModel, null, null, null, null, true)).to.equal('#REF!');
            });
            it('should return the string representation of a cell address with sheet', function () {
                expect(ooxOpGrammar.formatReference(docModel, s1, null, c1, null)).to.equal('Sheet2!B$3');
                expect(ooxUiGrammar.formatReference(docModel, s1, null, c1, null)).to.equal('Sheet2!B$3');
                expect(odfOpGrammar.formatReference(docModel, s1, null, c1, null)).to.equal('[$Sheet2.B$3]');
                expect(odfUiGrammar.formatReference(docModel, s1, null, c1, null)).to.equal('$Sheet2!B$3');
                expect(ooxOpGrammar.formatReference(docModel, s1, null, c1, null, true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, s1, null, c1, null, true)).to.equal('Sheet2!Z3S[1]');
                expect(odfOpGrammar.formatReference(docModel, s1, null, c1, null, true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, s1, null, c1, null, true)).to.equal('$Sheet2!R3C[1]');
            });
            it('should return the string representation of a cell range address with sheet', function () {
                expect(ooxOpGrammar.formatReference(docModel, s1, null, c1, c2)).to.equal('Sheet2!B$3:$D5');
                expect(ooxUiGrammar.formatReference(docModel, s1, null, c1, c2)).to.equal('Sheet2!B$3:$D5');
                expect(odfOpGrammar.formatReference(docModel, s1, null, c1, c2)).to.equal('[$Sheet2.B$3:.$D5]');
                expect(odfUiGrammar.formatReference(docModel, s1, null, c1, c2)).to.equal('$Sheet2!B$3:$D5');
                expect(ooxOpGrammar.formatReference(docModel, s1, null, c1, c2, true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, s1, null, c1, c2, true)).to.equal('Sheet2!Z3S[1]:Z[4]S4');
                expect(odfOpGrammar.formatReference(docModel, s1, null, c1, c2, true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, s1, null, c1, c2, true)).to.equal('$Sheet2!R3C[1]:R[4]C4');
            });
            it('should return the string representation of a reference error with sheet', function () {
                expect(ooxOpGrammar.formatReference(docModel, s1, null, null, null)).to.equal('Sheet2!#REF!');
                expect(ooxUiGrammar.formatReference(docModel, s1, null, null, null)).to.equal('Sheet2!#BEZUG!');
                expect(odfOpGrammar.formatReference(docModel, s1, null, null, null)).to.equal('[$Sheet2.#REF!#REF!]');
                expect(odfUiGrammar.formatReference(docModel, s1, null, null, null)).to.equal('$Sheet2!#BEZUG!');
                expect(ooxOpGrammar.formatReference(docModel, s1, null, null, null, true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, s1, null, null, null, true)).to.equal('Sheet2!#BEZUG!');
                expect(odfOpGrammar.formatReference(docModel, s1, null, null, null, true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, s1, null, null, null, true)).to.equal('$Sheet2!#BEZUG!');
            });
            it('should return the string representation of a reference with sheet range', function () {
                // cell address
                expect(ooxOpGrammar.formatReference(docModel, s1, s2, c1, null)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                expect(ooxUiGrammar.formatReference(docModel, s1, s2, c1, null)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                expect(odfOpGrammar.formatReference(docModel, s1, s2, c1, null)).to.equal('[$Sheet2.B$3:$\'Sheet 4\'.B$3]');
                expect(odfUiGrammar.formatReference(docModel, s1, s2, c1, null)).to.equal('$Sheet2:$\'Sheet 4\'!B$3');
                // cell range address
                expect(ooxOpGrammar.formatReference(docModel, s1, s2, c1, c2)).to.equal('\'Sheet2:Sheet 4\'!B$3:$D5');
                expect(ooxUiGrammar.formatReference(docModel, s1, s2, c1, c2)).to.equal('\'Sheet2:Sheet 4\'!B$3:$D5');
                expect(odfOpGrammar.formatReference(docModel, s1, s2, c1, c2)).to.equal('[$Sheet2.B$3:$\'Sheet 4\'.$D5]');
                expect(odfUiGrammar.formatReference(docModel, s1, s2, c1, c2)).to.equal('$Sheet2:$\'Sheet 4\'!B$3:$D5');
                // reference error
                expect(ooxOpGrammar.formatReference(docModel, s1, s2, null, null)).to.equal('\'Sheet2:Sheet 4\'!#REF!');
                expect(ooxUiGrammar.formatReference(docModel, s1, s2, null, null)).to.equal('\'Sheet2:Sheet 4\'!#BEZUG!');
                expect(odfOpGrammar.formatReference(docModel, s1, s2, null, null)).to.equal('[$Sheet2.#REF!#REF!:$\'Sheet 4\'.#REF!#REF!]');
                expect(odfUiGrammar.formatReference(docModel, s1, s2, null, null)).to.equal('$Sheet2:$\'Sheet 4\'!#BEZUG!');
            });
            it('should return the string representation of a sheet reference error', function () {
                // cell address
                expect(ooxOpGrammar.formatReference(docModel, s3, null, c1, null)).to.equal('#REF!!B$3');
                expect(ooxUiGrammar.formatReference(docModel, s3, null, c1, null)).to.equal('#BEZUG!!B$3');
                expect(odfOpGrammar.formatReference(docModel, s3, null, c1, null)).to.equal('[$#REF!.B$3]');
                expect(odfUiGrammar.formatReference(docModel, s3, null, c1, null)).to.equal('#BEZUG!!B$3');
                // cell range address
                expect(ooxOpGrammar.formatReference(docModel, s3, null, c1, c2)).to.equal('#REF!!B$3:$D5');
                expect(ooxUiGrammar.formatReference(docModel, s3, null, c1, c2)).to.equal('#BEZUG!!B$3:$D5');
                expect(odfOpGrammar.formatReference(docModel, s3, null, c1, c2)).to.equal('[$#REF!.B$3:.$D5]');
                expect(odfUiGrammar.formatReference(docModel, s3, null, c1, c2)).to.equal('#BEZUG!!B$3:$D5');
                // reference error
                expect(ooxOpGrammar.formatReference(docModel, s3, null, null, null)).to.equal('#REF!!#REF!');
                expect(ooxUiGrammar.formatReference(docModel, s3, null, null, null)).to.equal('#BEZUG!!#BEZUG!');
                expect(odfOpGrammar.formatReference(docModel, s3, null, null, null)).to.equal('[$#REF!.#REF!#REF!]');
                expect(odfUiGrammar.formatReference(docModel, s3, null, null, null)).to.equal('#BEZUG!!#BEZUG!');
            });
            it('should return string representation of a column range', function () {
                // relative->absolute
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('B:$D');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('B:$D');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('[.B$1:.$D$65536]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'))).to.equal('B:$D');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'), true)).to.equal('S[1]:S4');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('$3.$65535'), true)).to.equal('C[1]:C4');
                // absolute->relative
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('$B:D');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('$B:D');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('[.$B$1:.D$65536]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'))).to.equal('$B:D');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'), true)).to.equal('S2:S[3]');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('3.$65535'), true)).to.equal('C2:C[3]');
            });
            it('should return string representation of a single column', function () {
                // absolute
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('$B:$B');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('$B:$B');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('[.$B$1:.$B$65536]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'))).to.equal('$B:$B');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'), true)).to.equal('S2');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$1.$65535'), true)).to.equal('C2');
                // relative
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('B:B');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('B:B');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('[.B$1:.B$65536]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'))).to.equal('B:B');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'), true)).to.equal('S[1]');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('1.$0'), cell('1.$65535'), true)).to.equal('C[1]');
            });
            it('should return string representation of a column range with sheet', function () {
                // single sheet
                expect(ooxOpGrammar.formatReference(docModel, sh(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('Sheet2!$B:B');
                expect(ooxUiGrammar.formatReference(docModel, sh(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('Sheet2!$B:B');
                expect(odfOpGrammar.formatReference(docModel, sh(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('[$Sheet2.$B$1:.B$65536]');
                expect(odfUiGrammar.formatReference(docModel, sh(1), null, cell('$1.$0'), cell('1.$65535'))).to.equal('$Sheet2!$B:B');
                expect(ooxOpGrammar.formatReference(docModel, sh(1), null, cell('$1.$0'), cell('1.$65535'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, sh(1), null, cell('$1.$0'), cell('1.$65535'), true)).to.equal('Sheet2!S2:S[1]');
                expect(odfOpGrammar.formatReference(docModel, sh(1), null, cell('$1.$0'), cell('1.$65535'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, sh(1), null, cell('$1.$0'), cell('1.$65535'), true)).to.equal('$Sheet2!C2:C[1]');
                // sheet range
                expect(ooxOpGrammar.formatReference(docModel, sh(1), sh(3), cell('$1.$0'), cell('1.$65535'))).to.equal('\'Sheet2:Sheet 4\'!$B:B');
                expect(ooxUiGrammar.formatReference(docModel, sh(1), sh(3), cell('$1.$0'), cell('1.$65535'))).to.equal('\'Sheet2:Sheet 4\'!$B:B');
                expect(odfOpGrammar.formatReference(docModel, sh(1), sh(3), cell('$1.$0'), cell('1.$65535'))).to.equal('[$Sheet2.$B$1:$\'Sheet 4\'.B$65536]');
                expect(odfUiGrammar.formatReference(docModel, sh(1), sh(3), cell('$1.$0'), cell('1.$65535'))).to.equal('$Sheet2:$\'Sheet 4\'!$B:B');
            });
            it('should not return string representation of a column range', function () {
                // missing absolute flags
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('$B$1:$D65536');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('$B$1:$D65536');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('[.$B$1:.$D65536]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.65535'))).to.equal('$B$1:$D65536');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('$B1:$D$65536');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('$B1:$D$65536');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('[.$B1:.$D$65536]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$1.0'), cell('$3.$65535'))).to.equal('$B1:$D$65536');
                // wrong indexes
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('$B$2:$D$65536');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('$B$2:$D$65536');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('[.$B$2:.$D$65536]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$1.$1'), cell('$3.$65535'))).to.equal('$B$2:$D$65536');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('$B$1:$D$65535');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('$B$1:$D$65535');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('[.$B$1:.$D$65535]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$1.$0'), cell('$3.$65534'))).to.equal('$B$1:$D$65535');
            });
            it('should return string representation of a row range', function () {
                // relative->absolute
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('2:$4');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('2:$4');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('[.$A2:.$IV$4]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'))).to.equal('2:$4');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'), true)).to.equal('Z[1]:Z4');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.$3'), true)).to.equal('R[1]:R4');
                // absolute->relative
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('$2:4');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('$2:4');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('[.$A$2:.$IV4]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'))).to.equal('$2:4');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'), true)).to.equal('Z2:Z[3]');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.3'), true)).to.equal('R2:R[3]');
            });
            it('should return string representation of a single row', function () {
                // absolute
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('$2:$2');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('$2:$2');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('[.$A$2:.$IV$2]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'))).to.equal('$2:$2');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'), true)).to.equal('Z2');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$255.$1'), true)).to.equal('R2');
                // relative
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'))).to.equal('2:2');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'))).to.equal('2:2');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'))).to.equal('[.$A2:.$IV2]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'))).to.equal('2:2');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'), true)).to.equal('Z[1]');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.1'), cell('$255.1'), true)).to.equal('R[1]');
            });
            it('should return string representation of a row range with sheet', function () {
                // single sheet
                expect(ooxOpGrammar.formatReference(docModel, sh(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('Sheet2!$2:2');
                expect(ooxUiGrammar.formatReference(docModel, sh(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('Sheet2!$2:2');
                expect(odfOpGrammar.formatReference(docModel, sh(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('[$Sheet2.$A$2:.$IV2]');
                expect(odfUiGrammar.formatReference(docModel, sh(1), null, cell('$0.$1'), cell('$255.1'))).to.equal('$Sheet2!$2:2');
                expect(ooxOpGrammar.formatReference(docModel, sh(1), null, cell('$0.$1'), cell('$255.1'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, sh(1), null, cell('$0.$1'), cell('$255.1'), true)).to.equal('Sheet2!Z2:Z[1]');
                expect(odfOpGrammar.formatReference(docModel, sh(1), null, cell('$0.$1'), cell('$255.1'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, sh(1), null, cell('$0.$1'), cell('$255.1'), true)).to.equal('$Sheet2!R2:R[1]');
                // sheet range
                expect(ooxOpGrammar.formatReference(docModel, sh(1), sh(3), cell('$0.$1'), cell('$255.1'))).to.equal('\'Sheet2:Sheet 4\'!$2:2');
                expect(ooxUiGrammar.formatReference(docModel, sh(1), sh(3), cell('$0.$1'), cell('$255.1'))).to.equal('\'Sheet2:Sheet 4\'!$2:2');
                expect(odfOpGrammar.formatReference(docModel, sh(1), sh(3), cell('$0.$1'), cell('$255.1'))).to.equal('[$Sheet2.$A$2:$\'Sheet 4\'.$IV2]');
                expect(odfUiGrammar.formatReference(docModel, sh(1), sh(3), cell('$0.$1'), cell('$255.1'))).to.equal('$Sheet2:$\'Sheet 4\'!$2:2');
            });
            it('should not return string representation of a row range', function () {
                // missing absolute flags
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('$A$2:IV$4');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('$A$2:IV$4');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('[.$A$2:.IV$4]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('255.$3'))).to.equal('$A$2:IV$4');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('A$2:$IV$4');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('A$2:$IV$4');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('[.A$2:.$IV$4]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('0.$1'), cell('$255.$3'))).to.equal('A$2:$IV$4');
                // wrong indexes
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('$B$2:$IV$4');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('$B$2:$IV$4');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('[.$B$2:.$IV$4]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$1.$1'), cell('$255.$3'))).to.equal('$B$2:$IV$4');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('$A$2:$IU$4');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('$A$2:$IU$4');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('[.$A$2:.$IU$4]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.$1'), cell('$254.$3'))).to.equal('$A$2:$IU$4');
            });
            it('should return string representation of a sheet range', function () {
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('$1:$65536');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('$1:$65536');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('[.$A$1:.$IV$65536]');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'))).to.equal('$1:$65536');
                expect(ooxOpGrammar.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'), true)).to.equal('#N/A');
                expect(ooxUiGrammar.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'), true)).to.equal('Z1:Z65536');
                expect(odfOpGrammar.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'), true)).to.equal('#N/A');
                expect(odfUiGrammar.formatReference(docModel, null, null, cell('$0.$0'), cell('$255.$65535'), true)).to.equal('R1:R65536');
            });
            it('should handle simple/complex sheet names correctly', function () {
                var c1 = cell('0.0');
                // simple name
                expect(ooxOpGrammar.formatReference(docModel, sh('Sheet1'), null, c1, null)).to.equal('Sheet1!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('Sheet1'), null, c1, null)).to.equal('Sheet1!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet1'), null, c1, null)).to.equal('[$Sheet1.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet1'), null, c1, null)).to.equal('$Sheet1!A1');
                // whitespace
                expect(ooxOpGrammar.formatReference(docModel, sh('Sheet 2'), null, c1, null)).to.equal('\'Sheet 2\'!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('Sheet 2'), null, c1, null)).to.equal('\'Sheet 2\'!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet 2'), null, c1, null)).to.equal('[$\'Sheet 2\'.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet 2'), null, c1, null)).to.equal('$\'Sheet 2\'!A1');
                // leading digit
                expect(ooxOpGrammar.formatReference(docModel, sh('1Sheet'), null, c1, null)).to.equal('\'1Sheet\'!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('1Sheet'), null, c1, null)).to.equal('\'1Sheet\'!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('1Sheet'), null, c1, null)).to.equal('[$\'1Sheet\'.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('1Sheet'), null, c1, null)).to.equal('$\'1Sheet\'!A1');
                // operator
                expect(ooxOpGrammar.formatReference(docModel, sh('Sheet+1'), null, c1, null)).to.equal('\'Sheet+1\'!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('Sheet+1'), null, c1, null)).to.equal('\'Sheet+1\'!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet+1'), null, c1, null)).to.equal('[$\'Sheet+1\'.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet+1'), null, c1, null)).to.equal('$\'Sheet+1\'!A1');
                // simple sheet range
                expect(ooxOpGrammar.formatReference(docModel, sh('Sheet1'), sh('Sheet2'), c1, null)).to.equal('Sheet1:Sheet2!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('Sheet1'), sh('Sheet2'), c1, null)).to.equal('Sheet1:Sheet2!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet1'), sh('Sheet2'), c1, null)).to.equal('[$Sheet1.A1:$Sheet2.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet1'), sh('Sheet2'), c1, null)).to.equal('$Sheet1:$Sheet2!A1');
                // complex first sheet
                expect(ooxOpGrammar.formatReference(docModel, sh('Sheet 1'), sh('Sheet2'), c1, null)).to.equal('\'Sheet 1:Sheet2\'!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('Sheet 1'), sh('Sheet2'), c1, null)).to.equal('\'Sheet 1:Sheet2\'!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet 1'), sh('Sheet2'), c1, null)).to.equal('[$\'Sheet 1\'.A1:$Sheet2.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet 1'), sh('Sheet2'), c1, null)).to.equal('$\'Sheet 1\':$Sheet2!A1');
                // complex second sheet
                expect(ooxOpGrammar.formatReference(docModel, sh('Sheet1'), sh('Sheet 2'), c1, null)).to.equal('\'Sheet1:Sheet 2\'!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('Sheet1'), sh('Sheet 2'), c1, null)).to.equal('\'Sheet1:Sheet 2\'!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet1'), sh('Sheet 2'), c1, null)).to.equal('[$Sheet1.A1:$\'Sheet 2\'.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet1'), sh('Sheet 2'), c1, null)).to.equal('$Sheet1:$\'Sheet 2\'!A1');
                // ODF: relative sheets
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet1', false), null, c1, null)).to.equal('[Sheet1.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet1', false), null, c1, null)).to.equal('Sheet1!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet 2', false), null, c1, null)).to.equal('[\'Sheet 2\'.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet 2', false), null, c1, null)).to.equal('\'Sheet 2\'!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet1', false), sh('Sheet2'), c1, null)).to.equal('[Sheet1.A1:$Sheet2.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet1', false), sh('Sheet2'), c1, null)).to.equal('Sheet1:$Sheet2!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet1'), sh('Sheet2', false), c1, null)).to.equal('[$Sheet1.A1:Sheet2.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet1'), sh('Sheet2', false), c1, null)).to.equal('$Sheet1:Sheet2!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet1', false), sh('Sheet2', false), c1, null)).to.equal('[Sheet1.A1:Sheet2.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet1', false), sh('Sheet2', false), c1, null)).to.equal('Sheet1:Sheet2!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet 1', false), sh('Sheet2', false), c1, null)).to.equal('[\'Sheet 1\'.A1:Sheet2.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet 1', false), sh('Sheet2', false), c1, null)).to.equal('\'Sheet 1\':Sheet2!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Sheet1', false), sh('Sheet 2', false), c1, null)).to.equal('[Sheet1.A1:\'Sheet 2\'.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Sheet1', false), sh('Sheet 2', false), c1, null)).to.equal('Sheet1:\'Sheet 2\'!A1');
            });
            it('should treat booleans as complex sheet names', function () {
                var c1 = cell('0.0');
                expect(ooxOpGrammar.formatReference(docModel, sh('TRUE'), null, c1, null)).to.equal('\'TRUE\'!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('TRUE'), null, c1, null)).to.equal('TRUE!A1');
                expect(ooxOpGrammar.formatReference(docModel, sh('WAHR'), null, c1, null)).to.equal('WAHR!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('WAHR'), null, c1, null)).to.equal('\'WAHR\'!A1');
                expect(ooxOpGrammar.formatReference(docModel, sh('TRUE'), sh('WAHR'), c1, null)).to.equal('\'TRUE:WAHR\'!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('TRUE'), sh('WAHR'), c1, null)).to.equal('\'TRUE:WAHR\'!A1');
            });
            it('should treat cell addresses as complex sheet names', function () {
                var c1 = cell('0.0');
                expect(ooxOpGrammar.formatReference(docModel, sh('A1'), null, c1, null)).to.equal('\'A1\'!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('A1'), null, c1, null)).to.equal('\'A1\'!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('A1'), null, c1, null)).to.equal('[$\'A1\'.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('A1'), null, c1, null)).to.equal('$\'A1\'!A1');
                expect(ooxOpGrammar.formatReference(docModel, sh('IW1'), null, c1, null)).to.equal('IW1!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('IW1'), null, c1, null)).to.equal('IW1!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('IW1'), null, c1, null)).to.equal('[$IW1.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('IW1'), null, c1, null)).to.equal('$IW1!A1');
                expect(ooxOpGrammar.formatReference(docModel, sh('R1C1'), null, c1, null)).to.equal('\'R1C1\'!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('R1C1'), null, c1, null)).to.equal('\'R1C1\'!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('R1C1'), null, c1, null)).to.equal('[$\'R1C1\'.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('R1C1'), null, c1, null)).to.equal('$\'R1C1\'!A1');
                expect(ooxOpGrammar.formatReference(docModel, sh('Z1S1'), null, c1, null)).to.equal('Z1S1!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh('Z1S1'), null, c1, null)).to.equal('\'Z1S1\'!A1');
                expect(odfOpGrammar.formatReference(docModel, sh('Z1S1'), null, c1, null)).to.equal('[$Z1S1.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh('Z1S1'), null, c1, null)).to.equal('$Z1S1!A1');
                expect(ooxOpGrammar.formatReference(docModel, sh(1), sh('A1'), c1, null)).to.equal('\'Sheet2:A1\'!A1');
                expect(ooxUiGrammar.formatReference(docModel, sh(1), sh('A1'), c1, null)).to.equal('\'Sheet2:A1\'!A1');
                expect(odfOpGrammar.formatReference(docModel, sh(1), sh('A1'), c1, null)).to.equal('[$Sheet2.A1:$\'A1\'.A1]');
                expect(odfUiGrammar.formatReference(docModel, sh(1), sh('A1'), c1, null)).to.equal('$Sheet2:$\'A1\'!A1');
            });
        });

        describe('method "formatName"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('formatName');
                expect(ooxUiGrammar).to.respondTo('formatName');
            });
            it('should return the string representation', function () {
                var s1 = sh(1), s2 = sh(3), s3 = sh(-1);
                expect(ooxOpGrammar.formatName(docModel, null, 'name')).to.equal('name');
                expect(ooxUiGrammar.formatName(docModel, null, 'name')).to.equal('name');
                expect(ooxOpGrammar.formatName(docModel, s1, 'Name')).to.equal('Sheet2!Name');
                expect(ooxUiGrammar.formatName(docModel, s1, 'Name')).to.equal('Sheet2!Name');
                expect(ooxOpGrammar.formatName(docModel, s2, 'NAME')).to.equal('\'Sheet 4\'!NAME');
                expect(ooxUiGrammar.formatName(docModel, s2, 'NAME')).to.equal('\'Sheet 4\'!NAME');
                expect(ooxOpGrammar.formatName(docModel, s3, 'name')).to.equal('#REF!!name');
                expect(ooxUiGrammar.formatName(docModel, s3, 'name')).to.equal('#BEZUG!!name');
            });
            it('should handle simple/complex sheet names correctly', function () {
                expect(ooxOpGrammar.formatName(docModel, sh('Sheet1'), 'name')).to.equal('Sheet1!name');
                expect(ooxUiGrammar.formatName(docModel, sh('Sheet1'), 'name')).to.equal('Sheet1!name');
                expect(ooxOpGrammar.formatName(docModel, sh('Sheet 2'), 'name')).to.equal('\'Sheet 2\'!name');
                expect(ooxUiGrammar.formatName(docModel, sh('Sheet 2'), 'name')).to.equal('\'Sheet 2\'!name');
                expect(ooxOpGrammar.formatName(docModel, sh('1Sheet'), 'name')).to.equal('\'1Sheet\'!name');
                expect(ooxUiGrammar.formatName(docModel, sh('1Sheet'), 'name')).to.equal('\'1Sheet\'!name');
                expect(ooxOpGrammar.formatName(docModel, sh('Sheet+1'), 'name')).to.equal('\'Sheet+1\'!name');
                expect(ooxUiGrammar.formatName(docModel, sh('Sheet+1'), 'name')).to.equal('\'Sheet+1\'!name');
            });
            it('should treat booleans as complex sheet names', function () {
                expect(ooxOpGrammar.formatName(docModel, sh('TRUE'), 'name')).to.equal('\'TRUE\'!name');
                expect(ooxUiGrammar.formatName(docModel, sh('TRUE'), 'name')).to.equal('TRUE!name');
                expect(ooxOpGrammar.formatName(docModel, sh('WAHR'), 'name')).to.equal('WAHR!name');
                expect(ooxUiGrammar.formatName(docModel, sh('WAHR'), 'name')).to.equal('\'WAHR\'!name');
            });
            it('should treat cell addresses as complex sheet names', function () {
                expect(ooxOpGrammar.formatName(docModel, sh('A1'), 'name')).to.equal('\'A1\'!name');
                expect(ooxUiGrammar.formatName(docModel, sh('A1'), 'name')).to.equal('\'A1\'!name');
                expect(ooxOpGrammar.formatName(docModel, sh('IW1'), 'name')).to.equal('IW1!name');
                expect(ooxUiGrammar.formatName(docModel, sh('IW1'), 'name')).to.equal('IW1!name');
                expect(ooxOpGrammar.formatName(docModel, sh('R1C1'), 'name')).to.equal('\'R1C1\'!name');
                expect(ooxUiGrammar.formatName(docModel, sh('R1C1'), 'name')).to.equal('\'R1C1\'!name');
                expect(ooxOpGrammar.formatName(docModel, sh('Z1S1'), 'name')).to.equal('Z1S1!name');
                expect(ooxUiGrammar.formatName(docModel, sh('Z1S1'), 'name')).to.equal('\'Z1S1\'!name');
            });
        });

        describe('method "formatMacro"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('formatMacro');
                expect(ooxUiGrammar).to.respondTo('formatMacro');
            });
            var s1 = sh(1), s2 = sh(3), s3 = sh(-1);
            it('should return the string representation', function () {
                expect(ooxOpGrammar.formatMacro(docModel, null, 'func')).to.equal('func');
                expect(ooxUiGrammar.formatMacro(docModel, null, 'func')).to.equal('func');
                expect(ooxOpGrammar.formatMacro(docModel, s1, 'Func')).to.equal('Sheet2!Func');
                expect(ooxUiGrammar.formatMacro(docModel, s1, 'Func')).to.equal('Sheet2!Func');
                expect(ooxOpGrammar.formatMacro(docModel, s2, 'FUNC')).to.equal('\'Sheet 4\'!FUNC');
                expect(ooxUiGrammar.formatMacro(docModel, s2, 'FUNC')).to.equal('\'Sheet 4\'!FUNC');
                expect(ooxOpGrammar.formatMacro(docModel, s3, 'func')).to.equal('#REF!!func');
                expect(ooxUiGrammar.formatMacro(docModel, s3, 'func')).to.equal('#BEZUG!!func');
            });
            it('should omit UDF prefix in UI grammar', function () {
                expect(ooxOpGrammar.formatMacro(docModel, null, '_xludf.sum')).to.equal('_xludf.sum');
                expect(ooxUiGrammar.formatMacro(docModel, null, '_xludf.sum')).to.equal('sum');
                expect(ooxOpGrammar.formatMacro(docModel, s1, '_xludf.Sum')).to.equal('Sheet2!_xludf.Sum');
                expect(ooxUiGrammar.formatMacro(docModel, s1, '_xludf.Sum')).to.equal('Sheet2!Sum');
                expect(ooxOpGrammar.formatMacro(docModel, s2, '_xludf.SUM')).to.equal('\'Sheet 4\'!_xludf.SUM');
                expect(ooxUiGrammar.formatMacro(docModel, s2, '_xludf.SUM')).to.equal('\'Sheet 4\'!SUM');
                expect(ooxOpGrammar.formatMacro(docModel, s3, '_xludf.sum')).to.equal('#REF!!_xludf.sum');
                expect(ooxUiGrammar.formatMacro(docModel, s3, '_xludf.sum')).to.equal('#BEZUG!!sum');
            });
            it('should add UDF prefix in operations', function () {
                expect(ooxOpGrammar.formatMacro(docModel, null, 'sum')).to.equal('_xludf.sum');
                expect(ooxUiGrammar.formatMacro(docModel, null, 'sum')).to.equal('sum');
                expect(ooxOpGrammar.formatMacro(docModel, s1, 'Sum')).to.equal('Sheet2!_xludf.Sum');
                expect(ooxUiGrammar.formatMacro(docModel, s1, 'Sum')).to.equal('Sheet2!Sum');
                expect(ooxOpGrammar.formatMacro(docModel, s2, 'SUM')).to.equal('\'Sheet 4\'!_xludf.SUM');
                expect(ooxUiGrammar.formatMacro(docModel, s2, 'SUM')).to.equal('\'Sheet 4\'!SUM');
                expect(ooxOpGrammar.formatMacro(docModel, s3, 'sum')).to.equal('#REF!!_xludf.sum');
                expect(ooxUiGrammar.formatMacro(docModel, s3, 'sum')).to.equal('#BEZUG!!sum');
            });
        });

        describe('method "getBooleanName"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('getBooleanName');
                expect(ooxUiGrammar).to.respondTo('getBooleanName');
            });
            it('should return native boolean names', function () {
                expect(ooxOpGrammar.getBooleanName(false)).to.equal('FALSE');
                expect(ooxOpGrammar.getBooleanName(true)).to.equal('TRUE');
            });
            it('should return localized boolean names', function () {
                expect(ooxUiGrammar.getBooleanName(false)).to.equal('FALSCH');
                expect(ooxUiGrammar.getBooleanName(true)).to.equal('WAHR');
            });
            it('should return English boolean names', function () {
                expect(ooxEnGrammar.getBooleanName(false)).to.equal('FALSE');
                expect(ooxEnGrammar.getBooleanName(true)).to.equal('TRUE');
            });
        });

        describe('method "getBooleanValue"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('getBooleanValue');
                expect(ooxUiGrammar).to.respondTo('getBooleanValue');
            });
            it('should parse native boolean names', function () {
                expect(ooxOpGrammar.getBooleanValue('false')).to.equal(false);
                expect(ooxOpGrammar.getBooleanValue('False')).to.equal(false);
                expect(ooxOpGrammar.getBooleanValue('FALSE')).to.equal(false);
                expect(ooxOpGrammar.getBooleanValue('true')).to.equal(true);
                expect(ooxOpGrammar.getBooleanValue('True')).to.equal(true);
                expect(ooxOpGrammar.getBooleanValue('TRUE')).to.equal(true);
            });
            it('should parse localized boolean names', function () {
                expect(ooxUiGrammar.getBooleanValue('falsch')).to.equal(false);
                expect(ooxUiGrammar.getBooleanValue('Falsch')).to.equal(false);
                expect(ooxUiGrammar.getBooleanValue('FALSCH')).to.equal(false);
                expect(ooxUiGrammar.getBooleanValue('wahr')).to.equal(true);
                expect(ooxUiGrammar.getBooleanValue('Wahr')).to.equal(true);
                expect(ooxUiGrammar.getBooleanValue('WAHR')).to.equal(true);
            });
            it('should parse English boolean names', function () {
                expect(ooxEnGrammar.getBooleanValue('false')).to.equal(false);
                expect(ooxEnGrammar.getBooleanValue('False')).to.equal(false);
                expect(ooxEnGrammar.getBooleanValue('FALSE')).to.equal(false);
                expect(ooxEnGrammar.getBooleanValue('true')).to.equal(true);
                expect(ooxEnGrammar.getBooleanValue('True')).to.equal(true);
                expect(ooxEnGrammar.getBooleanValue('TRUE')).to.equal(true);
            });
            it('should return null for invalid names', function () {
                expect(ooxOpGrammar.getBooleanValue('1')).to.equal(null);
                expect(ooxOpGrammar.getBooleanValue('abc')).to.equal(null);
                expect(ooxOpGrammar.getBooleanValue('FALSCH')).to.equal(null);
                expect(ooxUiGrammar.getBooleanValue('1')).to.equal(null);
                expect(ooxUiGrammar.getBooleanValue('abc')).to.equal(null);
                expect(ooxUiGrammar.getBooleanValue('FALSE')).to.equal(null);
                expect(ooxEnGrammar.getBooleanValue('1')).to.equal(null);
                expect(ooxEnGrammar.getBooleanValue('abc')).to.equal(null);
                expect(ooxEnGrammar.getBooleanValue('FALSCH')).to.equal(null);
            });
        });

        describe('method "getErrorName"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('getErrorName');
                expect(ooxUiGrammar).to.respondTo('getErrorName');
            });
            it('should return native error names', function () {
                expect(ooxOpGrammar.getErrorName(ErrorCode.NULL)).to.equal('#NULL!');
                expect(ooxOpGrammar.getErrorName(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(ooxOpGrammar.getErrorName(ErrorCode.VALUE)).to.equal('#VALUE!');
                expect(ooxOpGrammar.getErrorName(ErrorCode.REF)).to.equal('#REF!');
                expect(ooxOpGrammar.getErrorName(ErrorCode.NAME)).to.equal('#NAME?');
                expect(ooxOpGrammar.getErrorName(ErrorCode.NUM)).to.equal('#NUM!');
                expect(ooxOpGrammar.getErrorName(ErrorCode.NA)).to.equal('#N/A');
                expect(ooxOpGrammar.getErrorName(ErrorCode.DATA)).to.equal('#GETTING_DATA');
            });
            it('should return localized error names', function () {
                expect(ooxUiGrammar.getErrorName(ErrorCode.NULL)).to.equal('#NULL!');
                expect(ooxUiGrammar.getErrorName(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(ooxUiGrammar.getErrorName(ErrorCode.VALUE)).to.equal('#WERT!');
                expect(ooxUiGrammar.getErrorName(ErrorCode.REF)).to.equal('#BEZUG!');
                expect(ooxUiGrammar.getErrorName(ErrorCode.NAME)).to.equal('#NAME?');
                expect(ooxUiGrammar.getErrorName(ErrorCode.NUM)).to.equal('#ZAHL!');
                expect(ooxUiGrammar.getErrorName(ErrorCode.NA)).to.equal('#NV');
                expect(ooxUiGrammar.getErrorName(ErrorCode.DATA)).to.equal('#GETTING_DATA');
            });
            it('should return English error names', function () {
                expect(ooxEnGrammar.getErrorName(ErrorCode.NULL)).to.equal('#NULL!');
                expect(ooxEnGrammar.getErrorName(ErrorCode.DIV0)).to.equal('#DIV/0!');
                expect(ooxEnGrammar.getErrorName(ErrorCode.VALUE)).to.equal('#VALUE!');
                expect(ooxEnGrammar.getErrorName(ErrorCode.REF)).to.equal('#REF!');
                expect(ooxEnGrammar.getErrorName(ErrorCode.NAME)).to.equal('#NAME?');
                expect(ooxEnGrammar.getErrorName(ErrorCode.NUM)).to.equal('#NUM!');
                expect(ooxEnGrammar.getErrorName(ErrorCode.NA)).to.equal('#N/A');
                expect(ooxEnGrammar.getErrorName(ErrorCode.DATA)).to.equal('#GETTING_DATA');
            });
        });

        describe('method "getErrorCode"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('getErrorCode');
                expect(ooxUiGrammar).to.respondTo('getErrorCode');
            });
            it('should parse native error names', function () {
                expect(ooxOpGrammar.getErrorCode('#NULL!')).to.equal(ErrorCode.NULL);
                expect(ooxOpGrammar.getErrorCode('#null!')).to.equal(ErrorCode.NULL);
                expect(ooxOpGrammar.getErrorCode('#DIV/0!')).to.equal(ErrorCode.DIV0);
                expect(ooxOpGrammar.getErrorCode('#VALUE!')).to.equal(ErrorCode.VALUE);
                expect(ooxOpGrammar.getErrorCode('#REF!')).to.equal(ErrorCode.REF);
                expect(ooxOpGrammar.getErrorCode('#NAME?')).to.equal(ErrorCode.NAME);
                expect(ooxOpGrammar.getErrorCode('#NUM!')).to.equal(ErrorCode.NUM);
                expect(ooxOpGrammar.getErrorCode('#N/A')).to.equal(ErrorCode.NA);
                expect(ooxOpGrammar.getErrorCode('#GETTING_DATA')).to.equal(ErrorCode.DATA);
            });
            it('should parse localized error names', function () {
                expect(ooxUiGrammar.getErrorCode('#NULL!')).to.equal(ErrorCode.NULL);
                expect(ooxUiGrammar.getErrorCode('#DIV/0!')).to.equal(ErrorCode.DIV0);
                expect(ooxUiGrammar.getErrorCode('#WERT!')).to.equal(ErrorCode.VALUE);
                expect(ooxUiGrammar.getErrorCode('#wert!')).to.equal(ErrorCode.VALUE);
                expect(ooxUiGrammar.getErrorCode('#BEZUG!')).to.equal(ErrorCode.REF);
                expect(ooxUiGrammar.getErrorCode('#NAME?')).to.equal(ErrorCode.NAME);
                expect(ooxUiGrammar.getErrorCode('#ZAHL!')).to.equal(ErrorCode.NUM);
                expect(ooxUiGrammar.getErrorCode('#NV')).to.equal(ErrorCode.NA);
                expect(ooxUiGrammar.getErrorCode('#GETTING_DATA')).to.equal(ErrorCode.DATA);
            });
            it('should parse English error names', function () {
                expect(ooxEnGrammar.getErrorCode('#NULL!')).to.equal(ErrorCode.NULL);
                expect(ooxEnGrammar.getErrorCode('#null!')).to.equal(ErrorCode.NULL);
                expect(ooxEnGrammar.getErrorCode('#DIV/0!')).to.equal(ErrorCode.DIV0);
                expect(ooxEnGrammar.getErrorCode('#VALUE!')).to.equal(ErrorCode.VALUE);
                expect(ooxEnGrammar.getErrorCode('#REF!')).to.equal(ErrorCode.REF);
                expect(ooxEnGrammar.getErrorCode('#NAME?')).to.equal(ErrorCode.NAME);
                expect(ooxEnGrammar.getErrorCode('#NUM!')).to.equal(ErrorCode.NUM);
                expect(ooxEnGrammar.getErrorCode('#N/A')).to.equal(ErrorCode.NA);
                expect(ooxEnGrammar.getErrorCode('#GETTING_DATA')).to.equal(ErrorCode.DATA);
            });
            it('should return null for invalid names', function () {
                expect(ooxOpGrammar.getErrorCode('abc')).to.equal(null);
                expect(ooxOpGrammar.getErrorCode('#NULL')).to.equal(null);
                expect(ooxOpGrammar.getErrorCode('NULL')).to.equal(null);
                expect(ooxOpGrammar.getErrorCode('#WERT!')).to.equal(null);
                expect(ooxUiGrammar.getErrorCode('abc')).to.equal(null);
                expect(ooxUiGrammar.getErrorCode('#NULL')).to.equal(null);
                expect(ooxUiGrammar.getErrorCode('NULL')).to.equal(null);
                expect(ooxUiGrammar.getErrorCode('#VALUE!')).to.equal(null);
                expect(ooxEnGrammar.getErrorCode('abc')).to.equal(null);
                expect(ooxEnGrammar.getErrorCode('#NULL')).to.equal(null);
                expect(ooxEnGrammar.getErrorCode('NULL')).to.equal(null);
                expect(ooxEnGrammar.getErrorCode('#WERT!')).to.equal(null);
            });
        });

        describe('method "getOperatorName"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('getOperatorName');
                expect(ooxUiGrammar).to.respondTo('getOperatorName');
            });
            it('should return native operator names', function () {
                expect(ooxOpGrammar.getOperatorName('add')).to.equal('+');
                expect(ooxOpGrammar.getOperatorName('le')).to.equal('<=');
                expect(ooxOpGrammar.getOperatorName('list')).to.equal(',');
                expect(ooxOpGrammar.getOperatorName('isect')).to.equal(' ');
                expect(odfOpGrammar.getOperatorName('add')).to.equal('+');
                expect(odfOpGrammar.getOperatorName('le')).to.equal('<=');
                expect(odfOpGrammar.getOperatorName('list')).to.equal('~');
                expect(odfOpGrammar.getOperatorName('isect')).to.equal('!');
            });
            it('should return localized operator names', function () {
                expect(ooxUiGrammar.getOperatorName('add')).to.equal('+');
                expect(ooxUiGrammar.getOperatorName('le')).to.equal('<=');
                expect(ooxUiGrammar.getOperatorName('list')).to.equal(';');
                expect(ooxUiGrammar.getOperatorName('isect')).to.equal(' ');
                expect(odfUiGrammar.getOperatorName('add')).to.equal('+');
                expect(odfUiGrammar.getOperatorName('le')).to.equal('<=');
                expect(odfUiGrammar.getOperatorName('list')).to.equal(';');
                expect(odfUiGrammar.getOperatorName('isect')).to.equal(' ');
            });
            it('should return English operator names', function () {
                expect(ooxEnGrammar.getOperatorName('add')).to.equal('+');
                expect(ooxEnGrammar.getOperatorName('le')).to.equal('<=');
                expect(ooxEnGrammar.getOperatorName('list')).to.equal(',');
                expect(ooxEnGrammar.getOperatorName('isect')).to.equal(' ');
                expect(odfEnGrammar.getOperatorName('add')).to.equal('+');
                expect(odfEnGrammar.getOperatorName('le')).to.equal('<=');
                expect(odfEnGrammar.getOperatorName('list')).to.equal(',');
                expect(odfEnGrammar.getOperatorName('isect')).to.equal(' ');
            });
            it('should return null for invalid keys', function () {
                expect(ooxOpGrammar.getOperatorName('abc')).to.equal(null);
                expect(ooxOpGrammar.getOperatorName('ADD')).to.equal(null);
                expect(ooxOpGrammar.getOperatorName('+')).to.equal(null);
            });
        });

        describe('method "getOperatorKey"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('getOperatorKey');
                expect(ooxUiGrammar).to.respondTo('getOperatorKey');
            });
            it('should parse native operator names', function () {
                expect(ooxOpGrammar.getOperatorKey('+')).to.equal('add');
                expect(ooxOpGrammar.getOperatorKey('<=')).to.equal('le');
                expect(ooxOpGrammar.getOperatorKey(',')).to.equal('list');
                expect(ooxOpGrammar.getOperatorKey(' ')).to.equal('isect');
                expect(odfOpGrammar.getOperatorKey('+')).to.equal('add');
                expect(odfOpGrammar.getOperatorKey('<=')).to.equal('le');
                expect(odfOpGrammar.getOperatorKey('~')).to.equal('list');
                expect(odfOpGrammar.getOperatorKey('!')).to.equal('isect');
            });
            it('should parse localized operator names', function () {
                expect(ooxUiGrammar.getOperatorKey('+')).to.equal('add');
                expect(ooxUiGrammar.getOperatorKey('<=')).to.equal('le');
                expect(ooxUiGrammar.getOperatorKey(';')).to.equal('list');
                expect(ooxUiGrammar.getOperatorKey(' ')).to.equal('isect');
                expect(odfUiGrammar.getOperatorKey('+')).to.equal('add');
                expect(odfUiGrammar.getOperatorKey('<=')).to.equal('le');
                expect(odfUiGrammar.getOperatorKey(';')).to.equal('list');
                expect(odfUiGrammar.getOperatorKey(' ')).to.equal('isect');
            });
            it('should parse English operator names', function () {
                expect(ooxEnGrammar.getOperatorKey('+')).to.equal('add');
                expect(ooxEnGrammar.getOperatorKey('<=')).to.equal('le');
                expect(ooxEnGrammar.getOperatorKey(',')).to.equal('list');
                expect(ooxEnGrammar.getOperatorKey(' ')).to.equal('isect');
                expect(odfEnGrammar.getOperatorKey('+')).to.equal('add');
                expect(odfEnGrammar.getOperatorKey('<=')).to.equal('le');
                expect(odfEnGrammar.getOperatorKey(',')).to.equal('list');
                expect(odfEnGrammar.getOperatorKey(' ')).to.equal('isect');
            });
            it('should return null for invalid names', function () {
                expect(ooxOpGrammar.getOperatorKey('abc')).to.equal(null);
                expect(ooxOpGrammar.getOperatorKey('add')).to.equal(null);
                expect(ooxOpGrammar.getOperatorKey(';')).to.equal(null);
                expect(ooxUiGrammar.getOperatorKey('abc')).to.equal(null);
                expect(ooxUiGrammar.getOperatorKey('add')).to.equal(null);
                expect(ooxUiGrammar.getOperatorKey(',')).to.equal(null);
                expect(ooxEnGrammar.getOperatorKey('abc')).to.equal(null);
                expect(ooxEnGrammar.getOperatorKey('add')).to.equal(null);
                expect(ooxEnGrammar.getOperatorKey(';')).to.equal(null);
            });
        });

        describe('method "getFunctionName"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('getFunctionName');
                expect(ooxUiGrammar).to.respondTo('getFunctionName');
            });
            it('should return native function names', function () {
                expect(ooxOpGrammar.getFunctionName('ABS')).to.equal('ABS');
                expect(ooxOpGrammar.getFunctionName('SUM')).to.equal('SUM');
                expect(ooxOpGrammar.getFunctionName('AGGREGATE')).to.equal('_xlfn.AGGREGATE');
                expect(odfOpGrammar.getFunctionName('ABS')).to.equal('ABS');
                expect(odfOpGrammar.getFunctionName('SUM')).to.equal('SUM');
                expect(odfOpGrammar.getFunctionName('AGGREGATE')).to.equal('COM.MICROSOFT.AGGREGATE');
            });
            it('should return localized function names', function () {
                expect(ooxUiGrammar.getFunctionName('ABS')).to.equal('ABS');
                expect(ooxUiGrammar.getFunctionName('SUM')).to.equal('SUMME');
                expect(ooxUiGrammar.getFunctionName('AGGREGATE')).to.equal('AGGREGAT');
                expect(odfUiGrammar.getFunctionName('ABS')).to.equal('ABS');
                expect(odfUiGrammar.getFunctionName('SUM')).to.equal('SUMME');
                expect(odfUiGrammar.getFunctionName('AGGREGATE')).to.equal('AGGREGAT');
            });
            it('should return English function names', function () {
                expect(ooxEnGrammar.getFunctionName('ABS')).to.equal('ABS');
                expect(ooxEnGrammar.getFunctionName('SUM')).to.equal('SUM');
                expect(ooxEnGrammar.getFunctionName('AGGREGATE')).to.equal('AGGREGATE');
                expect(odfEnGrammar.getFunctionName('ABS')).to.equal('ABS');
                expect(odfEnGrammar.getFunctionName('SUM')).to.equal('SUM');
                expect(odfEnGrammar.getFunctionName('AGGREGATE')).to.equal('AGGREGATE');
            });
            it('should return null for invalid keys', function () {
                expect(ooxOpGrammar.getFunctionName('123')).to.equal(null);
                expect(ooxOpGrammar.getFunctionName('abc')).to.equal(null);
                expect(ooxOpGrammar.getFunctionName('abs')).to.equal(null);
                expect(ooxOpGrammar.getFunctionName('ISLEAPYEAR')).to.equal(null);
                expect(ooxUiGrammar.getFunctionName('123')).to.equal(null);
                expect(ooxUiGrammar.getFunctionName('abc')).to.equal(null);
                expect(ooxUiGrammar.getFunctionName('abs')).to.equal(null);
                expect(ooxUiGrammar.getFunctionName('ISLEAPYEAR')).to.equal(null);
                expect(ooxEnGrammar.getFunctionName('123')).to.equal(null);
                expect(ooxEnGrammar.getFunctionName('abc')).to.equal(null);
                expect(ooxEnGrammar.getFunctionName('abs')).to.equal(null);
                expect(ooxEnGrammar.getFunctionName('ISLEAPYEAR')).to.equal(null);
            });
        });

        describe('method "getFunctionKey"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('getFunctionKey');
                expect(ooxUiGrammar).to.respondTo('getFunctionKey');
            });
            it('should parse native function names', function () {
                expect(ooxOpGrammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(ooxOpGrammar.getFunctionKey('abs')).to.equal('ABS');
                expect(ooxOpGrammar.getFunctionKey('SUM')).to.equal('SUM');
                expect(ooxOpGrammar.getFunctionKey('sum')).to.equal('SUM');
                expect(ooxOpGrammar.getFunctionKey('_xlfn.AGGREGATE')).to.equal('AGGREGATE');
                expect(odfOpGrammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(odfOpGrammar.getFunctionKey('abs')).to.equal('ABS');
                expect(odfOpGrammar.getFunctionKey('SUM')).to.equal('SUM');
                expect(odfOpGrammar.getFunctionKey('sum')).to.equal('SUM');
                expect(odfOpGrammar.getFunctionKey('COM.MICROSOFT.AGGREGATE')).to.equal('AGGREGATE');
            });
            it('should parse localized function names', function () {
                expect(ooxUiGrammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(ooxUiGrammar.getFunctionKey('abs')).to.equal('ABS');
                expect(ooxUiGrammar.getFunctionKey('SUMME')).to.equal('SUM');
                expect(ooxUiGrammar.getFunctionKey('summe')).to.equal('SUM');
                expect(ooxUiGrammar.getFunctionKey('AGGREGAT')).to.equal('AGGREGATE');
                expect(odfUiGrammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(odfUiGrammar.getFunctionKey('abs')).to.equal('ABS');
                expect(odfUiGrammar.getFunctionKey('SUMME')).to.equal('SUM');
                expect(odfUiGrammar.getFunctionKey('summe')).to.equal('SUM');
                expect(odfUiGrammar.getFunctionKey('AGGREGAT')).to.equal('AGGREGATE');
            });
            it('should parse English function names', function () {
                expect(ooxEnGrammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(ooxEnGrammar.getFunctionKey('abs')).to.equal('ABS');
                expect(ooxEnGrammar.getFunctionKey('SUM')).to.equal('SUM');
                expect(ooxEnGrammar.getFunctionKey('sum')).to.equal('SUM');
                expect(ooxEnGrammar.getFunctionKey('AGGREGATE')).to.equal('AGGREGATE');
                expect(odfEnGrammar.getFunctionKey('ABS')).to.equal('ABS');
                expect(odfEnGrammar.getFunctionKey('abs')).to.equal('ABS');
                expect(odfEnGrammar.getFunctionKey('SUM')).to.equal('SUM');
                expect(odfEnGrammar.getFunctionKey('sum')).to.equal('SUM');
                expect(odfEnGrammar.getFunctionKey('AGGREGATE')).to.equal('AGGREGATE');
            });
            it('should return null for invalid names', function () {
                expect(ooxOpGrammar.getFunctionKey('123')).to.equal(null);
                expect(ooxOpGrammar.getFunctionKey('abc')).to.equal(null);
                expect(ooxOpGrammar.getFunctionKey('SUMME')).to.equal(null);
                expect(ooxOpGrammar.getFunctionKey('ISLEAPYEAR')).to.equal(null);
                expect(ooxOpGrammar.getFunctionKey('FORMULATEXT')).to.equal(null);
                expect(ooxOpGrammar.getFunctionKey('FORMELTEXT')).to.equal(null);
                expect(ooxUiGrammar.getFunctionKey('123')).to.equal(null);
                expect(ooxUiGrammar.getFunctionKey('abc')).to.equal(null);
                expect(ooxUiGrammar.getFunctionKey('SUM')).to.equal(null);
                expect(ooxUiGrammar.getFunctionKey('ISTSCHALTJAHR')).to.equal(null);
            });
        });

        describe('method "getTableRegionName"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('getTableRegionName');
                expect(ooxUiGrammar).to.respondTo('getTableRegionName');
            });
            it('should return native region names', function () {
                expect(ooxOpGrammar.getTableRegionName('ALL')).to.equal('#All');
                expect(ooxOpGrammar.getTableRegionName('HEADERS')).to.equal('#Headers');
                expect(ooxOpGrammar.getTableRegionName('DATA')).to.equal('#Data');
                expect(ooxOpGrammar.getTableRegionName('TOTALS')).to.equal('#Totals');
                expect(ooxOpGrammar.getTableRegionName('ROW')).to.equal('#This Row');
                expect(odfOpGrammar.getTableRegionName('ALL')).to.equal('#All');
                expect(odfOpGrammar.getTableRegionName('HEADERS')).to.equal('#Headers');
                expect(odfOpGrammar.getTableRegionName('DATA')).to.equal('#Data');
                expect(odfOpGrammar.getTableRegionName('TOTALS')).to.equal('#Totals');
                expect(odfOpGrammar.getTableRegionName('ROW')).to.equal('#This Row');
            });
            it('should return localized region names', function () {
                expect(ooxUiGrammar.getTableRegionName('ALL')).to.equal('#Alle');
                expect(ooxUiGrammar.getTableRegionName('HEADERS')).to.equal('#Kopfzeilen');
                expect(ooxUiGrammar.getTableRegionName('DATA')).to.equal('#Daten');
                expect(ooxUiGrammar.getTableRegionName('TOTALS')).to.equal('#Ergebnisse');
                expect(ooxUiGrammar.getTableRegionName('ROW')).to.equal('#Diese Zeile');
                expect(odfUiGrammar.getTableRegionName('ALL')).to.equal('#Alle');
                expect(odfUiGrammar.getTableRegionName('HEADERS')).to.equal('#Kopfzeilen');
                expect(odfUiGrammar.getTableRegionName('DATA')).to.equal('#Daten');
                expect(odfUiGrammar.getTableRegionName('TOTALS')).to.equal('#Ergebnisse');
                expect(odfUiGrammar.getTableRegionName('ROW')).to.equal('#Diese Zeile');
            });
            it('should return English region names', function () {
                expect(ooxEnGrammar.getTableRegionName('ALL')).to.equal('#All');
                expect(ooxEnGrammar.getTableRegionName('HEADERS')).to.equal('#Headers');
                expect(ooxEnGrammar.getTableRegionName('DATA')).to.equal('#Data');
                expect(ooxEnGrammar.getTableRegionName('TOTALS')).to.equal('#Totals');
                expect(ooxEnGrammar.getTableRegionName('ROW')).to.equal('#This Row');
                expect(odfEnGrammar.getTableRegionName('ALL')).to.equal('#All');
                expect(odfEnGrammar.getTableRegionName('HEADERS')).to.equal('#Headers');
                expect(odfEnGrammar.getTableRegionName('DATA')).to.equal('#Data');
                expect(odfEnGrammar.getTableRegionName('TOTALS')).to.equal('#Totals');
                expect(odfEnGrammar.getTableRegionName('ROW')).to.equal('#This Row');
            });
            it('should return null for invalid keys', function () {
                expect(ooxOpGrammar.getTableRegionName('all')).to.equal(null);
                expect(ooxOpGrammar.getTableRegionName('#All')).to.equal(null);
                expect(ooxOpGrammar.getTableRegionName(' ')).to.equal(null);
            });
        });

        describe('method "getTableRegionKey"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('getTableRegionKey');
                expect(ooxUiGrammar).to.respondTo('getTableRegionKey');
            });
            it('should parse native region names', function () {
                expect(ooxOpGrammar.getTableRegionKey('#all')).to.equal('ALL');
                expect(ooxOpGrammar.getTableRegionKey('#All')).to.equal('ALL');
                expect(ooxOpGrammar.getTableRegionKey('#ALL')).to.equal('ALL');
                expect(ooxOpGrammar.getTableRegionKey('#Headers')).to.equal('HEADERS');
                expect(ooxOpGrammar.getTableRegionKey('#Data')).to.equal('DATA');
                expect(ooxOpGrammar.getTableRegionKey('#Totals')).to.equal('TOTALS');
                expect(ooxOpGrammar.getTableRegionKey('#This Row')).to.equal('ROW');
                expect(odfOpGrammar.getTableRegionKey('#All')).to.equal('ALL');
                expect(odfOpGrammar.getTableRegionKey('#Headers')).to.equal('HEADERS');
                expect(odfOpGrammar.getTableRegionKey('#Data')).to.equal('DATA');
                expect(odfOpGrammar.getTableRegionKey('#Totals')).to.equal('TOTALS');
                expect(odfOpGrammar.getTableRegionKey('#This Row')).to.equal('ROW');
            });
            it('should parse localized region names', function () {
                expect(ooxUiGrammar.getTableRegionKey('#alle')).to.equal('ALL');
                expect(ooxUiGrammar.getTableRegionKey('#Alle')).to.equal('ALL');
                expect(ooxUiGrammar.getTableRegionKey('#ALLE')).to.equal('ALL');
                expect(ooxUiGrammar.getTableRegionKey('#Kopfzeilen')).to.equal('HEADERS');
                expect(ooxUiGrammar.getTableRegionKey('#Daten')).to.equal('DATA');
                expect(ooxUiGrammar.getTableRegionKey('#Ergebnisse')).to.equal('TOTALS');
                expect(ooxUiGrammar.getTableRegionKey('#Diese Zeile')).to.equal('ROW');
                expect(odfUiGrammar.getTableRegionKey('#Alle')).to.equal('ALL');
                expect(odfUiGrammar.getTableRegionKey('#Kopfzeilen')).to.equal('HEADERS');
                expect(odfUiGrammar.getTableRegionKey('#Daten')).to.equal('DATA');
                expect(odfUiGrammar.getTableRegionKey('#Ergebnisse')).to.equal('TOTALS');
                expect(odfUiGrammar.getTableRegionKey('#Diese Zeile')).to.equal('ROW');
            });
            it('should parse English region names', function () {
                expect(ooxEnGrammar.getTableRegionKey('#all')).to.equal('ALL');
                expect(ooxEnGrammar.getTableRegionKey('#All')).to.equal('ALL');
                expect(ooxEnGrammar.getTableRegionKey('#ALL')).to.equal('ALL');
                expect(ooxEnGrammar.getTableRegionKey('#Headers')).to.equal('HEADERS');
                expect(ooxEnGrammar.getTableRegionKey('#Data')).to.equal('DATA');
                expect(ooxEnGrammar.getTableRegionKey('#Totals')).to.equal('TOTALS');
                expect(ooxEnGrammar.getTableRegionKey('#This Row')).to.equal('ROW');
                expect(odfEnGrammar.getTableRegionKey('#All')).to.equal('ALL');
                expect(odfEnGrammar.getTableRegionKey('#Headers')).to.equal('HEADERS');
                expect(odfEnGrammar.getTableRegionKey('#Data')).to.equal('DATA');
                expect(odfEnGrammar.getTableRegionKey('#Totals')).to.equal('TOTALS');
                expect(odfEnGrammar.getTableRegionKey('#This Row')).to.equal('ROW');
            });
            it('should return null for invalid names', function () {
                expect(ooxOpGrammar.getTableRegionKey('All')).to.equal(null);
                expect(ooxOpGrammar.getTableRegionKey('#Alle')).to.equal(null);
                expect(ooxOpGrammar.getTableRegionKey('#ThisRow')).to.equal(null);
                expect(ooxUiGrammar.getTableRegionKey('Alle')).to.equal(null);
                expect(ooxUiGrammar.getTableRegionKey('#All')).to.equal(null);
                expect(ooxUiGrammar.getTableRegionKey('#DieseZeile')).to.equal(null);
                expect(ooxEnGrammar.getTableRegionKey('All')).to.equal(null);
                expect(ooxEnGrammar.getTableRegionKey('#Alle')).to.equal(null);
                expect(ooxEnGrammar.getTableRegionKey('#ThisRow')).to.equal(null);
            });
        });

        describe('method "isSimpleTableColumn"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('isSimpleTableColumn');
                expect(ooxUiGrammar).to.respondTo('isSimpleTableColumn');
            });
            it('should return true for simple table columns', function () {
                expect(ooxOpGrammar.isSimpleTableColumn('Col1')).to.equal(true);
                expect(ooxUiGrammar.isSimpleTableColumn('Col1')).to.equal(true);
                expect(ooxOpGrammar.isSimpleTableColumn('Col\xA11')).to.equal(true);
                expect(ooxUiGrammar.isSimpleTableColumn('Col\xA11')).to.equal(true);
            });
            it('should return false for complex table columns', function () {
                expect(ooxOpGrammar.isSimpleTableColumn('Col 1')).to.equal(false);
                expect(ooxUiGrammar.isSimpleTableColumn('Col 1')).to.equal(false);
                expect(ooxOpGrammar.isSimpleTableColumn('Col!1')).to.equal(false);
                expect(ooxUiGrammar.isSimpleTableColumn('Col!1')).to.equal(false);
                expect(ooxOpGrammar.isSimpleTableColumn('Col_1')).to.equal(false);
                expect(ooxUiGrammar.isSimpleTableColumn('Col_1')).to.equal(false);
            });
            it('should return result for comma character according to grammar', function () {
                expect(ooxOpGrammar.isSimpleTableColumn('Col,1')).to.equal(false);
                expect(ooxUiGrammar.isSimpleTableColumn('Col,1')).to.equal(true);
                expect(ooxEnGrammar.isSimpleTableColumn('Col,1')).to.equal(false);
            });
        });

        describe('method "encodeTableColumn"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('encodeTableColumn');
                expect(ooxUiGrammar).to.respondTo('encodeTableColumn');
            });
            it('should return encoded table columns', function () {
                expect(ooxOpGrammar.encodeTableColumn('Col1')).to.equal('Col1');
                expect(ooxUiGrammar.encodeTableColumn('Col1')).to.equal('Col1');
                expect(ooxOpGrammar.encodeTableColumn(' !:*+{')).to.equal(' !:*+{');
                expect(ooxUiGrammar.encodeTableColumn(' !:*+{')).to.equal(' !:*+{');
                expect(ooxOpGrammar.encodeTableColumn('[]\'#@')).to.equal('\'[\']\'\'\'#@');
                expect(ooxUiGrammar.encodeTableColumn('[]\'#@')).to.equal('\'[\']\'\'\'#\'@');
            });
        });

        describe('method "decodeTableColumn"', function () {
            it('should exist', function () {
                expect(ooxOpGrammar).to.respondTo('decodeTableColumn');
                expect(ooxUiGrammar).to.respondTo('decodeTableColumn');
            });
            it('should return encoded table columns', function () {
                expect(ooxOpGrammar.decodeTableColumn('Col\'\'\'#\'1')).to.equal('Col\'#1');
                expect(ooxUiGrammar.decodeTableColumn('Col\'\'\'#\'1')).to.equal('Col\'#1');
            });
        });
    });

    // ========================================================================
});
