/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/funcs/matrixfuncs'
], function (AppHelper, SheetHelper, FormulaUtils, MatrixFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var m = SheetHelper.mat;

    function matrixMatcher(expected) {
        return function (result) {
            return _.isEqual(expected, result._array);
        };
    }

    // module MatrixFuncs =====================================================

    describe('Spreadsheet module MatrixFuncs', function () {

        // initialize the test
        var appPromise = AppHelper.createSpreadsheetApp('ooxml');
        var moduleTester = SheetHelper.createFunctionModuleTester(MatrixFuncs, appPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('MDETERM', function (MDETERM) {
            it('should return the matrix determinant', function () {
                expect(MDETERM(m([[1]]))).to.equal(1);
                expect(MDETERM(m([[3, 6], [1, 1]]))).to.equal(-3);
                expect(MDETERM(m([[0, 1, 2], [3, 4, 5], [6, 7, 8]]))).to.equal(0);
                expect(MDETERM(m([[3, 6, 1], [1, 1, 0], [3, 10, 2]]))).to.equal(1);
                expect(MDETERM(m([[-4, -2.5, 3], [5, 6, 4], [9, 10, -9]]))).to.equal(161.5);
                expect(MDETERM(m([[1, 3, 8, 5], [1, 3, 6, 1], [1, 1, 1, 0], [7, 3, 10, 2]]))).to.equal(88);
                expect(MDETERM(m([[1, 3, 8, 5, 1], [1, 3, 6, 1, 1], [1, 1, 1, 0, 1], [7, 3, 10, 2, 1], [1, 1, 1, 1, 1]]))).to.equal(-24);
                expect(MDETERM(m([[1, 0, 0], [0, 1, 0], [0, 0, 0]]))).to.equal(0);
            });
            it('should throw #VALUE! error code for non-square matrix', function () {
                expect(MDETERM(m([[1, 1], [1, 1], [1, 1]]))).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('MINVERSE', function (MINVERSE) {
            it('should return the inverse matrix', function () {
                expect(MINVERSE(m([[1, 0, 0], [0, 1, 0], [0, 0, 1]]))).to.satisfy(matrixMatcher([[1, 0, 0], [0, 1, 0], [0, 0, 1]]));
                expect(MINVERSE(m([[1, 2, 1], [3, 4, -1], [0, 2, 0]]))).to.satisfy(matrixMatcher([[0.25, 0.25, -0.75], [0, 0, 0.5], [0.75, -0.25, -0.25]]));
            });
            it('should throw #VALUE! error code for non-square matrix', function () {
                expect(MINVERSE(m([[1, 1], [1, 1], [1, 1]]))).to.equal(ErrorCode.VALUE);
            });
            it('should throw #NUM! error code for non-invertible matrix', function () {
                expect(MINVERSE(m([[1, 0, 0], [0, 1, 0], [0, 0, 0]]))).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('MMULT', function (MMULT) {
            it('should return the matrix product', function () {
                expect(MMULT(m([[2]]), m([[3]]))).to.satisfy(matrixMatcher([[6]]));
                expect(MMULT(m([[2]]), m([[3, 4]]))).to.satisfy(matrixMatcher([[6, 8]]));
                expect(MMULT(m([[2], [3]]), m([[4]]))).to.satisfy(matrixMatcher([[8], [12]]));
                expect(MMULT(m([[2, 3]]), m([[4], [5]]))).to.satisfy(matrixMatcher([[23]]));
                expect(MMULT(m([[2], [3]]), m([[4, 5]]))).to.satisfy(matrixMatcher([[8, 10], [12, 15]]));
                expect(MMULT(m([[2, 3], [4, 5]]), m([[6, 7], [8, 9]]))).to.satisfy(matrixMatcher([[36, 41], [64, 73]]));
            });
            it('should throw the #VALUE! error, if matrix sizes do not match', function () {
                expect(MMULT(m([[2]]), m([[3], [4]]))).to.equal(ErrorCode.VALUE);
                expect(MMULT(m([[2, 3]]), m([[4, 5]]))).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('MUNIT', function (MUNIT) {
            it('should return the unit matrix', function () {
                expect(MUNIT(1)).to.satisfy(matrixMatcher([[1]]));
                expect(MUNIT(2)).to.satisfy(matrixMatcher([[1, 0], [0, 1]]));
                expect(MUNIT(3)).to.satisfy(matrixMatcher([[1, 0, 0], [0, 1, 0], [0, 0, 1]]));
                expect(MUNIT(4)).to.satisfy(matrixMatcher([[1, 0, 0, 0], [0, 1, 0, 0], [0, 0, 1, 0], [0, 0, 0, 1]]));
            });
            it('should throw the #VALUE! error, if passed dimension is too small', function () {
                expect(MUNIT(0)).to.equal(ErrorCode.VALUE);
                expect(MUNIT(-1)).to.equal(ErrorCode.VALUE);
            });
            it('should throw the UNSUPPORTED error, if passed dimension is too large', function () {
                var maxSize = Math.max(FormulaUtils.MAX_MATRIX_ROW_COUNT, FormulaUtils.MAX_MATRIX_COL_COUNT);
                expect(MUNIT(maxSize + 1)).to.equal(FormulaUtils.UNSUPPORTED_ERROR);
            });
        });

        moduleTester.testFunction('SUMPRODUCT', function (SUMPRODUCT) {
            it('should return the sum product of its arguments', function () {
                expect(SUMPRODUCT(1, 2, m([[3]]))).to.equal(6);
                expect(SUMPRODUCT(m([[1, 2], [3, 4], [5, 6]]))).to.equal(21);
                expect(SUMPRODUCT(m([[1, 2]]), m([[3, 4]]), m([[5, 6]]))).to.equal(63);
            });
            it('should skip invalid values in matrixes', function () {
                expect(SUMPRODUCT(m([[1, 2]]), m([['a', 4]]), m([[5, 6]]))).to.equal(48);
            });
            it('should throw #VALUE! error code for invalid scalar values', function () {
                expect(SUMPRODUCT(1, 2, 'abc')).to.equal(ErrorCode.VALUE);
                expect(SUMPRODUCT(1, 2, '3')).to.equal(ErrorCode.VALUE);
                expect(SUMPRODUCT(1, 2, true)).to.equal(ErrorCode.VALUE);
                expect(SUMPRODUCT(1, 2, null)).to.equal(ErrorCode.VALUE);
            });
            it('should throw #VALUE! error code for non-matching matrix sizes', function () {
                expect(SUMPRODUCT(m([[3, 4, 1], [8, 6, 1], [1, 9, 1]]), m([[2, 7], [6, 7], [5, 3]]))).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('SUMX2MY2', function (SUMX2MY2) {
            it('should return the correct result', function () {
                expect(SUMX2MY2(1, m([[2]]))).to.equal(-3);
                expect(SUMX2MY2(m([[1, 2, 4]]), m([[2, 5, 1]]))).to.equal(-9);
                expect(SUMX2MY2(m([[1, 2, 4]]), m([[2, 'a', 1]]))).to.equal(12);
            });
            it('should ignore matrix orientation', function () {
                expect(SUMX2MY2(m([[1, 2, 4]]), m([[2], [5], [1]]))).to.equal(-9);
            });
            it('should return the #N/A error code for non-matching matrix sizes', function () {
                expect(SUMX2MY2(m([[1, 1], [1, 1]]), m([[1, 1]]))).to.equal(ErrorCode.NA);
            });
            it('should throw #VALUE! error code for invalid parameters', function () {
                expect(SUMX2MY2(1, true)).to.equal(ErrorCode.VALUE);
                expect(SUMX2MY2(1, 'abc')).to.equal(ErrorCode.VALUE);
                expect(SUMX2MY2(1, null)).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('SUMX2PY2', function (SUMX2PY2) {
            it('should return the correct result', function () {
                expect(SUMX2PY2(m([[1, 2, 4]]), m([[2, 5, 1]]))).to.equal(51);
                expect(SUMX2PY2(m([[1, 2, 4]]), m([[2, 'a', 1]]))).to.equal(22);
            });
            it('should ignore matrix orientation', function () {
                expect(SUMX2PY2(m([[1, 2, 4]]), m([[2], [5], [1]]))).to.equal(51);
            });
            it('should return the #N/A error code for non-matching matrix sizes', function () {
                expect(SUMX2PY2(m([[1, 1], [1, 1]]), m([[1, 1]]))).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('SUMXMY2', function (SUMXMY2) {
            it('should return the correct result', function () {
                expect(SUMXMY2(m([[1, 2, 4]]), m([[2, 5, 1]]))).to.equal(19);
                expect(SUMXMY2(m([[1, 2, 4]]), m([[2, 'a', 1]]))).to.equal(10);
            });
            it('should ignore matrix orientation', function () {
                expect(SUMXMY2(m([[1, 2, 4]]), m([[2], [5], [1]]))).to.equal(19);
            });
            it('should return the #N/A error code for non-matching matrix sizes', function () {
                expect(SUMXMY2(m([[1, 1], [1, 1]]), m([[1, 1]]))).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('TRANSPOSE', function (TRANSPOSE) {
            it('should return the transposed matrix', function () {
                expect(TRANSPOSE(m([[1]]))).to.satisfy(matrixMatcher([[1]]));
                expect(TRANSPOSE(m([[1, 2]]))).to.satisfy(matrixMatcher([[1], [2]]));
                expect(TRANSPOSE(m([[1], [2]]))).to.satisfy(matrixMatcher([[1, 2]]));
                expect(TRANSPOSE(m([[1, 2], [3, 4]]))).to.satisfy(matrixMatcher([[1, 3], [2, 4]]));
                expect(TRANSPOSE(m([[1, 2, 3], [4, 5, 6]]))).to.satisfy(matrixMatcher([[1, 4], [2, 5], [3, 6]]));
            });
        });
    });

    // ========================================================================
});
