/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/funcs/logicalfuncs'
], function (AppHelper, SheetHelper, LogicalFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var r3da = SheetHelper.r3da;
    var mat = SheetHelper.mat;

    // module LogicalFuncs ====================================================

    describe('Spreadsheet module LogicalFuncs', function () {

        // initialize the test
        var appPromise = AppHelper.createSpreadsheetApp('ooxml');
        var moduleTester = SheetHelper.createFunctionModuleTester(LogicalFuncs, appPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('AND', function (AND) {
            it('should return the logical AND of the arguments', function () {
                expect(AND(false)).to.equal(false);
                expect(AND(true)).to.equal(true);
                expect(AND(false, false)).to.equal(false);
                expect(AND(false, true)).to.equal(false);
                expect(AND(true, false)).to.equal(false);
                expect(AND(true, true)).to.equal(true);
                expect(AND(false, false, false)).to.equal(false);
                expect(AND(false, false, true)).to.equal(false);
                expect(AND(false, true, false)).to.equal(false);
                expect(AND(false, true, true)).to.equal(false);
                expect(AND(true, false, false)).to.equal(false);
                expect(AND(true, false, true)).to.equal(false);
                expect(AND(true, true, false)).to.equal(false);
                expect(AND(true, true, true)).to.equal(true);
            });
        });

        moduleTester.testFunction('FALSE', function (FALSE) {
            it('should return FALSE', function () {
                expect(FALSE()).to.equal(false);
            });
        });

        moduleTester.testFunction('IF', { toOperand: [1, 2] }, function (IF) {
            var mat1 = mat([[1]]);
            var ranges = r3da('0:0!A1:A1');
            it('should return first parameter, if condition is TRUE', function () {
                expect(IF(true, 2, 'x')).to.equal(2);
                expect(IF(true, 'a', 'x')).to.equal('a');
                expect(IF(true, false, 'x')).to.equal(false);
                expect(IF(true, ErrorCode.DIV0, 'x')).to.equal(ErrorCode.DIV0);
                expect(IF(true, null, 'x')).to.equal(null);
                expect(IF(true, mat1, 'x')).to.deep.equal(mat1);
                expect(IF(true, ranges, 'x')).to.deep.equal(ranges);
            });
            it('should return second parameter, if condition is FALSE', function () {
                expect(IF(false, 'x', 2)).to.equal(2);
                expect(IF(false, 'x', 'a')).to.equal('a');
                expect(IF(false, 'x', true)).to.equal(true);
                expect(IF(false, 'x', ErrorCode.DIV0)).to.equal(ErrorCode.DIV0);
                expect(IF(false, 'x', null)).to.equal(null);
                expect(IF(false, 'x', mat1)).to.deep.equal(mat1);
                expect(IF(false, 'x', ranges)).to.equal(ranges);
            });
            it('should return FALSE, if second parameter is missing, and condition is FALSE', function () {
                expect(IF(true, 2)).to.equal(2);
                expect(IF(false, 2)).to.equal(false);
            });
            it('should ignore error codes in unused parameter', function () {
                expect(IF(true, 2, ErrorCode.DIV0)).to.equal(2);
                expect(IF(false, ErrorCode.DIV0, 2)).to.equal(2);
            });
        });

        moduleTester.testFunction('IFERROR', function (IFERROR) {
            it('should return the first parameter, if not an error code', function () {
                expect(IFERROR(2, 'x')).to.equal(2);
                expect(IFERROR('a', 'x')).to.equal('a');
                expect(IFERROR(false, 'x')).to.equal(false);
            });
            it('should return the second parameter, if first is an error code', function () {
                expect(IFERROR(ErrorCode.DIV0, 2)).to.equal(2);
                expect(IFERROR(ErrorCode.NA, 'a')).to.equal('a');
                expect(IFERROR(ErrorCode.NAME, false)).to.equal(false);
                expect(IFERROR(ErrorCode.NULL, 2)).to.equal(2);
                expect(IFERROR(ErrorCode.NUM, 'a')).to.equal('a');
                expect(IFERROR(ErrorCode.REF, false)).to.equal(false);
                expect(IFERROR(ErrorCode.VALUE, 2)).to.equal(2);
                expect(IFERROR(ErrorCode.DIV0, ErrorCode.NAME)).to.equal(ErrorCode.NAME);
            });
        });

        moduleTester.testFunction('IFNA', function (IFNA) {
            it('should return the first parameter, if not the #N/A error code', function () {
                expect(IFNA(2, 'x')).to.equal(2);
                expect(IFNA('a', 'x')).to.equal('a');
                expect(IFNA(false, 'x')).to.equal(false);
                expect(IFNA(ErrorCode.DIV0, 'x')).to.equal(ErrorCode.DIV0);
                expect(IFNA(ErrorCode.NAME, 'x')).to.equal(ErrorCode.NAME);
                expect(IFNA(ErrorCode.NULL, 'x')).to.equal(ErrorCode.NULL);
                expect(IFNA(ErrorCode.NUM, 'x')).to.equal(ErrorCode.NUM);
                expect(IFNA(ErrorCode.REF, 'x')).to.equal(ErrorCode.REF);
                expect(IFNA(ErrorCode.VALUE, 'x')).to.equal(ErrorCode.VALUE);
            });
            it('should return the second parameter, if first is the #N/A error code', function () {
                expect(IFNA(ErrorCode.NA, 2)).to.equal(2);
                expect(IFNA(ErrorCode.NA, 'a')).to.equal('a');
                expect(IFNA(ErrorCode.NA, false)).to.equal(false);
                expect(IFNA(ErrorCode.NA, ErrorCode.NAME)).to.equal(ErrorCode.NAME);
            });
        });

        moduleTester.testFunction('IFS', { toOperand: [1, 3, 5, 7, 9, 11, 13] }, function (IFS) {
            var mat1 = mat([[1]]);
            var ranges = r3da('0:0!A1:A1');
            it('should return parameter following the first TRUE condition', function () {
                expect(IFS(true, 2, true, 'a', true, false, true, ErrorCode.DIV0, true, null, true, mat1, true, ranges)).to.equal(2);
                expect(IFS(false, 2, true, 'a', true, false, true, ErrorCode.DIV0, true, null, true, mat1, true, ranges)).to.equal('a');
                expect(IFS(false, 2, false, 'a', true, false, true, ErrorCode.DIV0, true, null, true, mat1, true, ranges)).to.equal(false);
                expect(IFS(false, 2, false, 'a', false, false, true, ErrorCode.DIV0, true, null, true, mat1, true, ranges)).to.equal(ErrorCode.DIV0);
                expect(IFS(false, 2, false, 'a', false, false, false, ErrorCode.DIV0, true, null, true, mat1, true, ranges)).to.equal(null);
                expect(IFS(false, 2, false, 'a', false, false, false, ErrorCode.DIV0, false, null, true, mat1, true, ranges)).to.deep.equal(mat1);
                expect(IFS(false, 2, false, 'a', false, false, false, ErrorCode.DIV0, false, null, false, mat1, true, ranges)).to.equal(ranges);
            });
            it('should return #N/A if all conditions are FALSE', function () {
                expect(IFS(false, 2, false, 'a', false, false, false, ErrorCode.DIV0, false, null, false, mat1, false, ranges)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('NOT', function (NOT) {
            it('should return the negated argument', function () {
                expect(NOT(false)).to.equal(true);
                expect(NOT(true)).to.equal(false);
            });
        });

        moduleTester.testFunction('OR', function (OR) {
            it('should return the logical OR of the arguments', function () {
                expect(OR(false)).to.equal(false);
                expect(OR(true)).to.equal(true);
                expect(OR(false, false)).to.equal(false);
                expect(OR(false, true)).to.equal(true);
                expect(OR(true, false)).to.equal(true);
                expect(OR(true, true)).to.equal(true);
                expect(OR(false, false, false)).to.equal(false);
                expect(OR(false, false, true)).to.equal(true);
                expect(OR(false, true, false)).to.equal(true);
                expect(OR(false, true, true)).to.equal(true);
                expect(OR(true, false, false)).to.equal(true);
                expect(OR(true, false, true)).to.equal(true);
                expect(OR(true, true, false)).to.equal(true);
                expect(OR(true, true, true)).to.equal(true);
            });
        });

        moduleTester.testFunction('TRUE', function (TRUE) {
            it('should return TRUE', function () {
                expect(TRUE()).to.equal(true);
            });
        });

        moduleTester.testFunction('XOR', function (XOR) {
            it('should return the logical XOR of the arguments', function () {
                expect(XOR(false)).to.equal(false);
                expect(XOR(true)).to.equal(true);
                expect(XOR(false, false)).to.equal(false);
                expect(XOR(false, true)).to.equal(true);
                expect(XOR(true, false)).to.equal(true);
                expect(XOR(true, true)).to.equal(false);
                expect(XOR(false, false, false)).to.equal(false);
                expect(XOR(false, false, true)).to.equal(true);
                expect(XOR(false, true, false)).to.equal(true);
                expect(XOR(false, true, true)).to.equal(false);
                expect(XOR(true, false, false)).to.equal(true);
                expect(XOR(true, false, true)).to.equal(false);
                expect(XOR(true, true, false)).to.equal(false);
                expect(XOR(true, true, true)).to.equal(true);
            });
        });
    });

    // ========================================================================
});
