/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, Parser, ErrorCode, FormulaUtils) {

    'use strict';

    // static class FormulaUtils ==============================================

    describe('Spreadsheet module FormulaUtils', function () {

        it('should exist', function () {
            expect(FormulaUtils).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var sqrt = Math.sqrt;
        var exp = Math.exp;
        var log = Math.log;

        // constants ----------------------------------------------------------

        describe('constant "MAX_PARAM_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_PARAM_COUNT').that.is.a('number').and.above(0);
            });
        });

        describe('constant "MAX_MATRIX_ROW_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_MATRIX_ROW_COUNT').that.is.a('number').and.above(0);
            });
        });

        describe('constant "MAX_MATRIX_COL_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_MATRIX_COL_COUNT').that.is.a('number').and.above(0);
            });
        });

        describe('constant "MAX_REF_LIST_SIZE"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_REF_LIST_SIZE').that.is.a('number').and.above(0);
            });
        });

        describe('constant "MAX_EVAL_TIME"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_EVAL_TIME').that.is.a('number').and.above(0);
            });
        });

        describe('constant "ScalarType"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('ScalarType').that.is.an('object');
                expect(FormulaUtils.ScalarType).to.have.a.property('NULL', 0);
                expect(FormulaUtils.ScalarType).to.have.a.property('NUMBER', 1);
                expect(FormulaUtils.ScalarType).to.have.a.property('STRING', 2);
                expect(FormulaUtils.ScalarType).to.have.a.property('BOOLEAN', 3);
                expect(FormulaUtils.ScalarType).to.have.a.property('ERROR', 4);
                expect(FormulaUtils.ScalarType).to.have.a.property('NULL_MAX', 5);
            });
        });

        describe('constant "UNSUPPORTED_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('UNSUPPORTED_ERROR').that.is.an('object');
                expect(FormulaUtils.UNSUPPORTED_ERROR).to.have.a.property('key', 'unsupported');
            });
        });

        describe('constant "CIRCULAR_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('CIRCULAR_ERROR').that.is.an('object');
                expect(FormulaUtils.CIRCULAR_ERROR).to.have.a.property('key', 'circular');
            });
        });

        describe('constant "INTERNAL_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('INTERNAL_ERROR').that.is.an('object');
                expect(FormulaUtils.INTERNAL_ERROR).to.have.a.property('key', 'internal');
            });
        });

        // logger -------------------------------------------------------------

        describe('should support the logger interface"', function () {
            expect(FormulaUtils).itself.to.respondTo('log');
            expect(FormulaUtils).itself.to.respondTo('info');
            expect(FormulaUtils).itself.to.respondTo('warn');
            expect(FormulaUtils).itself.to.respondTo('error');
        });

        // static methods -----------------------------------------------------

        describe('method "throwInternal"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('throwInternal');
            });
            it('should always throw', function () {
                expect(FormulaUtils.throwInternal).to.throw(ErrorCode).that.equals(FormulaUtils.INTERNAL_ERROR);
            });
        });

        describe('method "throwErrorCode"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('throwErrorCode');
            });
            it('should return scalar values', function () {
                expect(FormulaUtils.throwErrorCode(42)).to.equal(42);
                expect(FormulaUtils.throwErrorCode('abc')).to.equal('abc');
                expect(FormulaUtils.throwErrorCode(true)).to.equal(true);
                expect(FormulaUtils.throwErrorCode(new Date())).to.be.a('date');
                expect(FormulaUtils.throwErrorCode(null)).to.equal(null);
            });
            it('should throw error codes', function () {
                expect(FormulaUtils.throwErrorCode.bind(null, ErrorCode.NULL)).to.throw(ErrorCode).that.equals(ErrorCode.NULL);
                expect(FormulaUtils.throwErrorCode.bind(null, ErrorCode.VALUE)).to.throw(ErrorCode).that.equals(ErrorCode.VALUE);
            });
        });

        describe('method "trunc"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('trunc');
            });
            it('should round down positive numbers', function () {
                expect(FormulaUtils.trunc(1.1)).to.equal(1);
                expect(FormulaUtils.trunc(5.5)).to.equal(5);
                expect(FormulaUtils.trunc(9.9)).to.equal(9);
            });
            it('should round up negative numbers', function () {
                expect(FormulaUtils.trunc(-1.1)).to.equal(-1);
                expect(FormulaUtils.trunc(-5.5)).to.equal(-5);
                expect(FormulaUtils.trunc(-9.9)).to.equal(-9);
            });
            it('should return integers unmodified', function () {
                expect(FormulaUtils.trunc(-1)).to.equal(-1);
                expect(FormulaUtils.trunc(0)).to.equal(0);
                expect(FormulaUtils.trunc(1)).to.equal(1);
            });
        });

        describe('method "add"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('add');
            });
            it('should return the sum of two numbers', function () {
                expect(FormulaUtils.add(3, 2)).to.equal(5);
                expect(FormulaUtils.add(0, 2)).to.equal(2);
            });
            it('should return the concatenation of two strings', function () {
                expect(FormulaUtils.add('a', 'b')).to.equal('ab');
            });
        });

        describe('method "multiply"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('multiply');
            });
            it('should return the product of two numbers', function () {
                expect(FormulaUtils.multiply(3, 2)).to.equal(6);
            });
        });

        describe('method "divide"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('divide');
            });
            it('should return the quotient of two numbers', function () {
                expect(FormulaUtils.divide(3, 2)).to.equal(1.5);
                expect(FormulaUtils.divide(0, 2)).to.equal(0);
            });
            it('should throw the #DIV/0! error, if divisor is 0', function () {
                expect(FormulaUtils.divide.bind(null, 3, 0)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
                expect(FormulaUtils.divide.bind(null, 0, 0)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "modulo"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('modulo');
            });
            it('should return the remainder, if divisor is positive', function () {
                expect(FormulaUtils.modulo(-5, 3)).to.equal(1);
                expect(FormulaUtils.modulo(-4, 3)).to.equal(2);
                expect(FormulaUtils.modulo(-3, 3)).to.equal(0);
                expect(FormulaUtils.modulo(-2, 3)).to.equal(1);
                expect(FormulaUtils.modulo(-1, 3)).to.equal(2);
                expect(FormulaUtils.modulo(0, 3)).to.equal(0);
                expect(FormulaUtils.modulo(1, 3)).to.equal(1);
                expect(FormulaUtils.modulo(2, 3)).to.equal(2);
                expect(FormulaUtils.modulo(3, 3)).to.equal(0);
                expect(FormulaUtils.modulo(4, 3)).to.equal(1);
                expect(FormulaUtils.modulo(5, 3)).to.equal(2);
            });
            it('should return the remainder, if divisor is negative', function () {
                expect(FormulaUtils.modulo(-5, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(-4, -3)).to.equal(-1);
                expect(FormulaUtils.modulo(-3, -3)).to.equal(0);
                expect(FormulaUtils.modulo(-2, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(-1, -3)).to.equal(-1);
                expect(FormulaUtils.modulo(0, -3)).to.equal(0);
                expect(FormulaUtils.modulo(1, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(2, -3)).to.equal(-1);
                expect(FormulaUtils.modulo(3, -3)).to.equal(0);
                expect(FormulaUtils.modulo(4, -3)).to.equal(-2);
                expect(FormulaUtils.modulo(5, -3)).to.equal(-1);
            });
            it('should throw the #DIV/0! error, if divisor is 0', function () {
                expect(FormulaUtils.modulo.bind(null, 3, 0)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
                expect(FormulaUtils.modulo.bind(null, 0, 0)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "powerOOXML"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('powerOOXML');
            });
            it('should return the power of two numbers', function () {
                expect(FormulaUtils.powerOOXML(3, 0)).to.equal(1);
                expect(FormulaUtils.powerOOXML(3, 1)).to.equal(3);
                expect(FormulaUtils.powerOOXML(3, 2)).to.equal(9);
                expect(FormulaUtils.powerOOXML(3, 3)).to.equal(27);
                expect(FormulaUtils.powerOOXML(9, 0.5)).to.almostEqual(3);
                expect(FormulaUtils.powerOOXML(81, 0.25)).to.almostEqual(3);
                expect(FormulaUtils.powerOOXML(4, -1)).to.almostEqual(0.25);
                expect(FormulaUtils.powerOOXML(2, -2)).to.almostEqual(0.25);
                expect(FormulaUtils.powerOOXML(0, 1)).to.equal(0);
                expect(FormulaUtils.powerOOXML(0, 2)).to.equal(0);
                expect(FormulaUtils.powerOOXML(0, 0.5)).to.equal(0);
            });
            it('should throw the #NUM! error, if both numbers are 0', function () {
                expect(FormulaUtils.powerOOXML.bind(null, 0, 0)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
            });
            it('should throw the #DIV/0! error, if base is 0, and exponent is negative', function () {
                expect(FormulaUtils.powerOOXML.bind(null, 0, -1)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "powerODF"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('powerODF');
            });
            it('should return the power of two numbers', function () {
                expect(FormulaUtils.powerODF(3, 0)).to.equal(1);
                expect(FormulaUtils.powerODF(3, 1)).to.equal(3);
                expect(FormulaUtils.powerODF(3, 2)).to.equal(9);
                expect(FormulaUtils.powerODF(3, 3)).to.equal(27);
                expect(FormulaUtils.powerODF(9, 0.5)).to.almostEqual(3);
                expect(FormulaUtils.powerODF(81, 0.25)).to.almostEqual(3);
                expect(FormulaUtils.powerODF(4, -1)).to.almostEqual(0.25);
                expect(FormulaUtils.powerODF(2, -2)).to.almostEqual(0.25);
                expect(FormulaUtils.powerODF(0, 1)).to.equal(0);
                expect(FormulaUtils.powerODF(0, 2)).to.equal(0);
                expect(FormulaUtils.powerODF(0, 0.5)).to.equal(0);
            });
            it('should return 1, if both numbers are 0', function () {
                expect(FormulaUtils.powerODF(0, 0)).to.equal(1);
            });
            it('should throw the #NUM! error, if base is 0, and exponent is negative', function () {
                expect(FormulaUtils.powerODF.bind(null, 0, -1)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('property "POWER"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('POWER').that.is.an('object');
            });
            it('should contain resolvers for file formats', function () {
                expect(FormulaUtils.POWER).to.have.a.property('ooxml', FormulaUtils.powerOOXML);
                expect(FormulaUtils.POWER).to.have.a.property('odf', FormulaUtils.powerODF);
            });
        });

        describe('method "pow10"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('pow10');
            });
            it('should return the decimal power', function () {
                expect(FormulaUtils.pow10(0)).to.equal(1);
                expect(FormulaUtils.pow10(1)).to.equal(10);
                expect(FormulaUtils.pow10(2)).to.equal(100);
                expect(FormulaUtils.pow10(3)).to.equal(1000);
                expect(FormulaUtils.pow10(4)).to.equal(10000);
                expect(FormulaUtils.pow10(-1)).to.equal(0.1);
                expect(FormulaUtils.pow10(-2)).to.equal(0.01);
                expect(FormulaUtils.pow10(-3)).to.equal(0.001);
                expect(FormulaUtils.pow10(-4)).to.equal(0.0001);
            });
        });

        describe('method "log10"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('log10');
            });
            it('should return the decimal logarithm', function () {
                expect(FormulaUtils.log10(1)).to.equal(0);
                expect(FormulaUtils.log10(10)).to.almostEqual(1);
                expect(FormulaUtils.log10(100)).to.almostEqual(2);
                expect(FormulaUtils.log10(1000)).to.almostEqual(3);
                expect(FormulaUtils.log10(10000)).to.almostEqual(4);
                expect(FormulaUtils.log10(0.1)).to.almostEqual(-1);
                expect(FormulaUtils.log10(0.01)).to.almostEqual(-2);
                expect(FormulaUtils.log10(0.001)).to.almostEqual(-3);
                expect(FormulaUtils.log10(0.0001)).to.almostEqual(-4);
            });
        });

        describe('method "arctan2"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('arctan2');
            });
            it('should return the arc tangent of two numbers', function () {
                expect(FormulaUtils.arctan2(1, 0)).to.be.almostZero;
                expect(FormulaUtils.arctan2(1, 1)).to.almostEqual(0.25 * Math.PI);
                expect(FormulaUtils.arctan2(0, 1)).to.almostEqual(0.5 * Math.PI);
                expect(FormulaUtils.arctan2(-1, 1)).to.almostEqual(0.75 * Math.PI);
                expect(FormulaUtils.arctan2(-1, 0)).to.almostEqual(Math.PI);
                expect(FormulaUtils.arctan2(-1, -1)).to.almostEqual(-0.75 * Math.PI);
                expect(FormulaUtils.arctan2(0, -1)).to.almostEqual(-0.5 * Math.PI);
                expect(FormulaUtils.arctan2(1, -1)).to.almostEqual(-0.25 * Math.PI);
            });
            it('should throw the #DIV/0! error, if both numbers are 0', function () {
                expect(FormulaUtils.arctan2.bind(null, 0, 0)).to.throw(ErrorCode).that.equals(ErrorCode.DIV0);
            });
        });

        describe('method "sinh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('sinh');
            });
            it('should return the hyperbolic sine', function () {
                expect(FormulaUtils.sinh(-2)).to.almostEqual((exp(-2) - exp(2)) / 2);
                expect(FormulaUtils.sinh(-1)).to.almostEqual((exp(-1) - exp(1)) / 2);
                expect(FormulaUtils.sinh(0)).to.equal(0);
                expect(FormulaUtils.sinh(1)).to.almostEqual((exp(1) - exp(-1)) / 2);
                expect(FormulaUtils.sinh(2)).to.almostEqual((exp(2) - exp(-2)) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.sinh) { expect(FormulaUtils.sinh).to.equal(Math.sinh); }
            });
        });

        describe('method "cosh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('cosh');
            });
            it('should return the hyperbolic cosine', function () {
                expect(FormulaUtils.cosh(-2)).to.almostEqual((exp(-2) + exp(2)) / 2);
                expect(FormulaUtils.cosh(-1)).to.almostEqual((exp(-1) + exp(1)) / 2);
                expect(FormulaUtils.cosh(0)).to.equal(1);
                expect(FormulaUtils.cosh(1)).to.almostEqual((exp(1) + exp(-1)) / 2);
                expect(FormulaUtils.cosh(2)).to.almostEqual((exp(2) + exp(-2)) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.cosh) { expect(FormulaUtils.cosh).to.equal(Math.cosh); }
            });
        });

        describe('method "tanh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('tanh');
            });
            it('should return the hyperbolic tangent', function () {
                expect(FormulaUtils.tanh(-2)).to.almostEqual((exp(-4) - 1) / (exp(-4) + 1));
                expect(FormulaUtils.tanh(-1)).to.almostEqual((exp(-2) - 1) / (exp(-2) + 1));
                expect(FormulaUtils.tanh(0)).to.equal(0);
                expect(FormulaUtils.tanh(1)).to.almostEqual((exp(2) - 1) / (exp(2) + 1));
                expect(FormulaUtils.tanh(2)).to.almostEqual((exp(4) - 1) / (exp(4) + 1));
            });
            it('should use the native implementation', function () {
                if (Math.tanh) { expect(FormulaUtils.tanh).to.equal(Math.tanh); }
            });
        });

        describe('method "asinh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('asinh');
            });
            it('should return the argument resulting in the hyperbolic sine', function () {
                expect(FormulaUtils.asinh(-2)).to.almostEqual(log(sqrt(5) - 2));
                expect(FormulaUtils.asinh(-1)).to.almostEqual(log(sqrt(2) - 1));
                expect(FormulaUtils.asinh(0)).to.equal(0);
                expect(FormulaUtils.asinh(1)).to.almostEqual(log(sqrt(2) + 1));
                expect(FormulaUtils.asinh(2)).to.almostEqual(log(sqrt(5) + 2));
            });
            it('should use the native implementation', function () {
                if (Math.asinh) { expect(FormulaUtils.asinh).to.equal(Math.asinh); }
            });
        });

        describe('method "acosh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('acosh');
            });
            it('should return the argument resulting in the hyperbolic cosine', function () {
                expect(FormulaUtils.acosh(-1)).not.to.be.finite;
                expect(FormulaUtils.acosh(0)).not.to.be.finite;
                expect(FormulaUtils.acosh(1)).to.equal(0);
                expect(FormulaUtils.acosh(2)).to.almostEqual(log(2 + sqrt(3)));
                expect(FormulaUtils.acosh(3)).to.almostEqual(log(3 + sqrt(8)));
            });
            it('should use the native implementation', function () {
                if (Math.acosh) { expect(FormulaUtils.acosh).to.equal(Math.acosh); }
            });
        });

        describe('method "atanh"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('atanh');
            });
            it('should return the argument resulting in the hyperbolic tangent', function () {
                expect(FormulaUtils.atanh(-2)).not.to.be.finite;
                expect(FormulaUtils.atanh(-1)).not.to.be.finite;
                expect(FormulaUtils.atanh(-0.5)).to.almostEqual(log(1 / 3) / 2);
                expect(FormulaUtils.atanh(0)).to.equal(0);
                expect(FormulaUtils.atanh(0.5)).to.almostEqual(log(3) / 2);
                expect(FormulaUtils.atanh(1)).not.to.be.finite;
                expect(FormulaUtils.atanh(2)).not.to.be.finite;
            });
            it('should use the native implementation', function () {
                if (Math.atanh) { expect(FormulaUtils.atanh).to.equal(Math.atanh); }
            });
        });

        describe('method "acoth"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('acoth');
            });
            it('should return the argument resulting in the hyperbolic tangent', function () {
                expect(FormulaUtils.acoth(-3)).to.almostEqual(log(1 / 2) / 2);
                expect(FormulaUtils.acoth(-2)).to.almostEqual(log(1 / 3) / 2);
                expect(FormulaUtils.acoth(-1)).not.to.be.finite;
                expect(FormulaUtils.acoth(0)).not.to.be.finite;
                expect(FormulaUtils.acoth(1)).not.to.be.finite;
                expect(FormulaUtils.acoth(2)).to.almostEqual(log(3) / 2);
                expect(FormulaUtils.acoth(3)).to.almostEqual(log(2) / 2);
            });
            it('should use the native implementation', function () {
                if (Math.acoth) { expect(FormulaUtils.acoth).to.equal(Math.acoth); }
            });
        });

        describe('method "factorial"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('factorial');
            });
            it('should return the factorials of numbers', function () {
                expect(FormulaUtils.factorial(0)).to.equal(1);
                expect(FormulaUtils.factorial(1)).to.equal(1);
                expect(FormulaUtils.factorial(2)).to.equal(2);
                expect(FormulaUtils.factorial(3)).to.equal(6);
                expect(FormulaUtils.factorial(4)).to.equal(24);
                expect(FormulaUtils.factorial(5)).to.equal(120);
                expect(FormulaUtils.factorial(6)).to.equal(720);
                expect(FormulaUtils.factorial(7)).to.equal(5040);
            });
            it('should round the input parameter', function () {
                expect(FormulaUtils.factorial(4.5)).to.equal(24);
            });
            it('should throw #NUM! error code for invalid numbers', function () {
                expect(FormulaUtils.factorial.bind(FormulaUtils, -1)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
                expect(FormulaUtils.factorial.bind(FormulaUtils, 1000)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('method "binomial"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('binomial');
            });
            it('should return the binomial coefficient', function () {
                expect(FormulaUtils.binomial(0, 0)).to.equal(1);
                expect(FormulaUtils.binomial(1, 0)).to.equal(1);
                expect(FormulaUtils.binomial(2, 0)).to.equal(1);
                expect(FormulaUtils.binomial(3, 0)).to.equal(1);
                expect(FormulaUtils.binomial(6, 0)).to.equal(1);
                expect(FormulaUtils.binomial(6, 1)).to.equal(6);
                expect(FormulaUtils.binomial(6, 2)).to.equal(15);
                expect(FormulaUtils.binomial(6, 3)).to.equal(20);
                expect(FormulaUtils.binomial(6, 4)).to.equal(15);
                expect(FormulaUtils.binomial(6, 5)).to.equal(6);
                expect(FormulaUtils.binomial(6, 6)).to.equal(1);
                expect(FormulaUtils.binomial(20, 0)).to.equal(1);
                expect(FormulaUtils.binomial(20, 1)).to.equal(20);
                expect(FormulaUtils.binomial(20, 5)).to.equal(15504);
                expect(FormulaUtils.binomial(20, 10)).to.equal(184756);
                expect(FormulaUtils.binomial(20, 15)).to.equal(15504);
                expect(FormulaUtils.binomial(20, 19)).to.equal(20);
                expect(FormulaUtils.binomial(20, 20)).to.equal(1);
                expect(FormulaUtils.binomial(1028, 514)).to.be.above(7.1e307);
                expect(FormulaUtils.binomial(1083, 400)).to.be.above(1.4e308);
            });
            it('should throw #NUM! error code for invalid parameters', function () {
                expect(FormulaUtils.binomial.bind(null, -1, 0)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
                expect(FormulaUtils.binomial.bind(null, 0, -1)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
                expect(FormulaUtils.binomial.bind(null, 2, 3)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
                expect(FormulaUtils.binomial.bind(null, 1200, 600)).to.throw(ErrorCode).that.equals(ErrorCode.NUM);
            });
        });

        describe('method "erf"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('erf');
            });
            it('should return the result of the Gauss error function', function () {
                expect(FormulaUtils.erf(0.0)).to.equal(0);
                expect(FormulaUtils.erf(0.5)).to.be.closeTo(0.520499877813046, 1e-15);
                expect(FormulaUtils.erf(1.0)).to.be.closeTo(0.842700792949714, 1e-15);
                expect(FormulaUtils.erf(1.5)).to.be.closeTo(0.96610514647531, 1e-15);
                expect(FormulaUtils.erf(2.0)).to.be.closeTo(0.995322265019265, 1e-15);
                expect(FormulaUtils.erf(2.5)).to.be.closeTo(0.999593047982555, 1e-15);
                expect(FormulaUtils.erf(3.0)).to.be.closeTo(0.999977909503001, 1e-15);
                expect(FormulaUtils.erf(3.5)).to.be.closeTo(0.999999256901627, 1e-15);
                expect(FormulaUtils.erf(4.0)).to.be.closeTo(0.999999984582742, 1e-15);
                expect(FormulaUtils.erf(4.5)).to.be.closeTo(0.999999999803383, 1e-15);
                expect(FormulaUtils.erf(5.0)).to.be.closeTo(0.999999999998462, 1e-15);
                expect(FormulaUtils.erf(5.5)).to.be.closeTo(0.9999999999999927, 1e-15);
                expect(FormulaUtils.erf(6.0)).to.equal(1);
            });
            it('should be an odd function', function () {
                expect(FormulaUtils.erf(-1)).to.equal(-FormulaUtils.erf(1));
                expect(FormulaUtils.erf(-2)).to.equal(-FormulaUtils.erf(2));
                expect(FormulaUtils.erf(-3)).to.equal(-FormulaUtils.erf(3));
                expect(FormulaUtils.erf(-4)).to.equal(-FormulaUtils.erf(4));
                expect(FormulaUtils.erf(-5)).to.equal(-FormulaUtils.erf(5));
                expect(FormulaUtils.erf(-6)).to.equal(-FormulaUtils.erf(6));
            });
        });

        describe('method "erfc"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('erfc');
            });
            it('should return the result of the complementary Gauss error function', function () {
                expect(FormulaUtils.erfc(0.0)).to.equal(1);
                expect(FormulaUtils.erfc(0.5)).to.be.closeTo(1 - FormulaUtils.erf(0.5), 1e-15);
                expect(FormulaUtils.erfc(1.0)).to.be.closeTo(1 - FormulaUtils.erf(1.0), 1e-15);
                expect(FormulaUtils.erfc(1.5)).to.be.closeTo(1 - FormulaUtils.erf(1.5), 1e-15);
                expect(FormulaUtils.erfc(2.0)).to.be.closeTo(1 - FormulaUtils.erf(2.0), 1e-15);
                expect(FormulaUtils.erfc(2.5)).to.be.closeTo(1 - FormulaUtils.erf(2.5), 1e-15);
                expect(FormulaUtils.erfc(3.0)).to.be.closeTo(1 - FormulaUtils.erf(3.0), 1e-15);
                expect(FormulaUtils.erfc(3.5)).to.be.closeTo(1 - FormulaUtils.erf(3.5), 1e-15);
                expect(FormulaUtils.erfc(4.0)).to.be.closeTo(1 - FormulaUtils.erf(4.0), 1e-15);
                expect(FormulaUtils.erfc(4.5)).to.be.closeTo(1 - FormulaUtils.erf(4.5), 1e-15);
                expect(FormulaUtils.erfc(5.0)).to.be.closeTo(1 - FormulaUtils.erf(5.0), 1e-15);
                expect(FormulaUtils.erfc(5.5)).to.be.closeTo(1 - FormulaUtils.erf(5.5), 1e-15);
                expect(FormulaUtils.erfc(6.0)).to.be.closeTo(1 - FormulaUtils.erf(6.0), 1e-15);
            });
            it('should return non-zero results for large input parameters', function () {
                expect(FormulaUtils.erfc(6)).to.be.above(0); // erf(6) returns exactly 1; but erfc(6) must not return 0
                expect(FormulaUtils.erfc(7)).to.be.above(0);
                expect(FormulaUtils.erfc(8)).to.be.above(0);
                expect(FormulaUtils.erfc(9)).to.be.above(0);
            });
            it('should return correct results for negative numbers', function () {
                expect(FormulaUtils.erfc(-1)).to.equal(1 - FormulaUtils.erf(-1));
                expect(FormulaUtils.erfc(-2)).to.equal(1 - FormulaUtils.erf(-2));
                expect(FormulaUtils.erfc(-3)).to.equal(1 - FormulaUtils.erf(-3));
                expect(FormulaUtils.erfc(-4)).to.equal(1 - FormulaUtils.erf(-4));
                expect(FormulaUtils.erfc(-5)).to.equal(1 - FormulaUtils.erf(-5));
                expect(FormulaUtils.erfc(-6)).to.equal(1 - FormulaUtils.erf(-6));
            });
        });

        describe('method "slope"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('slope');
            });
        });

        describe('method "combineParsedFormats"', function () {

            var stdFormat, percentFormat, currFormat, fracFormat, dateFormat, timeFormat, dateTimeFormat;
            before(function () {
                stdFormat = Parser.parseFormatCode('ooxml', 'op', 'General');
                percentFormat = Parser.parseFormatCode('ooxml', 'op', '0%');
                currFormat = Parser.parseFormatCode('ooxml', 'op', '0[$€]');
                fracFormat = Parser.parseFormatCode('ooxml', 'op', '# ?/?');
                dateFormat = Parser.parseFormatCode('ooxml', 'op', 'YYYY-MM-DD');
                timeFormat = Parser.parseFormatCode('ooxml', 'op', 'hh:mm:ss');
                dateTimeFormat = Parser.parseFormatCode('ooxml', 'op', 'YYYY-MM-DD hh:mm:ss');
            });

            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('combineParsedFormats');
            });
            it('should return null without if any format is null', function () {
                expect(FormulaUtils.combineParsedFormats(null, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(stdFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(percentFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(currFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(fracFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(timeFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, stdFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, percentFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, currFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, fracFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, dateFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, timeFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, dateTimeFormat)).to.equal(null);
            });
            it('should skip standard category', function () {
                expect(FormulaUtils.combineParsedFormats(stdFormat, stdFormat)).to.equal(stdFormat);

                expect(FormulaUtils.combineParsedFormats(percentFormat, stdFormat)).to.equal(percentFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, stdFormat)).to.equal(currFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, stdFormat)).to.equal(fracFormat);
                expect(FormulaUtils.combineParsedFormats(dateFormat, stdFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(timeFormat, stdFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, stdFormat)).to.equal(dateTimeFormat);

                expect(FormulaUtils.combineParsedFormats(stdFormat, percentFormat)).to.equal(percentFormat);
                expect(FormulaUtils.combineParsedFormats(stdFormat, currFormat)).to.equal(currFormat);
                expect(FormulaUtils.combineParsedFormats(stdFormat, fracFormat)).to.equal(fracFormat);
                expect(FormulaUtils.combineParsedFormats(stdFormat, dateFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(stdFormat, timeFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(stdFormat, dateTimeFormat)).to.equal(dateTimeFormat);
            });
            it('should add different category commbinations', function () {
                expect(FormulaUtils.combineParsedFormats(percentFormat, percentFormat)).to.equal(percentFormat);
                expect(FormulaUtils.combineParsedFormats(percentFormat, currFormat)).to.equal(percentFormat);
                expect(FormulaUtils.combineParsedFormats(percentFormat, fracFormat)).to.equal(percentFormat);
                expect(FormulaUtils.combineParsedFormats(percentFormat, dateFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(percentFormat, timeFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(percentFormat, dateTimeFormat)).to.equal(dateTimeFormat);

                expect(FormulaUtils.combineParsedFormats(currFormat, percentFormat)).to.equal(currFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, currFormat)).to.equal(currFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, fracFormat)).to.equal(currFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, dateFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, timeFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, dateTimeFormat)).to.equal(dateTimeFormat);

                expect(FormulaUtils.combineParsedFormats(fracFormat, percentFormat)).to.equal(fracFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, currFormat)).to.equal(fracFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, fracFormat)).to.equal(fracFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, dateFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, timeFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, dateTimeFormat)).to.equal(dateTimeFormat);

                expect(FormulaUtils.combineParsedFormats(dateFormat, percentFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(dateFormat, currFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(dateFormat, fracFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(dateFormat, dateFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateFormat, timeFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateFormat, dateTimeFormat)).to.equal(null);

                expect(FormulaUtils.combineParsedFormats(timeFormat, percentFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(timeFormat, currFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(timeFormat, fracFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(timeFormat, dateFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(timeFormat, timeFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(timeFormat, dateTimeFormat)).to.equal(null);

                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, percentFormat)).to.equal(dateTimeFormat);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, currFormat)).to.equal(dateTimeFormat);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, fracFormat)).to.equal(dateTimeFormat);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, dateFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, timeFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, dateTimeFormat)).to.equal(null);
            });
        });

        describe('method "getScalarType"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('getScalarType');
            });
            var ScalarType = FormulaUtils.ScalarType;
            it('should recognize numbers', function () {
                expect(FormulaUtils.getScalarType(0)).to.equal(ScalarType.NUMBER);
                expect(FormulaUtils.getScalarType(3)).to.equal(ScalarType.NUMBER);
                expect(FormulaUtils.getScalarType(-3)).to.equal(ScalarType.NUMBER);
                expect(FormulaUtils.getScalarType(1 / 0)).to.equal(ScalarType.NUMBER);
                expect(FormulaUtils.getScalarType(Number.NaN)).to.equal(ScalarType.NUMBER);
            });
            it('should recognize strings', function () {
                expect(FormulaUtils.getScalarType('')).to.equal(ScalarType.STRING);
                expect(FormulaUtils.getScalarType('abc')).to.equal(ScalarType.STRING);
                expect(FormulaUtils.getScalarType('ABC')).to.equal(ScalarType.STRING);
            });
            it('should recognize booleans', function () {
                expect(FormulaUtils.getScalarType(false)).to.equal(ScalarType.BOOLEAN);
                expect(FormulaUtils.getScalarType(true)).to.equal(ScalarType.BOOLEAN);
            });
            it('should recognize error codes', function () {
                expect(FormulaUtils.getScalarType(ErrorCode.DIV0)).to.equal(ScalarType.ERROR);
                expect(FormulaUtils.getScalarType(ErrorCode.NULL)).to.equal(ScalarType.ERROR);
            });
            it('should recognize null', function () {
                expect(FormulaUtils.getScalarType(null)).to.equal(ScalarType.NULL);
                expect(FormulaUtils.getScalarType(null, false)).to.equal(ScalarType.NULL);
                expect(FormulaUtils.getScalarType(null, true)).to.equal(ScalarType.NULL_MAX);
            });
        });

        describe('method "isZero"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('isZero');
            });
            it('should return true for zero and denormalized numbers', function () {
                expect(FormulaUtils.isZero(0)).to.equal(true);
                expect(FormulaUtils.isZero(Utils.MIN_NUMBER / 2)).to.equal(true);
                expect(FormulaUtils.isZero(-Utils.MIN_NUMBER / 2)).to.equal(true);
            });
            it('should return false for non-zero numbers, infinite, and NaN', function () {
                expect(FormulaUtils.isZero(Utils.MIN_NUMBER)).to.equal(false);
                expect(FormulaUtils.isZero(-Utils.MIN_NUMBER)).to.equal(false);
                expect(FormulaUtils.isZero(1)).to.equal(false);
                expect(FormulaUtils.isZero(-1)).to.equal(false);
                expect(FormulaUtils.isZero(Number.POSITIVE_INFINITY)).to.equal(false);
                expect(FormulaUtils.isZero(Number.NEGATIVE_INFINITY)).to.equal(false);
                expect(FormulaUtils.isZero(Number.NaN)).to.equal(false);
            });
        });

        describe('method "equalNumbers"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('equalNumbers');
            });
            it('should return true, if the numbers are equal', function () {
                expect(FormulaUtils.equalNumbers(-3, -3)).to.equal(true);
                expect(FormulaUtils.equalNumbers(0, 0)).to.equal(true);
                expect(FormulaUtils.equalNumbers(3, 3)).to.equal(true);
                expect(FormulaUtils.equalNumbers(0, Utils.MIN_NUMBER / 2)).to.equal(true);
                expect(FormulaUtils.equalNumbers(1, 1 + 2 * Utils.MIN_NUMBER)).to.equal(true);
                expect(FormulaUtils.equalNumbers(100, 100 * (1 + 2 * Utils.MIN_NUMBER))).to.equal(true);
            });
            it('should return false, if the numbers are not equal', function () {
                expect(FormulaUtils.equalNumbers(-3, 0)).to.equal(false);
                expect(FormulaUtils.equalNumbers(3, 0)).to.equal(false);
                expect(FormulaUtils.equalNumbers(0, -3)).to.equal(false);
                expect(FormulaUtils.equalNumbers(0, 3)).to.equal(false);
                expect(FormulaUtils.equalNumbers(1, 1 + 1e-14)).to.equal(false);
                expect(FormulaUtils.equalNumbers(100, 100 + 1e-12)).to.equal(false);
            });
            it('should return false, if either number is not finite', function () {
                expect(FormulaUtils.equalNumbers(3, 1 / 0)).to.equal(false);
                expect(FormulaUtils.equalNumbers(1 / 0, 3)).to.equal(false);
                expect(FormulaUtils.equalNumbers(1 / 0, 1 / 0)).to.equal(false);
                expect(FormulaUtils.equalNumbers(3, Number.NaN)).to.equal(false);
                expect(FormulaUtils.equalNumbers(Number.NaN, 3)).to.equal(false);
                expect(FormulaUtils.equalNumbers(Number.NaN, Number.NaN)).to.equal(false);
            });
        });

        describe('method "equalStrings"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('equalStrings');
            });
            it('should return false, if first string is less than second string (ignoring case)', function () {
                expect(FormulaUtils.equalStrings('a', 'b')).to.equal(false);
                expect(FormulaUtils.equalStrings('A', 'b')).to.equal(false);
                expect(FormulaUtils.equalStrings('a', 'B')).to.equal(false);
                expect(FormulaUtils.equalStrings('A', 'B')).to.equal(false);
                expect(FormulaUtils.equalStrings('a', 'aa')).to.equal(false);
                expect(FormulaUtils.equalStrings('ab', 'ba')).to.equal(false);
                expect(FormulaUtils.equalStrings('', 'a')).to.equal(false);
            });
            it('should return true, if the strings are equal (ignoring case)', function () {
                expect(FormulaUtils.equalStrings('a', 'a')).to.equal(true);
                expect(FormulaUtils.equalStrings('a', 'A')).to.equal(true);
                expect(FormulaUtils.equalStrings('A', 'a')).to.equal(true);
                expect(FormulaUtils.equalStrings('A', 'A')).to.equal(true);
                expect(FormulaUtils.equalStrings('abc', 'abc')).to.equal(true);
                expect(FormulaUtils.equalStrings('', '')).to.equal(true);
            });
            it('should return false, if first string is greater than second string (ignoring case)', function () {
                expect(FormulaUtils.equalStrings('b', 'a')).to.equal(false);
                expect(FormulaUtils.equalStrings('B', 'a')).to.equal(false);
                expect(FormulaUtils.equalStrings('b', 'A')).to.equal(false);
                expect(FormulaUtils.equalStrings('B', 'A')).to.equal(false);
                expect(FormulaUtils.equalStrings('aa', 'a')).to.equal(false);
                expect(FormulaUtils.equalStrings('ba', 'ab')).to.equal(false);
                expect(FormulaUtils.equalStrings('a', '')).to.equal(false);
            });
            it('should return false, if first string is less than second string (regarding case)', function () {
                expect(FormulaUtils.equalStrings('a', 'A', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('a', 'b', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('a', 'B', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('a', 'c', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('a', 'C', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('a', 'aa', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('ba', 'bA', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('', 'a', { withCase: true })).to.equal(false);
            });
            it('should return true, if the strings are equal (regarding case)', function () {
                expect(FormulaUtils.equalStrings('a', 'a', { withCase: true })).to.equal(true);
                expect(FormulaUtils.equalStrings('A', 'A', { withCase: true })).to.equal(true);
                expect(FormulaUtils.equalStrings('abc', 'abc', { withCase: true })).to.equal(true);
                expect(FormulaUtils.equalStrings('', '', { withCase: true })).to.equal(true);
            });
            it('should return false, if first string is greater than second string (regarding case)', function () {
                expect(FormulaUtils.equalStrings('A', 'a', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('b', 'a', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('B', 'a', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('c', 'a', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('C', 'a', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('aa', 'a', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('bA', 'ba', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalStrings('a', '', { withCase: true })).to.equal(false);
            });
        });

        describe('method "equalScalars"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('equalScalars');
            });
            it('should compare null to other types', function () {
                expect(FormulaUtils.equalScalars(null, 0, { nullMode: 'less' })).to.equal(false);
                expect(FormulaUtils.equalScalars(null, 0, { nullMode: 'greater' })).to.equal(false);
                expect(FormulaUtils.equalScalars(null, '', { nullMode: 'less' })).to.equal(false);
                expect(FormulaUtils.equalScalars(null, '', { nullMode: 'greater' })).to.equal(false);
                expect(FormulaUtils.equalScalars(null, false, { nullMode: 'less' })).to.equal(false);
                expect(FormulaUtils.equalScalars(null, false, { nullMode: 'greater' })).to.equal(false);
                expect(FormulaUtils.equalScalars(null, ErrorCode.NULL, { nullMode: 'less' })).to.equal(false);
                expect(FormulaUtils.equalScalars(null, ErrorCode.NULL, { nullMode: 'greater' })).to.equal(false);
            });
            it('should compare numbers to other types', function () {
                expect(FormulaUtils.equalScalars(0, null, { nullMode: 'less' })).to.equal(false);
                expect(FormulaUtils.equalScalars(0, null, { nullMode: 'greater' })).to.equal(false);
                expect(FormulaUtils.equalScalars(0, '0')).to.equal(false);
                expect(FormulaUtils.equalScalars(0, false)).to.equal(false);
                expect(FormulaUtils.equalScalars(0, ErrorCode.NULL)).to.equal(false);
            });
            it('should compare strings to other types', function () {
                expect(FormulaUtils.equalScalars('', null, { nullMode: 'less' })).to.equal(false);
                expect(FormulaUtils.equalScalars('', null, { nullMode: 'greater' })).to.equal(false);
                expect(FormulaUtils.equalScalars('0', 0)).to.equal(false);
                expect(FormulaUtils.equalScalars('false', false)).to.equal(false);
                expect(FormulaUtils.equalScalars('#NULL!', ErrorCode.NULL)).to.equal(false);
            });
            it('should compare booleans to other types', function () {
                expect(FormulaUtils.equalScalars(false, null, { nullMode: 'less' })).to.equal(false);
                expect(FormulaUtils.equalScalars(false, null, { nullMode: 'greater' })).to.equal(false);
                expect(FormulaUtils.equalScalars(false, 0)).to.equal(false);
                expect(FormulaUtils.equalScalars(false, 'false')).to.equal(false);
                expect(FormulaUtils.equalScalars(false, ErrorCode.NULL)).to.equal(false);
            });
            it('should compare error codes to other types', function () {
                expect(FormulaUtils.equalScalars(ErrorCode.NULL, null)).to.equal(false);
                expect(FormulaUtils.equalScalars(ErrorCode.NULL, null, { nullMode: 'less' })).to.equal(false);
                expect(FormulaUtils.equalScalars(ErrorCode.NULL, null, { nullMode: 'greater' })).to.equal(false);
                expect(FormulaUtils.equalScalars(ErrorCode.NULL, 0)).to.equal(false);
                expect(FormulaUtils.equalScalars(ErrorCode.NULL, '#NULL!')).to.equal(false);
                expect(FormulaUtils.equalScalars(ErrorCode.NULL, false)).to.equal(false);
            });
            it('should compare numbers', function () {
                expect(FormulaUtils.equalScalars(0, 1)).to.equal(false);
                expect(FormulaUtils.equalScalars(1, 1)).to.equal(true);
                expect(FormulaUtils.equalScalars(1, 0)).to.equal(false);
                expect(FormulaUtils.equalScalars(3, 1 / 0)).to.equal(false);
            });
            it('should compare strings (ignoring case)', function () {
                expect(FormulaUtils.equalScalars('a', 'a')).to.equal(true);
                expect(FormulaUtils.equalScalars('a', 'A')).to.equal(true);
                expect(FormulaUtils.equalScalars('a', 'b')).to.equal(false);
                expect(FormulaUtils.equalScalars('a', 'B')).to.equal(false);
                expect(FormulaUtils.equalScalars('b', 'a')).to.equal(false);
                expect(FormulaUtils.equalScalars('B', 'a')).to.equal(false);
            });
            it('should compare strings (regarding case)', function () {
                expect(FormulaUtils.equalScalars('a', 'a', { withCase: true })).to.equal(true);
                expect(FormulaUtils.equalScalars('a', 'A', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalScalars('A', 'a', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalScalars('a', 'B', { withCase: true })).to.equal(false);
                expect(FormulaUtils.equalScalars('A', 'b', { withCase: true })).to.equal(false);
            });
            it('should compare booleans', function () {
                expect(FormulaUtils.equalScalars(false, false)).to.equal(true);
                expect(FormulaUtils.equalScalars(false, true)).to.equal(false);
                expect(FormulaUtils.equalScalars(true, false)).to.equal(false);
                expect(FormulaUtils.equalScalars(true, true)).to.equal(true);
            });
            it('should compare error codes', function () {
                expect(FormulaUtils.equalScalars(ErrorCode.NULL, ErrorCode.NULL)).to.equal(true);
                expect(FormulaUtils.equalScalars(ErrorCode.NULL, ErrorCode.DIV0)).to.equal(false);
                expect(FormulaUtils.equalScalars(ErrorCode.DIV0, ErrorCode.NULL)).to.equal(false);
            });
            it('should convert null value to zero', function () {
                expect(FormulaUtils.equalScalars(null, -1, { nullMode: 'convert' })).to.equal(false);
                expect(FormulaUtils.equalScalars(null, 0, { nullMode: 'convert' })).to.equal(true);
                expect(FormulaUtils.equalScalars(null, 0)).to.equal(true);
                expect(FormulaUtils.equalScalars(null, 1, { nullMode: 'convert' })).to.equal(false);
                expect(FormulaUtils.equalScalars(-1, null, { nullMode: 'convert' })).to.equal(false);
                expect(FormulaUtils.equalScalars(0, null, { nullMode: 'convert' })).to.equal(true);
                expect(FormulaUtils.equalScalars(0, null)).to.equal(true);
                expect(FormulaUtils.equalScalars(1, null, { nullMode: 'convert' })).to.equal(false);
            });
            it('should convert null value to empty string', function () {
                expect(FormulaUtils.equalScalars(null, '', { nullMode: 'convert' })).to.equal(true);
                expect(FormulaUtils.equalScalars(null, '')).to.equal(true);
                expect(FormulaUtils.equalScalars(null, 'a', { nullMode: 'convert' })).to.equal(false);
                expect(FormulaUtils.equalScalars('', null, { nullMode: 'convert' })).to.equal(true);
                expect(FormulaUtils.equalScalars('', null)).to.equal(true);
                expect(FormulaUtils.equalScalars('a', null, { nullMode: 'convert' })).to.equal(false);
            });
            it('should convert null value to FALSE', function () {
                expect(FormulaUtils.equalScalars(null, false, { nullMode: 'convert' })).to.equal(true);
                expect(FormulaUtils.equalScalars(null, false)).to.equal(true);
                expect(FormulaUtils.equalScalars(null, true, { nullMode: 'convert' })).to.equal(false);
                expect(FormulaUtils.equalScalars(null, true)).to.equal(false);
                expect(FormulaUtils.equalScalars(false, null, { nullMode: 'convert' })).to.equal(true);
                expect(FormulaUtils.equalScalars(false, null)).to.equal(true);
                expect(FormulaUtils.equalScalars(true, null, { nullMode: 'convert' })).to.equal(false);
                expect(FormulaUtils.equalScalars(true, null)).to.equal(false);
            });
            it('should not convert null value to error code', function () {
                expect(FormulaUtils.equalScalars(null, ErrorCode.NULL, { nullMode: 'convert' })).to.equal(false);
                expect(FormulaUtils.equalScalars(null, ErrorCode.NULL)).to.equal(false);
                expect(FormulaUtils.equalScalars(ErrorCode.NULL, null, { nullMode: 'convert' })).to.equal(false);
                expect(FormulaUtils.equalScalars(ErrorCode.NULL, null)).to.equal(false);
            });
            it('should compare two null values', function () {
                expect(FormulaUtils.equalScalars(null, null)).to.equal(true);
                expect(FormulaUtils.equalScalars(null, null, { nullMode: 'less' })).to.equal(true);
                expect(FormulaUtils.equalScalars(null, null, { nullMode: 'greater' })).to.equal(true);
                expect(FormulaUtils.equalScalars(null, null, { nullMode: 'convert' })).to.equal(true);
            });
        });

        describe('method "compareNumbers"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareNumbers');
            });
            it('should return -1, if first number is less than second number', function () {
                expect(FormulaUtils.compareNumbers(-3, 0)).to.equal(-1);
                expect(FormulaUtils.compareNumbers(0, 3)).to.equal(-1);
                expect(FormulaUtils.compareNumbers(1, 1 + 1e-14)).to.equal(-1);
                expect(FormulaUtils.compareNumbers(100, 100 + 1e-12)).to.equal(-1);
            });
            it('should return 0, if the numbers are equal', function () {
                expect(FormulaUtils.compareNumbers(-3, -3)).to.equal(0);
                expect(FormulaUtils.compareNumbers(0, 0)).to.equal(0);
                expect(FormulaUtils.compareNumbers(3, 3)).to.equal(0);
                expect(FormulaUtils.compareNumbers(0, Utils.MIN_NUMBER / 2)).to.equal(0);
                expect(FormulaUtils.compareNumbers(1, 1 + 2 * Utils.MIN_NUMBER)).to.equal(0);
                expect(FormulaUtils.compareNumbers(100, 100 * (1 + 2 * Utils.MIN_NUMBER))).to.equal(0);
            });
            it('should return 1, if first number is greater than second number', function () {
                expect(FormulaUtils.compareNumbers(3, 0)).to.equal(1);
                expect(FormulaUtils.compareNumbers(0, -3)).to.equal(1);
                expect(FormulaUtils.compareNumbers(1 + 1e-14, 1)).to.equal(1);
                expect(FormulaUtils.compareNumbers(100 + 1e-12, 100)).to.equal(1);
            });
            it('should return NaN, if either number is not finite', function () {
                expect(FormulaUtils.compareNumbers(3, 1 / 0)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(1 / 0, 3)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(1 / 0, 1 / 0)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(3, Number.NaN)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(Number.NaN, 3)).to.be.NaN;
                expect(FormulaUtils.compareNumbers(Number.NaN, Number.NaN)).to.be.NaN;
            });
        });

        describe('method "compareStrings"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareStrings');
            });
            it('should return -1, if first string is less than second string (ignoring case)', function () {
                expect(FormulaUtils.compareStrings('a', 'b')).to.equal(-1);
                expect(FormulaUtils.compareStrings('A', 'b')).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'B')).to.equal(-1);
                expect(FormulaUtils.compareStrings('A', 'B')).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'aa')).to.equal(-1);
                expect(FormulaUtils.compareStrings('ab', 'ba')).to.equal(-1);
                expect(FormulaUtils.compareStrings('', 'a')).to.equal(-1);
            });
            it('should return 0, if the strings are equal (ignoring case)', function () {
                expect(FormulaUtils.compareStrings('a', 'a')).to.equal(0);
                expect(FormulaUtils.compareStrings('a', 'A')).to.equal(0);
                expect(FormulaUtils.compareStrings('A', 'a')).to.equal(0);
                expect(FormulaUtils.compareStrings('A', 'A')).to.equal(0);
                expect(FormulaUtils.compareStrings('abc', 'abc')).to.equal(0);
                expect(FormulaUtils.compareStrings('', '')).to.equal(0);
            });
            it('should return 1, if first string is greater than second string (ignoring case)', function () {
                expect(FormulaUtils.compareStrings('b', 'a')).to.equal(1);
                expect(FormulaUtils.compareStrings('B', 'a')).to.equal(1);
                expect(FormulaUtils.compareStrings('b', 'A')).to.equal(1);
                expect(FormulaUtils.compareStrings('B', 'A')).to.equal(1);
                expect(FormulaUtils.compareStrings('aa', 'a')).to.equal(1);
                expect(FormulaUtils.compareStrings('ba', 'ab')).to.equal(1);
                expect(FormulaUtils.compareStrings('a', '')).to.equal(1);
            });
            it('should return -1, if first string is less than second string (regarding case)', function () {
                expect(FormulaUtils.compareStrings('a', 'A', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'b', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'B', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'c', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'C', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('a', 'aa', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('ba', 'bA', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareStrings('', 'a', { withCase: true })).to.equal(-1);
            });
            it('should return 0, if the strings are equal (regarding case)', function () {
                expect(FormulaUtils.compareStrings('a', 'a', { withCase: true })).to.equal(0);
                expect(FormulaUtils.compareStrings('A', 'A', { withCase: true })).to.equal(0);
                expect(FormulaUtils.compareStrings('abc', 'abc', { withCase: true })).to.equal(0);
                expect(FormulaUtils.compareStrings('', '', { withCase: true })).to.equal(0);
            });
            it('should return 1, if first string is greater than second string (regarding case)', function () {
                expect(FormulaUtils.compareStrings('A', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('b', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('B', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('c', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('C', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('aa', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('bA', 'ba', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareStrings('a', '', { withCase: true })).to.equal(1);
            });
        });

        describe('method "compareBooleans"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareBooleans');
            });
            it('should return the result of Boolean comparison', function () {
                expect(FormulaUtils.compareBooleans(false, false)).to.equal(0);
                expect(FormulaUtils.compareBooleans(false, true)).to.equal(-1);
                expect(FormulaUtils.compareBooleans(true, false)).to.equal(1);
                expect(FormulaUtils.compareBooleans(true, true)).to.equal(0);
            });
        });

        describe('method "compareErrorCodes"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareErrorCodes');
            });
            it('should return the result of error code comparison', function () {
                expect(FormulaUtils.compareErrorCodes(ErrorCode.NULL, ErrorCode.NULL)).to.equal(0);
                expect(FormulaUtils.compareErrorCodes(ErrorCode.NULL, ErrorCode.DIV0)).to.equal(-1);
                expect(FormulaUtils.compareErrorCodes(ErrorCode.DIV0, ErrorCode.NULL)).to.equal(1);
            });
        });

        describe('method "compareScalars"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('compareScalars');
            });
            it('should compare null to other types', function () {
                expect(FormulaUtils.compareScalars(null, 0, { nullMode: 'less' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(null, 0, { nullMode: 'greater' })).to.equal(1);
                expect(FormulaUtils.compareScalars(null, '', { nullMode: 'less' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(null, '', { nullMode: 'greater' })).to.equal(1);
                expect(FormulaUtils.compareScalars(null, false, { nullMode: 'less' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(null, false, { nullMode: 'greater' })).to.equal(1);
                expect(FormulaUtils.compareScalars(null, ErrorCode.NULL, { nullMode: 'less' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(null, ErrorCode.NULL, { nullMode: 'greater' })).to.equal(1);
            });
            it('should compare numbers to other types', function () {
                expect(FormulaUtils.compareScalars(0, null, { nullMode: 'less' })).to.equal(1);
                expect(FormulaUtils.compareScalars(0, null, { nullMode: 'greater' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(0, '0')).to.equal(-1);
                expect(FormulaUtils.compareScalars(0, false)).to.equal(-1);
                expect(FormulaUtils.compareScalars(0, ErrorCode.NULL)).to.equal(-1);
            });
            it('should compare strings to other types', function () {
                expect(FormulaUtils.compareScalars('', null, { nullMode: 'less' })).to.equal(1);
                expect(FormulaUtils.compareScalars('', null, { nullMode: 'greater' })).to.equal(-1);
                expect(FormulaUtils.compareScalars('0', 0)).to.equal(1);
                expect(FormulaUtils.compareScalars('false', false)).to.equal(-1);
                expect(FormulaUtils.compareScalars('#NULL!', ErrorCode.NULL)).to.equal(-1);
            });
            it('should compare booleans to other types', function () {
                expect(FormulaUtils.compareScalars(false, null, { nullMode: 'less' })).to.equal(1);
                expect(FormulaUtils.compareScalars(false, null, { nullMode: 'greater' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(false, 0)).to.equal(1);
                expect(FormulaUtils.compareScalars(false, 'false')).to.equal(1);
                expect(FormulaUtils.compareScalars(false, ErrorCode.NULL)).to.equal(-1);
            });
            it('should compare error codes to other types', function () {
                expect(FormulaUtils.compareScalars(ErrorCode.NULL, null)).to.equal(1);
                expect(FormulaUtils.compareScalars(ErrorCode.NULL, null, { nullMode: 'less' })).to.equal(1);
                expect(FormulaUtils.compareScalars(ErrorCode.NULL, null, { nullMode: 'greater' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(ErrorCode.NULL, 0)).to.equal(1);
                expect(FormulaUtils.compareScalars(ErrorCode.NULL, '#NULL!')).to.equal(1);
                expect(FormulaUtils.compareScalars(ErrorCode.NULL, false)).to.equal(1);
            });
            it('should compare numbers', function () {
                expect(FormulaUtils.compareScalars(0, 1)).to.equal(-1);
                expect(FormulaUtils.compareScalars(1, 1)).to.equal(0);
                expect(FormulaUtils.compareScalars(1, 0)).to.equal(1);
                expect(FormulaUtils.compareScalars(3, 1 / 0)).to.be.NaN;
            });
            it('should compare strings (ignoring case)', function () {
                expect(FormulaUtils.compareScalars('a', 'a')).to.equal(0);
                expect(FormulaUtils.compareScalars('a', 'A')).to.equal(0);
                expect(FormulaUtils.compareScalars('a', 'b')).to.equal(-1);
                expect(FormulaUtils.compareScalars('a', 'B')).to.equal(-1);
                expect(FormulaUtils.compareScalars('b', 'a')).to.equal(1);
                expect(FormulaUtils.compareScalars('B', 'a')).to.equal(1);
            });
            it('should compare strings (regarding case)', function () {
                expect(FormulaUtils.compareScalars('a', 'a', { withCase: true })).to.equal(0);
                expect(FormulaUtils.compareScalars('a', 'A', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareScalars('A', 'a', { withCase: true })).to.equal(1);
                expect(FormulaUtils.compareScalars('a', 'B', { withCase: true })).to.equal(-1);
                expect(FormulaUtils.compareScalars('A', 'b', { withCase: true })).to.equal(-1);
            });
            it('should compare booleans', function () {
                expect(FormulaUtils.compareScalars(false, false)).to.equal(0);
                expect(FormulaUtils.compareScalars(false, true)).to.equal(-1);
                expect(FormulaUtils.compareScalars(true, false)).to.equal(1);
                expect(FormulaUtils.compareScalars(true, true)).to.equal(0);
            });
            it('should compare error codes', function () {
                expect(FormulaUtils.compareScalars(ErrorCode.NULL, ErrorCode.NULL)).to.equal(0);
                expect(FormulaUtils.compareScalars(ErrorCode.NULL, ErrorCode.DIV0)).to.equal(-1);
                expect(FormulaUtils.compareScalars(ErrorCode.DIV0, ErrorCode.NULL)).to.equal(1);
            });
            it('should convert null value to zero', function () {
                expect(FormulaUtils.compareScalars(null, -1, { nullMode: 'convert' })).to.equal(1);
                expect(FormulaUtils.compareScalars(null, 0, { nullMode: 'convert' })).to.equal(0);
                expect(FormulaUtils.compareScalars(null, 1, { nullMode: 'convert' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(null, 0)).to.equal(0);
                expect(FormulaUtils.compareScalars(-1, null, { nullMode: 'convert' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(0, null, { nullMode: 'convert' })).to.equal(0);
                expect(FormulaUtils.compareScalars(0, null)).to.equal(0);
                expect(FormulaUtils.compareScalars(1, null, { nullMode: 'convert' })).to.equal(1);
            });
            it('should convert null value to empty string', function () {
                expect(FormulaUtils.compareScalars(null, '', { nullMode: 'convert' })).to.equal(0);
                expect(FormulaUtils.compareScalars(null, '')).to.equal(0);
                expect(FormulaUtils.compareScalars(null, 'a', { nullMode: 'convert' })).to.equal(-1);
                expect(FormulaUtils.compareScalars('', null, { nullMode: 'convert' })).to.equal(0);
                expect(FormulaUtils.compareScalars('', null)).to.equal(0);
                expect(FormulaUtils.compareScalars('a', null, { nullMode: 'convert' })).to.equal(1);
            });
            it('should convert null value to FALSE', function () {
                expect(FormulaUtils.compareScalars(null, false, { nullMode: 'convert' })).to.equal(0);
                expect(FormulaUtils.compareScalars(null, false)).to.equal(0);
                expect(FormulaUtils.compareScalars(null, true, { nullMode: 'convert' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(null, true)).to.equal(-1);
                expect(FormulaUtils.compareScalars(false, null, { nullMode: 'convert' })).to.equal(0);
                expect(FormulaUtils.compareScalars(false, null)).to.equal(0);
                expect(FormulaUtils.compareScalars(true, null, { nullMode: 'convert' })).to.equal(1);
                expect(FormulaUtils.compareScalars(true, null)).to.equal(1);
            });
            it('should not convert null value to error code', function () {
                expect(FormulaUtils.compareScalars(null, ErrorCode.NULL, { nullMode: 'convert' })).to.equal(-1);
                expect(FormulaUtils.compareScalars(null, ErrorCode.NULL)).to.equal(-1);
                expect(FormulaUtils.compareScalars(ErrorCode.NULL, null, { nullMode: 'convert' })).to.equal(1);
                expect(FormulaUtils.compareScalars(ErrorCode.NULL, null)).to.equal(1);
            });
            it('should compare two null values', function () {
                expect(FormulaUtils.compareScalars(null, null)).to.equal(0);
                expect(FormulaUtils.compareScalars(null, null, { nullMode: 'less' })).to.equal(0);
                expect(FormulaUtils.compareScalars(null, null, { nullMode: 'greater' })).to.equal(0);
                expect(FormulaUtils.compareScalars(null, null, { nullMode: 'convert' })).to.equal(0);
            });
        });

        describe('method "isValidMatrixSize"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('isValidMatrixSize');
            });
            it('should return true for valid matrix sizes', function () {
                expect(FormulaUtils.isValidMatrixSize(1, 1)).to.equal(true);
                expect(FormulaUtils.isValidMatrixSize(FormulaUtils.MAX_MATRIX_ROW_COUNT, 1)).to.equal(true);
                expect(FormulaUtils.isValidMatrixSize(1, FormulaUtils.MAX_MATRIX_COL_COUNT)).to.equal(true);
                expect(FormulaUtils.isValidMatrixSize(FormulaUtils.MAX_MATRIX_ROW_COUNT, FormulaUtils.MAX_MATRIX_COL_COUNT)).to.equal(true);
            });
            it('should return false for invalid matrix sizes', function () {
                expect(FormulaUtils.isValidMatrixSize(0, 1)).to.equal(false);
                expect(FormulaUtils.isValidMatrixSize(1, 0)).to.equal(false);
                expect(FormulaUtils.isValidMatrixSize(FormulaUtils.MAX_MATRIX_ROW_COUNT + 1, 1)).to.equal(false);
                expect(FormulaUtils.isValidMatrixSize(1, FormulaUtils.MAX_MATRIX_COL_COUNT + 1)).to.equal(false);
            });
        });

        describe('method "ensureMatrixSize"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('ensureMatrixSize');
            });
            it('should do nothing for valid matrix sizes', function () {
                expect(FormulaUtils.ensureMatrixSize(1, 1)).to.equal(undefined);
                expect(FormulaUtils.ensureMatrixSize(FormulaUtils.MAX_MATRIX_ROW_COUNT, 1)).to.equal(undefined);
                expect(FormulaUtils.ensureMatrixSize(1, FormulaUtils.MAX_MATRIX_COL_COUNT)).to.equal(undefined);
                expect(FormulaUtils.ensureMatrixSize(FormulaUtils.MAX_MATRIX_ROW_COUNT, FormulaUtils.MAX_MATRIX_COL_COUNT)).to.equal(undefined);
            });
            it('should throw for invalid matrix sizes', function () {
                expect(FormulaUtils.ensureMatrixSize.bind(null, 0, 1)).to.throw(ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
                expect(FormulaUtils.ensureMatrixSize.bind(null, 1, 0)).to.throw(ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
                expect(FormulaUtils.ensureMatrixSize.bind(null, FormulaUtils.MAX_MATRIX_ROW_COUNT + 1, 1)).to.throw(ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
                expect(FormulaUtils.ensureMatrixSize.bind(null, 1, FormulaUtils.MAX_MATRIX_COL_COUNT + 1)).to.throw(ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
            });
        });

        describe('method "implementFilterAggregation"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('implementFilterAggregation');
            });
            it('should return an aggregator function', function () {
                var resolver = FormulaUtils.implementFilterAggregation(1, _.identity, _.identity);
                expect(resolver).to.be.a('function');
            });
        });

        describe('method "implementParallelFilterAggregation"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('implementParallelFilterAggregation');
            });
            it('should return an aggregator function', function () {
                var resolver = FormulaUtils.implementParallelFilterAggregation(1, _.identity, _.identity);
                expect(resolver).to.be.a('function');
            });
        });

        describe('method "valueToString"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('valueToString');
            });
            it('should convert strings', function () {
                expect(FormulaUtils.valueToString('abc')).to.equal('"abc"');
                expect(FormulaUtils.valueToString('a"b"c')).to.equal('"a""b""c"');
                expect(FormulaUtils.valueToString('')).to.equal('""');
            });
            it('should convert dates', function () {
                var now = new Date();
                expect(FormulaUtils.valueToString(now)).to.equal(now.toISOString());
            });
            it('should convert null and booleans', function () {
                expect(FormulaUtils.valueToString(null)).to.equal('null');
                expect(FormulaUtils.valueToString(true)).to.equal('TRUE');
                expect(FormulaUtils.valueToString(false)).to.equal('FALSE');
            });
            it('should convert numbers', function () {
                expect(FormulaUtils.valueToString(42)).to.equal('42');
                expect(FormulaUtils.valueToString(-12.5)).to.equal('-12.5');
            });
            it('should convert error codes', function () {
                expect(FormulaUtils.valueToString(ErrorCode.REF)).to.equal('#REF');
            });
        });

        describe('method "logTokens"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('logTokens');
            });
        });
    });

    // ========================================================================
});
