/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/cellcollection'
], function (AppHelper, SheetHelper, IteratorUtils, SheetUtils, CellCollection) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var a = SheetHelper.a;
    var r = SheetHelper.r;
    var ra = SheetHelper.ra;
    var rangesMatcher = SheetHelper.mergedRangesMatcher;

    // class CellCollection ===================================================

    describe('Spreadsheet class CellCollection', function () {

        it('should exist', function () {
            expect(CellCollection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a0', attrs: {}, default: true },

            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'changeCells', sheet: 1, start: 'A1', contents: [[1, 2, 3, 4]] },
            { name: 'changeCells', sheet: 1, start: 'A3', contents: [[{ v: 10, f: 'SUM(A1:D1)' }, { v: 111, f: 'B6' }]] },
            { name: 'changeCells', sheet: 1, start: 'B6', contents: [[111]] },

            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'changeCells', sheet: 2, start: 'A2', contents: [[5, 6, 7, 8]] },
            { name: 'changeCells', sheet: 2, start: 'A4', contents: [[{ v: 26, f: 'SUM(A2:D2)' }, { v: '3', f: 'Sheet2!C1' }]] },

            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet4' },
            { name: 'changeRows', attrs: { row: { height: 556, visible: true } }, sheet: 3, start: 0 },
            { name: 'changeRows', attrs: { row: { visible: true } }, sheet: 3, start: 1, end: 8 },
            { name: 'changeCells', sheet: 3, start: 'B1', contents: [
                [1,    { v: 50,  r: 5 }, { v: 11, f: 'SUM(NamedRange_Sheet1)' },   '< NamedRange Sheet1',   null, null,     'Kg', 'Euro'],
                [100,  { v: 666, r: 5 }, { v: 21, f: 'SUM(NamedRange_Workbook)' }, '< NamedRange Workbook', null, 'apples', 1,    2.5],
                [50,   { v: 84,  r: 5 }, null, null, null, 'pears',  2, 6],
                [null, { v: 5,   r: 5 }, null, null, null, 'grapes', 1, 8],
                [null, { v: 2,   r: 5 }],
                [1],
                [10],
                [20],
                [1]
            ] },
            // data validation
            { name: 'insertValidation', sheet: 3, index: 0, ranges: [{ start: [9, 0] }], type: 'source', value1: '$B$1:$B$3' },
            // named ranges
            { name: 'insertName', sheet: 3, label: 'NamedRange_Sheet1', formula: 'Sheet4!$B$6:$B$7' },
            { name: 'insertName', label: 'NamedRange_Workbook', formula: 'Sheet4!$B$8:$B$9' },
            // chart drawing
            { name: 'insertDrawing', start: [3, 0], type: 'chart', attrs: { drawing: { startCol: 13, startColOffset: 0, startRow: 0, startRowOffset: 0, endCol: 17, endColOffset: 0, endRow: 5, endRowOffset: 0, anchorType: 'twoCell' }, chart: { type: 'column2d', chartStyleId: 2, stacking: 'clustered', chartColors: { id: 10, meth: 'cycle', schemeClr: [{ color: { type: 'scheme', value: 'accent1' } }, { color: { type: 'scheme', value: 'accent2' } }, { color: { type: 'scheme', value: 'accent3' } }, { color: { type: 'scheme', value: 'accent4' } }, { color: { type: 'scheme', value: 'accent5' } }, { color: { type: 'scheme', value: 'accent6' } }] }, varyColors: false }, line: { type: 'solid', width: 26, color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }] } }, fill: { type: 'solid', color: { type: 'scheme', value: 'background1' } } } },
            { name: 'insertChartDataSeries', start: [3, 0], series: 0, attrs: { series: { title: 'Sheet4!$L$1', names: 'Sheet4!$K$2:$K$4', values: 'Sheet4!$L$2:$L$4' }, line: { type: 'none' }, fill: { type: 'solid', color: { type: 'scheme', value: 'accent1' } } } },
            { name: 'insertChartDataSeries', start: [3, 0], series: 1, attrs: { series: { title: 'Sheet4!$M$1', names: 'Sheet4!$K$2:$K$4', values: 'Sheet4!$M$2:$M$4' }, line: { type: 'none' }, fill: { type: 'solid', color: { type: 'scheme', value: 'accent2' } } } },
            { name: 'setChartAxisAttributes', start: [3, 0], axis: 'x', attrs: { line: { type: 'solid', width: 26, color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }] } }, fill: { type: 'none' }, axis: { label: true }, character: { color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 65000 }, { type: 'lumOff', value: 35000 }] }, fontSize: 9 } } },
            { name: 'setChartAxisAttributes', start: [3, 0], axis: 'y', attrs: { line: { type: 'none', width: 18 }, fill: { type: 'none' }, axis: { label: true }, character: { color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 65000 }, { type: 'lumOff', value: 35000 }] }, fontSize: 9 } } },
            { name: 'setChartGridlineAttributes', start: [3, 0], axis: 'y', attrs: { line: { type: 'solid', width: 26, color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }] } }, fill: { type: 'none' } } },
            { name: 'setChartLegendAttributes', start: [3, 0], attrs: { legend: { pos: 'off' } } },
            // conditional formatting
            { name: 'insertCFRule', sheet: 3, id: '0', ranges: 'C1:C5', type: 'colorScale', colorScale: [{ t: 'formula', v: '$B$1', c: { type: 'rgb', value: 'FF7128' } }, { t: 'formula', v: '$B$2', c: { type: 'rgb', value: 'FFEF9C' } }], priority: 5 },
            { name: 'insertCFRule', sheet: 3, id: '1', ranges: 'D1:D5', type: 'colorScale', colorScale: [{ t: 'formula', v: '$B$1', c: { type: 'rgb', value: 'F8696B' } }, { t: 'formula', v: '$B$3', c: { type: 'rgb', value: 'FFEB84' } }, { t: 'formula', v: '$B$2', c: { type: 'rgb', value: '63BE7B' } }], priority: 4 },
            { name: 'insertCFRule', sheet: 3, id: '2', ranges: 'E1:E5', type: 'dataBar', dataBar: { r1: { t: 'formula', v: '$B$1' }, r2: { t: 'formula', v: '$B$2' }, c: { type: 'rgb', value: '638EC6' }, s: true }, priority: 3 },
            { name: 'insertCFRule', sheet: 3, id: '3', ranges: 'F1:F5', type: 'between', value1: '$B$1', value2: '$B$3', priority: 2, attrs: { cell: { fillColor: { type: 'rgb', value: '00B050' } }, rule: { type: 'between', value1: '$B$1', value2: '$B$3', priority: 2, stop: false, colorScale: [], dataBar: { } } } },
            { name: 'insertCFRule', sheet: 3, id: '4', ranges: 'G1:G5', type: 'less', value1: '$B$2', priority: 1, attrs: { cell: { fillColor: { type: 'rgb', value: '00B050' } }, rule: { type: 'less', value1: '$B$2', value2: '', priority: 1, stop: false, colorScale: [], dataBar: { } } } },

            { name: 'insertSheet', sheet: 4, sheetName: 'Sheet5' },
            { name: 'insertSheet', sheet: 5, sheetName: 'Sheet6' },
            { name: 'insertSheet', sheet: 6, sheetName: 'Sheet7' },
            { name: 'insertSheet', sheet: 7, sheetName: 'Sheet8' },

            // Sheet to test SharedFormulas with References
            { name: 'insertSheet', sheet: 8, sheetName: 'Sheet9' },
            { name: 'changeCells', sheet: 8, start: 'A1', contents: [
                [1, { f: 'SUM(A1, $A$1:$A$4)', si: 0, sr: { start: [1, 0], end: [4, 0] }, v: 11 }, { si: 0, v: 21 }, { si: 0, v: 31 }, { si: 0, v: 41 }],
                [2],
                [3],
                [4]
            ] },

            // Sheet to test SharedFormulas without References
            { name: 'insertSheet', sheet: 9, sheetName: 'Sheet10' },
            { name: 'changeCells', sheet: 9, start: 'A1', contents: [
                [{ f: 'SUM(_xlfn.CEILING.MATH(4.9),ABS(0.1))', si: 0, sr: { start: [0, 0], end: [4, 9] }, v: 5.1 }, { si: 0, v: 5.1, r: 4 }],
                { c: [{ si: 0, v: 5.1, r: 5 }], r: 9 }
            ] },

            { name: 'insertSheet', sheet: 10, sheetName: 'Sort-Sheet' },
            { name: 'changeCells', sheet: 10, start: [0, 0], contents: [
                ['headline', 'headline', 'headline', 9],
                [2, 2, 2, 'hier'],
                [3, 2, 5, 12],
                [4, 2, 4, 0],
                [5, 2, 1, 'bla']
            ] },

            { name: 'insertSheet', sheet: 11, sheetName: 'Sort-Sheet 2' },
            { name: 'changeCells', sheet: 11, start: [0, 0], contents: [
                ['headline', 2, 2, 4, 5],
                ['headline', 2, 5, 12, 4],
                ['headline', 2, 4, 0, 4],
                [9, 'hier', 1, 5, 'bla']
            ] }
        ];

        // initialize test document
        var docModel = null, sheetModel = null, cellCollection = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            sheetModel = docModel.getSheetModel(0);
            cellCollection = sheetModel.getCellCollection();
        });

        // color attribute values
        var AUTO = { type: 'auto' };
        var RED = { type: 'rgb', value: 'FF0000' };

        // border attribute values
        var NONE = { style: 'none' };
        var THIN = { style: 'solid', width: 26, color: AUTO };
        var DBL = { style: 'double', width: 79, color: AUTO };
        var THIN_RED = { style: 'solid', width: 26, color: RED };
        var DBL_RED = { style: 'double', width: 79, color: RED };

        // build border attributes by character shortcuts
        function borders() {
            var attrs = {};
            function add(keys, border) {
                _.each(keys, function (key) {
                    attrs[SheetUtils.getBorderName(key)] = border;
                });
            }
            for (var i = 0; i < arguments.length; i += 2) {
                add(arguments[i], arguments[i + 1]);
            }
            return attrs;
        }

        // creates a matcher function for Chai's satisfy() assertion that maches attribute values
        function expectAttributes(attrs, expected) {
            expect(attrs).to.be.an('object');
            _.each(expected, function (value, name) {
                expect(attrs).to.have.a.property(name).that.deep.equals(value);
            });
        }

        function expectCellAttributes(range, expected) {
            range = r(range);
            for (var address = range.start.clone(); address[1] <= range.end[1]; address[1] += 1) {
                for (address[0] = range.start[0]; address[0] <= range.end[0]; address[0] += 1) {
                    var cellAttrs = cellCollection.getCellAttributeSet(address).cell;
                    //console.log(address + ' attrs=' + JSON.stringify(cellAttrs));
                    expectAttributes(cellAttrs, expected);
                }
            }
        }

        // public methods -----------------------------------------------------

        describe('method "getUsedCols"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('getUsedCols');
            });
            it('should return the number of used columns', function () {
                expect(cellCollection.getUsedCols()).to.equal(0);
            });
        });

        describe('method "getUsedRows"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('getUsedRows');
            });
            it('should return the number of used rows', function () {
                expect(cellCollection.getUsedRows()).to.equal(0);
            });
        });

        describe('method "getUsedCount"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('getUsedCount');
            });
            it('should return the number of used columns and rows', function () {
                expect(cellCollection.getUsedCount(false)).to.equal(0);
                expect(cellCollection.getUsedCount(true)).to.equal(0);
            });
        });

        // operation generators -----------------------------------------------

        describe('method "parseCellValue"', function () {

            function expectParseResult(parseText, exp) {
                var parseResult = cellCollection.parseCellValue(parseText, a('A1'));
                expect(parseResult).to.have.a.property('v', exp.v);
                expect(parseResult).to.have.a.property('f', exp.f || null);
                if (exp.type) {
                    expect(parseResult).to.have.a.property('result').that.is.an('object');
                    expect(parseResult.result).to.have.a.property('type', exp.type);
                    expect(parseResult.result).to.have.a.property('value', exp.v);
                    if (exp.code) {
                        expect(parseResult.result).to.have.a.property('code', exp.code);
                    } else {
                        expect(parseResult.result).to.not.have.a.property('code');
                    }
                    if (exp.format) {
                        expect(parseResult.result).to.have.a.property('format').that.is.an('object');
                        expect(parseResult.result.format).to.have.a.property('formatCode', exp.format);
                    } else {
                        expect(parseResult.result).to.not.have.a.property('format');
                    }
                } else {
                    expect(parseResult).to.have.a.property('result', null);
                }
                if (exp.format && (exp.format !== 'General')) {
                    expect(parseResult).to.have.a.property('format', exp.format);
                } else {
                    expect(parseResult).to.not.have.a.property('format');
                }
            }

            it('should exist', function () {
                expect(cellCollection).to.respondTo('parseCellValue');
            });
            it('should parse literal values', function () {
                expectParseResult('', { v: null });
                expectParseResult('0', { v: 0 });
                expectParseResult('42,5', { v: 42.5 });
                expectParseResult('-42', { v: -42 });
                expectParseResult('wahr', { v: true });
                expectParseResult('Falsch', { v: false });
                expectParseResult('#Wert!', { v: ErrorCode.VALUE });
                expectParseResult('abc', { v: 'abc' });
                expectParseResult('0a', { v: '0a' });
                expectParseResult(' wahr', { v: ' wahr' });
                expectParseResult('true', { v: 'true' });
                expectParseResult('#value!', { v: '#value!' });
            });
            it('should parse formatted numbers', function () {
                expectParseResult('1e3', { v: 1000, format: '0.00E+00' });
                expectParseResult('-100 %', { v: -1, format: '0%' });
                expectParseResult('24.4.1977', { v: 28239, format: 'DD.MM.YYYY' });
            });
            it('should parse formula expressions', function () {
                expectParseResult('=1+1', { v: 2, f: '1+1', type: 'valid', format: 'General' });
                expectParseResult('=SUMME(1;2;3)', { v: 6, f: 'SUM(1,2,3)', type: 'valid', format: 'General' });
                expectParseResult('=DATUM(2015;1;1)', { v: 42005, f: 'DATE(2015,1,1)', type: 'valid', format: 'DD.MM.YYYY' });
                expectParseResult('+2*3', { v: 6, f: '+2*3', type: 'valid' });
                expectParseResult('-2*3', { v: -6, f: '-2*3', type: 'valid' });
                expectParseResult('=1+', { v: ErrorCode.NA, f: '1+', type: 'error', code: 'missing' });
            });
            it('should handle leading apostrophes', function () {
                expectParseResult('\'0', { v: '0' });
                expectParseResult('\'WAHR', { v: 'WAHR' });
                expectParseResult('\'#WERT!', { v: '#WERT!' });
                expectParseResult('\'=1+1', { v: '=1+1' });
            });
            it('should not parse special repeated characters as formulas', function () {
                expectParseResult('=', { v: '=' });
                expectParseResult('+', { v: '+' });
                expectParseResult('-', { v: '-' });
                expectParseResult('===', { v: '===' });
                expectParseResult('+++', { v: '+++' });
                expectParseResult('---', { v: '---' });
            });
        });

        describe('method "generateCellContentOperations"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateCellContentOperations');
            });
            it('should generate correct cell operations', function (done) {
                expect(cellCollection.getCellValue(a('A1'))).to.equal(null);
                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateCellContentOperations(generator, a('A1'), { v: 42 });
                expect(promise).to.respondTo('promise');
                promise.always(function (changed) {
                    expect(promise.state()).to.equal('resolved');
                    expect(changed).to.deep.equal(ra('A1:A1'));
                    expect(cellCollection.getCellValue(a('A1'))).to.equal(42);
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expect(cellCollection.getCellValue(a('A1'))).to.equal(null);
                    done();
                });
            });
        });

        describe('method "generateFillOperations"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateFillOperations');
            });
//            it('should generate correct cell operations', function () {
//            });
        });

        describe('method "generateBorderOperations"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateBorderOperations');
            });
            it('should create a border around a range of cells', function (done) {
                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateBorderOperations(generator, r('B2:E5'), borders('tblr', THIN));
                expect(promise).to.respondTo('promise');
                promise.always(function (changed) {
                    expect(promise.state()).to.equal('resolved');
                    expect(changed).to.satisfy(rangesMatcher('B2:E2 B3:B4 E3:E4 B5:E5'));
                    expectCellAttributes('B2:B2', borders('tl', THIN, 'br', NONE));
                    expectCellAttributes('C2:D2', borders('t', THIN, 'blr', NONE));
                    expectCellAttributes('E2:E2', borders('tr', THIN, 'bl', NONE));
                    expectCellAttributes('B3:B4', borders('l', THIN, 'tbr', NONE));
                    expectCellAttributes('E3:E4', borders('r', THIN, 'tbl', NONE));
                    expectCellAttributes('B5:B5', borders('bl', THIN, 'tr', NONE));
                    expectCellAttributes('C5:D5', borders('b', THIN, 'tlr', NONE));
                    expectCellAttributes('E5:E5', borders('br', THIN, 'tl', NONE));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes('B2:E5', borders('tblr', NONE));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                    done();
                });
            });
            it('should merge a new border over an existing border', function (done) {
                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateBorderOperations(generator, ra('E4:F6 C2:C2 C6:C6'), borders('tblr', DBL));
                expect(promise).to.respondTo('promise');
                promise.always(function (changed) {
                    expect(promise.state()).to.equal('resolved');
                    expect(changed).to.satisfy(rangesMatcher('E4:F6 C2:C2 C5:C6'));
                    expectCellAttributes('E4:E4', borders('tl', DBL, 'r', THIN, 'b', NONE));
                    expectCellAttributes('F4:F4', borders('tr', DBL, 'bl', NONE));
                    expectCellAttributes('E5:E5', borders('l', DBL, 'br', THIN, 't', NONE));
                    expectCellAttributes('F5:F5', borders('r', DBL, 'tbl', NONE));
                    expectCellAttributes('E6:E6', borders('bl', DBL, 'tr', NONE));
                    expectCellAttributes('F6:F6', borders('br', DBL, 'tl', NONE));
                    expectCellAttributes('C1:C1', borders('tblr', NONE));
                    expectCellAttributes('C2:C2', borders('tblr', DBL));
                    expectCellAttributes('C5:C5', borders('tblr', NONE));
                    expectCellAttributes('C6:C6', borders('tblr', DBL));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes('E4:E4', borders('r', THIN, 'tbl', NONE));
                    expectCellAttributes('E5:E5', borders('br', THIN, 'tl', NONE));
                    expectCellAttributes('E6:E6', borders('tblr', NONE));
                    expectCellAttributes('F4:F6', borders('tblr', NONE));
                    expectCellAttributes('C2:C2', borders('t', THIN, 'blr', NONE));
                    expectCellAttributes('C5:C5', borders('b', THIN, 'tlr', NONE));
                    expectCellAttributes('C6:C6', borders('tblr', NONE));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                    done();
                });
            });
            it('should fill a range with inner borders', function (done) {
                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateBorderOperations(generator, r('A10:E12'), borders('tblrhv', THIN));
                expect(promise).to.respondTo('promise');
                promise.always(function (changed) {
                    expect(promise.state()).to.equal('resolved');
                    expect(changed).to.satisfy(rangesMatcher('A10:E12'));
                    expectCellAttributes('A10:E12', borders('tblr', THIN));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes('A10:E12', borders('tblr', NONE));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                    done();
                });
            });
            it('should remove adjacent borders when deleting borders', function (done) {
                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateBorderOperations(generator, r('B11:B11'), borders('tblr', NONE));
                expect(promise).to.respondTo('promise');
                promise.always(function (changed) {
                    expect(promise.state()).to.equal('resolved');
                    expect(changed).to.satisfy(rangesMatcher('B10:B10 A11:C11 B12:B12'));
                    expectCellAttributes('B10:B10', borders('tlr', THIN, 'b', NONE));
                    expectCellAttributes('A11:A11', borders('tbl', THIN, 'r', NONE));
                    expectCellAttributes('B11:B11', borders('tblr', NONE));
                    expectCellAttributes('C11:C11', borders('tbr', THIN, 'l', NONE));
                    expectCellAttributes('B12:B12', borders('blr', THIN, 't', NONE));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes('A10:C12', borders('tblr', THIN));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                    done();
                });
            });
            it('should remove adjacent borders when setting different borders', function (done) {
                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateBorderOperations(generator, r('D11:D11'), borders('tblr', DBL));
                expect(promise).to.respondTo('promise');
                promise.always(function (changed) {
                    expect(promise.state()).to.equal('resolved');
                    expect(changed).to.satisfy(rangesMatcher('D10:D10 C11:E11 D12:D12'));
                    expectCellAttributes('D10:D10', borders('tlr', THIN, 'b', NONE));
                    expectCellAttributes('C11:C11', borders('tb', THIN, 'lr', NONE));
                    expectCellAttributes('D11:D11', borders('tblr', DBL));
                    expectCellAttributes('E11:E11', borders('tbr', THIN, 'l', NONE));
                    expectCellAttributes('D12:D12', borders('blr', THIN, 't', NONE));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes('D10:E12', borders('tblr', THIN));
                    expectCellAttributes('C11:C11', borders('tbr', THIN, 'l', NONE));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                    done();
                });
            });
        });

        describe('method "generateVisibleBorderOperations"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateVisibleBorderOperations');
            });
            it('should change border color of existing borders', function (done) {
                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateVisibleBorderOperations(generator, r('B1:D5'), { color: RED });
                expect(promise).to.respondTo('promise');
                promise.always(function (changed) {
                    expect(promise.state()).to.equal('resolved');
                    expect(changed).to.satisfy(rangesMatcher('B2:D2 B3:B5 D5:D5 C6:C6 E4:E5'));
                    expectCellAttributes('B2:B2', borders('tl', THIN_RED, 'br', NONE));
                    expectCellAttributes('C2:C2', borders('tblr', DBL_RED));
                    expectCellAttributes('D2:D2', borders('t', THIN_RED, 'blr', NONE));
                    expectCellAttributes('B3:B4', borders('l', THIN_RED, 'tbr', NONE));
                    expectCellAttributes('B5:B5', borders('bl', THIN_RED, 'tr', NONE));
                    expectCellAttributes('D5:D5', borders('b', THIN_RED, 'tlr', NONE));
                    expectCellAttributes('C6:C6', borders('t', DBL_RED, 'blr', DBL));
                    expectCellAttributes('E4:E4', borders('l', DBL_RED, 't', DBL, 'r', THIN, 'b', NONE));
                    expectCellAttributes('E5:E5', borders('l', DBL_RED, 'br', THIN, 't', NONE));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes('B2:B2', borders('tl', THIN, 'br', NONE));
                    expectCellAttributes('C2:C2', borders('tblr', DBL));
                    expectCellAttributes('D2:D2', borders('t', THIN, 'blr', NONE));
                    expectCellAttributes('B3:B4', borders('l', THIN, 'tbr', NONE));
                    expectCellAttributes('B5:B5', borders('bl', THIN, 'tr', NONE));
                    expectCellAttributes('D5:D5', borders('b', THIN, 'tlr', NONE));
                    expectCellAttributes('C6:C6', borders('tblr', DBL));
                    expectCellAttributes('E4:E4', borders('tl', DBL, 'r', THIN, 'b', NONE));
                    expectCellAttributes('E5:E5', borders('l', DBL, 'br', THIN, 't', NONE));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                    done();
                });
            });
        });

        describe('method "generateMergeCellsOperations"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateMergeCellsOperations');
            });
//            it('should generate correct cell operations', function () {
//            });
        });

        describe('method "generateMoveCellsOperations" 1', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateMoveCellsOperations');
            });
            it('should generate correct cell operations', function (done) {
                sheetModel = docModel.getSheetModel(1);
                cellCollection = sheetModel.getCellCollection();
                expect(cellCollection.getCellFormula(a('A3'), 'op')).to.equal('SUM(A1:D1)');
                expect(cellCollection.getCellFormula(a('B3'), 'op')).to.equal('B6');

                sheetModel = docModel.getSheetModel(2);
                cellCollection = sheetModel.getCellCollection();
                expect(cellCollection.getCellFormula(a('B4'), 'op')).to.equal('Sheet2!C1');

                sheetModel = docModel.getSheetModel(1);
                cellCollection = sheetModel.getCellCollection();

                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateMoveCellsOperations(generator, r('B1:B1048576'), 'right');
                expect(promise).to.respondTo('promise');

                promise.always(function () {

                    expect(promise.state()).to.equal('resolved');
                    expect(cellCollection.getCellFormula(a('A3'), 'op')).to.equal('SUM(A1:E1)');
                    expect(cellCollection.getCellFormula(a('C3'), 'op')).to.equal('C6');

                    sheetModel = docModel.getSheetModel(2);
                    cellCollection = sheetModel.getCellCollection();
                    expect(cellCollection.getCellFormula(a('B4'), 'op')).to.equal('Sheet2!D1');

                    done();
                });

            });
        });

        describe('method "generateMoveCellsOperations" 2', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateMoveCellsOperations');
            });

            it('should generate correct cell operations', function (done) {
                sheetModel = docModel.getSheetModel(3);
                cellCollection = sheetModel.getCellCollection();

                //data validation
                var validationCollection = sheetModel.getValidationCollection();
                expect(validationCollection.getValidationSettings(a('J1')).attributes.value1).to.equal('$B$1:$B$3');

                //name ranges
                expect(sheetModel.getNameCollection().getName('NamedRange_Sheet1').getFormula('op', a('C1'))).to.equal('Sheet4!$B$6:$B$7');
                expect(docModel.getNameCollection().getName('NamedRange_Workbook').getFormula('op', a('C1'))).to.equal('Sheet4!$B$8:$B$9');

                // Chart drawing
                var chartModel = sheetModel.getDrawingCollection().findModel([0]);
                testChartModel(chartModel, { title: 'Sheet4!$L$1', names: 'Sheet4!$K$2:$K$4', values: 'Sheet4!$L$2:$L$4' }, { title: 'Sheet4!$M$1', names: 'Sheet4!$K$2:$K$4', values: 'Sheet4!$M$2:$M$4' });

                // conditional formatting
                var condFormattingProperties = [
                    { colorScale: [{ t: 'formula', v: '$B$1', c: { type: 'rgb', value: 'FF7128' } }, { t: 'formula', v: '$B$2', c: { type: 'rgb', value: 'FFEF9C' } }] },
                    { colorScale: [{ t: 'formula', v: '$B$1', c: { type: 'rgb', value: 'F8696B' } }, { t: 'formula', v: '$B$3', c: { type: 'rgb', value: 'FFEB84' } }, { t: 'formula', v: '$B$2', c: { type: 'rgb', value: '63BE7B' } }] },
                    { dataBar: { r1: { t: 'formula', v: '$B$1' }, r2: { t: 'formula', v: '$B$2' }, c: { type: 'rgb', value: '638EC6' }, s: true } },
                    { value1: '$B$1', value2: '$B$3' },
                    { value1: '$B$2' }
                ];
                testCondFormatting(condFormattingProperties);

                // Move column to right
                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateMoveCellsOperations(generator, r('A1:A1048576'), 'right');
                expect(promise).to.respondTo('promise');

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');
                    //data validation
                    expect(validationCollection.getValidationSettings(a('K1')).attributes.value1).to.equal('$C$1:$C$3');

                    //name ranges
                    expect(sheetModel.getNameCollection().getName('NamedRange_Sheet1').getFormula('op', a('C1'))).to.equal('Sheet4!$C$6:$C$7');
                    expect(docModel.getNameCollection().getName('NamedRange_Workbook').getFormula('op', a('C1'))).to.equal('Sheet4!$C$8:$C$9');

                    // Chart drawing
                    testChartModel(chartModel, { title: 'Sheet4!$M$1', names: 'Sheet4!$L$2:$L$4', values: 'Sheet4!$M$2:$M$4' }, { title: 'Sheet4!$N$1', names: 'Sheet4!$L$2:$L$4', values: 'Sheet4!$N$2:$N$4' });

                    // conditional formatting
                    condFormattingProperties = [
                        { colorScale: [{ t: 'formula', v: '$C$1', c: { type: 'rgb', value: 'FF7128' } }, { t: 'formula', v: '$C$2', c: { type: 'rgb', value: 'FFEF9C' } }] },
                        { colorScale: [{ t: 'formula', v: '$C$1', c: { type: 'rgb', value: 'F8696B' } }, { t: 'formula', v: '$C$3', c: { type: 'rgb', value: 'FFEB84' } }, { t: 'formula', v: '$C$2', c: { type: 'rgb', value: '63BE7B' } }] },
                        { dataBar: { r1: { t: 'formula', v: '$C$1' }, r2: { t: 'formula', v: '$C$2' }, c: { type: 'rgb', value: '638EC6' }, s: true } },
                        { value1: '$C$1', value2: '$C$3' },
                        { value1: '$C$2' }
                    ];
                    testCondFormatting(condFormattingProperties);

                    done();
                });

            });
        });

        describe('method "hasRangeHeadline"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('hasRangeHeadline');
            });
            it('should detect headlines #1', function () {
                var sheet       = 10,

                    range1      = r('A1:D5'),
                    range2      = r('A2:D5');

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.hasRangeHeadline(range1, 'vertical')).to.equal(true);
                expect(cellCollection.hasRangeHeadline(range1, 'horizontal')).to.equal(false);

                expect(cellCollection.hasRangeHeadline(range2, 'vertical')).to.equal(false);
                expect(cellCollection.hasRangeHeadline(range2, 'horizontal')).to.equal(false);
            });
            it('should detect headlines #2', function () {
                var sheet       = 11,

                    range1      = r('A1:D5'),
                    range2      = r('B1:D5');

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                expect(cellCollection.hasRangeHeadline(range1, 'vertical')).to.equal(false);
                expect(cellCollection.hasRangeHeadline(range1, 'horizontal')).to.equal(true);

                expect(cellCollection.hasRangeHeadline(range2, 'vertical')).to.equal(false);
                expect(cellCollection.hasRangeHeadline(range2, 'horizontal')).to.equal(false);
            });
        });

        describe('method "generateSortOperations"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateSortOperations');
            });
            it('should generate correct cell operations', function (done) {
                var sheet       = 10,

                    col_from    = 'A',
                    col_to      = 'D',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C1'),
                    orderBy     = 'toggle',
                    options     = {
                        // direction: null,
                        // headers: null,
                        // colrow: null
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('A2'), val_before: 2,          val_after: 5            },
                        { adr: a('A3'), val_before: 3,          val_after: 2            },
                        { adr: a('A4'), val_before: 4,          val_after: 4            },
                        { adr: a('A5'), val_before: 5,          val_after: 3            },
                        { adr: a('B1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 2            },
                        { adr: a('C1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('C2'), val_before: 2,          val_after: 1            },
                        { adr: a('C3'), val_before: 5,          val_after: 2            },
                        { adr: a('C4'), val_before: 4,          val_after: 4            },
                        { adr: a('C5'), val_before: 1,          val_after: 5            },
                        { adr: a('D1'), val_before: 9,          val_after: 9            },
                        { adr: a('D2'), val_before: 'hier',     val_after: 'bla'        },
                        { adr: a('D3'), val_before: 12,         val_after: 'hier'       },
                        { adr: a('D4'), val_before: 0,          val_after: 0            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 12           }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, orderBy, options);

                expect(promise).to.respondTo('promise');

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_after);
                    });

                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                    done();
                });
            });
            it('should generate correct cell operations #2', function (done) {
                var sheet       = 10,

                    col_from    = 'A',
                    col_to      = 'D',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C1'),
                    orderBy     = 'toggle',
                    options     = {
                        // direction: null,
                        headers: false
                        // colrow: null
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 5            },
                        { adr: a('A2'), val_before: 2,          val_after: 2            },
                        { adr: a('A3'), val_before: 3,          val_after: 4            },
                        { adr: a('A4'), val_before: 4,          val_after: 3            },
                        { adr: a('A5'), val_before: 5,          val_after: 'headline'   },
                        { adr: a('B1'), val_before: 'headline', val_after: 2            },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 'headline'   },
                        { adr: a('C1'), val_before: 'headline', val_after: 1            },
                        { adr: a('C2'), val_before: 2,          val_after: 2            },
                        { adr: a('C3'), val_before: 5,          val_after: 4            },
                        { adr: a('C4'), val_before: 4,          val_after: 5            },
                        { adr: a('C5'), val_before: 1,          val_after: 'headline'   },
                        { adr: a('D1'), val_before: 9,          val_after: 'bla'        },
                        { adr: a('D2'), val_before: 'hier',     val_after: 'hier'       },
                        { adr: a('D3'), val_before: 12,         val_after: 0            },
                        { adr: a('D4'), val_before: 0,          val_after: 12           },
                        { adr: a('D5'), val_before: 'bla',      val_after: 9            }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, orderBy, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
            it('should generate correct cell operations (horizontal) #3', function (done) {
                var sheet       = 10,

                    col_from    = 'A',
                    col_to      = 'D',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C1'),
                    orderBy     = 'ascending',
                    options     = {
                        direction: 'horizontal',
                        headers: false,
                        colrow: 4
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('A2'), val_before: 2,          val_after: 2            },
                        { adr: a('A3'), val_before: 3,          val_after: 5            },
                        { adr: a('A4'), val_before: 4,          val_after: 4            },
                        { adr: a('A5'), val_before: 5,          val_after: 1            },
                        { adr: a('B1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 2            },
                        { adr: a('C1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('C2'), val_before: 2,          val_after: 2            },
                        { adr: a('C3'), val_before: 5,          val_after: 3            },
                        { adr: a('C4'), val_before: 4,          val_after: 4            },
                        { adr: a('C5'), val_before: 1,          val_after: 5            },
                        { adr: a('D1'), val_before: 9,          val_after: 9            },
                        { adr: a('D2'), val_before: 'hier',     val_after: 'hier'       },
                        { adr: a('D3'), val_before: 12,         val_after: 12           },
                        { adr: a('D4'), val_before: 0,          val_after: 0            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 'bla'        }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, orderBy, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
            it('should generate correct cell operations (horizontal) #4', function (done) {
                var sheet       = 10,

                    col_from    = 'A',
                    col_to      = 'D',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('A1'),
                    orderBy     = 'ascending',
                    options     = {
                        direction: 'horizontal',
                        headers: false,
                        colrow: 0
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 9            },
                        { adr: a('A2'), val_before: 2,          val_after: 'hier'       },
                        { adr: a('A3'), val_before: 3,          val_after: 12           },
                        { adr: a('A4'), val_before: 4,          val_after: 0            },
                        { adr: a('A5'), val_before: 5,          val_after: 'bla'        },
                        { adr: a('B1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 3            },
                        { adr: a('B4'), val_before: 2,          val_after: 4            },
                        { adr: a('B5'), val_before: 2,          val_after: 5            },
                        { adr: a('C1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('C2'), val_before: 2,          val_after: 2            },
                        { adr: a('C3'), val_before: 5,          val_after: 2            },
                        { adr: a('C4'), val_before: 4,          val_after: 2            },
                        { adr: a('C5'), val_before: 1,          val_after: 2            },
                        { adr: a('D1'), val_before: 9,          val_after: 'headline'   },
                        { adr: a('D2'), val_before: 'hier',     val_after: 2            },
                        { adr: a('D3'), val_before: 12,         val_after: 5            },
                        { adr: a('D4'), val_before: 0,          val_after: 4            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 1            }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, orderBy, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
        });

        /**
         * Helper function to test the chart model.
         */
        function testChartModel(chartModel, dataSeries1, dataSeries2) {
            var count = 0;
            chartModel.iterateTokenArrays(function (seriesIndex, tokenArrays) {

                _.each(tokenArrays, function (tokenArray, name) {
                    count++;
                    var formula = tokenArray.getFormula('op');
                    var expectFormula = seriesIndex === 0 ? dataSeries1[name] : dataSeries2[name];
                    expect(formula).to.equal(expectFormula);
                });

            });
            expect(count).to.equal(6);
        }

        function testCondFormatting(testProperties) {

            var condFormatCollection = sheetModel.getCondFormatCollection();
            var iterator = condFormatCollection.createModelIterator();

            IteratorUtils.forEach(iterator, function (ruleModel) {
                var rule = ruleModel.getExplicitAttributeSet(true).rule;
                var properties = {};
                switch (rule.type) {
                    case 'colorScale':
                        properties.colorScale = rule.colorScale;
                        break;
                    case 'dataBar':
                        properties.dataBar = rule.dataBar;
                        break;
                    default:
                        if (rule.value1) {
                            properties.value1 = rule.value1;
                        }
                        if (rule.value2) {
                            properties.value2 = rule.value2;
                        }
                }
                expect(properties).to.deep.equal(testProperties[parseInt(ruleModel.getRuleId(), 10)]);
            });
        }

        // Shared Formula Tests
        describe('method "generateMoveCellsOperations"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateMoveCellsOperations');
            });
            it('should generate correct cell operations for shared formulas with references', function (done) {
                sheetModel = docModel.getSheetModel(8);
                cellCollection = sheetModel.getCellCollection();
                expect(cellCollection.getCellFormula(a('B1'), 'op')).to.equal('SUM(A1, $A$1:$A$4)');
                expect(cellCollection.getCellModel(a('B1'))).to.have.a.property('si', 0);
                expect(cellCollection.getCellModel(a('B1')).sr).to.stringifyTo('B1:E1');
                expect(cellCollection.getCellFormula(a('C1'), 'op')).to.equal('SUM(B1, $A$1:$A$4)');
                expect(cellCollection.getCellModel(a('C1'))).to.have.a.property('si', 0);

                // Move the Column A to the right. The Range of the shared formula must be greater.
                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateMoveCellsOperations(generator, r('A1:A1048576'), 'right');
                expect(promise).to.respondTo('promise');

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getCellFormula(a('C1'), 'op')).to.equal('SUM(B1, $B$1:$B$4)');
                    expect(cellCollection.getCellModel(a('C1'))).to.have.a.property('si', 0);
                    expect(cellCollection.getCellModel(a('C1')).sr).to.stringifyTo('C1:F1');

                    expect(cellCollection.getCellFormula(a('D1'), 'op')).to.equal('SUM(C1, $B$1:$B$4)');
                    expect(cellCollection.getCellModel(a('D1'))).to.have.a.property('si', 0);

                    // Change the value of the last cell from the SharedFormula. The Range must be on Column smaller as before
                    var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                    var promise2 = cellCollection.generateCellContentOperations(generator, a('F1'), { v: 42 });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');
                        expect(cellCollection.getCellModel(a('C1')).sr).to.stringifyTo('C1:E1');

                        // Change the value of the first cell from the SharedFormula. The Range must be on Column smaller as before
                        generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                        var promise3 = cellCollection.generateCellContentOperations(generator, a('C1'), { v: 'leer', f: null });

                        promise3.always(function () {
                            expect(promise3.state()).to.equal('resolved');
                            expect(cellCollection.getCellValue(a('C1'))).to.equal('leer');

                            expect(cellCollection.getCellModel(a('D1')).sr).to.stringifyTo('D1:E1');
                            expect(cellCollection.getCellModel(a('D1'))).to.have.a.property('si', 0);
                            expect(cellCollection.getCellModel(a('E1'))).to.have.a.property('si', 0);

                            // Move the Column E, which it's included in the Range, to the right. The SharedFormula should be removed from the sheet.
                            var promise4 = cellCollection.generateMoveCellsOperations(generator, r('E1:E1048576'), 'right');
                            promise4.always(function () {
                                expect(promise4.state()).to.equal('resolved');

                                expect(cellCollection.getCellModel(a('D1'))).to.have.a.property('sr', null);
                                expect(cellCollection.getCellModel(a('D1'))).to.have.a.property('si', null);
                                expect(cellCollection.getCellModel(a('F1'))).to.have.a.property('si', null);

                                done();
                            });
                        });
                    });
                });

            });
            it('should generate correct cell operations for shared formulas without references', function (done) {
                sheetModel = docModel.getSheetModel(9);
                cellCollection = sheetModel.getCellCollection();
                expect(cellCollection.getCellFormula(a('A1'), 'op')).to.equal('SUM(_xlfn.CEILING.MATH(4.9),ABS(0.1))');
                expect(cellCollection.getCellModel(a('A1'))).to.have.a.property('si', 0);
                expect(cellCollection.getCellModel(a('A1')).sr).to.stringifyTo('A1:E10');
                expect(cellCollection.getCellFormula(a('B1'), 'op')).to.equal('SUM(_xlfn.CEILING.MATH(4.9),ABS(0.1))');
                expect(cellCollection.getCellModel(a('B1'))).to.have.a.property('si', 0);

                // Move the Column A to right. The Range of the SharedFormula must be moved one Column to the right.
                var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                var promise = cellCollection.generateMoveCellsOperations(generator, r('A1:A1048576'), 'right');
                expect(promise).to.respondTo('promise');

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    expect(cellCollection.getCellModel(a('B1'))).to.have.a.property('si', 0);
                    expect(cellCollection.getCellModel(a('B1')).sr).to.stringifyTo('B1:F10');

                    expect(cellCollection.getCellModel(a('F10'))).to.have.a.property('si', 0);

                    // Change the Value of the Cell F1. Nothing should happend with the SharedFormula
                    var generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                    var promise2 = cellCollection.generateCellContentOperations(generator, a('F1'), { v: 42 });

                    promise2.always(function () {
                        expect(promise2.state()).to.equal('resolved');
                        expect(cellCollection.getCellValue(a('F1'))).to.equal(42);
                        expect(cellCollection.getCellModel(a('B1')).sr).to.stringifyTo('B1:F10');

                        // Move the Column E to the right. The Range of the Column should greater as before.
                        generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                        var promise3 = cellCollection.generateMoveCellsOperations(generator, r('E1:E1048576'), 'right');

                        promise3.always(function () {
                            expect(promise3.state()).to.equal('resolved');
                            expect(cellCollection.getCellModel(a('B1')).sr).to.stringifyTo('B1:G10');
                            expect(cellCollection.getCellFormula(a('G10'), 'op')).to.equal('SUM(_xlfn.CEILING.MATH(4.9),ABS(0.1))');
                            expect(cellCollection.getCellModel(a('G10'))).to.have.a.property('si', 0);

                            // Move the Row 4 down. The SharedFormula range should greater as before.
                            generator = sheetModel.createOperationsGenerator({ applyImmediately: true });
                            var promise4 = cellCollection.generateMoveCellsOperations(generator, r('A4:XFD4'), 'down');
                            promise4.always(function () {
                                expect(promise4.state()).to.equal('resolved');
                                expect(cellCollection.getCellModel(a('B1')).sr).to.stringifyTo('B1:G11');

                                done();
                            });
                        });
                    });
                });
            });
        });
    });

    // ========================================================================
});
