/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/editframework/app/seleniumwrapper',
    'io.ox/office/textframework/utils/position'
], function (AppHelper, SeleniumWrapper, Position) {

    'use strict';

    // class SeleniumWrapper ==================================================

    describe('EditFramework class SeleniumWrapper', function () {

        it('should exist', function () {
            expect(SeleniumWrapper).to.be.a('object');
        });

        describe('SeleniumWrapper generic methods for Appsuite', function () {
            it('SeleniumWrapper.getAllFilesInFolder should exist', function () {
                expect(SeleniumWrapper.getAllFilesInFolder).to.be.a('function');
            });

            it('SeleniumWrapper.oxLaunch should exist', function () {
                expect(SeleniumWrapper.oxLaunch).to.be.a('function');
            });

            it('SeleniumWrapper.getAllFilesInFolder should exist', function () {
                expect(SeleniumWrapper.getAllFilesInFolder).to.be.a('function');
            });

            it('SeleniumWrapper.clearTrashFolder should exist', function () {
                expect(SeleniumWrapper.clearTrashFolder).to.be.a('function');
            });

            it('SeleniumWrapper.setSettingsStates should exist', function () {
                expect(SeleniumWrapper.setSettingsStates).to.be.a('function');
            });
        });

        describe('SeleniumWrapper methods for OXText', function () {

            var OPERATIONS = [
                { name: 'setDocumentAttributes',
                    attrs: {
                        document:  { defaultTabStop: 1270, zoom: { value: 100 } },
                        page:      { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 },
                        character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' },
                        paragraph: { lineHeight: { type: 'percent', value: 115 }, marginBottom: 352 }
                    }
                },
                { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading1', styleName: 'heading·1',
                    attrs: {
                        character: { bold: true, fontName: 'Times·New·Roman', fontSize: 14, color: { transformations: [{ type: 'shade', value: 74902 }], type: 'scheme', value: 'accent1' } },
                        paragraph: { marginTop: 846, outlineLevel: 0, nextStyleId: 'Normal' }
                    },
                    parent: 'Normal',
                    uiPriority: 9
                },
                { name: 'insertParagraph', start: [0] },
                { name: 'insertText', text: 'Hello World', start: [0, 0] }
            ];

            var transactionIDCounter = 0;

            // initialize test application
            var docModel = null, node = null;
            AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
                docModel = app.getModel();
                node = docModel.getNode();
            });

            function testWrappedFn(fnName, options) {
                var def = $.Deferred();
                var interval = null;

                var transactionID = 'tr' + String(transactionIDCounter);
                transactionIDCounter++;

                SeleniumWrapper[fnName](docModel.getApp(), transactionID, options);

                interval = window.setInterval(function () {
                    var res = SeleniumWrapper.get(this);
                    if (res) {
                        if (res.state) {
                            def.resolve(res.result);
                        } else {
                            def.reject(res.result);
                        }
                        window.clearInterval(interval);
                    }
                }.bind(transactionID), 50);

                return def;
            }

            it('SeleniumWrapper.applyOperations()', function (done) {
                expect(SeleniumWrapper.applyOperations).to.be.a('function');

                var ops = [
                    { name: 'splitParagraph',    start: [0, 11] },
                    { name: 'insertText',        start: [1, 0], text: 'lorem ipsum' }
                ];

                testWrappedFn('applyOperations', { ops: ops }).always(function () {
                    var secondParagraph  = $(Position.getParagraphElement(node, [1]));
                    expect(secondParagraph.text()).to.equal('lorem ipsum');
                    done();
                });

            });

            it('SeleniumWrapper.getAttributesFromTextApp()', function (done) {
                expect(SeleniumWrapper.getAttributesFromTextApp).to.be.a('function');

                docModel.getSelection().setTextSelection([1, 0], [1, 11]);
                docModel.setAttributes('paragraph', { styleId: 'Heading1' }, { clear: true });

                testWrappedFn('getAttributesFromTextApp', { family: 'character' }).always(function (attrs) {
                    expect(attrs.bold).to.equal(true);
                    done();
                });

            });

            it('SeleniumWrapper.setTextSelection()', function (done) {

                expect(SeleniumWrapper.setTextSelection).to.be.a('function');

                var positionOld = [0, 2];
                docModel.getSelection().setTextSelection(positionOld);
                expect(docModel.getSelection().getStartPosition()).to.deep.equal(positionOld);

                var start = [1, 5];
                var end = [1, 6];
                testWrappedFn('setTextSelection', { start: start, end: end }).always(function () {
                    expect(docModel.getSelection().getStartPosition()).to.deep.equal(start);
                    expect(docModel.getSelection().getEndPosition()).to.deep.equal(end);
                    done();
                });
            });
        });

        describe('SeleniumWrapper methods for Spreadsheet', function () {
            //TODO: write tests for NOCE branch

            it('SeleniumWrapper.getAttributesFromSpreadsheetApp should exist', function () {
                expect(SeleniumWrapper.getAttributesFromSpreadsheetApp).to.be.a('function');
            });

            it('SeleniumWrapper.getActiveCellEntry should exist', function () {
                expect(SeleniumWrapper.getActiveCellEntry).to.be.a('function');
            });

            it('SeleniumWrapper.getSheetName should exist', function () {
                expect(SeleniumWrapper.getSheetName).to.be.a('function');
            });
        });

    });

    // ========================================================================
});
