/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/simplemap', [
    'io.ox/office/tk/utils/iteratorutils'
], function (IteratorUtils) {

    'use strict';

    // class SimpleMap ========================================================

    /**
     * A simple helper class that represents an associative map of arbitrary
     * values with a few convenience accessor methods.
     *
     * @constructor
     */
    function SimpleMap() {

        this._map = {};

    } // class SimpleMap

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this map is empty.
     *
     * @returns {Boolean}
     *  Whether this map is empty.
     */
    SimpleMap.prototype.empty = function () {
        return _.isEmpty(this._map);
    };

    /**
     * Removes all elements from this map.
     *
     * @returns {SimpleMap}
     *  A reference to this instance.
     */
    SimpleMap.prototype.clear = function () {
        this._map = {};
        return this;
    };

    /**
     * Creates a shallow or deep clone of this map.
     *
     * @param {Function} [generator]
     *  A generator callback function that can be used to deeply clone the
     *  elements of this map. Receives the following parameters:
     *  (1) {Any} value
     *      The map element to be cloned.
     *  (2) {String} key
     *      The key of the map element.
     *  The return value of this function will be inserted as new map element
     *  into the clone. If omitted, the elements will be copied by reference
     *  into the clone.
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {SimpleMap}
     *  The clone of this map.
     */
    SimpleMap.prototype.clone = function (generator, context) {
        var clone = new SimpleMap();
        clone._map = _.mapObject(this._map, generator || _.identity, context);
        return clone;
    };

    /**
     * Returns whether this map contains an element with the specified key.
     *
     * @param {String} key
     *  The key to be checked.
     *
     * @returns {Boolean}
     *  Whether this map contains an element with the specified key.
     */
    SimpleMap.prototype.has = function (key) {
        return key in this._map;
    };

    /**
     * Returns the map element with the specified key.
     *
     * @param {String} key
     *  The key of the map element to be returned.
     *
     * @param {Any} [def]
     *  The default value that will be returned if this map does not contain an
     *  element with the specified key. If omitted, undefined will be returned.
     *
     * @returns {Any}
     *  The map element with the specified key; or the specified default value,
     *  if the map does not contain such an element.
     */
    SimpleMap.prototype.get = function (key, def) {
        return (key in this._map) ? this._map[key] : def;
    };

    /**
     * Returns any element from this map.
     *
     * @param {type} [def]
     *  The default value that will be returned if this map is empty. If
     *  omitted, undefined will be returned.
     *
     * @returns {Any}
     *  Any map element; or the specified default value, if the map is empty.
     *  Repeated invocations of this method will always return the same value
     *  unless the map has been modified.
     */
    SimpleMap.prototype.getAny = function (def) {
        for (var key in this._map) { return this._map[key]; }
        return def;
    };

    /**
     * Inserts an element into this map. An old existing map element will be
     * overwritten.
     *
     * @param {String} key
     *  The key for the map element to be inserted.
     *
     * @param {Any} value
     *  The map element to be inserted.
     *
     * @returns {Any}
     *  The map element passed to this method, for convenience.
     */
    SimpleMap.prototype.insert = function (key, value) {
        return (this._map[key] = value);
    };

    /**
     * Removes an element from this map.
     *
     * @param {String} key
     *  The key of the map element to be removed.
     *
     * @param {Any} [def]
     *  The default value that will be returned if this map does not contain an
     *  element with the specified key. If omitted, undefined will be returned.
     *
     * @returns {Any}
     *  The map element that has been removed; or the specified default value,
     *  if the map does not contain such an element.
     */
    SimpleMap.prototype.remove = function (key, def) {
        var value = this.get(key, def);
        delete this._map[key];
        return value;
    };

    /**
     * Moves an element in this map to a new key.
     *
     * @param {String} key
     *  The key of the map element to be moved.
     *
     * @param {String} newKey
     *  The new key of the map element.
     *
     * @returns {SimpleMap}
     *  A reference to this instance.
     */
    SimpleMap.prototype.move = function (key, newKey) {
        if (this.has(key)) { this.insert(newKey, this.remove(key)); }
        return this;
    };

    /**
     * Inserts all elements of the passed map into this map.
     *
     * @param {SimpleMap} map
     *  The map to be inserted into this map. Old elements in this map will be
     *  overwritten by the elements of the passed map.
     *
     * @returns {SimpleMap}
     *  A reference to this instance.
     */
    SimpleMap.prototype.merge = function (map) {
        map.forEach(function (value, key) { this._map[key] = value; }, this);
        return this;
    };

    /**
     * Returns the map element with the specified key. If such an element does
     * not exist, the generator callback function will be invoked to create a
     * new map element.
     *
     * @param {String} key
     *  The key of the map element to be returned.
     *
     * @param {Function} [generator]
     *  A generator callback function for missing map elements. Receives the
     *  following parameters:
     *  (1) {String} key
     *      The key of the map element to be generated.
     *  The return value of this function will be inserted as new map element.
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {Any}
     *  The map element for the specified key.
     */
    SimpleMap.prototype.getOrCreate = function (key, generator, context) {
        return (key in this._map) ? this._map[key] : (this._map[key] = generator.call(context, key));
    };

    /**
     * Returns the map element with the specified key. If such an element does
     * not exist, a new element will be created with the passed class
     * constructor.
     *
     * @param {String} key
     *  The key of the map element to be returned.
     *
     * @param {Function} Ctor
     *  A class constructor function for missing map elements. Receives all
     *  following parameters passed to this method.
     *
     * @param {Any} [...]
     *  All following parameters will be passed to the class constructor.
     *
     * @returns {Any}
     *  The map element for the specified key.
     */
    SimpleMap.prototype.getOrConstruct = function (key, Ctor) {
        return (key in this._map) ? this._map[key] : (this._map[key] = (function (args) {

            // performance shortcut for few number of arguments
            switch (args.length) {
                case 2: return new Ctor();
                case 3: return new Ctor(args[2]);
                case 4: return new Ctor(args[2], args[3]);
                case 5: return new Ctor(args[2], args[3], args[4]);
            }

            // construct a new instance with arbitrary number of arguments on the fly
            return (function () {
                function Helper() { return Ctor.apply(this, _.toArray(args).slice(2)); }
                Helper.prototype = Ctor.prototype;
                return new Helper();
            }());
        }(arguments)));
    };

    /**
     * Invokes the passed callback function, if this map contains an element
     * with the specified key.
     *
     * @param {String} key
     *  The key of a map element.
     *
     * @param {Function} callback
     *  The callback function that will be invoked if this map contains an
     *  element with the sepcified key. Receives the following parameters:
     *  (1) {Any} value
     *      The map element with the specified key.
     *
     * @param {Object} context
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The return value of the callback function, if it has been invoked with
     *  an existing map element; otherwise undefined.
     */
    SimpleMap.prototype.with = function (key, callback, context) {
        if (this.has(key)) { return callback.call(context, this._map[key]); }
    };

    /**
     * Returns the keys of all elements contained in this map as array.
     *
     * @returns {Array<String>}
     *  The keys of all elements contained in this map, in no specific order.
     */
    SimpleMap.prototype.keys = function () {
        return Object.keys(this._map);
    };

    /**
     * Returns all elements contained in this map as array.
     *
     * @returns {Array<Any>}
     *  The elements contained in this map, in no specific order.
     */
    SimpleMap.prototype.values = function () {
        return _.values(this._map);
    };

    /**
     * Returns an iterator that visits all elements in this map. The map
     * elements will be visited in no specific order.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the map has been visited completely. No more map
     *      elements are available; this result object does not contain any
     *      other properties!
     *  - {Any} value
     *      The map element currently visited. This property will be omitted,
     *      if the iterator is done (see property 'done').
     *  - {String} key
     *      The key of the map element currently visited. This property will be
     *      omitted, if the iterator is done (see property 'done').
     */
    SimpleMap.prototype.iterator = function () {
        return IteratorUtils.createObjectIterator(this._map);
    };

    /**
     * Invokes the passed callback function for all elements in this map. The
     * map elements will be visited in no specific order.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each map element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {SimpleMap}
     *  A reference to this instance.
     */
    SimpleMap.prototype.forEach = function (callback, context) {
        _.each(this._map, callback, context);
        return this;
    };

    /**
     * Returns whether the predicate callback function returns a truthy value
     * for at least one element in this map. The map elements will be visited
     * in no specific order.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this map until it returns a truthy value. After that, iteration will
     *  be stopped immediately, the remaining map elements will not be visited
     *  anymore. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Boolean}
     *  Whether the predicate callback function returns a truthy value for at
     *  least one element in this map.
     */
    SimpleMap.prototype.some = function (predicate, context) {
        return _.some(this._map, predicate, context);
    };

    /**
     * Returns whether the predicate callback function returns a truthy value
     * for all elements in this map. The map elements will be visited in no
     * specific order.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this map until it returns a falsy value. After that, iteration will
     *  be stopped immediately, the remaining map elements will not be visited
     *  anymore. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Boolean}
     *  Whether the predicate callback function returns a truthy value for all
     *  elements in this map.
     */
    SimpleMap.prototype.every = function (predicate, context) {
        return _.every(this._map, predicate, context);
    };

    /**
     * Invokes the passed callback function for all elements in this map, and
     * returns the return values as plain array. The map elements will be
     * visited in no specific order.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each element in this
     *  map. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *  The return value of the callback function will be inserted into the
     *  resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Array<Any>}
     *  The return values of the callback function, in no specific order.
     */
    SimpleMap.prototype.map = function (callback, context) {
        return _.map(this._map, callback, context);
    };

    /**
     * Invokes the passed predicate callback function for all elements in this
     * map, and returns a plain array with all elements resulting in a truthy
     * return value. The map elements will be visited in no specific order.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this map. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *  If the return value is truthy, the map element will be inserted into
     *  the resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Array<Any>}
     *  The map elements that passed the truth test, in no specific order.
     */
    SimpleMap.prototype.filter = function (predicate, context) {
        return _.filter(this._map, predicate, context);
    };

    /**
     * Invokes the passed predicate callback function for all elements in this
     * map, and returns a plain array with all elements resulting in a falsy
     * return value. The map elements will be visited in no specific order.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this map. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *  If the return value is falsy, the map element will be inserted into the
     *  resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Array<Any>}
     *  The map elements that do not pass the truth test, in no specific order.
     */
    SimpleMap.prototype.reject = function (predicate, context) {
        return _.reject(this._map, predicate, context);
    };

    /**
     * Invokes the passed callback function for all elements in this map, and
     * reduces the values to a single result value. The map elements will be
     * visited in no specific order.
     *
     * @param {Any} initial
     *  The initial result value passed to the first invocation of the callback
     *  function (if the map is empty, this value becomes the final result
     *  value immediately).
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each element in this
     *  map. Receives the following parameters:
     *  (1) {Any} prevValue
     *      The return value of the previous invocation of this callback
     *      function (or the initial value on first invocation).
     *  (2) {Any} value
     *      The value of the current map element.
     *  (3) {String} key
     *      The key of the current map element.
     *  The return value of the callback function will become the new result
     *  value that will be passed as 'prevValue' with the next map element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The final result (the last return value of the callback fucntion).
     */
    SimpleMap.prototype.reduce = function (initial, callback, context) {
        return _.reduce(this._map, callback, initial, context);
    };

    /**
     * Returns the values of a specific property from all object elements in
     * this map. The map elements will be visited in no specific order.
     *
     * @param {String} name
     *  The name of an element property.
     *
     * @returns {Array<Any>}
     *  The values of the specified property from all object elements in this
     *  map, in no specific order.
     */
    SimpleMap.prototype.pluck = function (name) {
        return _.pluck(this._map, name);
    };

    /**
     * Returns the first element in this map that passed a truth test according
     * to the specified predicate. The map elements will be visited in no
     * specific order.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this map until it returns a truthy value. After that, the element
     *  will be returned immediately. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Any}
     *  The first map elements that passes the truth test; or undefined, if no
     *  map element passes the truth test.
     */
    SimpleMap.prototype.find = function (predicate, context) {
        return _.find(this._map, predicate, context);
    };

    // exports ================================================================

    return SimpleMap;

});
