/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/tk/utils/driveutils', [
    'io.ox/files/api',
    'io.ox/core/folder/api',
    'io.ox/core/folder/node',
    'io.ox/core/folder/tree',
    'io.ox/core/tk/selection',
    'io.ox/core/permissions/permissions',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/core/capabilities',
    'settings!io.ox/files'
], function (FilesAPI, FolderAPI, FolderNode, TreeView, Selection, Permissions, Utils, KeyCodes, TriggerObject, Capabilities, DriveSettings) {

    'use strict';

    // static class DriveUtils ================================================

    var ERROR_MAP = {
        general:                { errorForErrorCode: true, error: 'GENERAL_UNKNOWN_ERROR'               },
        'FILE_STORAGE-0026':    { errorForErrorCode: true, error: 'GENERAL_FILE_NOT_FOUND_ERROR.LOAD'   },
        'IFO-0400':             { errorForErrorCode: true, error: 'WARNING_NO_PERMISSION_FOR_DOC'       },
        'IFO-0438':             { errorForErrorCode: true, error: 'GENERAL_FILE_NOT_FOUND_ERROR.LOAD'   }
    };

    // the exported DriveUtils class
    var DriveUtils = {};

    // dummy
    var dummyTree = {
        options: {
            customize: $.noop,
            disable: $.noop,
            filter: $.noop
        }
    };

    function optString(arg) {
        if (_.isNull(arg) || _.isUndefined(arg)) {
            return arg;
        } else {
            return String(arg);
        }
    }

    function optFolder(folder) {
        folder = optString(folder);
        if (!folder) {
            return DriveUtils.getMyFilesId();
        } else {
            return folder;
        }
    }

    function generalFailHandler(error) {
        if (error) {
            Utils.error('DriveUtils fn failed:' + error.error_desc + ' code:"' + error.code + '" error:"' + error.error + '"');

            var realError = ERROR_MAP[error.code];
            if (!realError) {
                realError = ERROR_MAP.general;
            }
            return realError;
        } else {
            Utils.error('DriveUtils fn failed');
        }
    }

    /**
     * instead of moving file in trash this function really deletes the assigned file
     *
     * @param {Object} file is a file-descriptor
     *
     * @returns {Deferred}
     *
     */
    DriveUtils.purgeFile = function (file) {
        return FilesAPI.remove([file], true);
    };

    /**
     * returns true if user is guest or anonymous
     *
     * if user has no MyFiles he is probably a guest
     *
     */
    DriveUtils.isGuest = function () {
        return _.isNull(DriveUtils.getMyFilesId());
    };

    /**
     * @param {String} folderId is the "id" of the folder, you want the content of.
     */
    DriveUtils.getAllFiles = function (folderId) {
        return FilesAPI.getAll(folderId);
    };

    /**
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [version]
     *   When false, no version will be appended. This can be useful for getting the most recent version.
     *   (for more details see function 'getUrl' in Api.js)
     *
     */
    DriveUtils.getFileUrl = function (file, type, options) {
        return FilesAPI.getUrl(file, type, options);
    };

    /**
     * @param {Object} file is a file-descriptor
     *
     *  this can have:
     *       {String} file.folder_id (parent folder)
     *       {String} file.id
     *       {String} file.filename
     *       {Number} file.version
     *       {String} file.source (for example: drive, mail etc)
     *       {String} file.attachment (don't know)
     *
     *  @returns {Deferred}
     *
     */
    DriveUtils.propagateChangeFile = function (file) {
        return FilesAPI.propagate('add:version', file).then(function () {
            return DriveUtils.getFile(file, { cache: false, columns: '1,2,3,5,20,23,108,700,702,703,704,705,707,710,711' });
        });
    };

    /**
     * @param {Object} file is a file-descriptor
     *  this can have:
     *       {String} file.folder_id (parent folder)
     *       {String} file.id
     *
     * @param {Object} options
     *  directly assigned to FilesAPI
     *
     * @returns {Deferred}
     */
    DriveUtils.getFile = function (file, options) {
        return FilesAPI.get(file, options).then(function (fileDesc) {
            return fileDesc;
        }, generalFailHandler);
    };

    /**
     * @returns {Deferred}
     */
    DriveUtils.propagateNewFile = function (file) {
        FilesAPI.propagate('add:file', file);
        return DriveUtils.getFile(file);
    };

    /**
     * returns true if file.source is drive or undefined
     *
     * @returns {Boolean}
     *  true if file is in drive
     */
    DriveUtils.isDriveFile = function (file) {
        if (file && !file.source) {
            Utils.warn('assigned file has no source. please change that.');
            //TODO: use new files-models
        }
        return file && (file.source === 'drive' || !file.source);
    };

    /**
     * returns the corresponding backbone based file model ('io.ox/files/api').
     *
     * already used by file ['io.ox/presenter/main'] > 'start-presentation' > getFileSuccess
     *
     * @returns {Deferred} Deferred with FileAPI.Model as result
     */
    DriveUtils.getFileModelFromDescriptor = function (fileDescriptor) {
        return DriveUtils.getFile(fileDescriptor).then(function (file) {
            var model = FilesAPI.pool.get('detail').get(_.cid(file));
            if (!model) {
                throw new Error('DriveUtils.getFileModelFromDescriptor() model is null ' + fileDescriptor);
            }
            return model.clone();
        });
    };

    // subclasses -------------------------------------------------------------

    /**
     * a generic FolderPicker for own groups of folders
     * the folders real structure is seen as breadcrumb view
     *
     * An instance of this class triggers the following events:
     * - 'change'
     *      After the selected folder has been changed (by user or by 'preselect')
     *      the 'change' event is triggered
     *
     * @extend TriggerObject
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. The following option is supported:
     *
     *  @param {String} [initOptions.preselect]
     *      Drive-File-Id to preselect a folder
     */
    DriveUtils.FlatFolderPicker = TriggerObject.extend({ constructor: function (initOptions) {

        var self = this;

        TriggerObject.call(this);

        var preselect = Utils.getStringOption(initOptions, 'preselect');
        var folderTree = $('<div class="folder-tree visible-selection dropzone" role="navigation" data-dropzones=".selectable">');
        var selectedFolder = null;

        function selectNode() {
            selectedFolder = folderTree.find('.folder.selected').attr('data-id');
            self.trigger('change');
        }

        // methods --------------------------------------------------------

        /**
         * adds a new group of folders
         *
         * @param {String} name
         *  localized String which is shown over the new group of folders
         *
         * @param {Array} folders
         *  Array of folder_ids which will be checked for users' "create" state
         *  (TODO: create as param???) and will be shown as breadcrumb view
         */
        this.addFolders = function (name, folders) {
            if (!folders.length) { return; }

            var treeContainer = $('<ul class="tree-container f6-target" role="tree">');
            treeContainer.appendTo(folderTree);

            var head = DriveUtils.newGroupNode(name);
            head.on('select', selectNode);
            head.render().$el.appendTo(treeContainer);

            _.each(folders, function (id) {
                //DriveUtils.deleteFromCache(id);

                var el = $('<li>');
                el.appendTo(treeContainer);

                DriveUtils.getCreateState(id).always(function (state) {

                    if (state) {
                        var node = DriveUtils.newFolderPickerNode(id);

                        node.on('select', selectNode);

                        DriveUtils.getPath(id).done(function (path) {
                            if (path.length > 2) {
                                var show = path[1].title;
                                for (var i = 2; i < path.length - 0; i++) {
                                    show += ' / ' + path[i].title;
                                }
                                show = DriveUtils.getShortenedTitle(show, 40);

                                node.render().$el.find('.folder-label').text(show);
                            }
                        });

                        el.replaceWith(node.render().$el);

                        if (id === preselect) {
                            node.render().$el.click();
                        }
                        head.render().$el.attr('data-id', treeContainer.find('li[data-id]').first().attr('data-id'));
                    } else {
                        el.remove();
                        if (treeContainer.children().length <= 1) {
                            treeContainer.remove();
                        }
                    }
                });
            });
        };

        /**
         * @returns {jQuery}
         *  The jQuery node for using this object in the DOM.
         */
        this.getNode = function () {
            return folderTree;
        };

        /**
         * @returns {Object}
         *  The current file selection file has folder_id (parentId)
         */
        this.getSelectedFile = function () {
            if (selectedFolder) {
                return { folder_id: selectedFolder };
            }
        };

    } }); // class FlatFolderPicker

    /**
     * with help of ox.TreeView this is a generic FolderPicker
     *
     * An instance of this class triggers the following events:
     * - 'change'
     *      After the selected folder has been changed (by user or by 'preselect')
     *      the 'change' event is triggered
     *
     * @extend TriggerObject
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. The following option is supported:
     *
     *  @param {String} [initOptions.preselect]
     *      Drive-File-Id to preselect a folder
     */
    DriveUtils.FolderPicker = TriggerObject.extend({ constructor: function (initOptions) {

        var self = this;
        var selectedFolder = null;

        TriggerObject.call(this);

        var preselect = Utils.getStringOption(initOptions, 'preselect');

        var folderTree = new TreeView({
            persistent: false,
            tabindex: 0,
            root: '9',
            all: false,
            module: 'infostore',
            abs: false,
            filter: DriveUtils.getPickerFilter(),
            highlightclass: Utils.SMALL_DEVICE ? 'visible-selection-smartphone' : 'visible-selection'
        });

        folderTree.render();
        var treeNode = folderTree.$el;

        // extend filesPane with CoreSelection functionalities, plus keyboard selection support
        Selection.extend(folderTree, treeNode, {});

        treeNode.addClass('drive-folder-list');

        folderTree.on('change virtual', function (folder_id) {
            selectedFolder = folder_id;
            self.trigger('change');
        });

        if (preselect) {
            DriveUtils.preselectFolder(folderTree, preselect);
        }

        // methods -----------------------------------------------------------------

        /**
         * @returns {jQuery}
         *  The jQuery node for using this object in the DOM.
         */
        this.getNode = function () {
            return treeNode;
        };

        /**
         * @returns {Object}
         *  the current file selection file has folder_id (parentId)
         */
        this.getSelectedFile = function () {
            if (selectedFolder) {
                return { folder_id: selectedFolder };
            }
        };

    } }); // class FolderPicker

    // private functions ------------------------------------------------------

    function callState(fnName, state, data) {
        return FolderAPI[fnName](state, data);
    }

    function getState(fnName, state, id) {
        checkId(id);
        return DriveUtils.getPath(id).then(function (list) {
            var last = _.last(list);
            return callState(fnName, state, last);
        }, function () {
            return null;
        });
    }

    function handleFolderPermissions(fnName, state, id) {
        return FolderAPI.get(id, { cache: false }).then(function (data) {

            var result = false;
            _.each(data.permissions, function (perm, index) {
                if (!index) {
                    //owner permission
                    return;
                }
                if (callState(fnName, state, { own_rights: perm.bits })) {
                    result = true;
                }
            });
            return result;
        });
    }

    function checkId(id) {
        if (!_.isString(id)) {
            Utils.error('id is no string:', id);
            throw new Error('missing identifier');
        }
    }

    function bindFolderNodeHandlers(el, node) {
        el.find('.folder-node>.folder-label div').css('overflow', 'visible');

        el.on('click', function () {
            var others = el.parent().parent().find('li.folder.selectable');
            others.removeClass('selected');
            others.attr('aria-selected', false);
            others.attr('tabindex', -1);

            el.addClass('selected');
            el.attr('aria-selected', true);
            el.attr('tabindex', 1);
            el.focus();

            node.trigger('select');
        });
        el.on('keydown', function (evt) {
            var offset = 0;
            if (evt.keyCode === KeyCodes.UP_ARROW) {
                offset--;
            } else if (evt.keyCode === KeyCodes.DOWN_ARROW) {
                offset++;
            } else {
                return;
            }
            var allFolders = el.parent().parent().find('li.folder.selectable:visible');
            var index = null;
            _.find(allFolders, function (folder, idx) {
                if (el[0].isSameNode(folder)) {
                    index = idx;
                    return true;
                }
            });
            $(allFolders[index + offset]).trigger('click');
        });
    }

    // deferred functions -----------------------------------------------------

    /**
     * checks if the assigned Drive-File-ID is the trash-folder
     * or in the trash-folder
     *  used by 'filter' of FolderPicker and its TreeView
     *
     * @param {String} id
     *  the Drive-File-ID of files or folders
     *
     * @returns {Deferred}
     *  resolves with {Boolean} true if it is the trashfolder
     */
    DriveUtils.getTrashState = function (id) {
        return getState('is', 'trash', id);
    };

    /**
     * checks if current user can create files in the folder of the assigned id
     *
     * @param {String} id
     *  the Drive-File-ID of files or folders
     *
     * @returns {Deferred}
     *  resolves with {Boolean} true if user can create files in the folder
     */
    DriveUtils.getCreateState = function (id) {
        return getState('can', 'create', id);
    };

    /**
     * checks if current user can write/edit files in the folder of the assigned id
     *
     * @param {String} id
     *  the Drive-File-ID of files or folders
     *
     * @returns {Deferred}
     *  resolves with {Boolean} true if user can write/edit files in the folder
     */
    DriveUtils.getWriteState = function (id) {
        return getState('can', 'write', id);
    };

    /**
     * checks if current user can edit the assigned file-descriptor
     *
     * @attention assigned folder must be already in use by OX-Drive
     * because this function is syncron and can only communicate with cache of FolderAPI
     *
     * @param {Object} file is a file-descriptor
     *  the Drive-File of files
     *
     * @returns {Boolean}
     *  return true if user can edit current file
     */
    DriveUtils.canModify = function (file) {
        var model = FilesAPI.pool.get('detail').get(_.cid(file));

        if (DriveUtils.isGuest()) {
            var permission = _.findWhere(model.get('object_permissions'), { entity: ox.user_id });
            return Boolean(permission && (permission.bits >= 2));
        } else {
            var folderModel = FolderAPI.pool.models[model.get('folder_id')];
            if (!folderModel) {
                Utils.warn('DriveUtils.canModify() is called to early, folder is not cached yet. better call DriveUtils.getWriteState() file: ' + file);
                return false;
            }
            return callState('can', 'write', folderModel.toJSON());
        }
    };

    DriveUtils.isShareable = function (file) {
        var folderModel = FolderAPI.pool.models[file.folder_id];
        if (!folderModel) {
            Utils.warn('DriveUtils.isShareable() is called to early, folder is not cached yet. better call DriveUtils.getWriteState() file: ' + file);
            return false;
        }
        return folderModel.isShareable();
    };

    /**
     * checks if assigned id is the trash folder
     *
     * @attention assigned folder must be already in use by OX-Drive
     * because this function is syncron and can only communicate with cache of FolderAPI
     *
     * @param {String} id
     *  the Drive-File-ID of files or folders
     *
     * @returns {Boolean}
     *  return true if folder is the trash folder
     */
    DriveUtils.isTrash = function (id) {
        var model = FolderAPI.pool.models[id];
        if (!model) {
            Utils.warn('DriveUtils.isTrash() is called to early, folder is not cached yet. better call DriveUtils.getTrashState() id: ' + id);
            return false;
        }
        return callState('is', 'trash', model.toJSON());
    };

    /**
     * gives the complete path of the assigned Drive-File-ID includes itself
     *
     * @param {String} id
     *  the Drive-File-ID of files or folders
     *
     * @returns {Deferred}
     *  resolves with {Array} path with all parent-folders and assigned file or folder
     */
    DriveUtils.getPath = function (id) {
        checkId(id);
        return FolderAPI.path(id);
    };

    /**
     * checks for all users which have permissions on assigned folder (except the owner)
     * if the have read access, if one has read access, promise is called with true
     *
     * @param {String} id
     *  the Drive-File-ID of files or folders
     *
     * @returns {Deferred}
     *  resolves with {Boolean} if one user has read permission (true), or no user (false)
     */
    DriveUtils.hasFolderReadPermissions = function (id) {
        return handleFolderPermissions('can', 'read', id);
    };

    // normal functions -------------------------------------------------------

    /**
     * return real Drive-File-ID of users documents folder
     * (My Files/Documents)
     *
     * @returns {String}
     *  Drive-File-ID
     */
    DriveUtils.getStandardDocumentsFolderId = function () {
        return optFolder(DriveSettings.get('folder/documents'));
    };

    /**
     * return real Drive-File-ID of users pictures folder
     * (My Files/Pictures)
     *
     * @returns {String}
     *  Drive-File-ID
     */
    DriveUtils.getStandardPicturesFolderId = function () {
        return optFolder(DriveSettings.get('folder/pictures'));
    };

    /**
     * return real Drive-File-ID of users template folder
     * (My Files/Document/Templates)
     *
     * @returns {String}
     *  Drive-File-ID
     */
    DriveUtils.getStandardTemplateFolderId = function () {
        return optFolder(DriveSettings.get('folder/templates'));
    };

    /**
     * return real Drive-File-ID of users trash folder
     * (Trash)
     *
     * @returns {String}
     *  Drive-File-ID
     */
    DriveUtils.getStandardTrashFolderId = function () {
        return optFolder(DriveSettings.get('folder/trash'));
    };

    /**
     * return real Drive-File-ID of users folder
     * (My Files)
     *
     * @returns {String}
     *  Drive-File-ID
     */
    DriveUtils.getMyFilesId = function () {
        return optString(FolderAPI.getDefaultFolder('infostore'));
    };

    /**
     * give a filter function, which is used by FolderPicker and its TreeView
     *
     * @returns {Function}
     *  a function that returns false if assigned model is the trash folder
     */
    DriveUtils.getPickerFilter = function () {
        return function (parent, model) {
            if (callState('is', 'trash', model.toJSON())) { return false; }
        };
    };

    /**
     * gives an extended FolderNode, which look similar like the FolderNode of
     * the FolderPicker and its TreeView.
     * But it is not used for a tree, it is used for a flat view.
     *
     * Reacts on KeyUp and KeyDown to select only one FolderNode at once and iss clickable.
     *
     * @returns {FolderNode}
     *  used by FolderPicker and its TreeView
     */
    DriveUtils.newFolderPickerNode = function (id) {
        var node = new FolderNode({
            indent: true,
            empty: true,
            subfolders: false,
            folder: id,
            tree: dummyTree,
            parent: dummyTree
        });
        var el = node.render().$el;
        el.find('.folder-node').css('pointer-events', 'none');
        bindFolderNodeHandlers(el, node);
        return node;
    };

    /**
     * gives a fake FolderNode, which look similar like the FolderNode of
     * the FolderPicker and its TreeView.
     * But it is not used for a tree, it is used for a flat view.
     *
     * Reacts on KeyUp and KeyDown to select only one FolderNode at once and iss clickable.
     *
     * @returns {Object}
     *  fake FolderNode
     */
    DriveUtils.newGroupNode = function (title) {
        var node = new TriggerObject();
        var el = $('<li class="section folder selectable"><div class="folder-node" role="presentation" style="padding-left: 0px;"><div class="folder-arrow"><i class="fa fa-caret-down"></i></div><div class="folder-icon"><i class="fa fa-fw"></i></div><div class="folder-label" style="font-weight: bold;"><div>' + title + '</div></div></div></li>');
        bindFolderNodeHandlers(el, node);
        var arrow = el.find('.folder-arrow>i');

        function open() {
            el.parent().find('li.folder').show();
            arrow.removeClass('fa-caret-right');
            arrow.addClass('fa-caret-down');
        }

        function close() {
            el.parent().find('li.folder:not(.section)').hide();
            arrow.removeClass('fa-caret-down');
            arrow.addClass('fa-caret-right');
        }

        el.on('keydown', function (evt) {
            if (evt.keyCode === KeyCodes.LEFT_ARROW) {
                close();
            } else if (evt.keyCode === KeyCodes.RIGHT_ARROW) {
                open();
            }
        });

        arrow.parent().on('click', function () {
            if (arrow.hasClass('fa-caret-down')) {
                close();
            } else {
                open();
            }
        });

        node.render = function () {
            return {
                $el: el
            };
        };

        return node;
    };

    /**
     * if the assigned title is longer than assigned length,
     * it returns a new String with "..." in the middle,
     * if not the original String is returned
     *
     * @param {String} title
     *
     * @param {Number} length
     *  max length of the result String
     *
     * @returns {String}
     *  the shortened title if it was longer than assigned length
     */
    DriveUtils.getShortenedTitle = function (title, length) {
        return FolderAPI.getFolderTitle(title, length);
    };

    /**
     * Drive only only fetches once per session the file and folder datas
     * with this function you can clear the cache of a single file or folder,
     * so the next call fetches the data fresh
     *
     * @param {String} id
     *  the Drive-File-ID of files or folders
     */
    DriveUtils.deleteFromCache = function (id) {
        try {
            delete FolderAPI.pool.models[id];
            //on this way i force a refresh of the folder, if its deleted
        } catch (e) {
            Utils.warn('cannot delete FolderAPI.pool.models', e);
        }
    };

    /**
     * show Drive permission dialog for assigned folder
     *
     * @param {String} Drive-File-ID
     *
     * @returns {jQuery.Promise}
     *  A promise for closing the dialog.
     */
    DriveUtils.showPermissionsDialog = function (id) {
        return Permissions.show(id);
    };

    /**
     * listens to TreeViews first appearance
     *
     * if an id is assigned this function fetches its path
     * and iterates through the tree until it finds the correct folder
     * and selects this folder
     *
     * if there is no folder assigned it selects the first folder
     *
     * @param {TreeView} treeView
     *
     * @param {String} id
     *  the Drive-File-ID of files or folders
     */
    DriveUtils.preselectFolder = function (treeView, id) {
        var timeout = null;
        function open() {
            window.clearTimeout(timeout);
            treeView.off('appear', open);
            if (id) {
                var preselect = function (index, path) {
                    var current = path[index].id;
                    var folderEl = treeView.$el.find('.folder[data-id="' + current + '"]');
                    if (folderEl.length) {
                        //listen to appear:id
                        var next = path[index + 1];
                        if (!next) {
                            //done
                            folderEl.click();
                            return;
                        }
                        treeView.once('appear:' + next.id, function () {
                            preselect(index + 1, path);
                        });
                        //simulate click
                        folderEl.find('.folder-arrow').click();
                    }
                };
                DriveUtils.getPath(id).done(function (path) {
                    preselect(1, path);
                });
            } else {
                var selection = treeView.selection;
                if (selection) {
                    selection.updateIndex().selectFirst();
                    treeView.$el.find('.selected').focus();
                }
            }
        }

        //stupid hacks
        treeView.once('appear', open);
        timeout = window.setTimeout(open, 250);
    };

    DriveUtils.preparePath = function (paths) {
        var names = _.pluck(_.rest(paths), 'title');
        _.each(names, function (name, index) {
            names[index] = name.replace(/\s/g, '\xA0');
        });
        var path = _.pluck(_.rest(paths), 'title').join('\xA0/ ');
        return path;
    };

    /**
     * Check if Contacts is enabled.
     *
     * @returns {Boolean}
     *  true if contacts is enabled, otherwise false.
     */
    DriveUtils.isContactsEnabled = function () {
        return  Capabilities.has('contacts');
    };

    // exports ================================================================

    return DriveUtils;

});
