/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/render/canvas', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/render/path'
], function (Utils, BaseObject, Path) {

    'use strict';

    // temporary canvas element used for content relocation
    var relocateCanvas = null;

    // private global functions ===============================================

    /**
     * Returns whether the passed value is a rectangle with a valid size.
     *
     * @param {Any} rectangle
     *  Any value that will be checked.
     *
     * @returns {Boolean}
     *  Whether the passed value is a rectangle with positive width and height.
     */
    function isValidRectangle(rectangle) {
        return _.isObject(rectangle) && (rectangle.width > 0) && (rectangle.height > 0);
    }

    /**
     * Returns a scaled clone of the passed rectangle.
     *
     * @param {Object} rectangle
     *  The rectangle to be scaled.
     *
     * @param {Number} scale
     *  The scaling factor, as floating-point value.
     *
     * @returns {Object}
     *  The scaled rectangle.
     */
    function getScaledRectangle(rectangle, scale) {
        return {
            left: rectangle.left * scale,
            top: rectangle.top * scale,
            width: rectangle.width * scale,
            height: rectangle.height * scale
        };
    }

    // class PathGenerator ====================================================

    /**
     * This class implements the rendering path API of a 2D rendering context
     * of a canvas element. It uses the native methods of the wrapped rendering
     * context, but applies a fixed global translation to all operations. This
     * is needed to workaround the limited floating-point precision provided by
     * the canvas (single-precision instead of double-precision, see bug 35653,
     * and http://stackoverflow.com/a/8874802).
     *
     * @constructor
     *
     * @param {CanvasRenderingContext2D} context
     *  The wrapped rendering context of a canvas element.
     *
     * @param {Number} x0
     *  The global translation on the X axis that will be applied to all path
     *  operations.
     *
     * @param {Number} y0
     *  The global translation on the Y axis that will be applied to all path
     *  operations.
     */
    function PathGenerator(context, x0, y0) {

        // the wrapped rendering context
        this.context = context;
        // the global X translation
        this.x0 = x0;
        // the global Y translation
        this.y0 = y0;

    } // class PathGenerator

    // path API ---------------------------------------------------------------

    PathGenerator.prototype.beginPath = function () {
        this.context.beginPath();
    };

    PathGenerator.prototype.save = function () {
        this.context.save();
    };

    PathGenerator.prototype.restore = function () {
        this.context.restore();
    };

    PathGenerator.prototype.translate = function (dx, dy) {
        this.context.translate(dx, dy);
    };

    PathGenerator.prototype.moveTo = function (x, y) {
        this.context.moveTo(x - this.x0, y - this.y0);
    };

    PathGenerator.prototype.lineTo = function (x, y) {
        this.context.lineTo(x - this.x0, y - this.y0);
    };

    PathGenerator.prototype.arcTo = function (x1, y1, x2, y2, r) {
        this.context.arcTo(x1 - this.x0, y1 - this.y0, x2 - this.x0, y2 - this.y0, r);
    };

    PathGenerator.prototype.quadraticCurveTo = function (cx, cy, x, y) {
        this.context.quadraticCurveTo(cx - this.x0, cy - this.y0, x - this.x0, y - this.y0);
    };

    PathGenerator.prototype.bezierCurveTo = function (cx1, cy1, cx2, cy2, x, y) {
        this.context.bezierCurveTo(cx1 - this.x0, cy1 - this.y0, cx2 - this.x0, cy2 - this.y0, x - this.x0, y - this.y0);
    };

    PathGenerator.prototype.arc = function (x, y, r, a1, a2, ccw) {
        this.context.arc(x - this.x0, y - this.y0, r, a1, a2, ccw);
    };

    PathGenerator.prototype.ellipse = function (x, y, rx, ry, a0, a1, a2, ccw) {
        this.context.ellipse(x - this.x0, y - this.y0, rx, ry, a0, a1, a2, ccw);
    };

    PathGenerator.prototype.closePath = function () {
        this.context.closePath();
    };

    // public methods ---------------------------------------------------------

    /**
     * Generates the passed path in the wrapped rendering context using its
     * native implementation.
     *
     * @param {Path} path
     *  The description of the rendering path to be generated.
     */
    PathGenerator.prototype.generatePath = function (path) {
        this.beginPath();
        path.ops.forEach(function (op) {
            this[op.name].apply(this, op.args);
        }, this);
    };

    // class ContextWrapper ===================================================

    /**
     * This wrapper class provides a higher-level API for a 2D rendering
     * context of a canvas element.
     *
     * @constructor
     *
     * @param {CanvasRenderingContext2D} context
     *  The wrapped rendering context of a canvas element.
     *
     * @param {Number} x0
     *  The global translation on the X axis that will be applied to all path
     *  operations.
     *
     * @param {Number} y0
     *  The global translation on the Y axis that will be applied to all path
     *  operations.
     */
    function ContextWrapper(context, x0, y0) {

        // generator for native paths in the rendering context
        var pathGenerator = new PathGenerator(context, x0, y0);

        // current font settings
        var fontStyle = {
            font: '10pt sans-serif',
            align: 'start',
            baseline: 'alphabetic'
        };

        // private methods ----------------------------------------------------

        /**
         * Sets the current line dash pattern.
         *
         * @param {Number|Array|Null} pattern
         *  The new dash pattern. See method ContextWrapper.setLineStyle() for
         *  details.
         */
        function initLineDash(pattern) {

            // convert parameter to pattern array
            if (_.isNumber(pattern)) {
                pattern = [pattern];
            } else if (!_.isArray(pattern)) {
                pattern = [];
            }

            // initialize line dash
            context.setLineDash(pattern);
        }

        // public methods -----------------------------------------------------

        /**
         * Sets the fill style for rendering the inner area of a path.
         *
         * @param {Object|String} options
         *  Fill style parameters. Omitting a property will not change the
         *  respective style setting.
         *  @param {String} [options.style]
         *      The fill style for rendering the inner fill area (CSS color
         *      descriptor, e.g. as '#RRGGBB' style, or in function style).
         *  For convenience, a simple string can be passed instead of an object
         *  with the property 'style'.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.setFillStyle = function (options) {

            // string shortcut to change the style only
            if (typeof options === 'string') {
                context.fillStyle = options;
                return this;
            }

            // new fill color/style
            if ('style' in options) {
                context.fillStyle = options.style;
            }

            return this;
        };

        /**
         * Sets the stroke style, and other line properties, for rendering the
         * outline of a path.
         *
         * @param {Object} options
         *  Stroke style parameters. Omitting a property will not change the
         *  respective style setting.
         *  @param {String} [options.style]
         *      The line style for rendering line segments of a path (CSS color
         *      descriptor, e.g. as '#RRGGBB' style, or in function style).
         *  @param {Number} [options.width]
         *      The width of the line segments.
         *  @param {Number} [options.cap]
         *      The style of the line end points. Supports all values supported
         *      by the property 'lineCap' of the native rendering context.
         *  @param {Number} [options.join]
         *      The style of the line connection points. Supports all values
         *      supported by the property 'lineJoin' of the native rendering
         *      context.
         *  @param {Array|Number|Null} [options.pattern]
         *      The line dash pattern, as pairs of numbers in a flat array.
         *      Each number in the array MUST be non-negative, and the sum of
         *      all numbers (the length of the pattern) MUST be greater than
         *      zero. The first entry of each pair in this array represents the
         *      length of a visible line segment, the second entry of each pair
         *      represents the length of the gap to the next line segment. By
         *      passing multiple pairs it is possible to specify more complex
         *      dash patterns like dash-dot-dotted lines etc. If the number of
         *      elements in the array is odd, the resulting dash pattern will
         *      be the concatenation of two copies of the array. If set to a
         *      number, uses a simple pattern with equally sized segments and
         *      gaps. If set to null, the current line dash pattern will be
         *      removed, and solid lines will be rendered.
         *  @param {Number} [options.patternOffset]
         *      The initial pattern offset for every subpath in a path stroked
         *      with a dash pattern.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.setLineStyle = function (options) {

            // new line color/style
            if ('style' in options) {
                context.strokeStyle = options.style;
            }

            // new line width
            if ('width' in options) {
                context.lineWidth = options.width;
            }

            // new line cap style
            if ('cap' in options) {
                context.lineCap = options.cap;
            }

            // new line join style
            if ('join' in options) {
                context.lineJoin = options.join;
            }

            // new line dash pattern
            if ('pattern' in options) {
                initLineDash(options.pattern);
            }

            // dash pattern offset
            if ('patternOffset' in options) {
                context.lineDashOffset = options.patternOffset;
            }

            return this;
        };

        /**
         * Sets the font style, and other font properties, for rendering of
         * texts in the canvas.
         *
         * @param {Object} options
         *  Font style parameters. Omitting a property will not change the
         *  respective style setting.
         *  @param {String} [options.font]
         *      The font styles, as expected by the property 'font' of the
         *      canvas rendering context. Initial value is '10pt sans-serif'.
         *  @param {String} [options.align]
         *      The horizontal alignment (accepts all values supported by the
         *      property 'textAlign' of a canvas rendering context).
         *  @param {String} [options.baseline]
         *      The text baseline mode (accepts all values supported by the
         *      property 'textBaseline' of a canvas rendering context).
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.setFontStyle = function (options) {
            _.extend(fontStyle, options);
            context.font = fontStyle.font;
            context.textAlign = fontStyle.align;
            context.textBaseline = fontStyle.baseline;
            return this;
        };

        /**
         * Sets the global opaqueness for all drawing operations (fill, stroke,
         * text, and images).
         *
         * @param {Number} alpha
         *  The opaqueness. The value 1 represents full opacity, the value 0
         *  represents full transparency.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.setGlobalAlpha = function (alpha) {
            context.globalAlpha = alpha;
            return this;
        };

        /**
         * Changes the translation of the coordinate system.
         *
         * @param {Number} dx
         *  The horizontal translation.
         *
         * @param {Number} dy
         *  The vertical translation.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.translate = function (dx, dy) {
            context.translate(dx, dy);
            return this;
        };

        /**
         * Changes the scaling factors of the coordinate system.
         *
         * @param {Number} fx
         *  The horizontal scaling factor.
         *
         * @param {Number} fy
         *  The vertical scaling factor.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.scale = function (fx, fy) {
            context.scale(fx, fy);
            return this;
        };

        /**
         * Saves the current state of the rendering context (all stroke, fill,
         * and text styles, transformation, clipping, etc.), sets up (or adds)
         * a rectangular clipping region, invokes the passed callback function,
         * and restores the saved context state afterwards.
         *
         * @param {Object|Null} rectangle
         *  The clipping rectangle, with the numeric properties 'left', 'top',
         *  'width', and 'height'. For convenience, can be set to null, if no
         *  clipping rectangle is available (conditional clipping without
         *  having to repeat the code executed in the callback).
         *
         * @param {Function} callback
         *  The callback function to be invoked with active clipping region.
         *  The symbol 'this' inside the callback function will be bound to
         *  this instance.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.clip = function (rectangle, callback) {
            if (rectangle) {
                context.save();
                context.beginPath();
                context.rect(rectangle.left - x0, rectangle.top - y0, rectangle.width, rectangle.height);
                context.clip();
            }
            callback.call(this);
            if (rectangle) {
                context.restore();
            }
            return this;
        };

        /**
         * Creates a new empty path object. Path objects are expected by other
         * rendering methods of this canvas wrapper. A caller may create as
         * many path objects as needed at the same time.
         *
         * @returns {Path}
         *  A new empty path object.
         */
        this.createPath = function () {
            return new Path();
        };

        /**
         * Renders the specified path into the wrapped canvas element. First,
         * the path will be filled according to the current fill style (see
         * method ContextWrapper.setFillStyle() for details). Afterwards, the
         * path outline will be rendered according to the current line style
         * (see method ContextWrapper.setLineStyle() for details).
         *
         * @param {Path} path
         *  The path to be rendered.
         *
         * @param {String} [mode='all']
         *  The rendering mode for the path. If specified, MUST be one of the
         *  following values:
         *  - 'all' (default): The path will be filled, and afterwards the
         *      path outline will be rendered.
         *  - 'fill': The path will be filled, but the path outline will NOT be
         *      rendered, regardless of the current line style.
         *  - 'stroke': The path outline will be rendered, but the path will
         *      NOT be filled, regardless of the current fill style.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawPath = function (path, mode) {

            // whether to actually fill and stroke the path
            var fill = mode !== 'stroke';
            var stroke = mode !== 'fill';

            // generate the path in the rendering context, and render the path
            if (fill || stroke) { pathGenerator.generatePath(path); }
            if (fill) { context.fill(); }
            if (stroke) { context.stroke(); }

            return this;
        };

        /**
         * Renders a straight line into the wrapped canvas element.
         *
         * @param {Number} x1
         *  The X coordinate of the starting point of the line.
         *
         * @param {Number} y1
         *  The Y coordinate of the starting point of the line.
         *
         * @param {Number} x2
         *  The X coordinate of the end point of the line.
         *
         * @param {Number} y2
         *  The Y coordinate of the end point of the line.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawLine = function (x1, y1, x2, y2) {
            return this.drawPath(new Path().pushLine(x1, y1, x2, y2), 'stroke');
        };

        /**
         * Renders a rectangle into the wrapped canvas element.
         *
         * @param {Number|Object} x
         *  The X coordinate of the starting point of the rectangle, or a
         *  complete rectangle with the numeric properties 'left', 'top',
         *  'width', and 'height'.
         *
         * @param {Number|String} y
         *  The Y coordinate of the starting point of the rectangle. If the
         *  parameter 'x' is a rectangle object, this parameter will be used as
         *  the optional parameter 'mode' (see below).
         *
         * @param {Number} w
         *  The width of the rectangle. If parameter 'x' is a rectangle object,
         *  this parameter will be ignored.
         *
         * @param {Number} h
         *  The height of the rectangle. If parameter 'x' is a rectangle
         *  object, this parameter will be ignored.
         *
         * @param {String} [mode='all']
         *  The rendering mode for the rectangle. See description of the method
         *  ContextWrapper.drawPath() for details.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawRect = function (x, y, w, h, mode) {
            if (typeof y === 'string') { mode = y; }
            return this.drawPath(new Path().pushRect(x, y, w, h), mode);
        };

        /**
         * Renders a circle into the wrapped canvas element.
         *
         * @param {Number} x
         *  The X coordinate of the center point of the circle.
         *
         * @param {Number} y
         *  The Y coordinate of the center point of the circle.
         *
         * @param {Number} r
         *  The radius of the circle.
         *
         * @param {String} [mode='all']
         *  The rendering mode for the circle. See description of the method
         *  ContextWrapper.drawPath() for details.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawCircle = function (x, y, r, mode) {
            return this.drawPath(new Path().pushCircle(x, y, r), mode);
        };

        /**
         * Renders an ellipse into the wrapped canvas element.
         *
         * @param {Number} x
         *  The X coordinate of the center point of the ellipse.
         *
         * @param {Number} y
         *  The Y coordinate of the center point of the ellipse.
         *
         * @param {Number} rx
         *  The radius of the ellipse on the X axis.
         *
         * @param {Number} ry
         *  The radius of the ellipse on the Y axis.
         *
         * @param {Number} [a=0]
         *  The rotation angle (radiant) of the ellipse.
         *
         * @param {String} [mode='all']
         *  The rendering mode for the circle. See description of the method
         *  ContextWrapper.drawPath() for details.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawEllipse = function (x, y, rx, ry, a, mode) {
            if (typeof a === 'string') { mode = a; a = 0; }
            return this.drawPath(new Path().pushEllipse(x, y, rx, ry, a), mode);
        };

        /**
         * Clears a rectangle (sets all pixels to full transparency) in the
         * wrapped canvas element.
         *
         * @param {Number|Object} x
         *  The X coordinate of the starting point of the rectangle, or a
         *  complete rectangle with the numeric properties 'left', 'top',
         *  'width', and 'height'.
         *
         * @param {Number} y
         *  The Y coordinate of the starting point of the rectangle (ignored,
         *  if parameter 'x' is a rectangle object).
         *
         * @param {Number} w
         *  The width of the rectangle (ignored, if parameter 'x' is a
         *  rectangle object).
         *
         * @param {Number} h
         *  The height of the rectangle (ignored, if parameter 'x' is a
         *  rectangle object).
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.clearRect = function (x, y, w, h) {
            if (_.isObject(x)) { y = x.top; w = x.width; h = x.height; x = x.left; }
            context.clearRect(x - x0, y - y0, w, h);
            return this;
        };

        /**
         * Draws the bitmap contents of the passed image source into the
         * wrapped canvas element.
         *
         * @param {CanvasImageSource|jQuery|Canvas} source
         *  The bitmap source object. Can be any DOM element supporting the DOM
         *  helper type CanvasImageSource (e.g. <img>, <canvas>, or <video>
         *  elements), or a jQuery collection wrapping a DOM element mentioned
         *  before, or an instance of the toolkit class Canvas itself.
         *
         * @param {Number|Object} p1
         *  Either the horizontal target coordinate as floating-point number,
         *  where to draw the entire unscaled source image into this context
         *  (in this case, parameter 'p2' MUST be a numeric coordinate too); or
         *  a rectangle object specifying the target area in this context to
         *  draw the scaled source image to.
         *
         * @param {Number|Object} [p2]
         *  If parameter 'p1' is a numeric coordinate, this parameter MUST BE
         *  the vertical target coordinate as floating-point number, where to
         *  draw the entire unscaled source image into this context. If
         *  parameter 'p1' is a rectangle, this parameter can be set to a
         *  rectangle specifying the partial area in the source image to be
         *  drawn into this context. If omitted, the entire source image will
         *  be scaled into the target rectangle. Note that in difference to the
         *  native method CanvasRenderingContext2D.drawImage(), the target
         *  rectangle (parameter 'p1') ALWAYS precedes the source rectangle
         *  (parameter 'p2')! If the bitmap source object is an instance of the
         *  class Canvas, its global translation and resolution will be taken
         *  into account when interpreting this parameter.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawImage = function (source, p1, p2) {

            // pixel resolution in source canvas
            var sres = 1;
            // global translation in source canvas
            var sx0 = 0, sy0 = 0;

            // convert canvas wrapper or jQuery objects to canvas DOM element
            if (source instanceof Canvas) {
                sres = source.getResolution();
                var sourceRect = source.getRectangle();
                sx0 = sourceRect.left;
                sy0 = sourceRect.top;
                source = source.getNode()[0];
            } else if (source instanceof $) {
                source = source[0];
            }

            if (_.isNumber(p1)) {
                // coordinates as numbers
                context.drawImage(source, p1 - x0, p2 - y0);
            } else if (_.isObject(p2)) {
                // source and target rectangle (native method expects source before target)
                context.drawImage(source, (p2.left - sx0) * sres, (p2.top - sy0) * sres, p2.width * sres, p2.height * sres, p1.left - x0, p1.top - y0, p1.width, p1.height);
            } else {
                // target rectangle only
                context.drawImage(source, p1.left - x0, p1.top - y0, p1.width, p1.height);
            }

            return this;
        };

        /**
         * Renders the specified text into the wrapped canvas element.
         *
         * @param {String} text
         *  The text to be rendered.
         *
         * @param {Number} x
         *  The X coordinate of the starting point of the text.
         *
         * @param {Number} y
         *  The Y coordinate of the starting point of the text.
         *
         * @param {String} [mode='all']
         *  The rendering mode for the text. See description of the method
         *  ContextWrapper.drawPath() for details.
         *
         * @returns {ContextWrapper}
         *  A reference to this instance.
         */
        this.drawText = function (text, x, y, mode) {

            // fill the text, if a fill style has been set
            if (mode !== 'stroke') {
                context.fillText(text, x - x0, y - y0);
            }

            // stroke the text outline, if a stroke style has been set
            if (mode !== 'fill') {
                context.strokeText(text, x - x0, y - y0);
            }

            return this;

        };

        /**
         * Returns the width of the passed string.
         *
         * @param {String} text
         *  The text whose width will be calculated.
         *
         * @returns {Number}
         *  The width of the passed string, as floating-point pixels.
         */
        this.getTextWidth = function (text) {
            return context.measureText(text).width;
        };

        /**
         * Returns the width of the passed single character. To increase
         * precision for different browsers (some browsers return a rounded
         * integral width for single characters), the character will be
         * repeated ten times, and a tenth of the width of the resulting
         * string will be returned.
         *
         * @param {String} char
         *  A single character.
         *
         * @returns {Number}
         *  The width of the passed character, as floating-point pixels.
         */
        this.getCharacterWidth = function (char) {
            return context.measureText(Utils.repeatString(char, 10)).width / 10;
        };

    } // class ContextWrapper

    // class Canvas ===========================================================

    /**
     * A wrapper class for a canvas DOM element with additional convenience
     * methods to render into the canvas.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {HTMLCanvasElement|jQuery} [initOptions.node]
     *      An existing <canvas> element to be wrapped by this instance. The
     *      current size of that canvas will be retained. By default, a new
     *      <canvas> element with zero size will be created automatically.
     *  @param {String} [initOptions.classes]
     *      Additional CSS classes (space-separated list) that will be added to
     *      the passed or created <canvas> element.
     */
    var Canvas = BaseObject.extend({ constructor: function (initOptions) {

        // self reference
        var self = this;

        // the canvas DOM element, as jQuery object
        var $canvas = $(Utils.getOption(initOptions, 'node', '<canvas width="0" height="0">'));

        // the canvas DOM element
        var canvas = $canvas[0];

        // the rendering context of the canvas
        var context = canvas.getContext('2d');

        // additional classes for the canvas element
        var classes = Utils.getStringOption(initOptions, 'classes');

        // the current rectangle represented by the canvas element
        var x0 = 0;
        var y0 = 0;
        var width = canvas.width;
        var height = canvas.height;

        // pixel resolution (physical pixels per canvas pixel in one direction)
        var res = 1;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // public methods -----------------------------------------------------

        /**
         * Returns the wrapped canvas element.
         *
         * @returns {jQuery}
         *  The wrapped canvas DOM element, as jQuery object.
         */
        this.getNode = function () {
            return $canvas;
        };

        /**
         * Returns the pixel resolution used by this canvas (the number of
         * physical pixels in each direction of the coordinate system
         * represented by one pixel offered by the API of this canvas).
         *
         * @returns {Number}
         *  The pixel resolution used by this canvas.
         */
        this.getResolution = function () {
            return res;
        };

        /**
         * Returns the rectangle currently represented by this canvas wrapper.
         *
         * @returns {Object}
         *  The rectangle currently represented by this canvas wrapper.
         */
        this.getRectangle = function () {
            return { left: x0, top: y0, width: width, height: height };
        };

        /**
         * Returns a data URL with the current contents of the canvas.
         *
         * @returns {String}
         *  A data URL with the current contents of the canvas.
         */
        this.getDataURL = function () {
            return canvas.toDataURL();
        };

        /**
         * Returns the current image data object of the entire canvas area.
         *
         * @returns {Array}
         *  The image data, as array of bytes. Each pixel in the canvas is
         *  represented by four array elements (red, green, blue, alpha).
         */
        this.getImageData = function () {
            return context.getImageData(0, 0, width, height).data;
        };

        /**
         * Returns details about a single pixel in the canvas.
         *
         * @param {Number} x
         *  The X coordinate of the pixel.
         *
         * @param {Number} y
         *  The Y coordinate of the pixel.
         *
         * @returns {Object}
         *  A descriptor object with the following properties:
         *  - {Number} r
         *      The value of the red channel, as unsigned 8-bit integer.
         *  - {Number} g
         *      The value of the green channel, as unsigned 8-bit integer.
         *  - {Number} b
         *      The value of the blue channel, as unsigned 8-bit integer.
         *  - {Number} a
         *      The value of the alpha channel, as unsigned 8-bit integer.
         *  - {String} hex
         *      The upper-case hexadecimal RRGGBB value, ignoring the opacity
         *      of the pixel.
         */
        this.getPixelColor = function (x, y) {
            var pixel = context.getImageData(x, y, 1, 1).data;
            var result = { r: pixel[0], g: pixel[1], b: pixel[2], a: pixel[3] };
            result.hex = ('00000' + (result.r * 0x10000 + result.g * 0x100 + result.b).toString(16)).substr(-6).toUpperCase();
            return result;
        };

        /**
         * Changes the size and global translation of the canvas element, and
         * clears all its contents.
         *
         * The global translation will not be set directly at the native canvas
         * element, but will be handled manually. This is needed to workaround
         * the limited floating-point precision provided by the canvas.
         * Browsers usually only implement single-precision instead of
         * double-precision as required by the W3C (see bug 35653, and
         * http://stackoverflow.com/a/8874802).
         *
         * @param {Object} rectangle
         *  The new rectangle represented by the canvas DOM element, with the
         *  following properties:
         *  @param {Number} [rectangle.left=0]
         *      Global horizontal translation, in pixels. A pixel drawn at this
         *      position will appear at the left border of the canvas area.
         *  @param {Number} [rectangle.top=0]
         *      Global vertical translation, in pixels. A pixel drawn at this
         *      position will appear at the top border of the canvas area.
         *  @param {Number} rectangle.width
         *      Width of the bitmap in the canvas area, in pixels. Will also be
         *      set as element size of the <canvas> DOM element.
         *  @param {Number} rectangle.height
         *      Height of the bitmap in the canvas area, in pixels. Will also
         *      be set as element size of the <canvas> DOM element.
         *
         * @param {Number} [resolution=1]
         *  Number of physical pixels in each direction of the coordinate
         *  system represented by one pixel offered by the API of this canvas.
         *
         * @returns {Canvas}
         *  A reference to this instance.
         */
        this.initialize = function (rectangle, resolution) {

            // rescue old settings for performance optimizations
            var oldWidth = width, oldHeight = height, oldRes = res;

            // set global translation
            x0 = Utils.getIntegerOption(rectangle, 'left', 0);
            y0 = Utils.getIntegerOption(rectangle, 'top', 0);
            width = rectangle.width;
            height = rectangle.height;

            // set new pixel resolution
            res = (typeof resolution === 'number') ? resolution : 1;

            // the new internal size of the canvas bitmap, according to pixel resolution
            var canvasWidth = Math.ceil(width * res);
            var canvasHeight = Math.ceil(height * res);
            var changedBitmapSize = (canvas.width !== canvasWidth) || (canvas.height !== canvasHeight);

            // do not touch canvas bitmap size, if it remains the same (may improve performance)
            if (changedBitmapSize) {
                canvas.width = canvasWidth;
                canvas.height = canvasHeight;
            }

            // initialize the visible size of the canvas element
            if ((oldWidth !== width) || (oldHeight !== height)) {
                $canvas.css({ width: width, height: height });
            }

            // initialize global transformation (has been reset after changing canvas size)
            if (changedBitmapSize || (oldRes !== res)) {
                context.setTransform(res, 0, 0, res, 0, 0);
            }

            // always clear all contents
            context.clearRect(0, 0, width, height);
            return this;
        };

        /**
         * Changes the rectangle represented by this canvas element. Moves the
         * existing bitmap contents to the new position, if the current
         * rectangle and the passed rectangle overlap each other.
         *
         * @param {Object} rectangle
         *  The new rectangle represented by this canvas element. See
         *  description of the method Canvas.initialize() for details.
         *
         * @param {Number} [scale=1]
         *  The scaling factor used when copying the old bitmap contents of
         *  this canvas element to the passed target rectangle.
         *
         * @returns {Array}
         *  The positions of all remaining rectangles that could not be filled
         *  with the old contents of the canvas element.
         */
        this.relocate = function (rectangle, scale) {

            // effective scaling factor
            var scaleFactor = (_.isNumber(scale) && (scale > 0)) ? scale : 1;
            // the old rectangle in the source canvas (previous zoom state)
            var sourceOldRectangle = this.getRectangle();
            // the new rectangle in the source canvas (scaled back to the previous zoom state)
            var sourceNewRectangle = getScaledRectangle(rectangle, 1 / scaleFactor);
            // the common part of old and new rectangle in source canvas (previous zoom state)
            var sourceCommonRectangle = Utils.getIntersectionRectangle(sourceOldRectangle, sourceNewRectangle);
            // the old rectangle in target canvas (scaled to the new zoom state)
            var targetOldRectangle = getScaledRectangle(sourceOldRectangle, scaleFactor);
            // the common part of old and new rectangle in target canvas (current zoom state)
            var targetCommonRectangle = Utils.getIntersectionRectangle(targetOldRectangle, rectangle);

            // if possible, copy common areas to secondary canvas, and swap the canvases
            if (isValidRectangle(sourceCommonRectangle) && isValidRectangle(targetCommonRectangle)) {

                // copy bitmap data to temporary canvas (using a temporary canvas element
                // is up to 100 times faster than using getImageData/putImageData)
                relocateCanvas.initialize(targetCommonRectangle, res).render(function (relocContext) {
                    relocContext.drawImage(self, targetCommonRectangle, sourceCommonRectangle);
                });

                // initialize this canvas element, and copy the (already scaled) image data back
                this.initialize(rectangle, res).render(function (thisContext) {
                    thisContext.drawImage(relocateCanvas, targetCommonRectangle);
                });

                // return the remaining empty areas in the canvas
                return Utils.getRemainingRectangles(rectangle, targetOldRectangle);
            }

            // nothing to copy (no common areas)
            this.initialize(rectangle);

            // return the entire area as resulting empty space
            return [rectangle];
        };

        /**
         * Saves the current state of the canvas (colors, transformation,
         * clipping, etc.), invokes the passed callback function, and restores
         * the canvas state afterwards.
         *
         * @param {Object} [clipRect]
         *  An optional clipping rectangle, with the numeric properties 'left',
         *  'top', 'width', and 'height'. Can be omitted completely (instead of
         *  being set to null or undefined).
         *
         * @param {Function} callback
         *  The callback function to be invoked. Receives the following
         *  parameters:
         *  (1) {ContextWrapper} context
         *      The context wrapper of the canvas element, which provides a
         *      more convenient rendering API than the native rendering context
         *      of the canvas, and that adds polyfills for missing features of
         *      the current browser.
         *  (2) {Number} width
         *      The current width of the canvas area, in pixels.
         *  (3) {Number} height
         *      The current height of the canvas area, in pixels.
         *  The symbol 'this' inside the callback function will be bound to
         *  this canvas wrapper instance.
         *
         * @returns {Canvas}
         *  A reference to this instance.
         */
        this.render = function (clipRect, callback) {

            // the context wrapper for this invocation
            var contextWrapper = new ContextWrapper(context, x0, y0);

            // adjust arguments (missing clipping rectangle)
            if (arguments.length === 1) {
                callback = clipRect;
                clipRect = null;
            }

            // apply clipping if specified, invoke callback function
            contextWrapper.clip(clipRect, function () {
                context.save();
                callback.call(this, contextWrapper, width, height);
                context.restore();
            });

            return this;
        };

        // initialization -----------------------------------------------------

        // add CSS class names passed to the constructor
        if (classes) { $canvas.addClass(classes); }

        // add polyfills for missing native CanvasRenderingContext2D methods
        if (!_.isFunction(context.ellipse)) {
            context.ellipse = function (x, y, rx, ry, a0, a1, a2, ccw) {
                this.translate(x, y);
                this.rotate(a0);
                this.scale(rx, ry);
                this.arc(0, 0, 1, a1, a2, ccw);
                this.scale(1 / rx, 1 / ry);
                this.rotate(-a0);
                this.translate(-x, -y);
            };
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            $canvas.remove();
            self = initOptions = $canvas = canvas = context = null;
        });

    } }); // class Canvas

    // global initialization ==================================================

    // create a global canvas object for relocation
    relocateCanvas = new Canvas();

    // exports ================================================================

    return Canvas;

});
