/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/locale/parser', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/utils/arraytemplate',
    'io.ox/office/tk/locale/localedata'
], function (Utils, DateUtils, ArrayTemplate, LocaleData) {

    'use strict';

    // all available text colors in format codes, as name to RGB map
    var TEXT_COLORS = {
        black: '000000',
        blue: '0000FF',
        cyan: '00FFFF',
        green: '00FF00',
        magenta: 'FF00FF',
        red: 'FF0000',
        white: 'FFFFFF',
        yellow: 'FFFF00'
    };

    // A regular expression matching a text color in brackets at the beginning of a string.
    // - Group 1: The name of the matched color, without the bracket characters.
    var TEXT_COLOR_TOKEN_RE = new RegExp('^\\[(' + _.keys(TEXT_COLORS).join('|') + ')\\]', 'i');

    // currency symbols that can appear outside of a currency bracket
    // (TODO: generate from locale data according to some smart filter rules?)
    var LITERAL_CURRENCY_SYMBOLS = ['$', '\xa3', '\u0e3f', '\u20a1', '\u20a9', '\u20aa', '\u20ab', '\u20ac', '\u20ae', '\uffe5'];

    // A regular expression matching a literal currency symbol at the beginning of a string.
    // - Group 1: The matched currency symbol.
    var LITERAL_CURRENCY_RE = new RegExp('^(' + LITERAL_CURRENCY_SYMBOLS.map(_.escapeRegExp).join('|') + ')');

    // A map for all locale correct month-names. Stores pieces of month-names also (like "Janua", or "Dezemb")
    var MONTH_MAP = (function () {
        var returnVal = {};
        _.each(DateUtils.getLocaleMonths(), function (monthString, monthIndex) {
            var // monthName in lower case
                monthName = monthString.toLowerCase(),
                // first three letters (they must match)
                firstLetters        = monthName.substr(0, 3),
                // the remaining letters of the month-name
                remainingLetters    = monthName.substr(3);

            returnVal[monthName] = monthIndex;
            returnVal[firstLetters] = monthIndex;

            // run through all remeining letters of the month
            _.each(remainingLetters, function (letter, index) {
                var indexString = firstLetters + remainingLetters.substr(0, (index + 1));
                // add string with growing month-name
                // ("january" = 'jan' + ('u', 'ua', 'uar', 'uary'))
                returnVal[indexString] = monthIndex;
            });
        });
        return returnVal;
    }());

    // creates and caches the regular expressions to detect currency-formats
    var getCurrencyMatchingRegExp = _.memoize(function (options) {
        var sign = _.escapeRegExp(LocaleData.CURRENCY),
            code = _.escapeRegExp(LocaleData.ISO_CURRENCY);

        if (Utils.getBooleanOption(options, 'leading', true)) {
            // checks a complete string and searches a leading currency-symbol e.g. "€ 213" or "€ string"
            // (means: "[currency][string]")
            return new RegExp('^(' + sign + '|' + code + '])\\s*(.*)$');
        } else {
            // checks only remaining strings like " €" from "12 €"
            // (means: "[space][currency]"")
            return new RegExp('^\\s*(' + sign + '|' + code + ')$');
        }

    }, function (options) {
        return Utils.getBooleanOption(options, 'leading', true);
    });

    // creates and caches the regular expressions to detect date-formats
    var getMonthRegExp = _.memoize(function (monthName, options) {

        function generator(monthName, options) {
            // monthName in lower case
            monthName = monthName.toLowerCase();

            var end             = (Utils.getBooleanOption(options, 'monthGroupEnd', false))      ? '$' : '',
                get             = (Utils.getBooleanOption(options, 'monthGroupGet', true))       ? ''  : '?:',
                mon             = monthName,
                regex           = [],
                letterRegEx     = null,
                index           = 0;

            do {
                mon = mon.substr(0, (monthName.length - index));
                regex[index] = mon;
                index++;
            } while (mon.length > 3);

            letterRegEx = (regex.length > 0) ? '(?:' + regex.join('|') + ')' + end : '';

            // return part of regex
            // e.g. = "(jan(?:uary|uar|ua|u)?$)"
            return '(' + get + letterRegEx + ')';
        }

        // empty month-string means returning all months
        if (monthName === '') {
            var returnVal = [];
            _.each(DateUtils.getLocaleMonths(), function (monthName, i) {
                returnVal[i] = generator(monthName, options);
            });
            return returnVal.join('|');
        } else {
            return generator(monthName, options);
        }
    }, function (monthName, options) {
        var end         = (Utils.getBooleanOption(options, 'monthGroupEnd', false)) ? '$' : '_',
            get         = (Utils.getBooleanOption(options, 'monthGroupGet', true)) ? '_' : '?:';

        return monthName + end + get;
    });

    //all codes for grammar 'op', this means the original english key
    //and all codes for 'ui', this means all translated codes
    var CODES = (function () {

        var op = {};
        var ui = {};

        _.each(LocaleData.TOKENS, function (val, key) {
            op[key] = key;
            ui[key] = val;
        });

        function createRegexp(dec, group, localeCodes) {

            var codes = {};

            codes.dec = dec;
            codes.group = group;

            // date/time category: year
            // /^Y+/i
            codes.expYear = new RegExp('^' + _.escapeRegExp(localeCodes.Y) + '+', 'i');

            // date/time category: month/minute; M... or m...
            // /^M+/i
            codes.expMonthMinute = new RegExp('^' + _.escapeRegExp(localeCodes.M) + '+', 'i');

            // date/time category: month
            // /^M{3,}/i
            codes.expMonthLong = new RegExp('^' + _.escapeRegExp(localeCodes.M) + '{3,}', 'i');

            // date/time category: day/weekday
            // /^D+/i
            codes.expDay = new RegExp('^' + _.escapeRegExp(localeCodes.D) + '+', 'i');

            // date/time category: total number of hours
            // /^\[(h+)\]/i
            codes.expTotalHour = new RegExp('^\\[(' + _.escapeRegExp(localeCodes.h) + '+)\\]', 'i');

            // date/time category: total number of minutes
            // /^\[(m+)\]/
            codes.expTotalMinute = new RegExp('^\\[(' + _.escapeRegExp(localeCodes.m) + '+)\\]', 'i');

            // date/time category: total number of seconds
            // /^\[(s+)\]/i
            codes.expTotalSecond = new RegExp('^\\[(' + _.escapeRegExp(localeCodes.s) + '+)\\]', 'i');

            // date/time category: hour
            // /^h+/i
            codes.expHour = new RegExp('^' + _.escapeRegExp(localeCodes.h) + '+', 'i');

            // date/time category: minute together with seconds
            // /^(m+)(?=\W+s)/i
            codes.expMinuteSecond = new RegExp('^(' + _.escapeRegExp(localeCodes.m) + '+)(?=\\W+' + _.escapeRegExp(localeCodes.s) + ')', 'i');

            // date/time category: second
            // /^s+/i
            codes.expSecond = new RegExp('^' + _.escapeRegExp(localeCodes.s) + '+', 'i');

            // decimal fractional part
            // /^\.[0?#,]+/i
            codes.expSign = new RegExp('^' + _.escapeRegExp(dec) + '[0?#' + _.escapeRegExp(group) + ']+', 'i');

            // integral digits
            // /^,?[0#?]+,?/
            codes.expNr = new RegExp('^' + _.escapeRegExp(group) + '?[0#?]+' + _.escapeRegExp(group) + '?');

            // grouping in integral digits
            // /,/
            codes.expGroup = new RegExp(_.escapeRegExp(group));

            // grouping in integral digits
            // /,/g
            codes.expGroupAll = new RegExp(_.escapeRegExp(group), 'g');

            // date/time category: weekday. Specifically for ODF
            codes.expWeekdayODS = /^N{2,4}/i;

            // date/time category: quarter. Specifically for ODF
            codes.expQuarterODS = /^QQ?/i;

            // date/time category: weeknumber. Specifically for ODF
            codes.expWeeknumberODS = /^WW/i;

            return codes;
        }

        return {
            op: createRegexp('.', ',', op),
            ui: createRegexp(LocaleData.DEC, LocaleData.GROUP, ui)
        };
    }());

    // class FormatToken ======================================================

    /**
     * A parsed token from a number format code.
     *
     * @constructor
     *
     * @property {String} type
     *  The type of this format token.
     *
     * @property {String} text
     *  The text represented by this format token, e.g. for literal text, or
     *  for specific delimiter tokens for numbers or dates.
     *
     * @property {Number} length
     *  The target length of this format token, e.g. for date/time tokens
     *  specifying the type or length of the date/time component.
     */
    function FormatToken(type, text, length) {

        this.type = type;
        this.text = text;
        this.length = length;

    } // class FormatToken

    // class FormatTokenArray =================================================

    /**
     * A typed array with instances of the class FormatToken.
     *
     * @constructor
     */
    var FormatTokenArray = ArrayTemplate.create(FormatToken);

    // class FormatSection ====================================================

    /**
     * Contains the parsed information of a single section in a format code.
     * Sections will be separated with semicolons in the format code and can be
     * used to define different number formats for specific intervals of
     * numbers, e.g. for positive and negative numbers.
     *
     * @constructor
     *
     * @property {String} formatCode
     *  The part of the original format code for this section.
     *
     * @property {FormatTokenArray} tokens
     *  An array with the parsed tokens of this section.
     *
     * @property {String} category
     *  The identifier of the category this format section is associated to.
     *  See property ParsedFormat.category for details.
     *
     * @property {String|Null} colorName
     *  The lower-case name of the text color to be used, if this format
     *  section is used to format a value. The value null does not change the
     *  original text color of the value.
     *
     * @property {String|Null} rgbColor
     *  The RGB value of the text color to be used, if this format section is
     *  used to format a value. The value null does not change the original
     *  text color of the value.
     *
     * @property {String|Null} operator
     *  The operator used to define the number interval covered by this code
     *  section, together with the property 'boundary'. Must be one of '<'
     *  (less than), '<=' (less than or equal), '>' (greater than), '>='
     *  (greater than or equal), '=' (equal), or '<>' (not equal). The value
     *  null represents a missing operator (this format section will match all
     *  numbers).
     *
     * @property {Number} boundary
     *  The boundary value used to define the number interval covered by this
     *  code section, together with the property 'operator'.
     *
     * @property {Boolean} autoMinus
     *  Whether this code section will cause to automatically add a leading
     *  minus sign when formatting negative numbers. Happens if the interval of
     *  this code section covers at least negative numbers, and non-negative
     *  numbers (zero and/or positive numbers).
     *
     * @property {Object} flags
     *  Internal state flags for the format section:
     *  @property {Number} flags.general
     *      Whether this section contains a 'General' format token.
     *  @property {Number} flags.integer
     *      Whether this section contains any format tokens for the integral
     *      part of numbers.
     *  @property {Number} flags.float
     *      Whether this section contains any format tokens for the fractional
     *      part of floating-point numbers.
     *  @property {Number} flags.percent
     *      The number of percent tokens contained in this section.
     *  @property {Boolean} flags.scientific
     *      Whether this section contains an exponent token resulting in
     *      scientific notation.
     *  @property {Boolean} flags.fraction
     *      Whether this section contains a fraction token with numerator and
     *      denominator.
     *  @property {Boolean} flags.currency
     *      Whether this section contains a currency symbol.
     *  @property {Boolean} flags.date
     *      Whether this section contains any date tokens.
     *  @property {Boolean} flags.time
     *      Whether this section contains any time tokens.
     *  @property {Boolean} flags.ampm
     *      Whether this section contains an AM/PM token that switches the time
     *      to 12-hours format.
     *  @property {Boolean} flags.text
     *      Whether this section contains a text token.
     *  @property {Boolean} flags.fill
     *      Whether this section contains a fill token (a character that fills
     *      the remaining space available for the formatted text).
     */
    function FormatSection() {

        this.formatCode = '';
        this.tokens = new FormatTokenArray();
        this.category = 'custom';

        this.colorName = null;
        this.rgbColor = null;

        this.operator = null;
        this.boundary = 0;
        this.autoMinus = true;

        this.flags = {
            general: false,
            integer: false,
            float: false,
            percent: 0,
            scientific: false,
            fraction: false,
            fractionCode: '',
            floatCode: '',
            currency: false,
            date: false,
            time: false,
            ampm: false,
            text: false,
            fill: false,
            group: false,
            ungroupText: ''
        };

    } // class FormatSection

    // public methods ---------------------------------------------------------

    /**
     * Adds a new format token without literal text, and without a specific
     * length to this code section.
     *
     * @param {String} type
     *  The type of the new format token.
     */
    FormatSection.prototype.pushToken = function (type) {
        this.tokens.push(new FormatToken(type, '', 0));
    };

    /**
     * Adds a new format token with literal text to this code section, or
     * extends the text of the last token, if it has the same type.
     *
     * @param {String} type
     *  The type of the new format token.
     *
     * @param {String} text
     *  The text represented by the new format token, e.g. for literal text, or
     *  for specific delimiter tokens for numbers or dates.
     */
    FormatSection.prototype.pushTextToken = function (type, text) {
        var token = this.tokens.last();
        if (token && (token.type === type)) {
            token.text += text;
        } else {
            this.tokens.push(new FormatToken(type, text, 0));
        }
    };

    /**
     * Adds a new literal text token to this code section.
     *
     * @param {String} text
     *  The text literal represented by the new token.
     */
    FormatSection.prototype.pushLiteralToken = function (text) {
        this.pushTextToken('lit', text);
    };

    /**
     * Adds a new date token to this code section.
     *
     * @param {String} type
     *  The exact type of the date token. Must be one of 'year', 'month', or
     *  'day'.
     *
     * @param {Number} length
     *  The length of the date token. The meaning of this value depends on the
     *  passed type:
     *  - 'year': 2 for short years, 4 for full years.
     *  - 'month': 1 for short month number, 2 for month number with leading
     *      zero, 3 for abbreviated month name, 4 for full month name, or 5 for
     *      the first character of the month name.
     *  - 'day': 1 for short day number, 2 for day number with leading zero, 3
     *      for abbreviated name of week day, or 4 for full name of week day.
     */
    FormatSection.prototype.pushDateToken = function (type, length) {
        this.tokens.push(new FormatToken(type, '', length));
        this.flags.date = true;
    };

    /**
     * Adds a new time token to this code section.
     *
     * @param {String} type
     *  The exact type of the time token. Must be one of 'hour', 'minute',
     *  'second', 'hours', 'minutes', or 'seconds' (the latter three types are
     *  for total number of hours, minutes, or seconds contained in a date).
     *
     * @param {Number} length
     *  The length of the time token. The meaning of this value depends on the
     *  passed type:
     *  - 'hour', 'minute', 'second': The length of the time token: 1 for short
     *      numbers, 2 for numbers with leading zero.
     *  - 'hours', 'minutes', 'seconds': The minimum length of the number, to
     *      be filled with leading zeros.
     */
    FormatSection.prototype.pushTimeToken = function (type, length) {
        this.tokens.push(new FormatToken(type, '', length));
        this.flags.time = true;
    };

    /**
     * Adds a new AM/PM token to this code section.
     *
     * @param {Number} length
     *  1, if the token represents the short variant ('a' or 'p'), or 2 for the
     *  long variant ('am' or 'pm').
     */
    FormatSection.prototype.pushAmPmToken = function (length) {
        this.pushTimeToken('ampm', length);
        this.flags.ampm = true;
    };

    /**
     * Sets the text color to be used to render a value formatted with this
     * format section.
     *
     * @param {String} colorName
     *  A valid predefined name of a text color.
     */
    FormatSection.prototype.setColor = function (colorName) {
        if (colorName in TEXT_COLORS) {
            this.colorName = colorName;
            this.rgbColor = TEXT_COLORS[colorName];
        } else {
            Utils.error('FormatSection.setColor(): invalid color "' + colorName + '"');
        }
    };

    /**
     * Changes the interval covered by this code section.
     *
     * @param {String} operator
     *  The operator used to define the number interval covered by this code
     *  section. See description of the class property 'operator' for details.
     *
     * @param {Number} boundary
     *  The boundary value used to define the number interval covered by this
     *  code section. See description of the class property 'boundary' for
     *  details.
     *
     * @param {Boolean} [def=false]
     *  Whether this invocation wants to initialize the default interval of
     *  this code section. If the section already contains a custom interval,
     *  it will not be changed.
     */
    FormatSection.prototype.setInterval = function (operator, boundary, def) {

        // do not change custom intervals, if default flag has been passed
        if (def && this.operator) { return; }

        // initialize 'autoMinus' flag, and check validity of the operator
        switch (operator) {
            case '<':  this.autoMinus = boundary > 0;  break;
            case '<=': this.autoMinus = boundary >= 0; break;
            case '>': // same auto-minus for both operators (e.g.: [>-1] and [>=-1] cover negative and zero, but [>0] and [>=0] do not)
            case '>=': this.autoMinus = boundary < 0;  break;
            case '=':  this.autoMinus = false;         break;
            case '<>': this.autoMinus = true;          break;
            default:
                Utils.error('FormatSection.setInterval(): invalid interval operator "' + operator + '"');
                return;
        }

        // store passed values, if operator is valid
        this.operator = operator;
        this.boundary = boundary;
    };

    /**
     * Returns whether this code section can be used to format the passed
     * number, according to the properties 'operator' and 'boundary'.
     *
     * @param {Number} number
     *  The number to be checked.
     *
     * @returns {Boolean}
     *  Whether this code section can be used to format the passed number.
     */
    FormatSection.prototype.contains = function (number) {
        switch (this.operator) {
            case '<':  return number < this.boundary;
            case '<=': return number <= this.boundary;
            case '>':  return number > this.boundary;
            case '>=': return number >= this.boundary;
            case '=':  return number === this.boundary;
            case '<>': return number !== this.boundary;
        }
        return true; // default for single code section covering all numbers
    };

    // static methods ---------------------------------------------------------

    /**
     * Parses the leading code section contained in the passed format code, and
     * creates a new instance of the class FormatSection if possible.
     *
     * @param {String} fileFormat
     *  The file format identifier specifying which format tokens will be
     *  recognized by the parser.
     *
     * @param {String} grammarId
     *  The identifier of a format grammar.
     *
     * @param {String} formatCode
     *  The number format code to be parsed.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.odf=false]
     *      If it is a ODF files, there a special date formats
     *
     * @returns {Object}
     *  The parser result, with the following properties:
     *  - {FormatSection|Null} section
     *      The new format section instance; or null, if the passed format code
     *      contains an unrecoverable syntax error.
     *  - {String|Null} remainder
     *      The remaining format code without the parsed code section; or null,
     *      if the entire format code has been parsed successfully.
     */
    FormatSection.parse = function (fileFormat, grammarId, formatCode) {

        // special behavior for ODF files
        var odf = fileFormat === 'odf';
        // the format grammar configuration
        var grammarCodes = CODES[grammarId];
        // the current code section
        var section = new FormatSection();
        // the internal flags of the code section
        var flags = section.flags;
        // the matches of the regular expressions
        var matches = null;
        // whether the parser has encountered a syntax error
        var parseError = false;
        // whether the parser has found a section separator
        var hasSeparator = false;
        // whether last date/time token was of type hour
        var lastWasHour = false;

        // rescue the original format code in the section instance
        section.formatCode = formatCode;

        // stop if a code section, or the entire format code has been consumed
        while (!parseError && !hasSeparator && (formatCode.length > 0)) {

            // try/finally to be able to continue the while-loop at any place
            try {

                // code section separator
                if ((matches = /^;/.exec(formatCode))) {
                    hasSeparator = true;
                    section.formatCode = section.formatCode.slice(0, -formatCode.length);
                    continue;
                }

                // 'General' format token
                if ((matches = /^GENERAL/i.exec(formatCode))) {
                    section.pushToken('general');
                    flags.general = true;
                    continue;
                }

                // decimal fractional part
                if ((matches = grammarCodes.expSign.exec(formatCode))) {
                    var ungroup = matches[0].replace(grammarCodes.expGroupAll, '');
                    section.pushTextToken('sign', ungroup);
                    flags.float = true;
                    flags.floatCode = ungroup;
                    continue;
                }

                // fractional notation
                if ((matches = /^[0?#]+\/[0-9?#]+/i.exec(formatCode))) {
                    section.pushTextToken('frac', matches[0]);
                    flags.fraction = true;
                    flags.fractionCode = matches[0];
                    continue;
                }

                // integral digits
                if ((matches = grammarCodes.expNr.exec(formatCode))) {
                    section.pushTextToken('nr', matches[0]);
                    flags.integer = true;

                    if (grammarCodes.expGroup.test(matches[0])) {
                        flags.group = true;
                        flags.ungroupText = matches[0].replace(grammarCodes.expGroupAll, '');
                    } else {
                        flags.ungroupText = String(matches[0]);
                    }

                    continue;
                }

                // scientific notation
                if ((matches = /^E[-+]([0?#]+)/i.exec(formatCode))) {
                    section.pushTextToken('scien', matches[1]);
                    flags.scientific = true;
                    continue;
                }

                // date/time: 12 hours, AM/PM marker
                if ((matches = /^AM\/PM/i.exec(formatCode))) {
                    section.pushAmPmToken(2);
                    continue;
                }

                // date/time: 12 hours, A/P marker
                if ((matches = /^A\/P/i.exec(formatCode))) {
                    section.pushAmPmToken(1);
                    continue;
                }

                // date/time category: year
                if ((matches = grammarCodes.expYear.exec(formatCode))) {
                    section.pushDateToken('year', Math.min(4, Utils.roundUp(matches[0].length, 2))); // only YY and YYYY allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: minute together with seconds
                if ((matches = grammarCodes.expMinuteSecond.exec(formatCode))) {
                    section.pushTimeToken('minute', Math.min(2, matches[1].length)); // m or mm allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: long month (3 signs or more)
                if ((matches = grammarCodes.expMonthLong.exec(formatCode))) {
                    section.pushDateToken('month', Math.min(5, matches[0].length)); // MMM to MMMMM allowed or mmm to mmmmm allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: month/minute; M... or m...
                // M/m or MM/mm with a Hour before or a Second after is always a Minute, in other cases it is a Month
                if ((matches = grammarCodes.expMonthMinute.exec(formatCode))) {
                    var length = matches[0].length;
                    if (lastWasHour && length <= 2) {
                        section.pushTimeToken('minute', Math.min(2, length)); // M or MM allowed in "h:M" or "h:MM"
                    } else {
                        section.pushDateToken('month', Math.min(5, length)); // M to MMMMM allowed
                    }
                    lastWasHour = false;
                    continue;
                }

                // date/time category: day/weekday
                if ((matches = grammarCodes.expDay.exec(formatCode))) {
                    section.pushDateToken('day', Math.min(4, matches[0].length)); // D to DDDD allowed
                    lastWasHour = false;
                    continue;
                }

                if (odf) {
                    // date/time category: weekday
                    if ((matches = grammarCodes.expWeekdayODS.exec(formatCode))) {
                        section.pushDateToken('day', Math.min(4, matches[0].length + 1)); // NN to NNN allowed
                        lastWasHour = false;
                        continue;
                    }

                    // date/time category: quarter of the year
                    if ((matches = grammarCodes.expQuarterODS.exec(formatCode))) {
                        section.pushDateToken('quarter', Math.min(2, matches[0].length)); // Q to QQ allowed
                        continue;
                    }

                    // date/time category: week number
                    if ((matches = grammarCodes.expWeeknumberODS.exec(formatCode))) {
                        section.pushDateToken('weeknumber', matches[0].length); // WW allowed
                        continue;
                    }
                }

                // date/time category: total number of hours
                if ((matches = grammarCodes.expTotalHour.exec(formatCode))) {
                    section.pushTimeToken('hours', matches[1].length); // [h], [hh], [hhh], ... (unlimited length)
                    lastWasHour = true;
                    continue;
                }

                // date/time category: total number of minutes
                if ((matches = grammarCodes.expTotalMinute.exec(formatCode))) {
                    section.pushTimeToken('minutes', matches[1].length); // [m], [mm], [mmm], ... (unlimited length)
                    continue;
                }

                // date/time category: total number of seconds
                if ((matches = grammarCodes.expTotalSecond.exec(formatCode))) {
                    section.pushTimeToken('seconds', matches[1].length); // [s], [ss], [sss], ... (unlimited length)
                    lastWasHour = false;
                    continue;
                }

                // date/time category: hour
                if ((matches = grammarCodes.expHour.exec(formatCode))) {
                    section.pushTimeToken('hour', Math.min(2, matches[0].length)); // h or hh allowed
                    lastWasHour = true;
                    continue;
                }

                // date/time category: second
                if ((matches = grammarCodes.expSecond.exec(formatCode))) {
                    section.pushTimeToken('second', Math.min(2, matches[0].length)); // s or ss allowed
                    lastWasHour = false;
                    continue;
                }

                // percent sign
                if ((matches = /^%/.exec(formatCode))) {
                    section.pushLiteralToken('%');
                    flags.percent += 1;
                    continue;
                }

                // string literal, enclosed in double-quotes
                if ((matches = /^"([^"]*)"/.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    continue;
                }

                // character literal, preceded by backspace
                if ((matches = /^\\(.)/.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    continue;
                }

                // whitespace (TODO: store and handle exact character width)
                if ((matches = /^_./.exec(formatCode))) {
                    section.pushLiteralToken(' ');
                    continue;
                }

                // repeated fill character
                if ((matches = /^\*(.)/.exec(formatCode))) {
                    // subsequent fill characters will be inserted once literally
                    // (only the first fill character in the section will be expanded)
                    if (flags.fill) {
                        section.pushLiteralToken(matches[1]);
                    } else {
                        section.pushTextToken('fill', matches[1]);
                        flags.fill = true;
                    }
                    continue;
                }

                // number interval
                if ((matches = /^\[(<=?|>=?|<>|=)([^[\];]+)\]/.exec(formatCode))) {
                    var parseResult = Parser.parseLeadingNumber(matches[2], { dec: grammarCodes.dec, sign: true, complete: true });
                    if (parseResult && (parseResult.remaining.length === 0)) {
                        section.setInterval(matches[1], parseResult.number);
                    } else {
                        parseError = true;
                    }
                    continue;
                }

                // text color
                if ((matches = TEXT_COLOR_TOKEN_RE.exec(formatCode))) {
                    section.setColor(matches[1].toLowerCase());
                    continue;
                }

                // locale data with hexadecimal LCID and calendar information
                if ((matches = /^\[\$-([0-9a-f]+)\]/i.exec(formatCode))) {
                    continue;
                }

                // currency symbol with hexadecimal LCID
                if ((matches = /^\[\$([^[\];-]+)-([0-9a-f]+)\]/i.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    flags.currency = true;
                    continue;
                }

                // currency symbol without hexadecimal LCID
                if ((matches = /^\[\$([^[\];-]+)\]/i.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    flags.currency = true;
                    continue;
                }

                // literal currency symbol not enclosed in brackets
                if ((matches = LITERAL_CURRENCY_RE.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    flags.currency = true;
                    continue;
                }

                // ignore other unsupported specifiers
                if ((matches = /^(\[[^[\];]*\])/.exec(formatCode))) {
                    Utils.warn('FormatSection.parse(): unsupported section specifier "' + matches[1] + '"');
                    continue;
                }

                // placeholder for text value
                if ((matches = /^@/.exec(formatCode))) {
                    section.pushToken('text');
                    flags.text = true;
                    continue;
                }

                // unmatched double-quote, unmatched opening bracket, or incomplete escape sequence: syntax error
                if (/^["\[\\_*]/.test(formatCode)) {
                    parseError = true;
                    continue;
                }

                // any non-matching token is considered a character literal
                matches = [formatCode[0]];
                section.pushLiteralToken(matches[0]);

            } finally {
                // remove the current match from the formula string
                formatCode = formatCode.slice(matches[0].length);
            }
        }

        // resolve the category identifier
        if (!parseError) {
            if (flags.text) {
                section.category = 'text';
                // do not allow any numeric tokens (currency symbols are allowed)
                parseError = flags.general || flags.integer || flags.float || flags.percent || flags.scientific || flags.fraction || flags.date || flags.time;
            } else if (flags.general && (section.tokens.length === 1)) {
                // standard format code (optionally with color)
                section.category = 'standard';
            } else if (!flags.general && (flags.integer || flags.float) && !flags.date && !flags.time) {
                // numeric format codes
                if (!flags.percent && !flags.scientific && !flags.fraction && !flags.currency) {
                    section.category = 'number';
                } else if (flags.percent && !flags.scientific && !flags.fraction && !flags.currency) {
                    section.category = 'percent';
                } else if (!flags.percent && flags.scientific && !flags.fraction && !flags.currency) {
                    section.category = 'scientific';
                } else if (!flags.percent && !flags.scientific && flags.fraction && flags.integer && !flags.float && !flags.currency) {
                    section.category = 'fraction';
                } else if (!flags.percent && !flags.scientific && !flags.fraction && flags.currency) {
                    section.category = 'currency';
                }
            } else if (!flags.general && !flags.integer && !flags.percent && !flags.scientific && !flags.fraction) {
                // date/time format codes
                if (flags.date && !flags.time && !flags.float) {
                    section.category = 'date';
                } else if (!flags.date && flags.time) {
                    section.category = 'time';
                } else if (flags.date && flags.time) {
                    section.category = 'datetime';
                }
            }
        }

        // create and return the parser result
        return {
            section: parseError ? null : section,
            remainder: hasSeparator ? formatCode : null
        };
    };

    // class FormatSectionArray ===============================================

    /**
     * A typed array with instances of the class FormatSection.
     *
     * @constructor
     */
    var FormatSectionArray = ArrayTemplate.create(FormatSection);

    // class ParsedFormat =====================================================

    /**
     * Represents a parsed number format code.
     *
     * @constructor
     *
     * @property {String} formatCode
     *  The original format code this instance has been created from.
     *
     * @property {FormatSectionArray} numberSections
     *  An array of format code sections for different number intervals.
     *
     * @property {FormatSection} textSection
     *  A dedicated code section to format string values.
     *
     * @property {Boolean} syntaxError
     *  Whether the format code that has been used to create this instance
     *  contains a syntax error.
     *
     * @property {String} category
     *  The identifier of the category this format code is associated to. Will
     *  contain one of the following values:
     *  - 'standard': Reserved for the standard format code 'General'.
     *  - 'number': All numeric format codes without special appearance.
     *  - 'scientific': All numeric format codes with scientific notation.
     *  - 'percent': All numeric format codes shown as percentage.
     *  - 'fraction': All numeric format codes shown as fractions.
     *  - 'currency': All numeric format codes with a currency symbol.
     *  - 'date': All format codes shown as a date (without time).
     *  - 'time': All format codes shown as a time (without date).
     *  - 'datetime': All format codes shown as combined date and time.
     *  - 'text': All format codes containing a text section only.
     *  - 'custom': All format codes not fitting into any other category.
     */
    function ParsedFormat(formatCode) {

        this.formatCode = formatCode;
        this.numberSections = new FormatSectionArray();
        this.textSection = null;
        this.syntaxError = false;
        this.category = 'custom';

    } // class ParsedFormat

    // public methods ---------------------------------------------------------

    /**
     * Parses the leading code section contained in the passed format code, and
     * inserts a new instance of the class FormatSection into this object if
     * possible. If the passed format code contains a syntax error, the public
     * flag 'syntaxError' will be set to true.
     *
     * @param {String} fileFormat
     *  The file format identifier specifying which format tokens will be
     *  recognized by the parser.
     *
     * @param {String} grammarId
     *  The identifier of a format grammar.
     *
     * @param {String} formatCode
     *  The number format code to be parsed.
     *
     * @returns {String|Null}
     *  The remaining format code without the parsed code section; or null, if
     *  the entire format code has been parsed successfully.
     */
    ParsedFormat.prototype.parseSection = function (fileFormat, grammarId, formatCode) {

        // result of the section parser (new code section, and remaining format code)
        var parseResult = FormatSection.parse(fileFormat, grammarId, formatCode);

        // missing code section indicates syntax error
        if (!parseResult.section) {
            this.syntaxError = true;
            return null;
        }

        // there can only be a single text sections at the end of the format code
        if (this.textSection) {
            this.syntaxError = true;
            return null;
        }

        // store the new code section in this instance
        if (parseResult.section.category === 'text') {
            this.textSection = parseResult.section;
        } else {
            this.numberSections.push(parseResult.section);
        }

        // return the remaining format code
        return parseResult.remainder;
    };

    /**
     * Returns an existing code section, according to the passed value.
     *
     * @param {Any} value
     *  The value used to pick a code section. For finite numbers, one of the
     *  numeric sections will be returned. For strings, the text section will
     *  be returned. For all other values, null will be returned.
     *
     * @returns {FormatSection|Null}
     *  The format code section for the passed value, if available; otherwise
     *  null.
     */
    ParsedFormat.prototype.getSection = function (value) {

        // do not return sections, if the format code contains a syntax error
        if (this.syntaxError) { return null; }

        // return text section for strings
        if (_.isString(value)) { return this.textSection; }

        // reject other values than finite numbers
        if (!_.isNumber(value) || !isFinite(value)) { return null; }

        // pick first matching section for numbers
        return this.numberSections.find(function (s) { return s.contains(value); });
    };

    /**
     * Returns whether this parsed format is the standard format code (category
     * is set to 'standard').
     *
     * @returns {Boolean}
     *  Whether this parsed format is the standard format code.
     */
    ParsedFormat.prototype.isStandard = function () {
        return this.category === 'standard';
    };

    /**
     * Returns whether this parsed format is in one of the categories 'date',
     * or 'datetime'.
     *
     * @returns {Boolean}
     *  Whether this parsed format is in a date category.
     */
    ParsedFormat.prototype.isAnyDate = function () {
        return (/^date(time)?$/).test(this.category);
    };

    /**
     * Returns whether this parsed format is in one of the categories 'time',
     * or 'datetime'.
     *
     * @returns {Boolean}
     *  Whether this parsed format is in a time category.
     */
    ParsedFormat.prototype.isAnyTime = function () {
        return (/^(date)?time$/).test(this.category);
    };

    /**
     * Returns whether this parsed format is in one of the categories 'date',
     * 'time', or 'datetime'.
     *
     * @returns {Boolean}
     *  Whether this parsed format is in one of the date/time categories.
     */
    ParsedFormat.prototype.isAnyDateTime = function () {
        return (/^(date|time|datetime)$/).test(this.category);
    };

    /**
     * Returns whether this parsed format has the same category as the passed
     * parsed format.
     *
     * @param {ParsedFormat} parsedFormat
     *  The parsed format whose category will be compared with the category of
     *  this instance.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.anyDateTime=false]
     *      If set to true, the categories 'date', 'time', and 'datetime' are
     *      considered to be equal.
     *
     * @returns {Boolean}
     *  Whether this and the other parsed format have the same category.
     */
    ParsedFormat.prototype.hasEqualCategory = function (parsedFormat, options) {

        // immediately return for exact category matches
        if (this.category === parsedFormat.category) { return true; }

        // ignore differences in date/time categories if specified
        var anyDateTime = Utils.getBooleanOption(options, 'anyDateTime', false);
        return anyDateTime && this.isAnyDateTime() && parsedFormat.isAnyDateTime();
    };

    // static class Parser ====================================================

    /**
     * A static class providing methods to parse numbers, dates, format codes,
     * and other locale-dependent source data.
     */
    var Parser = {};

    // constants --------------------------------------------------------------

    /**
     * The format token name for the standard number format.
     *
     * @constant
     */
    Parser.GENERAL = 'GENERAL';

    // static methods ---------------------------------------------------------

    /**
     * Converts the passed number to a string, using the decimal separator of
     * the current UI locale. This is a simple method that uses JavaScript's
     * conversion from numbers to strings.
     *
     * @param {Number} number
     *  The number to be converted.
     *
     * @returns {String}
     *  The passed number, converted to string, with the decimal separator of
     *  the current UI locale.
     */
    Parser.numberToString = function (number) {
        return String(number).replace(/\./g, LocaleData.DEC);
    };

    /**
     * Converts the passed string to a number, using the decimal separator of
     * the current UI locale. This is a simple method that uses JavaScript's
     * parseFloat() conversion function.
     *
     * @param {String} text
     *  The text to be converted to a number.
     *
     * @returns {Number}
     *  The passed number, converted to string. If the passed text does not
     *  represent a floating-point number, NaN will be returned. If the text
     *  consists of a number with following garbage (e.g. '123abc'), the
     *  leading number will be returned (same behavior as JavaScript's native
     *  parseFloat() function).
     */
    Parser.stringToNumber = function (text) {
        return parseFloat(text.replace(LocaleData.DEC, '.'));
    };

    /**
     * Tries to find a currency expression from the following part of the
     * passed string.
     *
     * @param {String} text
     *  The text potentially ends with a currency expression.
     *
     * @returns {Boolean}
     *  If a currency expression found at the end, return true. Otherwise false.
     */
    Parser.checkFollowingCurrency = (function () {
        var currencyTest = getCurrencyMatchingRegExp({ leading: false });

        function parseFollowingCurrency(text) {
            return (currencyTest.exec(text) !== null);
        }

        return parseFollowingCurrency;
    }());

    Parser.parseLeadingCurrency = (function () {
        return function (string, options) {
            var sign = Utils.getStringOption(options, 'sign', LocaleData.CURRENCY),
                code = Utils.getStringOption(options, 'code', LocaleData.ISO_CURRENCY),
                regExSign = _.escapeRegExp(sign),
                regExCode = _.escapeRegExp(code),
                regEx = new RegExp('^([-+]*)\\s*(' + regExSign + '|' + regExCode + ')\\s*(.+)$'),
                result = regEx.exec(string);

            return result ? [result[0], result[1] + result[3], result[2]] : null;
        };
    }());

    Parser.parseLeadingPercent = (function () {
        var regEx = /^([+-]?)\s*%\s*(.+)$/;
        return function (string) {
            var result = regEx.exec(string);
            return result ? [result[0], result[1] + result[2]] : null;
        };
    }());

    Parser.parseBrackets = (function () {
        var getBracketsRegExp = _.memoize(function (dec, cur) {
            dec = _.escapeRegExp(dec);
            cur = _.escapeRegExp(cur);
            var r =  new RegExp('^([' + cur + '%]?)\\s*(?:\\()\\s*([' + cur + '%]?)\\s*(\\d*' + dec + '\\d+|\\d+' + dec + '?)(.*)(?:\\))\\s*([' + cur + '%]?)$', 'i');
            return r;
        }, function (dec, cur) {
            return dec + cur;
        });

        return function (text, options) {
            var dec = Utils.getStringOption(options, 'dec', LocaleData.DEC),
                cur = Utils.getStringOption(options, 'cur', LocaleData.CURRENCY),
                result = getBracketsRegExp(dec, cur).exec(text);

            if (result === null) {
                return text;
            } else {
                return [result[1], result[2], '-', result[3], result[4], result[5]].join('');
            }
        };
    }());

    /**
     * Tries to parse a floating-point number from the leading part of the
     * passed string.
     *
     * @param {String} text
     *  The text potentially starting with a floating-point number.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.dec=LocaleData.DEC]
     *      If specified, a custom decimal separator character. By default, the
     *      decimal separator of the current UI language will be used.
     *  @param {Boolean} [options.sign=false]
     *      If set to true, the passed string may contain a leading sign
     *      character ('-' or '+').
     *  @param {Boolean|String} [options.group=false]
     *      If set to true, the group separator of the current UI language will
     *      be accepted in the parsed text. If set to false (or omitted), group
     *      separators will not be accepted at all. Additionally, this option
     *      can explicitly be set to a custom group separator character.
     *  @param {Boolean} [options.complete=false]
     *      If set to true, the entire passed string must represent a valid
     *      floating-point number. No remaining 'garbage' text will be
     *      accepted.
     *
     * @returns {Object|Null}
     *  A result descriptor, if the passed string starts with a valid floating-
     *  point number (according to the passed options); otherwise null. The
     *  result descriptor contains the following properties:
     *  - {Number} number
     *      The parsed floating-point number.
     *  - {String} text
     *      The leading text from the passed string that has been parsed to the
     *      floating-point number.
     *  - {String} sign
     *      The sign character. Will be the empty string, if no sign exists in
     *      the passed text, or if the option 'sign' has not been set
     *      (see above).
     *  - {Boolean} dec
     *      Whether the number in the parsed text contains a decimal separator.
     *  - {Boolean} scientific
     *      Whether the passed text contains a number in scientific notation
     *      (true), or in decimal notation (false).
     *  - {String} remaining
     *      The remaining characters from the passed string following the
     *      parsed floating-point number. Will be the empty string, if the
     *      option 'complete' has been set (see above).
     */
    Parser.parseLeadingNumber = (function () {

        // returns a regular expression used to match a floating-point number at the beginning of a string
        // - Group 1: The leading sign character (may be an empty string).
        // - Group 2: The floating-point number (decimal or scientific), without the sign character.
        // - Group 3: The integral exponent as string, if the number was given in scientific notation; or
        //      undefined, if the number was given in decimal notation.
        var getNumberRegExp = _.memoize(function (dec, group) {
            var intPattern = (group ? '\\d*(?:' + _.escapeRegExp(group) + '\\d{3})*' : '\\d*') + _.escapeRegExp(dec);
            return new RegExp('^([-+]?)((?:' + intPattern + '\\d+|' + intPattern + '?)(?:E([-+]?\\d+))?)', 'i');
        }, function (dec, group) {
            return dec + group;
        });

        var getGrpRegEx = _.memoize(function (group) {
            return new RegExp(_.escapeRegExp(group), 'g');
        });

        // the actual parseLeadingNumber() method
        function parseLeadingNumber(text, options) {

            // the decimal separator used to parse the number
            var dec = Utils.getStringOption(options, 'dec', LocaleData.DEC);
            var group = Utils.getBooleanOption(options, 'group', false) ? LocaleData.GROUP : Utils.getStringOption(options, 'group', '');

            // try to parse the passed string for a leading floating-point number
            var matches = getNumberRegExp(dec, group).exec(text);

            // reject leading sign unless specified in the passed options
            var sign = Utils.getBooleanOption(options, 'sign', false);
            if (!matches || (!sign && matches[1].length > 0)) { return null; }

            // the complete text of the parsed number (matched by the regular expression)
            var matchedText = matches[0];
            // the matched sign character
            var signText = matches[1];
            // the matched absolute number (with scientific exponent) as text
            var numberText = matches[2];

            // reject trailing garbage if specified in the passed options
            var complete = Utils.getBooleanOption(options, 'complete', false);
            if (complete && (matchedText.length < text.length)) { return null; }

            // remove group-separators
            var grouped = false;
            if (group) {
                var oldNumberText = numberText;
                numberText = numberText.replace(getGrpRegEx(group), '');
                grouped = numberText.length < oldNumberText.length;
            }

            // parse the floating-point number
            var number = parseFloat(numberText.replace(dec, '.'));

            if (!_.isFinite(number)) { return null; }
            if (signText === '-') { number = -number; }

            // build the result descriptor
            return {
                number: number,
                text: matchedText,
                sign: signText,
                dec: matchedText.indexOf(dec) >= 0,
                scientific: _.isString(matches[3]),
                remaining: text.substr(matchedText.length),
                grouped: grouped
            };
        }

        return parseLeadingNumber;
    }());

    /**
     * Tries to parse a date from the leading part of the passed string.
     *
     * @param {String} text
     *  The text potentially starting with a date.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.dec=LocaleData.DEC]
     *      A custom decimal separator character. By default, the decimal
     *      separator of the current UI language will be used. Needed to decide
     *      whether the period character is allowed as date separator (if the
     *      decimal separator is a period, it cannot be part of a date).
     *  @param {Boolean} [options.leadingMonth=LocaleData.LEADING_MONTH]
     *      If set to true, the parser will prefer months before days if in
     *      doubt. If omitted, the preference of the current UI locale will be
     *      used. Example: The text '1/2' may be converted to the date Jan-02
     *      instead of Feb-01.
     *  @param {Boolean} [options.leadingYear=LocaleData.LEADING_YEAR]
     *      If set to true, the parser will prefer a leading year if in doubt.
     *      If omitted, the preference of the current UI locale will be used.
     *      Example: The text '1/2/3' may be converted to the date 2001-Feb-03
     *      instead of 2003-Feb-01.
     *  @param {Number} [options.centuryThreshold=100]
     *      An integer in the range 0 to 100 that specifies how to convert a
     *      number below 100 to a 4-digit year. If omitted, all short years
     *      will be moved into the current century. See DateUtils.expandYear()
     *      for more details.
     *  @param {Boolean} [options.complete=false]
     *      If set to true, the entire passed string must represent a valid
     *      date. No remaining 'garbage' text will be accepted.
     *
     * @returns {Object|Null}
     *  A result descriptor, if the passed string starts with a valid date
     *  (according to the passed options); otherwise null. The result
     *  descriptor contains the following properties:
     *  - {Number|Null} D
     *      The parsed day (one-based); or null, if the text does not contain a
     *      day (i.e. month and year only).
     *  - {Number} M
     *      The parsed month (zero-based!).
     *  - {Number|Null} Y
     *      The parsed year; or null, if the text does not contain a year (i.e.
     *      day and month only).
     *  - {Date} date
     *      A UTC date object representing the parsed date. Defaults to the
     *      current year, and to the first day in the month, if the respective
     *      date component is missing in the passed text.
     *  - {String} text
     *      The leading text from the passed string that has been parsed to the
     *      date.
     *  - {String} remaining
     *      The remaining characters from the passed string following the
     *      parsed date. Will be the empty string, if the option 'complete' has
     *      been set (see above).
     */
    Parser.parseLeadingDate = (function () {

        // returns a regular expression used to match a specific date at the beginning of a string
        var getDateRegExp = _.memoize(function (comps, dec) {

            // character class for valid date separators
            var separators = (dec === '.') ? '[-/]' : '[-./]';

            var pattern = '';
            _.each(comps, function (comp) {

                // add separator character between date components
                if (pattern) { pattern += separators; }

                // add a capturing group for the current component
                switch (comp) {
                    case 'D':
                        pattern += '([12]\\d|3[01]|0?[1-9])'; // shortest alternative must be at the end
                        break;
                    case 'M':
                        // add standard month-numbers for month detection
                        // AND (new) add month-name regexp
                        pattern += '(1[0-2]|0?[1-9])';
                        break;
                    case 'Y':
                        pattern += '(\\d{4}|\\d\\d?)'; // shortest alternative must be at the end
                        break;
                }
            });

            pattern += '(?:\\s|$)';

            return new RegExp('^' + pattern, 'i');
        }, function (comps, dec) { return comps + dec; });

        // returns a regular expression used to match a specific date at the beginning of a string
        var getLongDateRegExp = _.memoize(function (comps) {

            // character class for valid date separators
            var separators = '[-,./\\s]+';

            var pattern = '';
            _.each(comps, function (comp) {

                // add separator character between date components
                if (pattern) { pattern += separators; }

                // add a capturing group for the current component
                switch (comp) {
                    case 'D':
                        pattern += '([12]\\d|3[01]|0?[1-9])'; // shortest alternative must be at the end
                        break;
                    case 'M':
                        // add standard month-numbers for month detection
                        // AND (new) add month-name regexp
                        pattern += '(' + getMonthRegExp('', { monthGroupGet: false }) + ')';
                        break;
                    case 'Y':
                        pattern += '(\\d{4}|\\d\\d?)'; // shortest alternative must be at the end
                        break;
                }
            });

            pattern += '(?:\\s|$)';

            return new RegExp('^' + pattern, 'i');
        }, function (comps, dec) { return comps + dec; });

        // tries to parse a date with the specified components
        function parseDate(text, comps, dec) {

            // try to match a date with the specified components
            var matches = getDateRegExp(comps, dec).exec(text);
            if (!matches) { matches = getLongDateRegExp(comps).exec(text); }
            if (!matches) { return null; }

            // the result object with the matched text
            var result = {
                D: null,
                M: null,
                Y: null,
                text: matches[0],
                remaining: text.substr(matches[0].length)
            };

            // convert the matched date components to numbers
            _.each(comps, function (comp, index) {
                var parsedVal = parseInt(matches[index + 1], 10);
                result[comp] = (_.isFinite(parsedVal)) ? parsedVal : matches[index + 1];
            });

            // zero-based month (by convention, as used everywhere else, e.g. in class Date)
            if (_.isFinite(result.M)) { result.M -= 1; }

            return result;
        }

        // the actual parseLeadingDate() method
        function parseLeadingDate(text, options) {

            // the decimal separator needed to select the valid date separators
            var dec = Utils.getStringOption(options, 'dec', LocaleData.DEC);
            // whether to prefer leading months and years in the text to be parsed
            var leadingMonth = Utils.getBooleanOption(options, 'leadingMonth', LocaleData.LEADING_MONTH);
            var leadingYear = Utils.getBooleanOption(options, 'leadingYear', LocaleData.LEADING_YEAR);

            // the day/month components for getDateRegExp() according to the native order of the current locale
            var DAY_MONTH_COMPS = leadingMonth ? 'MD' : 'DM';

            // all date patterns to be matched, in the correct order
            var ALL_COMPONENTS = [
                // start with complete dates (leading and trailing year, prefer native date order)
                leadingYear ? ('YMD') : (DAY_MONTH_COMPS + 'Y'),
                leadingYear ? (DAY_MONTH_COMPS + 'Y') : ('YMD'),
                // day/month without year
                DAY_MONTH_COMPS,
                // month/year without day (leading and trailing year, prefer native date order)
                leadingYear ? 'YM' : 'MY',
                leadingYear ? 'MY' : 'YM'
            ];

            // find the first matching date pattern (method some() will exit on first match)
            var result = null;
            ALL_COMPONENTS.some(function (comps) { return (result = parseDate(text, comps, dec)); });
            if (!result) { return null; }

            // reject trailing garbage if specified in the passed options
            var complete = Utils.getBooleanOption(options, 'complete', false);
            if (complete && (result.remaining.length > 0)) { return null; }

            // validate the day according to month and year
            var threshold = Utils.getIntegerOption(options, 'centuryThreshold', 100, 0, 100);
            var y = _.isNumber(result.Y) ? DateUtils.expandYear(result.Y, threshold) : new Date().getFullYear();
            var m = _.isNumber(result.M) ? result.M : _.isString(result.M) ? MONTH_MAP[result.M.toLowerCase()] : 0;
            var d = _.isNumber(result.D) ? result.D : 1;
            if (d > DateUtils.getDaysInMonth(y, m)) { return null; }

            // create the date property in the result
            result.date = DateUtils.makeUTCDate({ Y: y, M: m, D: d });

            return isFinite(result.date.getTime()) ? result : null;
        }

        return parseLeadingDate;
    }());

    /**
     * Tries to parse a time from the leading part of the passed string.
     *
     * @param {String} text
     *  The text potentially starting with a time.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.dec=LocaleData.DEC]
     *      If specified, a custom decimal separator character. By default, the
     *      decimal separator of the current UI language will be used. Needed
     *      to parse the fractional part of seconds (see option 'milliseconds'
     *      for details).
     *  @param {Boolean} [options.multipleDays=false]
     *      If set to true, the hours may be greater than the value 23,
     *      resulting in a time greater than or equal to an entire day.
     *  @param {Boolean} [options.milliseconds=false]
     *      If set to true, the seconds may contain a fractional part, e.g. in
     *      the parse text '12:23:34.567'.
     *  @param {Boolean} [options.complete=false]
     *      If set to true, the entire passed string must represent a valid
     *      time. No remaining 'garbage' text will be accepted.
     *
     * @returns {Object|Null}
     *  A result descriptor, if the passed string starts with a valid time
     *  (according to the passed options); otherwise null. The result
     *  descriptor contains the following properties:
     *  - {Number|Null} h
     *      The parsed hour; or null, if the text does not contain an hour
     *      (i.e. minutes and seconds only).
     *  - {Number} m
     *      The parsed minute.
     *  - {Number|Null} s
     *      The parsed second; or null, if the text does not contain a second
     *      (i.e. hours and minutes only).
     *  - {Number|Null} ms
     *      The parsed milliseconds (e.g. 500 for the text '12:34.5'); or null,
     *      if the text does not contain a fractional seconds part.
     *  - {Date} time
     *      A UTC date object representing the parsed time.
     *  - {String} text
     *      The leading text from the passed string that has been parsed to the
     *      time.
     *  - {Number} serial
     *      The parsed time, as floating-point number (fraction of a day).
     *  - {Boolean} ampm
     *      If true, the result contains a am/pm formatted time.
     *  - {String} remaining
     *      The remaining characters from the passed string following the
     *      parsed time. Will be the empty string, if the option 'complete' has
     *      been set (see above).
     */
    Parser.parseLeadingTime = (function () {

        // returns a regular expression used to match a specific time at the beginning of a string
        var getTimeRegExp = _.memoize(function (comps, dec) {
            var pattern = '';
            _.each(comps, function (comp) {

                // add separator character between time components
                if (pattern) { pattern += '\\s*' + ((comp === '0') ? _.escapeRegExp(dec) : ':') + '\\s*'; }

                // add a capturing group for the current component
                switch (comp) {
                    case 'h':
                        pattern += '(\\d{1,4})'; // accept up to 4 digits for hours (multiple days)
                        break;
                    case 'm':
                    case 's':
                        pattern += '([0-5]?\\d)';
                        break;
                    case '0':
                        pattern += '(\\d+)';
                }
            });

            return new RegExp('^' + pattern);
        }, function (comps, dec) { return comps + dec; });

        // creates and caches the regular expressions to detect am/pm time-formats
        var getAmPmMatchingRegExp = _.memoize(function (options) {
            if (Utils.getBooleanOption(options, 'leadingNumbers', false)) {
                return new RegExp('^(1[0-2]|[1-9])*\\s+([ap]m?)', 'i');
            } else {
                return new RegExp('^\\s*([ap]m?)\\s+', 'i');
            }
        }, function (options) { return Utils.getBooleanOption(options, 'leadingNumbers', false); });

        // tries to detect a am/pm time
        function parseAmPmTime(text, result, options) {
            var regex = getAmPmMatchingRegExp(options),
                regExResult = regex.exec(text);

            // the am/pm regex found something
            if (regExResult) {
                var // store the first letter of the 'am' or 'pm' string
                    firstLetter = regExResult[(regExResult.length - 1)].substr(0, 1).toLowerCase();

                // if the given result is empty
                if (result === null) {
                    // create result object
                    result = {
                        h: null,
                        m: null,
                        s: null,
                        ms: null,
                        text: text
                    };
                }

                // set the new real remaining string
                result.remaining = text.substr(regExResult[0].length);
                // set am/pm attribute to 'true'
                result.ampm = true;

                // parse the hour from the result, if we used the 'leadingNumbers'-regex
                if ((regExResult.length === 3) && regExResult[1]) {
                    result.h = parseInt(regExResult[1], 10);
                }

                // if it's a pm time, add 12 hours to 'h' attribute
                if (_.isNumber(result.h) && firstLetter === 'p') { result.h += 12; }

                // special behavior to solve 0:00/24:00 h am/pm problem
                if ((result.h === 12 && firstLetter === 'a') || (result.h === 24 && firstLetter === 'p')) {
                    result.h -= 12;
                }
            }

            return result;
        }

        // tries to parse a time with the specified components
        function parseTime(text, comps, dec) {

            // try to match a time with the specified components
            var matches = getTimeRegExp(comps, dec).exec(text);
            if (!matches) { return null; }

            // the result object with the matched text
            var result = {
                h: null,
                m: null,
                s: null,
                ms: null,
                text: matches[0],
                remaining: text.substr(matches[0].length)
            };

            // convert the matched time components to numbers
            _.each(comps, function (comp, index) {
                var match = matches[index + 1];
                // special handling for the fractional seconds part
                if (comp === '0') {
                    result.ms = parseFloat('0.' + match) * 1000;
                } else {
                    result[comp] = parseInt(match, 10);
                }
            });

            // if there were remaining letters, try to find the am/pm format
            result = (result.remaining.length > 0) ? parseAmPmTime(result.remaining, result, { leadingNumbers: true }) : result;

            return result;
        }

        // the actual parseLeadingTime() method
        function parseLeadingTime(text, options) {

            // the decimal separator used to parse the milliseconds
            var dec = Utils.getStringOption(options, 'dec', LocaleData.DEC);
            // whether to accept milliseconds
            var milliseconds = Utils.getBooleanOption(options, 'milliseconds', false);
            // all time patterns to be matched, in the correct order (TODO: AM/PM)
            var ALL_COMPONENTS = milliseconds ? ['hms0', 'hms', 'hm', 'ms0'] : ['hms', 'hm'];
            // find the first matching time pattern (method some() will exit on first match)
            var result = null;

            ALL_COMPONENTS.some(function (comps) { return (result = parseTime(text, comps, dec)); });

            // when normal time matching doesn't find anything, try to find a am/pm time
            if (!result || result.remaining.length > 0) {

                // if there is a result, but some letters are remaining
                if (result && result.remaining.length > 0) {
                    // try to find am/pm time expression
                    result = parseAmPmTime(result.remaining, result);
                }

                //
                if (!result) {
                    result = parseAmPmTime(text, result, { leadingNumbers: true });
                }

                // if am/pm time matching also doesn't match anything
                if (!result) {
                    return null;
                }
            }

            // reject trailing garbage if specified in the passed options
            var complete = Utils.getBooleanOption(options, 'complete', false);
            if (complete && (result.remaining.length > 0)) { return null; }

            // reject hours above 23 unless specified
            var multipleDays = Utils.getBooleanOption(options, 'multipleDays', false);
            if (!multipleDays && _.isNumber(result.h) && (result.h > 23)) { return null; }

            // create the time property in the result
            result.time = new Date(0);
            if (_.isNumber(result.h)) { result.time.setUTCHours(result.h); }
            if (_.isNumber(result.m)) { result.time.setUTCMinutes(result.m); }
            if (_.isNumber(result.s)) { result.time.setUTCSeconds(result.s); }
            if (_.isNumber(result.ms)) { result.time.setUTCMilliseconds(result.ms); }

            result.serial = result.time.getTime() / DateUtils.MSEC_PER_DAY;

            return result;
        }

        return parseLeadingTime;
    }());

    /**
     * Parses a format code, and returns an instance of the class ParsedFormat
     * containing all properties of the format code needed to format numbers or
     * strings.
     *
     * @param {String} fileFormat
     *  The file format identifier specifying which format tokens will be
     *  recognized by the parser.
     *
     * @param {String} grammarId
     *  The identifier of a format grammar. Supported values are:
     *  - 'op': Fixed token representations as used in operations, and the file
     *      format.
     *  - 'ui': The localized token representations according to the current UI
     *      language of the application.
     *
     * @param {String} formatCode
     *  The format code to be parsed.
     *
     * @returns {ParsedFormat}
     *  An instance of the class ParsedFormat containing all properties of the
     *  format code needed to format numbers or strings.
     */
    Parser.parseFormatCode = (function () {

        /**
         * Returns if the fraction on the give token index is a proper fraction.
         *
         * @param {Integer} tokenIndex
         *  index of the fraction token.
         *
         * @returns {Boolean}
         *  true if the fraction is proper, otherwise false.
         */
        function isProperFraction(tokens, tokenIndex) {
            var proper = false;
            var start = false;
            var index = tokenIndex - 1;
            if (index >= 1) {
                for (; index >= 0; index--) {
                    var token = tokens[index];
                    if (token.type === 'lit' || token.type === 'fill' || token.type === 'text') {
                        start = true;
                    } else if (start && token.type === 'nr') {
                        proper = true;
                        break;
                    } else {
                        break;
                    }
                }
            }
            return proper;
        }

        // creates a hash key for the memoize() cache
        function getConfigKey(fileFormat, grammarId, formatCode) {
            return fileFormat + ':' + grammarId + ':' + formatCode;
        }

        // parses a new format code
        function parseFormatCode(fileFormat, grammarId, formatCode) {

            // the resulting parsed format code
            var parsedFormat = new ParsedFormat(formatCode);

            // bug 46211: empty format code falls back to standard format
            if (formatCode.length === 0) { formatCode = 'General'; }

            // stop if the entire format code has been consumed, or a syntax error was found
            while (!parsedFormat.syntaxError && _.isString(formatCode)) {
                formatCode = parsedFormat.parseSection(fileFormat, grammarId, formatCode);
            }

            // initialize category of the format code (before adding default sections)
            var numSections = parsedFormat.numberSections;
            if (numSections.length === 0) {
                // 'text' category, if there are no numeric sections at all
                parsedFormat.category = 'text';
            } else if ((numSections.length === 1) && (numSections[0].category === 'standard')) {
                // 'standard' category only for standard format code (optionally with color)
                parsedFormat.category = 'standard';
            } else {
                // use the category of the numeric sections, if they are all equal
                var categories = _.unique(_.pluck(numSections, 'category'));
                parsedFormat.category = (categories.length === 1) ? categories[0] : 'custom';
            }

            // initialize default intervals of numeric code sections
            switch (numSections.length) {
                case 0:
                    // text format only: format numbers with standard number format
                    numSections.push(new FormatSection());
                    numSections[0].pushToken('general');
                    break;
                case 1:
                    // one number section for all numbers
                    break;
                case 2:
                    if (!numSections[0].operator) {
                        // two number sections: positive and zero, negative
                        numSections[0].setInterval('>=', 0, true);
                        numSections[1].setInterval('<', 0, true);
                    }
                    break;
                case 3:
                case 4:
                    // It's possible to create with excel a format like this [>1]###-####;[<1](###) ###-####;-#-;
                    // and a fourth numSections will created. BUT this numSection must be empty, otherwise the
                    // format is not valid.
                    if (numSections.length === 4) {
                        // The fourth numSections must be empty.
                        if (numSections[3].formatCode === '') {
                            numSections.pop(); // remove the useless numSection
                        } else {
                            parsedFormat.syntaxError = true;
                        }
                    }
                    if (numSections[0].operator) {
                        // If the first section is set and the second not, then the second is less than zero.
                        if (!numSections[1].operator) {
                            numSections[1].setInterval('<', 0, true);
                        }
                    } else {
                        // three number sections: positive, negative, zero
                        numSections[0].setInterval('>', 0, true);
                        numSections[1].setInterval('<', 0, true);
                        numSections[2].setInterval('=', 0, true);
                    }
                    break;
                default:
                    // invalid count of number sections
                    parsedFormat.syntaxError = true;
            }

            // Set the right number count for token of the type 'nr' and if a fraction is proper.
            var tokenIndex = [];
            var hasFraction = false;
            var rightNumbersCount = 0;
            numSections.forEach(function (section) {
                for (var t = section.tokens.length - 1; t >= 0; t--) {
                    var token = section.tokens[t];
                    if (token.type === 'nr') {
                        // set the count of numbers on the right side for number formats like this
                        // "####-##-#" > 3-1-0, to split the number 12345678 to 12345-67-8
                        token.rightNumbersCount = rightNumbersCount;
                        if (!hasFraction) {
                            tokenIndex.push(t);
                        }
                        rightNumbersCount += token.text.length;
                    } else if (token.type === 'frac') {
                        rightNumbersCount = 0;
                        hasFraction = true;
                        // set if the token is a proper fraction or not
                        token.isProperFraction = isProperFraction(section.tokens, t);
                    }
                }
                if (hasFraction) {
                    tokenIndex.forEach(function (index) {
                        var token = section.tokens[index];
                        delete token.rightNumbersCount;
                        // if a token of the type is 'nr' the cell shows as many 0 as the token text lenght
                        // e.g. token.text = '????' the cell shows '0000'
                        token.fractionText = token.text.replace(/./g, '0');
                    });
                }
                tokenIndex = [];
                hasFraction = false;
                rightNumbersCount = 0;
            });
            tokenIndex = null;

            // initialize missing text section
            if (!parsedFormat.textSection) {
                parsedFormat.textSection = FormatSection.parse(fileFormat, 'op', '@').section;
                // use color of first number section, but only if it is the standard format code
                if (numSections[0].colorName && (parsedFormat.category === 'standard')) {
                    parsedFormat.textSection.setColor(numSections[0].colorName);
                }
            } else if (parsedFormat.textSection.operator) {
                // text section with number interval is invalid
                parsedFormat.syntaxError = true;
            }

            return parsedFormat;
        }

        // create a function that caches all parsed formats
        return _.memoize(parseFormatCode, getConfigKey);
    }());

    // exports ================================================================

    return Parser;

});
