/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Kai Ahrens <kai.ahrens@open-xchange.com>
 */

define('io.ox/office/textframework/view/dialog/pagesettingsdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/dialogs',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/settings/units',
    'gettext!io.ox/office/textframework/main',
    'less!io.ox/office/textframework/view/dialog/pagesettingsdialog'
], function (Utils, Forms, Dialogs, Operations, Units, gt) {

    'use strict';

    /**
     * Localized string contents and read only arrays with page attributes
     */

    var // page-specific wording as of text-app
        //
        pageSettingsStr = /*#. Defining the page extents and page margins*/ gt('Page Settings'),
        pageFormatStr = /*#. Predefined standard page formats, e.g. A4*/ gt('Paper format'),
        pageOrientationStr = /*#. Orientation of the page, e.g. Portrait or Landscape*/ gt('Paper orientation'),
        pageMarginsStr = /*#. Predefined page margins for the edges of a paper*/ gt('Page margins'),
        pageSizeStr = /*#. Defining the page extents*/ gt('Paper size'),
        pageWidthStr = /*#. Defining the page width*/ gt('Width'),
        pageHeightStr = /*#. Defining the page height*/ gt('Height'),
        marginLeftStr = /*#. Defining the left page margin*/ gt('Left'),
        marginTopStr = /*#. Defining the top page margin*/ gt('Top'),
        marginRightStr = /*#. Defining the right page margin*/ gt('Right'),
        marginBottomStr = /*#. Defining the bottom page margin*/ gt('Bottom'),
        orientationPortraitStr = /*#. The 'Portrait' page orientation*/ gt('Portrait'),
        orientationLandscapeStr = /*#. The 'Landscape' page orientation*/ gt('Landscape'),
        marginNarrowStr = /*#. A set of predefined, 'Narrow' page margins*/ gt('Narrow'),
        marginNormalStr = /*#. A set of predefined, 'Normal' page margins*/ gt('Normal'),
        marginWideStr = /*#. A set of predefined, 'Wide' page margins*/ gt('Wide'),
        customStr = /*#. A custom extent*/ gt('Custom'),

        // slide-specific wording as of presentation-app
        //
        slideSettingsStr = /*#. Defining the slide extents*/ gt('Slide Settings'),
        slideFormatStr = /*#. Predefined standard slide formats, e.g. A4*/ gt('Slide format'),
        slideOrientationStr = /*#. Orientation of the slide, e.g. Portrait or Landscape*/ gt('Orientation'),
        slideSizeStr = /*#. Defining the slide extents*/ gt('Slide size'),

        // media-/app-specific format-setting data
        //
        FORMAT_SETTING_MAP = {

            // page-specific
            letter:     { value: 'letter', name: /*#. Predefined 'Letter' page format*/ gt('Letter'), width: 21590, height: 27940 },
            tabloid:    { value: 'tabloid', name: /*#. Predefined 'Tabloid' page format*/ gt('Tabloid'), width: 27900, height: 43200 },
            legal:      { value: 'legal', name: /*#. Predefined 'Legal' page format*/ gt('Legal'), width: 21590, height: 35560 },
            statement:  { value: 'statement', name: /*#. Predefined 'Statement' page format*/ gt('Statement'), width: 14000, height: 21600 },
            executive:  { value: 'executive', name: /*#. Predefined 'Executive' page format*/ gt('Executive'), width: 18400, height: 26700 },
            folio:      { value: 'folio', name: /*#. Predefined 'Folio' page format*/ gt('Folio'), width: 21600, height: 33000 },
            a3:         { value: 'a3', name: /*#. Predefined 'A3' page format*/ gt('A3'), width: 29700, height: 42000 },
            a4:         { value: 'a4', name: /*#. Predefined 'A4' page format*/ gt('A4'), width: 21000, height: 29700 },
            a5:         { value: 'a5', name: /*#. Predefined 'A5' page format*/ gt('A5'), width: 14800, height: 21000 },
            b4:         { value: 'b4', name: /*#. Predefined 'B4' page format*/ gt('B4'), width: 25000, height: 35300 },
            b5:         { value: 'b5', name: /*#. Predefined 'B5' page format*/ gt('B5'), width: 17600, height: 25000 },

            /**
             *  slide-specific:
             *  - width/height values need to be set paper format alike in order to handle orientation calculation properly for both use cases screen and paper.
             */
            screen_4_by_3:    { value: 'screen_4_by_3', name: /*#. Predefined 'Standard (4:3)' slide format for presentations */ gt('Standard (4:3)'), width: 19050, height: 25400 },
            screen_16_by_9:   { value: 'screen_16_by_9', name: /*#. Predefined 'Widescreen (16:9)' slide format for presentations */ gt('Widescreen (16:9)'), width: 14290, height: 25400 },
            screen_16_by_10:  { value: 'screen_16_by_10', name: /*#. Predefined 'Widescreen (16:10)' slide format for presentations */ gt('Widescreen (16:10)'), width: 15880, height: 25400 },
            screen_ultrawide: { value: 'screen_ultrawide', name: /*#. Predefined 'Ultra Widescreen' slide format for presentations */ gt('Ultra Widescreen'), width: 19050, height: 33870 }
        },

        PAGE_FORMAT_SETTING_KEY_LIST = [
            'letter',
            'tabloid',
            'legal',
            'statement',
            'executive',
            'folio',
            'a3',
            'a4',
            'a5',
            'b4',
            'b5'
        ],
        SLIDE_FORMAT_SETTING_KEY_LIST = [
            'screen_4_by_3',
            'screen_16_by_9',
            'screen_16_by_10',
            'screen_ultrawide',
            'letter',
            'a3',
            'a4',
            'b4',
            'b5'
        ],

        ORIENTATION_SETTING_LIST = [
            { value: 'portrait', name: orientationPortraitStr },
            { value: 'landscape', name: orientationLandscapeStr }
        ],

        MARGIN_SETTING_LIST = [
           { value: 'narrow', name: marginNarrowStr, left: 1270, top: 1270, right: 1270, bottom: 1270 },
           { value: 'normal', name: marginNormalStr, left: 2540, top: 2540, right: 2540, bottom: 2540 },
           { value: 'wide', name: marginWideStr, left: 5080, top: 5080, right: 5080, bottom: 5080 }
        ];

    function collectFormatSettings(collector, formatKey/*, idx, list*/) {
        if (formatKey in FORMAT_SETTING_MAP) {
            collector.push(FORMAT_SETTING_MAP[formatKey]);
        }
        return collector;
    }

    // class PageSettingsDialog ===============================================

    /**
     * The page settings modal dialog.
     *
     * Shows the currently set page width and page height, the page margins and some
     * other page attributes directly after executing the dialog.
     * Depending on the given values, a predefined page format as well as a predefined
     * set of page margins is tried to be determined and selected/shown accordingly.
     * The page attributes can be changed by various dropdown menu and spinfield controls.
     * The changes will be applied to the current document after clicking the 'Ok' button.
     *
     * The dialog itself is shown by calling this.execute. The caller can react on successful
     * finalization of the dialog within an appropriate done-Handler, chained to the
     * this.execute function call.
     *
     * @constructor
     *
     * @extends Dialogs.ModalDialog
     *
     * @param {TextView} docView
     *  The view instance containing this editor instance.
     */
    function PageSettingsDialog(docView) {

        var self = this,
            // Misc
            docModel = docView.getDocModel(),
            unit = Units.getStandardUnit(),
            step = Units.getStandardUnitStep(),
            // jQuery DOM objects
            jqFormat = null,
            jqOrientation = null,
            jqMargins = null,
            jqWidth = null,
            jqHeight = null,
            jqMarginTop = null,
            jqMarginBottom = null,
            jqMarginLeft = null,
            jqMarginRight = null,
            jqSizeTitle = null,
            jqMarginTitle = null,
            jqTable = null,
            jqTableVertMargins = null,
            jqTableHorzMargins = null,
            // Page data
            formatSettingList = null,
            isPresentationApp = docModel.useSlideMode(),
            initialPageAttrs  = _.copy(docModel.getStyleCollection('page').getElementAttributes(docModel.getNode()).page, true);

        // base constructor ---------------------------------------------------

        if (isPresentationApp) {

            // presentation-app specific text / text-label mapping
            pageSettingsStr     = slideSettingsStr;
            pageFormatStr       = slideFormatStr;
            pageOrientationStr  = slideOrientationStr;
            pageSizeStr         = slideSizeStr;

            // presentation-app specific format-settings
            formatSettingList = SLIDE_FORMAT_SETTING_KEY_LIST.reduce(collectFormatSettings, []);
        } else {
            // text-app specific format-settings
            formatSettingList = PAGE_FORMAT_SETTING_KEY_LIST.reduce(collectFormatSettings, []);
        }
        Dialogs.ModalDialog.call(this, { title: pageSettingsStr, width: (ox.mobile || isPresentationApp) ? 320 : 530 });

        // private methods ----------------------------------------------------

        /**
         * Creates and layouts all dialog controls, contained within the body of the
         * dialog.
         * This function, in conjunction with the dialogs' less file,
         * determines the general visual layout and functionality of the dialog
         */
        function initControls() {

            // create Dropdown links

            // PageFormat dropdown menu
            jqFormat = $('<div class="dropdown-link">').append(
                $('<label>').append($('<span>').text(pageFormatStr), $('<span>').text(_.noI18n(':'))),
                getDropdownMenu({ values: formatSettingList, handler: function (text) {
                    setPageFormat(_.findWhere(formatSettingList, { name: text }));
                } })
            );

            // PageOrientation dropdown menu
            jqOrientation = $('<div class="dropdown-link">').append(
                $('<label>').append($('<span>').text(pageOrientationStr), $('<span>').text(_.noI18n(':'))),
                getDropdownMenu({ values: ORIENTATION_SETTING_LIST, handler: function (text) {
                    setPageOrientation(_.findWhere(ORIENTATION_SETTING_LIST, { name: text }));
                } })
            );

            var dropDownContainer = $('<div class="dropdown-container">').append(jqFormat, jqOrientation);

            if (isPresentationApp) {

                jqFormat.addClass('control--slide-size');
                jqOrientation.addClass('control--orientation');

            } else {

                // PageMargins dropdown menu
                jqMargins = $('<div class="dropdown-link">').append(
                    $('<label>').append($('<span>').text(pageMarginsStr), $('<span>').text(_.noI18n(':'))),
                    getDropdownMenu({ values: MARGIN_SETTING_LIST, handler: function (text) {
                        setPageMargins(_.findWhere(MARGIN_SETTING_LIST, { name: text }));
                    } })
                );
                dropDownContainer.append(jqMargins);
            }

            // create controls

            // Width spinfield
            jqWidth = $(Forms.createSpinFieldMarkup({ label: pageWidthStr, tail: unit, min: cmToUnit(10), max: cmToUnit(55), step: step }));
            jqWidth.children('input').attr('tabindex', 10).on('change', function () {
                setPageValue(jqWidth, 'width', getPageValue($(this).val()));
            });

            // Height spinfield
            jqHeight = $(Forms.createSpinFieldMarkup({ label: pageHeightStr, tail: unit, min: cmToUnit(10), max: cmToUnit(55), step: step }));
            jqHeight.children('input').attr('tabindex', 11).on('change', function () {
                setPageValue(jqHeight, 'height', getPageValue($(this).val()));
            });

            // create title row with title element
            jqSizeTitle =  $('<h4>').text(pageSizeStr);

            if (!isPresentationApp) {

                jqSizeTitle.addClass('column40');

                // create title row with title element
                jqMarginTitle = $('<h4 class="column60">').text(pageMarginsStr);

                // MarginTop spinfield
                jqMarginTop = $(Forms.createSpinFieldMarkup({ label: marginTopStr, tail: unit, min: cmToUnit(0), max: cmToUnit(5), step: step }));
                jqMarginTop.children('input').attr('tabindex', 20).on('change', function () {
                    setPageValue(jqMarginTop, 'marginTop', getPageValue($(this).val()));
                });

                // MarginBottom spinfield
                jqMarginBottom = $(Forms.createSpinFieldMarkup({ label: marginBottomStr, tail: unit, min: cmToUnit(0), max: cmToUnit(5), step: step }));
                jqMarginBottom.children('input').attr('tabindex', 21).on('change', function () {
                    setPageValue(jqMarginBottom, 'marginBottom', getPageValue($(this).val()));
                });
                // MarginLeft spinfield
                jqMarginLeft = $(Forms.createSpinFieldMarkup({ label: marginLeftStr, tail: unit, min: cmToUnit(0), max: cmToUnit(5), step: step }));
                jqMarginLeft.children('input').attr('tabindex', 30).on('change', function () {
                    setPageValue(jqMarginLeft, 'marginLeft', getPageValue($(this).val()));
                });

                // MarginRight spinfield
                jqMarginRight = $(Forms.createSpinFieldMarkup({ label: marginRightStr, tail: unit, min: cmToUnit(0), max: cmToUnit(5), step: step }));
                jqMarginRight.children('input').attr('tabindex', 31).on('change', function () {
                    setPageValue(jqMarginRight, 'marginRight', getPageValue($(this).val()));
                });

                jqTable = $('<table class="extents">').append(
                    $('<tr>').append($('<td>').append(jqWidth)),
                    $('<tr>').append($('<td>').append(jqHeight))
                );

                jqTableVertMargins = $('<table class="margins">').append(
                    $('<tr>').append($('<td>').append(jqMarginTop)),
                    $('<tr>').append($('<td>').append(jqMarginBottom))
                );

                jqTableHorzMargins = $('<table class="margins">').append(
                    $('<tr>').append($('<td>').append(jqMarginLeft)),
                    $('<tr>').append($('<td>').append(jqMarginRight))
                );

                // append control divs to body
                self.getBody().append(
                    $('<form novalidate>').append(
                        dropDownContainer,
                        jqSizeTitle,
                        jqMarginTitle,
                        $('<hr class="ruler35">'),
                        $('<hr class="ruler250px">'),
                        jqTable,
                        jqTableVertMargins,
                        jqTableHorzMargins
                    )
                );

            } else {

                var $tr = $('<tr>');
                $tr.append($('<td>').append(jqWidth));
                $tr.append($('<td>').append(jqHeight));

                jqTable = $('<table>').append($tr);

                // append control divs to body
                self.getBody().append(
                    $('<form novalidate>').append(
                        dropDownContainer,
                        jqSizeTitle,
                        $('<hr>'),
                        jqTable
                    )
                );
            }
        }

        /**
         * Creates and returns a jQuery drop-down menu element
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Array<String>} [options.values]
         *      The array of strings to create the drop-down menu entries.
         *  @param {Function} [options.handler]
         *      The callback function to be called when a drop-down menu entry
         *      has been clicked.
         *
         * @returns {jQuery}
         *  The created menu node or null if no values where given.
         */
        function getDropdownMenu(options) {
            var menu = null,
                jqList = $('<ul class="dropdown-menu" role="menu">'),
                values = _.isObject(options) ? options.values : null,
                firstValue = (values && (values.length > 0)) ? values[0] : null;

            if (firstValue) {
                // create menu anchor
                menu = $('<span class="dropdown">').
                    append($('<a href="#">').
                        attr({ tabindex: 1, 'data-type': firstValue.value, 'aria-haspopup': true, 'data-toggle': 'dropdown' }).
                        addClass('text').
                        text(firstValue.name)).
                    append(jqList);

                // fill dropdown menu with given values and assign callback handler, if given
                _.each(values, function (object) {
                    jqList.append($('<li>').append($('<a>').
                        attr({ href: '#', 'data-value': object.value, role: 'menuitem' }).
                        addClass('dropdown-listentry').
                        text(object.name)).
                        on('click', function (evt) {
                            evt.preventDefault();
                            var curText = $(this).text();

                            // set currently selected entry as menu title
                            menu.find('a[data-toggle=dropdown]').text(curText);

                            if (_.isFunction(options.handler)) {
                                options.handler(curText);
                            }
                        }));
                });
            }

            return menu;
        }

        /**
         * Sets the current page width and page height
         * values for the given pageFormat object and
         * updates the corresponding controls.
         *
         * @param {Object} [pageFormat]
         *  The page format object, that contains the width and height
         *  properties to be set. Depending on the currently set page
         *  orientation value, the with and height values are exchanged with
         *  each other. Expected properties:
         *  @param {Number} pageFormat.width
         *      The page width in 100th millimeters to be set
         *  @param {Number} pageFormat.height
         *      The page height in 100th millimeters to be set
         */
        function setPageFormat(pageFormat) {
            if (_.isObject(pageFormat)) {
                if (!pageAttrs.orientation) {
                    pageAttrs.orientation = (isPresentationApp ? 'landscape' : 'portrait');
                }
                var isPortrait = (pageAttrs.orientation === 'portrait');

                setPageValue(jqWidth, 'width', isPortrait ? pageFormat.width :  pageFormat.height);
                setPageValue(jqHeight, 'height', isPortrait ? pageFormat.height : pageFormat.width);
            }
        }

        /**
         * Sets the current page orientation and updates the
         * corresponding controls.
         * Current width and height values are exchanged with
         * each other, if the orientation changes.
         *
         * @param {Object} [pageOrientation]
         *  The page orientation object, that contains the value for the
         *  current page orientation, with the following properties:
         *  @param {String} pageOrientation.value
         *      The page orientation string. This property needs to be set to
         *      'portrait' or 'landscape'.
         */
        function setPageOrientation(pageOrientation) {
            if (_.isObject(pageOrientation)) {
                var
                    orientation = (pageOrientation.value !== 'portrait') ? 'landscape' : 'portrait';

                if ((orientation !== pageAttrs.orientation) && (
                    ((orientation === 'portrait') && (pageAttrs.width > pageAttrs.height)) ||
                    ((orientation === 'landscape') && (pageAttrs.width < pageAttrs.height))
                )) {
                    var oldWidth = pageAttrs.width,
                        oldHeight = pageAttrs.height;

                    pageAttrs.orientation = orientation;

                    setPageValue(jqWidth, 'width', oldHeight);
                    setPageValue(jqHeight, 'height', oldWidth);
                }
            }
        }

        /**
         * Sets the current page margins as predefined set of values
         * and updates the corresponding page margin controls.
         *
         * @param {Object} [pageMargins]
         *  The optional PageMargins object, that contains the predefined
         *  values for the  page margins.
         *  Optional pproperties, that may be set at the pageMargins object.
         *  @param {Number} [pageMargins.left]
         *   The optional left page margin in 100th millimeters
         *  @param {Number} [pageMargins.top]
         *   The optional top page margin in 100th millimeters
         *  @param {Number} [pageMargins.right]
         *   The optional right page margin in 100th millimeters
         *  @param {Number} [pageMargins.bottom]
         *   The optional bottom page margin in 100th millimeters
         */
        function setPageMargins(pageMargins) {
            if (_.isObject(pageMargins)) {
                setPageValue(jqMarginLeft, 'marginLeft', pageMargins.left);
                setPageValue(jqMarginTop, 'marginTop', pageMargins.top);
                setPageValue(jqMarginRight, 'marginRight', pageMargins.right);
                setPageValue(jqMarginBottom, 'marginBottom', pageMargins.bottom);
            }
        }

        /**
         * Sets the current value and updates the corresponding page control,
         * depending on the given jQuery Node and the property key of the
         * current pageData Object.
         * All three parameters need to be given, if an update
         * is to be performed.
         *
         * @param {jQuery} jqNode
         *  The jQuery node that needs to be updated.
         *
         * @param {String} pageDataKey
         *  The property name of the current pageData object, whose value
         *  should be set accordingly.
         *
         * @param {Number} pageValue
         *   The page value to be set in 100th millimeters
         */
        function setPageValue(jqNode, pageDataKey, pageValue) {

            // set current page value and update corresponding control
            pageAttrs[pageDataKey] = pageValue;
            jqNode.children('input').val(getUIValue(pageValue));

            // update selection and text of dropdown menus
            var isPortrait = pageAttrs.orientation === 'portrait',
                width = Utils.round(isPortrait ? pageAttrs.width : pageAttrs.height, 10),
                height = Utils.round(isPortrait ? pageAttrs.height : pageAttrs.width, 10);

            selectDropdownEntry(jqFormat,
                _.findWhere(formatSettingList, {
                    width: width,
                    height: height
                })
            );

            selectDropdownEntry(jqOrientation,
                _.findWhere(ORIENTATION_SETTING_LIST, {
                    value: pageAttrs.orientation
                })
            );

            if (!isPresentationApp) {
                selectDropdownEntry(jqMargins,
                    _.findWhere(MARGIN_SETTING_LIST, {
                        left: pageAttrs.marginLeft,
                        top: pageAttrs.marginTop,
                        right: pageAttrs.marginTop,
                        bottom: pageAttrs.marginBottom
                    })
                );
            }
        }

        /**
         * Selects the appropriate menu entry of the given dropdown
         * menu and sets the text of the menu accordingly
         *
         * @param {jQuery} jqNode
         *  The jQuery menu node that needs to be updated.
         *
         * @param {Object} [dataObject]
         *  The optional dataObject, whose value property determines
         *  the menu entry to be selected
         *  @param {String} [dataObject.value]
         *      The optional string value, containing the id of the menu entry
         *      to be selected.
         */
        function selectDropdownEntry(jqNode, dataObject) {

            jqNode.find('.selected').removeClass('selected');

            if (dataObject && dataObject.value) {
                jqNode.find('a[data-toggle="dropdown"]').text(dataObject.name);
                jqNode.find('a[data-value="' + dataObject.value + '"]').addClass('selected');
            } else {
                jqNode.find('a[data-toggle="dropdown"]').text(customStr);
            }
        }

        /**
         * converts assigned centimeter to the general unit
         *
         * @param {Number} cm in centimeter
         *
         * @returns {Number}
         *  in general unit
         */
        function cmToUnit(cm) {
            return Utils.convertLength(cm, 'cm', unit, 0.01);
        }

        /**
         * Returns the UI value in mm, cm or inch for the given
         * page value in 100th millimeters.
         *
         * @param {Number} pageValue
         *  The page value to be converted in 100th millimeters.
         *
         * @returns {Number}
         *  The UI value in mm, cm or inch, depending on the used metric
         *  system, rounded to 2 decimal places
         */
        function getUIValue(pageValue) {
            return Utils.convertHmmToLength(pageValue, unit, 0.01);

        }

        /**
         * Returns the page value in 100th millimeters for the given
         * ui value in mm, cm or inch.
         *
         * @param {Number} uiValue
         *  The UI value in mm, cm or inch, depending on the used metric system,
         *
         * @returns {Number}
         *  The page value in 100th millimeters
         */
        function getPageValue(uiValue) {
            return Utils.convertLengthToHmm(uiValue, unit);
        }

        // public methods -----------------------------------------------------

        /**
         * Executes the dialog and performs all necessary steps, if the
         * user ended the dialog by clicking the Ok button
         *
         * @returns {jQuery.Promise}
         *  A promise representing the future result of the dialog.
         */
        this.show = _.wrap(this.show, function (show) {
            return show.call(this).done(function () {

                var newPageAttrs = {};
                _.each(pageAttrs, function (value, key) {
                    if (value !== initialPageAttrs[key]) {
                        newPageAttrs[key] = _.isNumber(value) ? Utils.round(value, 10) : value;
                    }
                });

                if (!_.isEmpty(newPageAttrs)) {
                    if (isPresentationApp) {

                      //docModel.createAndApplyOperations($.noop);
                      //docModel.reformatDocument(newPageAttrs); // process will use its own operations generator.

                        docModel.executeDelayed(function () {
                            this.reformatDocument(newPageAttrs);
                        });

                    } else {
                        docModel.createAndApplyOperations(function (generator) {

                            generator.generateOperation(Operations.SET_DOCUMENT_ATTRIBUTES, { attrs: { page: newPageAttrs } });

                            var paraStyles = docModel.getStyleCollection('paragraph');

                            function createTabStopOp(styleId) {

                                if (paraStyles.isDirty(styleId)) { return; }

                                var styleAttrs = paraStyles.getStyleSheetAttributeMap(styleId);
                                if (styleAttrs.paragraph && styleAttrs.paragraph.tabStops) {

                                    var oldWidth = initialPageAttrs.width - initialPageAttrs.marginLeft - initialPageAttrs.marginRight;
                                    var newWidth = pageAttrs.width - pageAttrs.marginLeft - pageAttrs.marginRight;

                                    _.each(styleAttrs.paragraph.tabStops, function (tabStop) {
                                        var relPos = tabStop.pos / oldWidth;
                                        tabStop.pos = Math.round(relPos * newWidth);
                                    });

                                    generator.generateOperation(Operations.CHANGE_STYLESHEET, { styleId: styleId, attrs: styleAttrs, type: 'paragraph' });
                                }
                            }
                            createTabStopOp(docModel.getDefaultHeaderTextDefinition().styleId);
                            createTabStopOp(docModel.getDefaultFooterTextDefinition().styleId);
                        });
                    }

                }
            }).always(function () {
                jqFormat = null;
                jqOrientation = null;
                jqMargins = null;
                jqWidth = null;
                jqHeight = null;
                jqMarginTop = null;
                jqMarginBottom = null;
                jqMarginLeft = null;
                jqMarginRight = null;
                jqSizeTitle = null;
                jqMarginTitle = null;
                jqTable = null;
                jqTableVertMargins = null;
                jqTableHorzMargins = null;

                formatSettingList = null;
                isPresentationApp = null;

                // blocking keyboard input during applying of operations
                docModel.setBlockKeyboardEvent(false);
                self = docModel = docView = null;
            });
        });

        // initialization -----------------------------------------------------

        var pageAttrs = _.copy(initialPageAttrs, true);

        // close dialog when losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // initialize the body element of the dialog
        var dialogCssCLassName = 'io-ox-office-text-dialog-page-settings';

        if (isPresentationApp) {
            dialogCssCLassName += ' slide-settings';
        }
        this.getBody()
            .addClass(dialogCssCLassName)
            .toggleClass('mobile', !!ox.mobile);

        // create the layout of the dialog
        initControls();

        // set initial control values and initial focus
        this.on('show', function () {
            // blocking keyboard input during applying of operations
            docModel.setBlockKeyboardEvent(true);

            setPageValue(jqWidth, 'width', pageAttrs.width);
            setPageValue(jqHeight, 'height', pageAttrs.height);

            if (!isPresentationApp) {

                setPageValue(jqMarginLeft, 'marginLeft', pageAttrs.marginLeft);
                setPageValue(jqMarginTop, 'marginTop', pageAttrs.marginTop);
                setPageValue(jqMarginRight, 'marginRight', pageAttrs.marginRight);
                setPageValue(jqMarginBottom, 'marginBottom', pageAttrs.marginBottom);
            }
            // us 102078426: dialogs on small devices should come up w/o keyboard,
            // so the focus shoud not be in an input element
            if (Utils.SMALL_DEVICE) {
                self.getOkButton().focus();
            } else {
                jqWidth.find('input').focus();
            }
        });

    } // class PageSettingsDialog

    // exports ================================================================

    // derive this class from class Dialogs.ModalDialog
    return Dialogs.ModalDialog.extend({ constructor: PageSettingsDialog });

});
