/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/view/controls', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/utils/paragraphspacing',
    'io.ox/office/editframework/view/editcontrols',
    'io.ox/office/textframework/view/labels',
    'io.ox/office/textframework/view/control/paragraphstylepicker',
    'io.ox/office/textframework/view/control/tablestylepicker',
    'io.ox/office/textframework/view/control/liststylepicker',
    'io.ox/office/textframework/view/control/changetrackbadge',
    'io.ox/office/textframework/view/control/drawingpositionpicker',
    'io.ox/office/textframework/view/control/commentfiltergroup',
    'io.ox/office/textframework/view/control/fieldformatlist',
    'io.ox/office/baseframework/view/control/compoundsplitbutton',
    'gettext!io.ox/office/textframework/main'
], function (Utils, LineHeight, ParagraphSpacing, EditControls, Labels, ParagraphStylePicker, TableStylePicker, ListStylePicker, ChangeTrackBadge, DrawingPositionPicker, CommentFilterGroup, FieldFormatList, CompoundSplitButton, gt) {

    'use strict';

    // static class TextControls ==============================================

    /**
     * Additional classes defining specialized GUI controls for the OX Text
     * application.
     *
     * @extends EditControls
     */
    var TextControls = _.extend({}, EditControls, {
        ParagraphStylePicker: ParagraphStylePicker,
        TableStylePicker: TableStylePicker,
        ChangeTrackBadge: ChangeTrackBadge,
        DrawingPositionPicker: DrawingPositionPicker,
        CommentFilterGroup: CommentFilterGroup,
        FieldFormatList: FieldFormatList
    });

    // class ParagraphAlignmentGroup ==========================================

    /**
     * A picker control for horizontal text alignment in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParagraphAlignmentGroup = EditControls.RadioGroup.extend({ constructor: function (initOptions) {

        EditControls.RadioGroup.call(this, Utils.extendOptions({
            tooltip: Labels.PARA_ALIGNMENT_LABEL
        }, initOptions));

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().attr({ 'aria-label': Labels.PARA_ALIGNMENT_LABEL, 'aria-labelledby': 'dummy' });

        _.each(Labels.PARA_ALIGNMENT_STYLES, function (style) {
            this.createOptionButton(style.value, { icon: style.icon, tooltip: style.label });
        }, this);

    } }); // class ParagraphAlignmentGroup

    // class ParagraphAlignmentPicker =========================================

    /**
     * A picker control for horizontal text alignment in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParagraphAlignmentPicker = EditControls.RadioList.extend({ constructor: function (initOptions) {

        EditControls.RadioList.call(this, Utils.extendOptions({
            icon: 'docs-para-align-left',
            tooltip: Labels.PARA_ALIGNMENT_LABEL,
            updateCaptionMode: 'icon'
        }, initOptions));

        _.each(Labels.PARA_ALIGNMENT_STYLES, function (style) {
            this.createOptionButton(style.value, { icon: style.icon, label: style.label });
        }, this);

    } }); // class ParagraphAlignmentPicker

    // class TextPositionGroup ================================================

    /**
     * A button group control for text position (subscript/superscript).
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.TextPositionGroup = EditControls.RadioGroup.extend({ constructor: function (initOptions) {

        EditControls.RadioGroup.call(this, Utils.extendOptions({
            tooltip: Labels.TEXT_POSITION_LABEL,
            toggleValue: 'baseline'
        }, initOptions));

        // set accessible name - WCAG 2.0 Success Criterion 4.1.2
        this.getNode().attr({ 'aria-label': Labels.TEXT_POSITION_LABEL, 'aria-labelledby': 'dummy' });

        this.createOptionButton('sub',   { icon: 'docs-font-subscript',   tooltip: gt('Subscript'),   dropDownVersion: { label: gt('Subscript') } })
            .createOptionButton('super', { icon: 'docs-font-superscript', tooltip: gt('Superscript'), dropDownVersion: { label: gt('Superscript') } });

    } }); // class TextPositionGroup

    // class LineHeightPicker =================================================

    /**
     * A picker control for the line height in paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.LineHeightPicker = EditControls.RadioGroup.extend({ constructor: function () {

        EditControls.RadioGroup.call(this, {
            dropDownVersion: { label: Labels.LINE_SPACING_LABEL }
        });

        this.createOptionButton(LineHeight.SINGLE,   { icon: 'docs-para-line-spacing-100', label: /*#. text line spacing in paragraphs */ gt('100%') })
            .createOptionButton(LineHeight._115,     { icon: 'docs-para-line-spacing-115', label: /*#. text line spacing in paragraphs */ gt('115%') })
            .createOptionButton(LineHeight.ONE_HALF, { icon: 'docs-para-line-spacing-150', label: /*#. text line spacing in paragraphs */ gt('150%') })
            .createOptionButton(LineHeight.DOUBLE,   { icon: 'docs-para-line-spacing-200', label: /*#. text line spacing in paragraphs */ gt('200%') });

    } }); // class LineHeightPicker

    // class ParagraphSpacingPicker ===========================================

    /**
     * A picker control for the 'marginBottom' controlled spacing
     * in between two or more selected paragraphs.
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    TextControls.ParagraphSpacingPicker = EditControls.RadioGroup.extend({ constructor: function () {

        EditControls.RadioGroup.call(this, {
            dropDownVersion: { label: Labels.PARAGRAPH_SPACING_LABEL }
        });

        var multiplier = ParagraphSpacing.setting.multiplier;

        this.createOptionButton(multiplier.NONE,   { label: /*#. no distance at all in between two or more selected paragraphs */ gt('None') })
            .createOptionButton(multiplier.NORMAL, { label: /*#. a normal distance in between two or more selected paragraphs  */ gt('Normal') })
            .createOptionButton(multiplier.WIDE,   { label: /*#. a large distance in between two or more selected paragraphs   */ gt('Wide') });

    } }); // class ParagraphSpacingPicker

    // class BulletListStylePicker ============================================

    TextControls.BulletListStylePicker = ListStylePicker.extend({ constructor: function (docView, docModel, initOptions) {

        ListStylePicker.call(this, docView, docModel.getPredefinedBulletListStyles(), Utils.extendOptions({
            icon: 'docs-list-bullet',
            tooltip: gt('Bullet list'),
            gridColumns: 4,
            dropDownVersion: { label: gt('Bullet list') }
        }, initOptions));

    } }); // class BulletListStylePicker

    // class NumberedListStylePicker ==========================================

    TextControls.NumberedListStylePicker = ListStylePicker.extend({ constructor: function (docView, docModel, initOptions) {

        ListStylePicker.call(this, docView, docModel.getPredefinedNumberedListStyles(), Utils.extendOptions({
            icon: 'docs-list-numbered',
            tooltip: gt('Numbered list'),
            gridColumns: 4,
            dropDownVersion: { label: gt('Numbered list') }
        }, initOptions));

    } }); // class NumberedListStylePicker

    // class UserFilterCompoundButton =========================================

    /**
     * Drop-down menu control for setting a user filter for the comments in the
     * comments layer.
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.UserFilterCompoundButton = EditControls.CompoundButton.extend({ constructor: function (docView) {

        // base constructor ---------------------------------------------------

        EditControls.CompoundButton.call(this, docView, {
            label: gt('Comment authors'),
            tooltip: gt('Show comment authors'),
            anchorBorder: 'right left bottom top',
            updateCaptionMode: 'icon',
            smallerVersion: { hideLabel: true }
        });

        // initialization -----------------------------------------------------

        this.addGroup('comment/authorFilter', new CommentFilterGroup(docView), { sticky: true });

    } }); // class UserFilterCompoundButton

    // class CommentDisplayModePicker ============================================

    /**
     * Drop-down menu control for modifying the highlight mode and visibility
     * of the comments in the comments layer.
     * Additionally a list with all comment authors can be used to filter
     * specific authors.
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.CommentDisplayModePicker = EditControls.CompoundButton.extend({ constructor: function (docView, options) {

        var smallDevice = Utils.getBooleanOption(options, 'smallDevice', false),
            // the group of all display mode buttons
            displayModeGroup = new EditControls.RadioGroup();

        // base constructor ---------------------------------------------------

        EditControls.CompoundButton.call(this, docView, {
            icon: 'docs-comment-show',
            label: gt('Markup'),
            tooltip: gt('Highlight comments mode'),
            updateCaptionMode: 'icon',
            smallerVersion: { hideLabel: true }
        });

        // private methods ----------------------------------------------------

        /**
         * Updates the icon after modifying the display mode.
         */
        function displayModeUpdateHandler(newValue, oldValue) {

            if (newValue === oldValue) {  }

            // TODO: Modify the icon
        }

        // initialization -----------------------------------------------------

        if (!smallDevice) {
            displayModeGroup
                .createOptionButton('all',      { label: gt('Highlight all comments') })
                .createOptionButton('selected', { label: gt('Highlight selected comments') });
        }

        displayModeGroup
            .createOptionButton('bubbles',  { label: gt('Show bubble comments') })
            // .createOptionButton('none',  { label: gt('Highlight no comments') })
            .createOptionButton('hidden',   { label: gt('Show no comments') });

        displayModeGroup.registerUpdateHandler(displayModeUpdateHandler);

        this.addGroup('comment/displayMode', displayModeGroup)
            .addSeparator()
            // adding the button, that contains a list of all users who created comments in the document
            .addGroup(null, new TextControls.UserFilterCompoundButton(docView));

    } }); // class CommentDisplayModePicker

    // class CommentDisplayViewPicker ============================================

    /**
     * Drop-down menu control for modifying the view of the comments layer. This is
     * only planned for testing purposes and can be removed in release version.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.CommentDisplayViewPicker = EditControls.RadioList.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        EditControls.RadioList.call(this, {
            icon: 'fa-cogs',
            label: _.noI18n('Comment view'),
            tooltip: _.noI18n('Set different comment view mode'),
            updateCaptionMode: 'icon',
            smallerVersion: {
                hideLabel: true
            }
        });

        // initialization -----------------------------------------------------

        this.createOptionButton('modern',     { icon: 'fa-indent',  label: _.noI18n('Modern') })
            .createOptionButton('avantgard',  { icon: 'fa-outdent', label: _.noI18n('Avantgard') })
            .createOptionButton('classic',    { icon: 'fa-list-alt',  label: _.noI18n('Classic') });

    } }); // class CommentDisplayViewPicker

    // class AnchoragePicker ============================================

    /**
     *
     * Drop-down menu control for position and text floating of drawing
     * objects in text documents.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.AnchoragePicker = EditControls.RadioList.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        EditControls.RadioList.call(this, {
            icon: 'docs-position-anchor',
            label: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Position'),
            tooltip: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Position'),
            updateCaptionMode: 'icon',
            smallerVersion: {
                hideLabel: true
            }
        });

        // initialization -----------------------------------------------------

        this.createOptionButton('inline',     { icon: 'docs-position-on-text',         label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with text') })
            .createOptionButton('paragraph',  { icon: 'docs-position-on-paragraph',    label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with paragraph') })
            .createOptionButton('margin',     { icon: 'docs-position-on-page-margin',  label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with page margin') })
            .createOptionButton('page',       { icon: 'docs-position-on-page',         label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Move with page') });
    } }); // class AnchoragePicker

    // class DrawingOrder ============================================

    /**
     *
     * Drop-down menu control for z-order of drawing
     * objects in text documents.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.DrawingOrder = EditControls.CompoundButton.extend({ constructor: function (docView, options) {

        // base constructor ---------------------------------------------------

        EditControls.CompoundButton.call(this, docView, Utils.extendOptions(options, { label: /*#. button for drawing objects z-index (depth), is used for overlapping objects to change their order */ gt.pgettext('drawing-pos', 'Reorder'), smallerVersion: { hideLabel: true } }));

        // initialization -----------------------------------------------------

        this.addGroup('drawing/order', new TextControls.Button({ value: 'front',       tooltip: /*#. button for drawing objects z-index (depth)*/ gt.pgettext('drawing-pos', 'Bring object to the front'),                  /*#. button for drawing objects z-index (depth)*/ label: gt.pgettext('drawing-pos', 'Bring to front') }))
            .addGroup('drawing/order', new TextControls.Button({ value: 'forward',     tooltip: /*#. button for drawing objects z-index (depth)*/ gt.pgettext('drawing-pos', 'Move object up one placement to the front'),  /*#. button for drawing objects z-index (depth)*/ label: gt.pgettext('drawing-pos', 'Bring forward') }))
            .addGroup('drawing/order', new TextControls.Button({ value: 'backward',    tooltip: /*#. button for drawing objects z-index (depth)*/ gt.pgettext('drawing-pos', 'Send object one placement to the back'),      /*#. button for drawing objects z-index (depth)*/ label: gt.pgettext('drawing-pos', 'Send backward') }))
            .addGroup('drawing/order', new TextControls.Button({ value: 'back',        tooltip: /*#. button for drawing objects z-index (depth)*/ gt.pgettext('drawing-pos', 'Send object to the back'),                    /*#. button for drawing objects z-index (depth)*/ label: gt.pgettext('drawing-pos', 'Send to back') }));
    } });

    // class DrawingPositionPicker ============================================

    /**
     *
     * Drop-down menu control for position and text floating of drawing
     * objects in text documents.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.DrawingPositionPicker = EditControls.RadioList.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        EditControls.RadioList.call(this, {
            icon: 'docs-drawing-inline',
            label: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Alignment'),
            tooltip: /*#. alignment and text floating of drawing objects in text documents */ gt.pgettext('drawing-pos', 'Drawing alignment'),
            updateCaptionMode: 'icon',
            smallerVersion: {
                hideLabel: true
            }
        });

        // initialization -----------------------------------------------------

        this//.createMenuSection('inline')
            // .createOptionButton('inline',       { section: 'inline', icon: 'docs-drawing-inline',       label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Inline with text') })
            .createMenuSection('left')
            .createOptionButton('left:none',    { section: 'left',   icon: 'docs-drawing-left-none',    label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Left aligned, no text wrapping') })
            .createOptionButton('left:right',   { section: 'left',   icon: 'docs-drawing-left-right',   label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Left aligned, text wraps at right side') })
            .createMenuSection('right')
            .createOptionButton('right:none',   { section: 'right',  icon: 'docs-drawing-right-none',   label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Right aligned, no text wrapping') })
            .createOptionButton('right:left',   { section: 'right',  icon: 'docs-drawing-right-left',   label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Right aligned, text wraps at left side') })
            .createMenuSection('center')
            .createOptionButton('center:none',  { section: 'center', icon: 'docs-drawing-center-none',  label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Centered, no text wrapping') })
            .createOptionButton('center:left',  { section: 'center', icon: 'docs-drawing-center-left',  label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Centered, text wraps at left side') })
            .createOptionButton('center:right', { section: 'center', icon: 'docs-drawing-center-right', label: /*#. drawing object position in paragraph */ gt.pgettext('drawing-pos', 'Centered, text wraps at right side') });

    } }); // class DrawingPositionPicker

    // class InTextStyleContextSubMenu ========================================

    /**
     *
     * @constructor
     *
     * @extends CompoundButton
     */
    TextControls.InTextStyleContextSubMenu = EditControls.CompoundButton.extend({ constructor: function (docView, initOptions) {

        var self = this;
        var docModel = docView.getDocModel();
        var changeStyleButton = null;

        // base constructor ---------------------------------------------------

        EditControls.CompoundButton.call(this, docView, Utils.extendOptions({
            label: Labels.PARAGRAPH_STYLE_LABEL
        }, initOptions));

        // initialization -----------------------------------------------------

        this.addGroup('paragraph/createstylesheet', new TextControls.Button({ label: Labels.CREATE_STYLE_LABEL, tooltip: Labels.CREATE_STYLE_LABEL_TOOLTIP }));

        this.on('group:focus', function () {
            var styleId = docModel.getAttributes('paragraph').styleId;

            //getTranslatedNameByStyleId call of StyleSheetMixin
            changeStyleButton = new TextControls.Button({
                label: gt('Change style "%1$s"', docModel.getTranslatedNameByStyleId(styleId)),
                tooltip: Labels.CHANGE_STYLE_LABEL_TOOLTIP,
                value: styleId
            });

            self.addGroup('paragraph/changestylesheet', changeStyleButton);
        });

        this.on('group:blur', function () {
            if (changeStyleButton) {
                self.removeGroup(changeStyleButton);
                changeStyleButton = null;
            }
        });

    } }); // class InTextStyleContextSubMenu

    // class InsertFieldPicker =================================================

    /**
     * Control to pick field from dropdown
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.InsertFieldPicker = EditControls.RadioList.extend({ constructor: function () {

        EditControls.RadioList.call(this, {
            icon: 'docs-field',
            label: gt('Field'),
            tooltip: gt('Insert field'),
            smallerVersion: {
                hideLabel: true
            }
        });

        this.createOptionButton('pagenumber',   { label: /*#. insert page number */ gt('Page number') })
            .createOptionButton('numpages',     { label: /*#. insert number of pages in document */ gt('Page count') })
            .createOptionButton('date',         { label: /*#. insert current date and time */ gt('Date & time') })
            .createOptionButton('filename',     { label: /*#. insert document name */ gt('Document name') })
            .createOptionButton('author',       { label: /*#. insert author's name */ gt('Author name') });

    } }); // class InsertFieldPicker

    // class PresentationDateNumberPicker =================================================

    /**
     * Control to pick field from dropdown
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.PresentationDateNumberPicker = EditControls.RadioList.extend({ constructor: function () {

        EditControls.RadioList.call(this, {
            icon: 'docs-field',
            label: gt('Field'),
            tooltip: gt('Insert field'),
            smallerVersion: {
                hideLabel: true
            }
        });

        this.createOptionButton('slidenum', { label: /*#. insert slide number */ gt('Slide number') })
            .createOptionButton('datetime', { label: /*#. insert current date and time */ gt('Date & time') });

    } }); // end of class PresentationDateNumberPicker

    // class VerticalAlignmentPicker ============================================

    /**
     * Drop-down menu control for modifying the view of the comments layer. This is
     * only planned for testing purposes and can be removed in release version.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.VerticalAlignmentPicker = EditControls.RadioList.extend({ constructor: function () {

        // base constructor ---------------------------------------------------

        EditControls.RadioList.call(this, {
            icon: 'fa-cogs',
            label: gt('Vertical alignment'),
            tooltip: gt('Set the vertical text alignment'),
            updateCaptionMode: 'icon',
            smallerVersion: {
                hideLabel: true
            }
        });

        // initialization -----------------------------------------------------

        this.createOptionButton('top',      { icon: 'fa-indent',   label: gt('Top') })
            .createOptionButton('centered', { icon: 'fa-outdent',  label: gt('Centered') })
            .createOptionButton('bottom',   { icon: 'fa-list-alt', label: gt('Bottom') });

    } }); // class VerticalAlignmentPicker

    // class DrawingAlignment ============================================

    /**
     *
     * Drop-down menu control for the alignment of selected drawings in text/presentation documents.
     *
     * @constructor
     *
     * @extends RadioList
     */
    TextControls.DrawingAlignment = EditControls.CompoundButton.extend({ constructor: function (docView, options) {

        // base constructor ---------------------------------------------------

        EditControls.CompoundButton.call(this, docView, Utils.extendOptions(options, { label: /*#. button to change the alignment of the selected drawing/s */ gt.pgettext('drawing-pos', 'Align'), smallerVersion: { hideLabel: true } }));

        // initialization -----------------------------------------------------

        this.addGroup('drawing/align', new TextControls.Button({ value: 'left',     /*#. button to change the drawing/s to left alignment*/ label: gt.pgettext('drawing-pos', 'Align left') }))
            .addGroup('drawing/align', new TextControls.Button({ value: 'center',   /*#. button to change the drawing/s to center alignment*/ label: gt.pgettext('drawing-pos', 'Align center') }))
            .addGroup('drawing/align', new TextControls.Button({ value: 'right',    /*#. button to change the drawing/s to right alignment*/ label: gt.pgettext('drawing-pos', 'Align right') }))
            .addGroup('drawing/align', new TextControls.Button({ value: 'top',      /*#. button to change the drawing/s to top alignment*/ label: gt.pgettext('drawing-pos', 'Align top') }))
            .addGroup('drawing/align', new TextControls.Button({ value: 'middle',   /*#. button to change the drawing/s to middle alignment*/ label: gt.pgettext('drawing-pos', 'Align middle') }))
            .addGroup('drawing/align', new TextControls.Button({ value: 'bottom',   /*#. button to change the drawing/s to bottom alignment*/ label: gt.pgettext('drawing-pos', 'Align bottom') }));
    } }); // class DrawingAlignment

    // exports ================================================================

    return TextControls;

});
