/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define.async('io.ox/office/textframework/model/numberformatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/locale/formatter'
], function (Utils, LocaleData, Parser, Formatter) {

    'use strict';

    // all predefined number format codes, mapped by category
    var categoryCodesDatabase = {};

    // load the localized number format definitions
    var numberFormatsPromise = LocaleData.loadResource('io.ox/office/textframework/resource/numberformats').done(function (data) {
        categoryCodesDatabase = data.categories;
    });

    // class NumberFormatter ==================================================

    /**
     * A number formatter for a text document.
     *
     * @constructor
     *
     * @extends Formatter
     *
     * @param {TextModel} docModel
     *  The document model containing this instance.
     */
    var NumberFormatter = Formatter.extend({ constructor: function (docModel) {

        // the file format may influence number formats
        var fileFormat = docModel.getApp().getFileFormat();

        // base constructor ---------------------------------------------------

        Formatter.call(this);

        // public methods -----------------------------------------------------

        /**
         * Returns the parsed number format descriptor for the specified number
         * format.
         *
         * @param {String} formatCode
         *  The format code to be parsed.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.grammarId='op']
         *      The identifier of a format grammar. See description of the
         *      static method Parser.parseFormatCode() for details.
         *
         * @returns {ParsedFormat}
         *  The parsed number format descriptor of a number format code.
         */
        this.getParsedFormat = function (formatCode, options) {
            var grammarId = Utils.getStringOption(options, 'grammarId', 'op');
            return Parser.parseFormatCode(fileFormat, grammarId, formatCode);
        };

        /**
         * Converts decimal into Roman number format.
         *
         * @param {Number} number
         *  Decimal number.
         *
         * @returns {String}
         *  Converted decimal number into string with Roman formatting.
         */
        this.toRoman = function (number) {
            var romanNumber = '',
                decimalTemplates = [1000, 900, 500, 400, 100, 90, 50, 40, 10, 9, 5, 4, 1],
                dtLength = decimalTemplates.length,
                romanLetters = ['M', 'CM', 'D', 'CD', 'C', 'XC', 'L', 'XL', 'X', 'IX', 'V', 'IV', 'I'];

            for (var i = 0; i < dtLength; i++) {
                while (number >= decimalTemplates[i]) {
                    romanNumber += romanLetters[i];
                    number -= decimalTemplates[i];
                }
            }
            return romanNumber;
        };

        /**
         * Converts decimal into alphabetic letter format.
         *
         * @param {Number} number
         *  Decimal number.
         *
         * @returns {String}
         *  Converted decimal number into string with alphabetic formatting.
         */
        this.toAlphabetic = function (number) {
            var alphabet = '',
                multiAlphabet = '',
                charArray = 'abcdefghijklmnopqrstuvwxyz',
                factor = null;

            if (!_.isNumber(number) || number < 1) {
                Utils.error('numberFormatter.toAlphabetic(): argument is not valid number: ', number);
                return;
            }
            if (number > 26) {
                factor = Math.floor(number / 26);
                number %= 26;
            }
            alphabet = charArray[number - 1]; // Zero based index of charArray!

            if (factor) {
                for (var i = 0; i <= factor; i++) {
                    multiAlphabet += alphabet;
                }
                return multiAlphabet;
            }
            return alphabet;
        };

        /**
         * Formats decimal number to have one dash before and one after number,
         * separated by one whitespace. Example: 1 to '- 1 -'.
         *
         * @param {Number} number
         *  Decimal number.
         *
         * @returns {String}
         *  Converted decimal number into string with dash formatting.
         */
        this.toArabicDash = function (number) {
            return '- ' + number + ' -';
        };

        /**
         * Converts decimal number to hexadecimal.
         *
         * @param {Number} number
         *  Number to be converted to hexadecimal.
         * @returns {String|null}
         *  Converted hex string.
         */
        this.decToHex = function (number) {
            if (!_.isNumber(number)) {
                Utils.error('numberFormatter.decToHex(): argument is not valid number: ', number);
                return;
            }
            return number.toString(16);
        };

        /**
         * Converts hexadecimal string to decimal number.
         *
         * @param {String} hexString
         *  String to be converted into decimal.
         * @returns {Number}
         *  Converted hex string to decimal number.
         */
        this.hexToDec = function (hexString) {
            return parseInt(hexString, 16);
        };

        /**
         * Converts decimal number to ordinal number in string format.
         *
         * @param {Number} number
         *  Number to be converted to ordinal.
         * @returns {String|null}
         *  Converted ordinal string.
         */
        this.toOrdinal = function (number) {
            if (!_.isNumber(number)) {
                Utils.error('numberFormatter.toOrdinal(): argument is not valid number: ', number);
                return;
            }
            return number + '.';
        };

        /**
         * Returns all available number format codes for the specified format
         * category and current UI language.
         *
         * @param {String} category
         *  The identifier of the number format category.
         *
         * @returns {Array|Null}
         *  The format codes for the specified category, or null if no format
         *  codes have been defined for the category. Each element in the
         *  returned array is an object with the following properties:
         *  - {String} value
         *      The actual format code.
         *  - {String} label
         *      A caption label with a formatted example number, intended to be
         *      used in GUI form controls.
         *  - {Boolean} [red=false]
         *      If set to true, the format code contains a [RED] color modifier
         *      for negative numbers.
         */
        this.getCategoryCodes = function (category) {
            var categoryFormats = categoryCodesDatabase[category.toLocaleLowerCase()];
            return _.isArray(categoryFormats) ? _.copy(categoryFormats, true) : null;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } }); // class NumberFormatter

    // exports ================================================================

    return numberFormatsPromise.then(_.constant(NumberFormatter));

});
