/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/clipboardhandlermixin', [
    'io.ox/office/tk/io',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/textframework/components/hyperlink/hyperlink',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/export',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/snapshot',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/drawinglayer/view/imageutil',
    'gettext!io.ox/office/textframework/main'
], function (IO, AttributeUtils, Border, Hyperlink, HyperlinkUtils, DOM, Export, Operations, Position, Snapshot, Utils, ImageUtil, gt) {

    'use strict';

    // determines whether the origin of the clipboard data is OX Text or OX Presentation
    var clipboardOrigin = '';

    // mix-in class ClipBoardHandlerMixin ======================================

    /**
     * A mix-in class for the clip board specific functions.
     *
     * @constructor
     */
    function ClipBoardHandlerMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Check, whether the clipboard operations contain only drawings (or content
         * inside drawings).
         * In the case of internal clipboard, a mixture of drawing and text selection
         * is not possible. Therefore the clipboard contains only drawings, if at least
         * one 'insertDrawing' operation is included in the list of operations. This
         * can include text operations in the drawing (text frame).
         * In the case of the external clipboard it is possible, that there is a text
         * selection next to a drawing selection. But this is not handled within this
         * function.
         *
         * @returns {Boolean}
         *  Whether only drawings are top level elements in the clip board.
         */
        function checkClipboardForDrawings(operations) {
            return _.filter(operations, function (operation) { return operation.name === Operations.DRAWING_INSERT; }).length > 0;
        }

        /**
         * Returns true if the given operations are only insertDrawing of type image.
         *
         * @param {Array} operations
         *  An array with the clipboard operations.
         *
         * @returns {Boolean}
         *  Whether the operations are only insertDrawing of type image.
         */
        function isOnlyInsertImage(operations) {
            return _.every(operations, function (operation) { return operation.name === Operations.DRAWING_INSERT && operation.type === 'image'; });
        }

        /**
         * Returns true if the array begins with the sub array.
         *
         * @param {Array} array
         *  The array to check if it begins with the sub array.
         *
         * @param {Array} sub
         *  The sub array to check the given array with.
         *
         * @returns {Boolean}
         *  Whether the array starts with the sub array.
         */
        function arrayStartsWith(array, sub) {
            if (!_.isArray(array) || !_.isArray(sub)) { return false; }
            return _.every(sub, function (v, i) {
                return array[i] === v;
            });
        }

        /**
         * Returns an array with all drawing nodes in the selection.
         *
         * @returns {Array}
         *  An array with all drawing nodes in the selection.
         */
        function getSelectedDrawingNodes() {
            // the selection object
            var selection = self.getSelection();

            if (selection.isMultiSelectionSupported() && selection.isMultiSelection()) {
                return selection.getAllSelectedDrawingNodes();
            } else {
                return [selection.getClosestSelectedDrawing()];
            }
        }

        /**
         * Returns all drawing nodes that are present on the currently selected slide.
         *
         * @returns {Array}
         *  An array with all drawing nodes of the selected slide.
         */
        function getDrawingNodesOfCurrentSlide() {
            // the selection object
            var selection = self.getSelection();
            // the slide position
            var slidePos = [_.first(selection.getStartPosition())];
            // the slide node
            var slide = Position.getParagraphElement(self.getRootNode(), slidePos);

            return $(slide).children(DOM.ABSOLUTE_DRAWING_SELECTOR);
        }

        /**
         * Returns the drawing attributes for the given drawing node.
         *
         * @param {jQuery|Node} drawingNode
         *  The drawing node that will get new values assigned.
         *
         * @returns {Object}
         *  The attribute set if existing, otherwise an empty object.
         */
        function getDrawingAttributes(drawingNode) {
            var attrs = null;

            if (self.useSlideMode() && self.isPlaceHolderDrawing(drawingNode)) {
             // the merged attributes of the specified place holder drawing
                attrs = self.getDrawingStyles().getElementAttributes(drawingNode);
            } else {
                // the explicit attributes at the specified drawing
                attrs = AttributeUtils.getExplicitAttributes(drawingNode);
            }

            return attrs;
        }

        /**
         * Calculates the minimum top/left position of the selected drawings.
         *
         * @returns {Object}
         *  An object with the minimum top and left position.
         */
        function getMinPositionForSelectedDrawings() {
            return getMinPositionForDrawings(getSelectedDrawingNodes());
        }

        /**
         * Calculates the minimum top/left position of the drawing selection
         * defined by the given drawing nodes.
         *
         * @param {Array} drawingNodes
         *  The selected drawing nodes.
         *
         * @returns {Object}
         *  An object with the minimum top and left position.
         */
        function getMinPositionForDrawings(drawingNodes) {
            var position = {};

            _.each(drawingNodes, function (node) {
                var attrs = getDrawingAttributes(node);
                if (attrs && attrs.drawing) {
                    // exchanges the top, left position if they are smaller than the current
                    if ((attrs.drawing.top > 0) && (!('top' in position) || (attrs.drawing.top < position.top))) {
                        position.top = attrs.drawing.top;
                    }
                    if ((attrs.drawing.left > 0) && (!('left' in position) || (attrs.drawing.left < position.left))) {
                        position.left = attrs.drawing.left;
                    }
                }
            });

            // make sure to have defaults
            return _.extend({ top: 0, left: 0 }, position);
        }

        /**
         * Moving the pasted drawing, so that it does not cover the original drawing
         * completely.
         */
        function shiftDrawingPositions(operations, anchor) {
            // the top and left offset to place the drawings with
            var offset = 500;
            // the position to place the next drawing when pasted from OX Text
            var currentPosition = anchor || { top: 0, left: 0 };
            // all drawings on the slide
            var allDrawings = getDrawingNodesOfCurrentSlide();
            // all attributes of the drawings on the slide
            var allDrawingAttrs = _.map(allDrawings, getDrawingAttributes);

            // returns true if the given operation is a drawing and not inside a group.
            function isRootLevelDrawing(operation) {
                return (operation && operation.name === Operations.DRAWING_INSERT && operation.start && operation.start.length === 2);
            }

            // returns true if the given position overlaps one of the existing drawings
            function isPositionOverlappingExistingDrawing(position) {
                // check the attributes of all drawings if position overlaps
                var result = _.any(allDrawingAttrs, function (attrs) {
                    var top = attrs.drawing.top;
                    var left = attrs.drawing.left;
                    var bottom = attrs.drawing.top + offset;
                    var right = attrs.drawing.left + offset;
                    // the overlapping rectangle is defined by the position plus an offset in width and height
                    return (position.top >= top && position.top < bottom && position.left >= left && position.left < right);
                });

                return result;
            }

            // get a drawing position that doesn't overlap existing drawings
            function getNonOverlappingDrawingPosition(operation) {
                var position = null;

                if (self.isClipboardOriginPresentation()) {
                    position = {
                        top:  operation.attrs && operation.attrs.drawing && operation.attrs.drawing.top || 0,
                        left: operation.attrs && operation.attrs.drawing && operation.attrs.drawing.left || 0
                    };
                } else {
                    position = currentPosition;
                    currentPosition.left += offset;
                    currentPosition.top  += offset;
                }

                while (isPositionOverlappingExistingDrawing(position)) {
                    position.top += offset;
                    position.left += offset;
                }

                return position;
            }

            // shift positions of insert drawing operations
            _.each(operations, function (operation) {
                var position = null;

                if (isRootLevelDrawing(operation)) {
                    // new non overlapping position
                    position = getNonOverlappingDrawingPosition(operation);
                    // set new position
                    if (position && operation.attrs && operation.attrs.drawing) {
                        operation.attrs.drawing.top = position.top;
                        operation.attrs.drawing.left = position.left;
                    }
                }
            });
        }

        /**
         * Transforms OX Text operations to be used with slide mode, e.g. OX Presentation.
         *
         * @param {Array} operations
         *  An array with the clipboard operations.
         *
         * @returns {Array}
         *  An array with the resulting operations.
         */
        function transformOperationsToSlideMode(operations) {
            // the operations generator
            var generator = self.createOperationsGenerator();
            // the selection object
            var selection = self.getSelection();
            // whether the selected drawing is a drawing that contains place holder attributes
            var isPlaceHolderDrawingSelected = self.isPlaceHolderDrawing(selection.getAnyTextFrameDrawing({ forceTextFrame: true }));
            // the default width of the textframe in hmm
            var defaultWidth = 7500;
            // the default drawing attributes
            var drawingAttrs = Utils.extendOptions({ width: defaultWidth }, self.getInsertDrawingAttibutes());
            // the default border attributes
            var lineAttrs = { color: { type: 'rgb', value: '000000' }, style: 'single', type: 'none', width: Border.getWidthForPreset('thin') };
            // the default fill color attributes
            var fillAttrs = { color: { type: 'auto' }, type: 'solid' };
            // the default attributes
            var attrs = { drawing: drawingAttrs, shape: { autoResizeHeight: true }, line: lineAttrs, fill: fillAttrs };
            // target for operation - if exists, it's for ex. header or footer
            var target = self.getActiveTarget();
            // the current operation to transform
            var operation = null;
            // the newly created operation
            var newOperation = null;
            // the index of the currently transformed operation
            var index = 0;
            // the resulting slide mode operations
            var resultOperations = [];
            // the new base position for root level operations
            var rootLevelStart = [0, 0];
            // the new base position for drawings
            var drawingStart = [0, 0];
            // the new base position for text frames
            var textFrameStart = [0, 0];
            // the positions of group drawings
            var groupPositions = [];
            // the new base position for groups
            var groupStart = [0, 0];
            // the list definitions taken from insertListStyle operations
            var listDefinitions = {};
            // the operations that need to be deleted finally in order to remove replacement spaces
            var replacementDeleteOperations = [];

            // check for operations first
            if (!operations || operations.length < 1) { return []; }
            // do nothing if paste target is not OX Text
            if (!self.useSlideMode()) { return operations; }
            // adding styleId (only for ODT)
            if (app.isODF()) { attrs.styleId = 'Frame'; }

            // transforms the operation positions for a top level paste.
            function transformPositionsToTopLevel(operation) {
                var newOperation = _.clone(operation);

                if (operation.start) {
                    // for root level operations just add the start prefix
                    if (operation.start.length < 3) {
                        newOperation.start = rootLevelStart.concat(operation.start);

                        if (operation.end) {
                            newOperation.end = rootLevelStart.concat(operation.end);
                        }

                    } else {
                        // for deeper level operations replace the start prefix
                        newOperation.start.splice(0, 2);
                        newOperation.start = textFrameStart.concat(newOperation.start);

                        if (operation.end) {
                            newOperation.end.splice(0, 2);
                            newOperation.end = textFrameStart.concat(newOperation.end);
                        }
                    }
                }

                return newOperation;
            }

            // transforms the operation positions for a group paste.
            function transformPositionsToGroup(operation) {
                var newOperation = _.clone(operation);

                if (operation.start) {
                    // replace the start prefix
                    newOperation.start.splice(0, 2);
                    //newOperation.start = drawingStart.concat(newOperation.start);
                    newOperation.start = groupStart.concat(newOperation.start);

                    if (operation.end) {
                        newOperation.end.splice(0, 2);
                        //newOperation.end = drawingStart.concat(newOperation.end);
                        newOperation.end = groupStart.concat(newOperation.end);
                    }
                }

                return newOperation;
            }

            // returns true if the operation is nested inside a group drawing
            function isInsideGroup(operation) {
                if (!operation || !operation.start || groupPositions.length < 1) { return false; }
                // check if operation.start begins with one of the collected group positions
                return _.some(groupPositions, function (groupPosition) {
                    return arrayStartsWith(operation.start, groupPosition);
                });
            }

            // returns true if the operation is a direct child of the group
            function isDirectGroupChild(operation) {
                if (!operation || !operation.start || groupPositions.length < 1) { return false; }

                // check if operation.start begins with one of the collected group positions
                return _.some(groupPositions, function (groupPosition) {
                    if (!arrayStartsWith(operation.start, groupPosition)) { return false; }
                    return (operation.start.length === (groupPosition.length + 1));
                });
            }

            // returns true if the operation contains a list definition
            function hasListDefinition(operation) {
                return (operation && _.isString(operation.listStyleId) && !_.isEmpty(operation.listStyleId) && _.isObject(operation.listDefinition));
            }

            // returns true if the operation contains list paragraph attributes
            function hasListAttributes(operation) {
                return (operation && operation.attrs && operation.attrs.paragraph && operation.attrs.paragraph.listStyleId && operation.attrs.styleId === self.getDefaultUIParagraphListStylesheet());
            }

            // maps OX Text list number formats to OX Presentation list numbering types
            function getNumberingType(numberFormat, levelText) {
                var numberingType;
                // the first part of the numbering type is defined by the number format
                switch (numberFormat) {
                    case 'lowerRoman':
                        numberingType = 'romanLc';
                        break;
                    case 'upperRoman':
                        numberingType = 'romanUc';
                        break;
                    case 'lowerLetter':
                        numberingType = 'alphaLc';
                        break;
                    case 'upperLetter':
                        numberingType = 'alphaUc';
                        break;
                    case 'decimal':
                    default:
                        numberingType = 'arabic';
                        break;
                }
                // the second part of the numbering type is defined by the level text
                if (/^\(%\d+\)/.test(levelText)) {
                    numberingType += 'ParenBoth';
                } else if (/^%\d+\)/.test(levelText)) {
                    numberingType += 'ParenR';
                } else {
                    numberingType += 'Period';
                }
                return numberingType;
            }

            // transform paragraph list attributes to slide mode
            function transformListAttributes(operation) {
                // the paragraph attributes of the operation
                var paragraph = operation && operation.attrs && operation.attrs.paragraph;
                // the list definition for the given list style id
                var listDefinition = null;
                // the definition for the current list level
                var levelDefinition = null;
                // the resulting paragraph attributes
                var result = {};

                if (_.isObject(paragraph)) {
                    // transform listLevel to level
                    if (_.isNumber(paragraph.listLevel)) {
                        result.level = paragraph.listLevel;
                    }
                    // provide only list level when pasting into place holder drawings
                    if (!isPlaceHolderDrawingSelected) {
                        // look for list definition
                        listDefinition = listDefinitions[paragraph.listStyleId];
                        // look for level definition
                        if (_.isObject(listDefinition) && _.isNumber(paragraph.listLevel)) {
                            levelDefinition = listDefinition['listLevel' + paragraph.listLevel];
                        }
                        // transform style informations
                        if (levelDefinition) {
                            // indentation
                            if (_.isNumber(levelDefinition.indentLeft)) {
                                result.indentLeft = levelDefinition.indentLeft;
                            }
                            if (_.isNumber(levelDefinition.indentFirstLine)) {
                                result.indentFirstLine = levelDefinition.indentFirstLine;
                            }
                            // number format
                            if (levelDefinition.numberFormat === 'bullet') {
                                // the definition of the bullet style
                                _.extend(result, {
                                    bullet: { type: 'character', character: levelDefinition.levelText || '\u25CF' },
                                    bulletFont: { followText: false, name: levelDefinition.fontName || 'Times New Roman' }
                                });
                            } else {
                                // the definition of the numbering style
                                _.extend(result, {
                                    bullet: { type: 'numbering', numType: getNumberingType(levelDefinition.numberFormat, levelDefinition.levelText) },
                                    bulletFont: { followText: true }
                                });
                            }
                        }
                    }

                    // delete OX Text style id
                    delete operation.attrs.styleId;
                    // replace paragraph attributes
                    operation.attrs.paragraph = result;
                }

                return operation;
            }

            if (isOnlyInsertImage(operations)) {
                //
                // special handling when pasting only images
                //
                // modifying positions for the clipboard operations
                for (; index < operations.length; index++) {
                    // the original operation
                    operation = operations[index];

                    newOperation = _.clone(operation);
                    newOperation.start = drawingStart;
                    resultOperations.push(newOperation);
                    // update start index for new drawings
                    drawingStart = Position.increaseLastIndex(drawingStart, 1);
                }

            } else if (selection.isTopLevelTextCursor()) {
                //
                // no text frame selected, paste target is the slide
                //
                // take the first operation
                operation = operations[index];
                // and make sure it inserts a text frame
                if (!(operation.name === Operations.DRAWING_INSERT && operation.type === 'shape')) {
                    // text frames are drawings of type shape
                    newOperation = { attrs: attrs, start: rootLevelStart, type: 'shape' };
                    self.extendPropertiesWithTarget(newOperation, target);
                    newOperation = generator.generateOperation(Operations.DRAWING_INSERT, newOperation);
                    resultOperations.push(_.clone(newOperation));

                } else {
                    // insert text frame operation already present
                    resultOperations.push(_.clone(operation));
                    index++;
                }

                // take the next operation
                operation = operations[index];
                // and make sure it inserts a paragraph
                if (!operation || operation.name !== Operations.PARA_INSERT) {
                    // add a paragraph into the shape
                    newOperation = { start: Position.appendNewIndex(rootLevelStart) };
                    self.extendPropertiesWithTarget(newOperation, target);
                    newOperation = generator.generateOperation(Operations.PARA_INSERT, newOperation);
                    resultOperations.push(_.clone(newOperation));

                } else {
                    // insert paragraph operation already present
                    newOperation = transformPositionsToTopLevel(operation);
                    resultOperations.push(newOperation);
                    index++;
                }

                // modifying positions for the clipboard operations
                for (; index < operations.length; index++) {
                    // the original operation
                    operation = operations[index];

                    // check if operation is nested inside a group
                    if (isInsideGroup(operation)) {
                        // check insert image operations for their positions:
                        // images are allowed as direct child of a group, but are forbidden at deeper positions (like an image iside a text frame inside a group).
                        if ((operation.name === Operations.DRAWING_INSERT) && (operation.type === 'image') && !isDirectGroupChild(operation))  {
                            // replace insert image with insert text operation to keep positions consistent,
                            newOperation = transformPositionsToGroup({ start: operation.start, text: ' ' });
                            self.extendPropertiesWithTarget(newOperation, target);
                            newOperation = generator.generateOperation(Operations.TEXT_INSERT, newOperation);
                            resultOperations.push(_.clone(newOperation));

                            // update start index for new drawings
                            drawingStart = Position.increaseLastIndex(drawingStart, 1);
                            // insert the image at the next valid root level position
                            newOperation = _.clone(operation);
                            newOperation.start = drawingStart;
                            resultOperations.push(newOperation);

                        } else if (operation.name === Operations.INSERT_LIST) {
                            // store list definition to transform list paragraphs
                            if (hasListDefinition(operation)) {
                                listDefinitions[operation.listStyleId] = operation.listDefinition;
                            }

                        } else if ((operation.name === Operations.COMPLEXFIELD_INSERT) || (operation.name === Operations.RANGE_INSERT && operation.type === 'field')) {
                            // replace complex field and range operations with insert text operation to keep positions consistent
                            newOperation = transformPositionsToGroup({ start: operation.start, text: '\u2060' });
                            self.extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.TEXT_INSERT, newOperation);
                            resultOperations.push(_.clone(generator.getLastOperation()));

                            // create a delete operation to finally remove replacement spaces again
                            newOperation = transformPositionsToGroup({ start: operation.start });
                            self.extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.DELETE, newOperation);
                            replacementDeleteOperations.push(_.clone(generator.getLastOperation()));

                        } else {
                            newOperation = transformPositionsToGroup(operation);
                            if (hasListAttributes(operation)) {
                                newOperation = transformListAttributes(newOperation);
                            }
                            resultOperations.push(newOperation);
                        }

                    } else {
                        // if we find an operation that inserts a drawing we need to update the base positions for drawings and for text frames.
                        if (operation.name === Operations.DRAWING_INSERT) {
                            // replace insert drawing with insert text operation to keep positions consistent,
                            // but only if the drawing's origin is OX Text
                            if (self.isClipboardOriginText()) {
                                newOperation = transformPositionsToTopLevel({ start: operation.start, text: ' ' });
                                self.extendPropertiesWithTarget(newOperation, target);
                                generator.generateOperation(Operations.TEXT_INSERT, newOperation);
                                resultOperations.push(_.clone(generator.getLastOperation()));
                            }

                            // update start index for new drawings
                            drawingStart = Position.increaseLastIndex(drawingStart, 1);
                            // update start index for new text frames
                            if (operation.type === 'shape') {
                                textFrameStart = _.clone(drawingStart);
                            }

                            // store group position; used to check if operation is nested inside a group drawing
                            if (operation.type === 'group') {
                                groupPositions.push(operation.start);
                                groupStart =  _.clone(drawingStart);
                            }

                            newOperation = _.clone(operation);
                            newOperation.start = drawingStart;
                            resultOperations.push(newOperation);

                        } else if (operation.name === Operations.INSERT_LIST) {
                            // store list definition to transform list paragraphs
                            if (hasListDefinition(operation)) {
                                listDefinitions[operation.listStyleId] = operation.listDefinition;
                            }

                        } else if ((operation.name === Operations.COMPLEXFIELD_INSERT) || (operation.name === Operations.RANGE_INSERT && operation.type === 'field')) {
                            // replace complex field and range operations with insert text operation to keep positions consistent
                            newOperation = transformPositionsToTopLevel({ start: operation.start, text: '\u2060' });
                            self.extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.TEXT_INSERT, newOperation);
                            resultOperations.push(_.clone(generator.getLastOperation()));

                            // create a delete operation to finally remove replacement spaces again
                            newOperation = transformPositionsToTopLevel({ start: operation.start });
                            self.extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.DELETE, newOperation);
                            replacementDeleteOperations.push(_.clone(generator.getLastOperation()));

                        } else {
                            newOperation = transformPositionsToTopLevel(operation);
                            if (hasListAttributes(operation)) {
                                newOperation = transformListAttributes(newOperation);
                            }
                            resultOperations.push(newOperation);
                        }
                    }
                }

            } else if (selection.isAnyTextFrameSelection()) {
                //
                // a text frame is selected, paste target for text is that frame
                //
                // modifying positions for the clipboard operations
                for (; index < operations.length; index++) {
                    // the original operation
                    operation = operations[index];

                    if (operation.name === Operations.INSERT_LIST) {
                        // store list definition to transform list paragraphs
                        if (hasListDefinition(operation)) {
                            listDefinitions[operation.listStyleId] = operation.listDefinition;
                        }

                    } else if ((operation.name === Operations.COMPLEXFIELD_INSERT) || (operation.name === Operations.RANGE_INSERT && operation.type === 'field')) {
                        // replace complex field and range operations with insert text operation to keep positions consistent
                        newOperation = { start: operation.start, text: '\u2060' };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.TEXT_INSERT, newOperation);
                        resultOperations.push(_.clone(generator.getLastOperation()));

                        // create a delete operation to finally remove replacement spaces again
                        newOperation = { start: operation.start };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.DELETE, newOperation);
                        replacementDeleteOperations.push(_.clone(generator.getLastOperation()));

                    } else if (hasListAttributes(operation)) {
                        newOperation = transformListAttributes(_.clone(operation));
                        resultOperations.push(newOperation);

                    } else {
                        resultOperations.push(_.clone(operation));
                    }
                }
            }

            replacementDeleteOperations.reverse();
            resultOperations = resultOperations.concat(replacementDeleteOperations);

            return resultOperations;
        }

        /**
         * Replaces a table including sub tables with a paragraph.
         * Therefore removes all table concerning operations, replacing them
         * with insert paragraph and insert text operations.
         *
         * @param {Array} operations
         *  An array with the clipboard operations.
         *
         * @returns {Array}
         *  An array with the resulting operations.
         */
        function makeTextFromTableOperations(operations) {
            // the operations generator
            var generator = self.createOperationsGenerator();
            // target for operation - if exists, it's for ex. header or footer
            var target = self.getActiveTarget();
            // the table root positions
            var tableRoots = [];
            // the position for the resulting text
            var textPosition = [0, 0];
            // the newly created operation
            var newOperation = null;
            // the resulting operations
            var resultOperations = [];

            // returns true if the position is inside a table.
            function isTablePosition(position) {
                // check position if it starts with one of the collected table root positions
                if ((!_.isArray(position)) || (tableRoots.length < 1)) { return false; }
                return _.some(tableRoots, function (rootPosition) {
                    return arrayStartsWith(position, rootPosition);
                });
            }

            _.each(operations, function (operation) {
                // check for insert table, operations that are inside the table and all the rest
                if (operation.name === Operations.TABLE_INSERT) {
                    // check for root or sub table
                    if ((tableRoots.length === 0) || !arrayStartsWith(operation.start, textPosition.slice(0, -1))) {
                        // create insert paragraph operation to replace insert table operation
                        newOperation = { start: _.clone(operation.start) };
                        self.extendPropertiesWithTarget(newOperation, target);
                        newOperation = generator.generateOperation(Operations.PARA_INSERT, newOperation);
                        resultOperations.push(_.clone(newOperation));
                        // set position for insert text operations
                        textPosition = Position.appendNewIndex(operation.start);
                    }
                    // remember table position
                    tableRoots.push(operation.start);

                } else if (isTablePosition(operation.start)) {
                    // shift position of insert text operation from the table into the target paragraph
                    if (operation.name === Operations.TEXT_INSERT) {
                        newOperation = _.clone(operation);
                        newOperation.start = _.clone(textPosition);
                        newOperation.text += ' ';
                        resultOperations.push(newOperation);
                        // update text position
                        textPosition = Position.increaseLastIndex(textPosition, newOperation.text.length);
                    }

                } else {
                    // non table operations
                    resultOperations.push(operation);
                }

            });

            return resultOperations;
        }

        /**
         * Processing drop of images.
         *
         * @param {jQuery.Event} event
         *  The jQuery browser event object.
         *
         * @param {Number[]} dropPosition
         *  The position for the drop operation.
         */
        function processDroppedImages(event, dropPosition) {

            var images = event.originalEvent.dataTransfer.files;

            // handle image date received from IO.readClientFileAsDataUrl
            function handleImageData(imgData) {
                self.insertImageURL(imgData, dropPosition);
            }

            //checks if files were dropped from the browser or the file system
            if (!images || images.length === 0) {
                self.insertImageURL(event.originalEvent.dataTransfer.getData('text'), dropPosition);
                return;
            } else {
                for (var i = 0; i < images.length; i++) {
                    var img = images[i];
                    var imgType = /image.*/;

                    //cancels insertion if the file is not an image
                    if (!img.type.match(imgType)) {
                        continue;
                    }

                    IO.readClientFileAsDataUrl(img).done(handleImageData);
                }
            }
        }

        /**
         * Generates operations needed to copy the current text selection to
         * the internal clipboard.
         *
         * @returns {Array}
         *  The operations array that represents the current selection.
         */
        function copyTextSelection() {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // zero-based index of the current content node
                targetPosition = 0,
                // result of the iteration process
                result = null,
                // OX Text default paragraph list style sheet
                listParaStyleId = self.getDefaultUIParagraphListStylesheet(),
                // indicates if we had to add a paragraph style
                listParaStyleInserted = false,
                // the applied list style ids
                listStyleIds = [],
                // the attribute property for the change track
                changesFamily = 'changes',
                // a logical helper start position, if a text frame is selected
                startPosition = null,
                // attributes of the contentNode
                attributes,
                // a new created operation
                newOperation = null,
                // the selection object
                selection = self.getSelection();

            // in the case of a text cursor selection inside a selected text frame, the text frame itself can be used for copying
            if (selection.getSelectionType() === 'text' && !self.hasSelectedRange() && selection.isAdditionalTextframeSelection()) {
                startPosition = Position.getOxoPosition(self.getNode(), selection.getSelectedTextFrameDrawing(), 0);
                selection.setTextSelection(startPosition, Position.increaseLastIndex(startPosition));
            }

            // ignoring empty selections
            if (selection.isTextCursor()) { return []; }

            // in case we are in header/footer and we have a selection that contains special page fields, don't copy
            if (self.getFieldManager().checkIfSpecialFieldsSelected()) {
                return [];
            }

            // visit the paragraphs and tables covered by the text selection
            result = selection.iterateContentNodes(function (contentNode, position, startOffset, endOffset) {

                var // whether this is a selection inside one paragraph -> no handling of paragraph attributes (42759)
                    isSingleParagraphSelection = _.isNumber(startOffset) && _.isNumber(endOffset),
                    // the list collection object
                    listCollection = self.getListCollection();

                // paragraphs may be covered partly
                if (DOM.isParagraphNode(contentNode)) {

                    // if we have a list add the list paragraph style and the list style
                    if (!isSingleParagraphSelection && self.getParagraphStyles().containsStyleSheet(listParaStyleId)) {

                        if (!listParaStyleInserted) {
                            self.generateMissingStyleSheetOperations(generator, 'paragraph', listParaStyleId);
                            listParaStyleInserted = true;
                        }

                        attributes = self.getParagraphStyles().getElementAttributes(contentNode);
                        if (listCollection && attributes.paragraph && attributes.paragraph.listStyleId && !_.contains(listStyleIds, attributes.paragraph.listStyleId)) {
                            newOperation = listCollection.getListOperationFromListStyleId(attributes.paragraph.listStyleId);
                            if (newOperation) {  // check for valid operation (38007)
                                generator.generateOperation(Operations.INSERT_LIST, newOperation);
                                listStyleIds.push(attributes.paragraph.listStyleId);
                            }
                        }

                    }

                    // first or last paragraph: generate operations for covered text components
                    if (_.isNumber(startOffset) || _.isNumber(endOffset)) {

                        // some selections might cause invalid logical positions (38095)
                        if (_.isNumber(startOffset) && _.isNumber(endOffset) && (endOffset < startOffset)) { return Utils.BREAK; }

                        // special handling for Firefox and IE when selecting list paragraphs.
                        // if the selection includes the end position of a non list paragraph
                        // above the list, we don't add an empty paragraph to the document
                        // fix for bug 29752
                        if ((_.browser.Firefox || _.browser.IE) &&
                                Position.getParagraphLength(self.getNode(), position) === startOffset &&
                                !DOM.isListLabelNode(contentNode.firstElementChild) &&
                                Utils.findNextNode(self.getNode(), contentNode, DOM.PARAGRAPH_NODE_SELECTOR) &&
                                DOM.isListLabelNode(Utils.findNextNode(self.getNode(), contentNode, DOM.PARAGRAPH_NODE_SELECTOR).firstElementChild)) {
                            return;
                        }

                        // generate a splitParagraph and setAttributes operation for
                        // contents of first paragraph (but for multiple-paragraph
                        // selections only)
                        if (!_.isNumber(endOffset)) {
                            generator.generateOperation(Operations.PARA_SPLIT, { start: [targetPosition, 0] });
                        }

                        // handling for all paragraphs, also the final (36825). But ignoring paragraph attributes in single paragraph selections (42759)
                        if (!isSingleParagraphSelection) {
                            generator.generateSetAttributesOperation(contentNode, { start: [targetPosition] }, { clearFamily: 'paragraph', ignoreFamily: changesFamily });
                        }

                        // operations for the text contents covered by the selection
                        generator.generateParagraphChildOperations(contentNode, [targetPosition], { start: startOffset, end: endOffset, targetOffset: 0, clear: true, ignoreFamily: changesFamily });

                    } else {

                        // skip embedded implicit paragraphs
                        if (DOM.isImplicitParagraphNode(contentNode)) { return; }

                        // generate operations for entire paragraph
                        generator.generateParagraphOperations(contentNode, [targetPosition], { ignoreFamily: changesFamily });
                    }

                // entire table: generate complete operations array for the table (if it is not an exceeded-size table)
                } else if (DOM.isTableNode(contentNode)) {

                    // skip embedded oversized tables
                    if (DOM.isExceededSizeTableNode(contentNode)) { return; }

                    // generate operations for entire table
                    generator.generateTableOperations(contentNode, [targetPosition], { ignoreFamily: changesFamily });

                } else {
                    Utils.error('Editor.copyTextSelection(): unknown content node "' + Utils.getNodeName(contentNode) + '" at position ' + JSON.stringify(position));
                    return Utils.BREAK;
                }

                targetPosition += 1;

            }, this, { shortestPath: true });

            // return operations, if iteration has not stopped on error
            return (result === Utils.BREAK) ? [] : generator.getOperations();
        }

        /**
         * Generates operations needed to copy the current cell range selection
         * to the internal clipboard.
         *
         * @returns {Array}
         *  The operations array that represents the current selection.
         */
        function copyCellRangeSelection() {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the selection object
                selection = self.getSelection(),
                // information about the cell range
                cellRangeInfo = selection.getSelectedCellRange(),
                // merged attributes of the old table
                oldTableAttributes = null,
                // explicit attributes for the new table
                newTableAttributes = null,
                // all rows in the table
                tableRowNodes = null,
                // relative row offset of last visited cell
                lastRow = -1,
                // the attribute property for the change track
                changesFamily = 'changes',
                // result of the iteration process
                result = null;
            // generates operations for missing rows and cells, according to lastRow/lastCol
            function generateMissingRowsAndCells(row/*, col*/) {

                // generate new rows (repeatedly, a row may be covered completely by merged cells)
                while (lastRow < row) {
                    lastRow += 1;
                    generator.generateOperationWithAttributes(tableRowNodes[lastRow], Operations.ROWS_INSERT, { start: [1, lastRow], count: 1, insertDefaultCells: false }, { ignoreFamily: changesFamily });
                }

                // TODO: detect missing cells, which are covered by merged cells outside of the cell range
                // (but do not generate cells covered by merged cells INSIDE the cell range)
            }

            if (!cellRangeInfo) {
                Utils.error('Editor.copyCellRangeSelection(): invalid cell range selection');
                return [];
            }

            // split the paragraph to insert the new table between the text portions
            generator.generateOperation(Operations.PARA_SPLIT, { start: [0, 0] });

            // generate the operation to create the new table
            oldTableAttributes = self.getTableStyles().getElementAttributes(cellRangeInfo.tableNode);
            newTableAttributes = AttributeUtils.getExplicitAttributes(cellRangeInfo.tableNode);
            newTableAttributes.table = newTableAttributes.table || {};
            newTableAttributes.table.tableGrid = oldTableAttributes.table.tableGrid.slice(cellRangeInfo.firstCellPosition[1], cellRangeInfo.lastCellPosition[1] + 1);
            if (newTableAttributes && newTableAttributes[changesFamily]) { delete newTableAttributes[changesFamily]; }
            generator.generateOperation(Operations.TABLE_INSERT, { start: [1], attrs: newTableAttributes });

            // all covered rows in the table
            tableRowNodes = DOM.getTableRows(cellRangeInfo.tableNode).slice(cellRangeInfo.firstCellPosition[0], cellRangeInfo.lastCellPosition[0] + 1);

            // visit the cell nodes covered by the selection
            result = selection.iterateTableCells(function (cellNode, position, row, col) {

                // generate operations for new rows, and for cells covered by merged cells outside the range
                generateMissingRowsAndCells(row, col);

                // generate operations for the cell
                generator.generateTableCellOperations(cellNode, [1, row, col], { ignoreFamily: changesFamily });
            });

            // missing rows at bottom of range, covered completely by merged cells (using relative cellRangeInfo, task 30839)
            generateMissingRowsAndCells(cellRangeInfo.lastCellPosition[0] - cellRangeInfo.firstCellPosition[0], cellRangeInfo.lastCellPosition[1] - cellRangeInfo.firstCellPosition[1] + 1);

            // return operations, if iteration has not stopped on error
            return (result === Utils.BREAK) ? [] : generator.getOperations();
        }

        /**
         * Generates operations needed to copy the current drawing selection to
         * the internal clipboard.
         *
         * @returns {Array}
         *  The operations array that represents the current selection.
         */
        function copyDrawingSelection() {
            // the operations generator
            var generator = self.createOperationsGenerator();
            // the attribute property for the change track
            var changesFamily = 'changes';
            // the start position for the drawing
            var startPosition = [0, 0];
            // the selected drawing nodes
            var drawingNodes = null;
            // target for operation - if exists, it's for ex. header or footer
            var target = self.getActiveTarget();

            // in case we are in header/footer and we have a selection that contains special page fields, don't copy
            if (self.getFieldManager().checkIfSpecialFieldsSelected()) {
                return [];
            }

            // the selected drawing nodes
            drawingNodes = getSelectedDrawingNodes();
            // generate operations
            _.each(drawingNodes, function (node) {
                // switching to node in drawing layer
                if (DOM.isDrawingPlaceHolderNode(node)) { node = DOM.getDrawingPlaceHolderNode(node); }

                // generate operations for the drawing (including its attributes)
                generator.generateDrawingOperations(node, startPosition, { ignoreFamily: changesFamily, target: target, allAttributes: true });

                // set start position for next drawing
                startPosition = Position.increaseLastIndex(startPosition, 1);
            });

            // the resulting operations
            return generator.getOperations();
        }

        /**
         * Generates operations needed to copy the current selected slide(s) selection to
         * the internal clipboard.
         *
         * @returns {Array}
         *  The operations array that represents the current selection.
         */
        function copySlides() {
            // the operations generator
            var generator = self.createOperationsGenerator();

            // the selected slide(s) in slide pane
            var selectedSlides = null;

            // the selected selected slide(s)
            selectedSlides = self.getSlidePaneSelection();
            // sort selected slides ascending
            if (_.isArray(selectedSlides)) {
                selectedSlides.sort(function (a, b) { return a - b; });
            }
            // generate operations
            _.each(selectedSlides, function (slideIndex) {
                var slideId = self.getIdOfSlideOfActiveViewAtIndex(slideIndex);
                var slideNode = self.getSlideById(slideId);
                var parentId = self.isLayoutOrMasterId(slideId) ? self.getMasterSlideId(slideId) : self.getLayoutSlideId(slideId);
                var operation;
                var options = {};

                // insert slide op TODO
                if (self.isMasterView()) {
                    operation = { start: null, target: parentId, id: null, attrs: { slide: { type: self.getSlideFamilyAttributeForSlide(slideId, 'slide', 'type') } } }; // start, target and id needs to be calculated on paste
                    generator.generateOperation(Operations.LAYOUT_SLIDE_INSERT, operation);
                    options.getDrawingListStyles = true;
                } else {
                    operation = { start: null, target: self.getSlideFamilyAttributeForSlide(parentId, 'slide', 'type') }; // start needs to be calculated on paste, target also
                    generator.generateOperation(Operations.SLIDE_INSERT, operation);
                }

                // generate operations for the drawing (including its attributes)
                generator.generateParagraphChildOperations(slideNode, [slideIndex], options);
            });

            // the resulting operations
            return generator.getOperations();
        }

        /**
         * Creates operations from the clipboard data returned by parseClipboard(...)
         * and applies them asynchronously.
         *
         * @param {Array} clipboardData
         *  The clipboard data array to create operations from.
         *
         * @param {Number[]} dropPosition
         *  An optional logical position used for pasting the content.
         */
        function createOperationsFromExternalClipboard(clipboardData, dropPosition) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the selection object
                selection = self.getSelection(),
                // the list collection object
                listCollection = self.getListCollection(),
                // the operation start position
                start = dropPosition ? dropPosition.start : selection.getStartPosition(),
                // the operation end position
                end = _.clone(start),
                // indicates if the previous operation was insertHyperlink
                hyperLinkInserted,
                // used to cancel operation preparing in iterateArraySliced
                cancelled = false,
                // the current paragraph style sheet id
                styleId,
                // the next free list style number part
                listStyleNumber = 1,
                // the default paragraph list style sheet id
                listParaStyleId = null,
                // the list stylesheet id
                listStyleId = null,
                // indicates if previous operations inserted a list
                listInserted = false,
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // the ratio of operation generation to apply the operations
                generationRatio = 0.2,
                // current element container node
                rootNode = self.getCurrentRootNode(target);

            // the operation after insertHyperlink needs to remove the hyperlink style again
            function removeHyperLinkStyle(start, end, nextEntry) {

                if (hyperLinkInserted && (!nextEntry || nextEntry.operation !== 'insertHyperlink')) {
                    // generate the 'setAttributes' operation to remove the hyperlink style
                    generator.generateOperation(Operations.SET_ATTRIBUTES, {
                        attrs: Hyperlink.CLEAR_ATTRIBUTES,
                        start: _.clone(start),
                        end: Position.decreaseLastIndex(end)
                    });
                    hyperLinkInserted = false;
                }
            }

            // checks the given position for a text span with hyperlink attributes
            function isHyperlinkAtPosition(position) {
                var span = Position.getSelectedElement(rootNode, Position.decreaseLastIndex(position), 'span'),
                    characterAttrs = AttributeUtils.getExplicitAttributes(span, { family: 'character', direct: true });

                return !!(characterAttrs && _.isString(characterAttrs.url));
            }

            // handle implicit paragraph
            function doHandleImplicitParagraph(start) {
                var position = _.clone(start),
                    paragraph;

                if (position.pop() === 0) {  // is this an empty paragraph?
                    paragraph = Position.getParagraphElement(rootNode, position);
                    if ((paragraph) && (DOM.isImplicitParagraphNode(paragraph)) && (Position.getParagraphNodeLength(paragraph) === 0)) {
                        // creating new paragraph explicitely
                        generator.generateOperation(Operations.PARA_INSERT, {
                            start: position
                        });
                    }
                }
            }

            // the list collection is only aware of the list styles that have already been applied by an 'insertList' operation,
            // so we need to handle the next free list style id generation ourselfs.
            function getFreeListStyleId() {

                var sFreeId = 'L';

                while (listCollection.hasListStyleId('L' + listStyleNumber)) {
                    listStyleNumber++;
                }
                sFreeId += listStyleNumber;
                listStyleNumber++;
                return sFreeId;
            }

            // parse entry and generate operations
            function generateOperationCallback(entry, index, dataArray) {

                var def = null,
                    lastPos,
                    position,
                    endPosition = null,
                    attributes,
                    listOperation = null,
                    hyperlinkStyleId,
                    styleAttributes;

                // next operation's start is previous operation's end
                start = _.clone(end);
                end = _.clone(end);
                lastPos = start.length - 1;

                // cancel the clipboard paste if we loose the edit rigths or if the 'cancel' button is clicked.
                if (cancelled || !self.getEditMode()) { return Utils.BREAK; }

                switch (entry.operation) {

                    case Operations.PARA_INSERT:
                        // generate the 'insertParagraph' operation
                        generator.generateOperation(Operations.PARA_SPLIT, {
                            start: _.clone(start)
                        });

                        end[lastPos - 1] += 1;
                        end[lastPos] = 0;

                        // set attributes only if it's not a list paragraph, these are handled in 'insertListElement'
                        if (entry.listLevel === -1) {
                            // use the nextStyleId if the current paragraph style defines one
                            styleAttributes = self.getStyleCollection('paragraph').getStyleSheetAttributeMap(styleId, 'paragraph');
                            styleId = styleAttributes.paragraph && styleAttributes.paragraph.nextStyleId;

                            if (listInserted) {
                                // remove list style
                                if ((!_.isString(styleId)) || (styleId === self.getDefaultUIParagraphListStylesheet())) {
                                    styleId = self.getDefaultUIParagraphStylesheet();
                                }
                                attributes = { styleId: styleId, paragraph: { listStyleId: null, listLevel: -1 } };
                                listInserted = false;

                            } else if (_.isString(styleId)) {
                                attributes = { styleId: styleId };
                            }

                            if (_.isString(styleId)) {
                                // generate operation to insert a dirty paragraph style into the document
                                self.generateMissingStyleSheetOperations(generator, 'paragraph', styleId);

                                // generate the 'setAttributes' operation
                                generator.generateOperation(Operations.SET_ATTRIBUTES, {
                                    start: end.slice(0, -1),
                                    attrs: attributes
                                });
                            }
                        }
                        break;

                    case Operations.TEXT_INSERT:
                        // generate the 'insertText' operation (but not for empty strings)
                        if (entry.data) {
                            generator.generateOperation(Operations.TEXT_INSERT, {
                                text: entry.data,
                                start: _.clone(start),
                                attrs: { changes: { inserted: null, removed: null, modified: null } } // 41152
                            });

                            end[lastPos] += entry.data.length;
                        }

                        removeHyperLinkStyle(start, end, dataArray[index + 1]);
                        break;

                    case Operations.TAB_INSERT:
                        // generate the 'insertTab' operation
                        generator.generateOperation(Operations.TAB_INSERT,
                            _.isObject(self.getPreselectedAttributes()) ? { start: _.clone(start), attrs: self.getPreselectedAttributes() } : { start: _.clone(start) }
                        );

                        end[lastPos] += 1;
                        removeHyperLinkStyle(start, end, dataArray[index + 1]);
                        break;

                    case Operations.DRAWING_INSERT:
                        if ((!_.isString(entry.data)) || (entry.data.length < 1)) { break; }
                        // check if we got a webkit fake URL instead of a data URI,
                        // there's currently no way to access the image data of a webkit fake URL
                        // and check for local file url
                        if ((entry.data.substring(0, 15) === 'webkit-fake-url') || (entry.data.substring(0, 7) === 'file://')) {
                            app.rejectEditAttempt('image');
                            break;
                        }

                        def = $.Deferred();

                        self.getImageSize(entry.data).then(function (size) {

                            // exit silently if we lost the edit rights
                            if (!self.getEditMode()) {  return $.when(); }

                            // check for base64 image data or image url
                            attributes = {
                                drawing: _.extend({}, size, self.getDefaultDrawingMargins()),
                                image: (entry.data.substring(0, 10) === 'data:image') ? { imageData: entry.data } : { imageUrl: entry.data }
                            };

                            // generate the 'insertDrawing' operation
                            generator.generateOperation(Operations.DRAWING_INSERT, {
                                start: _.clone(start),
                                type: 'image',
                                attrs: attributes
                            });

                            return $.when();
                        })
                        .then(function () {
                            end[lastPos] += 1;
                            removeHyperLinkStyle(start, end, dataArray[index + 1]);
                        }, function () {
                            app.rejectEditAttempt('image');
                        })
                        .always(function () {
                            // always resolve to continue processing paste operations
                            def.resolve();
                        });
                        break;

                    case 'insertHyperlink':
                        if (entry.data && _.isNumber(entry.length) && (entry.length > 0) && HyperlinkUtils.hasSupportedProtocol(entry.data) && HyperlinkUtils.isValidURL(entry.data)) {
                            hyperlinkStyleId = self.useSlideMode() ? null : self.getDefaultUIHyperlinkStylesheet();

                            // generate operation to insert a dirty character style into the document
                            if (_.isString(hyperlinkStyleId)) {
                                self.generateMissingStyleSheetOperations(generator, 'character', hyperlinkStyleId);
                            }

                            // the text for the hyperlink has already been inserted and the start position is right after this text,
                            // so the start for the hyperlink attribute is the start position minus the text length
                            position = _.clone(start);
                            position[lastPos] -= entry.length;
                            if (position[lastPos] < 0) {
                                position[lastPos] = 0;
                            }

                            // Task 35689, position of last character in range
                            endPosition = _.clone(end);
                            endPosition[lastPos] -= 1;

                            if (endPosition[lastPos] >= 0) { // avoiding invalid setAttribute operations

                                // generate the 'insertHyperlink' operation
                                generator.generateOperation(Operations.SET_ATTRIBUTES, {
                                    attrs: { styleId: hyperlinkStyleId, character: { url: entry.data } },
                                    start: position,
                                    end: endPosition
                                });

                                hyperLinkInserted = true;
                            }
                        }
                        break;

                    case 'insertList':
                        // for the list root level insert list style with all style levels
                        if (listCollection && entry.listLevel === -1) {
                            listOperation = listCollection.getListOperationFromHtmlListTypes(entry.type);
                            listStyleId = getFreeListStyleId();

                            // generate the 'insertList' operation
                            generator.generateOperation(Operations.INSERT_LIST, {
                                listStyleId: listStyleId,
                                listDefinition: listOperation.listDefinition
                            });
                        }
                        break;

                    case 'insertMSList':
                        if (listCollection) {
                            listOperation = listCollection.getListOperationFromListDefinition(entry.data);
                            listStyleId = getFreeListStyleId();

                            // generate the 'insertList' operation
                            generator.generateOperation(Operations.INSERT_LIST, {
                                listStyleId: listStyleId,
                                listDefinition: listOperation.listDefinition
                            });
                        }
                        break;

                    case 'insertListElement':
                        if (listStyleId && _.isNumber(entry.listLevel)) {
                            listParaStyleId = self.getDefaultUIParagraphListStylesheet();

                            // generate operation to insert a dirty paragraph style into the document
                            if (_.isString(listParaStyleId)) {
                                self.generateMissingStyleSheetOperations(generator, 'paragraph', listParaStyleId);
                            }

                            // generate the 'setAttributes' operation
                            generator.generateOperation(Operations.SET_ATTRIBUTES, {
                                start: start.slice(0, -1),
                                attrs: { styleId: listParaStyleId, paragraph: { listStyleId: listStyleId, listLevel: entry.listLevel } }
                            });

                            listInserted = true;
                        }
                        break;

                    // needed for additional paragraph inside a list element, from the second paragraph on they have no bullet or numberring
                    case 'insertListParagraph':
                        // generate the 'insertParagraph' operation
                        generator.generateOperation(Operations.PARA_SPLIT, {
                            start: _.clone(start)
                        });

                        end[lastPos - 1] += 1;
                        end[lastPos] = 0;

                        // set attributes only if it's a list paragraph
                        if (listInserted && _.isString(listParaStyleId) && entry.listLevel > -1) {

                            // generate the 'setAttributes' operation
                            generator.generateOperation(Operations.SET_ATTRIBUTES, {
                                start: end.slice(0, -1),
                                attrs: { styleId: listParaStyleId, paragraph: { listStyleId: null, listLevel: -1 } }
                            });
                        }
                        break;

                    default:
                        Utils.log('createOperationsFromExternalClipboard(...) - unhandled operation: ' + entry.operation);
                }

                return def;
            }

            // make sure that only one paste call is processed at the same time
            if (self.checkSetClipboardPasteInProgress()) { return $.when(); }

            // to paste at the cursor position don't create a paragraph as first operation
            if (clipboardData.length > 1 && clipboardData[0].operation === Operations.PARA_INSERT) {
                clipboardData.shift();
            } else if (clipboardData.length > 2 && clipboardData[0].operation === 'insertList' && clipboardData[1].operation === Operations.PARA_INSERT) {
                clipboardData.splice(1, 1);
            }

            // init the paragraph style sheet id
            styleId = AttributeUtils.getElementStyleId(Position.getLastNodeFromPositionByNodeName(rootNode, start, DOM.PARAGRAPH_NODE_SELECTOR));

            // make sure to remove the hyperlink style if we paste directly after a hyperlink
            hyperLinkInserted = isHyperlinkAtPosition(start);

            // init the next free list style number
            if (listCollection) { listStyleNumber = parseInt(listCollection.getFreeListId().slice(1, listCollection.getFreeListId().length), 10); }
            if (!_.isNumber(listStyleNumber)) {
                listStyleNumber = 1;
            }

            return self.getUndoManager().enterUndoGroup(function () {

                var // the deferred to keep the undo group open until it is resolved or rejected
                    undoDef = $.Deferred(),
                    // the apply actions promise
                    applyPromise = null,
                    // the generate operations deferred
                    generateDef = null,
                    // the generated operations
                    operations = null,
                    // a snapshot object
                    snapshot = null,
                    // whether there is a selection range before pasting
                    hasRange = selection.hasRange(),
                    // the selection range before pasting
                    rangeStart = null, rangeEnd = null,
                    // the promise for the deletion and generating and applying of operations
                    completePromise = null;

                // creating a snapshot
                if (hasRange) {
                    snapshot = new Snapshot(app);
                    rangeStart = _.clone(selection.getStartPosition());
                    rangeEnd = _.clone(selection.getEndPosition());
                }

                // delete current selection (if exists), has own progress bar
                completePromise = self.deleteSelected({ alreadyPasteInProgress: true, snapshot: snapshot })
                .then(function () {

                    // create operation to replace an implicit paragraph with a real one
                    doHandleImplicitParagraph(start);

                    // creating a snapshot
                    if (!snapshot) { snapshot = new Snapshot(app); }

                    // show a nice message for operation creation and for applying operations with cancel button
                    app.getView().enterBusy({
                        cancelHandler: function () {
                            cancelled = true;
                            // restoring the old document state
                            snapshot.apply();
                            // calling abort function for operation promise
                            app.enterBlockOperationsMode(function () { if (applyPromise && applyPromise.abort) { applyPromise.abort(); } });
                        },
                        warningLabel: gt('Sorry, pasting from clipboard will take some time.')
                    });

                    // generate operations
                    generateDef = self.iterateArraySliced(clipboardData, generateOperationCallback, { delay: 'immediate', infoString: 'Text: generateOperationCallback' });

                    // add progress handling
                    generateDef.progress(function (progress) {
                        // update the progress bar according to progress of the operations promise
                        app.getView().updateBusyProgress(generationRatio * progress);
                    });

                    return generateDef;
                })
                .then(function () {
                    // cancel the clipboard paste if we loose the edit rigths or if the 'cancel' button is clicked.
                    if (cancelled || !self.getEditMode()) { return $.Deferred().reject(); }
                })
                .then(function () {

                    var // the currently active target
                        target = self.getActiveTarget();

                    // cancel the clipboard paste if we loose the edit rigths or if the 'cancel' button is clicked.
                    if (cancelled || !self.getEditMode()) { return $.Deferred().reject(); }

                    operations = generator.getOperations();

                    // adding the information about the change tracking to the operations
                    if (self.getChangeTrack().isActiveChangeTracking()) {
                        self.getChangeTrack().handleChangeTrackingDuringPaste(operations);
                    } else {
                        self.getChangeTrack().removeChangeTrackInfoAfterSplitInPaste(operations);
                    }

                    // handling target positions (for example inside comments)
                    self.getCommentLayer().handlePasteOperationTarget(operations);
                    self.getFieldManager().handlePasteOperationTarget(operations);

                    // if pasting into header or footer that are currently active, extend operations with target
                    if (target) {
                        _.each(operations, function (operation) {
                            if (operation.name !== Operations.INSERT_STYLESHEET) {
                                operation.target = target;
                            }
                        });
                    }

                    // apply generated operations
                    applyPromise = self.applyOperations(operations, { async: true })
                    .progress(function (progress) {
                        // update the progress bar according to progress of the operations promise
                        app.getView().updateBusyProgress(generationRatio + (1 - generationRatio) * progress);
                    });

                    return applyPromise;

                });

                // handler for 'done', that is not triggered, if self is in destruction (document is closed, 42567)
                self.waitForSuccess(completePromise, function () {
                    selection.setTextSelection(end);
                });

                // handler for 'always', that is not triggered, if self is in destruction (document is closed, 42567)
                self.waitForAny(completePromise, function () {
                    if (cancelled && hasRange) { selection.setTextSelection(rangeStart, rangeEnd); }
                    // no longer blocking page break calculations (40107)
                    self.setBlockOnInsertPageBreaks(false);
                    // leaving the blocked async mode
                    self.leaveAsyncBusy();
                    // close undo group
                    undoDef.resolve();
                    // deleting the snapshot
                    if (snapshot) { snapshot.destroy(); }
                });

                return undoDef.promise();

            }); // enterUndoGroup()
        }

        /**
         * Uses a plain text string and interprets control characters, e.g.
         * \n, \t to have a formatted input for the editor.
         *
         * @param {String} plainText
         *  The plain text with optional control characters.
         *
         * @param {Number[]} [dropPosition]
         *  An optional position for the drop operation.
         */
        function insertPlainTextFormatted(plainText, dropPosition) {
            var lines, insertParagraph = false, result = [];

            if (_.isString(plainText) && plainText.length > 0) {
                if (self.getSelection().hasRange() && dropPosition.start) {
                    self.getSelection().setTextSelection(dropPosition.start); // clear selection range, #43275
                }

                lines = plainText.match(/[^\r\n]+/g);

                _(lines).each(function (line) {
                    if (insertParagraph) {
                        result.push({ operation: Operations.PARA_INSERT, depth: 0, listLevel: -1 });
                    }
                    result.push({ operation: Operations.TEXT_INSERT, data: line, depth: 0 });
                    insertParagraph = true;
                });

                createOperationsFromExternalClipboard(result, dropPosition);
            }
        }

        /**
         * Helper function that tells if user is going to paste slide(s), or more precisely,
         * if first operation in clipboard operations is insertSlide or insertLayoutSlide.
         *
         * @returns {Boolean}
         */
        function isSlideCopied() {

            if (!self.useSlideMode()) { return false; }

            var pasteOps = self.getClipboardOperations();
            var firstOp = pasteOps && _.first(pasteOps);

            return firstOp && (firstOp.name === 'insertSlide' || firstOp.name === 'insertLayoutSlide');
        }

        // public methods -----------------------------------------------------

        /**
         * Getting the clipboard data origin.
         * Returns "io.ox/office/text", "io.ox/office/presentation" or an empty String.
         *
         * @returns {String}
         *  The clipboard data origin.
         */
        this.getClipboardOrigin = function () {
            return clipboardOrigin;
        };

        /**
         * Returns true if the clipboard data origin is OX Text.
         *
         * @returns {Boolean}
         *  Whether the origin is OX Text.
         */
        this.isClipboardOriginText = function () {
            return (clipboardOrigin ===  'io.ox/office/text');
        };

        /**
         * Returns true if the clipboard data origin is OX Presentation.
         *
         * @returns {Boolean}
         *  Whether the origin is OX Presentation.
         */
        this.isClipboardOriginPresentation = function () {
            return (clipboardOrigin === 'io.ox/office/presentation');
        };

        /**
         * Setting the clipbard data origin.
         *
         * @param {String} origin
         *  The clipboard data origin.
         */
        this.setClipboardOrigin = function (origin) {
            clipboardOrigin = _.isString(origin) ? origin : '';
        };

        /**
         * Sets the browser focus to the clipboard node, and selects all its
         * contents.
         *
         * @param {jQuery} clipboardNode
         */
        this.grabClipboardFocus = function (clipboardNode) {

            // the browser selection
            var selection = window.getSelection();
            // a browser selection range object
            var docRange = null;

            // Clear the old browser selection.
            // Bug 28515, bug 28711: IE fails to clear the selection (and to modify
            // it afterwards), if it currently points to a DOM node that is not
            // visible anymore (e.g. the 'Show/hide side panel' button). Workaround
            // is to move focus to an editable DOM node which will cause IE to update
            // the browser selection object. The target container node cannot be used
            // for that, see comments above for bug 26283. Using another focusable
            // node (e.g. the body element) is not sufficient either. Interestingly,
            // even using the (editable) clipboard node does not work here. Setting
            // the new browser selection below will move the browser focus back to
            // the application pane.
            Utils.clearBrowserSelection();

            // set the browser selection
            try {
                docRange = window.document.createRange();
                docRange.setStart(clipboardNode[0], 0);
                docRange.setEnd(clipboardNode[0], clipboardNode[0].childNodes.length);
                selection.removeAllRanges();
                selection.addRange(docRange);
            } catch (ex) {
                Utils.error('ClipBoardHandlerMixin.grabClipboardFocus(): failed to select clipboard node: ' + ex);
            }
        };

        /**
         * Clears the browser selection from the clipboard node, and de-selects all its
         * contents.
         *
         */
        this.clearClipboardSelection = function () {

            // the browser selection
            var selection = window.getSelection();

            // Clear the old browser selection.
            // Bug 28515, bug 28711: IE fails to clear the selection (and to modify
            // it afterwards), if it currently points to a DOM node that is not
            // visible anymore (e.g. the 'Show/hide side panel' button). Workaround
            // is to move focus to an editable DOM node which will cause IE to update
            // the browser selection object. The target container node cannot be used
            // for that, see comments above for bug 26283. Using another focusable
            // node (e.g. the body element) is not sufficient either. Interestingly,
            // even using the (editable) clipboard node does not work here. Setting
            // the new browser selection below will move the browser focus back to
            // the application pane.
            Utils.clearBrowserSelection();

            // set the browser selection
            try {
                selection.removeAllRanges();
            } catch (ex) {
                Utils.error('ClipBoardHandlerMixin.clearClipboardSelection(): failed to select clipboard node: ' + ex);
            }
        };

        /**
         * Copies the current selection into the internal clipboard and deletes
         * the selection.
         *
         * @param {jQuery.Event} event
         *  The jQuery browser event object.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.cut = function (event) {

            var // the clipboard event data
                clipboardData = Utils.getClipboardData(event),
                // the selection object
                selection = self.getSelection();

            // set the internal clipboard data and
            // add the data to the clipboard event if the browser supports the clipboard api
            self.copy(event);

            // if the browser supports the clipboard api, use the copy function to add data to the clipboard
            // the iPad supports the api, but cut doesn't work
            if (clipboardData && !Utils.IOS) {

                // prevent default cut handling for desktop browsers, but not for touch devices
                if (!Utils.TOUCHDEVICE) {
                    event.preventDefault();
                }

                if (isSlideCopied()) {
                    self.deleteMultipleSlides(self.getSlidePaneSelection());
                    return;
                } else {
                    // delete current selection
                    return self.deleteSelected({ deleteKey: true }).done(function () {
                        selection.setTextSelection(selection.getStartPosition()); // setting the cursor position
                    });
                }
            }

            return self.executeDelayed(function () {
                if (isSlideCopied()) {
                    self.deleteMultipleSlides(self.getSlidePaneSelection());
                } else {
                    // focus and restore browser selection
                    selection.restoreBrowserSelection();
                    // delete restored selection
                    return self.deleteSelected({ deleteKey: true }).done(function () {
                        selection.setTextSelection(selection.getStartPosition()); // setting the cursor position
                    });
                }
            }, undefined, 'Text: cut');
        };

        /**
         * Copies the current selection into the internal clipboard and
         * attaches the clipboard data to the copy event, if the browser
         * supports the clipboard api.
         *
         * @param {jQuery.Event} event
         *  The jQuery browser event object.
         */
        this.copy = function (event) {

            var // the clipboard div
                clipboard,
                // the clipboard event data
                clipboardData = Utils.getClipboardData(event),
                // html clipboard data cleaned up for export
                htmlExportData,
                // the selection object
                selection = self.getSelection(),
                // start of current selection
                start = selection.getStartPosition(),
                // end of current selection
                end = selection.getEndPosition(),
                // information about the cell range and containing table
                cellRangeInfo,
                // is the cell defined by the start position the first cell in the row
                isFirstCell,
                // is the cell defined by the end position the last cell in the row
                isLastCell,
                // flag to determine if focus should be returned to slidepane or not
                returnFocusToSidepane = false;

            if (_.browser.IE && (end[0] - start[0]) > 200) {
                // bigger selection take more time to copy,
                // in IE it can take more than 10seconds,
                // so we warn the user
                /* eslint-disable */
                if (!window.confirm(gt('The selected text is very long. It will take some time to be copied. Do you really want to continue?'))) { return; }
                /* eslint-enable */
            }

            // generate a new unique id to identify the clipboard operations
            self.setClipboardId(ox.session + ':' + _.uniqueId());
            // set clipboard data origin
            self.setClipboardOrigin(app.getName());

            switch (selection.getSelectionType()) {

                case 'text':
                    if (self.useSlideMode()) {
                        // OX Presentation
                        if (selection.isAdditionalTextframeSelection() && selection.hasRange()) {
                            self.setClipboardOperations(copyTextSelection());
                        } else if (selection.isAdditionalTextframeSelection()) {
                            self.setClipboardOperations(copyDrawingSelection());
                        } else {
                            self.setClipboardOperations(copySlides());
                            returnFocusToSidepane = true;
                        }

                    } else {
                        // OX Text
                        cellRangeInfo = selection.getSelectedCellRange();
                        isFirstCell = $(Position.getLastNodeFromPositionByNodeName(self.getNode(), start, DOM.TABLE_CELLNODE_SELECTOR)).prev().length === 0;
                        isLastCell = $(Position.getLastNodeFromPositionByNodeName(self.getNode(), end, DOM.TABLE_CELLNODE_SELECTOR)).next().length === 0;

                        // if the selected range is inside the same table or parent table and
                        // the start position is the first cell in the start row and the end position
                        // is the last cell in the end row use table selection otherwise, use text selection.
                        if (cellRangeInfo && isFirstCell && isLastCell && !_.isEqual(cellRangeInfo.firstCellPosition, cellRangeInfo.lastCellPosition)) {
                            self.setClipboardOperations(copyCellRangeSelection());
                        } else {
                            self.setClipboardOperations(copyTextSelection());
                        }
                    }
                    break;

                case 'drawing':
                    if (self.useSlideMode()) {
                        self.setClipboardOperations(copyDrawingSelection());
                    } else {
                        self.setClipboardOperations(copyTextSelection());
                    }
                    break;

                case 'cell':
                    self.setClipboardOperations(copyCellRangeSelection());
                    break;

                default:
                    self.setClipboardId('');
                    Utils.error('Editor.copy(): unsupported selection type: ' + selection.getSelectionType());
            }

            htmlExportData = Export.getHTMLFromSelection(self, { clipboardId: self.getClipboardId(), clipboardOperations: self.getClipboardOperations(), clipboardOrigin: app.getName() });
            // set clipboard debug pane content
            self.trigger('debug:clipboard', htmlExportData);

            // if browser supports clipboard api add data to the event
            // chrome say it supports clipboard api, but it does not!!!
            // Bug 39428
            if (clipboardData) {
                // add operation data
                clipboardData.setData('text/ox-operations', JSON.stringify(self.getClipboardOperations()));

                // add operation data origin
                clipboardData.setData('text/ox-origin', app.getName());

                // add plain text and html of the current browser selection
                clipboardData.setData('text/plain', selection.getTextFromBrowserSelection());
                clipboardData.setData('text/html', htmlExportData);

                // prevent default copy handling for desktop browsers, but not for touch devices
                event.preventDefault();

                if (returnFocusToSidepane) { // can only be true in Presentation app
                    self.executeDelayed(function () {
                        self.clearClipboardSelection();
                        app.getView().getSlidePane().getSlidePaneContainer().focus();
                    }, undefined, 'Text: copy');
                }
            } else {

                // copy the currently selected nodes to the clipboard div and append it to the body
                clipboard = app.getView().createClipboardNode();

                clipboard.append(htmlExportData);

                // focus the clipboard node and select all of it's child nodes
                //selection.setBrowserSelectionToContents(clipboard);
                self.grabClipboardFocus(clipboard);

                self.executeDelayed(function () {
                    // set the focus back
                    if (returnFocusToSidepane) {  // can only be true in Presentation app
                        self.clearClipboardSelection();
                        app.getView().getSlidePane().getSlidePaneContainer().focus();
                    } else {
                        app.getView().grabFocus();

                    }

                    // remove the clipboard node
                    clipboard.remove();
                }, undefined, 'Text: copy');
            }
        };

        /**
         * Deletes the current selection and pastes the internal clipboard to
         * the resulting cursor position.
         *
         * @param {Number[]} dropPosition
         *  An optional logical position used for pasting the internal clipboard
         *  content.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.pasteInternalClipboard = function (dropPosition) {

            // check if clipboard contains something
            if (!self.hasInternalClipboard()) { return $.when(); }

            // make sure that only one paste call is processed at the same time
            if (self.checkSetClipboardPasteInProgress()) { return $.when(); }

            // Group all executed operations into a single undo action.
            // The undo manager returns the return value of the callback function.
            return self.getUndoManager().enterUndoGroup(function () {

                var // the deferred to keep the undo group open until it is resolved or rejected
                    undoDef = $.Deferred(),
                    // target position to paste the clipboard contents to
                    anchorPosition = null,
                    // the generated paste operations with transformed positions
                    operations = null,
                    // operation generator for additional insert style sheets operations
                    generator = self.createOperationsGenerator(),
                    // the next free list style number part
                    listStyleNumber = 1,
                    // the list style map
                    listStyleMap = {},
                    // a snapshot object
                    snapshot = null,
                    // whether the user aborted the pasting action
                    userAbort = false,
                    // whether there is a selection range before pasting
                    hasRange = false,
                    // the logical range position
                    rangeStart = null, rangeEnd = null,
                    // the list collection object
                    listCollection = self.getListCollection(),
                    // whether a operation was removed from the list of operations
                    operationRemoved = false,
                    // the selection object
                    selection = self.getSelection();

                // builds the list style map from insertListStyle operations, mapping the list style ids
                // of the source document to the list style ids of the destination document
                function createListStyleMap(operation) {

                    var listStyleId,
                        isKnownListStyle = false;

                    function getFreeListStyleId() {

                        var sFreeId = 'L';

                        while (listCollection.hasListStyleId('L' + listStyleNumber)) {
                            listStyleNumber++;
                        }
                        sFreeId += listStyleNumber;
                        listStyleNumber++;
                        return sFreeId;
                    }

                    if ((operation.name === Operations.INSERT_LIST) && _.isObject(operation.listDefinition)) {

                        // deep copy the operation before changing
                        operation = _.copy(operation, true);

                        // check if we already have a list style with the list definition and if not create a new id
                        listStyleId = listCollection.getListStyleIdForListDefinition(operation.listDefinition);

                        if (listStyleId) {
                            isKnownListStyle = true;
                        } else {
                            listStyleId = getFreeListStyleId();
                        }

                        listStyleMap[operation.listStyleId] = listStyleId;
                        operation.listStyleId = listStyleId;

                        // handling the base style id (that cannot be sent to the server)
                        if (operation.baseStyleId) {

                             // if the base style Id is the same, do not send any operation (filter cannot handle base style id)
                            if (isKnownListStyle && operation.baseStyleId === listCollection.getBaseStyleIdFromListStyle(operation.listStyleId)) {
                                operationRemoved = true;  // needs to be marked for cleanup
                                return null;
                            }

                            // but never send baseStyleId information to the server because filter cannot handle it correctly
                            delete operation.baseStyleId;
                        }

                        // make sure every list level has a listStartValue set
                        _.each(operation.listDefinition, function (listLevelDef) {
                            if (!('listStartValue' in listLevelDef)) {
                                listLevelDef.listStartValue = 1;
                            }
                        });
                    }

                    return operation;
                }

                // transforms a position being relative to [0,0] to a position relative to anchorPosition
                function transformPosition(position) {

                    var // the resulting position
                        resultPosition = null;

                    if ((position[0] === 0) && (position.length > 1)) {
                        // adjust text/drawing offset for first paragraph
                        resultPosition = anchorPosition.slice(0, -1);
                        resultPosition.push(anchorPosition[anchorPosition.length - 1] + position[1]);
                        resultPosition = resultPosition.concat(position.slice(2));
                    } else {
                        // adjust paragraph offset for following paragraphs
                        resultPosition = anchorPosition.slice(0, -2);
                        resultPosition.push(anchorPosition[anchorPosition.length - 2] + position[0]);
                        resultPosition = resultPosition.concat(position.slice(1));
                    }

                    return resultPosition;
                }

                // if the operation references a style try to add the style sheet to the document
                function addMissingStyleSheet(operation) {
                    if ((operation.name === Operations.TABLE_INSERT) && _.isObject(operation.attrs) && _.isString(operation.attrs.styleId)) {
                        // generate operation to insert a dirty table style into the document
                        self.generateMissingStyleSheetOperations(generator, 'table', operation.attrs.styleId);
                    }
                }

                // apply list style mapping to setAttributes and insertParagraph operations
                function mapListStyles(operation) {

                    if ((operation.name === Operations.SET_ATTRIBUTES || operation.name === Operations.PARA_INSERT) &&
                            operation.attrs && operation.attrs.paragraph && _.isString(operation.attrs.paragraph.listStyleId) &&
                        listStyleMap[operation.attrs.paragraph.listStyleId]) {

                        // apply the list style id from the list style map to the operation
                        operation.attrs.paragraph.listStyleId = listStyleMap[operation.attrs.paragraph.listStyleId];
                    }
                }

                // changes all drawing from floating to inline
                function mapDrawing(operation) {
                    if (operation.name === Operations.DRAWING_INSERT && operation.attrs && operation.attrs.drawing && !operation.attrs.drawing.inline) {
                        _.extend(operation.attrs.drawing, { inline: true, anchorHorBase: null, anchorHorAlign: null, anchorHorOffset: null, anchorVertBase: null, anchorVertAlign: null, anchorVertOffset: null, textWrapMode: null, textWrapSide: null });
                    }
                }

                // transforms the passed operation relative to anchorPosition
                function transformOperation(operation) {

                    // clone the operation to transform the positions (no deep clone,
                    // as the position arrays will be recreated, not modified inplace)
                    operation = _.clone(operation);

                    // transform position of operation (but not, if a target is defined (for example in comments)
                    if (_.isArray(operation.start) && !operation.target) {
                        // start may exist but is relative to position then
                        operation.start = transformPosition(operation.start);
                        // attribute 'end' only with attribute 'start'
                        if (_.isArray(operation.end)) {
                            operation.end = transformPosition(operation.end);
                        }
                        addMissingStyleSheet(operation);
                    }

                    // map list style ids from source to destination document
                    mapListStyles(operation);
                    // change drawing from floating to inline - fix for bug #32873
                    mapDrawing(operation);

                    var text = '  name="' + operation.name + '", attrs=';
                    var op = _.clone(operation);
                    delete op.name;
                    Utils.log(text + JSON.stringify(op));

                    return operation;
                }

                // helper function to check, if the operations can be pasted to the anchor position
                function isForbiddenPasting(anchorPosition, clipboardOperations) {

                    // helper function to find insertDrawing operations with type 'shape'
                    function containsInsertShapeDrawing(operations) {
                        return _.find(operations, function (operation) {
                            return operation.name && operation.name === 'insertDrawing' && operation.type && operation.type === 'shape';
                        });
                    }

                    // helper function to find specified operations in the list of operations
                    function containsSpecifiedOperations(operations, operationList) {
                        return _.find(operations, function (operation) {
                            return operation.name && _.contains(operationList, operation.name);
                        });
                    }

                    // pasting text frames into text frames or comments is not supported
                    if (containsInsertShapeDrawing(clipboardOperations)) {
                        if (Position.isPositionInsideTextframe(self.getNode(), anchorPosition)) {
                            // inform the user, that this pasting is not allowed
                            app.getView().yell({ type: 'info', message: gt('Pasting shapes into text frames is not supported.') });
                            return true;
                        } else if (self.isCommentFunctionality()) {
                            // inform the user, that this pasting is not allowed
                            app.getView().yell({ type: 'info', message: gt('Pasting shapes into comments is not supported.') });
                            return true;
                        }
                    }

                    // pasting tables or drawings into 'shapes with text content' in odf or into comments in odf is not supported
                    if (containsSpecifiedOperations(clipboardOperations, ['insertDrawing', 'insertTable']) && (self.isReducedOdfTextframeFunctionality() || self.isOdfCommentFunctionality())) {
                        // inform the user, that this pasting is not allowed
                        app.getView().yell({ type: 'info', message: gt('Pasting content into this object is not supported.') });
                        return true;
                    }

                    return false;
                }

                // finding a valid text cursor position after pasting
                function getFinalCursorPosition(defaultPosition, operations) {

                    var // the determined cursor position
                        cursorPosition = defaultPosition,
                        // the searched split paragraph operation
                        searchOperation = null,
                        // searched range end operation of type field
                        rangeEndFieldOperation = null,
                        // last operation
                        lastOperation = null,
                        // searching for a splitting of paragraph
                        searchOperationName = Operations.PARA_SPLIT,
                        // the position after splitting a paragraph
                        splitPosition = null;

                    if (operations && operations.length > 0) {
                        // setting the cursor after pasting.
                        // setting cursor after pasting complex field as last operation
                        lastOperation = _.last(operations);
                        // taking care of pasting range end of complex field as last operation, (#43699)
                        rangeEndFieldOperation = lastOperation.name === Operations.RANGE_INSERT && lastOperation.type === 'field' && lastOperation.position === 'end';
                        if (rangeEndFieldOperation) {
                            cursorPosition = Position.increaseLastIndex(lastOperation.start);
                        } else {
                            // Also taking care of splitted paragraphs (36471) -> search last split paragraph operation
                            searchOperation = _.find(operations.reverse(), function (operation) {
                                return operation.name && operation.name === searchOperationName;
                            });
                        }
                    }

                    if (searchOperation) {
                        // is this position after split behind the defaultPosition? Then is should be used
                        splitPosition = _.clone(searchOperation.start);
                        splitPosition.pop();  // the paragraph position
                        splitPosition = Position.increaseLastIndex(splitPosition);
                        splitPosition.push(0);  // setting cursor to start of second part of splitted paragraph
                        if (Position.isValidPositionOrder(cursorPosition, splitPosition)) { cursorPosition = splitPosition; }
                    }

                    return cursorPosition;
                }

                // applying the paste operations
                function doPasteInternalClipboard() {

                    var // the apply actions promise
                        applyPromise = null,
                        // the newly created operations
                        newOperations = null,
                        // the current target node
                        target = self.getActiveTarget(),
                        // whether the clipboard contains only text or also drawings
                        drawingInClipboard = checkClipboardForDrawings(self.getClipboardOperations()),
                        // the target position for drawings pasted from OX Text
                        textPasteDrawingTarget = selection.isTopLevelTextCursor() ? { top: 0, left: 0 } : getMinPositionForSelectedDrawings();

                    // init the next free list style number part
                    if (listCollection) { listStyleNumber = parseInt(listCollection.getFreeListId().slice(1, listCollection.getFreeListId().length), 10); }
                    if (!_.isNumber(listStyleNumber)) {
                        listStyleNumber = 1;
                    }

                    // paste clipboard to current cursor position
                    anchorPosition = dropPosition ? dropPosition.start : selection.getStartPosition();
                    if (anchorPosition.length >= 2) {
                        if (self.useSlideMode()) {
                            // determin anchor position depending on the selection and on the clipboard containing drawings
                            if (selection.isTopLevelTextCursor()) {
                                anchorPosition = self.getNextAvailablePositionInActiveSlide();
                            } else {
                                // do not append to the currently selected text frame if the clipboard contains a drawing
                                if (drawingInClipboard) {
                                    anchorPosition = self.getNextAvailablePositionInActiveSlide();
                                    selection.setTextSelection(anchorPosition);
                                }
                            }
                        } else {
                            // check for pasting comments into comments
                            if (self.getActiveTarget()) { self.setClipboardOperations(self.getCommentLayer().handleCommentOperationsForPasting(self.getClipboardOperations())); }
                            // doing some checks, if the content can be pasted into the destination element.
                            if (isForbiddenPasting(anchorPosition, self.getClipboardOperations())) {
                                return $.when();
                            }
                        }

                        Utils.info('Editor.pasteInternalClipboard()');
                        self.setBlockKeyboardEvent(true);

                        // creating a snapshot
                        if (!snapshot) { snapshot = new Snapshot(app); }

                        // show a nice message with cancel button
                        app.getView().enterBusy({
                            cancelHandler: function () {
                                userAbort = true;  // user aborted the pasting process
                                // restoring the document state
                                snapshot.apply();
                                // calling abort function for operation promise
                                app.enterBlockOperationsMode(function () { if (applyPromise && applyPromise.abort) { applyPromise.abort(); } });
                            },
                            warningLabel: gt('Sorry, pasting from clipboard will take some time.')
                        });

                        // map the operations
                        operations = listCollection ? _(self.getClipboardOperations()).map(createListStyleMap) : self.getClipboardOperations();
                        if (operationRemoved) { operations = Utils.removeFalsyItemsInArray(operations); }

                        // map operations from OX Text to OX Presenation
                        if (self.useSlideMode() && (!drawingInClipboard || self.isClipboardOriginText())) {
                            operations = makeTextFromTableOperations(operations);
                            operations = transformOperationsToSlideMode(operations);
                        }

                        // shifting the positions of the new inserted drawings
                        if (self.useSlideMode() && drawingInClipboard) {
                            shiftDrawingPositions(operations, textPasteDrawingTarget);
                        }

                        operations = _(operations).map(transformOperation);

                        // concat with newly generated operations
                        newOperations = generator.getOperations();
                        if (newOperations.length) {
                            operations = newOperations.concat(operations);
                        }

                        // adding change track information, if required
                        if (self.getChangeTrack().isActiveChangeTracking()) {
                            self.getChangeTrack().handleChangeTrackingDuringPaste(operations);
                        } else {
                            self.getChangeTrack().removeChangeTrackInfoAfterSplitInPaste(operations);
                        }

                        // handling target positions (for example inside comments)
                        self.getCommentLayer().handlePasteOperationTarget(operations);
                        self.getFieldManager().handlePasteOperationTarget(operations);

                        // check if destination paragraph for paste already has pageBreakBefore attribute
                        self.getPageLayout().handleManualPageBreakDuringPaste(operations);

                        self.doCheckImplicitParagraph(anchorPosition);

                        // if pasting into header or footer or comment, extend operations with target
                        if (target) {
                            _.each(operations, function (operation) {
                                if (operation.name !== Operations.INSERT_STYLESHEET) {
                                    operation.target = target;
                                }
                            });
                        }

                        // set clipboard debug pane content
                        self.trigger('debug:clipboard', operations);

                        // apply operations
                        applyPromise = self.applyOperations(operations, { async: true })
                        .progress(function (progress) {
                            app.getView().updateBusyProgress(progress);
                        });

                    } else {
                        Utils.warn('Editor.pasteInternalClipboard(): invalid cursor position');
                        applyPromise = $.Deferred().reject();
                    }

                    return applyPromise;
                }

                // special handling for selected drawings
                if (self.isDrawingSelected()) {
                    if (selection.isTextFrameSelection()) {
                        selection.setSelectionIntoTextframe(); // if a text frame is selected, the new content is inserted into the text frame
                    } else if (!self.useSlideMode()) {
                        selection.setTextSelection(selection.getStartPosition()); // if a drawing is selected in text mode, the new content is inserted before the drawing
                    }
                }

                hasRange = selection.hasRange();

                // creating a snapshot
                if (hasRange) {
                    snapshot = new Snapshot(app);
                    rangeStart = _.clone(self.getSelection().getStartPosition());
                    rangeEnd = _.clone(self.getSelection().getEndPosition());
                }

                // delete current selection for text mode only
                (self.useSlideMode() ? $.when() : (self.deleteSelected({ alreadyPasteInProgress: true, snapshot: snapshot })))
                .then(doPasteInternalClipboard)
                .always(function () {
                    // no longer blocking page break calculations (40107)
                    self.setBlockOnInsertPageBreaks(false);
                    // leaving busy mode
                    self.leaveAsyncBusy();
                    // close undo group
                    undoDef.resolve();
                    // deleting the snapshot
                    if (snapshot) { snapshot.destroy(); }
                    // setting the cursor range after cancel by user
                    if (userAbort && hasRange) { self.getSelection().setTextSelection(rangeStart, rangeEnd); }

                }).done(function () {
                    var lastOperation = _.last(operations);
                    var finalCursorPos = getFinalCursorPosition(self.getLastOperationEnd(), operations);
                    // in slide mode select the previously inserted image
                    if (self.useSlideMode() && lastOperation.name === Operations.DRAWING_INSERT && lastOperation.type === 'image') {
                        self.getSelection().setTextSelection(Position.decreaseLastIndex(finalCursorPos), finalCursorPos);
                    } else {
                        self.getSelection().setTextSelection(finalCursorPos);
                    }
                });

                return undoDef.promise();

            }, this); // enterUndoGroup()
        };

        /**
         * Pastes the clipboard containing copied slides behind the position of last selected slide.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.pasteSlidesInternal = function () {
            var operations = self.getClipboardOperations();
            var firstOp = operations && _.first(operations);
            var isFirstOpInsertSlide = firstOp && firstOp.name === 'insertSlide';
            var isFirstOpInsertLayout = firstOp && firstOp.name === 'insertLayoutSlide';

            // check if clipboard contains something
            if (!self.hasInternalClipboard()) { return $.when(); }

            // don't paste slides if it's not Presentation app
            if (!self.useSlideMode()) { return $.when(); }

            // don't allow pasting of standard slides into master view, and vice versa
            if (self.isMasterView()) {
                if (isFirstOpInsertSlide) {
                    return $.when();
                }
            } else {
                if (isFirstOpInsertLayout) {
                    return $.when();
                }
            }

            // make sure that only one paste call is processed at the same time
            if (self.checkSetClipboardPasteInProgress()) { return $.when(); }

            // Group all executed operations into a single undo action.
            // The undo manager returns the return value of the callback function.
            return self.getUndoManager().enterUndoGroup(function () {

                var // the deferred to keep the undo group open until it is resolved or rejected
                    undoDef = $.Deferred(),
                    // target position to paste the clipboard contents to
                    anchorPosition = null,
                    // operation generator for additional insert style sheets operations
                    generator = self.createOperationsGenerator(),
                    // a snapshot object
                    snapshot = null,
                    // whether the user aborted the pasting action
                    userAbort = false,
                    // whether there is a selection range before pasting
                    hasRange = false,
                    // the logical range position
                    rangeStart = null, rangeEnd = null,
                    // used for target of elements on pasted new layout slide
                    nextLayoutId = null,
                    // start position of layout slide
                    layoutSlideStart = null;

                // transforms a position being relative to [0,0] to a position relative to anchorPosition
                function transformPosition(position) {

                    var // the resulting position
                        resultPosition = position;

                    resultPosition.shift();
                    resultPosition.unshift(anchorPosition);

                    return resultPosition;
                }

                // transforms the passed operation relative to anchorPosition
                function transformOperation(operation) {

                    // clone the operation to transform the positions (no deep clone,
                    // as the position arrays will be recreated, not modified inplace)
                    operation = _.clone(operation);

                    if (operation.name === 'insertLayoutSlide') {
                        if (!nextLayoutId) {
                            nextLayoutId = self.getNextCustomLayoutId();
                        } else {
                            nextLayoutId = (parseInt(nextLayoutId, 10) + 1) + ''; // increase manually next layout Id, since the operations are not applied yet
                        }
                        operation.id = nextLayoutId;
                        operation.target = self.isMasterSlideId(self.getActiveSlideId()) ? self.getActiveSlideId() : self.getMasterSlideId(self.getActiveSlideId());
                        layoutSlideStart = _.isNull(layoutSlideStart) ? anchorPosition : layoutSlideStart;
                        operation.start = layoutSlideStart;
                        layoutSlideStart += 1;
                        anchorPosition = 0; // layout slides have start pos always 0
                    } else if (isFirstOpInsertLayout) {
                        operation.target  = nextLayoutId;
                    }

                    if (operation.name === 'insertSlide') {
                        anchorPosition += 1; // increase position for new slide
                        operation.start = [anchorPosition];
                        operation.target = self.getLayoutIdFromType(operation.target);
                    }

                    // transform position of operation
                    if (_.isArray(operation.start)) {
                        // start may exist but is relative to position then
                        operation.start = transformPosition(operation.start);
                        // attribute 'end' only with attribute 'start'
                        if (_.isArray(operation.end)) {
                            operation.end = transformPosition(operation.end);
                        }
                    }

                    var text = '  name="' + operation.name + '", attrs=';
                    var op = _.clone(operation);
                    delete op.name;
                    Utils.log(text + JSON.stringify(op));

                    return operation;
                }

                // applying the paste operations
                function doPasteSlidesInternal() {
                    var // the apply actions promise
                        applyPromise = null,
                        // the newly created operations
                        newOperations = null;

                    // paste clipboard to current cursor position
                    anchorPosition = self.getSlidePaneSelection() && self.getSlidePaneSelection().sort(function (a, b) { return a - b; });
                    anchorPosition = _.last(anchorPosition);
                    if (_.isNumber(anchorPosition)) {
                        Utils.info('Editor.pasteSlidesInternal()');
                        self.setBlockKeyboardEvent(true);

                        // creating a snapshot
                        if (!snapshot) { snapshot = new Snapshot(app); }

                        // show a message with cancel button
                        app.getView().enterBusy({
                            cancelHandler: function () {
                                userAbort = true;  // user aborted the pasting process
                                // restoring the document state
                                snapshot.apply();
                                // calling abort function for operation promise
                                app.enterBlockOperationsMode(function () { if (applyPromise && applyPromise.abort) { applyPromise.abort(); } });
                            },
                            warningLabel: gt('Sorry, pasting slides from clipboard will take some time.')
                        });

                        // map the operations
                        operations = _(operations).map(transformOperation);

                        // concat with newly generated operations
                        newOperations = generator.getOperations();
                        if (newOperations.length) {
                            operations = newOperations.concat(operations);
                        }

                        // set clipboard debug pane content
                        self.trigger('debug:clipboard', operations);

                        // apply operations
                        applyPromise = self.applyOperations(operations, { async: true })
                        .progress(function (progress) {
                            app.getView().updateBusyProgress(progress);
                        });

                    } else {
                        Utils.warn('Editor.pasteInternalClipboard(): invalid cursor position');
                        applyPromise = $.Deferred().reject();
                    }

                    return applyPromise;
                }

                doPasteSlidesInternal()
                .always(function () {
                    // leaving busy mode
                    self.leaveAsyncBusy();
                    // close undo group
                    undoDef.resolve();
                    // deleting the snapshot
                    if (snapshot) { snapshot.destroy(); }
                    // setting the cursor range after cancel by user
                    if (userAbort && hasRange) { self.getSelection().setTextSelection(rangeStart, rangeEnd); }

                }).done(function () {
                    self.changeToSlide(nextLayoutId || self.getSlideIdByPosition(anchorPosition));
                    self.executeDelayed(function () {
                        self.clearClipboardSelection();
                        app.getView().getSlidePane().getSlidePaneContainer().focus();
                    });
                });

                return undoDef.promise();

            }, this); // enterUndoGroup()
        };

        /**
         * Handler function for the events 'paste' and 'beforepaste'.
         *
         * @param {jQuery.Event} event
         *  The jQuery browser event object.
         */
        this.paste = function (event) {

            var // the clipboard div
                clipboard,
                // the clipboard event data
                clipboardData = Utils.getClipboardData(event),
                // the list items of the clipboard event data
                items = clipboardData && clipboardData.items,
                // the list of mime types of the clipboard event data
                types = clipboardData && clipboardData.types,
                // the operation data from the internal clipboard
                eventData,
                // the operation data origin
                eventDataOrigin,
                // the file reader
                reader,
                // the list item of type text/html
                htmlEventItem = null,
                // the list item of type text/plain
                textEventItem = null,
                // the file list item
                fileEventItem = null,
                // the event URL data
                urlEventData = null;

            // bug 44261
            // we had a case where a paste event could not be canceld by calling prevent default in the key down handler,
            // so we need to check for blocked keyboard events here as well.
            if (self.getBlockKeyboardEvent()) {
                event.preventDefault();
                return false;
            }

            if (!self.getEditMode()) {
                //paste via burger-menu in FF and Chrome must be handled
                app.rejectEditAttempt();
                event.preventDefault();
                return;
            }

            // handles the result of reading file data from the file blob received from the clipboard data api
            function onLoadHandler(evt) {
                var data = evt && evt.target && evt.target.result;

                if (data && data.substring(0, 10) === 'data:image') {
                    createOperationsFromExternalClipboard([{ operation: Operations.DRAWING_INSERT, data: data, depth: 0 }]);
                } else {
                    app.rejectEditAttempt('image');
                }
            }

            // returns true if the html clipboard has a matching clipboard id set
            function isHtmlClipboardIdMatching(html) {
                return ($(html).find('#ox-clipboard-data').attr('data-ox-clipboard-id') === self.getClipboardId());
            }

            // returns the operations attached to the html clipboard, or null
            function getHtmlAttachedOperations(html) {
                var operations;

                try {
                    operations = JSON.parse($(html).find('#ox-clipboard-data').attr('data-ox-operations') || '{}');
                } catch (e) {
                    Utils.warn('getHtmlAttachedOperations', e);
                    operations = null;
                }

                return operations;
            }

            // returns the origin attached to the html clipboard, or an empty String
            function getHtmlAttachedOrigin(html) {
                return ($(html).find('#ox-clipboard-data').attr('data-ox-origin') || '');
            }

            // if the browser supports the clipboard api, look for operation data
            // from the internal clipboard to handle as internal paste.
            if (clipboardData) {
                eventData = clipboardData.getData('text/ox-operations');
                eventDataOrigin = clipboardData.getData('text/ox-origin');
                if (eventData) {
                    // prevent default paste handling for desktop browsers, but not for touch devices
                    if (!Utils.TOUCHDEVICE) {
                        event.preventDefault();
                    }

                    // set the operations from the event to be used for the paste
                    self.setClipboardOperations((eventData.length > 0) ? JSON.parse(eventData) : []);
                    self.setClipboardOrigin(eventDataOrigin);
                    if (isSlideCopied()) {
                        self.pasteSlidesInternal();
                    } else {
                        self.pasteInternalClipboard();
                    }
                    return;
                }

                // check if clipboardData contains a html item
                htmlEventItem = _.find(items, function (item) { return item.type.toLowerCase() === 'text/html'; });

                // Chrome doesn't paste images into a (content editable) div, check if clipboardData contains an image item
                fileEventItem = _.find(items, function (item) { return item.type.toLowerCase().indexOf('image') !== -1; });

                // check if we have a mime type to get an URL from
                urlEventData = clipboardData.getData(_.find(types, function (type) { return type.toLowerCase().indexOf('text/uri-list') !== -1; }));

                // check if clipboardData contains a plain item
                textEventItem = _.find(items, function (item) { return item.type.toLowerCase() === 'text/plain'; });

                if (htmlEventItem || textEventItem) {
                    (htmlEventItem || textEventItem).getAsString(function (content) {
                        var div, ops;

                        // set clipboard debug pane content
                        self.trigger('debug:clipboard', content);

                        // clean up html to put only harmless html into the <div>
                        div = $('<div>').html(Utils.parseAndSanitizeHTML(content));

                        if (htmlEventItem) {
                            if (isHtmlClipboardIdMatching(div)) {
                                // if the clipboard id matches it's an internal copy & paste
                                if (isSlideCopied()) {
                                    self.pasteSlidesInternal();
                                } else {
                                    self.pasteInternalClipboard();
                                }

                            } else {
                                self.setClipboardOrigin(getHtmlAttachedOrigin(div));
                                ops = getHtmlAttachedOperations(div);
                                if (_.isArray(ops)) {
                                    // it's not an internal copy & paste, but we have clipboard operations piggy backed to use for clipboardOperations
                                    self.setClipboardOperations(ops);
                                    if (isSlideCopied()) {
                                        self.pasteSlidesInternal();
                                    } else {
                                        self.pasteInternalClipboard();
                                    }

                                } else {
                                    // use html clipboard
                                    ops = self.parseClipboard(div);
                                    createOperationsFromExternalClipboard(ops);
                                }
                            }
                        } else {
                            //text only
                            ops = self.parseClipboardText(div.text());
                            createOperationsFromExternalClipboard(ops);
                        }
                    });

                } else if (fileEventItem) {
                    reader = new window.FileReader();
                    reader.onload = onLoadHandler;
                    reader.readAsDataURL(fileEventItem.getAsFile());

                } else if (urlEventData && ImageUtil.hasUrlImageExtension(urlEventData)) {
                    createOperationsFromExternalClipboard([{ operation: Operations.DRAWING_INSERT, data: urlEventData, depth: 0 }]);
                }

                if (htmlEventItem || fileEventItem || urlEventData || textEventItem) {
                    // prevent default paste handling of the browser
                    event.preventDefault();
                    return;
                }
            }

            // append the clipboard div to the body and place the cursor into it
            clipboard = app.getView().createClipboardNode();

            // focus and select the clipboard container node
            //self.getSelection().setBrowserSelectionToContents(clipboard);
            self.grabClipboardFocus(clipboard);

            // read pasted data
            self.executeDelayed(function () {

                var clipboardData,
                    operations;

                // set the focus back
                app.getView().grabFocus();

                if (isHtmlClipboardIdMatching(clipboard)) {
                    // if the clipboard id matches it's an internal copy & paste
                    if (isSlideCopied()) {
                        self.pasteSlidesInternal();
                    } else {
                        self.pasteInternalClipboard();
                    }
                } else {
                    // look for clipboard origin
                    self.setClipboardOrigin(getHtmlAttachedOrigin(clipboard));

                    // look for clipboard operations
                    operations = getHtmlAttachedOperations(clipboard);
                    if (_.isArray(operations)) {
                        // it's not an internal copy & paste, but we have clipboard operations piggy backed to use for clipboardOperations
                        self.setClipboardOperations(operations);
                        if (isSlideCopied()) {
                            self.pasteSlidesInternal();
                        } else {
                            self.pasteInternalClipboard();
                        }
                    } else {
                        // set clipboard debug pane content
                        self.trigger('debug:clipboard', clipboard);
                        // use html clipboard
                        clipboardData = self.parseClipboard(clipboard);
                        createOperationsFromExternalClipboard(clipboardData);
                    }
                }

                // remove the clipboard node
                clipboard.remove();
                if (self.useSlideMode()) {
                    self.clearClipboardSelection();
                    app.getView().getSlidePane().getSlidePaneContainer().focus();
                }
            }, undefined, 'Text: paste');
        };

        /**
         * Removes all attributes that should not be used with clipboard operations.
         * Doesn't create a copy, modifies the given operations.
         *
         * @param {Array} operations
         *  The array with the clipboard operations to clean.
         *
         * @returns {Array}
         *  The array with the cleaned clipboard operations.
         */
        this.cleanClipboardOperations = function (operations) {
            // for copy/paste slides presentation family should be preserved
            var firstOp = operations && _.first(operations);
            var isPasteSlideMode = firstOp && (firstOp.name === 'insertSlide' || firstOp.name === 'insertLayoutSlide');

            _.each(operations, function (operation) {
                if (_.isObject(operation.attrs)) {
                    // delete empty URL attribute to avoid generation of a hyper link
                    if (operation.attrs.character && _.isEmpty(operation.attrs.character.url)) {
                        delete operation.attrs.character.url;
                    }

                    // remove presentation family for insert drawing operations. copied placeholder drawings are pasted as default drawings
                    if ((operation.name === Operations.DRAWING_INSERT) && (operation.attrs.presentation) && !isPasteSlideMode) {
                        delete operation.attrs.presentation;
                    }
                }
            });

            return operations;
        };

        /**
         * Handler function to process drop events from the browser.
         *
         * @param {Object} event
         *  The drop event sent by the browser.
         */
        this.processDrop = function (event) {

            event.preventDefault();

            if (self.getEditMode() !== true) {
                return false;
            }
            // inactive selection in header&footer
            if (event.target.classList.contains('inactive-selection') || $(event.target).parents('.inactive-selection').length > 0) {
                self.getNode().find('.drop-caret').remove();
                return false;
            }

            // inactive selection in comment node
            if (DOM.isNodeInsideComment(event.target) && (!self.getCommentLayer().isCommentTarget(self.getActiveTarget()) || !DOM.isSurroundingCommentNodeActive(event.target, self.getActiveTarget()))) {
                self.getNode().find('.drop-caret').remove();
                return false;
            }

            var activeRootNode = self.getSelection().getRootNode();
            var files = event.originalEvent.dataTransfer.files,
                dropX = event.originalEvent.clientX,
                dropY = event.originalEvent.clientY,
                dropPosition = Position.getOxoPositionFromPixelPosition(activeRootNode, dropX, dropY);

            // validating the drop position. This must be a text position. The parent must be a paragraph
            if (dropPosition && dropPosition.start && !Position.getParagraphElement(activeRootNode, _.initial(dropPosition.start))) {
                if (Position.getParagraphElement(activeRootNode, dropPosition.start)) {
                    dropPosition.start.push(0);
                } else {
                    return false;
                }
            }

            if (!files || files.length === 0) {

                // try to find out what type of data has been dropped
                var types = event.originalEvent.dataTransfer.types,
                    detectedDropDataType = null, div = null, url = null, text = null,
                    assuranceLevel = 99, lowerCaseType = null, operations = null;

                if (types && types.length > 0) {

                    _(types).each(function (type) {

                        lowerCaseType = type.toLowerCase();

                        if (lowerCaseType === 'text/ox-operations') {
                            operations = event.originalEvent.dataTransfer.getData(type);
                            // set the operations from the event to be used for the paste
                            self.setClipboardOperations((operations.length > 0) ? JSON.parse(operations) : []);
                            assuranceLevel = 0;
                            detectedDropDataType = 'operations';

                        } else if (lowerCaseType === 'text/html') {

                            var html = event.originalEvent.dataTransfer.getData(type);
                            if (html && assuranceLevel > 1) {
                                // clean up html to put only harmless html into the <div>
                                div = $('<div>').html(Utils.parseAndSanitizeHTML(html));
                                if (div.children().length > 0) {
                                    // Unfortunately we sometimes get broken html from Firefox (D&D from Chrome).
                                    // So it's better to use the plain text part.
                                    assuranceLevel = 1;
                                    detectedDropDataType = 'html';
                                }
                            }
                        } else if (lowerCaseType === 'text/uri-list' ||
                                    lowerCaseType === 'url') {

                            var list = event.originalEvent.dataTransfer.getData(type);
                            if (list && (list.length > 0) && (assuranceLevel > 3)) {
                                assuranceLevel = 3;
                                detectedDropDataType = 'link';
                                url = list;
                            }
                        } else if (lowerCaseType === 'text/x-moz-url') {
                            // FF sometimes (image D&D Chrome->FF) provides only this type
                            // instead of text/uri-list so we are forced to support it, too.
                            var temp = event.originalEvent.dataTransfer.getData(type);
                            // x-moz-url is defined as link + '\n' + caption
                            if (temp && temp.length > 0 && (assuranceLevel > 2)) {
                                var array = temp.split('\n');
                                url = array[0];
                                if (array.length > 1) {
                                    text = array[1];
                                }
                                assuranceLevel = 2;
                                detectedDropDataType = 'link';
                            }
                        } else if (lowerCaseType === 'text/plain' || lowerCaseType === 'text') {

                            var plainText = event.originalEvent.dataTransfer.getData(type);
                            if (plainText && (assuranceLevel > 4)) {
                                assuranceLevel = 4;
                                detectedDropDataType = 'text';
                                text = plainText;
                            }
                        }
                    });
                } else {
                    // IE sometimes don't provide any types but they are accessible getData()
                    // So try to check if we have a Url to check for.
                    url = event.originalEvent.dataTransfer.getData('Url');
                    if (url && url.length > 0) {
                        detectedDropDataType = 'link';
                    }
                }

                if (detectedDropDataType === 'operations') {
                    // use clipboard code to insert operations to document
                    self.pasteInternalClipboard(dropPosition);
                } else if (detectedDropDataType === 'html') {
                    if (div && div.children().length > 0) {
                        // drag&drop detected html
                        var ops = self.parseClipboard(div);
                        createOperationsFromExternalClipboard(ops, dropPosition);
                    }
                } else if (detectedDropDataType === 'link') {
                    // insert detected hyperlink
                    var setText = text || url;
                    if (setText && setText.length) {
                        if (ImageUtil.hasUrlImageExtension(url)) {
                            self.insertImageURL(url, dropPosition);
                        } else {
                            self.insertHyperlinkDirect(url, setText);
                        }
                    }
                } else if (detectedDropDataType === 'text') {
                    if (text && text.length > 0) {
                        insertPlainTextFormatted(text, dropPosition);
                    }
                } else {
                    // fallback try to use 'text' to at least get text
                    text = event.originalEvent.dataTransfer.getData('text');
                    if (text && text.length > 0) {
                        insertPlainTextFormatted(text, dropPosition);
                    }
                }
            } else {
                if (app.isODF() && (DOM.isNodeInsideComment(event.target) || self.getCommentLayer().isCommentTarget(self.getActiveTarget()))) {
                    // odf doesn't support inserting images inside comments, see #43789
                    self.getNode().find('.drop-caret').remove();
                    return false;
                }
                processDroppedImages(event, dropPosition);
            }

            // always clean caret on every drop
            self.executeDelayed(function () {
                self.getNode().find('.drop-caret').remove();
                // set selection to dropped position
                if (dropPosition) {
                    self.getSelection().setTextSelection(dropPosition.start);
                }
            }, { delay: 500 }, 'Text: processDrop');

            return false;
        };

        /**
         * Handler to process dragStart event from the browser.
         *
         * @param {jQuery.event} event
         *  The browser event sent via dragStart
         */
        this.processDragStart = function (event) {

            var // the data transfer object
                dataTransfer = event && event.originalEvent && event.originalEvent.dataTransfer,
                // the data transfer operations
                dataTransferOperations = null,
                // the selection object
                selection = self.getSelection();

            switch (selection.getSelectionType()) {

                case 'text':
                case 'drawing':
                    dataTransferOperations = copyTextSelection();
                    break;

                case 'cell':
                    dataTransferOperations = copyCellRangeSelection();
                    break;

                default:
                    Utils.error('Editor.processDragStart(): unsupported selection type: ' + selection.getSelectionType());
            }

            // if browser supports DnD api add data to the event
            if (dataTransfer) {
                // add operation data
                if (!_.browser.IE) {
                    if (dataTransferOperations) {
                        dataTransfer.setData('text/ox-operations', JSON.stringify(dataTransferOperations));
                    }
                    // add plain text and html of the current browser selection
                    dataTransfer.setData('text/plain', selection.getTextFromBrowserSelection());
                    dataTransfer.setData('text/html', Export.getHTMLFromSelection(self));
                } else {
                    // IE just supports 'Text' & Url. Text is more generic
                    dataTransfer.setData('Text', selection.getTextFromBrowserSelection());
                }
            }
        };

        /**
         * Handler for 'dragover' event.
         * - draws a overlay caret on the 'dragover' event, which indicates in which position
         * the dragged content should be inserted to.
         *
         * @param {jQuery.Event} event
         *  The browser event sent via dragOver
         */
        this.processDragOver = (function (event) {

            var // the dragOver event
                dragOverEvent = event;

            function directCallback(event) {
                dragOverEvent = event;
            }

            function deferredCallback() {

                var // the page node
                    pageNode = self.getNode(),
                    activeRootNode = pageNode; // at begining active root is page node, if h&f is active, change it later

                // comment node is active
                if (DOM.isNodeInsideComment(dragOverEvent.target)) {
                    if (!self.getCommentLayer().isCommentTarget(self.getActiveTarget()) || !DOM.isSurroundingCommentNodeActive(dragOverEvent.target, self.getActiveTarget())) {
                        return;
                    }
                    activeRootNode = self.getSelection().getRootNode();
                }
                // inactive selection in header&footer
                if (dragOverEvent.target.classList.contains('inactive-selection') || $(dragOverEvent.target).parents('.inactive-selection').length > 0) {
                    return;
                }
                // header/footer edit state is active
                if (self.isHeaderFooterEditState()) {
                    if (!dragOverEvent.target.classList.contains('marginal')) {
                        return; // return if target is outside of marginal root node
                    }
                    activeRootNode = self.getSelection().getRootNode(); // root node that is currently active, needed for getting DOM position
                }

                var caretOxoPosition = Position.getOxoPositionFromPixelPosition(activeRootNode, dragOverEvent.originalEvent.clientX, dragOverEvent.originalEvent.clientY);
                if (!caretOxoPosition || caretOxoPosition.start.length === 0) { return; }
                var dropCaret = pageNode.find('.drop-caret'),
                    zoom = app.getView().getZoomFactor(),
                    caretPoint = Position.getDOMPosition(activeRootNode, caretOxoPosition.start),
                    caretCoordinate = Utils.getCSSPositionFromPoint(caretPoint, pageNode, zoom),
                    caretLineHeight = caretPoint.node.parentNode.style.lineHeight;
                // adapt drop caret height to the content it is hovering, and reposition the caret
                dropCaret.css('height', caretLineHeight);
                dropCaret.css({
                    top: caretCoordinate.top + 'px',
                    left: caretCoordinate.left + 'px'
                });
                dropCaret.show();
            }

            return self.createDebouncedMethod(directCallback, deferredCallback,  { delay: 50, maxDelay: 100, infoString: 'Text: processDragOver' });
        }());

        /**
         * Handler for 'dragenter' event of the editor root node.
         * - create and appends drop caret, if user drags over editor root node
         *
         * @param {jQuery.Event} event
         *  The browser event sent via dragEnter
         */
        this.processDragEnter = function (event) {

            var // the page node
                pageNode = self.getNode();

            if (pageNode.find(event.originalEvent.target).length > 0 || DOM.isPageNode(event.originalEvent.target)) {
                var collabOverlay = pageNode.find('.collaborative-overlay'),
                    dropCaret = pageNode.find('.drop-caret');
                if (dropCaret.length === 0) {
                    dropCaret = $('<div>').addClass('drop-caret');
                    collabOverlay.append(dropCaret);
                }
            }
        };

        /**
         * Handler for 'dragleave' event of the editor root node.
         * - clears drop caret, if user drags out of the editor root node.
         *
         * @param {jQuery.Event} event
         *  The browser event sent via dragLeave
         */
        this.processDragLeave = function (event) {
            if (DOM.isPageNode(event.originalEvent.target)) {
                self.getNode().find('.drop-caret').remove();
            }
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class ClipBoardHandlerMixin

    // exports ================================================================

    return ClipBoardHandlerMixin;

});
