/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/format/characterstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/textframework/utils/dom'
], function (Utils, Color, LineHeight, StyleCollection, DOM) {

    'use strict';

    var // definitions for character attributes
        DEFINITIONS = {

            fontName: {
                def: 'Arial',
                format: function (element, fontName) {
                    element.css('font-family', this.getCssFontFamily(fontName));
                },
                preview: true // 'true' calls the own 'format' method
            },

            fontSize: {
                def: 11,
                // format: done together with escapement in the format handler
                preview: function (element, fontSize) {
                    fontSize = Utils.round(10 + (fontSize - 10) / 1.5, 0.1);
                    element.css('font-size', Utils.minMax(fontSize, 6, 22) + 'pt');
                }
            },

            bold: {
                def: false,
                format: function (element, state) {
                    element.css('font-weight', state ? 'bold' : 'normal');
                },
                preview: true // 'true' calls the own 'format' method
            },

            italic: {
                def: false,
                format: function (element, state) {
                    element.css('font-style', state ? 'italic' : 'normal');
                },
                preview: true // 'true' calls the own 'format' method
            },

            underline: {
                def: false
            },

            strike: {
                def: 'none'
            },

            vertAlign: { def: 'baseline' },

            color: {
                def: Color.AUTO,
                // format: color will be set in format handler, depending on fill colors
                preview: function (element, color) {
                    var colorModel = this.resolveColor(Color.parseJSON(color), 'text');
                    if (colorModel.y > 0.95) {
                        //white text on white background is bad, we do the MS-Word trick (Bug 40872)
                        element.css('color', 'black');
                    } else {
                        element.css('color', colorModel.css);
                    }
                }
            },

            fillColor: {
                def: Color.AUTO,
                format: function (element, color) {
                    element.css('background-color', this.getCssColor(color, 'fill'));
                }
            },

            language: {
                def: '',
                format: function (element, value) {
                    element.attr('lang', value);
                }
            },

            url: {
                def: '',
                scope: 'element',
                format: function (element, url) {
                    if (_.isString(url) && (url.length > 0)) {
                        element.attr('title', url);
                        element.attr('data-hyperlink-url', url);
                    } else {
                        element.removeAttr('title');
                        element.removeAttr('data-hyperlink-url');
                    }
                }
            },

            autoDateField: {
                def: '',
                format: function (element, value) {
                    if (value) {
                        element.attr('data-auto-date', value);
                    }
                }
            },

            // special attributes

            spellerror: {
                def: false,
                scope: 'element',
                format: function (element, state) {
                    element.toggleClass('spellerror', state);
                },
                special: true
            },

            highlight: {
                def: false,
                scope: 'element',
                format: function (element, state) {
                    element.toggleClass('highlight', state);
                },
                special: true
            }

        },

        PARENT_RESOLVERS = {
            paragraph: function (span) { return span.closest(DOM.PARAGRAPH_NODE_SELECTOR); }
        };

    // class CharacterStyles ==================================================

    /**
     * Contains the style sheets for character formatting attributes. The CSS
     * formatting will be written to text span elements contained somewhere in
     * the paragraph elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    function CharacterStyles(docModel) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'character', {
            parentResolvers: PARENT_RESOLVERS,
            formatHandler: updateCharacterFormatting,
            families: 'changes'
        });

        // private methods ----------------------------------------------------

        /**
         * Will be called for every text span whose character attributes have
         * been changed.
         *
         * @param {jQuery} textSpan
         *  The text span whose character attributes have been changed, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateCharacterFormatting(textSpan, mergedAttributes) {

            var // the parent paragraph of the node (may be a grandparent)
                paragraph = $(textSpan).closest(DOM.PARAGRAPH_NODE_SELECTOR),
                // the closest cell node (if it exists)
                cell = $(textSpan).closest(DOM.TABLE_CELLNODE_SELECTOR),
                // the closest text frame content node (if it exists)
                textframeContent = $(textSpan).closest(DOM.TEXTFRAMECONTENT_NODE_SELECTOR),
                // the paragraph style container
                paragraphStyles = docModel.getStyleCollection('paragraph'),
                // the merged attributes of the paragraph
                paragraphAttributes = paragraphStyles.getElementAttributes(paragraph).paragraph,
                // the character attributes of the passed attribute map
                characterAttributes = mergedAttributes.character,
                // effective text color, font size, and vertical offset
                textColor = characterAttributes.color,
                fontSize = characterAttributes.fontSize,
                textDecoration = self.getCssTextDecoration(characterAttributes),
                position = 'static',
                bottom = null,
                // the background color of the cell and text frame
                cellBackgroundColor = null, textFrameBackgroundColor = null,
                // an array containing all affected fill colors of text, paragraph and cell
                allFillColors = [],
                // the effective line height, that might be reduced inside comment
                effectiveLineHeight = paragraphAttributes.lineHeight,
                // the maximum font size inside a comment
                commentMaxFontSize = 10;

            // restriction of max font size for small devices
            if (Utils.SMALL_DEVICE && fontSize > 16) {
                fontSize = 16;
            }

            // Checking also the background-color of an affected cell, if there is one (28988).
            // For performance reasons, not using attributes (might be assigned to row or table, too),
            // but checking CSS style 'background-color' of the table directly.
            if (cell.length > 0) {
                cellBackgroundColor = Color.parseCSS(cell.css('background-color'), true);
                if (cellBackgroundColor) { allFillColors.push(cellBackgroundColor.toJSON()); }
            }

            // not only checking table cells, but also text frames, for their background-color (36385)
            if (textframeContent.length > 0) {
                textFrameBackgroundColor = Color.parseJSON(textframeContent.css('background-color'), true);
                if (textFrameBackgroundColor) { allFillColors.push(textFrameBackgroundColor.toJSON()); }
            }

            // adding fill color of paragraph and character (order is important)
            allFillColors.push(paragraphAttributes.fillColor);
            allFillColors.push(characterAttributes.fillColor);

            // calculate effective text color, according to fill colors
            // or theme if it's hyperlink, #46253
            if (DOM.isHyperlinkNode(textSpan) && docModel.useSlideMode()) {
                textColor = '#' + docModel.getTheme().getSchemeColor('hyperlink', '00f');
                textDecoration = 'underline';
            } else {
                textColor = self.getCssTextColor(textColor, allFillColors);
            }

            // calculate font height and vertical alignment (according to escapement)
            switch (characterAttributes.vertAlign) {
                case 'sub':
                    position = 'relative';
                    bottom = 0;
                    fontSize = Utils.round(fontSize * 0.66, 0.1);
                    break;
                case 'super':
                    position = 'relative';
                    bottom = Utils.convertToEM(fontSize * 0.5, 'pt');
                    fontSize = Utils.round(fontSize * 0.66, 0.1);
                    break;
            }

            // inside comments the paragraph bottom margin must be 0 (even if this was not applied via operation),
            // the font size is set to a maximum value and the line height is set to single.
            if (!docModel.getApp().isODF() && !docModel.getCommentLayer().isEmpty() && DOM.isNodeInsideComment(textSpan)) {
                if (fontSize > commentMaxFontSize) {
                    characterAttributes.fontSize = commentMaxFontSize;
                    fontSize = commentMaxFontSize;
                }
                effectiveLineHeight = LineHeight.SINGLE;
            }

            // update CSS of the text span
            textSpan.css({
                color: textColor,
                fontSize: Utils.convertToEM(fontSize, 'pt'),
                textDecoration: textDecoration,
                position: position,
                bottom: bottom
            });

            // update line height due to changed font settings
            self.updateElementLineHeight(textSpan, effectiveLineHeight, characterAttributes);

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(textSpan, mergedAttributes);

            // TODO: set bullet character formatting according to paragraph attributes
            // Bug 30794: We want have a consistent list-label formatting, therefore only the
            // code in updateList in editor should be responsible to set character formatting.
            // This should be extended as the user is currently only able to change the list-label
            // formatting via character attributes in paragraph styles.
            //$(paragraph).find('> ' + DOM.LIST_LABEL_NODE_SELECTOR + ' > span').css('font-size', characterAttributes.fontSize + 'pt');
        }

        // public methods -----------------------------------------------------

        /**
         * Sets the text line height of the specified DOM element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose line height will be changed. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} lineHeight
         *  The new line height value. Must contain the properties 'type' and
         *  'value'.
         *
         * @param {Object} attributes
         *  Character formatting attributes influencing the normal line height.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         */
        this.updateElementLineHeight = function (element, lineHeight, charAttributes) {

            var calced = this.calculateLineHeight(lineHeight, charAttributes);
            $(element).first().css('line-height', calced.val + calced.type);
        };

        /**
         * calculate text line height for assigned lineHeight options.
         *
         * @param {Object} lineHeight
         *  The new line height value. Must contain the properties 'type' and
         *  'value'.
         *
         * @param {Object} attributes
         *  Character formatting attributes influencing the normal line height.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         *  @return {Object}
         *      object with type & value
         *      { val: 5, type: 'px'} or { val: 35.1, type: '%'}
         */
        this.calculateLineHeight = function (lineHeight, charAttributes) {

            var // the font collection
                fontCollection = docModel.getFontCollection(),
                // effective line height in pixels (start with passed value, converted from 1/100 mm)
                height = Utils.convertHmmToLength(lineHeight.value, 'px'),
                // type could be 'px' or '%
                type = 'px',
                //
                normalLineHeight = 0;

            if (lineHeight.type === 'fixed') {
                //line height as passed
            } else {
                normalLineHeight = fontCollection.getNormalLineHeight(charAttributes);

                // calculate effective line height
                switch (lineHeight.type) {
                    case 'leading':
                        height += normalLineHeight;
                        break;
                    case 'atLeast':
                        height = Math.max(height, normalLineHeight);
                        break;
                    case 'percent':
                        height = normalLineHeight * lineHeight.value / 100;
                        break;
                    case 'normal':
                        height = normalLineHeight;
                        break;
                    default:
                        Utils.error('CharacterStyles.updateElementLineHeight(): invalid line height type');
                }

                //transform to relative size, (lineHeight / fontSize)
                height = (height / normalLineHeight) * 100;
                type = '%';
            }

            return { val: height, type: type };
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributeDefinitions('character', DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = self = null;
        });

    } // class CharacterStyles

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: CharacterStyles });

});
