/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/textframework/components/field/basefield', [
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/textframework/utils/textutils'
], function (DateUtils, TriggerObject, Utils) {

    'use strict';

    // class BaseField ============================================

    /**
     * An instance of this class represents the base class for all fields in the
     * edited document.
     *
     * @constructor
     *
     */
    function BaseField(app) {

        var
            self = this,
            // instance of the model
            model = null,
            // instance of the number formatter
            numberFormatter = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        // methods ------------------------------------------------------------

        /**
         * Helper functions to determine type of field from instruction
         */
        this.isNumPages = function (type) {
            return (/NUMPAGES/i).test(type);
        };

        this.isPageNumber = function (type) {
            return (/PAGE/i).test(type) && !(/PAGEREF/i).test(type);
        };

        this.isCurrentDate = function (type) {
            return (/^DATE/i).test(type);
        };

        this.isCurrentTime = function (type) {
            return (/^TIME/i).test(type);
        };

        /*
        function isPrintDate(type) {
            return (/PRINTDATE/i).test(type);
        }

        function isSaveDate(type) {
            return (/SAVEDATE/i).test(type);
        }
        */

        this.isFileName = function (type) {
            return (/FILE\-?NAME/i).test(type);
        };

        this.isAuthor = function (type) {
            return type === 'author' || type === 'author-name' || type === 'AUTHOR';
        };

        /**
         * Parse and trim instruction string. Remove escape chars,
         * format specific @ and * switches, and trims whitespaces.
         *
         * @param {String} instruction
         */
        this.trimFormatInstruction = function (instruction) {
            var tempFormat = instruction.replace(/"/g, '');

            if (!tempFormat) { return null; }

            tempFormat = tempFormat.replace(/[@*]/g, '');

            return Utils.trimString(tempFormat);
        };

        /**
         * From raw passed instruction string, extract type,
         * and clean up remaining instruction for future formatting.
         *
         * @param {String} instruction
         *  Instruction to be cleaned up.
         *
         * @returns {Object}
         */
        this.cleanUpAndExtractType = function (instruction) {
            var type = null,
                isDateTime = null;

            instruction = instruction.replace(/\\\* MERGEFORMAT/g, ''); // ignoring merge format for now
            instruction = instruction.replace(/\\\./g, '.'); // strip escape characters
            instruction = instruction.split('\\');
            if (_.isArray(instruction) && instruction.length > 0) {
                type =  Utils.trimString(instruction[0]);

                if ((self.isCurrentDate(type) || self.isCurrentTime(type))) {
                    isDateTime = true;
                }
                if (instruction.length > 1) {
                    if (instruction.length > 2) { // can happen than field can have combined more than one format
                        // first format always wins
                        if (isDateTime) {
                            if (instruction[1].indexOf('@') === 0) {
                                instruction = self.trimFormatInstruction(instruction[1]);
                            }
                        } else {
                            if (instruction[1].indexOf('*') === 0) {
                                instruction = self.trimFormatInstruction(instruction[1]);
                            }
                        }
                    } else {
                        instruction = self.trimFormatInstruction(instruction[1]);
                    }

                } else {
                    instruction = null;
                }
                instruction = instruction || 'default'; // convert null to default format
            }
            return { type: type, instruction: instruction };
        };

        /**
         * Converts number for page and numPages fields into proper format code.
         *
         * @param {Number} number
         *  Number to be inserted into field.
         *
         * @param {String} format
         *  Format of the number inserted into field.
         *
         * @param {Boolean} [numPages]
         *  NumPages field have some formats different from PageNum.
         *
         * @returns {String}
         *  Formatted number.
         */
        this.formatPageFieldInstruction = function (number, format, numPages) {
            if (number) {
                if (format) {
                    if (app.isODF()) {
                        if (format === 'I') {
                            number = numberFormatter.toRoman(number);
                        } else if (format === 'i') {
                            number = numberFormatter.toRoman(number).toLowerCase();
                        } else if (format === 'A') {
                            number = numberFormatter.toAlphabetic(number).toUpperCase();
                        } else if (format === 'a') {
                            number = numberFormatter.toAlphabetic(number);
                        } else {
                            number = number.toString();
                        }
                    } else {
                        if ((/ROMAN/).test(format)) {
                            number = numberFormatter.toRoman(number);
                        } else if ((/roman/).test(format)) {
                            number = numberFormatter.toRoman(number).toLowerCase();
                        } else if ((/ALPHABETIC/).test(format)) {
                            number = numberFormatter.toAlphabetic(number).toUpperCase();
                        } else if ((/alphabetic/).test(format)) {
                            number = numberFormatter.toAlphabetic(number);
                        } else if ((/ArabicDash/).test(format) && !numPages) {
                            number = numberFormatter.toArabicDash(number);
                        } else if (format.indexOf('Hex') > -1 && numPages) {
                            number = numberFormatter.decToHex(number);
                        } else if (format.indexOf('Ordinal') > -1 && numPages) {
                            number = numberFormatter.toOrdinal(number);
                        } else {
                            number = number.toString();
                        }
                    }
                } else {
                    Utils.warn('basefield.formatPageFieldInstruction(): missing format type!');
                }
                return number;
            }

            Utils.error('basefield.formatPageFieldInstruction(): missing number!');
        };

        /**
         * Create current date(time) value string with passed format type.
         *
         * @param {String} formatCode
         * @return {String}
         */
        this.getDateTimeRepresentation = function (formatCode) {
            var parsedFormat = numberFormatter.getParsedFormat(formatCode);
            var serial = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            return numberFormatter.formatValue(parsedFormat, serial);
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
            numberFormatter = model.getNumberFormatter();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = numberFormatter = null;
        });

    } // class BaseField

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: BaseField });
});
